/* fppLib.c - floating-point coprocessor support library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
02j,14nov88,dnw  documentation.
02i,29aug88,gae  documentation.
02h,06jul88,jcf  fixed bug in fppTaskRegsSet introduced v02a.
02g,22jun88,dnw  name tweaks.
		 changed to add task switch hook when fppInit() is called,
		   instead of first time fppCreateHook() is called.
02f,30may88,dnw  changed to v4 names.
02e,28may88,dnw  cleaned-up fppProbe.
02d,29apr88,jcf  removed unnecessary intLock () within fppTaskSwitch().
02c,31mar88,gae  fppProbe() now done in install routine fppInit() and
		   hooks only added if true.
02b,18mar88,gae  now supports both MC68881 & MC68882.
02a,25jan88,jcf  make kernel independent.
01e,20feb88,dnw  lint
01d,05nov87,jlf  documentation
01c,22oct87,gae  changed fppInit to use task create/delete facilities.
	   +jcf  made fppExitTask ... use pTcb not pTcbx
01b,28sep87,gae  removed extraneous logMsg's.
01a,06aug87,gae  written/extracted from vxLib.c
*/

/*
DESCRIPTION
This library provides a low-level interface to the MC68881/MC68882 coprocessor.
To activate this floating point coporcessor support,
fppInit(1) must be called before any tasks using the coprocessor are spawned.
This is done by the root task, usrRoot(2), in usrConfig(1).

Routines fppTaskRegsShow, fppTaskRegsSet, and fppTaskRegsGet allow inspection
and setting of copocessor registers on a per task basis.  FppProbe checks
the presence of the MC68881/MC68882 coprocessor.
Excepting fppProbe, users should not normally use these routines but
use the higher level facilities in dbgLib(1) and usrLib(1).

VX_FP_TASK OPTION BIT
Saving and restoring the floating point context adds to the context switch
time of a task (approximately 10us on a 20Mhz 68020).
Therefore the floating point context is NOT saved and restored for EVERY task.
Only those tasks spawned with the VX_FP_TASK task option bit will
have a valid floating point context saved and restored.

  IF A TASK WILL EXECUTE FLOATING POINT INSTRUCTIONS,
  THE TASK MUST BE SPAWNED WITH THE ``VX_FP_TASK'' TASK OPTION BIT.

INTERRUPT LEVEL
The floating point context is NOT saved and restored for interrupt
service routines connected with intConnect(1).  However, an interrupt
service routine can easily save and restore the floating point context
itself, if necessary, by calling routines in fppALib(2).

SEE ALSO: MC68881/MC68882 User's Manual
*/

#include "vxWorks.h"
#include "taskLib.h"
#include "memLib.h"
#include "iv68k.h"
#include "fppLib.h"

IMPORT fppProbeTrap ();
IMPORT FUNCPTR intVecGet ();
IMPORT TCBX *taskTcbX ();

/* forward declarations */

LOCAL STATUS fppCreateHook();
LOCAL VOID fppDeleteHook ();
LOCAL VOID fppSwitchHook ();


/*******************************************************************************
*
* fppInit - initialize floating-point coprocessor support
*
* This routine must be called before using the floating-point coprocessor.
* It is usually called by the root task, usrRoot(2), in usrConfig(1).
*/

VOID fppInit ()
    {
    if (fppProbe () == OK)
	{
	taskCreateHookAdd (fppCreateHook);
	taskDeleteHookAdd (fppDeleteHook);
	taskSwitchHookAdd (fppSwitchHook);
	}
    }
/*******************************************************************************
*
* fppTaskRegsShow - print contents of a task's floating-point registers
*
* This routine prints to standard out the contents of a task's 
* floating-point registers, in the following format:
*
*   F0 = ...  F4 = ...  Task = ...
*   F1 = ...  F5 = ...  PCR  = ...
*   F2 = ...  F6 = ...  PSR  = ...
*   F3 = ...  F7 = ...  PIAR = ...
*
* Nothing is printed if floating-point registers are not supported.
*/

VOID fppTaskRegsShow (task)
    FAST int task;	/* task to print registers for */

    {
    double fpregs [FP_NUM_REGS];	/* task's floating-point registers */
    int fpcr;
    int fpsr;
    int fpiar;

    if (fppTaskRegsGet (task, fpregs, &fpcr, &fpsr, &fpiar) == OK)
	{
	printf ("\n");
	printf ("F0 = %20g       F4 = %20g       Task = %8x\n",
		fpregs[0], fpregs[4], task);
	printf ("F1 = %20g       F5 = %20g       PCR  = %8x\n",
		fpregs[1], fpregs[5], fpcr);
	printf ("F2 = %20g       F6 = %20g       PSR  = %8x\n",
		fpregs[2], fpregs[6], fpsr);
	printf ("F3 = %20g       F7 = %20g       PIAR = %8x\n",
		fpregs[3], fpregs[7], fpiar);
	}
    }
/*******************************************************************************
*
* fppTaskRegsGet - get a task's floating-point registers from TCB
*
* This routine returns, in the locations whose pointers are
* passed as parameters, the task's floating-point registers and its
* PCR, PSR and PIAR.  The floating-point registers are returned in
* an array containing the 8 registers.
*
* NOTE
* This routine only works well if the specified task is not that
* of the calling task.  If a task tries to discover its own registers,
* the values will be stale, i.e. leftover from the last task switch.
*
* RETURNS: OK or ERROR if no floating-point support or invalid state
*
* SEE ALSO: fppTaskRegsSet(2)
*/

STATUS fppTaskRegsGet (task, fpregs, pFpcr, pFpsr, pFpiar)
    int task;		/* task to get info about         */
    double fpregs[];	/* floating-point registers (0-7) */
    int *pFpcr;		/* control register               */
    int *pFpsr;		/* status register                */
    int *pFpiar;	/* instruction address register   */

    {
    int i;
    FAST FP_CONTEXT *pFpContext;
    FAST TCBX *pTcbX = taskTcbX (task);

    if (pTcbX == NULL)
	return (ERROR);

    pFpContext = (FP_CONTEXT *)pTcbX->fpContext;

    if (pFpContext == (FP_CONTEXT *)NULL)
	return (ERROR);			/* no coprocessor support */
    else 
	if (pFpContext->stateFrame [0] == 0)
	    return (ERROR);		/* invalid state frame */
	else
	    {
	    /* normal/idle state */

	    for (i = 0; i < FP_NUM_REGS; i++)
		fppDxtoD (&fpregs [i], &pFpContext->fpx[i]);

	    *pFpcr  = pFpContext->fpcr;
	    *pFpsr  = pFpContext->fpsr;
	    *pFpiar = pFpContext->fpiar;
	    }

    return (OK);
    }
/*******************************************************************************
*
* fppTaskRegsSet - set a task's floating-point registers 
*
* This routine loads the given values into the given TCB.
* The registers are in an array containing f0-f7.
*
* RETURNS: OK | ERROR if no floating-point support or invalid state
*
* SEE ALSO: fppTaskRegsGet(2)
*/

STATUS fppTaskRegsSet (task, fpregs, fpcr, fpsr, fpiar)
    int task;		/* task whose registers are to be set */
    double fpregs[];	/* floating-point registers (0-7)     */
    int fpcr;		/* control register                   */
    int fpsr;		/* status register                    */
    int fpiar;		/* instruction address register       */

    {
    FAST TCBX *pTcbX = taskTcbX (task);
    FAST FP_CONTEXT *pFpContext;
    int i;

    if (pTcbX == NULL)
	return (ERROR);

    pFpContext = (FP_CONTEXT *)pTcbX->fpContext;

    if (pFpContext == (FP_CONTEXT *)NULL)
	return (ERROR);			/* no coprocessor support */
    else
	if (pFpContext->stateFrame [0] == 0)
	    return (ERROR);			/* invalid state frame */
	else
	    {
	    /* normal/idle state */
	    
	    for (i = 0; i < FP_NUM_REGS; i++)
		fppDtoDx (&pFpContext->fpx[i], &fpregs [i]);

	    pFpContext->fpcr  = fpcr;
	    pFpContext->fpsr  = fpsr;
	    pFpContext->fpiar = fpiar;
	    }

    return (OK);
    }
/*******************************************************************************
*
* fppCreateHook - initialize floating-point coprocessor support for task
*
* RETURNS: OK | ERROR if unable to allocate storage
*/

LOCAL STATUS fppCreateHook (pTcbX)
    FAST TCBX *pTcbX;

    {
    /* check for option bit and presence of floating-point coprocessor */

    if (pTcbX->options & VX_FP_TASK)
	{
	/* allocate space for saving context and registers */

	pTcbX->fpContext = malloc (sizeof (FP_CONTEXT));
	if (pTcbX->fpContext == NULL)
	    return (ERROR);		/* out of memory! */

	taskLock ();
	fppSave ((FP_CONTEXT *)pTcbX->fpContext);
	taskUnlock ();
	}

    return (OK);
    }
/*******************************************************************************
*
* fppDeleteHook - quit floating-point coprocessor support for task
*
* This routine just free's the memory used for the floating-point
* context save area.
*/

LOCAL VOID fppDeleteHook (pTcbX)
    FAST TCBX *pTcbX;	/* task to end support for */

    {
    /* is the floating-point task bit set in the task option word? */

    if ((pTcbX->options & VX_FP_TASK) && (pTcbX->fpContext != NULL))
	free (pTcbX->fpContext);
    }
/*******************************************************************************
*
* fppSwitchHook - switch task floating-point coprocessor registers
*
* This routine is the task switch hook that implements the task
* floating-point coprocessor registers facility.
* It swaps the current and saved values of all the task coprocessor
* registers of the out-going and in-coming tasks.
*/

LOCAL VOID fppSwitchHook (pOldTcbX, pNewTcbX)
    TCBX *pOldTcbX;
    TCBX *pNewTcbX;

    {
    FAST char *pOldFpContext = pOldTcbX->fpContext;
    FAST char *pNewFpContext = pNewTcbX->fpContext;

    /* save task coprocessor registers of old task */

    if (pOldFpContext != NULL)
	fppSave ((FP_CONTEXT *)pOldFpContext);

    /* restore task coprocessor registers of new task */

    if (pNewFpContext != NULL)
	fppRestore ((FP_CONTEXT *)pNewFpContext);
    }
/*******************************************************************************
*
* fppProbe - probe for presence of floating-point coprocessor
*
* This routine finds out whether there is an MC68881/MC68882
* floating-point coprocessor in the system.
*
* IMPLEMENTATION
* This routine functions by setting the illegal coprocessor opcode trap
* vector to fppProbeTrap and then executing a coprocessor instruction.
* If the instruction causes an exception then fppProbeTrap will return ERROR.
* Note that this routine saves and restores the illegal coprocessor opcode
* trap vector that was there prior to this call.
*
* The actual probe is only done the first time this routine is called.
* The result is stored in a static and returned on subsequent
* calls without actually probing.
*
* RETURNS:
*    OK if floating-point coprocessor present,
*    otherwise ERROR
*/

STATUS fppProbe ()

    {
    static int fppProbed = -2;		/* -2 = not done, -1 = ERROR, 0 = OK */
    FUNCPTR oldVec;

    if (fppProbed == -2)
	{
	oldVec = intVecGet (IV_LINE_1111_EMULATOR);	/* save error vector */
	intVecSet (IV_LINE_1111_EMULATOR, fppProbeTrap);/* replace error vec */

	fppProbed = fppProbeSup ();	/* execute coprocessor instruction */

	intVecSet (IV_LINE_1111_EMULATOR, oldVec);	/* replace old err vec*/
	}

    return (fppProbed);
    }
