/* excLib.c - exception handling library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
02r,21apr89,jcf  changed excExcHandle to taskDelay () after excJobAdd ()
		  instead of taskSuspend ().
02q,14jan89,jcf  changed handling of uninitialized interrupts before excTask
		  is spawned so messages are now sent to sysExcMsg.
		 changed excIntPrint () to excIntInfoShow () so it more closely
		  parallels handling of exceptional exceptions.
02p,10nov88,dnw  fixed bug in determining when exception occured at int level.
		 fixed bug in null terminating sysExcMsg.
02o,20oct88,jcf  fixed bug in which excTask was setting itself breakable.
02n,19aug88,gae  documentation, made excInfoShow no manual.
02m,04jul88,jcf  null terminated sysExcMsg, so bootroms can print it out.
02l,30jun88,rdc  changed sigDeliverSignal to sigDeliver.
02k,22jun88,dnw  name tweaks.
		 removed special treatment of taskRegsSet() for vrtx (now done
		   in vrtx taskRegsSet() itself).
02j,05jun88,dnw  changed dbgSetTaskBreakable() to taskOptionsSet().
		 changed taskSpawn/taskCreate args.
02i,30may88,dnw  changed to v4 names.
02h,29may88,dnw  changed to preserve task unbreakable option after exception
		   handling.
		 cleaned-up interaction with sigLib.
02g,28may88,dnw  changed to save exception info in tcbx.
02f,24may88,gae  cleanup, lint.  Fixed bug in excExcFix() of printing bogus
	   +ecs  ESF.
02e,13may88,rdc  arghh! fixed a compiler warning.
02d,13may88,rdc  added signal stuff.
02c,28apr88,ecs  removed something foolish from 02b.
02b,24mar88,ecs  changed excPrintEsf to sprint if exception happened during
		    an interrupt.
		 changed excExcFix to print the task id in hex, and as a string.
		 changed to use logMsg instead of printErr.
		 added include of taskLib.h.
02a,26jan88,jcf  made kernel independent.
01n,13nov87,jcf  changed call to sysToMonitor into reboot
	    ecs  documentation.
01m,03nov87,dnw  changed to let system pick excTask id;
		   result available in excTaskId.
		 fixed bug not printing specific msgs for well-known interrupts.
		 changed to use INUM_TO_IVEC where appropriate.
		 changed TAS retry to only happen on bus errors, not adrs errs.
		 changed to count number of TAS bus errors in excTasErrors
		   (just out of curiosity).
01l,01nov87,jcf  added code to retry aborted RMW cycles in excExcFix
01k,01apr87,ecs  hushed lint in excExcFix.
01j,25mar87,jlf  documentation.
		 added copyright.
01i,01mar87,dnw  fixes to 01h mods.
01h,26feb87,rdc  modifications for VRTX 3.2.
01g,21dec86,dnw  changed to not get include files from default directories.
01f,27oct86,llk  set interrupt vectors with intSetVec.
01e,04sep86,jlf  documentation.
01d,31jul86,llk  uses new spawn
01c,26jul86,dnw  changed to not call external exception routine if exception
		   is uninitialized interrupt.
		 changed to use BSR table to distinquish interrupts on 68000.
		 changed to use printErr instead of printf.
01b,03jul86,dnw  documentation.
01a,03apr86,dnw  extracted from dbgLib.c
*/

/*
This module provides facilities for handling 680x0 exceptions.
When installed, this package will safely trap and report exceptions
caused by program errors in VxWorks tasks,
and will report occurrence of any interrupts that have not been explicitly
connected to other handlers.

INITIALIZATION
Initialization of the exception handling package is in two parts.
First, the routine excVecInit is called to set all the 680x0
exception, trap, and interrupt vectors to the default handlers provided
by this module.
This does not involve any VxWorks kernel facilities,
and so is normally done very early in the system start-up routine
usrInit (2) in usrConfig (1), with interrupts disabled.

The rest of this package is initialized by calling excInit.
This routine spawns the exception support task, excTask, and creates
the pipe used to communicate with it.
This initialization uses VxWorks' kernel facilities and the pipe driver,
and so must be performed in the root task, usrRoot (2) in usrConfig (1),
after the pipe driver has been installed.

Exceptions or uninitialized interrupts that occur after the vectors 
have been initialized by the excVecInit routine, but before the
the excInit routine has initialized the rest of this package,
simply cause a trap to the ROM monitor.

NORMAL EXCEPTION HANDLING
When a program error generates an exception
(divide by 0, bus error, address error, etc.)
the task that was executing when the error occurred is
suspended and a description of the exception is printed on standard output.
The VxWorks kernel and other tasks in the system continue uninterrupted.
The suspended task may then be examined with the usual VxWorks routines,
including ti (2) for task information, and tt (2) for a stack trace.
It may be possible to fix the task and resume its execution 
(with the routine tr (2)).
However, tasks that have aborted in this way are often unsalvageable.
The task can be deleted in the usual way with the routine "td".

When an interrupt occurs which has not been connected to a handler,
the default handler provided by this module just prints a description
of the interrupt on standard output.

ADDITIONAL EXCEPTION HANDLING ROUTINE
Additional exception processing can be added from outside of
this module by calling the routine excHookAdd.
This routine allows a routine to be specified that will be called
whenever a hardware exception occurs.  This routine will be called
at the end of the normal exception handling.

EXCTASK - TASK LEVEL SUPPORT
A special task, excTask, is spawned by excInit to perform 
certain exception handling functions that need to be done at task level.
Do not suspend it, delete it, or change its priority.

DBGLIB
The facilities of excLib, including excTask, are used by dbgLib
to support breakpoints and single-stepping and additional exception
handling functions.

SIGLIB
SigLib provides a higher level UNIX compatible interface for
hardware and software exceptions.  If a "sigvec" is initialized
for the appropriate hardware exception/interrupt, eg. BUS ERROR == SIGSEGV,
excLib will use the signal mechanism instead.

SEE ALSO: dbgLib (1), sigLib (1), intLib (1)
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "esf68k.h"
#include "iv68k.h"
#include "taskLib.h"
#include "ioLib.h"
#include "sysLib.h"
#include "taskLib.h"


#if (CPU==MC68000)
IMPORT int excBsrTbl [];	/* table of BSRs in excALib.s */
#else
IMPORT excStub ();		/* exception driver in excALib.s */
IMPORT excIntStub ();		/* interrupt driver in excALib.s */
#endif

#define MAX_ARGS	6	/* max args to task level call */

/* vector numbers of specific exceptions */

#define LOW_VEC		2	/* lowest vector initialized */
#define BUS_ERROR_VEC	2	/* bus error vector */
#define ADRS_ERROR_VEC	3	/* address error vector */
#define FORMAT_ERROR	14	/* last program error vector */
#define TRAP_0_VEC	32	/* start of trap vectors */
#define USER_VEC_START	64	/* start of user interrupt vectors */
#define HIGH_VEC	255	/* highest vector initialized */


typedef struct		/* EXCMSG - message to excTask */
    {
    FUNCPTR funct;		/* function to call */
    int arg[MAX_ARGS];		/* args to pass to funct */
    } EXCMSG;

/* Exception error messages.  These are used by the exception printing routine.
   Exception numbers are the same as used by the CPU */

LOCAL char *excMsgs [] =
    {
    NULL,				/* reset sp */
    NULL,				/* reset pc */
    "Bus Error",
    "Address Error",
    "Illegal Instruction",
    "Zero Divide",
    "CHK Trap",
    "TRAPV Trap",
    "Privelege Violation",
    "Trace Exception",
    "Unimplemented Opcode 1010",
    "Unimplemented Opcode 1111",
    NULL,				/* 12 unassigned */
    "Coprocessor Protocol Violation",	/* 68020 only */
    "Format Error",			/* 68010, 68020 only */
    "Uninitialized Interrupt",
    NULL, NULL, NULL, NULL,		/* 16-23 unassigned */
    NULL, NULL, NULL, NULL,
    "Spurious Interrupt",
    };


/* global variables */

/* excTask parameters */

int excTaskId;
int excTaskPriority	= 0;
int excTaskOptions	= VX_SUPERVISOR_MODE | VX_UNBREAKABLE;
int excTaskStackSize	= 3000;

char *excTaskPipeName	= "/pipe/exc";

/* local variables */

LOCAL int excPipe = NONE;	/* fd of pipe to excTask */
LOCAL int excMsgsLost;		/* count of messages to excTask lost */
LOCAL int excTasErrors;		/* count of TAS bus errors - just curiosity */
LOCAL FUNCPTR excExcepHook;	/* add'l rtn to call when exceptions occur */



/* forward declarations */

LOCAL VOID excIntInfoShow ();
LOCAL VOID excExcFix ();
VOID excTask ();

/*******************************************************************************
*
* excVecInit - initialize exception/interrupt vectors
*
* This routine sets all exception vectors to point to the appropriate
* default exception handlers.
* All vectors from vector 2 (address 0x0008) to vector 255 (address 0x03fc)
* are initialized.
* (Vectors 0 and 1 contain the reset stack pointer and program counter.)
*
* WHEN TO CALL
* This routine is usually called from the system start-up routine,
* usrInit (2) in usrConfig (1), before interrupts are enabled.
*
* RETURNS: OK (always)
*/

STATUS excVecInit ()

    {
    FAST int vecNum;

#if (CPU==MC68000)
    /* make exception vectors point to proper place in bsr table */

    for (vecNum = LOW_VEC; vecNum <= HIGH_VEC; ++vecNum)
	intVecSet (INUM_TO_IVEC (vecNum), (FUNCPTR) &excBsrTbl[vecNum]);

#else

    /* make all exception vectors point to either generic exception or
     * interrupt trap handler */

    for (vecNum = LOW_VEC; vecNum <= HIGH_VEC; ++vecNum)
	intVecSet (INUM_TO_IVEC (vecNum),
		   programError (vecNum) ? excStub : excIntStub);

#endif

    return (OK);
    }
/*******************************************************************************
*
* excInit - initialize exception handling package
*
* This routine installs the exception handling package.
* This involves 
*
*   - spawning excTask, which handles exception handling 
*     functions that must happen at task level,
*
*   - creating the pipe used to communicate with excTask.
*
* WHEN TO CALL
* It is usually desirable to install the exception handling
* facilities as early as possible in system initialization,
* in the root task usrRoot (2) in usrConfig (1).
* It should not, however, be called until the pipe driver has been installed.
*
* RETURNS:
* OK, or
* ERROR if can't open a pipe or can't spawn excTask
*/

STATUS excInit ()

    {
    if (pipeDevCreate (excTaskPipeName, 10, sizeof (EXCMSG)) != OK ||
	(excPipe = open (excTaskPipeName, UPDATE, 0)) < 0)
	{
	return (ERROR);
	}

    excTaskId = taskSpawn ("excTask", excTaskPriority,
			   excTaskOptions, excTaskStackSize,
			   excTask, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);

    return (excTaskId == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* excHookAdd - specify routine to be called on exceptions
*
* This routine allows a routine to be specified that will be called
* whenever a hardware exception occurs.  This routine will be called
* at the end of the normal exception handling, which includes suspending
* the task that incurred the exception and printing out all pertinent
* information about the exception.
*
* The additional exception handling routine should be declared as follows:
* .CS
*
*   VOID myHandler (task, vecNum, pEsf)
*      int task;	    /* id of offending task *
*      int vecNum;	    /* exception vector number *
*      ESF0 *pEsf;          /* pointer to exception stack frame *
*
* .CE
*
* The "task" is the id of the task that was running when the exception
* occurred.
*
* This facility is normally used by dbgLib (1) to activate its exception
* mechanism.  If an application provides its own handler, it will deactivate
* dbgLib's mechanism.
*/

VOID excHookAdd (excepHook)
    FUNCPTR excepHook;	/* routine to be called when exceptions occur */

    {
    excExcepHook = excepHook;
    }
/*******************************************************************************
*
* excJobAdd - request a task-level function call from interrupt level
*
* This routine allows interrupt level code to request a function call
* to be made by excTask at task-level.
*
* VARARGS1
* ARGSUSED
* NOMANUAL
*/

STATUS excJobAdd (func, arg1, arg2, arg3, arg4, arg5, arg6)
    FUNCPTR func;
    int arg1, arg2, arg3, arg4, arg5, arg6;

    {
    EXCMSG excmsg;	/* message to excTask gets formed here */


    bcopy ((char *)&func, (char *)&excmsg.funct,
	   sizeof (func) + MAX_ARGS * sizeof (arg1));

    if (write (excPipe, (char *)&excmsg, sizeof (excmsg)) != sizeof (excmsg))
	{
	++excMsgsLost;
	return (ERROR);
	}
    
    return (OK);
    }
/*******************************************************************************
*
* excTask - exception handling task-level support task
*
* This routine is spawned as a task to support the exception handling package.
* It is spawned by excInit (2) with a priority of 0.  Its purpose is to
* perform functions that can not be done at interrupt or trap level.
*
* Do not suspend, delete, or change the priority of this task.
*
* SEE ALSO: excInit (2)
*/

VOID excTask ()

    {
    EXCMSG excmsg;	/* messages from interrupt level go here */
    int newMsgsLost;
    int nchars;
    static int oldMsgsLost = 0;

    FOREVER
	{
	/* call each function that has been queued for us to do */

	do
	    {
	    if (read (excPipe, (char *)&excmsg, sizeof (excmsg)) <= 0)
		{
		logMsg ("excTask: error reading msg, status = 0x%x.\n",
			errnoGet());
		break;
		}

	    (*(FUNCPTR)excmsg.funct) (excmsg.arg[0], excmsg.arg[1],
				      excmsg.arg[2], excmsg.arg[3],
				      excmsg.arg[4], excmsg.arg[5]);
	    }
	while ((ioctl (excPipe, FIONREAD, &nchars) == OK) && nchars > 0);


	/* check to see if interrupt level lost any more calls */

	if ((newMsgsLost = excMsgsLost) != oldMsgsLost)
	    {
	    logMsg ("%d messages from interrupt level lost.\n",
		    newMsgsLost - oldMsgsLost);

	    oldMsgsLost = newMsgsLost;
	    }
	}
    }
/*******************************************************************************
*
* excIntHandle - interrupt level handling of interrupts
*
* This routine handles interrupts. It is never be called except 
* from the special assembly language interrupt stub routine.
*
* It prints out a bunch of pertinent information about the trap that
* occurred via excTask.
*
* NOMANUAL
*/

VOID excIntHandle (vecNum, pEsf)
    int vecNum;		/* exception vector number */
    ESF0 *pEsf;		/* pointer to exception stack frame */

    {
    /* if excTask is not initialized, we can't handle it in the usual way.  
     * Instead, we save info in sysExcMsg and trap to rom monitor.
     */

    if (excPipe == NONE)		/* no exception task */
	{
	/* note: 1st byte of exception msg will get clobbered in bootConfig */

        excIntInfoShow (vecNum, pEsf->pc, pEsf->sr);

	reboot (BOOT_WARM_AUTOBOOT);
	return;			/* in case user tries to return from monitor */
	}

    /* send info to excTask */

    (void) excJobAdd (excIntInfoShow, vecNum, pEsf->pc, pEsf->sr);
    }
/*******************************************************************************
*
* excIntInfoShow - print out uninitialized interrupt info
*/

LOCAL VOID excIntInfoShow (vecNum, pc, sr)
    int vecNum;
    INSTR *pc;
    unsigned short sr;

    {
    printExc ("Interrupt to uninitialized vector number %d (0-255).\n",
	      vecNum);

    /* translate to vecNum to English if possible */

    if ((vecNum < NELEMENTS (excMsgs)) && (excMsgs [vecNum] != NULL))
	printExc ("Equivalent to vector-> %s\n", excMsgs [vecNum]);

    printExc ("Program Counter: 0x%08x\n", pc);
    printExc ("Status Register: 0x%04x\n", sr);
    }
/*******************************************************************************
*
* excExcHandle - interrupt level handling of exceptions
*
* This routine handles exception traps. It is never be called except 
* from the special assembly language interrupt stub routine.
*
* It prints out a bunch of pertinent information about the trap that
* occurred via excTask.
*
* Note that this routine runs in the context of the task that got the exception.
*
* NOMANUAL
*/

VOID excExcHandle (vecNum, pEsf, regs)
    int vecNum;		/* exception vector number */
    ESF0 *pEsf;		/* pointer to exception stack frame */
    int *regs;		/* pointer to register info on stack */

    {
    EXC_INFO excInfo;
    int origOptions;

    /* if this was a bus error involving a RMW cycle (TAS instruction) we
     * return to the handler to retry it.
     */

#if (CPU==MC68000)
    if ((vecNum == BUS_ERROR_VEC) &&				/* BERR! */
	(((ESF68K_BA *)pEsf)->fn & 0x800))			/* RMW cycle */
#else
    if (((((pEsf->vectorOffset & 0xf000) >> 12) == 10) ||
	 (((pEsf->vectorOffset & 0xf000) >> 12) == 11)) &&	/* BERR! */
	(((ESFA *)pEsf)->fn & 0x80))				/* RMW cycle */
#endif
	{
	++excTasErrors;				/* keep count of TAS errors */
	return;					/* retry the instruction */
	}

    /* if excTask is not initialized or
     * we got a program error at interrupt level
     * then we can't handle it in the usual way.  
     * Instead, we save info in sysExcMsg and trap to rom monitor.
     */

    if ((excPipe == NONE) || (intCount () > 0))
	{
	excGetInfoFromESF (vecNum, pEsf, &excInfo);

	/* note: 1st byte of exception msg will get clobbered in bootConfig */

	printExc (" \nException at interrupt level:\n");
	excInfoShow (&excInfo);		/* print the message into sysExcMsg */
	printExc ("Regs at 0x%x\n", regs);

	reboot (BOOT_WARM_AUTOBOOT);
	return;			/* in case user tries to return from monitor */
	}

    /* send info to excTask */

    taskOptionsGet (0, &origOptions);			/* save original opts */
    taskOptionsSet (0, VX_UNBREAKABLE, VX_UNBREAKABLE);	/* set unbreakable */

    (void) excJobAdd (excExcFix, taskIdSelf(), vecNum, pEsf, regs, origOptions);

    FOREVER
	taskDelay (1);	/* delay until excTask handles us */
    }
/*******************************************************************************
*
* excExcFix - task level handling of exceptions
*
* This routine handles tasks that have encountered an exception.
* This routine suspends the offending task, repairs its tcb so that the
* the task's apparent context is that of when the exception occurred,
* and prints out info about the exception.
*/

LOCAL VOID excExcFix (taskId, vecNum, pEsf, regs, origOptions)
    FAST int taskId;	/* task id of offending task */
    FAST int vecNum;	/* vector number of exception */
    ESF0 *pEsf;		/* pointer to exception stack frame */
    int *regs;		/* pointer to register info on stack */
    int origOptions;	/* original task options prior to exception processing*/

    {
    ESFB esfBuf;
    int esfSize;
    char *sp;
    TCBX *pTcbX;
    FAST EXC_INFO *pExcInfo;
    
    pTcbX = taskTcbX (taskId);		/* get ptr to tcbx */
    if (pTcbX == NULL)
	{
	logMsg ("excExcFix: invalid task id %#x\n", taskId);
	return;
	}

    pExcInfo = &pTcbX->excInfo;		/* get ptr to exc info in tcbx */

    taskIdDefault (taskId);		/* update default task */

    if (!(origOptions & VX_UNBREAKABLE)) /* make task breakable again */
	taskOptionsSet (taskId, VX_UNBREAKABLE, 0);

    /* copy the ESF to local buffers -
     * this is necessary only because in vrtx some registers
     * are stored on top of the stack, right where the ESF is now */

    bcopy ((char *) pEsf, (char *) &esfBuf, sizeof (ESFB));

    /* set the tasks exception info and calculate ESF size */

    esfSize = excGetInfoFromESF (vecNum, (ESF0 *) &esfBuf, pExcInfo);
    if (esfSize == ERROR)
	{
	logMsg ("excExcFix: invalid ESF type %#x\n", pExcInfo->funcCode);
	return;
	}

    /* fix task stack pointer and registers */

    sp = (char *) pEsf + esfSize;
    taskRegsSet (taskId, regs, &regs[8], sp, pExcInfo->statusReg, pExcInfo->pc);

    /* if there's a signal handler for the task, deliver the appropriate
     * signal for the exception */

    if (excDeliverSignal (taskId, vecNum) == ERROR)
	{
	taskSuspend (taskId);

	/* print exception info */

	excInfoShow (pExcInfo);
	printExc ("Task: %#x \"%s\"\007\n", taskId, taskName (taskId));
	}

    /* call any specified additional routine */

    if (excExcepHook != NULL)
	(* excExcepHook) (taskId, vecNum, &esfBuf);
    }
/*****************************************************************************
*
* excGetInfoFromESF - get relevent info from exception stack frame
*
* RETURNS: size of specified ESF
*/

LOCAL int excGetInfoFromESF (vecNum, pEsf, pExcInfo)
    FAST int vecNum;
    FAST ESF0 *pEsf;
    EXC_INFO *pExcInfo;

    {
    pExcInfo->vecNum = vecNum;
    pExcInfo->valid  = EXC_VEC_NUM;

#if (CPU==MC68000)
    if ((vecNum == BUS_ERROR_VEC) || (vecNum == ADRS_ERROR_VEC))
	{
	/* Its an address or bus error */

	pExcInfo->valid     |= EXC_PC | EXC_STATUS_REG | EXC_ACCESS_ADDR |
			       EXC_FUNC_CODE | EXC_INSTR_REG;
	pExcInfo->pc         = ((ESF68K_BA *)pEsf)->pc;
	pExcInfo->statusReg  = ((ESF68K_BA *)pEsf)->sr;
	pExcInfo->accessAddr = ((ESF68K_BA *)pEsf)->aa;
	pExcInfo->funcCode   = ((ESF68K_BA *)pEsf)->fn;
	pExcInfo->instrReg   = ((ESF68K_BA *)pEsf)->ir;

	return (sizeof (ESF68K_BA));
	}
    else
	{
	pExcInfo->valid    |= EXC_PC | EXC_STATUS_REG;
	pExcInfo->pc        = ((ESF68K *)pEsf)->pc;
	pExcInfo->statusReg = ((ESF68K *)pEsf)->sr;

	return (sizeof (ESF68K));
	}
#else

    /* switch on ESF type */

    switch ((pEsf->vectorOffset & 0xf000) >> 12)
	{
	case 0:
	case 1: 
	    pExcInfo->valid    |= EXC_PC | EXC_STATUS_REG;
	    pExcInfo->pc        = pEsf->pc;
	    pExcInfo->statusReg = pEsf->sr;

	    return (sizeof (ESF0));

	case 2:
	    pExcInfo->valid     |= EXC_PC | EXC_STATUS_REG | EXC_ACCESS_ADDR;
	    pExcInfo->pc         = ((ESF2 *)pEsf)->pc;
	    pExcInfo->statusReg  = ((ESF2 *)pEsf)->sr;
	    pExcInfo->accessAddr = ((ESF2 *)pEsf)->aa; 

	    return (sizeof (ESF2));

	case 8:
	    pExcInfo->valid     |= EXC_PC | EXC_STATUS_REG | EXC_ACCESS_ADDR |
				   EXC_FUNC_CODE | EXC_INSTR_REG;
	    pExcInfo->pc         = ((ESF8 *)pEsf)->pc;
	    pExcInfo->statusReg  = ((ESF8 *)pEsf)->sr;
	    pExcInfo->accessAddr = ((ESF8 *)pEsf)->aa;
	    pExcInfo->funcCode   = ((ESF8 *)pEsf)->fn;
	    pExcInfo->instrReg   = ((ESF8 *)pEsf)->ir; 

	    return (sizeof (ESF8));

	case 9:
	    pExcInfo->valid     |= EXC_PC | EXC_STATUS_REG | EXC_ACCESS_ADDR;
	    pExcInfo->pc         = ((ESF9 *)pEsf)->pc;
	    pExcInfo->statusReg  = ((ESF9 *)pEsf)->sr;
	    pExcInfo->accessAddr = ((ESF9 *)pEsf)->aa; 

	    return (sizeof (ESF9));

	case 10:
	    pExcInfo->valid     |= EXC_PC | EXC_STATUS_REG | EXC_ACCESS_ADDR |
				   EXC_FUNC_CODE | EXC_INSTR_REG;
	    pExcInfo->pc         = ((ESFA *)pEsf)->pc;
	    pExcInfo->statusReg  = ((ESFA *)pEsf)->sr;
	    pExcInfo->accessAddr = ((ESFA *)pEsf)->aa;
	    pExcInfo->funcCode   = ((ESFA *)pEsf)->fn;
	    pExcInfo->instrReg   = ((ESFA *)pEsf)->instPipeC; 

	    return (sizeof (ESFA));

	case 11:
	    pExcInfo->valid     |= EXC_PC | EXC_STATUS_REG | EXC_ACCESS_ADDR |
				   EXC_FUNC_CODE | EXC_INSTR_REG;
	    pExcInfo->pc         = ((ESFA *)pEsf)->pc;
	    pExcInfo->statusReg  = ((ESFA *)pEsf)->sr;
	    pExcInfo->accessAddr = ((ESFA *)pEsf)->aa;
	    pExcInfo->funcCode   = ((ESFA *)pEsf)->fn;
	    pExcInfo->instrReg   = ((ESFA *)pEsf)->instPipeC; 

	    return (sizeof (ESFB));

	default:
	    pExcInfo->valid   |= EXC_INVALID_TYPE;
	    pExcInfo->funcCode = ((pEsf->vectorOffset & 0xf000) >> 12);

	    return (ERROR);
	}
#endif
    }
/*******************************************************************************
*
* excInfoShow - print exception info
*
* NOMANUAL
*/

VOID excInfoShow (pExcInfo)
    FAST EXC_INFO *pExcInfo;

    {
    FAST int valid  = pExcInfo->valid;
    FAST int vecNum = pExcInfo->vecNum;

    /* print each piece of info if valid */

    if (valid & EXC_VEC_NUM)
	{
	if ((vecNum < NELEMENTS (excMsgs)) && (excMsgs [vecNum] != NULL))
	    printExc ("\n%s\n", excMsgs [vecNum]);	/* specific error msg */
	else
	    printExc ("\nTrap to uninitialized vector number %d (0-255).\n",
		      vecNum);
	}

    if (valid & EXC_PC)
	printExc ("Program Counter: 0x%08x\n", pExcInfo->pc);

    if (valid & EXC_STATUS_REG)
	printExc ("Status Register: 0x%04x\n", pExcInfo->statusReg);

    if (valid & EXC_ACCESS_ADDR)
	printExc ("Access Address : 0x%08x\n", pExcInfo->accessAddr);

    if (valid & EXC_INSTR_REG)
	printExc ("Instruction    : 0x%04x\n", pExcInfo->instrReg);

    if (valid & EXC_FUNC_CODE)
#if (CPU==MC68000)
	printExc ("Access Type    : %s %c FC%x\n",
		  (pExcInfo->funcCode & 0x10) ? "Read": "Write",
		  (pExcInfo->funcCode & 0x08) ? 'I' : 'N',
		  pExcInfo->funcCode & 0x07);
#else
	printExc ("Special Status : 0x%04x\n", pExcInfo->funcCode);
#endif

    if (valid & EXC_INVALID_TYPE)
	printExc ("Invalid ESF type 0x%x\n", pExcInfo->funcCode);
    }
/*******************************************************************************
*
* printExc - print error message, into sysExcMsg if necessary
*
* VARARGS1
*/

LOCAL VOID printExc (fmt, args)
    char *fmt;		/* format string to print from */
    int args;		/* 1st of up to 5 optional arguments for fmt string */

    {
    int *pArgs = &args;	/* pointer to optional arguments */

    if ((excPipe == NONE) || (intCount () > 0))
	{
	/* Exception happened during exception processing, or before
	 * excTask could be initialized. Tack the message onto the end
	 * of a well-known location. */

	sysExcMsg += sprintf (sysExcMsg, fmt, pArgs[0], pArgs[1], pArgs[2],
			      pArgs[3], pArgs[4]) - 1;	/* subtract EOS */
	}
    else
	printErr (fmt, pArgs[0], pArgs[1], pArgs[2], pArgs[3], pArgs[4]);
    }
/*****************************************************************************
*
* excDeliverSignal - signal a task that has an exception
*
* RETURNS: OK if the signal was delivered successfully, otherwise ERROR
*/

LOCAL STATUS excDeliverSignal (taskId, vecNum)
    FAST int taskId;	/* task to deliver signal to */
    int vecNum;		/* exception vector number */

    {
    FAST int signal;
    FAST int code;

    /* map the exception to a signal and a "code" */

    switch (vecNum)
	{
	case 2:		/* BUS ERROR */
	    /* ADA catches SIGSEGV for nonexistant memory access.
	     * This should really be SIGBUS */

/* XXX	    signal = SIGBUS; */
/* XXX	    code   = BUS_BUSERR; */

	    signal = SIGSEGV; 
	    code   = NULL; 
	    break;

	case 3:		/* ADDRESS ERROR */
	    signal = SIGBUS;
	    code   = BUS_ADDERR;
	    break;
	
	case 4:		/* ILLEGAL INSTRUCTION */
	    signal = SIGILL;
	    code   = ILL_ILLINSTR_FAULT;
	    break;

	case 5:		/* ZERO DIVIDE */
	    signal = SIGFPE;
	    code   = FPE_INTDIV_TRAP;
	    break;

	case 6:		/* CHK TRAP */
	    signal = SIGFPE;
	    code   = FPE_CHKINST_TRAP;
	    break;
	
	case 7:		/* TRAPV TRAP */
	    signal = SIGFPE;
	    code   = FPE_TRAPV_TRAP;
	    break;

	case 8:		/* PRIVILEGE VIOLATION */
	    signal = SIGILL;
	    code   = ILL_PRIVVIO_FAULT;
	    break;

	case 9:		/* TRACE EXCEPTION */
	    signal = SIGTRAP;
	    code   = NULL;
	    break;

	case 10:	/* LINE 1010 EMULATOR */
	    signal = SIGEMT;
	    code   = EMT_EMU1010;
	    break;

	case 11:	/* LINE 1111 EMULATOR */
	    signal = SIGEMT;
	    code   = EMT_EMU1111;
	    break;

	case 13:	/* COPROCESSOR PROTOCAL VIOLATION */
	    signal = SIGILL;
	    code   = ILL_ILLINSTR_FAULT;
	    break;

	case 14:	/* FORMAT ERROR */
	    signal = SIGFMT;
	    code   = NULL;
	    break;

	/* floating point coprocessor stuff */

	case 48:	/* COMPARE UNORDERED */
	    signal = SIGFPE;
	    code   = FPE_FLTBSUN_TRAP;
	    break;

	case 49:	/* INEXACT RESULT */
	    signal = SIGFPE;
	    code   = FPE_FLTINEX_TRAP;
	    break;

	case 50:	/* DIVIDE BY ZERO */
	    signal = SIGFPE;
	    code   = FPE_FLTDIV_TRAP;
	    break;

	case 51:	/* UNDERFLOW */
	    signal = SIGFPE;
	    code   = FPE_FLTUND_TRAP;
	    break;
	
	case 52:	/* OPERAND ERROR */
	    signal = SIGFPE;
	    code   = FPE_FLTOPERR_TRAP;
	    break;

	case 53:	/* OVERFLOW */
	    signal = SIGFPE;
	    code   = FPE_FLTOVF_TRAP;
	    break;
	    
	case 54:	/* SIGNALING NAN */
	    signal = SIGFPE;
	    code   = FPE_FLTNAN_TRAP;
	    break;
	
	default:	/* not an exception we know about */
	    return (ERROR);
	}

    /* try to deliver the signal to the task */

    return (sigRaise (taskId, signal, code));
    }
/*******************************************************************************
*
* programError - determine if exception is program error
*
* RETURNS:
*   TRUE if exception indicates program error,
*   FALSE if hardware interrupt or failure.
*/

LOCAL BOOL programError (vecNum)
    int vecNum;		/* exception vector number */

    {
    return (((BUS_ERROR_VEC <= vecNum) && (vecNum <= FORMAT_ERROR)) ||
	    ((TRAP_0_VEC <= vecNum ) && (vecNum < USER_VEC_START)));
    }
