/* dgTest.c - datagram test sender/receiver demo */

/*
modification history
--------------------
01a,10jun88,gae  written based on rdc's test programs.
*/

/*
DESCRIPTION
This demonstration module shows how to send and receive UDP packets
from VxWorks to UNIX and any combination thereof.
Firstly, a receiver, or receivers, are initiated on VxWorks and/or
UNIX.  It will wait for packets on an agreed port number.
Then a sender, again on either UNIX or VxWorks, begins sending packets.
The receiver will terminate after an agreed number of packets are sent.

EXAMPLE
    -> sp dgReceiver
    -> dgSender "127.1"
    ... output from dgReceiver
    -> dgHelp

A few global variables are defined: verbose, terminate, and broadcast.
If verbose is true then extra diagnostics are printed.
If terminate is set to true while dgReceiver is running then it will
exit gracefully (after the first packet).
If broadcast is set to true then dgSender will broadcast the packets.
*/

#include "vxWorks.h"
#include "types.h"
#include "netdb.h"
#include "inetLib.h"
#include "in.h"
#include "socket.h"
#include "ioLib.h"

#define	TORF(x)	((x) ? "True" : "False")

#define PACKET_SIZE	100 	/* size of UDP packets */
#define PACKET_NUM	500	/* maximum number of packets */

IMPORT char sysBootHost[];	/* VxWorks boot hostname */

typedef struct	/* PACKET */
    {
    int sequence;
    int spare;
    char data [PACKET_SIZE - 2 * sizeof (int)];
    } PACKET;

BOOL broadcast       = FALSE;	/* when TRUE dgSender will broadcast */
BOOL verbose         = FALSE;	/* when TRUE extra diagnostics are displayed */
BOOL terminate       = FALSE;	/* when TRUE dgReceiver will quit */
BOOL is42            = FALSE;	/* when FALSE socket is set for broadcast */
int dgPort           = 1101;	/* default */
int dgPackets        = 5;	/* default */
int fionread         = FIONREAD;/* VxWorks value (different from UNIX) */

LOCAL char *dgSendHelp =
"show defaults\n\
-b    broadcast         = %s\n\
-h    host              = %s\n\
-p    port number       = %d\n\
-n    number of packets = %d\n\
-v    verbose           = %s\n";

LOCAL char *dgRecvHelp =
"show defaults\n\
-p    port number       = %d\n\
-n    number of packets = %d\n\
-v    verbose           = %s\n";

/*******************************************************************************
*
* dgSender - send UDP datagrams
*/

VOID dgSender (host, port, packets)
    char *host;		/* NULL = use default */
    int port;		/* 0 = use default */
    int packets;	/* 0 = use default */

    {
    struct in_addr destNet;
    u_long inetaddr;
    struct sockaddr_in sockaddr;
    int sockaddrLen = sizeof (sockaddr);
    int optval = 1;
    PACKET packet;
    int ix;
    int s;

    if (host == NULL)
	host = sysBootHost;

    if ((inetaddr = hostGetByName (host)) == ERROR &&
	(inetaddr = inet_addr (host)) == ERROR)
	{
	printf ("invalid host: <%s>\n", host);
	return;
	}

    if (broadcast)
	{
	destNet  = inet_makeaddr (inet_netof (inetaddr), INADDR_BROADCAST);
	inetaddr = destNet.s_addr;

	if (verbose)
	    printf ("broadcast ");
	}

    if (verbose)
	printf ("inet address = %#x\n", inetaddr);

    s = socket (AF_INET, SOCK_DGRAM, 0);	/* get a udp socket */

    if (s < 0)
	{
	printf ("socket error (errno = %#x)\n", errnoGet ());
	return;
	}

    if (!is42)
	{
	/* BSD 4.2 doesn't have or require turning on of broadcasting */

	if (setsockopt (s, SOL_SOCKET, SO_BROADCAST,
			(caddr_t) &optval, sizeof (optval)) < 0)
	    {
	    printf ("setsockopt error (errno = %#x)\n", errnoGet ());
	    return;
	    }
	}

    if (port == 0)
	port = dgPort;

    if (packets == 0)
	packets = dgPackets;

    bzero (&packet, sizeof (PACKET));

    bzero (&sockaddr, sizeof (sockaddr));
    sockaddr.sin_family      = AF_INET;
    sockaddr.sin_addr.s_addr = inetaddr;
    sockaddr.sin_port        = port;

    /* send the datagrams */

    for (ix = 0; ix < packets; ix++)
	{
	packet.sequence = ix;

	/* send it */

	if (sendto (s, &packet, sizeof (packet), 0, 
		    &sockaddr, sizeof (sockaddr)) == -1)
	    {
	    printf ("sendto error on packet # %d (errno = %#x)\n",
		    ix, errnoGet ());
	    close (s);
	    return;
	    }
	}

    printf ("sent %d packets\n", ix);

    close (s);
    }
/*******************************************************************************
*
* dgReceiver -
*/

VOID dgReceiver (port, packets)
    int port;
    int packets;

    {
    struct sockaddr_in sockaddr;
    int sockaddrLen = sizeof (sockaddr);
    int optval = 1;
    int nBytes;
    int ix;
    BOOL missing;
    char packetList [PACKET_NUM];	/* boolean array of received packets */
    PACKET packet;
    int nPacketsReceived = 0;
    int s = socket (AF_INET, SOCK_DGRAM, 0);

    if (s < 0)
	{
	printf ("socket error (errno = %#x)\n", errnoGet ());
	return;
	}

#ifdef	REUSE
    if (setsockopt (s, SOL_SOCKET, SO_REUSEADDR,
		     (caddr_t) &optval, sizeof (optval)) < 0)
	{
	printf ("setsockopt error (errno = %#x)\n", errnoGet ());
	return;
	}
#endif	REUSE

    if (port == 0)
	port = dgPort;

    if (packets == 0)
	packets = dgPackets;

    /* clear out the packetList */
    bzero (packetList, sizeof (packetList));

    bzero (&sockaddr, sizeof (sockaddr));

    sockaddr.sin_family      = AF_INET;
    sockaddr.sin_port        = port;
   
    if (bind (s, &sockaddr, sizeof (sockaddr)) == ERROR)
	{
	printf ("bind error (errno = %#x)\n", errnoGet ());
	return;
	}

    if (getsockname (s, &sockaddr, &sockaddrLen) == ERROR)
	{
	printf ("getsockname error (errno = %#x)\n", errnoGet ());
	return;
	}

    terminate = FALSE;

    while (packets > nPacketsReceived)
	{
#ifdef	POLL
	if (ioctl (s, fionread, &nBytes) == ERROR)
	    {
	    printf ("ioctl error (errno = %#x)\n", errnoGet ());
	    close (s);
	    return;
	    }

	if (nBytes > 0)
#endif	POLL
	    {
	    nBytes = recvfrom (s, &packet, sizeof (PACKET), 0,
				&sockaddr, &sockaddrLen);
	    if (verbose)
		printf ("received packet # %d\n", packet.sequence);
	    packetList [packet.sequence] = TRUE;
	    nPacketsReceived++;
	    }
	if (terminate)
	    {
	    printf ("terminated\n");
	    break;
	    }
	}

    missing = FALSE;
    nPacketsReceived = 0;
    for (ix = 0; ix < packets; ix++)
	{
	if (packetList[ix] == TRUE)
	    nPacketsReceived++;
	else
	    {
	    missing = TRUE;
	    printf ("%d\n", ix);
	    }
	}

    printf ("\n%smissing packets\n", missing ? "" : "no ");
    printf ("%d packets received\n", nPacketsReceived);

    close (s);
    }
/*******************************************************************************
*
* dgHelp -
*/

VOID dgHelp ()

    {
    printf ("dgHelp                             show this list\n");
    printf ("dgReceiver [port],[packets]        receive datagrams\n");
    printf ("dgSender [host],[port],[packets]   send datagrams\n");
    printf ("\n");
    printf ("broadcast                = %s\n", TORF(broadcast));
    printf ("host                     = %s\n", sysBootHost);
    printf ("port number              = %d\n", dgPort);
    printf ("number of packets        = %d\n", dgPackets);
    printf ("verbose                  = %s\n", TORF(verbose));
    printf ("\n");
    }
