/* sysLib.c - SUN 3/e system dependent library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01p,15apr89,jcf  fixed sysBusToLocal ().
           +gae   wrote sysDualPortMap () and sysDualPortUnmap ().
		  enable DVMA in sysHwInit ().
		  rewrote sysDumpMMU () to be real pretty.
	          fixed page number calculation for other Sun-3s.
01o,04apr89,jcf  fixed sysBusToLocal ().
01n,29oct88,gae  moved ieEnetAddr here from usrConfig.c.
		 fixed sysBusTas documentation and made it log error message.
01m,09sep88,gae  documentation.
01k,08aug88,gae  changed to use proper z8530 definitions.  moved sysUsart*
		   stuff to tyCoDrv.c.  Turned sysFlash into sysLEDSet.
01j,08jun88,gae  Revised code/documentation for uniformity across targets.
		 sysClkConnect now returns STATUS.
		 added sysClk{Disable,Enable}, sysClkRateSet() doesn't enable.
		 added sysAuxClk{Disable,Enable}.
		 fixed sysIntAck to return NULL.
		 added sysProcNumGet.
		 UNTESTED!
01i,30may88,dnw  changed to v4 names.
01h,27apr88,ecs  added sysExcMsg.
01i,30may88,dnw  changed to v4 names.
01h,27apr88,ecs  added sysExcMsg.
01g,07apr88,jcf  fixed erroneous comment.
01f,28mar88,jcf  fixed initMMU to work with old and new Sun roms.
		  DVMA is not given any address space.
		  sysMemTop is fixed at 3 MB.
01e,24nov87,jcf  documentation.
01d,20nov87,dnw  changed sysIntGen to take interrupt vec num instead of vec adrs
		 changed to use new defines in config.h
01c,19nov87,jcf  documentation.
01b,13nov87,ecs  documentation.
01a,29mar87,jcf  Written, by modifying v01j of isi 68020 version.
		 added sysClkConnect(), sysIntGen(), sysMailboxEnable(),
		   sysBusToLocalAdrs().
		 added more sys... global variables.
		 added sysToMonitor().
		 deleted sysMailBoxAdrs().
		 deleted sysGetProcNum().
		 changed sysIntAck() to only do software ack for levels 2,4,7.
		 changed sysLocalToBusAdrs() to take adrs space arg.
		 changed aux clock to log msg if int occurs while unconnected.
		 added sysMailboxConnect routine
*/

/*
DESCRIPTION
This library contains a set of routines to manipulate the primary functions
of the CPU board.  The goal is to provide a board-independant interface on
which VxWorks and application code can be built in a system-independant way.
Not every feature of every board is supported by this library; a particular
board may have various extensions to the capabilities described here.
Also not every board will support all the functions provided by this library.
And some boards provide some of the functions of this library with hardware
switches, jumpers, or PALs, instead of software controllable registers.

The funtions addressed here include:

    initialization functions:
	- initialize hardware to known state
	- identify the system

    memory/address space functions:
	- get on-board memory limit
	- map from local to bus and bus to local address spaces
	- enable/disable cache memory
	- set/get non-volatile RAM

    bus interrupt functions:
	- enable/disable bus interrupt levels
	- generate bus interrupts

    serial channel functions (see tyCoDrv):
	- enable/disable serial channel interrupts
	- set serial channel baud rates
	- get/put bytes from a serial channel

    clock/timer functions:
       - enable/disable timer interrupts
       - set timer periodic rate

    mailbox/location monitor functions:
       - enable mailbox/location monitor interrupts

SPECIAL CONSIDERATIONS
Because the SUN 3/e only has 64K of EPROM, SUN bootroms must be
used.  The trap vector is used to return to the bootroms.  Also,
we rely on the SUN bootroms set up MMU page tables.

*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "vme.h"
#include "memLib.h"
#include "sysLib.h"
#include "sun3e.h"
#include "config.h"

IMPORT VOID logMsg ();


/* globals */

int   sysBus      = BUS;		/* system bus type (VME_BUS, etc) */
int   sysCpu      = CPU;		/* system cpu type (MC680x0) */
char *sysBootLine = BOOT_LINE_ADRS;	/* address of boot line */
char *sysExcMsg   = EXC_MSG_ADRS;	/* catastrophic message area */
int   sysProcNum;			/* processor number of this cpu */
int   sysFlags;				/* boot flags */
char  sysBootHost[BOOT_FIELD_LEN];	/* name of host from which we booted */
char  sysBootFile[BOOT_FIELD_LEN];	/* name of file from which we booted */

char ieEnetAddr [6] = IE_ENET_ADDR;
/* {0x08, 0x00, 0x14, 0x20, 0x01, 0x90} */



/* forward declarations */

LOCAL VOID sysClkInt ();

/* locals */

LOCAL int sysClkTicksPerSecond = 100;	/* fixed */
LOCAL FUNCPTR sysClkRoutine    = NULL;
LOCAL int sysClkArg            = NULL;
LOCAL int auxClkTicksPerSecond = 60;
FUNCPTR sysAuxClkRoutine       = logMsg;/* not LOCAL, used in tyCoDrv */
int sysAuxClkArg               = (int) "auxiliary clock interrupt\n";
LOCAL BOOL sysClkIsConnected   = FALSE;
LOCAL BOOL sysClkRunning       = FALSE;
LOCAL BOOL auxClkRunning       = FALSE;


/*******************************************************************************
*
* sysModel - return model name of the system CPU
*
* Use this routine to find the model name of the system CPU.
*
* RETURNS: pointer to string "SUN-3/E"
*/

char *sysModel ()

    {
    return ("SUN-3/E");
    }
/*******************************************************************************
*
* sysHwInit - initialize hardware
*
* This routine initializes various features of the board.
* It is normally called from usrInit (2) in usrConfig (1).
* This routine must be called in supervisor mode.
*
* Setup up clock handler.
* Setup monitor handler.
* Setup Memory Managment Unit (MMU)
*/

VOID sysHwInit ()

    {
    int delay;

    /* initialize SUN's clock interrupt handler and sysToMon vector */

    intVecSet (IV_AUTOVEC_7, (FUNCPTR) SUN_CLK_HANDLER);
    intVecSet (IV_TRAP_3, (FUNCPTR) SUN_TO_MONITOR);

    sysInitMMU ();

    sysDualPortUnmap (0x0,0xfe000);	/* initially don't dual port memory */

    /* reset the serial controllers */

    *SUN_SCCA = SCC_WR0_SEL_WR9;	/* master interrupt reg */
    *SUN_SCCA = SCC_WR9_HDWR_RST;	/* reset SCC 1 */

    /* Enable floating point chip and the DVMA */

    sysCsWB (SUN_ENABLE << 28, (char) sysCsRB (SUN_ENABLE << 28) | 0x60);

#if	FALSE
 /* If we reset the the keyboard/ mouse uart, we will lose the sun bootrom */

   *SUN_KYBC = SCC_WR0_SEL_WR9;		 /* master interrupt reg */
   *SUN_KYBC = SCC_WR9_HDWR_RST;	 /* reset SCC 2 */
#endif	FALSE

    for (delay = 0; delay < 5000; delay++)	/* Wait for reset */
	;

    sysCsWB (SUN_LED << 28, 0xff);		/* LEDs all off */
    }
/*******************************************************************************
*
* sysInitMMU - initialize the memory management unit
*
* The memory management unit is initialized, in part, by the sun boot roms.
* This code adds to their map devices such as the excellan ethernet board.
* Eventually we may choose to get rid of the boot proms and take things into
* our own hands.  The arguments for this are that the target could be used
* with non-sun hosts but the arguments against are: a 64k prom, a horribly
* complicated MMU which must first be dealt with in boot state, no other hosts
* have the rarpd . . .
* The user is warned against fooling with the map unless they really understand
* the context register, boot state bit, the segment map, page map, accessing
* control space.  Be especially warned that a modification to the context
* register causes an immediate switch and can thus sweep the code out from
* under you.
*
* We tried 32 bit data accesses to standard VME space but the 3E did not perform
* long word accesses on odd short boundaries.  So we left it as 16 bit data
* accesses.
*
* The map as it stands after this routine is exited is:
* (changes to the original map are double starred)
*
*			 ----RAM MAP----
* Virtual Address Range   Size    Seg #     PMEG #   Page #     Device
* -----------------------------------------------------------------------
**00000000 - 003FFFFF     4M   000 - 01F   00 - 1F  0 - F   MAIN MEMORY
**00400000 - 007FFFFF     4M   020 - 03F   20 - 3F  0 - F   EXTENSION MEM
**00800000 - 00AFFFFF     3M   040 - 057   40 - 57  0 - F   EXTENSION MEM
*
*        	         ----VME MAP----
**00C00000 - 01BDFFFF 16M-128K 060 - 0DE   60 - DE  0 - F   24b Add. 16b Dat.
**01BE0000 - 01BEFFFF    64K   0DF         DF       0 - 7   24b Add. 16b Dat.
**01BF0000 - 01BFFFFF    64K   0DF         DF       8 - F   16b Add. 32b Dat.
**01C00000 - 01C3FFFF   256K   0E0 - 0E1   E0 - E1  0 - F   32b Add. 32b Dat.
*
*		    ----SUN BOOTROMS SET UP----
* 01C40000 - 01DFFFFF  1.75M   0E2 - 0EF   E2 - EF  0 - F   **BOOTROM USE
*							       THESE SEGS.
*							    (ctrl-x needs them)
*
* 01E00000 - 0FDFFFFF    BIG   0E0 - 7EF     FF     0 - F   **INVALID
*
* 0FE00000 - 0FE01FFF     8K      7F0        FE       0     KYBD/MOUSE
* 0FE02000 - 0FE03FFF     8K      7F0        FE       1     SIO A/B 
* 0FE04000 - 0FE05FFF     8K      7F0        FE       2     EEPROM
* 0FE06000 - 0FE07FFF     8K      7F0        FE       3     TOD CLOCK 
* 0FE08000 - 0FE09FFF     8K      7F0        FE       4     MEMERR REG
* 0FE0A000 - 0FE0BFFF     8K      7F0        FE       5     INT REG
* 0FE0C000 - 0FE17FFF    48K      7F0        FE     6 - B   **UNUSED
* 0FE18000 - 0FE1FFFF    32K      7F0        FE     C - F   **INVALID
*
* 0FE20000 - 0FE5FFFF   256K   7F1 - 7F2   FD - FC  0 - F   VIDEO FRAME BUFFER
*
* 0FE60000 - 0FE61FFF     8K      7F3        FB       0     MONITOR STK
* 0FE62000 - 0FE7FFFF   120K      7F3        FB     1 - F   **INVALID
*
* 0FE80000 - 0FEBFFFF   256K   7F4 - 7F5   FA - F9  0 - F   **INVALID
* 0FEC0000 - 0FEDFFFF   128K      7F6        FF     0 - F   **INVALID 
*
* 0FEE0000 - 0FEEFFFF    64K      7F7        F7     0 - 7   **INVALID 
* 0FEF0000 - 0FEFFFFF    64K      7F7        F7     8 - F   PROM
*
* 0FF00000 - 0FFFFFFF     1M    7F8 - 7FE  58 - 5F  0 - F   DUAL PORT (DVMA)
*
* 0FFFE000 - 0FFFFFFF     8K      7FF        F8       F     MON_SHORT_PG
*
* UNASSIGNED PMEGS:  NONE
*/

LOCAL VOID sysInitMMU ()

    {
    UINT segment;	/* virtual address segment */
    UINT page;		/* virtual address page */
    UINT physical;	/* physical address */
    UINT virtual;	/* virtual address */
    char pmeg;	 	/* page map entry group */

    /* validate extension memory */
    for (segment = 0x000; segment < 0x058; segment++)
	{
	pmeg = (char) segment;
	sysSegSet (segment, pmeg);
	for (page = 0x0; page < 0x10; page++)
	    {
	    physical = (segment << 17) + (page << 13);	/* one to one */
	    virtual = (segment << 17) + (page << 13);
	    sysPageSet (virtual, physical, SUN_MEM);
	    }
	}

    /* provide vme bus interface */

    /* 24bit address 16bit data */			/* XXX 32bit some day */
    for (segment = 0x060; segment < 0x0df; segment++)
	{
	pmeg = (char) segment;
	sysSegSet (segment, pmeg);
	for (page = 0x0; page < 0x10; page++)
	    {
	    physical = 0xff000000 + ((segment - 0x60) << 17) + (page << 13);
	    virtual = (segment << 17) + (page << 13);
	    sysPageSet (virtual, physical, SUN_VME16);	/* XXX 32bit some day */
	    }
	}

    segment = 0xdf;
    pmeg = (char) segment;
    sysSegSet (segment, pmeg);
    for (page = 0x0; page < 0x8; page++)
	{
	physical = 0xff000000 + ((segment - 0x60) << 17) + (page << 13);
	virtual = (segment << 17) + (page << 13);
	sysPageSet (virtual, physical, SUN_VME32);
	}

    /* 16bit address 16bit data */
    for (page = 0x8; page < 0x10; page++)
	{
	physical = 0xffff0000 + ((page - 8) << 13);
	virtual = (segment << 17) + (page << 13);
	sysPageSet (virtual, physical, SUN_VME16);
	}


    /* 32bit address 32bit data */
    for (segment = 0x0e0; segment < 0x0e2; segment++)
	{
	pmeg = (char) segment;
	sysSegSet (segment, pmeg);
	for (page = 0x0; page < 0x10; page++)
	    {
	    physical = 0x00000000 + ((segment - 0xe0) << 17) + (page << 13);
	    virtual = (segment << 17) + (page << 13);
	    sysPageSet (virtual, physical, SUN_VME32);
	    }
	}

    /* allocate the pmegs to the dual port memory see sysDualPort () */

    for (segment = 0x7f8; segment < 0x7ff; segment++)
	sysSegSet (segment, (char) (segment - 0x7a0));

    }
/****************************************************************************** *
* sysDualPortMap - dual port local ram
*
* This routine dual ports the a piece of local memory starting at the specified
* address.  The size of the piece is specified in bytes by the size parameter.
* The size is rounded up to the nearest page boundary.
* The DVMA responds to the lowest megabyte of the VME extended (32 bit address)
* and the the VME standard (24 bit address) spaces.  The parameter vmeAdrs
* specifies the vmeAdrs that will translate to the specified local address.
*
* NOTE
* If one dual ports two local addresses that translate to the same vme
* address.  The translation for the most recent sysDualPortMap () is used.
* 
* CAVEATS
* The Sun 3/e can only dual port 1MB of its memory, so a size greater than
* this is not allowed.
* 
* The last page (8k) of the vmeAddress megabyte is reserved by sun,
* (MON_SHORT_PG) and should not be changed.  So don't specify a region that
* will occupy the vme address 0xfe000 thru 0x100000.
*/  

STATUS sysDualPortMap (localAddress, vmeAddress, size)
    UINT localAddress;
    UINT vmeAddress;
    UINT size;

    {
    UINT phys;					/* physical address */

    if (vmeAddress + size > 0xfe000)		/* last 8k reserved */
	return (ERROR);

    for (phys = localAddress; phys < localAddress + size; phys += 0x2000)
	sysPageSet (0xff00000 + vmeAddress + phys - localAddress, phys,
		    SUN_MEM);

    return (OK);
    }
/******************************************************************************
*
* sysDualPortUnmap - unmap dual porting of local ram
*
* This routine unmaps the dual porting of memory from the specified 
* VME address.  The size of the piece is specified in bytes by the
* size parameter.  The size is rounded up to the nearest page boundary.
* 
* The last 8k of the 1MB the DVMA responds to is used by the sun bootroms
* (MON_SHORT_PG) and should not be changed.
*/  

STATUS sysDualPortUnmap (vmeAddress, size)
    UINT vmeAddress;
    UINT size;

    {
    UINT vme;						/* vme address */

    if (vmeAddress + size > 0xfe000)			/* last 8k reserved */
	return (ERROR);

    /* make pages invalid */

    for (vme = vmeAddress; vme < vmeAddress + size; vme += 0x2000)
	sysCsWL (SUN_PAGE_BASE + 0xff00000 + vme, 0);

    return (OK);
    }

/*******************************************************************************
*
* sysMemTop - get top of memory address
*
* This routine returns the address of the first missing byte of memory.
*
* NOTE SUN3E:
* This board has 4M, however the fourth megabyte is used by SUN's bootroms.
* If control-x and reboot are to function correctly this should not be
* altered.  However, if more memory is required, change this routine to
* return 4 Mbytes instead.
*
* RETURNS: address of the first missing byte of memory
*/

char *sysMemTop ()

    {
    return ((char *) 0x300000);
    }

/*******************************************************************************
*
* sysClkConnect - connect routine to system clock interrupt
*
* This routine connects the given function to the system clock interrupt.
* Auxiliary clock interrupts are not enabled.
* It is normally called from usrRoot (2) in usrConfig (1) to connect
* usrClock (2) to the system clock interrupt.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), usrClock (2)
*/

STATUS sysClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each system clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    if (!sysClkIsConnected &&
	intConnect (INUM_TO_IVEC (INT_VEC_CLOCK), sysClkInt, 0) == ERROR)
	{
	return (ERROR);
	}

    sysClkIsConnected = TRUE;

    sysClkRoutine     = routine;
    sysClkArg         = arg;

    return (OK);
    }
/*******************************************************************************
*
* sysClkDisable - turn off system clock interrupts
*/

VOID sysClkDisable ()

    {
    if (sysClkRunning)
	{
	*SUN_IR	&= 0xfe;		/* turn off interrupts */
	*SUN_IR 	&= 0x5f;		/* turn off lvl 7, and 5 */
	*SUN_RTC_CMD = RTC_CMD_DIS_INT;	/* interrupt off */
	*SUN_RTC_CMD = RTC_CMD_ENA_INT;	/* enable interrupting */
	*SUN_IR 	|= 0x01;		/* turn on interrupts */

	sysClkRunning = FALSE;
	}
    }
/*******************************************************************************
*
* sysClkEnable - turn system clock interrupts on
*/

VOID sysClkEnable ()

    {
    if (!sysClkRunning)
	{
	/* clock fixed at 100 Hz.
	 * preserve other interrupt fields in the interrupt reg.
	 */

	*SUN_IR	&= 0xfe;		/* turn off interrupts */
	*SUN_IR 	&= 0x5f;		/* turn off lvl 7, and 5 */
	*SUN_RTC_CMD = RTC_CMD_DIS_INT;	/* interrupt off */
	*SUN_IR 	|= IR_CLOCK5;		/* turn on lvl 5 */
	*SUN_RTC_CMD = RTC_CMD_ENA_INT;	/* enable interrupting */
	*SUN_IR 	|= 0x01;		/* turn on interrupts */

	sysClkRunning = TRUE;
	}
    }
/*******************************************************************************
*
* sysClkInt - clock interrupt handler
*
* This routine handles the clock interrupt.  It is attached to the clock
* interrupt vector by the routine sysClkConnect (2).
* The appropriate routine is called and the interrupts are acknowleged.
*/

LOCAL VOID sysClkInt ()

    {
    if (sysClkRoutine != NULL)
	(*(FUNCPTR) sysClkRoutine) (sysClkArg);

    sysClkAck ();
    }
/*******************************************************************************
*
* sysClkRateGet - get rate of system clock
*
* This routine is used to find out the system clock speed.
*
* RETURNS: number of ticks per second of the system clock
*
* SEE ALSO: sysClkRateSet (2)
*/

int sysClkRateGet ()
    
    {
    return (sysClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysClkRateSet - set rate of system clock
*
* This routine sets the clock rate of the system clock.
* It is normally called by usrRoot (2) in usrConfig (1).
*
* NOTE SUN3E:
* The clock rate is not adjustable.
* This routine has no effect.
*
* SEE ALSO: sysClkRateGet (2)
*
* ARGSUSED
*/

VOID sysClkRateSet (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    }

/*******************************************************************************
*
* sysAuxClkConnect - connect a routine to the auxiliary clock interrupt
*
* This routine connects a user routine to the auxiliary clock interrupt.
* Auxiliary clock interrupts are not enabled.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), sysAuxClkDisconnect (2)
*/

STATUS sysAuxClkConnect (routine, parm)
    FUNCPTR routine;	/* routine called at each auxiliary clock interrupt */
    int parm;		/* argument with which to call routine */

    {
    sysAuxClkInit ();		/* init the channel */

    sysAuxClkRoutine = routine;
    sysAuxClkArg     = parm;

    return (OK);
    }
/*******************************************************************************
*
* sysAuxClkDisconnect - clear the auxiliary clock routine.
* 
* This routine disables the auxiliary clock interrupt, stops the timer,
* and disconnects the routine currently connected to the auxiliary clock
* interrupt.
*
* SEE ALSO: sysAuxClkConnect (2)
*/

VOID sysAuxClkDisconnect ()

    {
    /* disable the auxiliary clock interrupt */

    sysAuxClkDisable ();

    /* connect dummy routine, just in case */

    sysAuxClkConnect (logMsg, (int) "auxiliary clock interrupt\n");
    }
/*******************************************************************************
*
* sysAuxClkDisable - turn off auxiliary clock interrupts
*/

VOID sysAuxClkDisable ()

    {
    char zero = 0;

    if (auxClkRunning)
	{
	*SUN_MSC = SCC_WR0_SEL_WR1;
	*SUN_MSC = zero;			/* disable the works */

	auxClkRunning = FALSE;
	}
    }
/*******************************************************************************
*
* sysAuxClkEnable - turn auxiliary clock interrupts on
*/

VOID sysAuxClkEnable ()

    {
#define	HZ	4915200
    int tc;

    if (!auxClkRunning)
	{
	tc = HZ / (2 * auxClkTicksPerSecond) - 2;

	if (tc < 1 || tc > 65535)	/* 16 bits */
	    {
	    printErr ("sysAuxClkEnable: %d out of range\n",
			auxClkTicksPerSecond);
	    return;
	    }

	*SUN_MSC = SCC_WR0_SEL_WR12;
	*SUN_MSC = LSB(tc);
	*SUN_MSC = SCC_WR0_SEL_WR13;
	*SUN_MSC = MSB(tc);

	*SUN_MSC = SCC_WR0_SEL_WR14;
	*SUN_MSC = SCC_WR14_BR_SRC | SCC_WR14_BR_EN;
	*SUN_MSC = SCC_WR0_SEL_WR1;
	*SUN_MSC = SCC_WR1_EXT_INT_EN;		/* turn on interrupts */

	auxClkRunning = TRUE;
	}
    }
/*******************************************************************************
*
* sysAuxClkRateGet - get the auxiliary timer frequency
*
* This routine finds out the auxiliary clock speed.
*
* RETURNS: number of ticks per second of the auxiliary clock
*
* SEE ALSO: sysAuxClkRateSet (2)
*/

int sysAuxClkRateGet ()

    {
    return (auxClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysAuxClkRateSet - set the auxiliary timer frequency
*
* This routine sets the clock rate of the auxiliary clock.
* Auxiliary clock interrupts are not enabled.
*
* SEE ALSO: sysAuxClkRateGet (2)
*/

VOID sysAuxClkRateSet (ticksPerSecond)
    int ticksPerSecond;	    /* clock interrupts per second, 0 stops clock */

    {
    if (ticksPerSecond > 0)
	auxClkTicksPerSecond = ticksPerSecond;

    if (auxClkRunning)
	{
	sysAuxClkDisable ();
	sysAuxClkEnable ();
	}
    }
/*******************************************************************************
*
* sysAuxClkInit - initialize the auxiliary clock channel (the mouse channel)
*
* Initialize the SCC as per the Zilog initialization sequence.
* This routine must be called in supervisor mode.
*/

LOCAL VOID sysAuxClkInit ()

    {
    int oldlevel;		/* current interrupt level mask */
    int delay;			/* delay for reset */
    char *cp;			/* control port */
    char zero = 0;

    oldlevel = intLock ();	/* disable interrupts during init */
    cp = SUN_MSC;
    *cp = SCC_WR0_SEL_WR9;			/* master interrupt reg */
    *cp = SCC_WR9_CH_B_RST;			/* reset channel B */

    for (delay = 0; delay < 5000; delay++);	/* Wait for reset */

    *cp = SCC_WR0_SEL_WR4;
    *cp = SCC_WR4_1_STOP;		/* 1x clock */

    *cp = SCC_WR0_SEL_WR3;
    *cp = SCC_WR3_RX_8_BITS;		/* 8 bit word */

    *cp = SCC_WR0_SEL_WR5;
    *cp = SCC_WR5_TX_8_BITS;		/* 8 bit word */

    *cp = SCC_WR0_SEL_WR6;		/* clear sync regs. */
    *cp = zero;

    *cp = SCC_WR0_SEL_WR7;
    *cp = zero;

    *cp = SCC_WR0_SEL_WR9;
    *cp = SCC_WR9_MIE;			/* All interrupts, no vector */

    *cp = SCC_WR0_SEL_WR10;
    *cp = zero;				/* clear sync, loop, poll */

    *cp = SCC_WR0_SEL_WR11;
    *cp = SCC_WR11_RX_BR_GEN | SCC_WR11_TX_BR_GEN | SCC_WR11_TRXC_OI |
          SCC_WR11_OUT_BR_GEN;

    *cp = SCC_WR0_SEL_WR12;		/* LSB of baud constant. */
    *cp = zero;
    *cp = SCC_WR0_SEL_WR13;		/* MSB of baud constant. */
    *cp = 0xff;

    *cp = SCC_WR0_SEL_WR14;		/* misc control bits */
    *cp = SCC_WR14_BR_SRC;

    /* enable the extern. channel */

    *cp = SCC_WR0_RST_TX_CRC;		/* reset tx crc */

    *cp = SCC_WR0_SEL_WR14;		/* misc control bits */
    *cp = SCC_WR14_BR_SRC | SCC_WR14_BR_EN;

    *cp = SCC_WR0_SEL_WR15;		/* external/status interrupt control */
    *cp = SCC_WR15_ZERO_CNT;		/* Zero count ext int */

    *cp = SCC_WR0_RST_INT;		/* reset external interrupts */
    *cp = SCC_WR0_RST_INT;		/* twice */

    /* all interrupts are masked out: enable external interrupt later */

    intUnlock (oldlevel);
    }

/*******************************************************************************
*
* sysLocalToBusAdrs - convert local address to bus address
*
* Given a local memory address, this routine returns the VME address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unable to get to that local address from the bus
*
* SEE ALSO: sysBusToLocalAdrs (2)
*/

STATUS sysLocalToBusAdrs (adrsSpace, localAdrs, pBusAdrs)
    int adrsSpace;      /* bus address space in which busAdrs resides,     */
                        /* use address modifier codes as defined in vme.h, */
                        /* such as VME_AM_STD_SUP_DATA                     */
    char *localAdrs;    /* local address to convert                        */
    char **pBusAdrs;    /* where to return bus address                     */

    {
    if ((localAdrs < LOCAL_MEM_LOCAL_ADRS) || (localAdrs >= sysMemTop ()))
	{
	/* this is off-board memory - just return local address */
	*pBusAdrs = localAdrs;
	}
    else
	{
	/* this is on-board memory - map to bus address space;
	 *   the following memory mapping is established in sysProcNumSet ():
	 *   - only processor 0 has memory on bus,
	 *   - the memory is placed in STD space at address LOCAL_MEM_BUS_ADRS.
	 */

	if ((adrsSpace != VME_AM_STD_SUP_PGM) &&
	     (adrsSpace != VME_AM_STD_SUP_DATA) &&
	     (adrsSpace != VME_AM_STD_USR_PGM) &&
	     (adrsSpace != VME_AM_STD_USR_DATA))
	    return (ERROR);

	*pBusAdrs = localAdrs + LOCAL_MEM_BUS_ADRS - LOCAL_MEM_LOCAL_ADRS;
	}

    return (OK);
    }
/*******************************************************************************
*
* sysBusToLocalAdrs - convert bus address to local address
*
* Given a VME memory address, this routine returns the local address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unknown address space
*
* SEE ALSO: sysLocalToBusAdrs (2)
*/

STATUS sysBusToLocalAdrs (adrsSpace, busAdrs, pLocalAdrs)
    int adrsSpace;	/* bus address space in which busAdrs resides,     */
                        /* use address modifier codes as defined in vme.h, */
                        /* such as VME_AM_STD_SUP_DATA                     */
    char *busAdrs;	/* bus address to convert                          */
    char **pLocalAdrs;	/* where to return local address                   */

    {
    switch (adrsSpace)
	{
	case VME_AM_SUP_SHORT_IO:
	case VME_AM_USR_SHORT_IO:
	    *pLocalAdrs = (char *) (0x01bf0000 + (int) busAdrs);
	    return (OK);

	case VME_AM_STD_SUP_ASCENDING:
	case VME_AM_STD_SUP_PGM:
	case VME_AM_STD_SUP_DATA:
	case VME_AM_STD_USR_ASCENDING:
	case VME_AM_STD_USR_PGM:
	case VME_AM_STD_USR_DATA:
	    *pLocalAdrs = (char *) (0x00c00000 + (int) busAdrs);
	    return (OK);

	case VME_AM_EXT_SUP_ASCENDING:
	case VME_AM_EXT_SUP_PGM:
	case VME_AM_EXT_SUP_DATA:
	case VME_AM_EXT_USR_ASCENDING:
	case VME_AM_EXT_USR_PGM:
	case VME_AM_EXT_USR_DATA:
	    *pLocalAdrs = (char *) (0x01c00000 + (int) busAdrs);
	    return (OK);

	default:
	    return (ERROR);
	}
    }
/*******************************************************************************
*
* sysIntEnable - enable interrupt level
*
* This routine enables the specified VME interrupt level.
*
* NOTE SUN3E:
* VME interrupts are not software controllable.
* This routine has no effect.
*
* RETURNS: OK
*
* ARGSUSED
*/

STATUS sysIntEnable (intLevel)
    int intLevel;	/* interrupt level to enable */

    {
    return (OK);
    }
/*******************************************************************************
*
* sysIntDisable - disable interrupt level
*
* This routine disables the specified interrupt level.
*
* NOTE SUN3E:
* VME interrupts are not software controllable.
* This routine has no effect.
*
* RETURNS: OK
*
* ARGSUSED
*/

STATUS sysIntDisable (intLevel)
    int intLevel;	/* interrupt level to disable */

    {
    return (OK);
    }
/*******************************************************************************
*
* sysBusIntAck - acknowledge interrupt
*
* Acknowledges the specified interrupt.
*
* NOTE SUN3E:
* This board automatially acknowledges device.
* This routine has no effect.
* On some boards this routine returns the vector put on
* bus by interrupting device.
*
* RETURNS: NULL
*
* ARGSUSED
*/

int sysBusIntAck (intLevel)
    int intLevel;	/* interrupt level to acknowledge */

    {
    return (NULL);
    }
/*******************************************************************************
*
* sysBusIntGen - generate interrupt
*
* This routine generates a VME bus interrupt.
*
* NOTE SUN3E:
* This routine has no effect.
*
* RETURNS: ERROR, as the SUN3E is incapable of generating a VME bus interrupt
*
* ARGSUSED
*/

STATUS sysBusIntGen (level, vector)
    int level;		/* VME bus interrupt level to generate (1-7) */
    int vector;		/* interrupt vector to generate (0-255) */

    {
    return (ERROR);
    }

/*******************************************************************************
*
* sysMailboxConnect - connect routine to the mailbox interrupt
*
* This routine connects the given function to the mailbox interrupt.
*
* RETURNS: ERROR since there is no mailbox facility
*
* SEE ALSO: intConnect (2)
*
* ARGSUSED
*/

STATUS sysMailboxConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each mailbox interrupt */
    int arg;		/* argument with which to call routine */

    {
    return (ERROR);
    }
/*******************************************************************************
*
* sysMailboxEnable - enable mailbox interrupt
*
* This routine enables the mailbox interrupt.
*
* RETURNS: ERROR, since there is no mailbox facility
*
* ARGSUSED
*/

STATUS sysMailboxEnable (mailboxAdrs)
    char *mailboxAdrs;		/* mailbox address */

    {
    return (ERROR);
    }
/*******************************************************************************
*
* sysProcNumGet - get processor number
*
* This routine returns the processor number previously set with 
* sysProcNumSet (2).
*
* RETURNS: processor number
*
* SEE ALSO: sysProcNumSet (2)
*/

int sysProcNumGet ()

    {
    return (sysProcNum);
    }
/*******************************************************************************
*
* sysProcNumSet - set processor number
*
* Set the processor number for this CPU.  Processor numbers should be
* unique on a single backplane.
*
* SEE ALSO: sysProcNumGet (2)
*/

VOID sysProcNumSet (procNum)
    int procNum;		/* processor number */

    {
    sysProcNum = procNum;
    }
/*******************************************************************************
*
* sysBusTas - test and set across vme bus
*
* This routine does a 680x0 test-and-set instruction across the backplane.
*
* NOTE
* The SUN 3/E hardware cannot do a test-and-set.
* A warning message is printed, via logMsg (2), the first time this
* routine is called, and then every 100 times.
*
* RETURNS: FALSE (failure)
*/

BOOL sysBusTas (addr)
    char *addr;		/* address to be tested and set */

    {
    static BOOL warning;

    if (warning++ % 100 == 0)
	logMsg ("sysBusTas: cannot do test-and-set\n");

    return (FALSE);
    }

/* miscellaneous support routines */

/*******************************************************************************
*
* sysSegSet - change the MMU
*
* This changes the segment map to put out the specified PMEG (page map entry
* group) when given the specified virtual address. The PMEG 0xff is reserved
* for invalid virtual addresses.  In other words, a virtual address must
* produce some PMEG even if it is invalid and that PMEG is 0xff.
*/

VOID sysSegSet (segment, pmeg)
    UINT segment;	/* segment you want to change */
    char pmeg;		/* PMEG to be output */

    {
    sysCsWB (SUN_SEG_BASE + (segment << 17), pmeg);
    }
/*******************************************************************************
*
* sysSegGet - get the segment entry (pmeg) for a particular virtual address
*
* This returns the entry in the segment table corresponding to the virtual
* address given.
*/

UINT sysSegGet (virtual)
    UINT virtual;	/* virtual address you want the segment entry for */

    {
    return (sysCsRB (SUN_SEG_BASE + virtual));
    }
/*******************************************************************************
*
* sysPageSet - change the mmu map
*
* This changes the map so that a virtual address will map to the physical
* address specified.  The change is only made for the context in which it is
* called which should be 0.  THIS ROUTINE MUST BE CALLED AFTER sysSegSet BECAUSE
* the VIRTUAL ADDRESS UNDERGOES THE SEGMENT TABLE TRANSLATION BEFORE ACCESSING
* the PAGE TABLE.  If you haven't got a valid PMEG then you are in trouble.
* The type bits are necessary because they are sent out of the page table entry
* to inform the CPU of what type of space we are accessing.
* Type 0 = Main Memory, Type 1 = I/O space, Type 2 = 16 bit data VME bus
* access, type 3 = 32 bit data VME bus access. (Note that the 68020 uses
* dynamic bus sizing capability to determine the bus width for a particular
* access.  If one had a 16 bit device, however, 32 bit accesses would cause
* trouble.  Sun uses a special space that breaks 32 bit accesses into two 16 bit
* accesses.  This space is not currently given any area in the virtual map.
* There are no protection rings supported here, for now; currently all entries
* are set up so task and supervisor level can read and write the space.
*
* ERROR CHECKING
* All that is done is a check to see if one is accessing a
* virtual address which the segment translation produces a pmeg 0xff
* which is reserved for invalid virtual address accesses.
*/

VOID sysPageSet (virtual, physical, type)
    UINT virtual;	/* virtual address to be translated */
    UINT physical;	/* physical address to be translated into */
    char type;		/* see h/sun3e.h for space types */

    {
    if (sysSegGet (virtual) != 0xff)
	sysCsWL (SUN_PAGE_BASE + virtual, MMU_PAGE_ALL_OK |
					  ((type & 3) << 26) |
					  ((physical >> 13) & 0x7ffff));
    }
/*******************************************************************************
*
* sysPageGet - get the page entry for a particular virtual address
*
* This returns the entry in the page table corresponding to the virtual address
* given.
*/

UINT sysPageGet (virtual)
    UINT virtual;	/* virtual address you want the page table entry for */

    {
    return (sysCsRL (SUN_PAGE_BASE + virtual));
    }
/*******************************************************************************
*
* sysDumpMMU - dump the tables to stdOut
*
* This routine dumps the memory management unit is a slightly more readable
* format.
* NOMANUAL
*/

VOID sysDumpMMU (virtual)
    UINT virtual;
    
    {
    printf ("%8s %3s %8s %8s  %1s %1s %1s %1s %2s %1s %1s\n",
	    "virtual", "seg", "page", "physical",
	    "v", "w", "s", "x", "ty", "a", "m");

    for (; virtual < (UINT) 0x10000000; virtual += 0x2000)
        if (sysPageGet (virtual) != 0x20000000)
            {
	    unsigned int pg = sysPageGet (virtual);
	    unsigned int p = (pg >> 24) & 0xff;
	    unsigned int v = (p & 0x80) >> 7;
	    unsigned int w = (p & 0x40) >> 6;
	    unsigned int s = (p & 0x20) >> 5;
	    unsigned int x = (p & 0x10) >> 4;
	    unsigned int t = (p & 0x0c) >> 2;
				    /* 00 main, 01 i/o, 10 vme16, 11 vme32 */
	    unsigned int a = (p & 0x02) >> 1;
	    unsigned int m = (p & 0x01);

	    printf ("%8x %3x %8x %8x  %1d %1d %1d %1d %01d%01d %1d %1d\n",
		    virtual, (unsigned char)sysSegGet (virtual),
		    pg,
		   (virtual & 0x1fff)+((pg & 0x7ffff) << 13),
		   v, w, s, x, (t & 0x2) >> 1, (t & 0x1), a, m);
            }
    }
/*******************************************************************************
*
* sysLEDSet - set LEDs
*
* This routine sets and re-sets the CPU board LEDs.
* Useful for interrupt level debugging.
*
* NOMANUAL
*/

VOID sysLEDSet (led)
    int led;		/* led number from 0 - 7 */

    {
    static char orig = 0xff;	/* all off in sysHwInit */

    orig ^= (1 << led);		/* toggle value */

    sysCsWB (SUN_LED << 28, orig);
    }
