/* tyCoDrv.c - The MVME147 on-board Z8530 SCC tty handler */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01b,09aug87,gae  clean up.
01a,26apr88,dfm  written, by modifying 01g of the iv24 version.
*/

/*
DESCRIPTION
This is the driver for the Motorola MVME-147 on-board serial ports.
It uses the SCC's only in asynch mode.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, tyCoDrv () to
initialize the driver, and tyCoDevCreate (...) to create devices.

TYCODRV
Before using the driver, it must be initialized by calling the routine:

    tyCoDrv ()

This routine should be called exactly once, before any reads, writes, or
tyCoDevCreates.  Normally, it is called from usrRoot.

CREATING TERMINAL DEVICES
Before a terminal can be used, it must be created.  This is done
with the tyCoDevCreate call.

.CS

    STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
	char *name;	/* Name to use for this device *
	int channel;	/* Physical channel for this device (0..3) *
	int rdBufSize;	/* Read buffer size, in bytes *
	int wrtBufSize;	/* Write buffer size, in bytes *

.CE

For instance, to create the device "/tyCo/0", with buffer sizes of 512 bytes,
the proper call would be:

.CS
   tyCoDevCreate ("/tyCo/0", 0, 512, 512);

.CE

Each port to be used should have exactly one device associated with it
by calling this routine.
There are four channels on the MVME-147 board, channels 0..3.

IOCTL
This driver responds to all the same ioctl codes as a normal ty driver.
Any baud rate between 0 and 38.4K baud is available.

*/

#include "vxWorks.h"
#include "iv68k.h"
#include "ioLib.h"
#include "iosLib.h"
#include "tyLib.h"
#include "config.h"


#define DEFAULT_BAUD	9600

typedef struct			/* TY_CO_DEV */
    {
    TY_DEV tyDev;
    BOOL created;	/* true if this device has really been created */
    char *cr;		/* control register I/O address */
    char *dr;		/* data port I/O address */
    char intVec;	/* interrupt vector address */
    } TY_CO_DEV;

LOCAL TY_CO_DEV tyCoDv [] =		/* device descriptors */
    {
    {{{{NULL}}}, FALSE, SERIAL_Z8530_A+2, SERIAL_Z8530_A+3, INT_VEC_Z8530_A},
    {{{{NULL}}}, FALSE, SERIAL_Z8530_A+0, SERIAL_Z8530_A+1, INT_VEC_Z8530_A},
    {{{{NULL}}}, FALSE, SERIAL_Z8530_B+2, SERIAL_Z8530_B+3, INT_VEC_Z8530_B},
    {{{{NULL}}}, FALSE, SERIAL_Z8530_B+0, SERIAL_Z8530_B+1, INT_VEC_Z8530_B},
    };

LOCAL int tyCoDrvNum;		/* driver number assigned to this driver */

/* forward declarations */

LOCAL VOID tyCoStartup ();
LOCAL int tyCoOpen ();
LOCAL int tyCoRead ();
LOCAL int tyCoWrite ();
LOCAL STATUS tyCoIoctl ();
LOCAL VOID tyCoInt ();


/*******************************************************************************
*
* tyCoDrv - ty driver initialization routine
*
* This routine initializes the serial driver, sets up interrupt vectors,
* and performs hardware initialization of the serial ports.
*/

STATUS tyCoDrv ()

    {
    /* check if driver already installed */
       
    if (tyCoDrvNum > 0)                     
	return (OK);


    /* the MVME-147 board uses one interrupt level, shared between the
     * two SCC chips. In order to eliminate confusion, we just let
     * the SCC chips generate their own vectors. Therefore, we need to
     * connect two vectors, one for each SCC chip.
     */

    (void) intConnect (INUM_TO_IVEC (INT_VEC_Z8530_A), tyCoInt, 0);
    (void) intConnect (INUM_TO_IVEC (INT_VEC_Z8530_B), tyCoInt, 1);

    tyCoHrdInit ();

    tyCoDrvNum = iosDrvInstall (tyCoOpen, (FUNCPTR) NULL, tyCoOpen,
				(FUNCPTR) NULL, tyCoRead, tyCoWrite, tyCoIoctl);

    return (tyCoDrvNum == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* tyCoDevCreate - create a device for the onboard ports
*
* This routine creates a device on one of the serial ports.  Each port
* to be used should have exactly one device associated with it, by calling
* this routine.
*/

STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
    char     *name;		/* Name to use for this device */
    FAST int  channel;		/* Physical channel for this device (0 or 4) */
    int	      rdBufSize;	/* Read buffer size, in bytes */
    int       wrtBufSize;	/* Write buffer size, in bytes */

    {
    FAST TY_CO_DEV *pTyCoDv;

    if (tyCoDrvNum <= 0)
	{
	errnoSet (S_ioLib_NO_DRIVER);
	return (ERROR);
	}

    /* if this doesn't represent a valid channel, don't do it */

    if (channel >= NELEMENTS (tyCoDv))
	return (ERROR);

    pTyCoDv = &tyCoDv [channel];


    /* if there is a device already on this channel, don't do it */

    if (pTyCoDv->created)
	return (ERROR);


    /* initialize the ty descriptor */

    if (tyDevInit (&pTyCoDv->tyDev, rdBufSize, wrtBufSize, tyCoStartup) != OK)
	return (ERROR);


    /* initialize the channel hardware */

    tyCoInitChannel (channel);


    /* Mark the device as having been created, and add the device to
     * the I/O system */

    pTyCoDv->created = TRUE;

    return (iosDevAdd (&pTyCoDv->tyDev.devHdr, name, tyCoDrvNum));
    }
/*******************************************************************************
*
* tyCoHrdInit - initialize the usart.
*
* This routine initializes the on-board usarts for the VxWorks environment. 
* This routine must be called in supervisor mode.
*/

LOCAL VOID tyCoHrdInit ()

    {
    FAST int   oldlevel;	/* current interrupt level mask */

    oldlevel = intLock ();		/* disable interrupts during init */

    tyCoResetChannel (0);	/* two ports per channel */
    tyCoResetChannel (2);	/* two ports per channel */

    /* enable the interrupts for the serial chips, on the PCC chip. */

    *SERIAL_INT_CTL = INT_LVL_TY | 0x08; /* enabled, vectors from chip */

    intUnlock (oldlevel);
    } 
/********************************************************************************
* tyCoResetChannel - reset a single channel
*/

LOCAL VOID tyCoResetChannel (channel)
    int channel;

    {
    FAST char *cr = tyCoDv [channel].cr;        /* SCC control reg adr */
    int delay;

    *cr = SCC_WR0_SEL_WR9;      /* write register 9 - master int ctrl */
 
    *cr = ((channel % 2) == 0) ?
        SCC_WR9_CH_A_RST :      /* reset channel A */
        SCC_WR9_CH_B_RST;       /* reset channel B */
 
    for (delay = 0; delay < 1000; delay++);     /* spin wheels for a moment */
    } 
/*******************************************************************************
*
* tyCoInitChannel - initialize a single channel
*/

LOCAL VOID tyCoInitChannel (channel)
    int channel;

    {
    FAST TY_CO_DEV *pTyCoDv = &tyCoDv [channel];
    FAST char	   *cr = pTyCoDv->cr;	/* SCC control reg adr */
    FAST int   	    baudConstant;
    FAST int        oldlevel;		/* current interrupt level mask */

    oldlevel = intLock ();		/* disable interrupts during init */

    /* initialize registers */

    *cr = SCC_WR0_SEL_WR4;              /* write reg 4 - misc parms and modes */    *cr = SCC_WR4_1_STOP | SCC_WR4_16_CLOCK;

    *cr = SCC_WR0_SEL_WR1;		/* int and xfer mode */
    *cr = SCC_WR1_INT_ALL_RX | SCC_WR1_TX_INT_EN;

    *cr = SCC_WR0_SEL_WR2;		/* interrupt vector */
    *cr = pTyCoDv->intVec;

    *cr = SCC_WR0_SEL_WR3;		/* write reg 3 - receive parms */
    *cr = SCC_WR3_RX_8_BITS | SCC_WR3_RX_EN;

    *cr = SCC_WR0_SEL_WR5;		/* tx parms */
    *cr = SCC_WR5_TX_8_BITS | SCC_WR5_TX_EN | SCC_WR5_RTS | SCC_WR5_DTR;

    *cr = SCC_WR0_SEL_WR10;		/* misc tx/rx control */
    *cr = 0;				/* clear sync, loop, poll */

    *cr = SCC_WR0_SEL_WR11;		/* clock mode */
    *cr = SCC_WR11_RX_BR_GEN | SCC_WR11_TX_BR_GEN | SCC_WR11_OUT_BR_GEN;

    *cr = SCC_WR0_SEL_WR14;		/* misc control bits */
    *cr = SCC_WR14_BR_SRC | SCC_WR14_BR_EN | SCC_WR14_SRC_BR;

    *cr = SCC_WR0_SEL_WR15;		/* external/status interrupt cntrl */
    *cr = 0;

    *cr = SCC_WR0_RST_INT;		/* reset ext interrupts */
    *cr = SCC_WR0_ERR_RST;		/* reset errors */

    *cr = SCC_WR0_SEL_WR9;				/* master interrupt control */
    *cr = SCC_WR9_MIE;			/* enable interrupts */
/**#    *cr = SCC_WR9_MIE | SCC_WR9_NV;	/* enable interrupts */

    /* calculate the baud rate constant, from the input clock freq.
     * assumes that the input clock is the system cpu clock / 4, and
     * the divide by 16 bit is set (done in WR4 above). */

    baudConstant = ((SYS_CPU_FREQ / 128) / DEFAULT_BAUD) - 2;

    *cr = SCC_WR0_SEL_WR12;		/* LSB of baud constant */
    *cr = (char)baudConstant;		/* write LSB */
    *cr = SCC_WR0_SEL_WR13;		/* MSB of baud constant */
    *cr = (char)(baudConstant >> 8);	/* write MSB */

    intUnlock (oldlevel);
    }

/*******************************************************************************
*
* tyCoOpen - open file to usart
*
* ARGSUSED
*/

LOCAL int tyCoOpen (pTyCoDv, name, mode)
    TY_CO_DEV *pTyCoDv;
    char      *name;
    int        mode;

    {
    return ((int) pTyCoDv);
    }
/*******************************************************************************
*
* tyCoRead - task level read routine
*
* This routine fields all read calls to the MVME-147 usart.
*/

LOCAL int tyCoRead (pTyCoDv, buffer, maxbytes)
    TY_CO_DEV *pTyCoDv;
    char      *buffer;
    int        maxbytes;

    {
    return (tyRead ((TY_DEV_ID) pTyCoDv, buffer, maxbytes));
    }
/*******************************************************************************
*
* tyCoWrite - task level write routine
*
* This routine fields all write calls to the MVME-147 usart.
*/

LOCAL int tyCoWrite (pTyCoDv, buffer, nbytes)
    TY_CO_DEV *pTyCoDv;
    char      *buffer;
    int        nbytes;

    {
    return (tyWrite ((TY_DEV_ID) pTyCoDv, buffer, nbytes));
    }

/*******************************************************************************
*
* tyCoIoctl - special device control
*
* This routine handles baud rate requests, and passes all other requests
* to tyIoctl.
*/

LOCAL STATUS tyCoIoctl (pTyCoDv, request, arg)
    TY_CO_DEV *pTyCoDv;	/* device to control */
    int        request;	/* request code */
    int        arg;	/* some argument */

    {
    FAST int     oldlevel;		/* current interrupt level mask */
    FAST int     baudConstant;
    FAST STATUS  status;
    FAST char   *cr;			/* SCC control reg adr */

    switch (request)
	{
	case FIOBAUDRATE:

	    if (arg < 50 || arg > 38400)
	        {
		status = ERROR;		/* baud rate out of range */
		break;
	        }

	    /* calculate the baud rate constant, from the input clock freq.
	       assumes that the input clock is the system cpu clock / 4, and
	       the divide by 16 bit is set (done in tyCoHrdInit). */

	    baudConstant = ((SYS_CPU_FREQ / 128) / arg) - 2;

	    cr = pTyCoDv->cr;

	    /* disable interrupts during chip access */

	    oldlevel = intLock ();

	    *cr = 0x0c;			/* LSB of baud constant */
	    *cr = (char)baudConstant;	/* write LSB */
	    *cr = 0x0d;			/* MSB of baud constant */
	    *cr = (char)(baudConstant >> 8); /* write MSB */

	    intUnlock (oldlevel);

	    status = OK;
	    break;

	default:
	    status = tyIoctl ((TY_DEV_ID) pTyCoDv, request, arg);
	    break;
	}
    return (status);
    }

/*******************************************************************************
*
* tyCoInt - interrupt level processing
*
* This routine handles interrupts from both of the SCC's.
* We determine from the paramter which SCC interrupted us, then look at
* the code to find out which channel, and what kind of interrupt.
*/

LOCAL VOID tyCoInt (sccNum)
    int sccNum;

    {
    FAST char 	    *cr;
    FAST char        intStatus;
    FAST TY_CO_DEV  *pTyCoDv;
    char 	     outChar;

    /* We need to find out which channel interrupted.  We need to read
     * the B channel of the interrupting SCC to find out which channel
     * really interrupted.  Note that things are set up so that the A
     * channel is channel 0, even though on the chip it is the one with
     * the higher address
     */

    pTyCoDv = &tyCoDv [(sccNum * 2) + 1];
    cr = pTyCoDv->cr;

    *cr = SCC_WR0_SEL_WR2;			/* read reg 2 */
    intStatus = *cr;

    if ((intStatus & 0x08) != 0)
	{				/* the A channel interrupted */
	--pTyCoDv;
	cr = pTyCoDv->cr;
	}

    switch (intStatus & 0x06)
	{
	case 0x00:			/* Tx Buffer Empty */
	    if (pTyCoDv->created && (tyITx (&pTyCoDv->tyDev, &outChar) == OK))
		*(pTyCoDv->dr) = outChar;
	    else
		{
		/* no more chars to xmit now.  reset the tx int,
		 * so the SCC doesn't keep interrupting us. */

		*cr = SCC_WR0_RST_TX_INT;
		}
	    break;

	case 0x04:			/* RxChar Avail */
	    if (pTyCoDv->created)
		tyIRd (&pTyCoDv->tyDev, *(pTyCoDv->dr));
	    break;

	case 0x02:			/* External Status Change */
	case 0x06:			/* Special receive condition */
	    break;			/* ignore */

	}


    *cr = SCC_WR0_RST_HI_IUS;	/* Reset the interrupt in the Z8530 */
    }
/*******************************************************************************
*
* tyCoStartup - transmitter startup routine
*
* Call interrupt level character output routine.
*/

LOCAL VOID tyCoStartup (pTyCoDv)
    TY_CO_DEV *pTyCoDv;		/* ty device to start up */

    {
    char outChar;

    if (tyITx (&pTyCoDv->tyDev, &outChar) == OK)
	*(pTyCoDv->dr) = outChar;
    }
