/* tyCoDrv.c - The Integrated Solutions VME68k30 on-board USART tty handler */

/*
modification history
--------------------
*/

/*
DESCRIPTION
This is the driver for the Integrated Solutions VME68k30 on-board serial ports.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, tyCoDrv to
initialize the driver, and tyCoDevCreate to create devices.

TYCODRV
Before using the driver, it must be initialized by calling the routine:
.CS
    tyCoDrv ()
.CE
This routine should be called exactly once, before any reads, writes, or
tyCoDevCreates.  Normally, it is called from usrRoot.

CREATING TERMINAL DEVICES
Before a terminal can be used, it must be created.  This is done
with the tyCoDevCreate call.
Each port to be used should have exactly one device associated with it,
by calling this routine.

.CS
    STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
	char *name;	/* Name to use for this device *
	int channel;	/* Physical channel for this device (0 or 1) *
	int rdBufSize;	/* Read buffer size, in bytes *
	int wrtBufSize;	/* Write buffer size, in bytes *
.CE

For instance, to create the device "/tyCo/0", with buffer sizes of 512 bytes,
the proper call would be:
.CS
   tyCoDevCreate ("/tyCo/0", 0, 512, 512);
.CE
IOCTL
This driver responds to all the same ioctl codes as a normal ty driver.
The baud rates available are 50, 110, 134, 200, 300, 600, 1200, 1050,
2400, 4800, 7200, 9600, 38400.

*/

#include "vxWorks.h"
#include "ioLib.h"
#include "iosLib.h"
#include "memLib.h"
#include "wdLib.h"
#include "tyLib.h"
#include "vme68k30.h"

typedef struct			/* BAUD */
    {
    int rate;		/* a baud rate */
    char csrVal;	/* rate to write to the csr reg to get that baud rate */
    } BAUD;

typedef struct			/* T/R register set */
    {
    char mr1;		/* mode registers 1 and 2 */
#define mr2 mr1
    char sr;		/* status register */
#define csr sr		/* clock select register */
    char cr;		/* command register */
    char thr;		/* Tx holding register */
#define rhr thr		/* Rx holding register */
    } TRREG;

typedef struct			/* TY_CO_DEV */
    {
    TY_DEV tyDev;
    BOOL created;	/* true if this device has really been created */
    TRREG *regs;	/* pointer to registers on OCTART */
    int blkno;		/* register block number for this channel */
    char rem;		/* bit for receive-enable mask */
    char tem;		/* bit for transmit-enable mask */
    char baud;		/* baud rate code */
    } TY_CO_DEV;

LOCAL TY_CO_DEV tyCoDv [N_USART_CHANNELS] =	/* device descriptors */
    {
    {{{{(NODE *)NULL}}}, FALSE, (TRREG *)VME_U_CH0_ADDR,0, VME_U_INT_RXA,
	VME_U_INT_TXA},
    {{{{(NODE *)NULL}}}, FALSE, (TRREG *)VME_U_CH1_ADDR,0, VME_U_INT_RXB,
	VME_U_INT_TXB},
    {{{{(NODE *)NULL}}}, FALSE, (TRREG *)VME_U_CH2_ADDR,1, VME_U_INT_RXA,
	VME_U_INT_TXA},
    {{{{(NODE *)NULL}}}, FALSE, (TRREG *)VME_U_CH3_ADDR,1, VME_U_INT_RXB,
	VME_U_INT_TXB},
    {{{{(NODE *)NULL}}}, FALSE, (TRREG *)VME_U_CH4_ADDR,2, VME_U_INT_RXA,
	VME_U_INT_TXA},
    {{{{(NODE *)NULL}}}, FALSE, (TRREG *)VME_U_CH5_ADDR,2, VME_U_INT_RXB,
	VME_U_INT_TXB},
    {{{{(NODE *)NULL}}}, FALSE, (TRREG *)VME_U_CH6_ADDR,3, VME_U_INT_RXA,
	VME_U_INT_TXA},
    {{{{(NODE *)NULL}}}, FALSE, (TRREG *)VME_U_CH7_ADDR,3, VME_U_INT_RXB,
	VME_U_INT_TXB},
    };

/* baudTable is a table of the available baud rates, and the values to write
   to the csr reg to get those rates */

LOCAL BAUD baudTable [] =
    {
    {50,	0x00}, {110,	0x11}, {134,	0x22}, {200,	0x33},
    {300,	0x44}, {600,	0x55}, {1200,	0x66}, {1050,	0x77},
    {2400,	0x88}, {4800,	0x99}, {7200,	0xaa}, {9600,	0xbb},
    {38400,	0xcc}
    };

LOCAL int tyCoNum;		/* driver number assigned to this driver */

/* forward declarations */

LOCAL VOID tyCoStartup ();
LOCAL int tyCoOpen ();
LOCAL int tyCoRead ();
LOCAL int tyCoWrite ();
LOCAL STATUS tyCoIoctrl ();
LOCAL VOID txInt ();
LOCAL VOID rxInt ();

/***********************************************************************
*
* tyCoDrv - ty driver initialization routine
*
* This routine initializes the serial driver, sets up interrupt vectors,
* and performs hardware initialization of the serial ports.
*
* This routine must be called in supervisor state, since it does
* physical I/O directly.
*/

STATUS tyCoDrv ()

    {
    /* the 68030 board uses a single auto-vectored (level 4) interrupt
       for all OCTART exception processing; you have to poll the chip's
       interrupt status register to find the cause of the interrupt. The
       actual interrupt handling routine is in sysLib.c.  All we need
       to do here is connect to it using sysUsartConnect. */

    sysUsartConnect (rxInt, txInt);

    tyCoHrdInit ();

    tyCoNum = iosDrvInstall (tyCoOpen, (FUNCPTR) NULL, tyCoOpen, (FUNCPTR) NULL,
			     tyCoRead, tyCoWrite, tyCoIoctrl);

    return (OK);
    }
/******************************************************************
*
* tyCoDevCreate - create a device for the onboard ports
*
* This routine creates a device on one of the serial ports.  Each port
* to be used should have exactly one device associated with it, by calling
* this routine.
*/

STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
    char *name;		/* Name to use for this device */
    int channel;	/* Physical channel for this device (0 thru 7) */
    int rdBufSize;	/* Read buffer size, in bytes */
    int wrtBufSize;	/* Write buffer size, in bytes */

    {
    /* if there is a device already on this channel, don't do it */

    if (tyCoDv [channel].created)
	return (ERROR);

    /* initialize the ty descriptor, and turn on the bit for this
     * receiver in the interrupt mask */

    if (tyDevInit ((TY_DEV_ID) &tyCoDv [channel], 
		    rdBufSize, wrtBufSize, tyCoStartup) != OK)

	return (ERROR);

    sysImrSet (tyCoDv [channel].blkno, tyCoDv [channel].rem, 0);

    /* Mark the device as having been created, and add the device to
     * the io sys */

    tyCoDv [channel].created = TRUE;
    return (iosDevAdd ((DEV_HDR *) &tyCoDv [channel], name, tyCoNum));
    }
/******************************************************************
*
* tyCoHrdInit - initialize the usart.
*
* This routine initializes the isi on-board usart for the wrs/VRTX
* environment. 
*
* This routine must be called in supervisor mode, since it accesses I/O
* space.
*
*/

LOCAL VOID tyCoHrdInit ()

    {
    int oldlevel;		/* current interrupt level mask */
    int channel;

    oldlevel = intLock ();	 /* disable interrupts during init */

    /* reset and enable all USART channels (8) */
    for (channel = 0; channel < N_USART_CHANNELS; channel++)
	{
	/* reset the channel */
	afcwb (&tyCoDv[channel].regs->cr, VME_U_RX_DIS | VME_U_TX_DIS);
	afcwb (&tyCoDv[channel].regs->cr, VME_U_MR_RST);
	afcwb (&tyCoDv[channel].regs->cr, VME_U_RX_RST);
	afcwb (&tyCoDv[channel].regs->cr, VME_U_TX_RST);
	afcwb (&tyCoDv[channel].regs->cr, VME_U_ERR_RST);
	afcwb (&tyCoDv[channel].regs->cr, VME_U_BRK_RST);

	/* set comm to 8 bits, no parity,  one stop bit */
	afcwb (&tyCoDv[channel].regs->mr1, VME_U_NO_PAR | VME_U_8_BITS);
	afcwb (&tyCoDv[channel].regs->mr2, VME_U_1_STOP);

	/* set baud rate to 9600 */
	afcwb (&tyCoDv[channel].regs->csr, baudTable[11].csrVal);
	tyCoDv[channel].baud = baudTable[11].csrVal;

	/* enable the receivers and transmitters on each channel */
	afcwb (&tyCoDv[channel].regs->cr, VME_U_RX_EN | VME_U_TX_EN);

	/* all interrupts are masked out: the receiver interrupt will be
	   enabled in tyCoDevCreate */
	}

    intUnlock (oldlevel);
    } 

/**********************************************************************
*
* tyCoOpen - open file to isi usart
*
* ARGSUSED
*/

LOCAL int tyCoOpen (pTyCoDv, name, mode)
    TY_CO_DEV *pTyCoDv;
    char *name;
    int mode;

    {
    return ((int) pTyCoDv);
    }
/**********************************************************************
*
* tyCoRead - Task level read routine for isi usart.
*
* This routine fields all read calls to the isi usart.  It
* calls tyRead with a pointer to the appropriate element of ty_dev.
*/

LOCAL int tyCoRead (pTyCoDv, buffer, maxbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int maxbytes;

    {
    return (tyRead ((TY_DEV_ID) pTyCoDv, buffer, maxbytes));
    }
/**********************************************************************
*
* tyCoWrite - Task level write routine for isi usart.
*
* This routine fields all write calls to the isi usart.  It
* calls tyWrite with a pointer to the appropriate element of ty_dev.
*/

LOCAL int tyCoWrite (pTyCoDv, buffer, nbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int nbytes;

    {
    return (tyWrite ((TY_DEV_ID) pTyCoDv, buffer, nbytes));
    }
/***********************************************************************
*
* tyCoIoctrl - special device control
*
* This routine handles baud rate requests, and passes all other requests
* to tyIoctl.
*/

LOCAL STATUS tyCoIoctrl (pTyCoDv, request, arg)
    TY_CO_DEV *pTyCoDv;	/* device to control */
    int request;	/* request code */
    int arg;		/* some argument */

    {
    int i;
    STATUS status;

    switch (request)
	{
	case FIOBAUDRATE:
	    status = ERROR;
	    for (i = 0; i < NELEMENTS (baudTable); i++)
		{
		if (baudTable [i].rate == arg)
		    {
		    /* disable xmtr and rcvr */
		    afcwb (&pTyCoDv->regs->cr, VME_U_RX_DIS | VME_U_TX_DIS);

		    /* set comm to 8 bits, no parity,  one stop bit */
		    afcwb (&pTyCoDv->regs->mr1, VME_U_NO_PAR | VME_U_8_BITS);
		    afcwb (&pTyCoDv->regs->mr2, VME_U_1_STOP);

		    /* set baud rate */
		    afcwb (&pTyCoDv->regs->csr, baudTable [i].csrVal);
		    pTyCoDv->baud = baudTable[i].csrVal;

		    /* enable the receivers and transmitters on each channel */
		    afcwb (&pTyCoDv->regs->cr, VME_U_RX_EN | VME_U_TX_EN);
		    status = OK;
		    break;
		    }
		}
	    break;

	default:
	    status = tyIoctl ((TY_DEV_ID) pTyCoDv, request, arg);
	    break;
	}
    return (status);
    }

/**********************************************************************
*
* rxInt - handle a receiver interrupt
*
* This routine gets called by sysUsartInt to handle a receiver interrupt.
*/

LOCAL VOID rxInt (channel)
    FAST int channel;

    {
    char ch;

    ch = afcrb (&tyCoDv [channel].regs->rhr);
    tyIRd ((TY_DEV_ID) &tyCoDv [channel], ch);
    }

/**********************************************************************
*
* txInt - handle a transmitter interrupt
*
* This routine gets called by sysUsartInt to handle a xmitter interrupt.
* If there is another character to be transmitted, it sends it.  If
* not, or if a device has never been created for this channel, we just
* disable the interrupt.
*/

LOCAL VOID txInt (channel)
    FAST int channel;

    {
    char outChar;

    if ((tyCoDv [channel].created) &&
	(tyITx ((TY_DEV_ID) &tyCoDv [channel], &outChar) == OK))

	afcwb (&tyCoDv [channel].regs->thr, outChar);

    else

	/* turn off the transmitter */

	sysImrSet (tyCoDv [channel].blkno, 0, tyCoDv [channel].tem);
    }

/**********************************************************************
*
* tyCoStartup - Transmitter startup routine
*
* Call interrupt level character output routine for isi usart.
*/

LOCAL VOID tyCoStartup (pTyCoDv)
    TY_CO_DEV *pTyCoDv;		/* ty device to start up */

    {
    /* all we gotta do is enable the transmitter and it should interrupt
     * to write the next char out. */

    sysImrSet (pTyCoDv->blkno, pTyCoDv->tem, 0);
    }

/**********************************************************************
*
* tyCoReset - reset serial port
*
*/
tyCoReset (channel)
int channel;
{
    if (!tyCoDv[channel].created)
	return (ERROR);

    /* disable rx and tx temporarily */
    afcwb (&tyCoDv[channel].regs->cr, VME_U_RX_DIS | VME_U_TX_DIS);

    /* set comm to 8 bits, no parity,  one stop bit */
    afcwb (&tyCoDv[channel].regs->mr1, VME_U_NO_PAR | VME_U_8_BITS);
    afcwb (&tyCoDv[channel].regs->mr2, VME_U_1_STOP);

    /* reset baud rate */
    afcwb (&tyCoDv[channel].regs->csr, tyCoDv [channel].baud);

    /* enable the receivers and transmitters on each channel */
    afcwb (&tyCoDv[channel].regs->cr, VME_U_RX_EN | VME_U_TX_EN);
    return (OK);
}
