/* sysLib.c - Integrated Solutions VME68225 system dependent library */

static char *copyright = "Copyright 1988, Wind River Systems, Inc.";

/*
DESCRIPTION
This library contains a set of routines to manipulate the primary functions
of the CPU board.  The goal is to provide a board-independant interface on
which vxWorks and application code can be built in a system-independant way.
Not every feature of every board is supported by this library; a particular
board may have various extensions to the capabilities described here.
Also not every board will support all the functions provided by this library.
And some boards provide some of the functions of this library with hardware
switches, jumpers, or PALs, instead of software controllable registers.

The funtions addressed here include:

    initialization functions:
	- initialize hardware to known state
	- identify the system

    memory/address space functions:
	- get on-board memory limit
	- map from local to bus and bus to local address spaces
	- enable/disable cache memory
	- set/get non-volatile RAM

    bus interrupt functions:
	- enable/disable bus interrupt levels
	- generate bus interrupts

    serial channel functions (see tyCoDrv):
	- enable/disable serial channel interrupts
	- set serial channel baud rates
	- get/put bytes from a serial channel

    clock/timer functions:
       - enable/disable timer interrupts
       - set timer periodic rate

    mailbox/location monitor functions:
       - enable mailbox/location monitor interrupts

BUGS
The routine sysToMonitor (2) doesn't work because the trap vector
is squirreled away after usrInit (2) sets up vxWorks exception vectors.
The auxiliary clock doesn't work.
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "vme.h"
#include "memLib.h"
#include "sysLib.h"
#include "ioLib.h"
#include "config.h"

IMPORT char end;	/* end of system, inserted automatically by ld */

IMPORT FUNCPTR intVecGet ();
IMPORT VOID logMsg ();


/* globals */

int   sysBus      = BUS;		/* system bus type (VME_BUS, etc) */
int   sysCpu      = CPU;		/* system cpu type (MC680x0) */
int   sysCommType = USE_POLL;		/* system inter-processor comm. type */
char *sysBootLine = BOOT_LINE_ADRS;	/* address of boot line */
char *sysExcMsg   = EXC_MSG_ADRS;	/* catastrophic message area */
int   sysProcNum;			/* processor number of this cpu */
int   sysFlags;				/* boot flags */
char  sysBootHost[BOOT_FIELD_LEN];	/* name of host from which we booted */
char  sysBootFile[BOOT_FIELD_LEN];	/* name of file from which we booted */

/* locals */

LOCAL int clkTicksPerSecond	= 60;	/* fixed */
LOCAL int sysClkRunning		= FALSE;
LOCAL int auxClkTicksPerSecond	= 60;
LOCAL FUNCPTR sysAuxClkRoutine	= NULL;
LOCAL int sysAuxClkArg		= NULL;
LOCAL int auxClkRunning		= FALSE;

LOCAL char *memTop = NULL;
LOCAL FUNCPTR usartTxInt	= NULL;
LOCAL FUNCPTR usartRxInt	= NULL;


/*******************************************************************************
*
* sysModel - return model name of the system CPU
*
* Use this routine to find the model name of the system CPU.
*
* RETURNS: pointer to string "ISI 68225"
*/

char *sysModel ()

    {
    return ("ISI 68225");
    }
/*******************************************************************************
*
* sysHwInit - initialize hardware
*
* This routine initializes various features of the board.
* It is normally called from usrInit (2) in usrConfig (1).
* This routine must be called in supervisor mode.
*
* Save rom monitor address.
* Initialize Memory Management Unit (MMU).
* Setup context register (CTX).
*/

VOID sysHwInit ()

    {
    /* save the address of the rom monitor transfer address */

    /* initialize MMU */

    sysInitMMU ();

    /* set up the CTX reg: clock off, parity off, context = 0. */

    *VME_CTX = 0;

    /* turn off interrupts on the DUART */

    sysImrSet (0, 0);
    }
/*******************************************************************************
*
* sysInitMMU - Initialize the 68020 memory management
*/

LOCAL VOID sysInitMMU ()

    {
    int ptp;
    int tbuf;
    int page;
    int nPages;
    int nValidPtps;
    long ptpVal;
    char context;

    /* Set up ptt.  We put it at the top of memory, then modify memTop
       so the memory essentially 'doesn't exist'. The ptt must be at
       an even page boundary. */

    (void) sysMemTop ();	/* to make sure memTop gets calculated */

    nPages = (int) memTop / VME_PAGE_SIZE;
    memTop = (char *) (((int) memTop - (nPages * 4)) & (~ (VME_PAGE_SIZE - 1)));

    for (page = 0; page < nPages; page++)
	*((ULONG *) memTop + page) = page | VME_TBBE_VALID;

    /* set up the ptp buffer.  All contexts use the same ptt. Each ptp
       entry points at 512 ptte's in the ptt. */

    nValidPtps = (nPages + 511) / 512;
    for (context = 0; context < 8; context++)
	{
	*VME_CTX = context;
	for (ptp = 0; ptp < VME_NPTPS; ptp++)
	    {
	    if (ptp < nValidPtps)
		{
		ptpVal = (int) memTop >> 12;
		if (ptp & 1)

		    /* If this is an odd-numbered ptp, It needs to have
		       its LSB set, which is a high bit in the ptp entry */

		    ptpVal |= VME_PTP_LSB;
		}
	    else
		ptpVal = VME_PTP_INVALID;
	    
	    /* Write the ptp */

	    *((long *) (VME_PTP_BASE + (ptp * VME_PTP_INCR))) = ptpVal;
	    }
	}

    /* Zero the system and user translation buffers, so the entries
       will be pulled in from our ptt. */

    for (tbuf = 0; tbuf < VME_NTBUFS; tbuf++)
	{
	*((long *) (VME_TBUF_BASE_SYS + (tbuf * VME_TBUF_INCR))) = 0;
	*((long *) (VME_TBUF_BASE_USR + (tbuf * VME_TBUF_INCR))) = 0;
	}
    }
/*******************************************************************************
*
* sysMemTop - get top of memory address
*
* This routine returns the address of the first missing byte of memory.
*
* It starts probing at the end of bss, then tries to read a byte
* at every 4K boundary until it finds one that can't be read.
* This routine must be called in supervisor mode, the first time only.
* After that, it can be called in user mode as well.
*
* RETURNS: address of the first missing byte of memory
*/

char *sysMemTop ()

    {
    char bitBucket;
    char orig;

    if (memTop == NULL)
	{
	/* Look for the top of memory.  Starting at the first even page
	   boundary after _end, probe each page, then write it and
	   read it back, just to be sure. */

	memTop = (char *) ((((int) &end) + (VME_PAGE_SIZE - 1)) &
			    (~ (VME_PAGE_SIZE - 1)));

	FOREVER
	    {
	    if (vxMemProbe (memTop, READ, 1, &bitBucket) != OK)
		break;

	    orig = *memTop;

	    *memTop = 0x55;
	    if (*memTop != 0x55)
		break;

	    *memTop = orig;

	    if(((int)memTop + VME_PAGE_SIZE) > (int)0xDFFFFE)
		break;

	    memTop += VME_PAGE_SIZE;
	    }

	/* memTop is now pointing to the first non-existent address */

	/* NOTE: sysInitMMU will munge pointer further! */
	}

    return (memTop);
    }
/*******************************************************************************
*
* sysToMonitor - transfer to rom monitor
*
* This routine transfers control to the rom monitor.  It is usually called
* only by the routine reboot, which services control-x, and bus errors at
* interrupt level.  In special circumstances, however, the user may wish
* to introduce a new startType such that a special bootrom facility would be
* enabled.
*
* RETURNS: OK (if we ever continue from the rom monitor).
*/

STATUS sysToMonitor (startType)
    int startType;	/* parameter is passed to ROM to tell it how to boot *
			   The possible type are defined in h/sysLib.h */
    {
    (* ((FUNCPTR) (ROM_BASE_ADRS + 8))) (startType);

    return (OK);	/* in case we ever continue from rom monitor */
    }

/*******************************************************************************
*
* sysClkConnect - connect routine to system clock interrupt
*
* This routine connects the given function to the system clock interrupt.
* System clock interrupts are not enabled.
* It is normally called from usrRoot (2) in usrConfig (1) to connect
* usrClock (2) to the system clock interrupt.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), usrClock (2)
*/

STATUS sysClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each system clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    return (intConnect (INUM_TO_IVEC (INT_VEC_CLOCK), routine, arg));
    }
/*******************************************************************************
*
* sysClkDisable - turn off system clock interrupts
*/

VOID sysClkDisable ()

    {
    *VME_CTX = ~VME_CTX_CLOCK | ((~sysProcNum & 3) << 5);
    sysClkRunning = FALSE;
    }
/*******************************************************************************
*
* sysClkEnable - turn system clock interrupts on
*/

VOID sysClkEnable ()

    {
    /* Enable the clock. The clock always goes 60 Hz. We must also
       preserve the mailbox interrupt address field in the CTX reg. */

    *VME_CTX = VME_CTX_CLOCK | ((~sysProcNum & 3) << 5);
    sysClkRunning = TRUE;
    }
/*******************************************************************************
*
* sysClkRateGet - get rate of system clock
*
* This routine is used to find out the system clock speed.
*
* RETURNS: number of ticks per second of the system clock
*
* SEE ALSO: sysClkRateSet (2)
*/

int sysClkRateGet ()
    
    {
    return (clkTicksPerSecond);
    }
/*******************************************************************************
*
* sysClkRateSet - set rate of system clock
*
* This routine sets the clock rate of the system clock.
* System clock interrupts are not enabled.
* It is normally called by usrRoot (2) in usrConfig (1).
*
* NOTE IS20:
* The clock rate is not adjustable.
* The speed is always 60 Hz.
* Thus this routine has no effect.
*
* SEE ALSO: sysClkRateGet (2)
*
* ARGSUSED
*/

VOID sysClkRateSet (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    }

/*******************************************************************************
*
* sysAuxClkConnect - connect a routine to the auxiliary clock interrupt
*
* This routine connects a user routine to the auxiliary clock interrupt.
* Auxiliary clock interrupts are not enabled.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), sysAuxClkDisconnect (2)
*/

STATUS sysAuxClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each auxiliary clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    sysAuxClkRoutine = routine;
    sysAuxClkArg     = arg;

    return (OK);
    }
/*******************************************************************************
*
* sysAuxClkDisconnect - clear the auxiliary clock routine.
* 
* This routine disables the auxiliary clock interrupt, stops the timer,
* and disconnects the routine currently connected to the auxiliary clock
* interrupt.
*
* SEE ALSO: sysAuxClkConnect (2), sysAuxClkDisable (2)
*/

VOID sysAuxClkDisconnect ()

    {
    /* disable the auxiliary clock interrupt */

    sysAuxClkDisable ();

    /* connect dummy routine, just in case */

    sysAuxClkConnect (logMsg, (int) "auxiliary clock interrupt\n");
    }
/*******************************************************************************
*
* sysAuxClkDisable - turn off auxiliary clock interrupts
*/

VOID sysAuxClkDisable ()

    {
    char dummy;

    sysImrSet (0, VME_U_INT_TIMER);
    auxClkRunning = FALSE;

    /* can't stop C/T -- change clock source */
    dummy = (char)VME_U_ACR_CTR_EXT;
    *VME_U_ACR = dummy;

    if (*VME_U_STOP_TMR); /* EOI on the timer */
    }
/*******************************************************************************
*
* sysAuxClkEnable - turn auxiliary clock interrupts on
*/

int sysAuxClkEnable ()

    {
    unsigned int ctr;

    /* calculate the divide ration, and write it to the timer chip.  If
       ticksPerSecond == 0, write a vl of 0 to stop the clock. */

    ctr = (auxClkTicksPerSecond == 0) ? 0
				      : (3686400 / (2 * auxClkTicksPerSecond));

#ifdef	DEBUG
    printf ("ctr = 0x%x, CTUR = 0x%x, CTLR = 0x%x.\n", ctr,
	    ((ctr & 0xff00) >> 8), (ctr & 0x00ff));
#endif	DEBUG

    if ((ctr & 0xffff0000) == 0)
	{
	*VME_U_ACR = (char)VME_U_ACR_TMR_XTAL;
	*VME_U_CTUR = (ctr & 0xff00) >> 8;
	*VME_U_CTLR = (ctr & 0x00ff);

	sysImrSet (VME_U_INT_TIMER, 0);
	if (*VME_U_START_TMR); /* EOI on the timer */

	/* set up USART timer to use crystal */
	*VME_U_ACR = VME_U_ACR_TMR_XTAL;

	auxClkRunning = TRUE;
	return (OK);
	}
    else
	{
	printErr ("sysAuxClkEnable: invalid clock rate (%d)\n",
		  auxClkTicksPerSecond);
	auxClkRunning = FALSE;
	return (ERROR);
	}
    }
/*******************************************************************************
*
* sysAuxClkRateGet - get the auxiliary timer frequency
*
* This routine finds out the auxiliary clock speed.
*
* RETURNS: number of ticks per second of the auxiliary clock
*
* SEE ALSO: sysAuxClkRateSet (2)
*/

int sysAuxClkRateGet ()

    {
    return (auxClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysAuxClkRateSet - set the auxiliary timer frequency
*
* This routine sets the clock rate of the auxiliary clock.
* Auxiliary clock interrupts are not enabled.
*
* SEE ALSO: sysAuxClkRateGet (2), sysAuxClkEnable (2)
*/

VOID sysAuxClkRateSet (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second;
			     * 0 stops the clock */

    {
    if (ticksPerSecond == 0) {
	auxClkRunning = FALSE;
	sysClkDisable ();
	return (OK);
    }

    if (ticksPerSecond > 0)
	auxClkTicksPerSecond = ticksPerSecond;

    if (auxClkRunning)
	{
	sysAuxClkDisable ();
	sysAuxClkEnable ();
	}
    }

/*******************************************************************************
*
* sysLocalToBusAdrs - convert local address to bus address
*
* Given a local memory address, this routine returns the VME address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unable to get to that local address from the bus
*
* SEE ALSO: sysBusToLocalAdrs (2)
*
* ARGSUSED
*/

STATUS sysLocalToBusAdrs (adrsSpace, localAdrs, pBusAdrs)
    int adrsSpace;	/* bus address space in which busAdrs resides;
			 * use address modifier codes as defined in vme.h,
			 * such as VME_AM_STD_SUP_DATA */
    char *localAdrs;	/* local address to convert */
    char **pBusAdrs;	/* where to return bus address */

    {
    *pBusAdrs = localAdrs;
    return (OK);
    }
/*******************************************************************************
*
* sysBusToLocalAdrs - convert bus address to local address
*
* Given a VME memory address, this routine returns the local address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unknown address space
*
* SEE ALSO: sysLocalToBusAdrs (2)
*/

STATUS sysBusToLocalAdrs (adrsSpace, busAdrs, pLocalAdrs)
    int adrsSpace;	/* bus address space in which busAdrs resides;
			 * use address modifier codes as defined in vme.h,
			 * such as VME_AM_STD_SUP_DATA */
    char *busAdrs;	/* bus address to convert */
    char **pLocalAdrs;	/* where to return local address */

    {
    switch (adrsSpace)
	{
	case VME_AM_SUP_SHORT_IO:
	case VME_AM_USR_SHORT_IO:
	    *pLocalAdrs = (char *) (VME_SHORTIO | (int) busAdrs);
	    return (OK);

	case VME_AM_STD_SUP_ASCENDING:
	case VME_AM_STD_SUP_PGM:
	case VME_AM_STD_SUP_DATA:
	case VME_AM_STD_USR_ASCENDING:
	case VME_AM_STD_USR_PGM:
	case VME_AM_STD_USR_DATA:
	    *pLocalAdrs = (char *) (VME_STANDARD | (int) busAdrs);
	    return (OK);

	case VME_AM_EXT_SUP_ASCENDING:
	case VME_AM_EXT_SUP_PGM:
	case VME_AM_EXT_SUP_DATA:
	case VME_AM_EXT_USR_ASCENDING:
	case VME_AM_EXT_USR_PGM:
	case VME_AM_EXT_USR_DATA:
	    if (busAdrs < (char *) 0x03000000)
		return (ERROR);

	    *pLocalAdrs = (char *) (VME_EXTENDED | (int) busAdrs);
	    return (OK);

	default:
	    return (ERROR);
	}
    }

/*******************************************************************************
*
* sysIntDisable - disable interrupt level
*
* This routine disables the specified interrupt level.
*
* NOTE IS20:
* Interrupt levels are jumperable but not software controllable.
* This routine has no effect.
*
* RETURNS: OK
*
* ARGSUSED
*/

STATUS sysIntDisable (intLevel)
    int intLevel;	/* interrupt level to disable */

    {
    return (OK);
    }
/*******************************************************************************
*
* sysIntEnable - enable interrupt level
*
* This routine enables the specified VME interrupt level.
*
* NOTE IS20:
* VME interrupts are jumperable but not software controllable.
* This routine has no effect.
*
* RETURNS: OK
*
* ARGSUSED
*/

STATUS sysIntEnable (intLevel)
    int intLevel;	/* interrupt level to enable */

    {
    return (OK);
    }
/*******************************************************************************
*
* sysBusIntAck - acknowledge interrupt
*
* This routine acknowledges the specified interrupt.
*
* NOTE IS20:
* If jumpered correctly, the VME interrupts will be acknowledged automatically.
* This routine has no effect.
*
* RETURNS: NULL
*
* ARGSUSED
*/

int sysBusIntAck (intLevel)
    int intLevel;		/* interrupt level to acknowledge */

    {
    return (NULL);
    }

/*******************************************************************************
*
* sysMailboxConnect - connect routine to the mailbox interrupt
*
* This routine connects the given function to the mailbox interrupt.
*
* RETURNS: ERROR since there is no mailbox facility
*
* SEE ALSO: intConnect (2)
*
* ARGSUSED
*/

STATUS sysMailboxConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each mailbox interrupt */
    int arg;		/* argument with which to call routine */

    {
    return (ERROR);
    }
/*******************************************************************************
*
* sysMailboxEnable - enable mailbox interrupt
*
* This routine enables the mailbox interrupt.
*
* RETURNS: ERROR since there is no mailbox facility
*
* ARGSUSED
*/

STATUS sysMailboxEnable (mailboxAdrs)
    char *mailboxAdrs;		/* mailbox address */

    {
    return (ERROR);
    }

/*******************************************************************************
*
* sysIpiConnect - connect routine to the interprocessor interrupt
*
* This routine connects the given function to the interprocessor interrupt.
* For the 68225, this just uses intConnect
* It is expected that this will be used only with the vb driver, so
* there is no corresponding disconnect function.
*
* SEE ALSO: intConnect (2)
*/
STATUS sysIpiConnect(routine, arg)
FUNCPTR routine;
int arg;
{
    intConnect ((FUNCPTR *) 0x70, routine, arg);
    return;
}

/*******************************************************************************
*
* sysProcNumGet - get processor number
*
* This routine returns the processor number previously set with 
* sysProcNumSet (2).
*
* RETURNS: processor number
*/

sysProcNumGet ()
    {
    register int n = 0;

    n = (~(*VME_CTX))&VME_CTX_CNODE;
    if (n == VME_CTX_CNODE)
	n = 0;
    return n;
    }

/*******************************************************************************
*
* sysProcNumSet - set processor number
*
* Set the processor number for this CPU.  Processor numbers should be
* unique on a single backplane.
*
* NOTE IS20:
* The context register is set appropriately.
*/

VOID sysProcNumSet (procNum)
    int procNum;		/* processor number */

    {
    register int n = 0;

    n = (~(*VME_CTX))&VME_CTX_CNODE;
    if (n == VME_CTX_CNODE)
	n = 0;
    sysProcNum = n;
    return n;
    }
/*******************************************************************************
*
* sysBusTas - test and set across VME bus
*
* This routine does a 680x0 test-and-set instruction across the backplane.
*
* RETURNS: TRUE (successful set) or FALSE (failure).
*/

BOOL sysBusTas (addr)
    char *addr;		/* address to be tested and set */

    {
    return (vxTas (addr));
    }


/* miscellaneous support routines */

/*******************************************************************************
*
* sysImrSet - set and clear bits in the USART's interrupt mask register
*
* This routine sets and clears bits in the USART's IMR.
*
* This routine sets and clears bits in a local copy of the IMR, then
* writes that local copy to the USART.  This means that all changes to
* the IMR must go through this routine.  Otherwise, any direct changes
* to the IMR would be lost the next time this routine is called.
*
* NOTE: This routine must be called in supervisor mode.
*/

VOID sysImrSet (setBits, clearBits)
    char setBits;	/* which bits to set in the IMR */
    char clearBits;	/* which bits to clear in the IMR */

    {
    static char imr = 0;	/* our copy of the interrupt mask reg */

    imr = (imr | setBits) & (~clearBits);
    *VME_U_IMR = imr;
    }
/*******************************************************************************
*
* sysUsartInt - interrupt level processing for VME68225 USART
*
* This routine handles an interrupt from the on-board USART.  The interrupt
* is decoded by checking the uart's interrupt status reg, and the appropriate 
* routine invoked.  
*
* This routine is located here because we use the extra timer on the USART
* chip as the auxiliary timer, which is handled here.
*/

LOCAL VOID sysUsartInt ()

    {
    FAST char isr = *VME_U_ISR;

    if (isr & VME_U_INT_TXA)
	{
	/* transmitter channel A interrupt.  call USART routine, if
	   there is one, else issue an EOI */

	if (usartTxInt == NULL)
	    sysImrSet (0, VME_U_INT_TXA); 	/* EOI */
	else
	    (*usartTxInt) (0);
	}

    if (isr & VME_U_INT_TXB)
	{
	/* transmitter channel B interrupt.  call USART routine, if
	   there is one, else issue an EOI */

	if (usartTxInt == NULL)
	    sysImrSet (0, VME_U_INT_TXB); 	/* EOI */
	else
	    (*usartTxInt) (1);
	}

    if ((isr & VME_U_INT_RXA) && (usartRxInt != NULL))
	(*usartRxInt) (0);			/* receiver channel A */

    if ((isr & VME_U_INT_RXB) && (usartRxInt != NULL))
	(*usartRxInt) (1);			/* receiver channel B */

    if (isr & VME_U_INT_TIMER)		/* on-chip timer */
	{
	if (sysAuxClkRoutine != NULL)
	    (*sysAuxClkRoutine) (sysAuxClkArg);

	if (*VME_U_STOP_TMR);	/* EOI on the timer */
	}
    }
/*******************************************************************************
*
* sysUsartConnect - set interrupt routines for the USART
*
* This routine connects the serial driver interrupt routines to the
* interrupt routine.  It should be called once, from the serial driver.
*/

VOID sysUsartConnect (recvRoutine, xmitRoutine)
    FUNCPTR recvRoutine;	/* receive routine */
    FUNCPTR xmitRoutine;	/* transmit routine */

    {
    static BOOL usartConnected = FALSE;

    if (!usartConnected)
    	(void) intConnect (INUM_TO_IVEC (INT_VEC_CONSOLE), sysUsartInt, NULL);

    usartConnected = TRUE;
    usartTxInt = xmitRoutine;
    usartRxInt = recvRoutine;
    }

/*
 * vdma resource allocator for the 68020 board 
 * under uniworks
 */

/* the is68k225 doesn't have vdma capabilities, so
 * these routines are really no-ops
 */

/************************************************************************/
/* 
 * sysInitVdma
 */
sysInitVdma ()
{
    return (0);
}

/************************************************************************/
/* 
 * sysSetVdma
 *
 * set the vdma registers for a buffer at
 * location buff of size nbytes
 *
 * returns address of window requested which is always
 * the same as the address requested
 */
char *sysSetVdma (buff, nbytes)
char *buff;
int nbytes;
{
    return (buff);
}

/************************************************************************/
/* 
 * sysGiveVdma
 *
 */
sysGiveVdma (vmeoff)
int vmeoff;
{
    return (0);
}


k_putc (ch)
char ch;
{
    register int i;

    *(char *)0x58000006 = ch & 0x7f;
    for (i = 0; i < 10000; i++);
}

/************************************************************************/
/* 
 * sysDataCacheFlush
 */
sysDataCacheFlush ()
{
    return (OK);
}

/************************************************************************/
/* 
 * sysDataCacheFlushEntry
 */
sysDataCacheFlushEntry (addr, num)
char *addr;
int num;
{
    return (OK);
}

k_puts(){}
