/* tyDUSCCDrv.c - The Force 37 DUSCC serial driver */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;


/*
modification history
--------------------
01a,07dec88,jcf  written by modifying mz7122/tyCoDrv.c v01c.
*/

/*
DESCRIPTION
This is the driver for the Force 37 DUSCC chip.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, tyDUSCCDrv to
initialize the driver, and tyDUSCCDevCreate to create devices.

TYCODRV
Before using the driver, it must be initialized by calling the routine:
.CS
    STATUS tyDUSCCDrv ()
.CE
The status returned is OK.

This routine should be called exactly once, before any reads, writes, or
tyDUSCCDevCreates.  Normally, it is called from usrRoot.

CREATING TERMINAL DEVICES
Before a terminal can be used, it must be created.  This is done
with the tyDUSCCDevCreate call.
Each port to be used should have exactly one device associated with it,
by calling this routine.

.CS
    STATUS tyDUSCCDevCreate (name, channel, rdBufSize, wrtBufSize)
	char *name;	/* Name to use for this device
	int channel;	/* Physical channel for this device, always 0
	int rdBufSize;	/* Read buffer size, in bytes
	int wrtBufSize;	/* Write buffer size, in bytes
.CE

For instance, to create the device "/tyDUSCC/0", with buffer sizes of 512 bytes,
the proper call would be:
.CS
   tyDUSCCDevCreate ("/tyDUSCC/0", 0, 512, 512);
.CE
IOCTL
This driver responds to all the same ioctl codes as a normal ty driver.

*/

#include "vxWorks.h"
#include "iv68k.h"
#include "ioLib.h"
#include "iosLib.h"
#include "memLib.h"
#include "tyLib.h"
#include "frc37.h"


typedef struct			/* BAUD */
    {
    int rate;		/* a baud rate */
    char xmtVal;	/* rate to write to the Transmitter Timing Register */
    char rcvVal;	/* rate to write to the Reciever Timing Register */
    } BAUD;

typedef struct			/* TY_CO_DEV */
    {
    TY_DEV tyDev;
    BOOL created;	/* true if this device has really been created */
    char *rdr;		/* receiver data port */
    char *xdr;		/* transmitter data port */
    char *ttr;		/* transmitter timer register */
    char *rtr;		/* receiver timer register */
    char *ier;		/* interrupt enable register */
    char rrdy; 		/* receiver ready bit */
    char xrdy;		/* transmitter ready bit */
    } TY_CO_DEV;

LOCAL TY_CO_DEV tyDUSCCDv [N_CHANNELS] =	/* device descriptors */
    {
    {{{{NULL}}}, FALSE,
	DUSCC_RXFIFOA     (FRC37_DUSCC_BASE_ADRS),
	DUSCC_TXFIFOA     (FRC37_DUSCC_BASE_ADRS),
	DUSCC_TTRA        (FRC37_DUSCC_BASE_ADRS),
	DUSCC_RTRA        (FRC37_DUSCC_BASE_ADRS),
	DUSCC_IERA 	  (FRC37_DUSCC_BASE_ADRS),
	DUSCC_GSR_A_RXRDY,
	DUSCC_GSR_A_TXRDY },
    {{{{NULL}}}, FALSE,
	DUSCC_RXFIFOB     (FRC37_DUSCC_BASE_ADRS),
	DUSCC_TXFIFOB     (FRC37_DUSCC_BASE_ADRS),
	DUSCC_TTRB        (FRC37_DUSCC_BASE_ADRS),
	DUSCC_RTRB        (FRC37_DUSCC_BASE_ADRS),
	DUSCC_IERB        (FRC37_DUSCC_BASE_ADRS),
	DUSCC_GSR_B_RXRDY,
	DUSCC_GSR_B_TXRDY },
    };

#define CHANNEL_A	0		/* for structure references */
#define CHANNEL_B	1		/* for structure references */

/* baudTable is a table of the available baud rates, and the values to write
   to the csr reg to get those rates */

LOCAL BAUD baudTable [] =
    {
    {50,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_50,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_50 },
    {75,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_75,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_75 },
    {110,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_110,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_110},
    {134,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_134_5,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_134_5},
    {150,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_150,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_150},
    {200,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_200,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_200},
    {300,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_300,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_300},
    {600,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_600,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_600},
    {1050,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_1050,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_1050},
    {1200,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_1200,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_1200},
    {2000,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_2000,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_2000},
    {2400,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_2400,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_2400},
    {4800,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_4800,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_4800},
    {9600,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_9600,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_9600},
    {19200,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_19200,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_19200},
    {38400,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_38400,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_38400} };

LOCAL int tyDUSCCDrvNum;	/* driver number assigned to this driver */

/* forward declarations */

LOCAL VOID tyDUSCCStartup ();
LOCAL int tyDUSCCOpen ();
LOCAL int tyDUSCCRead ();
LOCAL int tyDUSCCWrite ();
LOCAL STATUS tyDUSCCIoctl ();
LOCAL VOID tyDUSCCTxInt ();
LOCAL VOID tyDUSCCRxInt ();

/*******************************************************************************
*
* tyDUSCCDrv - ty driver initialization routine
*
* This routine initializes the driver, sets up interrupt vectors,
* and performs hardware initialization of the ports.
*
* This routine must be called in supervisor state, since it does
* physical I/O directly.
*/

STATUS tyDUSCCDrv ()

    {
    /* check if driver already installed */
     
    if (tyDUSCCDrvNum > 0)
	return (OK);
		    
    /* Connect channel A's vectors */

    intConnect (INUM_TO_IVEC(INT_VEC_DUSCC + DUSCC_INT_A_RXRDY),
		tyDUSCCRxInt, CHANNEL_A);
    intConnect (INUM_TO_IVEC(INT_VEC_DUSCC + DUSCC_INT_A_TXRDY),
		tyDUSCCTxInt, CHANNEL_A);

    /* Connect channel B's vectors */

    intConnect (INUM_TO_IVEC(INT_VEC_DUSCC + DUSCC_INT_B_RXRDY),
		tyDUSCCRxInt, CHANNEL_B);
    intConnect (INUM_TO_IVEC(INT_VEC_DUSCC + DUSCC_INT_B_TXRDY),
		tyDUSCCTxInt, CHANNEL_B);

    tyDUSCCHrdInit ();

    tyDUSCCDrvNum = iosDrvInstall (tyDUSCCOpen, (FUNCPTR) NULL, tyDUSCCOpen,
				   (FUNCPTR) NULL, tyDUSCCRead, tyDUSCCWrite,
				   tyDUSCCIoctl);

    return (tyDUSCCDrvNum == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* tyDUSCCDevCreate - create a device for the onboard ports
*
* This routine creates a device on one of the ports.  Each port
* to be used should have exactly one device associated with it, by calling
* this routine.
*/

STATUS tyDUSCCDevCreate (name, channel, rdBufSize, wrtBufSize)
    char *name;		/* Name to use for this device */
    int channel;	/* Physical channel for this device (0 or 1) */
    int rdBufSize;	/* Read buffer size, in bytes */
    int wrtBufSize;	/* Write buffer size, in bytes */

    {
    if (tyDUSCCDrvNum <= 0)
	{
	errnoSet (S_ioLib_NO_DRIVER);
	return (ERROR);
	}
				     
    /* if this device already exists, don't create it */

    if (tyDUSCCDv [channel].created)
	return (ERROR);

    /* initialize the ty descriptor, and turn on the bit for this
     * receiver in the interrupt mask */

    if (tyDevInit ((TY_DEV_ID) &tyDUSCCDv [channel], 
		    rdBufSize, wrtBufSize, tyDUSCCStartup) != OK)

	return (ERROR);

    *(tyDUSCCDv [channel].ier) = DUSCC_IER_RXRDY;

    /* Mark the device as having been created, and add the device to
     * the io sys */

    tyDUSCCDv [channel].created = TRUE;

    return (iosDevAdd ((DEV_HDR *) &tyDUSCCDv [channel], name, tyDUSCCDrvNum));
    }
/*******************************************************************************
*
* tyDUSCCHrdInit - initialize the DUSCC
*
* This routine initializes the DUSCC for the VxWorks environment. 
*
* This routine must be called in supervisor mode, since it accesses I/O space.
*/

LOCAL VOID tyDUSCCHrdInit ()

    {
    int oldlevel;		/* current interrupt level mask */

    oldlevel = intLock ();

    /* 8 data bits, 1 stop bit, no parity, set for 9600 baud */

	/* channel A initialiazation */

    *DUSCC_CMR1A (FRC37_DUSCC_BASE_ADRS)= DUSCC_CMR1_ASYNC;
    *DUSCC_CMR2A (FRC37_DUSCC_BASE_ADRS)= DUSCC_CMR2_DTI_POLL_OR_INT;
    *DUSCC_TPRA (FRC37_DUSCC_BASE_ADRS)	= DUSCC_TPR_ASYNC_1 | DUSCC_TPR_8BITS;
    *DUSCC_TTRA (FRC37_DUSCC_BASE_ADRS)	= DUSCC_TTR_CLK_BRG |
					  DUSCC_TTR_BAUD_9600;
    *DUSCC_RPRA (FRC37_DUSCC_BASE_ADRS)	= DUSCC_RPR_8BITS;
    *DUSCC_RTRA (FRC37_DUSCC_BASE_ADRS)	= DUSCC_RTR_CLK_BRG |
					  DUSCC_RTR_BAUD_9600;
    *DUSCC_OMRA (FRC37_DUSCC_BASE_ADRS)	= DUSCC_OMR_TXRDY_FIFO_EMPTY |
			 		  DUSCC_OMR_TX_RES_CHAR_LENGTH_TPR;
    *DUSCC_CCRA (FRC37_DUSCC_BASE_ADRS)	= DUSCC_CCR_TX_ENABLE_TX;
    *DUSCC_CCRA (FRC37_DUSCC_BASE_ADRS)	= DUSCC_CCR_RX_ENABLE_RX;

	/* channel B initialiazation */

    *DUSCC_CMR1B (FRC37_DUSCC_BASE_ADRS)= DUSCC_CMR1_ASYNC;
    *DUSCC_CMR2B (FRC37_DUSCC_BASE_ADRS)= DUSCC_CMR2_DTI_POLL_OR_INT;
    *DUSCC_TPRB (FRC37_DUSCC_BASE_ADRS)	= DUSCC_TPR_ASYNC_1 | DUSCC_TPR_8BITS;
    *DUSCC_TTRB (FRC37_DUSCC_BASE_ADRS)	= DUSCC_TTR_CLK_BRG |
					  DUSCC_TTR_BAUD_9600;
    *DUSCC_RPRB (FRC37_DUSCC_BASE_ADRS)	= DUSCC_RPR_8BITS;
    *DUSCC_RTRB (FRC37_DUSCC_BASE_ADRS)	= DUSCC_RTR_CLK_BRG |
                                          DUSCC_RTR_BAUD_9600;
    *DUSCC_OMRB (FRC37_DUSCC_BASE_ADRS)	= DUSCC_OMR_TXRDY_FIFO_EMPTY |
					  DUSCC_OMR_TX_RES_CHAR_LENGTH_TPR;
    *DUSCC_CCRB	(FRC37_DUSCC_BASE_ADRS)	= DUSCC_CCR_TX_ENABLE_TX;
    *DUSCC_CCRB	(FRC37_DUSCC_BASE_ADRS)	= DUSCC_CCR_RX_ENABLE_RX;

    /* all interrupts are masked out: the receiver interrupt will be enabled
       in the tyDUSCCDevCreate */

    intUnlock (oldlevel);
    } 
/*******************************************************************************
*
* tyDUSCCOpen - open file to DUSCC
*
* ARGSUSED
*/

LOCAL int tyDUSCCOpen (pTyDUSCCDv, name, mode)
    TY_CO_DEV *pTyDUSCCDv;
    char *name;
    int mode;

    {
    return ((int) pTyDUSCCDv);
    }
/*******************************************************************************
*
* tyDUSCCRead - task level read routine for DUSCC
*
* This routine fields all read calls to the DUSCC.
*/

LOCAL int tyDUSCCRead (pTyDUSCCDv, buffer, maxbytes)
    TY_CO_DEV *pTyDUSCCDv;
    char *buffer;
    int maxbytes;

    {
    return (tyRead ((TY_DEV_ID) pTyDUSCCDv, buffer, maxbytes));
    }
/*******************************************************************************
*
* tyDUSCCWrite - task level write routine for DUSCC
*
* This routine fields all write calls to the DUSCC.
*/

LOCAL int tyDUSCCWrite (pTyDUSCCDv, buffer, nbytes)
    TY_CO_DEV *pTyDUSCCDv;
    char *buffer;
    int nbytes;

    {
    return (tyWrite ((TY_DEV_ID) pTyDUSCCDv, buffer, nbytes));
    }
/*******************************************************************************
*
* tyDUSCCIoctl - special device control
*
* This routine handles baud rate requests, and passes all other requests
* to tyIoctl.
*/

LOCAL STATUS tyDUSCCIoctl (pTyDUSCCDv, request, arg)
    TY_CO_DEV *pTyDUSCCDv;	/* device to control */
    int request;	/* request code */
    int arg;		/* some argument */

    {
    int i;
    STATUS status;

    switch (request)
	{
	case FIOBAUDRATE:
	    status = ERROR;
	    for (i = 0; i < NELEMENTS (baudTable); i++)
		{
		if (baudTable [i].rate == arg)	/* lookup baud rate value */
		    {
		    *pTyDUSCCDv->ttr = baudTable [i].xmtVal;
		    *pTyDUSCCDv->rtr = baudTable [i].rcvVal;
		    status = OK;
		    break;
		    }
		}
	    break;

	default:
	    status = tyIoctl ((TY_DEV_ID) pTyDUSCCDv, request, arg);
	    break;
	}
    return (status);
    }

/*******************************************************************************
*
* tyDUSCCRxInt - handle a receiver interrupt
*
* This routine gets called at interrupt level to handle a receiver A interrupt.
*/

LOCAL VOID tyDUSCCRxInt (channel)
    FAST int channel;

    {
    tyIRd ((TY_DEV_ID) &tyDUSCCDv [channel], (char) *tyDUSCCDv [channel].rdr);
    *DUSCC_GSR (FRC37_DUSCC_BASE_ADRS) = tyDUSCCDv [channel].rrdy;
    }
/*******************************************************************************
*
* tyDUSCCTxInt - handle a transmitter interrupt
*
* This routine gets called from interrupt level to handle a xmitter interrupt.
* If there is another character to be transmitted, it sends it.  If
* not, or if a device has never been created for this channel, we just
* disable the interrupt.
*/

LOCAL VOID tyDUSCCTxInt (channel)
    FAST int channel;

    {
    char outChar;

    if ((tyDUSCCDv [channel].created) &&
	(tyITx ((TY_DEV_ID) &tyDUSCCDv [channel], &outChar) == OK))
	{
	*(tyDUSCCDv [channel].xdr) = outChar;
	*DUSCC_GSR (FRC37_DUSCC_BASE_ADRS) = tyDUSCCDv [channel].xrdy;
	}
    else
	{
	/* turn off the transmitter */

	*(tyDUSCCDv [channel].ier) &= ~DUSCC_IER_TXRDY;
	}
    }
/*******************************************************************************
*
* tyDUSCCStartup - transmitter startup routine
*
* Call interrupt level character output routine for DUSCC.
*/

LOCAL VOID tyDUSCCStartup (pTyDUSCCDv)
    TY_CO_DEV *pTyDUSCCDv;		/* ty device to start up */

    {
    char outChar;

    if ((pTyDUSCCDv->created) &&
	(tyITx ((TY_DEV_ID) pTyDUSCCDv, &outChar) == OK))
	{
	*(pTyDUSCCDv->ier) |= DUSCC_IER_TXRDY;
	*(pTyDUSCCDv->xdr) = outChar;
	*DUSCC_GSR (FRC37_DUSCC_BASE_ADRS) = pTyDUSCCDv->xrdy;
	}
    }
