/* tyCoDrv.c - MFP tty handler */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01a,07dec88,jcf  written by modifying 01h of mv133/tyMFPDrv.c version.
*/

/*
DESCRIPTION
This is the driver for the Multi-Function Peripheral Chip used
on the Force-37.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, tyCoDrv to
initialize the driver, and tyCoDevCreate to create devices.

TYCODRV
Before using the driver, it must be initialized by calling the routine:
.CS
    tyCoDrv ()
.CE
This routine should be called exactly once, before any reads, writes, or
tyCoDevCreates.  Normally, it is called from usrRoot.

CREATING TERMINAL DEVICES
Before a terminal can be used, it must be created.  This is done
with the tyCoDevCreate call.
Each port to be used should have exactly one device associated with it,
by calling this routine.

.CS
    STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
	char *name;	/* Name to use for this device
	int channel;	/* Physical channel for this device, always 0
	int rdBufSize;	/* Read buffer size, in bytes
	int wrtBufSize;	/* Write buffer size, in bytes
.CE

For instance, to create the device "/tyCo/0", with buffer sizes of 512 bytes,
the proper call would be:
.CS
   tyCoDevCreate ("/tyCo/0", 0, 512, 512);
.CE
IOCTL
This driver responds to all the same ioctl codes as a normal ty driver.
The baud rates available are 150,300, 600, 1200, 2400, 4800, and 9600.
*/

#include "vxWorks.h"
#include "ioLib.h"
#include "iosLib.h"
#include "memLib.h"
#include "wdLib.h"
#include "tyLib.h"
#include "iv68k.h"
#include "frc37.h"


typedef struct			/* TY_CO_DEV */
    {
    TY_DEV tyDev;
    BOOL created;	/* true if this device has really been created */
    char *ucr;		/* MFP control register */
    char *rsr;		/* receiver status register */
    char *tsr;		/* transmit status register */
    char *udr;		/* MFP data register */
    } TY_CO_DEV;

LOCAL TY_CO_DEV tyCoDv;

LOCAL int tyCoDrvNum;		/* driver number assigned to this driver */

typedef struct			/* BAUD */
    {
    int rate;		/* a baud rate */
    char timeConstant;  /* time constant to write to the timer C register */
    } BAUD;

/* The following rates assume that the divide by 16 on usart control
 * register, and the divide by 4 on timer c prescale are selected.
 */

LOCAL BAUD baudTable [] =
    {
    {150,	192}, {300,	 96}, {600,	48}, {1200,	24},
    {2400,	 12}, {4800,	  6}, {9600,	 3},
    };

/* forward declarations */

LOCAL VOID tyCoStartup ();
LOCAL int tyCoOpen ();
LOCAL int tyCoRead ();
LOCAL int tyCoWrite ();
LOCAL STATUS tyCoIoctl ();
LOCAL VOID tyCoInt ();


/*******************************************************************************
*
* tyCoDrv - ty driver initialization routine
*
* This routine initializes the serial driver, sets up interrupt vectors,
* and performs hardware initialization of the serial ports.
*
* This routine must be called in supervisor state, since it does
* physical I/O directly.
*/

STATUS tyCoDrv ()

    {
    /* check if driver already installed */

    if (tyCoDrvNum > 0)
	return (OK);

    /* Connect the MFP interrupt vector to the appropriate routine */

    intConnect (INUM_TO_IVEC(INT_VEC_MFP), tyCoInt, NULL);

    tyCoHrdInit ();

    tyCoDrvNum = iosDrvInstall (tyCoOpen, (FUNCPTR) NULL, tyCoOpen,
				 (FUNCPTR) NULL, tyCoRead, tyCoWrite,
				 tyCoIoctl);

    /* initialize the DUSCC driver here... its as good a place as any */

    tyDUSCCDrv ();

    tyDUSCCDevCreate ("/tyDUSCC/0", 0, 512, 512);
    tyDUSCCDevCreate ("/tyDUSCC/1", 1, 512, 512);

    return (tyCoDrvNum == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* tyCoDevCreate - create a device for the onboard ports
*
* This routine creates a device on one of the serial ports.  Each port
* to be used should have exactly one device associated with it, by calling
* this routine.
* NOTE: The channel parameter is ignored since the MFP chip only has one
* serial port.  The parameter is included only to make the calling
* sequence compatible with other versions. Thus ARGSUSED.
* ARGSUSED
*/

STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
    char *name;		/* Name to use for this device */
    int channel;	/* Physical channel for this device (0 only!) */
    int rdBufSize;	/* Read buffer size, in bytes */
    int wrtBufSize;	/* Write buffer size, in bytes */

    {
    int temp;

    if (tyCoDrvNum <= 0)
	{
	errnoSet (S_ioLib_NO_DRIVER);
	return (ERROR);
	}

    /* if this device already exists, don't recreate it */

    if (tyCoDv.created)
	return (ERROR);

    /* initialize the ty descriptor, and turn on the bit for this
     * receiver in the interrupt mask */

    tyCoDv.ucr = MFP_UCR(FRC37_MFP_BASE_ADRS);
    tyCoDv.rsr = MFP_RSR(FRC37_MFP_BASE_ADRS);
    tyCoDv.tsr = MFP_TSR(FRC37_MFP_BASE_ADRS);
    tyCoDv.udr = MFP_UDR(FRC37_MFP_BASE_ADRS);

    if (tyDevInit ((TY_DEV_ID) &tyCoDv, 
		    rdBufSize, wrtBufSize, tyCoStartup) != OK)
	{
	return (ERROR);
	}

    /* enable the receiver */

    temp = *MFP_IMRA(FRC37_MFP_BASE_ADRS);
    temp |= MFP_A_RX_FULL;
    *MFP_IMRA(FRC37_MFP_BASE_ADRS) = temp;

    /* mark the device as created, and add the device to the I/O system */

    tyCoDv.created = TRUE;

    return (iosDevAdd ((DEV_HDR *) &tyCoDv, name, tyCoDrvNum));
    }
/*******************************************************************************
*
* tyCoHrdInit - initialize the Force-37 MFP
*
* This routine initializes the Force-37 MFP for the VxWorks environment. 
*/

LOCAL VOID tyCoHrdInit ()

    {
    int oldlevel = intLock ();		/* LOCK INTERRUPTS */
    int temp;


    /* 8 data bits, 1 stop bit, no parity */

    *MFP_UCR(FRC37_MFP_BASE_ADRS) = MFP_UCR_16X | MFP_UCR_8BIT | MFP_UCR_1STOP;

    /* set timer c for /4 prescale */
  
    temp = *MFP_TCDCR(FRC37_MFP_BASE_ADRS);
    temp = MFP_TCDCR_C_DELAY_4 | (temp & MFP_TCDCR_D_MASK);
    *MFP_TCDCR(FRC37_MFP_BASE_ADRS) = temp;

    *MFP_TCDR(FRC37_MFP_BASE_ADRS) = 0x03;	/* set for 9600 baud */

    /* enable the receivers and transmitters on both channels */

    *MFP_RSR(FRC37_MFP_BASE_ADRS) = MFP_RSR_RX_ENABLE;	/* enable rx status */
    *MFP_TSR(FRC37_MFP_BASE_ADRS) = MFP_TSR_TX_ENABLE;	/* enable tx status */


    temp = *MFP_IERA(FRC37_MFP_BASE_ADRS);
    temp |= MFP_A_RX_FULL;		/* enable recv interrupts */
    *MFP_IERA(FRC37_MFP_BASE_ADRS) = temp;

    temp = *MFP_IERA(FRC37_MFP_BASE_ADRS);
    temp |= MFP_A_TX_EMPTY;		/* enable transmit interrupts */
    *MFP_IERA(FRC37_MFP_BASE_ADRS) = temp;

    /* The interrupt mask register (IMRA) is still masking out receiver and
     * transmitter interrupts.  The receiver interrupt will be enabled in
     * tyCoDevCreate.  The transmitter interrupt will be enabled in
     * tyCoStartup.
     */

    intUnlock (oldlevel);		/* UNLOCK INTERRUPTS */
    } 
/*******************************************************************************
*
* tyCoOpen - open file to the Force-37 MFP
*
* ARGSUSED
*/

LOCAL int tyCoOpen (pTyCoDv, name, mode)
    TY_CO_DEV *pTyCoDv;
    char *name;
    int mode;

    {
    return ((int) pTyCoDv);
    }
/*******************************************************************************
*
* tyCoRead - task level read routine for Force-37 MFP
*
* This routine fields all read calls to the Force-37 MFP.  It
* calls tyRead with a pointer to the appropriate element of ty_dev.
*/

LOCAL int tyCoRead (pTyCoDv, buffer, maxbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int maxbytes;

    {
    return (tyRead ((TY_DEV_ID) pTyCoDv, buffer, maxbytes));
    }
/*******************************************************************************
*
* tyCoWrite - Task level write routine for Force-37 MFP
*
* This routine fields all write calls to the Force-37 MFP.
*/

LOCAL int tyCoWrite (pTyCoDv, buffer, nbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int nbytes;

    {
    return (tyWrite ((TY_DEV_ID) pTyCoDv, buffer, nbytes));
    }
/*******************************************************************************
*
* tyCoIoctl - special device control
*
* This routine handles FIOBAUDRATE requests and passes all others to tyIoctl.
*/

LOCAL STATUS tyCoIoctl (pTyCoDv, request, arg)
    TY_CO_DEV *pTyCoDv;	/* device to control */
    int request;	/* request code */
    int arg;		/* some argument */

    {
    int i;
    int temp;
    STATUS status;

    switch (request)
	{
	case FIOBAUDRATE:

	    /* set timer c for /4 prescale */
	  
	    temp = *MFP_TCDCR(FRC37_MFP_BASE_ADRS);
	    temp = MFP_TCDCR_C_DELAY_4 | (temp & MFP_TCDCR_D_MASK);
	    *MFP_TCDCR(FRC37_MFP_BASE_ADRS) = temp;

	    status = ERROR;
	    for (i = 0; i < NELEMENTS (baudTable); i++)
		{
		if (baudTable [i].rate == arg)	/* lookup time constant */
		    {
		    *MFP_TCDR(FRC37_MFP_BASE_ADRS) = baudTable [i].timeConstant;
		    status = OK;
		    break;
		    }
		}
	    break;

	default:
	    status = tyIoctl ((TY_DEV_ID) pTyCoDv, request, arg);
	    break;
	}

    return (status);
    }
/*******************************************************************************
*
* tyCoStartup - transmitter startup routine
*
* Call interrupt level character output routine for Force-37 MFP.
*/

LOCAL VOID tyCoStartup (pTyCoDv)
    TY_CO_DEV *pTyCoDv;		/* ty device to start up */

    {
    char outchar;
    int temp;

    /* all we have to do is enable the transmitter and it should interrupt us
     * to write the next char out.
     */
    if (tyITx ((TY_DEV_ID) pTyCoDv, &outchar) == OK)
	*pTyCoDv->udr = outchar;

    /* turn the transmitter on */

    temp = *MFP_IMRA(FRC37_MFP_BASE_ADRS);
    temp |= MFP_A_TX_EMPTY;
    *MFP_IMRA(FRC37_MFP_BASE_ADRS) = temp;
    }

/******************************************************************************
*
* tyCoInt - single interrupt service routine for all MFP interrupts.
*
* The Force-37 IRHDL chip does not support the MFP providing a vector which
* includes status.  In other words, all MFP interrupts end up here.  We
* must demultiplex them and invoke the appropriate routine.
*
* CAVEATS:
* The only MFP interrupts acknowledged currently are the receiver full and
* the transmitter empty.  The other interrupts may be demultiplexed in the
* same manner as below by extending the 'else if' construct.
* The decission not to demultiplex all MFP interrupts was based on keeping
* the interrupt service routine as minimal as possible.
*/  

LOCAL VOID tyCoInt ()

    {
    char outChar;
    int temp;

    /* handle any transceiver interrupts */

    if (*MFP_IPRA(FRC37_MFP_BASE_ADRS) & MFP_A_TX_EMPTY)
	{
	if ((tyCoDv.created) &&
	    (tyITx ((TY_DEV_ID) &tyCoDv, &outChar) == OK))
	    {
	    *(tyCoDv.udr) = outChar;
	    }
	else
	    {

	    /* turn off the transmitter */

	    temp = *MFP_IMRA(FRC37_MFP_BASE_ADRS);
	    temp &= ~MFP_A_TX_EMPTY;
	    *MFP_IMRA(FRC37_MFP_BASE_ADRS) = temp;
	    }

	/* acknowledge MFP transceiver interrupt */

	*MFP_IPRA(FRC37_MFP_BASE_ADRS) = ~MFP_A_TX_EMPTY;
	*MFP_ISRA(FRC37_MFP_BASE_ADRS) = ~MFP_A_TX_EMPTY;
	}
	

    /* handle any receiver interrupts */

    else if (*MFP_IPRA(FRC37_MFP_BASE_ADRS) & MFP_A_RX_FULL)
	{
	tyIRd ((TY_DEV_ID) &tyCoDv, *tyCoDv.udr);

	/* acknowledge MFP receiver interrupt */

	*MFP_IPRA(FRC37_MFP_BASE_ADRS) = ~MFP_A_RX_FULL;
	*MFP_ISRA(FRC37_MFP_BASE_ADRS) = ~MFP_A_RX_FULL;
	}

    /* other MFP interrupts can be demultiplexed as above */
    }
