/* sysLib.c - Force SYS68K/CPU-31 system dependent library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;


/*
modification history
--------------------
01a,17jan89,jcf  written by modifying 01a of frc30 version.
*/

/*
DESCRIPTION
This library contains a set of routines to manipulate the primary functions
of the CPU board.  The goal is to provide a board-independant interface on
which VxWorks and application code can be built in a system-independant way.
Not every feature of every board is supported by this library; a particular
board may have various extensions to the capabilities described here.
Also not every board will support all the functions provided by this library.
And some boards provide some of the functions of this library with hardware
switches, jumpers, or PALs, instead of software controllable registers.

The funtions addressed here include:

    initialization functions:
	- initialize hardware to known state
	- identify the system

    memory/address space functions:
	- get on-board memory limit
	- map from local to bus and bus to local address spaces
	- enable/disable cache memory
	- set/get non-volatile RAM

    bus interrupt functions:
	- enable/disable bus interrupt levels
	- generate bus interrupts

    serial channel functions (see tyCoDrv):
	- enable/disable serial channel interrupts
	- set serial channel baud rates
	- get/put bytes from a serial channel

    clock/timer functions:
       - enable/disable timer interrupts
       - set timer periodic rate

    mailbox/location monitor functions:
       - enable mailbox/location monitor interrupts
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "vme.h"
#include "ioLib.h"
#include "sysLib.h"
#include "config.h"

IMPORT char end;		/* end of system, created automatically by ld */

IMPORT VOID logMsg ();

/* globals */

int   sysBus      = BUS;		/* system bus type (VME_BUS, etc) */
int   sysCpu      = CPU;		/* system cpu type (MC680x0) */
char *sysBootLine = BOOT_LINE_ADRS;	/* address of boot line */
char *sysExcMsg   = EXC_MSG_ADRS;	/* catastrophic message area */
int   sysProcNum;			/* processor number of this cpu */
int   sysFlags;				/* boot flags */
char  sysBootHost[BOOT_FIELD_LEN];	/* name of host from which we booted */
char  sysBootFile[BOOT_FIELD_LEN];	/* name of file from which we booted */


/* locals */

LOCAL FUNCPTR sysClkRoutine     = NULL;	/* routine to call on clock interrupt */
LOCAL int sysClkArg             = NULL;	/* its argument */
LOCAL int sysClkTicksPerSecond  = 60;	/* system clock rate */
LOCAL BOOL sysClkRunning        = FALSE;/* system clock enabled? */

LOCAL FUNCPTR sysAuxClkRoutine  = NULL;	/* routine to call on clock interrupt */
LOCAL int sysAuxClkArg          = NULL;	/* its argument */
LOCAL int auxClkTicksPerSecond	= 60;	/* auxiliary clock rate */
LOCAL BOOL auxClkIsConnected    = FALSE;/* hooked up to interrupt yet? */
LOCAL BOOL auxClkIsRunning      = FALSE;/* auxiliary clock enabled? */

LOCAL BOOL sysPit1IsConnected   = FALSE;/* hooked up to interrupt yet? */

LOCAL FUNCPTR sysMboxRoutine    = NULL;	/* routine to call on mbox. interrupt */
LOCAL int sysMboxArg            = NULL;	/* its argument */
LOCAL BOOL sysMboxIsConnected   = FALSE;/* hooked up to interrupt yet? */

/* forward declarations */

LOCAL VOID sysMboxInt ();
LOCAL VOID sysPit1Int ();
LOCAL VOID sysAuxClkInt ();
LOCAL VOID sysAbortInt ();


/*******************************************************************************
*
* sysModel - return model name of the system CPU
*
* This routine returns a pointer to a string with the model name of this CPU.
*
* RETURNS: pointer to string "Force SYS68K/CPU-31".
*/

char *sysModel ()

    {
    return ("Force SYS68K/CPU-31");
    }
/**********************************************************************
*
* sysHwInit - initialize hardware
*
* This routine initializes the board hardware.
* It is normally called from usrInit (2) in usrConfig (1).
* This routine must be called in supervisor mode.
*
* The timers are initialized and turned off.
*/

VOID sysHwInit ()

    {
    char zero = 0; /* used to force compilers to use move inst instead of clr */

    /* Inhibit cacheing of certain regions of the address space (ie. i/o) */

    sysCacheInhibit ();

    /* Initialize PIT1 ports */

    *PIT_TCR (FRC31_PIT1_BASE_ADRS)   = TIMER_CTL_5;	/* disable timer */
    *PIT_PGCR (FRC31_PIT1_BASE_ADRS)  = H12_ENABLE |    /* h4,h3,h2,h1 ints. */
				        H34_ENABLE;	/*  enabled, pmode=0 */
    *PIT_PSRR (FRC31_PIT1_BASE_ADRS)  = PIRQ;           /* pc5 = pirq,  */
							/*  pc6 = fp sense */

    *PIT_PACR (FRC31_PIT1_BASE_ADRS)  = PORT_SUBMODE_3;	/* submode 1X */
    *PIT_PADDR (FRC31_PIT1_BASE_ADRS) = zero;		/* all inputs */

    *PIT_PBCR (FRC31_PIT1_BASE_ADRS)  = PORT_SUBMODE_3;	/* submode 1X */
    *PIT_PBDDR (FRC31_PIT1_BASE_ADRS) = 0xff;		/* all outputs */

    *PIT_PCDDR (FRC31_PIT1_BASE_ADRS) = zero;	/* all inputs */

    *PIT_PIVR (FRC31_PIT1_BASE_ADRS)  = INT_VEC_PIT1;
    *PIT_TVIR (FRC31_PIT1_BASE_ADRS)  = INT_VEC_PIT1;

    /* Initialize PIT2 ports */

    *PIT_TCR (FRC31_PIT2_BASE_ADRS)   = TIMER_CTL_5;	/* disable timer */
    *PIT_PGCR (FRC31_PIT2_BASE_ADRS)  = zero;		/* no Hx ints */
    *PIT_PSRR (FRC31_PIT2_BASE_ADRS)  = zero;           /* */

    *PIT_PACR (FRC31_PIT2_BASE_ADRS)  = PORT_SUBMODE_0;	/* submode 00 */
    *PIT_PADDR (FRC31_PIT2_BASE_ADRS) = zero;		/* all inputs */

    *PIT_PBCR (FRC31_PIT2_BASE_ADRS)  = PORT_SUBMODE_3;	/* submode 1X */
    *PIT_PBDDR (FRC31_PIT2_BASE_ADRS) = zero;		/* all inputs */

    *PIT_PCDDR (FRC31_PIT2_BASE_ADRS) = 0x21;	/* bits 0,5 outputs */

    *PIT_PIVR (FRC31_PIT2_BASE_ADRS)  = INT_VEC_PIT2;
    *PIT_TVIR (FRC31_PIT2_BASE_ADRS)  = INT_VEC_PIT2;

    /* Initialize the DUSCC1 Ports */

        /* reset all of the gizmo's on the DUSCC1 chip */

    *DUSCC_CCRA (FRC31_DUSCC1_BASE_ADRS) = DUSCC_CCR_TX_RESET_TX;
    *DUSCC_CCRA (FRC31_DUSCC1_BASE_ADRS) = DUSCC_CCR_RX_RESET_RX;
    *DUSCC_CCRA (FRC31_DUSCC1_BASE_ADRS) = DUSCC_CCR_CT_STOP;
    *DUSCC_CCRA (FRC31_DUSCC1_BASE_ADRS) = DUSCC_CCR_DPLL_DISABLE_DPLL;

    *DUSCC_CCRB (FRC31_DUSCC1_BASE_ADRS) = DUSCC_CCR_TX_RESET_TX;
    *DUSCC_CCRB (FRC31_DUSCC1_BASE_ADRS) = DUSCC_CCR_RX_RESET_RX;
    *DUSCC_CCRB (FRC31_DUSCC1_BASE_ADRS) = DUSCC_CCR_CT_STOP;
    *DUSCC_CCRB (FRC31_DUSCC1_BASE_ADRS) = DUSCC_CCR_DPLL_DISABLE_DPLL;

	/* initialize vector register */

    *DUSCC_IVR (FRC31_DUSCC1_BASE_ADRS)  = INT_VEC_DUSCC1;

        /* enable a, enable b, modify vec w/stat, present vec to cpu */

    *DUSCC_ICR (FRC31_DUSCC1_BASE_ADRS)  = DUSCC_ICR_CHAN_B_MAST_INT_EN |
				 	   DUSCC_ICR_CHAN_A_MAST_INT_EN |
					   DUSCC_ICR_VECT_INCLDS_STATUS |
					   DUSCC_ICR_VECT_MODE_VECT;

        /* rest of DUSCC1 initialization done in tyCoHrdInit */

    /* Initialize the FGA-002 */

	/* disable dual porting of memory.  It is enabled in sysProcNumSet() */
    *FGA_ENAMCODE(FRC31_FGA_BASE_ADRS)   = 0x0;	

	/* dual port from the beginning of RAM */
    *FGA_MAINUM(FRC31_FGA_BASE_ADRS)	 = 0x0;		/* local base = 0x0 */
    *FGA_MAINUU(FRC31_FGA_BASE_ADRS)	 = 0x0;		/* (start of RAM)   */

						
	/* set VME dual port base address */
    *FGA_VMEPAGE(FRC31_FGA_BASE_ADRS)    = (LOCAL_MEM_BUS_ADRS &
					    0xf0000000) >> 28;	/* A31-A28 */
    						
    *FGA_BOTTOMPAGEU(FRC31_FGA_BASE_ADRS)= (LOCAL_MEM_BUS_ADRS &
					    0x0ff00000) >> 20;	/* A27-A20 */
    							
    *FGA_BOTTOMPAGEL(FRC31_FGA_BASE_ADRS)= (LOCAL_MEM_BUS_ADRS & 
					    0x000ff000) >> 12;  /* A19-A12 */

	/* set VME dual port top access location */
    *FGA_TOPPAGEU(FRC31_FGA_BASE_ADRS)   = ((ULONG)(LOCAL_MEM_BUS_ADRS - 1 +
					    sysMemTop ()) & 0x0ff00000) >> 20;
								/* A27-A20 */

    *FGA_TOPPAGEL(FRC31_FGA_BASE_ADRS)   = ((ULONG)(LOCAL_MEM_BUS_ADRS - 1 +
					    sysMemTop ()) & 0x000ff000) >> 12;
								/* A19-A12 */

	/* enable RMW cycles, no cache for off board accesses or local i/o */

    *FGA_CTL15(FRC31_FGA_BASE_ADRS)    |= FGA_CTL15_SHAREDRMW |
					  FGA_CTL15_CINHOFFBOARD |
					  FGA_CTL15_CINHLIO;

	/* set mailbox accesses to be either user or supervisor */
    *FGA_CTL5(FRC31_FGA_BASE_ADRS)     |= FGA_CTL5_VME_A16_BOTH;


	/* initialize vector register */
    *FGA_CTL3(FRC31_FGA_BASE_ADRS)	= INT_VEC_FGA >> 4;

	/* initialize interrupt control registers */
    *FGA_ICRMBOX0(FRC31_FGA_BASE_ADRS)	= INT_LVL_MBOX;
    *FGA_ICRMBOX1(FRC31_FGA_BASE_ADRS)	= INT_LVL_MBOX;
    *FGA_ICRMBOX2(FRC31_FGA_BASE_ADRS)	= INT_LVL_MBOX;
    *FGA_ICRMBOX3(FRC31_FGA_BASE_ADRS)	= INT_LVL_MBOX;
    *FGA_ICRMBOX4(FRC31_FGA_BASE_ADRS)	= INT_LVL_MBOX;
    *FGA_ICRMBOX5(FRC31_FGA_BASE_ADRS)	= INT_LVL_MBOX;
    *FGA_ICRMBOX6(FRC31_FGA_BASE_ADRS)	= INT_LVL_MBOX;
    *FGA_ICRMBOX7(FRC31_FGA_BASE_ADRS)	= INT_LVL_MBOX;

    *FGA_ICRTIM0(FRC31_FGA_BASE_ADRS)	= INT_LVL_FGA_TIMER;

    *FGA_ICRFMB0REF(FRC31_FGA_BASE_ADRS)= INT_LVL_MBOX;
    *FGA_ICRFMB1REF(FRC31_FGA_BASE_ADRS)= INT_LVL_MBOX;
    *FGA_ICRFMB0MES(FRC31_FGA_BASE_ADRS)= INT_LVL_MBOX;
    *FGA_ICRFMB1MES(FRC31_FGA_BASE_ADRS)= INT_LVL_MBOX;
    *FGA_ICRDMANORM(FRC31_FGA_BASE_ADRS)= INT_LVL_DMA;
    *FGA_ICRDMAERR(FRC31_FGA_BASE_ADRS)	= INT_LVL_DMA;

    *FGA_ICRPARITY(FRC31_FGA_BASE_ADRS)	= INT_LVL_PARITY;

    *FGA_ICRVME1(FRC31_FGA_BASE_ADRS)	= 1;	/* one to one makes sense */
    *FGA_ICRVME2(FRC31_FGA_BASE_ADRS)	= 2;
    *FGA_ICRVME3(FRC31_FGA_BASE_ADRS)	= 3;
    *FGA_ICRVME4(FRC31_FGA_BASE_ADRS)	= 4;
    *FGA_ICRVME5(FRC31_FGA_BASE_ADRS)	= 5;
    *FGA_ICRVME6(FRC31_FGA_BASE_ADRS)	= 6;
    *FGA_ICRVME7(FRC31_FGA_BASE_ADRS)	= 7;

    /* extended interrupt control registers */
    *FGA_ICRABORT(FRC31_FGA_BASE_ADRS)	= INT_LVL_ABORT | FGA_ICR_EDGE |
					  FGA_ICR_AUTOCLEAR;
    *FGA_ICRACFAIL(FRC31_FGA_BASE_ADRS)	= INT_LVL_ACFAIL | FGA_ICR_AUTOCLEAR;
    *FGA_ICRSYSFAIL(FRC31_FGA_BASE_ADRS)= INT_LVL_SYSFAIL | FGA_ICR_AUTOCLEAR;

    *FGA_ICRLOCAL0(FRC31_FGA_BASE_ADRS)	= INT_LVL_RTC | FGA_ICR_AUTOCLEAR;
    *FGA_ICRLOCAL1(FRC31_FGA_BASE_ADRS)	= INT_LVL_FDC | FGA_ICR_AUTOCLEAR;
    *FGA_ICRLOCAL2(FRC31_FGA_BASE_ADRS)	= INT_LVL_PIT1 | FGA_ICR_AUTOCLEAR;
    *FGA_ICRLOCAL3(FRC31_FGA_BASE_ADRS)	= INT_LVL_PIT2 | FGA_ICR_AUTOCLEAR;
    *FGA_ICRLOCAL4(FRC31_FGA_BASE_ADRS)	= INT_LVL_DUSCC1 | FGA_ICR_AUTOCLEAR;
    *FGA_ICRLOCAL5(FRC31_FGA_BASE_ADRS)	= 0x0;	/* N/C; disable */
    *FGA_ICRLOCAL6(FRC31_FGA_BASE_ADRS)	= 0x0;	/* N/C; disable */
    *FGA_ICRLOCAL7(FRC31_FGA_BASE_ADRS)	= 0x0;	/* N/C; disable */

	/* set FGA to allow DUSCC chips to provide vector */
    *FGA_LOCALIACK(FRC31_FGA_BASE_ADRS) = FGA_LOCALIACK_EXT_500NS;

	/* setup VMEbus arbitration registers */
	
	/* NOTE THAT THE ARBITRATION JUMPER DETERMINES IF THIS ARBITRATION
	 * WILL TAKE PLACE.  THE ARBITER IS ENABLED FOR sysProcNum == 0 in
	 * sysProcNumSet ().
	 */

    *FGA_CTL7(FRC31_FGA_BASE_ADRS) |= FGA_CTL7_RAT_64US;	     /* RAT */
    *FGA_CTL7(FRC31_FGA_BASE_ADRS) &= ~FGA_CTL7_NO_RELEASE_ON_BCLR;  /* RBCLR */
    *FGA_CTL8(FRC31_FGA_BASE_ADRS) |= FGA_CTL8_FAIR_ARB;	     /* FAIR */

	/* disable FMB channels 1 and 0; slot code set in sysProcNumSet */
    *FGA_CTL5(FRC31_FGA_BASE_ADRS) = FGA_CTL5_VME_A16_BOTH;


	/* enable a few choice interrupts */

    *FGA_ICRLOCAL2(FRC31_FGA_BASE_ADRS)	|= FGA_ICR_ENABLE;	/* PIT1 */
    *FGA_ICRLOCAL4(FRC31_FGA_BASE_ADRS)	|= FGA_ICR_ENABLE;	/* DUSCC1 */

    /* The RTC and FDC are not initialized here but in their drivers */
    }
/*******************************************************************************
*
* sysMemTop - get top of memory address
*
* This routine returns the address of the first missing byte of memory.
*
* The PIT controller has 3 input bits on port b which have the memory size
* encoded.
*
* RETURNS: address of the first missing byte of memory
*/

char *sysMemTop ()

    {
    static char *memTop;

    switch ((*PIT_PBDR (FRC31_PIT2_BASE_ADRS)) & MEM_SIZE_MASK)
	{
	case 0x7:
	    memTop = (char *) 0x400000;	/* 4 MB, no nibble mode */
	    break;
	case 0x6:
	    memTop = (char *) 0x100000;	/* 1 MB, no nibble mode */
	    break;
	default : 			/* calculate adrs */
	    memTop = (char *)(0x2000000 >> ((*PIT_PBDR (FRC31_PIT2_BASE_ADRS)) &
     			                     MEM_SIZE_MASK));
	    break;
	}

    return (memTop);
    }
/*******************************************************************************
*
* sysToMonitor - transfer to rom monitor
*
* This routine transfers control to the rom monitor.  It is usually called
* only by the routine reboot, which services control-X, and bus errors at
* interrupt level.  In special circumstances, however, the user may wish
* to introduce a new startType such that a special bootrom facility would be
* enabled.
*
* RETURNS: OK (if we ever continue from the rom monitor)
*
* INTERNAL
* Note that the "WARM" restart address is at:
*    ROM_BASE_ADRS + 16: (8 bytes for vectors + 8 bytes for cold start code)
*/
 
STATUS sysToMonitor (startType)
    int startType;  /* parameter is passed to ROM to tell it how to boot, */
                    /* possible types are defined in sysLib.h             */

    {
    (* ((FUNCPTR) ((int)ROM_BASE_ADRS + 16))) (startType);

    return (OK);    /* in case we ever continue from rom monitor */
    }

/*******************************************************************************
*
* sysClkConnect - connect routine to system clock interrupt
*
* This routine connects the given function to the system clock interrupt.
* System clock interrupts are not enabled.
* It is normally called from usrRoot (2) in usrConfig (1) to connect
* usrClock (2) to the system clock interrupt.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), usrClock (2)
*/

STATUS sysClkConnect (routine, arg)
    FUNCPTR routine;
    int arg;

    {
    if (!sysPit1IsConnected &&
	intConnect (INUM_TO_IVEC (INT_VEC_CLOCK), sysPit1Int, 0) == ERROR)
	{
	return (ERROR);
	}

    sysClkRoutine = routine;
    sysClkArg     = arg;

    sysPit1IsConnected = TRUE;


    /* This routine is called early in vxWorks initialization so we connect
     * other useful interrupt handlers here.
     */

    /* connect abort switch */

    *FGA_ICRABORT(FRC31_FGA_BASE_ADRS) |= FGA_ICR_ENABLE;	/* enable */

    (void) intConnect (INUM_TO_IVEC(INT_VEC_ABORT), sysAbortInt, 0);

    return (OK);
    }
/*******************************************************************************
*
* sysClkDisable - turn off system clock interrupts
*/

VOID sysClkDisable ()

    {
    if (sysClkRunning)
	{
	/* disable the PIT1 Timer */

	*PIT_TCR (FRC31_PIT1_BASE_ADRS) = TIMER_CTL_5;	/* disable timer */

	sysClkRunning = FALSE;
	}
    }
/*******************************************************************************
*
* sysClkEnable - turn system clock interrupts on
*/

VOID sysClkEnable ()

    {
    unsigned int tc;

    /* A 5-bit prescaler (divide-by-32) is assumed to be used
     * on the PIT1 clock input.
     */

    tc = CLOCK_FREQ / (sysClkTicksPerSecond * 32);

    /* write the timer value */

    *PIT_CPRL (FRC31_PIT1_BASE_ADRS) = tc;
    *PIT_CPRM (FRC31_PIT1_BASE_ADRS) = tc >> 8;
    *PIT_CPRH (FRC31_PIT1_BASE_ADRS) = tc >> 16;

    /* enable the PIT1 Timer */

    *PIT_TCR (FRC31_PIT1_BASE_ADRS) = TIMER_CTL_5 | TIMER_ENABLE;

    sysClkRunning = TRUE;
    }
/*******************************************************************************
*
* sysClkRateGet - get rate of system clock
*
* This routine is used to find out the system clock speed.
*
* RETURNS: number of ticks per second of the system clock
*
* SEE ALSO: sysClkRateSet (2)
*/

int sysClkRateGet ()
    
    {
    return (sysClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysClkRateSet - set rate of system clock
*
* This routine sets the clock rate of the system clock.
* System clock interrupts are not enabled.
* It is normally called by usrRoot (2) in usrConfig (1).
*
* SEE ALSO: sysClkRateGet (2), sysClkEnable (2)
*/

VOID sysClkRateSet (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
	sysClkTicksPerSecond = ticksPerSecond;
	
    if (sysClkRunning)
	{
	sysClkDisable ();
	sysClkEnable ();
	}
    }

/*******************************************************************************
*
* sysAuxClkConnect - connect a routine to the auxiliary clock interrupt
*
* This routine connects a user routine to the auxiliary clock interrupt.
* Auxiliary clock interrupts are not enabled.
*
* RETURNS: OK or ERROR if unable to connnect to interrupt
*
* SEE ALSO: intConnect (2), sysAuxClkDisconnect (2)
*/

STATUS sysAuxClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each auxiliary clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    /* connect the routine to the interrupt vector */

    if (!auxClkIsConnected &&
	intConnect (INUM_TO_IVEC (INT_VEC_AUX_CLOCK),
		    sysAuxClkInt, NULL) == ERROR)
	{
	return (ERROR);
	}

    sysAuxClkRoutine   = routine;
    sysAuxClkArg       = arg;

    auxClkIsConnected  = TRUE;

    return (OK);
    }
/*******************************************************************************
*
* sysAuxClkDisconnect - clear the auxiliary clock routine
* 
* This routine disables the auxiliary clock interrupt and disconnects
* the routine currently connected to the auxiliary clock interrupt.
*
* SEE ALSO: sysAuxClkConnect (2)
*/

VOID sysAuxClkDisconnect ()

    {
    /* disable the auxiliary clock interrupt */
     
    sysAuxClkDisable ();
 
    /* connect the dummy routine, just in case */
 
    sysAuxClkConnect (logMsg, (int)"auxiliary clock interrupt\n");
    }
/********************************************************************************
* sysAuxClkDisable - turn off auxiliary clock interrupts
*/
 
VOID sysAuxClkDisable ()
 
    {
    if (auxClkIsRunning)
	{
	/* stop timer */
     
	*DUSCC_CCRB(FRC31_DUSCC1_BASE_ADRS) = DUSCC_CCR_CT_STOP;

	auxClkIsRunning = FALSE;
	}
    }
/********************************************************************************
* sysAuxClkEnable - turn auxiliary clock interrupts on
*/
 
VOID sysAuxClkEnable ()
 
    {
    FAST int tc;

    /* calculate the divide ratio, and write it to the timer chip */

    tc = AUX_CLOCK_FREQ / auxClkTicksPerSecond;

    *DUSCC_CTPRHB(FRC31_DUSCC1_BASE_ADRS) = MSB(tc);
    *DUSCC_CTPRLB(FRC31_DUSCC1_BASE_ADRS) = LSB(tc);

    /* enable and start clock interrupts */

    *DUSCC_CTCRB(FRC31_DUSCC1_BASE_ADRS)  = DUSCC_CTCR_ZERO_DET_INTR_ENABLE |
					    DUSCC_CTCR_CLK_X1_CLK;
    *DUSCC_CCRB(FRC31_DUSCC1_BASE_ADRS)	 = DUSCC_CCR_CT_START;

    auxClkIsRunning = TRUE;
    }
/*******************************************************************************
*
* sysAuxClkRateGet - get the auxiliary timer frequency
*
* This routine finds out the auxiliary clock speed.
*
* RETURNS: number of ticks per second of the auxiliary clock
*
* SEE ALSO: sysAuxClkRateSet (2)
*/

int sysAuxClkRateGet ()

    {
    return (auxClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysAuxClkRateSet - set rate of auxiliary clock
*
* This routine sets the clock rate of the auxiliary clock.
* Auxiliary clock interrupts are not enabled.
*
* SEE ALSO: sysAuxClkRateGet (2)
*/

VOID sysAuxClkRateSet (ticksPerSecond)
    FAST int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
        auxClkTicksPerSecond = ticksPerSecond;
     
    if (auxClkIsRunning)
        {
        sysAuxClkDisable ();
        sysAuxClkEnable ();
	}
    }

/*******************************************************************************
*
* sysLocalToBusAdrs - convert local address to bus address
*
* Given a local memory address, this routine returns the VME address
* that would have to be accessed to get to that byte.
*
*
* RETURNS: OK or ERROR if mapping not possible.
*
* SEE ALSO: sysBusToLocalAdrs (2)
*
* ARGSUSED
*/

STATUS sysLocalToBusAdrs (adrsSpace, localAdrs, pBusAdrs)
    int adrsSpace;	/* bus address space in which busAdrs resides */
    char *localAdrs;	/* local address to convert */
    char **pBusAdrs;	/* where to return bus address */
 
    {
    if ((localAdrs < LOCAL_MEM_LOCAL_ADRS) || (localAdrs >= sysMemTop ()))
	{
	/* this is off-board memory - just return local address */

	*pBusAdrs = localAdrs;
	return (OK);
	}
    else
	{
	/* this is on-board memory - map to bus address space;
	 *   the following memory mapping is established in sysProcNumSet():
	 *   - only processor 0 has memory on bus,
	 *   - the memory is placed in EXT space at
	 *      address LOCAL_MEM_BUS_ADRS.
	 */

	switch (adrsSpace)
	    {
	    case VME_AM_SUP_SHORT_IO:
	    case VME_AM_USR_SHORT_IO:
		return (ERROR);			/* no A16 access to RAM */

	    case VME_AM_STD_SUP_PGM:
	    case VME_AM_STD_SUP_DATA:
	    case VME_AM_STD_USR_PGM:
	    case VME_AM_STD_USR_DATA:
		return (ERROR);			/* no A24 access to RAM */

	    case VME_AM_EXT_SUP_PGM:
	    case VME_AM_EXT_SUP_DATA:
	    case VME_AM_EXT_USR_PGM:
	    case VME_AM_EXT_USR_DATA:
		*pBusAdrs = localAdrs + LOCAL_MEM_BUS_ADRS;
		return (OK);

	    default:
		return (ERROR);
	    }
	}
    }
/*******************************************************************************
*
* sysBusToLocalAdrs - convert bus address to local address
*
* Given a VME memory address, this routine returns the local address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK or ERROR if unknown address space
*
* SEE ALSO: sysLocalToBusAdrs (2)
*
* ARGSUSED
*/

STATUS sysBusToLocalAdrs (adrsSpace, busAdrs, pLocalAdrs)
    int adrsSpace;	/* bus address space in which busAdrs resides */
    char *busAdrs;	/* bus address to convert */
    char **pLocalAdrs;	/* where to return local address */

    {
    switch (adrsSpace)
        {
        case VME_AM_SUP_SHORT_IO:
        case VME_AM_USR_SHORT_IO:
            *pLocalAdrs = (char *) (0xfcff0000 |
			            (0x0000ffff & (unsigned int) busAdrs));
            return (OK);
 
        case VME_AM_STD_SUP_ASCENDING:
        case VME_AM_STD_SUP_PGM:
        case VME_AM_STD_SUP_DATA:
        case VME_AM_STD_USR_ASCENDING:
        case VME_AM_STD_USR_PGM:
        case VME_AM_STD_USR_DATA:
	    if ((unsigned int) busAdrs > (unsigned int) 0x00ffffff)
		return (ERROR);
	    *pLocalAdrs = (char *) (0xfd000000 | (unsigned int) busAdrs);
	    return (OK);
 
        case VME_AM_EXT_SUP_ASCENDING:
        case VME_AM_EXT_SUP_PGM:
        case VME_AM_EXT_SUP_DATA:
        case VME_AM_EXT_USR_ASCENDING:
        case VME_AM_EXT_USR_PGM:
        case VME_AM_EXT_USR_DATA:
            if ((unsigned int) busAdrs < (unsigned int) sysMemTop () ||
		(unsigned int) busAdrs  > (unsigned int) 0xfaffffff)
		{
		return (ERROR);
		}
	    *pLocalAdrs = busAdrs;
	    return (OK);
 
        default:
            return (ERROR);
        }
    }
/*******************************************************************************
*
* sysIntDisable - disable bus interrupt level
*
* This routine disables the specified bus interrupt level.
*
* NOTE:
* Interrupt levels are disabled by masking interrupts on the FGA-002 chip.
*
* RETURNS: OK, or ERROR if intLevel not in range 1-7
*/

STATUS sysIntDisable (intLevel)
    int intLevel;	/* interrupt level to disable */

    {
    switch (intLevel)
	{
	case 1:
	    *FGA_ICRVME1(FRC31_FGA_BASE_ADRS) &= ~FGA_ICR_ENABLE;
	    break;
	case 2:
	    *FGA_ICRVME2(FRC31_FGA_BASE_ADRS) &= ~FGA_ICR_ENABLE;
	    break;
	case 3:
	    *FGA_ICRVME3(FRC31_FGA_BASE_ADRS) &= ~FGA_ICR_ENABLE;
	    break;
	case 4:
	    *FGA_ICRVME4(FRC31_FGA_BASE_ADRS) &= ~FGA_ICR_ENABLE;
	    break;
	case 5:
	    *FGA_ICRVME5(FRC31_FGA_BASE_ADRS) &= ~FGA_ICR_ENABLE;
	    break;
	case 6:
	    *FGA_ICRVME6(FRC31_FGA_BASE_ADRS) &= ~FGA_ICR_ENABLE;
	    break;
	case 7:
	    *FGA_ICRVME7(FRC31_FGA_BASE_ADRS) &= ~FGA_ICR_ENABLE;
	    break;
	default:
	    return (ERROR);
        }
    return (OK);
    }
/*******************************************************************************
*
* sysIntEnable - enable bus interrupt level
*
* This routine enables the specified bus interrupt level.
*
* NOTE:
* Interrupt levels are enable by unmasking interrupts on the FGA-002 chip.
*
* RETURNS: OK, or ERROR if intLevel not in range 1-7
*/

STATUS sysIntEnable (intLevel)
    int intLevel;	/* interrupt level to enable */

    {
    switch (intLevel)
	{
	case 1:
	    *FGA_ICRVME1(FRC31_FGA_BASE_ADRS) |= FGA_ICR_ENABLE;
	    break;
	case 2:
	    *FGA_ICRVME2(FRC31_FGA_BASE_ADRS) |= FGA_ICR_ENABLE;
	    break;
	case 3:
	    *FGA_ICRVME3(FRC31_FGA_BASE_ADRS) |= FGA_ICR_ENABLE;
	    break;
	case 4:
	    *FGA_ICRVME4(FRC31_FGA_BASE_ADRS) |= FGA_ICR_ENABLE;
	    break;
	case 5:
	    *FGA_ICRVME5(FRC31_FGA_BASE_ADRS) |= FGA_ICR_ENABLE;
	    break;
	case 6:
	    *FGA_ICRVME6(FRC31_FGA_BASE_ADRS) |= FGA_ICR_ENABLE;
	    break;
	case 7:
	    *FGA_ICRVME7(FRC31_FGA_BASE_ADRS) |= FGA_ICR_ENABLE;
	    break;
	default:
	    return (ERROR);
        }
    return (OK);
    }
/*******************************************************************************
*
* sysBusIntAck - acknowledge interrupt
*
* This routine acknowledges a specific interrupt.
*
* The FGA-002 acknowledges the interrupt for us so this routine has no effect.
*
* RETURNS: NULL
*
* ARGSUSED
*/

int sysBusIntAck (intLevel)
    int intLevel;

    {
    return (NULL);
    }
/*******************************************************************************
*
* sysBusIntGen - generate interrupt
*
* This routine generates a VME bus interrupt.
*
* NOTE:
* This board cannot generate a VME bus interrupt so this routine has no effect.
*
* RETURNS: ERROR as board is unable to generate a VME bus interrupt
*
* ARGSUSED
*/

STATUS sysBusIntGen (level, vector)
    int level;		/* VME bus interrupt level to generate (1-7) */
    int vector;		/* interrupt vector to generate (0-255) */

    {
    return (ERROR);
    }

/*******************************************************************************
*
* sysMailboxConnect - connect routine to the mailbox interrupt
*
* This routine connects the given function to the mailbox interrupt.
*
* RETURNS: OK
*
* SEE ALSO: intConnect (2)
*
* ARGSUSED
*/
 
STATUS sysMailboxConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each mailbox interrupt */
    int arg;		/* argument with which to call routine */
 
    {
    if (!sysMboxIsConnected &&
	intConnect (INUM_TO_IVEC (INT_VEC_MBOX), sysMboxInt, 0) == ERROR)
	{
	return (ERROR);
	}

    sysMboxRoutine = routine;
    sysMboxArg     = arg;

    sysMboxIsConnected = TRUE;

    return (OK);
    }
/*******************************************************************************
*
* sysMailboxEnable - enable mailbox interrupt
*
* This routine enables the first mailbox of the FGA-002.
*
* RETURNS: OK
*
* ARGSUSED
*/
 
STATUS sysMailboxEnable (mailboxAdrs)
    char *mailboxAdrs;

    {
    *FGA_ICRMBOX0(FRC31_FGA_BASE_ADRS) |= FGA_ICR_ENABLE;

    return (OK);
    }
/*******************************************************************************
*
* sysProcNumGet - get processor number
*
* This routine returns the processor number previously set with 
* sysProcNumSet (2).
*
* RETURNS: processor number
*/

int sysProcNumGet ()

    {
    return (sysProcNum);
    }
/*******************************************************************************
*
* sysProcNumSet - set processor number
*
* Set the processor number for this CPU.  Processor numbers should be
* unique on a single backplane.
*/

VOID sysProcNumSet (procNum)
    int procNum;

    {
    sysProcNum = procNum; /* set sysProcNum; FRC31_FGA_VME_ADRS depends on it */

    if (procNum == 0)
	{
	/* enable system controller for processor 0 unless
	 * SYSFLG_NO_SYS_CONTROLLER flag is set in sysFlags
	 */

	if ((sysFlags & SYSFLG_NO_SYS_CONTROLLER) == 0)
	    *FGA_CTL1(FRC31_FGA_BASE_ADRS) |= FGA_CTL1_SCON;     /* arbitrate */

	/* dual port memory; dual port address setup in sysHwInit() */

	*FGA_ENAMCODE(FRC31_FGA_BASE_ADRS)  = FGA_ENAMCODE_EXTUSRDAT_RW |
					      FGA_ENAMCODE_EXTSUPDAT_RW;
	}

    /* set VME short address space base for FGA-002; each proc w/ unique adrs */
    *FGA_MYVMEPAGE(FRC31_FGA_BASE_ADRS)	 = (FRC31_FGA_VME_ADRS >> 8);

    /* set FMB slot code */
    *FGA_FMBCTL(FRC31_FGA_BASE_ADRS)	|= (procNum & FGA_FMBCTL_SLOT_MASK);

    }
/*******************************************************************************
*
* sysBusTas - test and set across VME bus
*
* This routine does a 680x0 test-and-set instruction across the backplane.
*
* RETURNS: TRUE (successful set) or FALSE (failure)
*/

BOOL sysBusTas (addr)
    char *addr;

    {
    return (vxTas (addr));
    }

/* miscellaneous support routines */

/*******************************************************************************
*
* sysSysfailConnect - connect routine to the sysfail signal
*
* This routine connects the given function to the sysfail signal.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2)
*/

STATUS sysSysfailConnect (routine, arg)
    FUNCPTR routine;
    int arg;

    {
    *FGA_ICRSYSFAIL(FRC31_FGA_BASE_ADRS) |= FGA_ICR_ENABLE;

    return (intConnect (INUM_TO_IVEC (INT_VEC_SYSFAIL), routine, arg));
    }
/*******************************************************************************
*
* sysAcfailConnect - connect routine to the acfail signal
*
* This routine connects the given function to the acfail signal.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2)
*/
STATUS sysAcfailConnect (routine, arg)
    FUNCPTR routine;
    int arg;

    {
    *FGA_ICRACFAIL(FRC31_FGA_BASE_ADRS) |= FGA_ICR_ENABLE;

    return (intConnect (INUM_TO_IVEC (INT_VEC_ACFAIL), routine, arg));
    }
/*******************************************************************************
*
* sysFrontPanelSwitches - read front panel switches
*
* This routines returns the value of the 8 bit rotary switch on the front panel.
*/

int sysFrontPanelSwitches ()

    {
    return (*PIT_PAAR (FRC31_PIT1_BASE_ADRS));
    }
/*******************************************************************************
*
* sysPit1Int - interrupt level processing for the PIT1 chip 
*
* This routine handles the interrupts associated with the PIT1 chip.
* It is attached to the interrupt vector by sysClkConnect (2).
*/

LOCAL VOID sysPit1Int ()

    {
    if (*PIT_TSR (FRC31_PIT1_BASE_ADRS) & ZERO_DET_STATUS)
	{
	if (sysClkRoutine != NULL)
	    (* sysClkRoutine) (sysClkArg);	/* call system clock routine */

	*PIT_TSR (FRC31_PIT1_BASE_ADRS) = 1;	/* acknowledge timer int */
	}
    }
/*******************************************************************************
*
* sysMboxInt - interrupt level processing for the mailbox
*
* This routine handles the interrupts associated with the FGA-002 MBOX0.
* It is attached to the interrupt vector by sysMailboxConnect (2).
*/

LOCAL VOID sysMboxInt ()

    {
    if (sysMboxRoutine != NULL)
	(* sysMboxRoutine) (sysMboxArg);	/* call mailbox routine */

    *FGA_MBOX0(FRC31_FGA_BASE_ADRS) = 0x0;	/* acknowledge interrupt */
    }
/*******************************************************************************
*
* sysAuxClkInt - interrupt level processing for auxiliary clock
*
* This routine handles the auxiliary clock interrupt.  It is attached to the
* clock interrupt vector by the routine sysAuxClkConnect (2).
* The appropriate routine is called and the interrupt is acknowleged.
*/

LOCAL VOID sysAuxClkInt ()

    {
    if (sysAuxClkRoutine != NULL)
	(*sysAuxClkRoutine) (sysAuxClkArg);	/* call system clock routine */

    /* acknowledge interrupt */

    *DUSCC_ICTSRB(FRC31_DUSCC1_BASE_ADRS) = DUSCC_ICTSR_CT_ZERO_COUNT;
    }
/*******************************************************************************
*
* sysAbortInt - handle NMI from ABORT switch on front panel
*/

LOCAL VOID sysAbortInt ()

    {
    *FGA_ISABORT(FRC31_FGA_BASE_ADRS) = 0x0;	/* acknowledge interrupt */

    sysToMonitor (BOOT_WARM_NO_AUTOBOOT);
    }
/******************************************************************************
*
* sysNvRamGet - get contents out of non-volatile RAM
*
* Copies non-volatile memory into string.
* The string will be terminated with an EOS.
*/

VOID sysNvRamGet (string, strLen, offset)
    char *string;    /* where to copy non-volatile RAM           */
    int strLen;      /* maximum number of bytes to copy          */
    int offset;      /* (even) byte offset into non-volatile RAM */

    {
    bcopyBytes (NV_BOOT_LINE + offset, string, strLen);
    string [strLen] = EOS;

    if (string [0] == EOS)
	strcpy (string, DEFAULT_BOOT_LINE);
    }
/*******************************************************************************
*
* sysNvRamSet - write to non-volatile RAM
*
* Copy 'string' into non-volatile RAM.
*
* INTERNAL
* Note that although the NVRAM looks just like normal memory, it is
* a single byte wide chip and will break long words into four transfers.
* Thus there is no significant speed advantage to longs and bcopyBytes (2)
* is used instead just to be safe.  config.h defines the base
* of the boot line space in the NVRAM.
*/

VOID sysNvRamSet (string, strLen, offset)
    char *string;     /* string to be copied into non-volatile RAM */
    int strLen;       /* maximum number of bytes to copy           */
    int offset;       /* (even) byte offset into non-volatile RAM  */
    {
    bcopyBytes (string, NV_BOOT_LINE + offset, strLen);
    }
