/* sysLib.c - Force SYS68K/CPU-21/29/32 system dependent library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01r,26apr89,shl  changed "warm start" address for consistency with romInit.s.
01p,26jan89,tja  fixed sysAuxClkDisable.
01o,09sep88,gae  documentation.
01n,05aug88,dnw  made sysMemTop work as advertised now that enp init is fixed.
01m,01aug88,gae  changes for frc32. fixed aux clock. fixed sysInt{Dis,En}able.
		 added Abort switch support.
01l,26jul88,gae  fixed sysBusToLocal () and sysLocalToBus ().
		 fixed sysAuxClkDisable ().  Documentation.
01k,11jul88,gae  updated to Terry Arden's frc21/9 version.
	   +tja  ported CPU-29 into existing CPU-21 functions;
		 converted to m68153 macro usage
01j,08jun88,gae  Revised code/documentation for uniformity across targets.
		 sysClkConnect now returns STATUS.
		 added sysClk{Disable,Enable}, sysClkRateSet() doesn't enable.
		 added null routines sysAuxClk{Disable,Enable}.
		 fixed sysIntAck to return NULL.
		 added sysProcNumGet.
01h,27apr88,ecs  added sysExcMsg.
01f,30may88,dnw  changed to v4 names.
01e,27apr88,ecs  added sysExcMsg.
		 added include of ioLib.h.
01d,28feb88,dnw  moved clock int ack here instead of doing it in usrConfig.c.
01c,21oct87,tja  checked for ASCU2 board presence in sysSetProcNum
		 added access code to real-time clock on ASCU2
01b,28sep87,tja	 updated for vxWorks release 3.1
01a,29may87,tja  written by modifying 01k of hkv2f version.
*/

/*
DESCRIPTION
This library contains a set of routines to manipulate the primary functions
of the CPU board.  The goal is to provide a board-independant interface on
which VxWorks and application code can be built in a system-independant way.
Not every feature of every board is supported by this library; a particular
board may have various extensions to the capabilities described here.
Also not every board will support all the functions provided by this library.
And some boards provide some of the functions of this library with hardware
switches, jumpers, or PALs, instead of software controllable registers.

The funtions addressed here include:

    initialization functions:
	- initialize hardware to known state
	- identify the system

    memory/address space functions:
	- get on-board memory limit
	- map from local to bus and bus to local address spaces
	- enable/disable cache memory
	- set/get non-volatile RAM

    bus interrupt functions:
	- enable/disable bus interrupt levels
	- generate bus interrupts

    serial channel functions (see tyCoDrv):
	- enable/disable serial channel interrupts
	- set serial channel baud rates
	- get/put bytes from a serial channel

    clock/timer functions:
       - enable/disable timer interrupts
       - set timer periodic rate

    mailbox/location monitor functions:
       - enable mailbox/location monitor interrupts

HARDWARE VARIATIONS
Force makes at least three different boards that have been tested with
this port.  The SYS68K/CPU-21, CPU-29 and CPU-32.  Allegedly, the CPU-20
will work as well.  Of these, then four, different boards, about four
sub-classes of varying clock rates, memory size, etc. exist for each one.
In sysHwInit (2) the three different boards are discerned.  The CPU-21
is assumed to have only .5 M of memory.  The CPU-29 and CPU-32 have
a special bit pattern in their second BIM chip that provides an I.D..

AUTHOR
Original port of both Force SYS68K/CPU-21 and SYS68K/CPU-29 by
Terry Arden of MacMillan-Bloedel Research, B.C. Canada.
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "vme.h"
#include "ioLib.h"
#include "memLib.h"
#include "sysLib.h"
#include "config.h"

IMPORT char end;		/* end of system, created automatically by ld */

IMPORT VOID logMsg ();

/* globals */

int   sysBus      = BUS;		/* system bus type (VME_BUS, etc) */
int   sysCpu      = CPU;		/* system cpu type (MC680x0) */
char *sysBootLine = BOOT_LINE_ADRS;	/* address of boot line */
char *sysExcMsg   = EXC_MSG_ADRS;	/* catastrophic message area */
int   sysProcNum;			/* processor number of this cpu */
int   sysFlags;				/* boot flags */
char  sysBootHost[BOOT_FIELD_LEN];	/* name of host from which we booted */
char  sysBootFile[BOOT_FIELD_LEN];	/* name of file from which we booted */

/* locals */

LOCAL FUNCPTR sysClkRoutine     = NULL;	/* routine to call on clock interrupt */
LOCAL int sysClkArg             = NULL;	/* its argument */
LOCAL int sysClkTicksPerSecond  = 60;	/* system clock rate */
LOCAL BOOL sysClkIsConnected    = FALSE;/* hooked up to interrupt yet? */
LOCAL BOOL sysClkRunning        = FALSE;/* system clock enabled? */

LOCAL FUNCPTR sysAuxClkRoutine  = NULL;	/* routine to call on clock interrupt */
LOCAL int sysAuxClkArg          = NULL;	/* its argument */
LOCAL int auxClkTicksPerSecond	= 60;	/* auxiliary clock rate */
LOCAL BOOL auxClkIsConnected    = FALSE;/* hooked up to interrupt yet? */
LOCAL BOOL auxClkRunning        = FALSE;/* auxiliary clock enabled? */

LOCAL int sysTarget;			/* FRC_CPU_21, FRC_CPU_29, FRC_CPU_32 */

/* forward declarations */

LOCAL VOID sysClkInt ();
LOCAL VOID sysAuxClkInt ();
LOCAL VOID sysAbortInt ();


/*******************************************************************************
*
* sysModel - return model name of the system CPU
*
* This routine returns a pointer to a string with the model name of this CPU.
*
* RETURNS: pointer to string "Force SYS68K/CPU-21", or whatever
*/

char *sysModel ()

    {
    switch (sysTarget)
	{
	case FRC_CPU_21:
	    return ("Force SYS68K/CPU-21");

	case FRC_CPU_29:
	    return ("Force SYS68K/CPU-29");

	case FRC_CPU_32:
	    return ("Force SYS68K/CPU-32");

	default:
	    return ("Force SYS68K/UNKNOWN");
	}
    }
/**********************************************************************
*
* sysHwInit - initialize hardware
*
* This routine initializes the board hardware.
* It is normally called from usrInit (2) in usrConfig (1).
* This routine must be called in supervisor mode.
*
* First determine whether the target is a CPU-21, CPU-29, or CPU-32.
* The timers are initialized and turned off.
* The BIM(s) are setup.
*
* NOTE:
* The routine sysMemTop () is used to differentiate between CPUs.
*/

VOID sysHwInit ()

    {
    char zero = 0;

    /* if only a half megabyte of memory then must be a CPU-21 */

    if (sysMemTop () == (char *)0x80000)
	sysTarget = FRC_CPU_21;
    else 
	{
	/* read CPU type from bits 3-7:
	 *   9 => CPU-29
	 *  12 => CPU-32
	 */

	sysTarget = ((((*PIT_PBAR (PIT_BASE_2)) >> 3) & 0x1f) == 0x9) ?
		    FRC_CPU_29 : FRC_CPU_32;
	}

    /* setup vector registers on BIM chips */

    *BIM_VR1 (BIM_BASE_1) = INT_VEC_ABORT;	/* abort */
    *BIM_VR2 (BIM_BASE_1) = INT_VEC_CO_MPCC;	/* console MPCC */
    *BIM_VR3 (BIM_BASE_1) = INT_VEC_PIT1_TIMER;	/* PI/T 1 timer (CLOCK) */
    *BIM_VR4 (BIM_BASE_1) = INT_VEC_PIT1_PORT;	/* PI/T 1 port */

    if (sysTarget != FRC_CPU_21)
	{
	*RTC_CTR_E  = 0xff;			/* disable interrupts */
	*BIM_VR1 (BIM_BASE_2) = INT_VEC_RTC;	/* rtc */

	/* vector 2 not used; optional MPCC 2 interrupt source */

	*BIM_VR3 (BIM_BASE_2) = INT_VEC_PIT2_TIMER;	/* PI/T 2 timer (AUX) */
	*BIM_VR4 (BIM_BASE_2) = INT_VEC_PIT2_PORT;	/* PI/T 2 port */
	}

    /* configure control registers on BIMs for
     * flag auto-clear, interrupt enable, and interrupt level
     */
    *BIM_CR1 (BIM_BASE_1) = BIM_FAC|BIM_IRE|INT_LVL_ABORT;	  
    *BIM_CR2 (BIM_BASE_1) = BIM_FAC|BIM_IRE|INT_LVL_CO_MPCC;  
    *BIM_CR3 (BIM_BASE_1) = BIM_FAC|BIM_IRE|INT_LVL_PIT1_TIMER;
    *BIM_CR4 (BIM_BASE_1) = BIM_FAC|BIM_IRE|INT_LVL_PIT1_PORT; 

    if (sysTarget != FRC_CPU_21)
	{
	*BIM_CR1 (BIM_BASE_2) = BIM_FAC|BIM_IRE|INT_LVL_RTC;
	*BIM_CR3 (BIM_BASE_2) = BIM_FAC|BIM_IRE|INT_LVL_PIT2_TIMER;
	*BIM_CR4 (BIM_BASE_2) = BIM_FAC|BIM_IRE|INT_LVL_PIT2_PORT;
	}

    /* initialize PI/T ports */

    *PIT_TCR (PIT_BASE_1)   = 0xe0;	/* disable timer interrupt */
    *PIT_PGCR (PIT_BASE_1)  = 0x30;	/* h4,h3,h2,h1 ints enabled pmode = 0 */
    *PIT_PSRR (PIT_BASE_1)  = 0x08;	/* pc5=pirq, pc6=port C */
    *PIT_PACR (PIT_BASE_1)  = 0x80;	/* submode 1X */
    *PIT_PADDR (PIT_BASE_1) = zero;	/* all inputs */
    *PIT_PBCR (PIT_BASE_1)  = 0x84;	/* submode 1X, H4 interrupt enabled */

/* XXX see below
    *PIT_PBDR (PIT_BASE_1)  = 0x01;	/* disable all VME ints,ROBCLR enabled*/
    *PIT_PBDR (PIT_BASE_1)  = 0xff;	/* enable all VME ints,ROBCLR enabled */

    *PIT_PBDDR (PIT_BASE_1) = 0xff;	/* all outputs */
    *PIT_PCDR (PIT_BASE_1)  = 0x07;	/* arbiter is RAT, ROR */
    *PIT_PCDDR (PIT_BASE_1) = 0x87;	/* bits 0-2 arbiter func, bit 7 mem32 */

    if (sysTarget != FRC_CPU_21)
	{
	*PIT_TCR (PIT_BASE_2)   = 0xe0; /* disable timer interrupt */
	*PIT_PGCR (PIT_BASE_2)  = zero; /* pmode = 0 */
	*PIT_PSRR (PIT_BASE_2)  = 0x08; /* pc5=pirq, pc6=port C */
	*PIT_PACR (PIT_BASE_2)  = 0xc0; /* submode 1X */
	*PIT_PADDR (PIT_BASE_2) = zero; /* all inputs */

	if (sysTarget == FRC_CPU_29)
	    *PIT_PBCR (PIT_BASE_2)  = 0xc0; /* submode 1X, H4 is input, high */
	else
	    *PIT_PBCR (PIT_BASE_2)  = 0xf8; /* submode 1X, H4 is output, low */

	*PIT_PBDR (PIT_BASE_2)  = zero; /* inputs not used */
	*PIT_PCDR (PIT_BASE_2)  = 0xc0; /* set space0/space1 to 1 */

	if (sysTarget == FRC_CPU_29)
	    *PIT_PCDDR (PIT_BASE_2) = 0xc7; /* drive pc0-pc2, pc6-pc7 */
	else
	    *PIT_PCDDR (PIT_BASE_2) = zero; /* drive pc4, pc6-pc7 */
	}

    /* XXX
     * we've enabled all interrupts 'cuz, not having a CPU-21 manual, it
     * appears that some things may interrupt on different levels on the CPU-29.
     * It would be a whole lot better if, in the appropriate place, the
     * following was done:
     *
     * sysIntEnable (INT_LVL_ABORT);
     * sysIntEnable (INT_LVL_CO_MPCC);
     * ...
     */
    }
/*******************************************************************************
*
* sysMemTop - get top of memory address
*
* This routine returns the address of the first missing byte of memory.
*
* It starts probing at the end of bss, then tries to read a byte
* at every 4K boundary until it finds one that can't be read.
* This routine must be called in supervisor mode, the first time only.
* After that, it can be called in user mode as well.
*
* RETURNS: address of the first missing byte of memory
*/

char *sysMemTop ()

    {
#ifdef	NO_PROBE
    /* assume 1 Meg. board ... use when probing not safe */
    static char *memTop = 0x100000;	/* top of memory address */
#else
    static char *memTop = NULL;		/* top of memory address */
#endif	NO_PROBE
    char bitBucket;

#define	PAGE_SIZE	0x1000

    if (memTop == NULL)
	{
	/* Look for the top of memory starting at the first even page
	 * boundary after _end. */

	memTop = (char *) ((((int) &end) + (PAGE_SIZE - 1)) &
			    (~ (PAGE_SIZE - 1)));

	/* search a possible 4 Mbytes */

	while ((vxMemProbe (memTop, READ, 1, &bitBucket) == OK) &&
	       (memTop < (char *) 0x400000))
	    {
	    memTop += PAGE_SIZE;
	    }

	/* memTop is now pointing to the first non-existent address */
	}

    return (memTop);
    }
/*******************************************************************************
*
* sysToMonitor - transfer to rom monitor
*
* This routine transfers control to the rom monitor.  It is usually called
* only by the routine reboot, which services control-X, and bus errors at
* interrupt level.  In special circumstances, however, the user may wish
* to introduce a new startType such that a special bootrom facility would be
* enabled.
*
* RETURNS: OK (if we ever continue from the rom monitor)
*
* INTERNAL
* Note that the "WARM" restart address is at:
*    ROM_BASE_ADRS + 38: (8 bytes for vectors + 30 bytes for cold start code)
*/
 
STATUS sysToMonitor (startType)
    int startType;  /* parameter is passed to ROM to tell it how to boot, */
                    /* possible types are defined in sysLib.h             */

    {
    (* ((FUNCPTR) ((int)ROM_BASE_ADRS + 38))) (startType);

    return (OK);    /* in case we ever continue from rom monitor */
    }

/*******************************************************************************
*
* sysClkConnect - connect routine to system clock interrupt
*
* This routine connects the given function to the system clock interrupt.
* System clock interrupts are not enabled.
* It is normally called from usrRoot (2) in usrConfig (1) to connect
* usrClock (2) to the system clock interrupt.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), usrClock (2)
*/

STATUS sysClkConnect (routine, arg)
    FUNCPTR routine;
    int arg;

    {
    if (!sysClkIsConnected &&
	intConnect (INUM_TO_IVEC (INT_VEC_CLOCK), sysClkInt, 0) == ERROR)
	{
	return (ERROR);
	}

    sysClkRoutine = routine;
    sysClkArg     = arg;

    sysClkIsConnected = TRUE;


    /* connect abort switch (this is as good as place as any) */

    (void) intConnect (INUM_TO_IVEC(INT_VEC_ABORT), sysAbortInt, 0);

    return (OK);
    }
/*******************************************************************************
*
* sysAbortInt - handle interrupt from ABORT switch on front panel
*/

LOCAL VOID sysAbortInt ()

    {
    while (*PIT_PCDR (PIT_BASE_1) & 0x10 == 0);	/* while switch pressed */

    sysToMonitor (BOOT_WARM_NO_AUTOBOOT);
    }
/*******************************************************************************
*
* sysClkDisable - turn off system clock interrupts
*/

VOID sysClkDisable ()

    {
    if (sysClkRunning)
	{
	/* disable the PI/T Timer */

	*PIT_TCR (PIT_BASE_1) = 0xe0;

	sysClkRunning = FALSE;
	}
    }
/*******************************************************************************
*
* sysClkEnable - turn system clock interrupts on
*/

VOID sysClkEnable ()

    {
    unsigned int tc;

    /* A 5-bit prescaler (divide-by-32) is assumed to be used
     * on the PI/T clock input.
     */

    tc = CLOCK_FREQ / (sysClkTicksPerSecond * 32);

    /* write the timer value */

    *PIT_CPRL (PIT_BASE_1) = tc;
    *PIT_CPRM (PIT_BASE_1) = tc >> 8;
    *PIT_CPRH (PIT_BASE_1) = tc >> 16;

    /* enable the PI/T Timer */

    *PIT_TCR (PIT_BASE_1) = 0xe1;

    sysClkRunning = TRUE;
    }
/*******************************************************************************
*
* sysClkInt - clock interrupt handler
*
* This routine handles the clock interrupt.  It is attached to the clock
* interrupt vector by the routine sysClkConnect (2).
* The appropriate routine is called and the interrupts are acknowleged.
*/

LOCAL VOID sysClkInt ()

    {
    if (sysClkRoutine != NULL)
	(* sysClkRoutine) (sysClkArg);	/* call system clock routine */

    *PIT_TSR (PIT_BASE_1) = 1;		/* acknowledge timer interrupt */
    }
/*******************************************************************************
*
* sysClkRateGet - get rate of system clock
*
* This routine is used to find out the system clock speed.
*
* RETURNS: number of ticks per second of the system clock
*
* SEE ALSO: sysClkRateSet (2)
*/

int sysClkRateGet ()
    
    {
    return (sysClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysClkRateSet - set rate of system clock
*
* This routine sets the clock rate of the system clock.
* System clock interrupts are not enabled.
* It is normally called by usrRoot (2) in usrConfig (1).
*
* SEE ALSO: sysClkRateGet (2), sysClkEnable (2)
*/

VOID sysClkRateSet (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
	sysClkTicksPerSecond = ticksPerSecond;
	
    if (sysClkRunning)
	{
	sysClkDisable ();
	sysClkEnable ();
	}
    }

/*******************************************************************************
*
* sysAuxClkConnect - connect a routine to the auxiliary clock interrupt
*
* This routine connects a user routine to the auxiliary clock interrupt.
* Auxiliary clock interrupts are not enabled.
*
* NOTE:
* There is no auxiliary clock on the CPU-21.
*
* RETURNS: OK or ERROR if there is no auxiliary clock
*
* ARGSUSED
*/

STATUS sysAuxClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each auxiliary clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    if (sysTarget == FRC_CPU_21)
	return (ERROR);

    if (!auxClkIsConnected &&
	intConnect (INUM_TO_IVEC (INT_VEC_PIT2_TIMER), sysAuxClkInt,0) == ERROR)
	{
	return (ERROR);
	}

    sysAuxClkRoutine  = routine;
    sysAuxClkArg      = arg;

    auxClkIsConnected = TRUE;
    return (OK);
    }
/*******************************************************************************
*
* sysAuxClkDisconnect - disconnect a routine from the auxiliary clock interrupt
*
* This routine disables the auxiliary clock interrupt and disconnects
* the routine currently connected to the auxiliary clock interrupt.
*
* NOTE:
* There is no auxiliary clock on the CPU-21.
*/

VOID sysAuxClkDisconnect ()

    {
    if (sysTarget == FRC_CPU_21)
	return;

    /* disable the auxiliary clock interrupt */
 
    sysAuxClkDisable ();
      
    /* connect dummy routine, just in case */

    sysAuxClkConnect (logMsg, (int) "auxiliary clock interrupt\n");
    }
/*******************************************************************************
*
* sysAuxClkDisable - turn off auxiliary clock interrupts
*
* NOTE:
* There is no auxiliary clock on the CPU-21.
*/

VOID sysAuxClkDisable ()

    {
    if (sysTarget == FRC_CPU_21)
	return;

    if (auxClkRunning)
	{
	/* disable the PI/T 2 Timer */

	*PIT_TCR (PIT_BASE_2) = 0xe0;

	auxClkRunning = FALSE;
	}
    }
/*******************************************************************************
*
* sysAuxClkEnable - turn auxiliary clock interrupts on
*
* NOTE:
* There is no auxiliary clock on the CPU-21.
*/

VOID sysAuxClkEnable ()

    {
    unsigned int tc;

    if (sysTarget == FRC_CPU_21)
	return;

    /* A 5-bit prescaler (divide-by-32) is assumed to be used
     * on the PI/T clock input.
     */

    tc = CLOCK_FREQ / (auxClkTicksPerSecond * 32);

    /* write the timer value */

    *PIT_CPRL (PIT_BASE_2) = tc;
    *PIT_CPRM (PIT_BASE_2) = tc >> 8;
    *PIT_CPRH (PIT_BASE_2) = tc >> 16;

    /* enable the PI/T Timer */

    *PIT_TCR (PIT_BASE_2) = 0xe1;

    auxClkRunning = TRUE;
    }
/*******************************************************************************
*
* sysAuxClkInt - auxiliary clock interrupt handler
*
* This routine handles the auxiliary clock interrupt.
* It is attached to the auxiliary clock
* interrupt vector by the routine sysAuxClkConnect (2).
* The appropriate routine is called and the interrupts are acknowleged.
*
* NOTE FRC21:
* There is no auxiliary clock on the CPU-21.
* This routine is not used.
*/

LOCAL VOID sysAuxClkInt ()

    {
    if (sysAuxClkRoutine != NULL)
	(* sysAuxClkRoutine) (sysAuxClkArg); /* call auxiliary clock routine */

    *PIT_TSR (PIT_BASE_2) = 1;	/* acknowledge auxiliary timer interrupt */
    }
/*******************************************************************************
*
* sysAuxClkRateGet - get rate of auxiliary clock
*
* Finds out the auxiliary clock speed.
*
* NOTE FRC21:
* There is no auxiliary clock on the CPU-21.
*
* RETURNS: number of ticks per second of the auxiliary clock
*
* SEE ALSO: sysAuxClkRateSet (2)
*/

int sysAuxClkRateGet ()
    
    {
    return (auxClkTicksPerSecond);
    }
/**********************************************************************
*
* sysAuxClkRateSet - set rate of auxiliary clock
*
* This routine sets the clock rate of the auxiliary clock.
* Auxiliary clock interrupts are not enabled.
*
* NOTE FRC21:
* There is no auxiliary clock on the CPU-21.
* This routine has no effect.
*
* SEE ALSO: sysAuxClkConnect (2), sysAuxClkRateGet (2)
*
* ARGSUSED
*/

VOID sysAuxClkRateSet (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {                                                                   
    if (ticksPerSecond > 0)
	auxClkTicksPerSecond = ticksPerSecond;

    if (auxClkRunning)
	{
	sysAuxClkDisable ();
	sysAuxClkEnable ();
	}
    }

/*******************************************************************************
*
* sysLocalToBusAdrs - convert local address to bus address
*
* Given a local memory address, this routine returns the VME address
* that would have to be accessed to get to that byte.
*
* RETURNS: ERROR since the board is not dual-ported
*
* SEE ALSO: sysBusToLocalAdrs (2)
*
* ARGSUSED
*/

STATUS sysLocalToBusAdrs (adrsSpace, localAdrs, pBusAdrs)
    int adrsSpace;	/* bus address space in which busAdrs resides */
    char *localAdrs;	/* local address to convert */
    char **pBusAdrs;	/* where to return bus address */
 
    {
    return (ERROR);
    }
/*******************************************************************************
*
* sysBusToLocalAdrs - convert bus address to local address
*
* Given a VME memory address, this routine returns the local address
* that would have to be accessed to get to that byte.
*
* NOTE:
* May not be accurate for the CPU-21.
*
* RETURNS: OK or ERROR if unknown address space
*
* SEE ALSO: sysLocalToBusAdrs (2)
*
* ARGSUSED
*/

STATUS sysBusToLocalAdrs (adrsSpace, busAdrs, pLocalAdrs)
    int adrsSpace;	/* bus address space in which busAdrs resides */
    char *busAdrs;	/* bus address to convert */
    char **pLocalAdrs;	/* where to return local address */

    {
    switch (adrsSpace)
        {
        case VME_AM_SUP_SHORT_IO:
        case VME_AM_USR_SHORT_IO:
            *pLocalAdrs = (char *) (0xfbff0000 | 
			 (0x0000ffff & (unsigned int) busAdrs));
            return (OK);
 
        case VME_AM_STD_SUP_ASCENDING:
        case VME_AM_STD_SUP_PGM:
        case VME_AM_STD_SUP_DATA:
        case VME_AM_STD_USR_ASCENDING:
        case VME_AM_STD_USR_PGM:
        case VME_AM_STD_USR_DATA:
	    if ((unsigned int) busAdrs > (unsigned int) 0x00ffffff)
		return (ERROR);
	    *pLocalAdrs = (char *) (0xfd000000 | (unsigned int) busAdrs);
	    return (OK);
 
        case VME_AM_EXT_SUP_ASCENDING:
        case VME_AM_EXT_SUP_PGM:
        case VME_AM_EXT_SUP_DATA:
        case VME_AM_EXT_USR_ASCENDING:
        case VME_AM_EXT_USR_PGM:
        case VME_AM_EXT_USR_DATA:
            if ((unsigned int) busAdrs < (unsigned int) sysMemTop () ||
		(unsigned int) busAdrs  > (unsigned int) 0xfaffffff)
		{
		return (ERROR);
		}
	    *pLocalAdrs = busAdrs;
	    return (OK);
 
        default:
            return (ERROR);
        }
    }
/*******************************************************************************
*
* sysIntDisable - disable interrupt level
*
* This routine disables the specified interrupt level.
*
* NOTE:
* Interrupt levels are disabled by writing to port B of the PI/T.
* To disable IRQn clear bit PBn where 1<=n<=7.
*
* RETURNS: OK, or ERROR if intLevel not in range 1-7
*/

STATUS sysIntDisable (intLevel)
    int intLevel;	/* interrupt level to disable */

    {
    if ((intLevel < 1) || (intLevel > 7))
	return (ERROR);

    *PIT_PBDR (PIT_BASE_1) &= ~(1 << intLevel);
		
    return (OK);
    }
/*******************************************************************************
*
* sysIntEnable - enable interrupt level
*
* This routine enables the specified interrupt level.
*
* NOTE:
* Interrupt levels are enabled by writing to port B of the PI/T.
* To enable IRQn, set bit PBn where 1<=n<=7.
*
* RETURNS: OK, or ERROR if intLevel not in range 1-7
*/

STATUS sysIntEnable (intLevel)
    int intLevel;	/* interrupt level to enable */

    {
    if ((intLevel < 1) || (intLevel > 7))
	return (ERROR);

    *PIT_PBDR (PIT_BASE_1) |= (1 << intLevel);

    return (OK);
    }
/*******************************************************************************
*
* sysBusIntAck - acknowledge interrupt
*
* This routine acknowledges a a specific interrupt.
*
* NOTE FRC21:
* The BIM controller supports an automatic IACKIN/IACKOUT daisy chain.
* On some boards this routine returns the vector put on
* bus by interrupting device.
* This routine has no effect.
*
* RETURNS: NULL
*
* ARGSUSED
*/

int sysBusIntAck (intLevel)
    int intLevel;

    {
    return (NULL);
    }
/*******************************************************************************
*
* sysBusIntGen - generate interrupt
*
* This routine generates a VME bus interrupt.
*
* NOTE:
* These boards cannot generate a VME bus interrupt.
* This routine has no effect.
*
* RETURNS: ERROR as board is unable to generate a VME bus interrupt
*
* ARGSUSED
*/

STATUS sysBusIntGen (level, vector)
    int level;		/* VME bus interrupt level to generate (1-7) */
    int vector;		/* interrupt vector to generate (0-255) */

    {
    return (ERROR);
    }

/*******************************************************************************
*
* sysMailboxConnect - connect routine to the mailbox interrupt
*
* This routine connects the given function to the mailbox interrupt.
*
* RETURNS: ERROR, as there are no mailbox facility
*
* SEE ALSO: intConnect (2)
*
* ARGSUSED
*/
 
STATUS sysMailboxConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each mailbox interrupt */
    int arg;		/* argument with which to call routine */
 
    {
    return (ERROR);
    }
/*******************************************************************************
*
* sysMailboxEnable - enable mailbox interrupt
*
* This routine enables the mailbox interrupt.
*
* RETURNS: ERROR, as there are no mailbox facility
*
* ARGSUSED
*/
 
STATUS sysMailboxEnable (mailboxAdrs)
    char *mailboxAdrs;

    {
    return (ERROR);
    }
/*******************************************************************************
*
* sysProcNumGet - get processor number
*
* This routine returns the processor number previously set with 
* sysProcNumSet (2).
*
* RETURNS: processor number
*/

int sysProcNumGet ()

    {
    return (sysProcNum);
    }
/*******************************************************************************
*
* sysProcNumSet - set processor number
*
* Set the processor number for this CPU.  Processor numbers should be
* unique on a single backplane.
*/

VOID sysProcNumSet (procNum)
    int procNum;

    {
    sysProcNum = procNum;
    }
/*******************************************************************************
*
* sysBusTas - test and set across VME bus
*
* This routine does a 680x0 test-and-set instruction across the backplane.
*
* RETURNS: TRUE (successful set) or FALSE (failure)
*/

BOOL sysBusTas (addr)
    char *addr;

    {
    return (vxTas (addr));
    }

/* miscellaneous support routines */

/*******************************************************************************
*
* sysFrontPanelSwitches - read front panel switches
*
* This routines returns the value of the 8 bit dip switch on the front panel.
*/

int sysFrontPanelSwitches ()

    {
    return (*PIT_PAAR (PIT_BASE_1));
    }
