/* vrtxALib.s - VxWorks/VRTX kernel assembly library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
	.data
	.globl	_copyright_wind_river
	.long	_copyright_wind_river

/*
modification history 
--------------------
06h,24aug88,gae  documentation.
06g,12aug88,jcf  changed to match new intLib.c.
06f,22jun88,dnw  name tweaks.
06e,05jun88,dnw  changed vxCurrentTask to taskIdCurrent.
		 changed to set vrtx module number in error status values.
06d,30may88,dnw  changed to v4 names.
06c,09mar88,ecs  VRTX/32: added vrtxQjam, vrtxQEcreate, vrtxFcreate,
		    vrtxFdelete, vrtxFpost, vrtxFpend, vrtxFclear,
		    vrtxFinquiry, vrtxScreate, vrtxSdelete, vrtxSpost,
		    vrtxSpend, and vrtxSinquiry.
		 changed names of routines from 'vx...' to 'vrtx...'.
06b,19mar88,dnw  changed movb -> moveb for motorola assembler compatibility.
06a,19jan88,jcf  reworked for version 4.
05w,17feb88,ecs  removed is68020, fltused, and f68881_used to hostALib.s.
05v,13feb88,dnw  added .data before .asciz above, for Intermetrics assembler.
05u,06jan88,rdc  modifications for new vxMyTaskId.
05t,05jan88,dpg  fixed vxPcreate to pass 32 bit block size.
05s,16dec87,jlf  made vxQaccept save and restore d2, to fix bug.
05r,15dec87,gae  documentation.
05q,04dec87,rdc  changed flt68881_used to f68881_used.
05p,16nov87,rdc  added flt68881_used for sun.
05o,04nov87,ecs  documentation.
05n,22oct87,jcf  fixed bad comment
05m,21oct87,jcf  changed traps to vrtx into jsr to _sysKernelTrap.
		 fixed bugs in vxTcreateHook. 
		  don't allow createHook routines to be done.
		 fixed bug in vxTaskEntry to add 40 to sp instead of 44.
		 changed name of vxTdelete to vrtxTdelete so only we call it.
		 deleted deleteHook; delete routines are done in vxTdelete.
05n,06aug87,gae  moved floating-point coproccessor routines to fppALib.s.
05m,19may87,gae  added floating-point coproccessor support routines:
		   vxFpProbeTrap(), vxFpProbeSup(), vxFpSave(), vxFpRestore(),
		   and special type conversion routines dtodx(), dxtod().
05l,02apr87,jlf  changed tas() to vxTas().
05k,27mar87,dnw  changed ERROR to -1 because some compilers complain.
		 moved is68020 flag here from sysALib.s.
		 documentation.
05j,27mar87,rdc  moved tas() here from unixALib.s
05i,25mar87,jlf  added stuff for vxAddTaskCreateRtn
		 added fltused (moved here from vxInit.s)
		 documentation
05h,25feb87,dnw  added vxMemTrap() and vxMemProbeSup().
		 added vxQinquiry.
05g,21dec86,dnw  changed to not get include files from default directories.
05f,01nov86,dnw  Changed "moveml" instructions to use Motorola style register
		   lists, which are now handled by "aspp".
		 Changed "mov[bwl]" to "move[bwl]" for compatiblity w/Sun as.
05e,04sep86,jlf  documentation.
05d,20jun86,rdc  added vxTdeleteHook.
05c,04apr86,dnw  added vxVrtxInit and vxVrtxGo.
		 fixed routines that didn't save/restore enough registers:
		   vxTpriority(), vxTinquiry(), vxPcreate(), vxPextend(),
		   vxQpost(), vxQpend(), vxQcreate(), vxTXcreate().
05b,08mar86,rdc  added conditional compiles for 68020
05a,08feb86,dnw  changed name from vxLib.s to vxALib.s.
		 changed vxTScreate to vxTXcreate with add'l arg for tcbx.
		 deleted obsolete/unused routines: vxTcreate, vxGetC, vxPutC, 
		   vxWaitC, vxRxchr, vxIpost, and vxIQpost
		 added vxTcreateHook, vxTswitchHook, and vxTaskEntry, previously
		   in vxsLib and spnALib.
04e,03oct85,ecs  changed vxPcreate parameter 'id' from a short to a long.
		 changed vxQcreate parameter 'count' from a TINY to a short.
04d,21jul85,jlf  documentation.
04c,24jun85,rdc  conditional compile for system V which requires
		    truncated symbols. 
04b,19jun85,rdc  changed .globls for 4.2 as.
04a.22may85,jlf  translated from asm to as.
03a,08apr85,rdc  modifications for vrtx version3:  new calling format,
		    delay and timeout prms are long, trashed UIPOST and UIQPOST.
02c,07sep84,jlf  added copyright notice, and some comments.
02b,14aug84,dnw  fixed errors in documentation for vxTinquiry.
02a,08aug84,ecs  changed nonzero vrtx return codes to cause a call to
		    setStatus, and return ERROR. (All except vxGtime & vxGetc)
01i,26jun84,ecs  added LINK & UNLK instructions to make these functions
		    look like C to the stacktrace program.
01h,18may84,dnw  removd non-vrtx stuff.  now in system dependent library.
		    added vxTScreate and modified vxTcreate, for stack mods.
01g,03aug83,dnw  added vxProbe.
01f,22jul83,ecs  added vxIntDrop and vxUbug.
01e,01jul83,jlf  changed vxSetInt to not lock out, and to always do a UI_EXIT.
01d,24jun83,dnw  added lock-out of interrupts in vxSetInt to avoid losing
		    UI_EXITs requested by interrupt handlers.
01c,01jun83,ecs  added _vxSetIn & vxSetInt, _vxIntLo & _vxIntUn
01b,15mar83,ecs  save only necessary regs
01a,22feb83,ecs  written
*/

/*
DESCRIPTION
This library contains support for VRTX facilities.

C callable interface routines are provided for most VRTX system calls.  
Each interface routine gets the parameters passed to it in a normal 
C fashion (on the stack) and puts them in the proper registers where 
VRTX expects to find them, then traps to VRTX.

RETURNING VRTX STATUS CODES
If a VRTX system call returns an error status 
(non-zero in d0 on return from VRTX),
the interface routines set the task error status to the VRTX error code
returned, then return the value ERROR to the calling routine.  That routine
can find out the error code that VRTX returned by calling errnoGet ().
If all went well with VRTX, the interface routines return OK.

TASK CREATION
No direct access is provided to the VRTX task creation system calls.
Instead all VxWorks tasks should be created by the taskSpawn (2) or
taskCreate (2) calls in taskLib (1).

CHARACTER I/O
VxWorks does not use or support the character I/O facility
provided by the VRTX system calls SC_GETC, SC_PUTC, SC_WAITC,
UI_RXCHR, and UI_TXRDY.  Instead, VxWorks supplies a complete
device independent I/O system and a set of device drivers.

NOTE
Many routines in this module "link" and "unlk" the C frame pointer
(a6) although they don't use it in any way.  This is only for the benefit of
the stack trace facility, to allow it to properly trace tasks executing within
these routines.
*/

#define ASMLANGUAGE
#include "vxWorks.h"
#include "asm.h"

	.text
	.even

	/* internals */

	.globl _intEnt
	.globl _intExit
	.globl _vrtxAccept
	.globl _vrtxGblock
	.globl _vrtxGtime
	.globl _vrtxLock
	.globl _vrtxPcreate
	.globl _vrtxPend
	.globl _vrtxPextend
	.globl _vrtxPost
	.globl _vrtxQaccept
	.globl _vrtxQcreate
	.globl _vrtxQinquiry
	.globl _vrtxQpend
	.globl _vrtxQpost
	.globl _vrtxRblock
	.globl _vrtxStime
	.globl _vrtxTXcreate
	.globl _vrtxTaskEntry
	.globl _vrtxTcreateHook
	.globl _vrtxTdelay
	.globl _vrtxTdelete
	.globl _vrtxTimer
	.globl _vrtxTinquiry
	.globl _vrtxTpriority
	.globl _vrtxTresume
	.globl _vrtxTslice
	.globl _vrtxTsuspend
	.globl _vrtxTswitchHook
	.globl _vrtxUnlock
	.globl _vrtxGo
	.globl _vrtxInit

#if (CPU == MC68020)
	.globl _vrtxFclear
	.globl _vrtxFcreate
	.globl _vrtxFdelete
	.globl _vrtxFinquiry
	.globl _vrtxFpend
	.globl _vrtxFpost
	.globl _vrtxQEcreate
	.globl _vrtxQjam
	.globl _vrtxScreate
	.globl _vrtxSdelete
	.globl _vrtxSinquiry
	.globl _vrtxSpend
	.globl _vrtxSpost
#endif (CPU == MC68020)

	/* externals */

	.globl _taskSwitchTable		/* table of task switch routines */
	.globl _taskIdCurrent		/* for use with taskIdSelf () */

#if	FALSE
/* a6 offsets to arguments */

#define ARG1	8
#define ARG1W	10
#define ARG2	12
#define ARG2W	14
#define ARG3	16
#define ARG3W	18
#define ARG4	20
#define ARG5	24
#define ARG6	28
#define ARG7	32
#endif	FALSE

	.text
	.even

/*******************************************************************************
*
* vrtxInit - initialize VRTX
*
* This routine is a C interface to the VRTX system call VRTX_INIT.
*
* RETURNS: OK or VRTX error code
*
* SEE ALSO: VRTX User's Guide

* int vrtxInit ()

*/

_vrtxInit:
	moveq	#0x30,d0	/* VRTX_INIT */
	jsr	_sysKernelTrap	/* system trap */
	rts
/*******************************************************************************
*
* vrtxGo - start VRTX multi-tasking
*
* This routine is a C interface to the VRTX system call VRTX_GO.
*
* RETURNS: Doesn't!
*
* SEE ALSO: VRTX User's Guide

* int vrtxGo ()

*/

_vrtxGo:
	movew	#0x3000,sr	/* enable interrupts */
	moveq	#0x31,d0	/* VRTX_GO */
	jsr	_sysKernelTrap	/* system trap */

/*******************************************************************************
*
* vrtxTXcreate - task create with stack address and tcb extension
*
* This routine is the low-level access to the VRTX task creation facility.
* It should NOT be called directly.  
* Normal VxWorks tasks should be created by the "spawn" routines in vrtxLib.
*
* This routine provides a C callable interface to the VRTX SC_TCREATE
* system call to create a task, with the vxWorks task creation extensions 
* that allow task stack addresses and a tcb extension to be specified
* on task creation.
* These extensions are made possible by the added vxWorks task creation
* hook, vrtxTcreateHook, which MUST be specified in the VRTX configuration table
* in usrConfig (1).
*
* Note that this routine allows specification of both the bottom of
* stack and the initial stack pointer.  This allows the caller
* to push initial arguments on the stack for the new task 
* (as in the case of the "spawn" routines).
*
* RETURNS: OK or ERROR
*	
* If ERROR, errnoSet (2) will be called with one of the following values:
*	1 = id number already assigned
*	2 = no TCBs available
*
* NOMANUAL

* vrtxTXcreate (priority, id, mode, entryAddress,
*	      bottomSP, initialSp, tcbExtension)
*     TINY priority;		/* priority of the new task (0 - 255)  *
*     TINY id;			/* id number of the new task (0 - 255) *
*     int mode;			/* (0)user or (1)system mode           *
*     FUNCPTR entryAddress;	/* starting address of the new task    *
*     char *bottomSP;		/* bottom of the stack                 *
*     char *initialSP;		/* initial stack pointer               *
*     char *tcbExtension;	/* address of new task's tcb extension *

*/

_vrtxTXcreate:
	link	a6,#0
	moveml	d2-d3/a2-a3,a7@-	/* save regs */

	movel	a6@(ARG1),d1		/* get priority */
	movel	a6@(ARG2),d2		/* get id */
	movel	a6@(ARG3),d3		/* get mode */
	movel	a6@(ARG4),a0		/* get entry address */
	movel	a6@(ARG5),a1		/* Tcreate hook expects bot SP in a1 */
	movel	a6@(ARG6),a2		/* Tcreate hook expects init SP in a2 */
	movel	a6@(ARG7),a3		/* Tcreate hook expects tcbx in a3 */
	moveq	#0x0,d0			/* SC_TCREATE */
	jsr	_sysKernelTrap		/* system trap */

	moveml	a7@+,d2-d3/a2-a3	/* restore regs */
	bra	return

/*******************************************************************************
*
* vrtxTdelete - vrtx task delete
*
* This routine serves as a C interface to VRTX system call SC_TDELETE.
* THIS SHOULD NEVER BE CALLED.  USE taskDelete INSTEAD!
* The difference being, taskDelete reclaims the stack allocated for a task
* and this routine does not.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with the following value:
*	1 = task id error (no task with specified id number)
*
* NOMANUAL

* int vrtxTdelete (id)
*     int id;		/* task id number *

*/

_vrtxTdelete:
	link	a6,#0
	movel	a6@(ARG1),d1		/* get id */
	moveq	#0x1,d0			/* SC_TDELETE */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vrtxTsuspend - suspend task
*
* This routine serves as a C interface to VRTX system call SC_TSUSPEND.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with the following value:
*	1 = task id error (no task with specified id number)
*
* SEE ALSO: VRTX User's Guide

* int vrtxTsuspend (id)
*     int id;		/* task id number *

*/

_vrtxTsuspend:
	link	a6,#0
	movel	a6@(ARG1),d1		/* get id */
	moveq	#0x2,d0			/* SC_TSUSPEND */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vrtxTresume - resume task
*
* This routine serves as a C interface to VRTX system call SC_TRESUME.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with the following value:
*	1 = task id error (no task with specified id number)
*
* SEE ALSO: VRTX User's Guide

* int vrtxTresume (id)
*     int id;		/* task id number *

*/

_vrtxTresume:
	link	a6,#0
	movel	a6@(ARG1),d1		/* get id */
	moveq	#0x3,d0			/* SC_TRESUME */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vrtxTpriority - task priority change
*
* This routine serves as a C interface to VRTX system call SC_TPRIORITY.
* It changes the priority of an existing task.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with the following value:
*	1 = task id error (no task with specified id number)
*
* SEE ALSO: VRTX User's Guide

* int vrtxTpriority (id, priority)
*     int id;		/* task id number    *
*     int priority;	/* new task priority *

*/

_vrtxTpriority:
	link	a6,#0
	movel	d2,a7@-			/* save reg d2 */

	movel	a6@(ARG1),d1		/* get id */
	movel	a6@(ARG2),d2		/* get priority */
	moveq	#0x4,d0			/* SC_TPRIORITY */
	jsr	_sysKernelTrap		/* system trap */

	movel	a7@+,d2			/* restore reg d2 */
	bra	return

/*******************************************************************************
*
* vrtxTinquiry - task inquiry
*
* This routine serves as a C interface to VRTX system call SC_TINQUIRY.
* It is used to obtain priority and status information about tasks.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with the following value:
*	1 = task id error
*
* SEE ALSO: VRTX User's Guide

* int vrtxTinquiry (id, tid, priority, tstatus, tcbAddr)
*     int id;		/* task id, or 0 to inquire about calling task *
*     int *tid;		/* where to return the task's id number        *
*     int *priority;	/* where to return the task's priority         *
*     int *tstatus;	/* where to return the task's status           *
*     VX_TCB **tcbAddr;	/* where to return pointer to the task's TCB   *

*/

_vrtxTinquiry:
	link	a6,#0
	moveml	d2-d3,a7@-		/* save regs */

	movel	a6@(ARG1),d1		/* get id */
	moveq	#0x5,d0			/* SC_TINQIRY */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG2),a1		/* get pointer to id */
	movel	d1,a1@			/* return id */
	movel	a6@(ARG3),a1		/* get pointer to priority */
	movel	d2,a1@			/* return priority */
	movel	a6@(ARG4),a1		/* get pointer to tstatus */
	movel	d3,a1@			/* return tstatus */
	movel	a6@(ARG5),a1		/* get pointer to TCBaddress */
	movel	a0,a1@			/* return TCBaddress */

	moveml	a7@+,d2-d3		/* restore regs */
	bra	return

/*******************************************************************************
*
* vrtxLock - disable task rescheduling
*
* This routine serves as a C interface to VRTX system call SC_LOCK.
* It prevents task rescheduling until vrtxUnlock (2) is called.  Interrupts
* will continue to be serviced.
*
* SEE ALSO: intLock(2), vrtxUnlock(2), VRTX User's Guide

* VOID vrtxLock ()

*/

_vrtxLock:
	link	a6,#0
	moveq	#0x20,d0		/* SC_LOCK */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vrtxUnlock - enable task rescheduling
*
* This routine serves as a C interface to VRTX system call SC_UNLOCK.
*
* SEE ALSO: vrtxLock(2), VRTX User's Guide

* VOID vrtxUnlock ()

*/

_vrtxUnlock:
	link	a6,#0
	moveq	#0x21,d0		/* SC_UNLOCK */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vrtxGblock - get memory block
*
* This routine serves as a C interface to VRTX system call SC_GBLOCK.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with one of the following values:
*	 3 = no memory blocks available
*	14 = no such partition.
*
* SEE ALSO: VRTX User's Guide

* int vrtxGblock (id, address)
*     short id;	        /* partition id number                    *
*     char *address;	/* receives the logical adrs of the block *

*/

_vrtxGblock:
	link	a6,#0
	movew	a6@(ARG1W),d1		/* get id */
	moveq	#0x6,d0			/* SC_GBLOCK */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG2),a1		/* get pointer to address */
	movel	a0,a1@			/* return address */
	bra	return

/*******************************************************************************
*
* vrtxRblock - release memory block
*
* This routine serves as a C interface to VRTX system call SC_RBLOCK.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with one of the following values:
*	 4 = not a memory block
*	14 = no such partition
*
* SEE ALSO: VRTX User's Guide

* int vrtxRblock (id, address)
*     short id;		/* partition id number          *
*     char *address;	/* logical address of the block *

*/

_vrtxRblock:
	link	a6,#0
	movew	a6@(ARG1W),d1		/* get id */
	movel	a6@(ARG2),a0		/* get address */
	moveq	#0x7,d0			/* SC_RBLOCK */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vrtxPcreate - create memory partition
*
* This routine serves as a C interface to VRTX system call SC_PCREATE.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with one of the following values:
*	 3 = no memory available
*	14 = partition id number already assigned
*
* SEE ALSO: VRTX User's Guide

* int vrtxPcreate (id, psize, bsize, paddress)
*     int id;		/* partition id number             *
*     int psize;	/* partition size in bytes         *
*     int bsize;	/* partition's block size in bytes *
*     char *paddress;	/* partition starting address      *

*/

_vrtxPcreate:
	link	a6,#0
	moveml	d2-d3,a7@-		/* save regs */

	movel	a6@(ARG1),d1		/* get id */
	movel	a6@(ARG2),d2		/* get partition size */
	movel	a6@(ARG3),d3		/* get block size */
	movel	a6@(ARG4),a0		/* get partition address */
	moveq	#0x22,d0		/* SC_PCREATE */
	jsr	_sysKernelTrap		/* system trap */

	moveml	a7@+,d2-d3		/* restore regs */
	bra	return

/*******************************************************************************
*
* vrtxPextend - extend memory partition
*
* This routine serves as a C interface to VRTX system call SC_PEXTEND.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with one of the following values:
*	 3 = no memory available
*	14 = no such partition id number 
*
* SEE ALSO: VRTX User's Guide

* int vrtxPextend (id, esize, eaddress)
*     short id;		/* partition id number        *
*     int esize;	/* extension size in bytes    *
*     char *eaddress;	/* extension starting address *

*/

_vrtxPextend:
	link	a6,#0
	movel	d2,a7@-			/* save reg d2 */

	movew	a6@(ARG1W),d1		/* get id */
	movel	a6@(ARG2),d2		/* get extension size */
	movel	a6@(ARG3),a0		/* get extension address */
	moveq	#0x23,d0		/* SC_PEXTEND */
	jsr	_sysKernelTrap		/* system trap */

	movel	a7@+,d2			/* restore reg d2 */
	bra	return

/*******************************************************************************
*
* vrtxPost - post a message
*
* This routine serves as a C interface to VRTX system call SC_POST.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with one of the following values:
*	5 = mailbox already in use
*	6 = zero message
*
* SEE ALSO: VRTX User's Guide

* int vrtxPost (pMailBox, message)
*     int *pMailBox;	/* pointer to mail box	*
*     int message;	/* message to post	*

*/

_vrtxPost:
	link	a6,#0
	movel	a6@(ARG1),a0		/* get mailbox address */
	movel	a6@(ARG2),d1		/* get message */
	moveq	#0x8,d0			/* SC_POST */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vrtxPend - pend for message
*
* This routine serves as a C interface to VRTX system call SC_PEND.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with the following value:
*	10 = timeout
*
* SEE ALSO: VRTX User's Guide

* int vrtxPend (pMailBox, timeout, pMsg)
*     int *pMailBox;		/* pointer to mail box			*
*     unsigned timeout;		/* # of ticks before returning timeout	*
*     int *pMsg;		/* pointer where to return msg		*

*/

_vrtxPend:
	link	a6,#0
	movel	a6@(ARG1),a0		/* get mail box address */
	movel	a6@(ARG2),d1		/* get timeout */
	moveq	#0x9,d0			/* SC_PEND */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG3),a0		/* get pointer to msg */
	movel	d1,a0@			/* return message */
	bra	return

/*******************************************************************************
*
* vrtxAccept - accept a message
*
* This routine serves as a C interface to VRTX system call SC_ACCEPT.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with the following value:
*	11 = no message present
*
* SEE ALSO: VRTX User's Guide

* int vrtxAccept (pMailBox, pMsg)
*     int *pMailBox;	/* pointer to mail box		*
*     int *pMsg;	/* pointer where to return msg	*

*/

_vrtxAccept:
	link	a6,#0
	movel	a6@(ARG1),a0		/* get mailbox address */
	moveq	#0x25,d0		/* SC_ACCEPT */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG2),a0		/* get pointer to message */
	movel	d1,a0@			/* return message */
	bra	return

/*******************************************************************************
*
* vrtxQpost - post message to queue
*
* This routine serves as a C interface to VRTX system call SC_QPOST.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with one of the following values:
*	12 = no such queue
*	13 = queue full
*
* SEE ALSO: VRTX User's Guide

* int vrtxQpost (qid, message)
*     short qid;	/* queue id number *
*     int message;	/* message to post *

*/

_vrtxQpost:
	link	a6,#0
	movel	d2,a7@-			/* save reg d2 */

	movew	a6@(ARG1W),d1		/* get qid */
	movel	a6@(ARG2),d2		/* get message */
	moveq	#0x26,d0		/* SC_QPOST */
	jsr	_sysKernelTrap		/* system trap */

	movel	a7@+,d2			/* restore reg d2 */
	bra	return

#if (CPU == MC68020)
/*******************************************************************************
*
* vrtxQjam - jam message to beginning of queue
*
* This routine serves as a C interface to VRTX32 system call SC_QJAM.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with one of the following values:
*	12 = no such queue
*	13 = queue full
*
* SEE ALSO: VRTX32 User's Guide

* int vrtxQjam (qid, message)
*     short qid;	/* queue id number *
*     int message;	/* message to jam  *

*/

_vrtxQjam:
	link	a6,#0
	movel	d2,a7@-			/* save reg d2 */

	movew	a6@(ARG1W),d1		/* get qid */
	movel	a6@(ARG2),d2		/* get message */
	moveq	#0x1e,d0		/* SC_QJAM */
	jsr	_sysKernelTrap		/* system trap */

	movel	a7@+,d2			/* restore reg d2 */
	bra	return
#endif (CPU == MC68020)
/*******************************************************************************
*
* vrtxQpend - pend for message from queue
*
* This routine serves as a C interface to VRTX system call SC_QPEND.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with one of the following values:
*	10 = time-out
*	12 = no such queue
*
* SEE ALSO: VRTX User's Guide

* int vrtxQpend (qid, timeout, pMsg)
*     short qid;		/* queue id number			*
*     unsigned timeout;		/* # of ticks before returning timeout	*
*     int *pMsg;		/* pointer where to return msg		*

*/

_vrtxQpend:
	link	a6,#0
	movel	d2,a7@-			/* save reg d2 */

	movew	a6@(ARG1W),d1		/* get qid */
	movel	a6@(ARG2),d2		/* get timeout */
	moveq	#0x27,d0		/* SC_QPEND */
	jsr	_sysKernelTrap		/* system trap */

	cmpl	#0xa,d0			/* did it time out? */
	beq	timedout		/* then there's no message to return */
	movel	a6@(ARG3),a0		/* get pointer to msg */
	movel	d2,a0@			/* return message */
timedout:

	movel	a7@+,d2			/* restore reg d2 */
	bra	return

/*******************************************************************************
*
* vrtxQaccept - accept message from queue
*
* This routine serves as a C interface to VRTX system call SC_QACCEPT.
*
* RETURNS: OK or ERROR
*	 
* If ERROR, errnoSet (2) will be called with one of the following values:
*	11 = no message present
*	12 = no such queue
*
* SEE ALSO: VRTX User's Guide

* int vrtxQaccept (qid, pMsg)
*     short qid;	/* queue id number		*
*     int *pMsg;	/* pointer where to return msg	*

*/

_vrtxQaccept:
	link	a6,#0
	movel	d2,a7@-			/* save reg d2 */

	movew	a6@(ARG1W),d1		/* get qid */
	moveq	#0x28,d0		/* SC_QACCEPT */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG2),a0		/* get pointer to msg */
	movel	d2,a0@			/* return message */

	movel	a7@+,d2			/* restore reg d2 */
	bra	return

/*******************************************************************************
*
* vrtxQcreate - create message queue
*
* This routine serves as a C interface to VRTX system call SC_QCREATE.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with one of the following values:
*	 3 = no memory available
*	12 = queue id number already assigned
*
* SEE ALSO: VRTX User's Guide

* int vrtxQcreate (qid, count)
*     short qid;	/* queue id number                *
*     short count;	/* number of entries in the queue *

*/

_vrtxQcreate:
	link	a6,#0
	movel	d2,a7@-			/* save reg d2 */

	movew	a6@(ARG1W),d1		/* get qid */
	movew	a6@(ARG2W),d2		/* get count */
	moveq	#0x29,d0		/* SC_QCREATE */
	jsr	_sysKernelTrap		/* system trap */

	movel	a7@+,d2			/* restore reg d2 */
	bra	return

#if (CPU == MC68020)
/*******************************************************************************
*
* vrtxQEcreate - create FIFO message queue
*
* This routine serves as a C interface to VRTX32 system call SC_QECREATE.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with one of the following values:
*	 3 = no memory available
*	12 = queue id number already assigned
*
* SEE ALSO: VRTX32 User's Guide

* int vrtxQEcreate (qid, count, pendOrder)
*     short qid;        /* queue id number                 *
*     short count;      /* number of entries in the queue  *
*     int pendOrder;    /* 0: pend tasks in priority order *
*                       /* 1: pend tasks in FIFO order     *

*/

_vrtxQEcreate:
	link	a6,#0
	moveml	d2-d3,a7@-		/* save regs */

	movew	a6@(ARG1W),d1		/* get qid */
	movew	a6@(ARG2W),d2		/* get count */
	movel	a6@(ARG3),d3		/* get count */
	moveq	#0x1f,d0		/* SC_QECREATE */
	jsr	_sysKernelTrap		/* system trap */

	moveml	a7@+,d2-d3		/* restore regs */
	bra	return
#endif (CPU == MC68020)
/*******************************************************************************
*
* vrtxQinquiry - message queue status inquiry
*
* This routine serves as a C interface to VRTX system call SC_QINQUIRY.
*
* RETURNS: OK or ERROR
*	 
* If ERROR, errnoSet (2) will be called with the following value:
*	12 = no such queue
*
* SEE ALSO: VRTX User's Guide

* int vrtxQinquiry (qid, pMsg, pNum)
*     short qid;	/* queue id number			*
*     int *pMsg;	/* ptr where to return msg		*
*     int *pNum;	/* ptr where to return count of msgs	*

*/

_vrtxQinquiry:
	link	a6,#0
	moveml	d2-d3,a7@-		/* save regs */

	movew	a6@(ARG1W),d1		/* get qid */
	moveq	#0x2a,d0		/* SC_QINQUIRY */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG2),a0		/* get pointer to msg */
	movel	d2,a0@			/* return message */

	movel	a6@(ARG3),a0		/* get pointer to count */
	andl	#0xffff,d3		/* make sure upper byte is clear */
	movel	d3,a0@			/* return count */

	moveml	a7@+,d2-d3		/* restore regs */
	bra	return

#if (CPU == MC68020)
/*******************************************************************************
*
* vrtxFcreate - create event flag group
*
* This routine serves as a C interface to VRTX32 system call SC_FCREATE.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with the following value:
*	48 = no control blocks available
*
* SEE ALSO: VRTX32 User's Guide

* int vrtxFcreate (pGID)
*     UINT *pGID;	/* where to return event flag group ID number *

*/

_vrtxFcreate:
	link	a6,#0

	moveq	#0x17,d0		/* SC_FCREATE */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG1),a0		/* get pointer to group ID number */
	movel	d1,a0@			/* return group ID number */

	bra	return

/*******************************************************************************
*
* vrtxFdelete - delete event flag group
*
* This routine serves as a C interface to VRTX32 system call SC_FDELETE.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with the following value:
*	49 = event flag group id error
*	50 = tasks pending on event flag group
*
* SEE ALSO: VRTX32 User's Guide

* int vrtxFdelete (efgid, forceDelete)
*     UINT efgid;	/* event flag group id number             *
*     int forceDelete;	/* 0: delete efg only if no tasks pending *
*               	/* 1: delete efg, ready all pending tasks *

*/

_vrtxFdelete:
	link	a6,#0
	movel	d2,a7@-			/* save reg */

	movel	a6@(ARG1),d1		/* get efgid */
	movel	a6@(ARG2),d2		/* get forceDelete */
	moveq	#0x18,d0		/* SC_FDELETE */
	jsr	_sysKernelTrap		/* system trap */

	movel	a7@+,d2			/* restore reg */
	bra	return

/*******************************************************************************
*
* vrtxFpost - post event to event flag group
*
* This routine serves as a C interface to VRTX32 system call SC_FPOST.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with one of the following values:
*	49 = event flag group ID error
*	52 = event flag already set
*
* SEE ALSO: VRTX32 User's Guide

* int vrtxFpost (efgid, eventFlags)
*     UINT efgid;	/* event flag group ID number              *
*     UINT eventFlags;	/* mask of event flags in event flag group *

*/

_vrtxFpost:
	link	a6,#0
	movel	d2,a7@-			/* save reg */

	movel	a6@(ARG1),d1		/* get efgid */
	movel	a6@(ARG2),d2		/* get eventFlags */
	moveq	#0x1a,d0		/* SC_FPOST */
	jsr	_sysKernelTrap		/* system trap */

	movel	a7@+,d2			/* restore reg */
	bra	return

/*******************************************************************************
*
* vrtxFpend - pend on event flag group
*
* This routine serves as a C interface to VRTX32 system call SC_FPEND.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with the following value:
*	10 = timeout
*	49 = event flag group ID error
*	51 = event flag group is deleted
*
* SEE ALSO: VRTX32 User's Guide

* int vrtxFpend (efgid, timeout, eventFlags, maskOpt, pEfg)
*     UINT efgid;     /* event flag group ID number                      *
*     UINT timeout;   /* number of ticks before returning timeout        *
*     UINT eventFlags;/* mask of event flags in event flag group         *
*     int maskOpt;    /* 0: pend on OR mask, any spec'd flag readies task*
*                     /* 1: pend on AND mask, all spec'd flags are true  *
*     UINT *pEfg;     /* pointer to where to return event flag group     *

*/

_vrtxFpend:
	link	a6,#0
	moveml	d2-d4,a7@-		/* save regs */

	movel	a6@(ARG1),d1		/* get efgid */
	movel	a6@(ARG2),d2		/* get timeout */
	movel	a6@(ARG3),d3		/* get eventFlags */
	movel	a6@(ARG4),d4		/* get maskOpt */
	moveq	#0x19,d0		/* SC_FPEND */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG5),a0		/* get pEfg */
	movel	d2,a0@			/* return event flag group */

	moveml	a7@+,d2-d4		/* restore regs */
	bra	return

/*******************************************************************************
*
* vrtxFclear - clear events in event flag group
*
* This routine serves as a C interface to VRTX32 system call SC_FCLEAR.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with the following value:
*	49 = event flag group ID error
*
* SEE ALSO: VRTX32 User's Guide

* int vrtxFclear (efgid, eventFlags, pEfg)
*     UINT efgid;	/* event flag group ID number              *
*     UINT eventFlags;	/* mask of event flags in event flag group *
*     UINT *pEfg;	/* return old event flag group here        *

*/

_vrtxFclear:
	link	a6,#0
	movel	d2,a7@-			/* save reg */

	movel	a6@(ARG1),d1		/* get efgid */
	movel	a6@(ARG2),d2		/* get eventFlags */
	moveq	#0x1b,d0		/* SC_FCLEAR */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG3),a0		/* get pEfg */
	movel	d2,a0@			/* return event flag group */

	movel	a7@+,d2			/* restore reg */
	bra	return

/*******************************************************************************
*
* vrtxFinquiry - event flag group inquiry
*
* This routine serves as a C interface to VRTX32 system call SC_FINQUIRY.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR
*	 
* If ERROR, errnoSet (2) will be called with the following value:
*	49 = event flag group ID error
*
* SEE ALSO: VRTX32 User's Guide

* int vrtxFinquiry (efgid, pEfg)
*     UINT efgid;	/* event flag group ID number   *
*     UINT *pEfg;	/* return event flag group here *

*/

_vrtxFinquiry:
	link	a6,#0
	movel	d2,a7@-			/* save reg */

	movel	a6@(ARG1),d1		/* get efgid */
	moveq	#0x1c,d0		/* SC_FINQUIRY */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG2),a0		/* get pEfg */
	movel	d2,a0@			/* return event flag group */

	movel	a7@+,d2			/* restore reg */
	bra	return

/*******************************************************************************
*
* vrtxScreate - create semaphore
*
* This routine serves as a C interface to VRTX32 system call SC_SCREATE.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with the following value:
*	48 = no control blocks available
*
* SEE ALSO: VRTX32 User's Guide

* int vrtxScreate (initVal, pendOrder, pSem)
*     UINT initVal;     /* initial value of semaphore, (0-65535) *
*     int pendOrder;    /* 0: pend tasks in priority order       *
*                       /* 1: pend tasks in FIFO order           *
*     UINT *pSem;       /* return semaphore ID number here       *

*/

_vrtxScreate:
	link	a6,#0
	movel	d2,a7@-			/* save reg */

	movel	a6@(ARG1),d1		/* get initVal */
	movel	a6@(ARG2),d2		/* get pendOrder */
	moveq	#0x2b,d0		/* SC_SCREATE */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG3),a0		/* get pSem */
	movel	d1,a0@			/* return semaphore ID number */

	movel	a7@+,d2			/* restore reg */
	bra	return

/*******************************************************************************
*
* vrtxSdelete - delete semaphore
*
* This routine serves as a C interface to VRTX32 system call SC_SDELETE.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with the following value:
*	49 = semaphore id error
*	50 = tasks pending on semaphore (when forceDelete = 0)
*
* SEE ALSO: VRTX32 User's Guide

* int vrtxSdelete (semId, forceDelete)
*     UINT semId;       /* semaphore id number                          *
*     int forceDelete;  /* 0: delete semaphore only if no tasks pending *
*                       /* 1: delete semaphore, ready all pending tasks *

*/

_vrtxSdelete:
	link	a6,#0
	movel	d2,a7@-			/* save reg */

	movel	a6@(ARG1),d1		/* get semId */
	movel	a6@(ARG2),d2		/* get forceDelete */
	moveq	#0x2c,d0		/* SC_SDELETE */
	jsr	_sysKernelTrap		/* system trap */

	movel	a7@+,d2			/* restore reg */
	bra	return

/*******************************************************************************
*
* vrtxSpost - post unit to semaphore
*
* This routine serves as a C interface to VRTX32 system call SC_SPOST.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	
* If ERROR, errnoSet (2) will be called with one of the following values:
*	49 = semaphore ID error
*	52 = semaphore overflow
*
* SEE ALSO: VRTX32 User's Guide

* int vrtxSpost (semId)
*     UINT semId;	/* semaphore id number *

*/

_vrtxSpost:
	link	a6,#0

	movel	a6@(ARG1),d1		/* get semId */
	moveq	#0x2e,d0		/* SC_SPOST */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vrtxSpend - pend on semaphore
*
* This routine serves as a C interface to VRTX32 system call SC_SPEND.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR.
*	 
* If ERROR, errnoSet (2) will be called with the following value:
*	10 = timeout
*	49 = semaphore ID error
*	51 = semaphore is deleted
*
* SEE ALSO: VRTX32 User's Guide

* int vrtxSpend (semId, timeout)
*     UINT semId;	/* semaphore id number                      *
*     UINT timeout;	/* number of ticks before returning timeout *

*/

_vrtxSpend:
	link	a6,#0
	movel	d2,a7@-			/* save reg */

	movel	a6@(ARG1),d1		/* get semId */
	movel	a6@(ARG2),d2		/* get timeout */
	moveq	#0x2d,d0		/* SC_SPEND */
	jsr	_sysKernelTrap		/* system trap */

	movel	a7@+,d2			/* restore reg */
	bra	return

/*******************************************************************************
*
* vrtxSinquiry - semaphore inquiry
*
* This routine serves as a C interface to VRTX32 system call SC_SINQUIRY.
* It exists only in systems with the VRTX32 kernel.
*
* RETURNS: OK or ERROR
*	 
* If ERROR, errnoSet (2) will be called with the following value:
*	49 = semaphore ID error
*
* SEE ALSO: VRTX32 User's Guide

* int vrtxSinquiry (semId, pSemVal)
*     UINT semId;	/* semaphore ID number         *
*     UINT *pSemVal;	/* return semaphore value here *

*/

_vrtxSinquiry:
	link	a6,#0
	movel	d2,a7@-			/* save reg */

	movel	a6@(ARG1),d1		/* get semId */
	moveq	#0x2f,d0		/* SC_SINQUIRY */
	jsr	_sysKernelTrap		/* system trap */

	movel	a6@(ARG2),a0		/* get pSemVal */
	movel	d2,a0@			/* return semaphore */

	movel	a7@+,d2			/* restore reg */
	bra	return

#endif (CPU == MC68020)
/*******************************************************************************
*
* vrtxGtime - get time
*
* This routine serves as a C interface to VRTX system call SC_GTIME.
* It returns the current number of "ticks" in the VRTX clock counter.
*
* RETURNS: Clock counter contents.
*
* SEE ALSO: VRTX User's Guide

* unsigned int vrtxGtime ()

*/

_vrtxGtime:
	link	a6,#0
	moveq	#0xa,d0			/* SC_GTIME */
	jsr	_sysKernelTrap		/* system trap */

	movel	d1,d0			/* time into d0 */
	unlk	a6
	rts

/*******************************************************************************
*
* vrtxStime - set time
*
* This routine serves as a C interface to VRTX system call SC_STIME.
* It sets the system clock to the specified time, in "ticks".
*
* SEE ALSO: VRTX User's Guide

* VOID vrtxStime (time)
*     unsigned time;	/* new clock counter value *

*/

_vrtxStime:
	link	a6,#0
	movel	a6@(ARG1),d1		/* get time */
	moveq	#0xb,d0			/* SC_STIME */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vrtxTdelay - delay task
*
* This routine serves as a C interface to VRTX system call SC_TDELAY.
* It suspends the calling task for the specified number of clock ticks.
*
*
* SEE ALSO: VRTX User's Guide

* VOID vrtxTdelay (delay)
*     int delay;	/* # of ticks for which task will be suspended *

*/

_vrtxTdelay:
	link	a6,#0
	movel	a6@(ARG1),d1		/* get delay */
	moveq	#0xc,d0			/* SC_TDELAY */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vrtxTslice - enable round-robin scheduling
*
* This routine serves as a C interface to VRTX system call SC_TSLICE.
* If interval == 0, round-robin scheduling is disabled.
*
* SEE ALSO: VRTX User's Guide

* VOID vrtxTslice (interval)
*     short interval;	/* number of ticks per task (0 to disable) *

*/

_vrtxTslice:
	link	a6,#0
	movew	a6@(ARG1W),d1		/* get interval */
	moveq	#0x15,d0		/* SC_TSLICE */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* vrtxTimer - announce timer interrupt
*
* This routine serves as a C interface to VRTX system call UI_TIMER.
* It is called by the system clock interrupt handler
* to announce a system "tick" to VRTX.
*
* SEE ALSO: VRTX User's Guide

* VOID vrtxTimer ()

*/

_vrtxTimer:
	link	a6,#0
	moveq	#0x12,d0		/* UI_TIMER */
	jsr	_sysKernelTrap		/* system trap */

	bra	return

/*******************************************************************************
*
* return - universal exit
*
* All C callable routines in this library except vrtxGtime exit 
* via here. If the VRTX return code in d0 is not 0, the vrtx module number
* is or'd with d0 and errnoSet is called, and ERROR is returned in d0.
*
* NOMANUAL
*/

return:
	tstl	d0			/* OK? */
	beq	byebye			/* normal exit */

	orl	#0x10000,d0		/* or M_vrtx into errno */
	movel	d0,a7@-			/* push VRTX errno */
	jsr	_errnoSet		/* set task's status word */
	addql	#4,a7
	moveq	#-1,d0			/* return (ERROR) */

byebye:
	unlk	a6
	rts

/*******************************************************************************
*
* vrtxTaskEntry - task startup code following spawn
*
* This hunk of code is the initial entry point to every task created via
* the "spawn" routines.  Spawn has put the true entry point of the
* task into the tcb extension before creating the task,
* and then pushed exactly ten arguments (although the task may use
* fewer) onto the stack.  This code picks up the real entry point and calls it.
* Upon return, the 10 task args are popped, and the result of the main
* routine is passed to "exit" which terminates the task.
* This way of doing things has several purposes.  First a task is easily
* "restartable" (see "restart" routine in vrtxLib.c) since the real
* entry point is available in the tcb extension.  Second, the call to the main
* routine is a normal call including the usual stack clean-up afterwards,
* which means that debugging stack trace facilities will handle the call of
* the main routine properly.  
*
* NOMANUAL
*/

_vrtxTaskEntry:
	movel	#0,a6		/* make sure frame pointer is 0 */
#if (CPU == MC68020)
	movel	a0@(28),a0	/* get pointer to tcb extension */
#else (CPU != MC68020)
	movel	a0@(8),a0	/* get pointer to tcb extension */
#endif (CPU)
	movel	a0@,a0		/* entry adr is 1st member of tcb extension */
	jsr	a0@		/* call main routine */
	addl	#40,a7		/* pop args to main routine */
	movel	d0,a7@-		/* pass result to exit */
	jsr	_exit		/* gone for good */

/*******************************************************************************
*
* intEnt - enter interrupt level
*
* IntEnt should be jumped to, in place of the VRTX UI_ENTER call, when
* entering an interrupt.  This normally happens automatically, from
* the stub built by intConnect (2).
*
* IntEnt should NEVER be called from C.
*
* SEE ALSO: intConnect (2)

* VOID intEnt ()

*/

_intEnt:
	addql	#1,_intCnt	/* Bump the counter */

#if (CPU != MC68020)
	movel	d0,a7@-		/* save d0 */
	moveq	#0x16,d0	/* UI_ENTER */
	jsr	_sysKernelTrap	/* trap to VRTX */
	movel	a7@+,d0		/* restore d0 */
#endif (CPU)

	rts

/*******************************************************************************
*
* intExit - leave interrupt level
*
* This routine should be jumped to in place of the VRTX UI_EXIT call. This
* routine can NEVER be called from C.  It can only be jumped to from 
* assembly code.

* intExit ()

*/

_intExit:
	subql	#1,_intCnt	/* debump the counter */
	movel	d0,a7@-		/* store d0 on stack for VRTX */
	moveq	#0x11,d0	/* UI_EXIT */
	jmp	_sysKernelTrap	/* trap to VRTX (never to return here) */

/*******************************************************************************
*
* vrtxTcreateHook - task creation hook
*
* This routine should be specified in the VRTX configuration table,
* in usrConfig (1), as the task creation extension routine.
* This addition extends the VRTX SC_TCREATE system call to include 
* specification of the new task's stack address instead of using a 
* VRTX allocated stack, and allows specification of a TCB extension 
* address.  The following additional parameters are thus expected 
* in a trap to the VRTX SC_TCREATE system call:
*
*	a1 = bottom of stack
*	a2 = initial stack pointer
*	a3 = TCB extension pointer
*
* Note that the initial stack pointer is different from the bottom of
* stack if the caller has pushed initial arguments on the stack for
* the new task (as in the case of the "spawn" routines).
*
* If no bottom of stack is specified, i.e. TCREATE was called with a1 = zero, 
* then the new task's stack pointers are not modified.  This allows the
* normal VRTX stack allocation mechanism to be used if desired.
*
* On entry, a1 contains a pointer to the task's TCB.
* VRTX constructs the stack extension that contains the extra TCB stuff
* in its own workspace (this is pointed to by TCB.SP).  We must copy this
* over to the real stack that was allocated by the user (or by taskSpawn).
* Also, the TCB.USP must point to the initial stack location, while
* the TCB.SP must point to the extra TCB stuff on the stack (see VRTX
* documentation). THIS IS NOT A C CALLABLE ROUTINE.
*
* SEE ALSO: taskCreateHookAdd (2)

* vrtxTcreateHook ()

*/

#define TCB_EXT	0x08
#define TCB_A1	0x2c
#define TCB_A2	0x30
#define TCB_A3	0x34
#define TCB_SSP	0x38
#define TCB_USP	0x3c
#define TCB_BOS	0x40

_vrtxTcreateHook:
	movel	a1@(TCB_A3),a1@(TCB_EXT) /* copy TCB.a3 to TCB.tcb_extension */

	tstl	a1@(TCB_A2)		/* check for stack pointer specified */
	beq	tcend			/* no, leave vrtx stack pointers */

	moveml	d0-d2/a0-a3,a7@-	/* save regs */

	movel	a1@(TCB_SSP),a3		/* the SP of the new task points to */
					/* VRTX workspace; must be copied */
					/* to the task's stack. */
	movel	a1@(TCB_A2),a2		/* TCB.a2 should point to the */
					/* new stack */
#if (CPU == MC68020)
	subl	#28,a2	
	moveq	#13,d0			/* loop counter: copy 14 words */
#endif
#if (CPU == MC68010)
	subl	#28,a2	
	moveq	#13,d0			/* loop counter: copy 14 words */
#endif
#if (CPU == MC68000)
	subl	#26,a2	
	moveq	#12,d0			/* loop counter: copy 13 words */
#endif
	
copystk:
	movew	a3@+,a2@+		/* copy VRTX workspace -> task stack */
	dbf	d0,copystk

	movel	a1@(TCB_A2),d0		/* get sp out of TCB.a2 */
	movel	d0,a1@(TCB_USP)		/* put it in TCB.usp */
#if (CPU == MC68020)
	subl	#28,d0			/* move sp down past new stuff */
#endif
#if (CPU == MC68010)
	subl	#28,d0			/* move sp down past new stuff */
#endif
#if (CPU == MC68000)
	subl	#26,d0			/* move sp down past new stuff */
#endif
	movel	d0,a1@(TCB_SSP)		/* put it in TCB.sp */
	moveml	a7@+,d0-d2/a0-a3	/* restore regs */
tcend:
	rts

/*******************************************************************************
*
* vrtxTswitchHook - task switch hook
*
* This routine should be specified in the VRTX configuration table,
* as the task switch extension routine.
* It loops through a NULL terminated table of C task switch routines
* that have been specified by calls to taskSwitchHookAdd (2).
* Each routine in the table is called with pointers to
* the old and new tasks' TCB extensions on each task
* switch.  THIS IS NOT A C CALLABLE ROUTINE.
*
* SEE ALSO: taskSwitchHookAdd (2)

* vrtxTswitchHook ()

*/

_vrtxTswitchHook:
	moveml	d0-d2/a0-a2,a7@-	/* save regs */

	moveb	a2@(0xd),_taskIdCurrent + 3 /* update pointer to current task */

	movel	a2@(0x08),a7@-		/* push pointer to new tcbX */
	movel	a1@(0x08),a7@-		/* push pointer to old tcbX*/

	movel	#_taskSwitchTable,a2	/* get adrs of task switch rtn list */

tsloop:
	movel	a2@,d0			/* get next task switch rtn */
	beq	tsend			/* quit if end of list */

	movel	d0,a1
	jsr	a1@			/* call routine */

	addl	#4,a2			/* bump to next task switch routine */
	bra	tsloop			/* loop */

tsend:
	addl	#8,a7			/* clean up stack */

	moveml	a7@+,d0-d2/a0-a2	/* restore regs */
	rts
