/* kernelLib.c - VxWorks/VRTX kernel library */

/*
modification history
--------------------
01e,21apr89,jcf  added kernelType.
01d,24aug88,gae  documentation.
01c,29jun88,jcf  extended kernelInit to spawn a new routine kernelRoot
		   which does any kernel specific work before calling
		   usrRoot ().
01b,24mar88,ecs  VRTX32 version.
01a,19jan88,jcf  written.
*/

/*
DESCRIPTION
This library provides a generic interface to the VRTX kernel
management routines.  Kernel management includes:

    - initialize the kernel,
    - start the kernel,
    - get the version number of the kernel,
    - turn on/off round robin (time slice) scheduling.

SEE ALSO: "Architecture", taskLib (1), tickLib (1)
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "taskLib.h"
#include "memLib.h"
#include "iv68k.h"

#if (CPU == MC68020)
#include "vrtx32.h"
#else (CPU != MC68020)
#include "vrtx.h"
#endif (CPU)


IMPORT VOID vrtxent ();			/* vtrx trap entry */
IMPORT int vrtxCfTblVec;		/* vtrx config table int vec entry */
IMPORT VRTX_CONFIG_TABLE usrCftbl;	/* vrtx config table */
IMPORT int intLockLevel;		/* vxWorks interrupt lock out level */


#define MEM_TAG	32		/* this is the size of the memory tag */
				/* placed at the end of memory by memInit */

/* globals */

int kernelType = KERNEL_UNINIT;		/* initially uninitalized */

/* forward declarations */

VOID kernelRoot ();


/*******************************************************************************
*
* kernelInit - initialize kernel
*
* This routine initializes and starts the kernel.  It should only be called
* once.  The routine `rootRtn' is the user's startup code that subsequently
* installs system facilities (eg. initialize the I/O system),
* spawns tasks, etc..
*
* INTERNAL
* The routine kernelRoot is called before the user's root routine so that
* memory management can be initialized.
*
* ARGSUSED
*/

VOID kernelInit (trapNum, rootRtn, rootStackSize, memPoolStart,
		 memPoolEnd, intStackSize, lockOutLevel)
    int trapNum;                /* kernel trap number             */
    FUNCPTR rootRtn;            /* user startup routine           */
    int rootStackSize;          /* stack size for startup routine */
    char *memPoolStart;         /* beginning of memory            */
    char *memPoolEnd;           /* end of memory                  */
    int intStackSize;           /* interrupt stack size           */
    int lockOutLevel;           /* interrupt lock out level (1-7) */
 
    {
    char *bottomOfStack = (char *)
			     (memPoolEnd - (ULONG) sizeof (TCBX) - MEM_TAG);

    /* initialize vxWorks interrupt lock out level */
    intLockLevel = lockOutLevel;

    /*
    * set up memory as follows
    * ------------------------ <--- memPoolEnd;		HIGH MEMORY
    * |                      |
    * |     memPoolTag       |    We have to leave room for this tag so we
    * |     < 32 bytes       |    can add these areas to the free mem pool
    * ------------------------    without clobbering the TCBX, which would
    * |                      |    make it impossible to delete the root task.
    * |        TCBX          |
    * |                      |
    * ------------------------ <--- bottomOfStack;
    * |                      |
    * |      ROOT STACK      |    rootStackSize includes the memPoolTag and TCBX
    * |                      |
    * ------------------------
    * ------------------------ <--- tempMemTop;
    * |                      |
    * |   FREE MEMORY POOL   |
    * |                      |
    * ------------------------ <--- memPoolStart      LOW  MEMORY
    */

    /* set kernel config table pointer and kernel trap */

    intVecSet (TRAPNUM_TO_IVEC (trapNum), vrtxent);
    intVecSet ((FUNCPTR *) vrtxCfTblVec, (FUNCPTR) (int)&usrCftbl);

    /* initialize kernel and spawn the root task to do the rest of the work */

    bzero (usrCftbl.wrkspcAdrs, usrCftbl.wrkspcSize);   /* clear workspace */

    vrtxInit ();					/* initialize kernel */

    taskActivate (taskCreate ("rootTask", 0, VX_SUPERVISOR_MODE, bottomOfStack,
			      rootStackSize - sizeof (TCBX) - MEM_TAG,
			      (TCBX *) bottomOfStack, kernelRoot,
			      (int) memPoolStart,
			      (int) memPoolEnd, rootRtn,
			      rootStackSize));

    vrtxGo ();						/* start kernel */
    }
/*******************************************************************************
*
* kernelVersion - get kernel version
*
* RETURNS: pointer to string of format `kernel version xxx.xxx'
*/

char *kernelVersion ()

    {
#if (CPU == MC68020)
    return ("VRTX32/68020 version 1.05");
#else (CPU != MC68020)
    return ("VRTX version 3.2");
#endif (CPU)
    }
/*******************************************************************************
*
* kernelTimeSlice - set time slice to specified ticks
*
* This routine enables round-robin selection among tasks of same priority.
*
* RETURNS: OK
*/

STATUS kernelTimeSlice (ticks)
    int ticks;		/* number of ticks for each task (0 to disable) */

    {
    vrtxTslice (ticks);
    return (OK);
    }
/*******************************************************************************
*
* kernelRoot - temp root
*
* This routine is the first task to run under VxWorks.  In initializes the
* memory management, calls a user specified root routine (usually usrRoot(1)),
* and then cleans up after itself.
*/  

LOCAL VOID kernelRoot (memPoolStart, memPoolEnd, rootRtn, rootStackSize)
    char *memPoolStart;
    ULONG memPoolEnd;
    FUNCPTR rootRtn;
    ULONG rootStackSize;

    {
    ULONG tempMemTop = memPoolEnd - rootStackSize;

    kernelType = KERNEL_VRTX;	/* variable as to what kernel we're using */

    /* find the top of memory and initialize memory pool.
     * rootStackSize + sizeof (TCBX) bytes at the top of memory are NOT added
     * to the pool now, because kernelRoot's stack is at the top of memory.
     * Our last act, below, will be to add that missing top of memory to
     * the pool.
     */

    memInit (memPoolStart, (unsigned) (tempMemTop - (int)memPoolStart));

    (* rootRtn) ();		/* call rootRtn */


    /* add to memory pool our stack, which was excluded from initial
     * pool in memInit above.  We assume (hope!) that this memory will not
     * be allocated until after we have exited and gone away.
     */

    memAddToPool ((char *) tempMemTop, (unsigned) rootStackSize);

    taskDelete (0);		/* commit suicide */
    }
