/* semLib.c - VxWorks/pSOS semaphore library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01g,08sep88,jcf  got semClear right... this time for sure.
01f,07sep88,gae  fudged 01e.
01e,01sep88,jcf  reworked semClear so it tells you when you got the semaphore.
01d,24aug88,gae  documentation.
01c,07jul88,jcf  lint.
01b,29may88,jcf  fixed bad call to taskTcbX () in semTake.
01a,28jan88,jcf  written.
*/

/*
DESCRIPTION
This library provides a generic interface to the pSOS kernel semaphores.
It translates calls from VxWorks into pSOS calls contained in psosALib (1).
Semaphore mangement includes:

    - create a semaphore,
    - take a semaphore,
    - give a semaphore,
    - delete a semaphore,
    - empty a semaphore.
 
SEMAPHORES
This implementation provides a binary semaphore facility.  Semaphores can
be used to control access to shared devices or data structures, and
to synchronize multiple tasks, or task level and interrupt level processes.

A semaphore is just a memory location that is accessible to all the processes
that need it.  The semaphore must be initialized (by semInit) before it
is ever used.  Thereafter, every time the semaphore is given (semGive) it
becomes full, and every time it is taken (semTake), it becomes empty.

TASK STATE CHANGES
When a task takes a semaphore, that task will become pended if the semaphore
is currently empty.  Any number of tasks may be simultaneously pended on
the same semaphore.  If the semaphore is full when the semTake is done, the
calling task continues running, and the semaphore is made empty.  Only
one task may have a semaphore at a time.

When a task gives a semaphore, that semaphore becomes full.  PSOS immediately
does a scan, to see if any other tasks are currently pended on that semaphore
(because they had done a semTake).  If there are one or more such tasks, the
highest priority one is made ready to run.  If that task is higher priority
than the task doing the semGive, that task will now get the CPU, and the
giving task will be preempted.  If no other tasks, or only lower priority
tasks, are pended on the semaphore, the giving task will keep running.

INTERRUPT LEVEL USE
Semaphores may be given, but not taken, from interrupt level.  An interrupt
level process that gives a semaphore will not be preempted, no matter who
becomes ready, but the task that had been running when the interrupt
occurred may be preempted when the interrupt level code is done.

PROTOCOL HINTS
Although any number of tasks may give, get, and initialize the same semaphore,
you must be careful about when some of these actions are taken.

There is no danger in any number of processes doing semTakes whenever they
like.  SemGive's and semInit's, however, should be controlled.  If semGive's
are done only by one process, there is no problem.  If they are done by
multiple processes, only a process that already has the semaphore should
give it.  Otherwise, multiple copies of the semaphore could be running
around, and mutual-exclusion will no longer be in effect.  SemInit's should
only be performed if some process (perhaps the same one that does the semInit)
is going to do a semGive without doing a semTake first.

DEFICIENCIES
There is no mechanism to give back semaphores automatically
when tasks are suspended or deleted.  If a task ceases execution unexpectedly
(a bus error, deleted from the shell, etc.) any semaphores that it currently
has taken will never be given again, unless there is a watchdog or something
in operation that can do it.  Since the I/O system uses semaphores, this
can cause a device to get hung up if a task doing I/O to that device has
blown up.
 
SEE ALSO: "Architecture", taskLib (1), kernelLib (1), psosALib (1)
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "semLib.h"
#include "memLib.h"
#include "psos.h"

/*******************************************************************************
*
* semCreate - initialize semaphore
*
* Create a semaphore.  The semaphore is initially empty.  After
* initialization, it must be given before it can be taken.
*
* RETURNS: SEM_ID or NULL if unable to allocate memory
*/

SEM_ID semCreate ()

    {
    SEM_ID semId = (SEM_ID) malloc ((unsigned) sizeof (SEMAPHORE));

    if (semId != NULL)
	{
	lstInit (&semId->semList);	/* initialize semaphore list */
	semId->count = 0;		/* initially taken */
	}

    return (semId);			/* return the SEM_ID */
    }
/*******************************************************************************
*
* semGive - give semaphore
*
* Gives the semaphore.  If a higher priority task has already taken
* the semaphore (so that it is now pended waiting for it), that task
* will now become ready to run, and preempt the task that does the semGive.
* If the semaphore is already full (it has been given but not taken) this
* call is essentially a no-op.
*
* SEE ALSO: psosSignalV (2)
*/

VOID semGive (semId)
    FAST SEM_ID semId;	/* semaphore id to give */

    {
    FAST TCBX *pTcbX;
    FAST NODE *pNode;
    FAST int oldLevel;

    if (semId->count == 1)		/* semaphore already given */
	return;

    oldLevel = intLock ();		/* LOCK interrupts */

    if (semId->count == 1)		/* test again now we're locked out */
	intUnlock (oldLevel);		/* UNLOCK interrupts */
    else
	{
	/* semaphore not previously available; give it now */

    	pNode = (NODE *) (lstGet (&semId->semList));	/* dequeue tcbx */
	if (pNode != NULL)
	    {
	    intUnlock (oldLevel);			/* UNLOCK interrupts */
	    pTcbX = (TCBX *) ((int) pNode - OFFSET (TCBX, semNode));
	    psosSignalV (pTcbX->taskId, PSOS_EVENT_14);	/* wake up the task */
	    }
	else
	    {
	    semId->count = 1;		/* give the semaphore */
	    intUnlock (oldLevel);	/* UNLOCK interrupts */
	    }
	}
    }
/*******************************************************************************
*
* semTake - take semaphore
*
* Takes the semaphore.  If the semaphore is empty (it has not been given
* since the last semTake or semInit), this task will become pended until
* the semaphore becomes available by some other process doing a semGive
* of it.  If the semaphore is already available, this call will empty
* the semaphore, so no other process can take it till this task gives it
* back, and this task will continue running.
*
* WARNING
* This routine may not be used from interrupt level.
*
* SEE ALSO: psosWaitV (2)
*/

VOID semTake (semId)
    SEM_ID semId;	/* semaphore id to take */

    {
    int dummy;
    TCBX *pTcbX = taskTcbX (0);
    int oldLevel = intLock ();		/* LOCK interrupts */

    if (semId->count == 0)		/* semaphore is not here */
	{
    	lstAdd (&semId->semList, &pTcbX->semNode);	/* enqueue */
	intUnlock (oldLevel);				/* UNLOCK interrupts */

	/* wait until semaphore is given */
	psosWaitV (PSOS_EVENT_14, PSOS_EVENT_COND_OR, 0, &dummy);
	}
    else
	{
	semId->count = 0;		/* take semaphore */
	intUnlock (oldLevel);		/* UNLOCK interrupts */
	}
    }
/******************************************************************************
*
* semDelete - delete semaphore
*
* Frees the memory allocated for the semaphore.  A deleted semaphore can
* not be resurrected.  If the semaphore was declared as a global variable
* and not created via semCreate then this should NOT be called.
*
* WARNING
* This routine may not be used from interrupt level.
*/  

VOID semDelete (semId)
    SEM_ID semId;	/* semaphore id to delete */

    {
    /* someday we may signal the tasks pended on the semaphore */
    free ((char *) semId);
    }
/******************************************************************************
*
* semClear - take semaphore if semaphore is available
* 
* If the semaphore is available, it is taken, otherwise no action is taken.
* This routine never preempts the caller.  If semaphore was not available
* this routine returns ERROR.
*
* RETURNS: OK or ERROR if semaphore was unavailable
*/  

STATUS semClear (semId)
    SEM_ID semId;	/* semaphore id to empty */

    {
    STATUS status = ERROR;		/* assume the worst */
    int oldLevel = intLock ();		/* LOCK INTERRUPTS */


    if (semId->count != 0)
	{
	semId->count = 0;	/* clear the semaphore count */
	status = OK;		/* we are the only one who has it */
	}

    intUnlock (oldLevel);		/* UNLOCK INTERRUPTS */

    return (status);
    }
/******************************************************************************
*
* semInit - initialize a declared semaphore
*
* In some instances a semaphore cannot be created with semCreate (2)
* but is global variable.  This routine must be called to initialize it.
*
* SEE ALSO: semCreate (2)
*/  

VOID semInit (semId)
    SEM_ID semId;	/* pointer to the semaphore to initialize */

    {
    lstInit (&semId->semList);
    semId->count = 0;
    }
