/* unixLib.c - unix kernal compatability library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01p,15oct88,dnw  changed name of sleep() to ksleep() to avoid confusion
		   with C library routine of same name.
		 changed 'panic' to not suspend unless panicSuspend=TRUE.
01o,22jun88,dnw  removed ovbcopy(), copyin(), copyout(), and imin() which
		   are now macros in systm.h.
		 made spl...() use taskIdCurrent directly.
		 made spl...() return 0/1 for "previous level" instead of
		   task id.
01n,29may88,dnw  removed bcmp() and bzero() now in bLib.
		 changed to v4 names.
01m,04may88,jcf  changed splSem to splSemId, and sem calls for new semLib.
01l,28jan88,jcf  made kernel independent.
01k,23jan88,rdc  brought back spl routines because of size issues.
01j,04jan88,rdc  made spl{net,imp,x} macro's in systm.h
01i,17nov87,dnw  removed printStatus from perror.
01h,13nov87,rdc  added perror;
01g,22oct87,ecs  delinted.
01f,02may87,dnw  added suspend to panic().
01e,03apr87,ecs  added copyright.
01d,02apr87,jlf  delinted.
01c,27feb87,dnw  change myTaskId() to vxMyTaskId().
01b,22dec86,dnw  added bcmp.
01a,28jul86,rdc  written.
*/

/*
DESCRIPTION
This library provides routines that simulate or replace Unix kernel functions
that are used in the network code.  This includes the spl...() processor
level locking routines, wakeup() and sleep() (renamed ksleep() to avoid
confusion with the C library routine called sleep()), perror() and panic(),
and a checksum routine.
*/

#include "semLib.h"
#include "vxWorks.h"
#include "types.h"
#include "mbuf.h"
#include "in.h"
#include "in_systm.h"

IMPORT ULONG taskIdCurrent;	/* id of cur task, faster than taskIdSelf() */

/* GLOBAL VARIABLES */

SEM_ID splSemId;	/* the semaphore used by splnet, splimp, etc */

int splTid;		/* task id of the process who has the 
			 * network processor level semaphore */

BOOL panicSuspend;	/* TRUE = suspend task causing panic */

/******************************************************************************
*
* splnet - set network processor level
*
* splnet may be used to insure mutual exclusion among processes which
* share common network data structures.  In unixland, this is done by
* setting the processor interrupt level.  Here in vxWorksLand, however,
* we use a simple semaphore mechanism.  splx should be called after
* completion of the critical section.  We must check first to see if we're 
* already at network level.
*
* RETURNS: previous "processor level"
*/

int splnet ()

    {
    if (taskIdCurrent == splTid)
	return (1);

    semTake (splSemId);
    splTid = taskIdCurrent;
    return (0);
    }
/******************************************************************************
*
* splimp - set imp processor level
*
* This routine is intentical to splnet().
* Currently, all network interlocking uses the same network semaphore.
*
* RETURNS: previous "processor level"
*/

int splimp ()

    {
    if (taskIdCurrent == splTid)
	return (1);

    semTake (splSemId);
    splTid = taskIdCurrent;
    return (0);
    }
/******************************************************************************
*
* splx - set processor level
*
* splx is used in unix to restore the processor level.
* It is used in vxWorks in conjunction with splnet() and splimp() to provide
* mutual exclusion of network data structures.  splx marks the end of the
* critical section.
*/

VOID splx (x)
    int x;	/* processor level to restore */
    {
    if (x == 0)
	{
	splTid = 0;
	semGive (splSemId);
	}
    }

/***********************************************************************
*
* ksleep -  put a process to sleep
*
* In Unix, processes can sleep on any value (typically an address).
* In VxWorks, we substitute a semaphore mechanism so sleep()
* calls have to be changed to provide the address of a semaphore.
*
* Note: In Unix there are two "sleep" routines.  In the kernel, the sleep
* routine causes a process to block until a wakeup is done on the same
* value that sleep was called with.  This routine, and the wakeup routine
* below, are replacements for that mechanism.  However, in the C library
* there is another sleep() routine that will suspend a process for a
* specified number of seconds.  Since VxWorks users porting an application
* may have calls to the latter but not the former, there is potential for
* vast confusion.  Thus this routine has been renamed "ksleep" ("kernel sleep")
* and all calls to it in network code have been changed.
*/

VOID ksleep (semId)
    SEM_ID semId;

    {
    BOOL hadSplSem;

    /* first see if we've got the network semaphore (splSemId);
     * if so, give it, and then take it back when we are awakened. */

    hadSplSem = (splTid == taskIdCurrent);
    if (hadSplSem)
	{
	splTid = 0;
	semGive (splSemId);
	}

    semTake (semId);

    if (hadSplSem)
	{
	semTake (splSemId);
	splTid = taskIdCurrent;
	}

    }

/******************************************************************************
*
* wakeup - wakeup a sleeping process
*/

VOID wakeup (semId)
    SEM_ID semId;
    {
    semGive (semId);
    }

/******************************************************************************
*
* in_cksum - Checksum routine for Internet Protocol family headers 
*/

in_cksum(m, len)
    FAST struct mbuf *m;
    FAST int len;
    {
    FAST u_short sum = 0;
    FAST int mlen = 0;
    FAST int slen = 0;

    while ((m != NULL) && (len != 0))
	{
	if ((mlen = m->m_len) > 0)
	    {
	    if (len < mlen)
		mlen = len;
	    sum = cksum (sum, mtod(m, u_short *), mlen, slen);
	    slen += mlen;
	    len -= mlen;
	    }
	m = m->m_next;
	}

    if (len != 0)
	logMsg ("cksum: out of data\n");

    return (~sum);
    }

/******************************************************************************
*
* perror - print error
*
* perror prints the given string plus the calling task's taskid and current
* status value on stderr.
*
*/

VOID perror (s)
    char *s;
    {
    printErr ("\nt%d: %s; status=0x%x\n", taskIdSelf (), s, errnoGet ());
    }

/******************************************************************************
*
* panic - something is very wrong
*
* panic simply logs a message to the console prepended with "panic".
* If the panic occurs at task level, the current task is suspended.
*/

VOID panic (msg)
    char *msg;
    {
    logMsg ("panic: %s\n", msg);

    if (!intContext () && panicSuspend)
	taskSuspend (0);
    }
