/* if_ring.c - network interface ring routines */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
02c,03apr87,ecs  added copyright.
02b,27mar87,rdc  changed all tas() to sysBusTas().
02a,02feb87,jlf  Removed CLEAN ifdefs.
*/

#include "if_ring.h"

#define	DELAY_COUNT	5000	/* how high to count when waiting on TAS */

/*
 * Initialize the ring pointed to by "rp" to contain "size" slots.
 *  The ring is set up to be empty and unlocked.
 *  Size had better be a power of 2!
 */
ringinit(rp, size)
  register struct ring *rp;
  short size;
  {
	rp->r_tas = rp->r_rdidx = rp->r_wrtidx = 0;
	rp->r_size = size;
}

/*
 * Add a new entry "v" to the ring "rp".
 *  Returns 0 if ring was full, else the new number of entries in the ring.
 *  Uses test-and-set routine, allowing multi-processor interlocked access
 *  to the ring IF the ring is in VME memory.  
 *  Masks all interrupts, in order to prevent reentrancy and to
 *  minimize ring lockout.
 */
int
ringput(rp, v)
  register struct ring *rp;
  int v;
  {
	register int x = DELAY_COUNT;	/* don't wait forever for lock */
	register int s;

	for (;;) {		/* disable interrupts, then lock the ring */
		s = intLock ();
		if (sysBusTas(&rp->r_tas))
			break;
		intUnlock (s);
		if (--x == 0)
			return 0;	/* ring is stuck! */
	}
	if ((x = (rp->r_wrtidx + 1) & (rp->r_size - 1)) != rp->r_rdidx) {
		rp->r_slot[rp->r_wrtidx] = v;
		rp->r_wrtidx = x;
	}				/* if x == rdidx then ring is full */
	if ((x -= rp->r_rdidx) < 0)
		x += rp->r_size;
	rp->r_tas = 0;			/* unlock ring */
	intUnlock (s);
	return x;	/* rtrn new number of entries, or 0 if ring was full */
}

/*
 * Extract next entry from the ring "rp".
 *  Returns 0 if ring is empty, else the extracted entry.
 *  Uses test-and-set routine, allowing multi-processor interlocked access
 *  to the ring IF the ring is in VME memory.  
 *  Masks all interrupts, in order to prevent reentrancy and to
 *  minimize ring lockout.
 */
int
ringget(rp)
  register struct ring *rp;
  {
	register int x = DELAY_COUNT;	/* don't wait forever for lock */
	register int s;

	for (;;) {		/* disable interrupts, then lock the ring */
		s = intLock ();
		if (sysBusTas(&rp->r_tas))
			break;
		intUnlock (s);
		if (--x == 0)
			return 0;	/* ring is stuck! */
	}

	if (rp->r_rdidx != rp->r_wrtidx) {
		x = rp->r_slot[rp->r_rdidx];
		rp->r_rdidx = (rp->r_rdidx + 1) & (rp->r_size - 1);
	} else
		x = 0;
	rp->r_tas = 0;			/* unlock ring */
	intUnlock (s);
	return x;	/* return 0 if ring was empty; else extracted value */
}

int
ringempty(rp)
  register struct ring *rp;
  {
	return rp->r_rdidx == rp->r_wrtidx;
}
