/* if_ie.c - SUN 3/e IE ethernet driver */

/* Copyright (c) 1986 by Sun Microsystems, Inc.  */

/*
modification history
--------------------
01l,05aug88,gae  cleaned up. Doesn't use memory manager. Allows multiple units.
01k,29jul88,gae  now works under 4.0... didn't malloc the scp, scb etc..
		 and reduced memory partition respectively.
01j,24jun88,rdc  added IFF_BROADCAST to if_flags.
01i,13jun88,dnw  changed memPartInit to memPartCreate.
		 fixed missing intUnlocks to iecustart().
01h,12jun88,dnw  changed cpybytes to bcopyBytes.
01g,30may88,dnw  changed to v4 names.
01f,21apr88,gae  added include of ioLib.h for READ/WRITE/UPDATE.
01e,28jan88,jcf  made kernel independent.
01d,18nov87,dnw  changed ieattach to take int vec num instead of vec adrs.
01c,17nov87,jcf  changed iereset to work.
01b,29oct87,rdc  fixed race due to interrupts not being locked out at splnet.
01a,23oct87,rdc  written.
*/


/*
DESCRIPTION:

Sun Intel Ethernet Controller interface driver.

The only user callable routine is:

   ieattach (unit, addr, ivec, ilevel)

INTERNAL

Memory layout:

	----------------- <--- TOP = base+0x20000
	|     SCP	| <--- base + 0x20000 - 0xa
	-----------------
	-----------------
	|		| <----	end of partition
	~		~
	|		|
	|  MEM_SIZE-SCP	|	MEM_SIZE = 0x200000
	|		|
	| tbuf's 24 bit	|
	| rbuf's 24 bit	|
	| trfd's 16 bit	|
	| rrfd's 16 bit	|
	|		|
	| scb		|
	| iscp		|
	| cbnop		|
	| cbsync	|
	|		| <---	base memory partition = IE_ADDR + 0x20000
	-----------------	MEM_OFFSET = 0x200000
	-----------------
	|		|
	~		~
	|		|
	-----------------
	-----------------
	|  CSR		| <---	+ 0x01ff02
	-----------------
	-----------------
	|		|
	~		~
	| IE_ADDR	| <---	0x00300000
	-----------------
*/

/*
 * Notes on the driver's data structures
 * (see also definitions in if_ievar.h)
 *
 * Transmit Buffer Descriptor (tbd) organization:
 *	Tbds are configured as a ring carved out of a contiguous chunk
 *	of the control block map.  The ring contains two regions, the
 *	set of descriptors currently in use and the set of free descriptors.
 *	The driver maintains pointers to the start of these two regions
 *	(respectively, es_utbd and es_ftbd).  The routine iexmitdone
 *	reclaims in-use tbds, converting them back into free tbds, and
 *	ietbdget does the opposite.  Both routines treat the descriptor
 *	ring as a pair of queues, with tbds moving from the front of one
 *	queue to the end of the other.
 *
 *	The organization above has the property that link fields are
 *	never changed once they've been initialized.  The only operation
 *	on them is to move to the next tbd.  This property allows the
 *	time optimization of adding a ietbd_link field that's redundant
 *	with the chip-recognized ietbd_next field, but that is a regular
 *	pointer instead of an offset from the chip's memory base.  The
 *	driver can move to the next descriptor more quickly using this
 *	field.
 *
 *	The tbd ring is allocated in ieinitmem, which calls ie_tbdinit
 *	to do the actual work.
 *
 * Transmit Frame Descriptor (tfd) organization:
 *	Tfds are configured as a pool with free descriptors chained
 *	into a free list through the ietfd_flink field.  As with tbds
 *	and tbufs, the routines iestartout and iexmitdone respectively
 *	consume and free tfds.
 *
 *	Tfds can almost be organized into a ring in the same way tbds
 *	are.  The sticking point is that other command block types
 *	can (and do) intervene between consecutive tfds, the culprits
 *	being the no-op commands generated by iedog.
 *
 *	Tfds are allocated in ieinitmem, which calls ie_tbdinit to do
 *	the actual work.
 *
 * Transmit Buffer (tbuf) organization:
 *	Wherever possible the driver tries to transmit directly from the
 *	mbufs constituting an output packet.  However, this isn't always
 *	possible, so the driver uses tbufs to hold non-mbuf packet data.
 *	Each tbuf is large enough to hold a complete packet and, when in
 *	use, contains a complete trailing packet suffix -- that is, all
 *	of the packet from some point on.  Thus, there's at most one tbuf
 *	per outgoing packet.  Free tbufs are chained together into a free
 *	list rooted at es_tbuf_free.  When a tbuf is in use, it's referenced
 *	from two places, the last tbd of the corresponding packet chain
 *	and from the packet's tfd.  The tfd reference is redundant, but
 *	convenient for reclaiming the tbuf when transmission is complete
 *	or requeueing is required.
 *
 *	Tbufs are allocated in ieinitmem, which calls ie_tbufinit to do
 *	the actual work.
 *
 * Receive Buffer Descriptor (rbd) organization:
 *	Rbds are organized into a ring containing an embedded list that
 *	completely fills the ring.  The list itself has two halves.  The
 *	front half contains rbds that the chip has filled from incoming
 *	packets, but that the driver has not yet processed.  The back half
 *	contains empty rbds that the chip has not yet filled.  The list
 *	head always points to the next rbd the driver is to process.  The
 *	chip keeps track of the start of the back half of the list; the
 *	driver maintains no explicit record of its location, but can
 *	determine that it has reached it by discovering an rbd with its
 *	ierbd_used field cleared.
 *
 * Receive Frame Descriptor (rfd) organization:
 *	Rfds are organized completely analogously to rbds.
 *
 * Receive Buffer (rbuf) organization:
 *	Receive buffers can be in one of three states:
 *		free:		inactive and available for use
 *		attached:	awaiting packet reception and visible to
 *				the chip as a packet receptacle
 *		loaned:		contains packet data that is being processed
 *				by the higher-level protocol code
 *	Free rbufs are chained together through their ierbuf_next fields
 *	into a free list rooted at es_rbuf_free.  Attached rbufs can be
 *	found only by working back from the descriptors in the rbd ring.
 *	Loaned buffers can be detected only by virtue of their not being
 *	in either of the other two states.  However, we do maintain a
 *	count of loaned rbufs in es_loaned, so that we can tell whether
 *	or not it's safe to reallocate driver memory.
 *
 *	Note that driver memory can be (re)allocated only when no rbufs
 *	are on loan.
 */

#include "vxWorks.h"
#include "iv68k.h"
#include "memLib.h"
#include "ioLib.h"
#include "param.h"
#include "mbuf.h"
#include "socket.h"
#include "errno.h"
#include "utime.h"
#include "ioctl.h"
#include "if.h"
#include "in_systm.h"
#include "if_ether.h"
#include "if_ievar.h"
#include "in_var.h"
#include "ip.h"

#include "if_iereg.h"
#include "if_tie.h"
/*
#include "if_mie.h"
#include "if_obie.h"
*/

IMPORT VOID ipintr ();
IMPORT char ieEnetAddr[];

#define NIE 2		/* max number of ie controllers */

/* controller types */

#define	IE_MB	1	/* Multibus (possibly embedded in VME) */
#define	IE_OB	2	/* On main CPU board ("On-Board") */
#define	IE_TE	3	/* 3E Ethernet board "Three E" */

/* also see commented out include's above */
#undef	IE_MB
#undef	IE_OB

#define	DEBUGIE	/* conditional compilation symbol for debugging */

#undef	DEBUGIE

/* Controls whether ether header lives in tfd/rfd or in start
 * of mbuf chain/buffer for packet.  Current thinking is that it's
 * more efficient to keep it in the mbuf chain, as this
 * conforms to assumptions made elsewhere about data placement
 * in the first mbuf and thus lets ether_pullup avoid allocating
 * a fresh mbuf when transmitting.
 */

#define	IE_FDADDRx

/* delay values */

#define IEDELAY		400000	/* delay period (in us) before giving up */
#define IEKLUDGE	    20	/* delay period (in us) to make chip work */

#define	CBSYNC_SIZE	(max(max((max(sizeof(struct iecb),\
			    sizeof(struct ieconf))),sizeof(struct ietdr)),\
			    sizeof(struct ieiaddr)))

#define	CBMISC_SIZE	(sizeof(struct iecb)+sizeof(struct ieiscp)+\
			 sizeof(struct iescb))


STATUS ieattach ();
LOCAL int ieinit ();
LOCAL int ieioctl ();
LOCAL int ieoutput ();
LOCAL int iereset ();
LOCAL int ieintr ();
LOCAL struct mbuf *ieget ();
LOCAL VOID ieintrTL ();
LOCAL VOID iestat ();
LOCAL char *ie_alloc ();
LOCAL struct mbuf *copy_to_mbufs ();
LOCAL caddr_t from_ieaddr ();
LOCAL caddr_t from_ieoff ();
LOCAL ieaddr_t to_ieaddr ();
LOCAL ieaddr_t to_ieaddr_opt ();
LOCAL ieoff_t to_ieoff ();
LOCAL ieint_t to_ieint ();

#define from_ieint	to_ieint


/* ethernet software status per interface */
LOCAL struct ie_softc *ie_softc [NIE];

/* initial values for resource allocation (patchable)  */

int	ie_tbds  = 128;	/* Generous, but they're cheap */
int	ie_tfds  =  40;	/* Ditto */
int	ie_tbufs =  15;	/* Probably too many */
int	ie_rbds  = 40;	/* 11 quite adequate except possibly for promisc. */
int	ie_rfds  = 35;	/* Must be < ie_rbds to avoid suspected chip bug. */
int	ie_rbufs = 45;	/* Must be >= ie_rbds */

/* set to 1 to enable retransmission after the chip detects
 * a failed packet transmission attempt.
 */
int	ie_retransmit = 1;

/* Reserve and release the statically allocated CB for synchronously
 * executed commands.  Note that the space allocated for es_cbsync must
 * be large enough to hold the largest CB type synccb_res is called with.
 * Once we're completely certain that calls to iesynccmd
 * can't/don't nest, eliminate the counter manipulation.
 */
#ifdef	DEBUGIE
#define synccb_res(es, t) \
			((es)->es_synccmd ? (t)0 : \
				((es)->es_synccmd = 1, (t)(es)->es_cbsync))
#define synccb_rel(es)	((es)->es_synccmd = 0)
#else	DEBUGIE
#define synccb_res(es, t) \
			((t)(es)->es_cbsync)
#define synccb_rel(es)
#endif	DEBUGIE


#ifdef	DEBUGIE
/* Verify that X is within the range reachable
 * from es->es_base by a 16-bit chip offset.
 */
#define	ie_chkbounds(X) \
{ \
	caddr_t	y = (caddr_t)(X); \
\
	if (y < es->es_base || y > es->es_base + 0xffff) { \
		printf("X"); \
		printf(" at %x: not addressible (base at %x)\n", \
			y, es->es_base); \
		panic ("ie_alloc botch"); \
	} \
}
#endif	DEBUGIE

#define CDELAY(c, n) { \
			int nTicks = n * sysClkRateGet () / 1000000; \
			while (nTicks-- > 0) \
			    { \
			    if (c) \
				break; \
			    taskDelay (1); \
			    } \
			} 


/*******************************************************************************
*
* ieattach -
*
* Interface exists; make available by filling in network interface
* record.  System will initialize the interface when it is ready
* to accept packets.
*
* RETURNS: OK or ERROR
*/

STATUS ieattach (unit, addr, ivec, ilev)
    int unit;
    caddr_t addr;
    int ivec;
    int ilev;

    {
    FAST struct ifnet *ifp;
    FAST struct ie_softc *es;

    /* allocate storage for softc for this unit */

    es = (struct ie_softc *)malloc (sizeof (struct ie_softc));

    if (es == NULL)
	return (ERROR);

    bzero (es, sizeof (struct ie_softc));

    ie_softc[unit] = es;

#ifdef	DEBUGIE
    printf ("ieattach: debugging on\n");
#endif	DEBUGIE

    /* Record chip address. At the same time, set up the interrupt
     * vector for the 3E board by writing the vector to the ivec
     * register on the board.
     */

    if (ieprobe (addr, unit) <= 0 ||
	intConnect (INUM_TO_IVEC (ivec), ieintr, unit) == ERROR)
	{
	(void) free ((char *)es);
	return (ERROR);
	}

    /* set minimum memory consumption */

    es->es_base = (caddr_t) addr + IE_TE_MEM_OFFSET;

    es->es_tie = (struct tie_device *) (addr + IE_TE_CSREG_OFFSET);
    es->es_tie->tie_ivec = ivec;

    /* XXX FAKE OUR ETHERNET ADDR */

    bcopy (ieEnetAddr, (char *)es->es_enaddr, 6);

    ifp = &es->es_if;

    ifp->if_unit   = unit;
    ifp->if_name   = "ie";
    ifp->if_mtu    = ETHERMTU;
    ifp->if_init   = ieinit;
    ifp->if_ioctl  = ieioctl;
    ifp->if_output = ieoutput;
    ifp->if_reset  = iereset;
    ifp->if_flags  = IFF_BROADCAST;

    /* setup chip memory buffers (after unit's been set) */

    iegetmem (es);
    ieinitmem (es);

    sysIntEnable (ilev);

    iechipreset (es);

    if_attach (ifp);

    return (OK);
    }
/*******************************************************************************
* ieprobe - probe for device
*
* The 3E board can be distinguished from the Multibus board by the bits
* in the csr that always read as zero.  The board is configured so that
* reg points directly to the csr register.
*
* Also, for the 3E board, we map in more bytes (0x11 vs 0x1) so that the
* interrupt vector is mapped in also.
*
* RETURNS: 0 if device not present
*/

LOCAL int ieprobe (reg, unit)
    caddr_t	reg;
    int	unit;

    {
    register struct ie_softc *es = ie_softc[unit];
    register struct tie_device *tie = (struct tie_device *)reg;
    int value;

    /* Look for the 3E board.  We write all 0xffff
     * and read it back.  Most of the bits in the csr
     * are always 0, so we should get 0xf000 for
     * the 3E board.
     *
     * The bit at 0x0800 is the interrupt line from the
     * 82586.  It may sometimes be on, but we expect it to
     * be off when probe is called.  This shuld always be true
     * when we boot.
     */

    value = 0xffff;
    if (vxMemProbe ((char*)tie, WRITE, 2, (char *)&value) == ERROR) 
	return (0);
    
    es->es_type = IE_TE;

    return (sizeof (struct tie_device));
    }
/*******************************************************************************
*
* iereset - reset interface following a system reset
*/

LOCAL VOID iereset (unit)
    int unit;

    {
    FAST struct ie_softc *es = ie_softc[unit];

    es->es_if.if_flags &= ~IFF_RUNNING;

    iechipreset (es);
    }
/*******************************************************************************
*
* ieinit -
*
* Unconditionally restart the interface from ground zero.
* If the driver's memory resources haven't yet been allocated,
* do it here.
*/

LOCAL VOID ieinit (unit)
    int unit;

    {
    struct ie_softc *es = ie_softc[unit];
    register struct mbuf *m;
    int s = splnet();

    iechipreset(es);

    es->es_if.if_flags &= ~(IFF_UP|IFF_RUNNING|IFF_PROMISC);

    /* Drain the output queue, against the possibility that
     * something has occurred to invalidate the packets in it
     * (such as our Ethernet address changing).
     */
    while (es->es_if.if_snd.ifq_head)
	{
	IF_DEQUEUE(&es->es_if.if_snd, m);
#ifdef	DEBUGIE
	iechkmbuf (m);
#endif	DEBUGIE
	m_freem (m);
	}

    es->es_cbhead = NULL;

    if (!iechipinit(es))
	goto exit;

    iedoaddr (es);
    /* XXX: should check with DUMP */

    /* hang out receive buffers and start any pending writes */
    es->es_if.if_flags |= IFF_UP|IFF_RUNNING;
    ierustart (es);
    iesplice (es);
    iestartout (unit);

exit:
    (void) splx(s);
    }
/*******************************************************************************
*
* ieintr - handle ethernet interrupts
*/

LOCAL VOID ieintr (unit)
    int unit;

    {
    register struct ie_softc *es = ie_softc[unit];
    register struct iescb *scb = es->es_scb;
    register int cmd;

    iechkcca (es);

    if (scb->ie_cmd != 0)
	{
	logMsg ("ieintr: lost sync\n");
	(void) netJobAdd (iestat, es);
	ieca (es); /* XXX */
	return;
	}

    /* This can be done faster with something like:
     * cmd = (scb->ie_status) &
     *	    (IECMD_ACK_CX|IECMD_ACK_FR|IECMD_ACK_CNR|IECMD_ACK_RNR);
     */
    cmd = 0;
    if (scb->ie_cx)
	cmd |= IECMD_ACK_CX;
    if (scb->ie_fr)
	cmd |= IECMD_ACK_FR;
    if (scb->ie_cnr)
	cmd |= IECMD_ACK_CNR;
    if (scb->ie_rnr)
	cmd |= IECMD_ACK_RNR;

    if (cmd == 0) 
	{
#ifdef DEBUGIE
	logMsg ("ieintr: spurious interrupt\n");
#endif DEBUGIE
	cmd = IECMD_ACK_CX+IECMD_ACK_FR+IECMD_ACK_CNR+IECMD_ACK_RNR;
	}

    scb->ie_cmd = cmd;

    (void) netJobAdd (ieintrTL, unit, cmd);
    ieca (es);
    }
/*******************************************************************************
*
* ieintrTL - interrupt handling at task level
*/

LOCAL VOID ieintrTL (unit, cmd)
    FAST int unit;
    register int cmd;

    {
    register struct ie_softc *es = ie_softc[unit];

    /* if interface is not just right, avoid touching cbs */
    if ((es->es_if.if_flags&(IFF_UP|IFF_RUNNING)) != (IFF_UP|IFF_RUNNING))
	{
	printf ("ieintrTL: interface not right\n"); /* XXX */
	return;
	}

    if (cmd & (IECMD_ACK_RNR|IECMD_ACK_FR))
	ierecv (es);

    if (cmd & (IECMD_ACK_CNR|IECMD_ACK_CX))
	iecbdone (es);
    else if (es->es_cbhead == NULL && es->es_if.if_snd.ifq_head)
	{
	printf ("ieintrTL: lost interrupt, resetting\n");

	/* Experience shows that proceeding normally doesn't
	 * suffice, so hit the chip with a hammer.
	 */

	ieinit (unit);
	}
    }
/*******************************************************************************
*
* ieioctl - process an ioctl request
*
* RETURNS: ERROR
*/

LOCAL STATUS ieioctl (ifp, cmd, data)
    register struct ifnet *ifp;
    int cmd;
    caddr_t data;

    {
    int unit = ifp->if_unit;
    register struct ie_softc *es = ie_softc[unit];
    FAST struct ifaddr *ifa = (struct ifaddr *) data;
    int error = 0;
    int s;

    s = splnet();

    switch (cmd)
	{
	case SIOCSIFADDR:
	    ifp->if_flags |= IFF_UP;
	    ieinit (ifp->if_unit);

	    switch (ifa->ifa_addr.sa_family)
		{
#ifdef INET
		case AF_INET:
		    ((struct arpcom *)ifp)->ac_ipaddr = IA_SIN(ifa)->sin_addr;
		    arpwhohas ((struct arpcom *)ifp, &IA_SIN(ifa)->sin_addr);
		    break;
#endif
		}

	case SIOCGIFADDR:
		bcopy ((char *) &es->es_enaddr[0],
		       (char *) &((struct ifreq *)data)->ifr_addr.sa_data[0],
		       6);
		break;

	case SIOCGIFFLAGS:
	    *(short *)data = ifp->if_flags;
	    break;

	case SIOCSIFFLAGS:
	    {
	    short flags = *(short *)data;
	    int up;
	    int promisc;

	    up = flags & IFF_UP;

	    if (up != (ifp->if_flags & IFF_UP))
		ieinit(unit); /* going up/down */

	    if (ifp->if_flags & IFF_UP)
		flags |= (IFF_UP|IFF_RUNNING);
	    else
		flags &= ~(IFF_UP|IFF_RUNNING);

	    promisc = flags & IFF_PROMISC;

	    if (promisc != (ifp->if_flags & IFF_PROMISC))
		{
		if ((ifp->if_flags & IFF_UP) == 0)
		    {
		    flags &= ~IFF_PROMISC;
		    error = ENETDOWN;
		    }
		else
		    {
		    register struct ieconf *ic;

		    ic = synccb_res(es, struct ieconf *);

#ifdef	DEBUGIE
		    if (!ic)
			panic ("ieioctl: nested iesynccmd");
#endif	DEBUGIE

		    iedefaultconf (ic);

		    ic->ieconf_promisc = promisc ? 1 : 0;
		    iesynccmd (es, &ic->ieconf_cb);
		    synccb_rel (es);
		    }
		}

	    ifp->if_flags = flags;
	    }
	break;

	default:
	    error = EINVAL;
	    break;
	}

    (void) splx(s);
    return (error);
    }
/*******************************************************************************
*
* iedoaddr - tell the chip its ethernet address
*/

LOCAL VOID iedoaddr (es)
    register struct ie_softc *es;

    {
    register struct ieiaddr *iad;

    iad = synccb_res (es, struct ieiaddr *);

#ifdef	DEBUGIE
    if (!iad)
	panic ("iedoaddr: iad");
#endif	DEBUGIE

    bzero((caddr_t)iad, sizeof (struct ieiaddr));

    iad->ieia_cb.iec_cmd = IE_IADDR;
    bcopy ((char *)es->es_enaddr, (char *)iad->ieia_addr, 6);
    iesynccmd (es, &iad->ieia_cb);
    synccb_rel (es);
    }
/*******************************************************************************
*
* ierecv -
*
* Process completed input packets and recycle resources;
* only called from interrupt level by ieintr.
*
* If we could count on there being exactly one receive buffer
* per receive frame, this code could be simplified.  The rest
* of the driver conforms to this assumption.  Seemingly, the
* only way it could be violated is by reception of a giant
* packet.  Todo: instrument to see whether we ever do receive
* multiple buffer frames.
*/

LOCAL VOID ierecv (es)
    register struct ie_softc *es;

    {
    int eof;
    int e;
    register struct ierbd *rbd;
    register struct iescb *scb = es->es_scb;
    register struct ierfd *rfd = es->es_rfdhd;

    if (!rfd)			/* not initialized */
	return;

top:
    /* process each received frame */

    while (rfd->ierfd_done)
	{
	ieread (es, rfd);

	/* reclaim resources associated with the frame */

	if (rfd->ierfd_rbd != IENULLOFF)
	    {
	    rbd = es->es_rbdhd;
#ifdef	DEBUGIE
	    /* Sanity check.  Verify that we agree with the
	     * frame descriptor about which rbd is current.
	     */

	    if (rbd != (struct ierbd *) from_ieoff(es, (ieoff_t)rfd->ierfd_rbd))
		panic ("ierecv rbd");

#endif	DEBUGIE
	    /*
	     * Gobble up all rbd/buffer pairs associated with
	     * the frame, stopping if we catch up to the spot
	     * where the chip is filling buffers.
	     */
	    while (rbd && rbd->ierbd_used)
		{

#ifdef	DEBUGIE
		/* Sanity check.  Verify that the current rfd
		 * points to the head of the filled rbd queue.
		 */
		if (rbd != es->es_rbdhd)
		    panic ("ierecv rbdhd");
#endif	DEBUGIE

		/* Advance the rbd list head and tail one
		 * notch around the ring.
		 */
		rbd->ierbd_el = 1;
		eof = rbd->ierbd_eof;
		rbd->ierbd_status = 0;

		/* Put rbd/buffer back onto receive list. */

		es->es_rbdtl->ierbd_el = 0;
		es->es_rbdtl = rbd;
		rbd = es->es_rbdhd = rbd->ierbd_link;

		if (eof)
		    break;
		}
	    }

	/*
	 * Advance the rfd list head and tail
	 * one notch around the ring.
	 */
	rfd->ierfd_rbd = IENULLOFF;
	rfd->ierfd_el = 1;
	rfd->ierfd_status = 0;
	/* Put frame descriptor back onto receive list. */
	es->es_rfdtl->ierfd_el = 0;
	es->es_rfdtl = rfd;
	rfd = es->es_rfdhd = rfd->ierfd_link;
	}

    /* Update cumulative statistics.  (These relate to everything
     * that's come in since we last updated, not to a single packet.)
     */

    if (e = scb->ie_crcerrs)
	{
	/* count of CRC errors */
	scb->ie_crcerrs = 0;
	e = from_ieint(e);
	es->es_if.if_ierrors += e;
	es->es_ierr.crc += e;
	}
    if (e = scb->ie_alnerrs)
	{
	/* count of alignment errors */
	scb->ie_alnerrs = 0;
	e = from_ieint(e);
	es->es_if.if_ierrors += e;
	es->es_ierr.aln += e;
	}
    if (e = scb->ie_rscerrs)
	{
	/* count of discarded packets */
	scb->ie_rscerrs = 0;
	e = from_ieint(e);
	es->es_if.if_ierrors += e;
	es->es_ierr.rsc += e;
	}
    if (e = scb->ie_ovrnerrs)
	{
	/* count of overrun packets */
	scb->ie_ovrnerrs = 0;
	e = from_ieint(e);
	es->es_if.if_ierrors += e;
	es->es_ierr.ovr += e;
	}

    /* If the receive unit is still chugging along, we're done.
     * Otherwise, it must have caught up with us and entered
     * the No Resources state, and we must restart it.
     */

    if (scb->ie_rus == IERUS_READY)
	    return;

    es->es_runotready++;		/* important instrumentation var! */

    /* the following test must be made when we know chip is quiet */

    if (es->es_rfdhd->ierfd_done)
	goto top;		/* more snuck in */

    /* reset our data structures and restart */

    es->es_rfdhd->ierfd_rbd = to_ieoff(es, (caddr_t)es->es_rbdhd);
    iechkcca (es);

    scb->ie_rfa = to_ieoff(es, (caddr_t)es->es_rfdhd);
    scb->ie_cmd = IECMD_RU_START;

    ieca (es);
    }
/*******************************************************************************
*
* ieread - move info from driver toward protocol interface
*/

LOCAL VOID ieread (es, rfd)
    register struct ie_softc *es;
    register struct ierfd *rfd;

    {
    register struct ierbd *rbd;
    struct ether_header	*header;
    struct mbuf *m;
    int length, off;
    int resid;
    caddr_t buffer;
    FAST struct ifqueue *inq;

    es->es_if.if_ipackets++;

    if (!rfd->ierfd_ok)
	{
	es->es_if.if_ierrors++;
	printf("receive error\n");
	return;
	}

    if (rfd->ierfd_rbd == IENULLOFF)
	panic ("ieread: rbd = IENULLOFF\n");
    else
	{
	rbd = (struct ierbd *)from_ieoff(es, (ieoff_t)rfd->ierfd_rbd);

	/* There is more work we would do here if the rbs
	 * were smaller than IE_BUFSZ, namely chase down
	 * the segments; but for ease of programming we
	 * allocate the receiver buffers large enough
	 * so that the input packet is contiguous.
	 */

	if (!rbd->ierbd_eof)
	    {
	    /* length > IE_BUFSZ  */
	    printf("giant packet\n");
	    es->es_if.if_ierrors++;
	    return;
	    }

	length = (rbd->ierbd_cnthi << 8) + rbd->ierbd_cntlo;
	}

#ifdef	IE_FDADDR
    header = (struct ether_header *)rfd->ierfd_dhost;
    buffer = from_ieaddr(es, rbd->ierbd_buf);

#else	IE_FDADDR
    header = (struct ether_header *) from_ieaddr(es, (ieaddr_t) rbd->ierbd_buf);
    buffer = ((caddr_t) header) + sizeof (struct ether_header);

    if (length >= sizeof (struct ether_header))
	length -= sizeof (struct ether_header);
    else
	length = 0;
#endif	IE_FDADDR

    header->ether_type = ntohs((u_short)header->ether_type);

#define	exdataaddr(header, off, type)	((type)(((caddr_t)((header)+1)+(off))))

    if (header->ether_type >= ETHERTYPE_TRAIL &&
	header->ether_type < ETHERTYPE_TRAIL+ETHERTYPE_NTRAILER)
	{
	off = (header->ether_type - ETHERTYPE_TRAIL) * 512;

	if (off >= ETHERMTU)
	    return;		/* sanity */

	header->ether_type = ntohs(*exdataaddr(header, off, u_short *));
	resid = ntohs(*(exdataaddr(header, off+2, u_short *)));

	if (off + resid > length)
	    return;		/* sanity */

	length = off + resid;
	}
    else
	off = 0;

    if (length == 0)
	return;		/* sanity */

    m = (struct mbuf *) ieget (buffer, length, off, &es->es_if);

    if (m == 0)
	return;
	
    if (off)
	{
	struct ifnet *ifp;

	ifp = *(mtod(m, struct ifnet **));
	m->m_off += 2 * sizeof (u_short);
	m->m_len -= 2 * sizeof (u_short);
	*(mtod(m, struct ifnet **)) = ifp;
	}

#ifdef	DEBUGIE
	iechkmbuf(m);
#endif	DEBUGIE

    switch (header->ether_type)
	{
#ifdef INET
	case ETHERTYPE_IP:
	    (void) netJobAdd (ipintr, 0);
	    inq = &ipintrq;
	    break;

	case ETHERTYPE_ARP:
	    arpinput(&es->es_ac, m);
	    return;

#endif
	default:
	    m_freem(m);
	    return;
	}

    {
    int s = splnet (); 

    if (IF_QFULL(inq))
	{
	IF_DROP(inq);
	m_freem(m);
	splx (s); 
	return;
	}

    IF_ENQUEUE(inq, m);

    splx (s); 
    }
    }

/* allocation and deallocation routines for descriptors and buffers */

/*******************************************************************************
*
* ie_alloc - simple-minded allocator for ie driver
*/

LOCAL char *ie_alloc (es, len)
    FAST struct ie_softc *es;
    FAST int len;

    {
    FAST char *mem = es->es_memspace;

    if ((int)(es->es_memspace += len) > (int)(es->es_base + es->es_memsize))
	panic ("ie_alloc: out of memory");

    return (mem);
    }
/*******************************************************************************
*
* ie_tbdinit -
*
* Allocate and initialize the ring of transmit buffer descriptors.
* Assumes that the ring is currently unallocated.
*/

LOCAL VOID ie_tbdinit(es, ntbds)
    register struct ie_softc *es;
    register int ntbds;

    {
    struct ietbd *tbd_start;
    register struct ietbd *tbd;
    int tbd_rsize;

    if (ntbds <= 0)
	panic ("ie_tbdinit: ridiculous request");

    es->es_ftbds = es->es_tbds = ntbds;

    /* grab a chunk of control block space to hold the descriptor ring */

    tbd_rsize = ntbds * sizeof (struct ietbd);
    tbd_start = (struct ietbd *) ie_alloc (es, tbd_rsize);

    es->es_tbdring = tbd_start;

#ifdef	DEBUGIE
    ie_chkbounds (es->es_tbdring);
    ie_chkbounds (es->es_tbdring+es->es_tfds-1);
#endif	DEBIGIE

    /* Link the tbds together into a ring, setting up the
     * ietbd_next and ietbd_link fields in tandem.
     */
    for (tbd = tbd_start; tbd < tbd_start + ntbds - 1; )
	{
	register struct ietbd *ptbd = tbd++;

	ptbd->ietbd_next = to_ieoff (es, (caddr_t)tbd);
	ptbd->ietbd_link = tbd;
	}

    tbd->ietbd_next = to_ieoff (es, (caddr_t)tbd_start);
    tbd->ietbd_link = tbd_start;

    /* initialize queue head pointers */
    es->es_utbd = es->es_ftbd = tbd_start;
    }
/*******************************************************************************
*
* ie_tfdinit -
*
* Allocate and initialize the pool of transmit frame descriptors.
* Assumes the pool is currently unallocated.
*/

LOCAL VOID ie_tfdinit(es, ntfds)
    register struct ie_softc *es;
    register int ntfds;

    {
    register struct ietfd *tfd_pool;
    int tfd_psize;
    register int i;

    if (ntfds <= 0)
	panic ("ie_tfdinit: ridiculous request");

    es->es_tfds = ntfds;

    /* carve the pool out of the resource map */

    tfd_psize = ntfds * sizeof (struct ietfd);
    tfd_pool = (struct ietfd *) ie_alloc (es, tfd_psize);

    if (!tfd_pool)
	panic ("ie_tfdinit: no rmap space");

    es->es_tfdpool = tfd_pool;

#ifdef	DEBUGIE
    ie_chkbounds (es->es_tfdpool);
    ie_chkbounds (es->es_tfdpool+es->es_tfds-1);
#endif	DEBUGIE

    /* chain the tfds together into a free list */

    for (i = 1; i < ntfds; i++)
	tfd_pool[i - 1].ietfd_flink = &tfd_pool[i];

    tfd_pool[ntfds - 1].ietfd_flink = (struct ietfd *)0;

    es->es_tfd_free = tfd_pool;
    }
/*******************************************************************************
*
* ie_tbufinit -
*
* Allocate and initialize the transmit buffer pool.
* Assumes the pool is currently unallocated.
*/

LOCAL VOID ie_tbufinit(es, ntbufs)
    register struct ie_softc *es;
    register int ntbufs;

    {
    register union ietbuf *tbuf_pool;
    int tbuf_psize;
    register int i;

    if (ntbufs <= 0)
	panic ("ie_tbufinit: ridiculous request");

    es->es_tbufs = ntbufs;

    /* carve the pool out of the resource map */

    tbuf_psize = ntbufs * sizeof (union ietbuf);
    tbuf_pool = (union ietbuf *) ie_alloc (es, tbuf_psize);

    es->es_tbufpool = tbuf_pool;

    /* chain the tbufs together into a free list */

    for (i = 1; i < ntbufs; i++)
	tbuf_pool[i - 1].ietbuf_next = &tbuf_pool[i];

    tbuf_pool[ntbufs - 1].ietbuf_next = (union ietbuf *)0;

    es->es_tbuf_free = tbuf_pool;
    }
/*******************************************************************************
*
* ie_rbdinit -
*
* Allocate and initialize the ring of receive buffer descriptors.
* Assumes that the ring is currently unallocated.
*/

LOCAL VOID ie_rbdinit (es, nrbds)
    register struct ie_softc *es;
    register int nrbds;

    {
    struct ierbd *rbd_start;
    register struct ierbd *rbd;
    int rbd_rsize;

    if (nrbds <= 0)
	panic ("ie_rbdinit: ridiculous request");

    es->es_rbds = nrbds;

    /* grab a chunk of control block space to hold the descriptor ring */

    rbd_rsize = nrbds * sizeof (struct ierbd);
    rbd_start = (struct ierbd *) ie_alloc (es, rbd_rsize);

    es->es_rbdring = rbd_start;
#ifdef	DEBUGIE
    ie_chkbounds(es->es_rbdring);
    ie_chkbounds(es->es_rbdring+es->es_rbds-1);
#endif	DEBUGIE

    /* Clear the rbds.  Ierustart counts on virgin rbds having
     * null ierbd_rbuf fields, as it must attach rbufs to such
     * rbds but leave others untouched.
     */
    bzero ((caddr_t )rbd_start, rbd_rsize);

    /* Link the rbds together into a ring, setting up the
     * ierbd_next and ierbd_link fields in tandem.
     */

    for (rbd = rbd_start; rbd < rbd_start + nrbds - 1; )
	{
	register struct ierbd	*prbd = rbd++;

	prbd->ierbd_next = to_ieoff(es, (caddr_t)rbd);
	prbd->ierbd_link = rbd;
	}

    rbd->ierbd_next = to_ieoff(es, (caddr_t)rbd_start);
    rbd->ierbd_link = rbd_start;

    /* initialize list pointers */
    es->es_rbdhd = rbd_start;
    es->es_rbdtl = rbd;
    }

#ifdef	DEBUGIE
#ifdef	LINT
/*******************************************************************************
*
* print_rbd -
*
* Assumes unit 0.
*/
LOCAL VOID print_rbd (nrbds)
    int nrbds;

    {
    struct ierbd *rbd_start = ie_softc[0]->es_rbdhd;
    register struct ierbd *rbd;

    for (rbd = rbd_start; rbd < rbd_start + nrbds - 1; ) {
	    register struct ierbd	*prbd = rbd++;

	    printf ("rbd = %x rbd next = %x\n", prbd, prbd->ierbd_link);
    }
    }
#endif	LINT
#endif	DEBUGIE

/*******************************************************************************
*
* ie_rfdinit -
*
* Allocate and initialize the ring of receive frame descriptors.
* Assumes that the ring is currently unallocated.
*/
LOCAL VOID ie_rfdinit (es, nrfds)
    register struct ie_softc *es;
    register int nrfds;

    {
    struct ierfd *rfd_start;
    register struct ierfd *rfd;
    int rfd_rsize;

    if (nrfds <= 0)
	panic ("ie_rfdinit: ridiculous request");

    es->es_rfds = nrfds;

    /* grab a chunk of control block space to hold the descriptor ring */

    rfd_rsize = nrfds * sizeof (struct ierfd);
    rfd_start = (struct ierfd *) ie_alloc (es, rfd_rsize);

    es->es_rfdring = rfd_start;
#ifdef	DEBUGIE
    ie_chkbounds(es->es_rfdring);
    ie_chkbounds(es->es_rfdring+es->es_rfds-1);
#endif	DEBUGIE

    /* Link the rfds together into a ring, setting up the
     * ierfd_next and ierfd_link fields in tandem.
     */

    for (rfd = rfd_start; rfd < rfd_start + nrfds - 1; )
	{
	register struct ierfd	*prfd = rfd++;

	prfd->ierfd_next = to_ieoff(es, (caddr_t)rfd);
	prfd->ierfd_link = rfd;
	}

    rfd->ierfd_next = to_ieoff(es, (caddr_t)rfd_start);
    rfd->ierfd_link = rfd_start;

    /* initialize list pointers */
    es->es_rfdhd = rfd_start;
    es->es_rfdtl = rfd;
    }
/*******************************************************************************
*
* ie_rbufinit -
*
* Allocate and initialize the receive buffer pool.
* Assumes the pool is currently unallocated.
*/

LOCAL VOID ie_rbufinit (es, nrbufs)
    register struct ie_softc *es;
    register int nrbufs;

    {
    register struct ierbuf *rbuf_pool;
    register int i;
    int rbuf_psize;

    if (nrbufs <= 0)
	panic ("ie_rbufinit: ridiculous request");

    es->es_rbufs = nrbufs;

    /* carve the pool out of the resource map */

    rbuf_psize = nrbufs * sizeof (struct ierbuf);
    rbuf_pool = (struct ierbuf *) ie_alloc (es, rbuf_psize);

    es->es_rbufpool = rbuf_pool;

    /* chain the rbufs together into a free list */

    for (i = 1; i < nrbufs; i++)
	rbuf_pool[i - 1].ierbuf_next = &rbuf_pool[i];

    rbuf_pool[nrbufs - 1].ierbuf_next = (struct ierbuf *)0;

    es->es_rbuf_free = rbuf_pool;
    }
/*******************************************************************************
*
* ietbdget -
*
* Extend the tbd list ending at tbd by transferring the head of
* the free list to it.  Tbd is assumed to be the end of the chain
* of tbds depending from the current tfd.  If nonnull, it must
* coincide with the in-use list tail.  Return the address of the
* new chain tail or NULL if there were no free tbds.
*
* This routine critically depends on the permanent arrangement of
* tbds into a ring with embedded free and in-use queues biting
* each others' tails.
*
* RETURNS: pointer to ietbd, or 0
*/
LOCAL struct ietbd *ietbdget (es, tbd)
    register struct ie_softc *es;
    register struct ietbd *tbd;

    {
    register struct ietbd *ntbd;

    if (es->es_ftbds == 0)
	return ((struct ietbd *) 0);

    ntbd = es->es_ftbd;
    es->es_ftbd = es->es_ftbd->ietbd_link;
    es->es_ftbds--;

    /* Move the list end marker for this frame
     * to the end of its newly extended tbd list.
     */
    if (tbd)
	tbd->ietbd_eof = 0;

    ntbd->ietbd_eof = 1;

    return (ntbd);
    }
/*******************************************************************************
*
* iestartout -
*
* Start or restart output to wire.  If there are sufficient
* resources available add a new frame to the transmit queue.
* Otherwise, wait until later.
*/

LOCAL VOID iestartout (unit)
    int	unit;

    {
    register struct ie_softc *es = ie_softc [unit];
    register struct ietfd *tfd;
    struct ietbd *sv_ftbd;
    u_short sv_ftbds;
    int qlen0;

    /* gather some statistics */
    qlen0 = es->es_if.if_snd.ifq_len;

    if (qlen0 > 0)
	{
	es->es_started++;
	if (qlen0 > 1)
	    es->es_started2++;
	}

    /* Loop until there are no more packets ready to send or we
     * have insufficient resources left to send another one.
     */
    while (es->es_if.if_snd.ifq_head && es->es_tfd_free &&
	    es->es_tbuf_free && es->es_ftbds > 0)
	{
	register struct mbuf	*m;
	register struct ietbd	*tbd = 0;
	register struct ietbd	*tbdlist = 0;
	struct ether_header	*ec;
	int			len;

	IF_DEQUEUE(&es->es_if.if_snd, m);

	/* Record tbd ring state for later restoration in the
	 * event of inability to complete setting up this frame.
	 */
	sv_ftbd = es->es_ftbd;
	sv_ftbds = es->es_ftbds;

	tfd = es->es_tfd_free;

#ifdef	DEBUGIE
	if (!tfd)
	    panic ("tfd free list botch");
#endif	DEBUGIE

	es->es_tfd_free = tfd->ietfd_flink;

	ec = mtod (m, struct ether_header *);
#ifdef	IE_FDADDR
	/*
	 * We've configured the chip so that the ether header lives
	 * in the frame descriptor rather than the first buffer.
	 * The chip will insert the source address.
	 */
	*(struct ether_addr *)tfd->ietfd_dhost = ec->ether_dhost;
	tfd->ietfd_type = ec->ether_type;
	m->m_len -= sizeof (struct ether_header);
	m->m_off += sizeof (struct ether_header);
#else	IE_FDADDR
	/*
	 * Insert source address.  If we end up running configured
	 * this way by default, this assignment should be moved
	 * into ether_output, since both this and the LANCE drivers
	 * would otherwise end up doing it themselves.
	 */
	bcopy ((char *)es->es_enaddr, (char *)ec->ether_shost, 6);

#endif	IE_FDADDR

	/*
	 * Record sufficient information in the frame descriptor
	 * to allow the packet to be reconstructed if need be.
	 */
/* XXX		tfd->ietfd_mbuf = m; THIS GETS ZEROED LATER */
	tfd->ietfd_tail = (union ietbuf *) 0;
	tfd->ietfd_tlen = 0;

	/*
	 * Avoid suspected bus contention problem by pulling
	 * the packet's initial buffer up to be long enough
	 * to give the chip a chance to track down successive
	 * buffers while DMA-ing out the contents of the first
	 * one.  Intel documentation claims that the worst case
	 * requirement is 70 bytes.  We choose to be more
	 * conservative.
	 */
#ifdef	IE_FDADDR
#define	IE_MINTU0	(100 - sizeof (struct ether_header))
#else	IE_FDADDR
#define	IE_MINTU0	100
#endif	IE_FDADDR

	/* don't bother - we copy into a tbuf anyway */
/* XXX		tfd->ietfd_mbuf = m = ether_pullup(m, IE_MINTU0); */
	if (!m) {
		printf("out of mbufs: output packet dropped\n");
		goto cleanup;
	}

	/*
	 * Force packet size up to minimum allowed.  Ether_pullup
	 * guarantees that we can adjust the mbuf size to the
	 * required amount (assumes MLEN > ETHERMIN).  If the
	 * ether header lives in the packet body, the required
	 * length goes up accordingly.
	 */
#ifdef	IE_FDADDR
#define	IE_ETHERMIN	ETHERMIN
#else	IE_FDADDR
#define	IE_ETHERMIN	(ETHERMIN + sizeof (struct ether_header))
#endif	IE_FDADDR
/* XXX		if (m->m_len < IE_ETHERMIN) */
/* XXX			m->m_len = IE_ETHERMIN; */

	tbd = ietbdget(es, tbd);
#ifdef	DEBUGIE
	if (!tbd)
	    panic ("iestartout null tbd");

#endif	DEBUGIE
	if (!tbdlist)
	    tbdlist = tbd;

	{
	/* must copy the mbuf and adjust bookkeeping information accordingly */

	union ietbuf	*tbuf = es->es_tbuf_free;

#ifdef	DEBUGIE
	if (!tbuf)
	    logMsg ("iestartout: null tbuf\n");
#endif	DEBUGIE

	es->es_outcpy++;	/* collect some statistics */

	tfd->ietfd_mbuf = (struct mbuf *)0;

	es->es_tbuf_free = tbuf->ietbuf_next;

	tfd->ietfd_tail = tbuf;
	tfd->ietfd_tlen = len = copy_from_mbufs ((char *)tbuf->ietbuf_data, m);

	m_freem (m);

	if (len < IE_ETHERMIN)
	    len = IE_ETHERMIN;

	tfd->ietfd_tlen = len; 

	tbd->ietbd_buf = to_ieaddr(es, (caddr_t)tbuf->ietbuf_data);
	tbd->ietbd_cntlo = len & 0xff;
	tbd->ietbd_cnthi = len >> 8;

	/* copy_from_mbufs has consumed the entire packet remainder */
	}

	tfd->ietfd_tbd = to_ieoff(es, (caddr_t)tbdlist);
	tfd->ietfd_cmd = IE_TRANSMIT;
	iedocb (es, (struct iecb *)tfd);
    }

    iecustart (es);
    return;

cleanup:
    if (tfd)
	{
	union ietbuf *tbuf = tfd->ietfd_tail;

	if (tfd->ietfd_mbuf)
	    m_freem (tfd->ietfd_mbuf);

	if (tbuf)
	    {
	    tbuf->ietbuf_next = es->es_tbuf_free;
	    es->es_tbuf_free = tbuf;
	    }
	tfd->ietfd_flink = es->es_tfd_free;
	es->es_tfd_free = tfd;
	}

    /* Restore the tbd ring to the state recorded at
     * the start of setting up the aborted packet.
     */
    es->es_ftbd = sv_ftbd;
    es->es_ftbds = sv_ftbds;

    iecustart(es);
    }
/*******************************************************************************
*
* iexmitdone -
*
* Update error counters and free up resources after transmitting a packet.
* Called by iecuclean at splnet or hardware level 3.
*
* Under certain conditions we're willing to make a second try at sending
* a packet when the first attempt fails.  The constraints are: the facility
* is enabled in the first place; no packet has gone out since the failed
* one (this prevents out of order transmissions); and we haven't already
* retried the packet (this prevents the driver from eating up the entire
* system when the net breaks).
*/

LOCAL VOID iexmitdone(es, tfd)
    register struct ie_softc *es;
    register struct ietfd *tfd;

    {
    register struct ietbd *tbd, *ntbd;
    union ietbuf *tbuf;
    register int canretransmit = ie_retransmit && tfd->ietfd_el &&
				 !es->es_pendreque;
    int s = splnet ();	/* XXX */

    if (tfd->ietfd_ok)
	{
	es->es_if.if_collisions += tfd->ietfd_ncoll;
	es->es_if.if_opackets++;
	if (tfd->ietfd_defer)
	es->es_iedefer++;
	if (tfd->ietfd_heart)
	es->es_ieheart++;
	}
    else
	{
	es->es_if.if_oerrors++;
	if (tfd->ietfd_xcoll)
	    {
	    es->es_trtry++;
	    if (!canretransmit)
		{
		printf("Ethernet jammed\n");
		}
	    }
	if (tfd->ietfd_nocarr)
	    {
	    es->es_tnocar++;
	    if (!canretransmit)
		{
		printf("no carrier\n");
		}
	    }
	if (tfd->ietfd_nocts)
	    {
	    es->es_noCTS++;
	    if (!canretransmit)
		{
		printf("no CTS\n");
		}
	    }
	if (tfd->ietfd_underrun)
	es->es_xmiturun++;
	}

    /* Verify that the tbds associated with the just-transmitted
     * frame appear at the front of the queue of in-use tbds and
     * transfer them to the free queue, reclaiming their associated
     * resources.
     */

    tbd = tfd->ietfd_tbd == IENULLOFF ? (struct ietbd *) 0
		    : (struct ietbd *) from_ieoff(es, (ieoff_t)tfd->ietfd_tbd);
    for (; tbd; tbd = ntbd)
	{
	ntbd = tbd->ietbd_eof ? (struct ietbd *) 0
		  : (struct ietbd *) from_ieoff(es, (ieoff_t)tbd->ietbd_next);

	if (tbd != es->es_utbd)
	    {
	    /* Checking here may be overkill.
	     * We may wish to move this check to immediately before the loop.
	     */
#ifdef	DEBUGIE
	    printf("tbd: %x, es->es_utbd: %x\n", tbd, es->es_utbd);
	    printf("\tes->es_tbdring: %x\n", es->es_tbdring);
#endif	DEBUGIE
	    panic ("iexmitdone: tbds out of sync");
	    }

	/* reclaim descriptor */
	es->es_utbd = es->es_utbd->ietbd_link;

	if (++es->es_ftbds > es->es_tbds)
	    panic ("iexmitdone: too many free tbds");
	}

    /* requeue if necessary and possible.
     * If successful, ie_requeue will set pendreque.
     */

    es->es_pendreque = 0;

    if (!tfd->ietfd_ok && canretransmit)
	ie_requeue(es, tfd);

    /* reclaim the frame's buffer space */
    if (tfd->ietfd_mbuf)
	m_freem(tfd->ietfd_mbuf);

    if (tbuf = tfd->ietfd_tail)
	{
	tbuf->ietbuf_next = es->es_tbuf_free;
	es->es_tbuf_free = tbuf;
	}

    tfd->ietfd_flink = es->es_tfd_free;
    es->es_tfd_free = tfd;

    splx (s);	/* XXX */
    }
/*******************************************************************************
*
* ie_requeue -
*
* Put the packet described by tfd back on the output queue.
* Adjust bookkeeping fields to reflect the altered state.
* Assumes called with the output queue locked.  Also assumes
* that caller will reclaim any resources left recorded in tfd.
*/

LOCAL VOID ie_requeue (es, tfd)
    struct ie_softc *es;
    register struct ietfd *tfd;

    {
    register struct mbuf *mtail = (struct mbuf *)0,
			    *m;
#ifdef	IE_FDADDR
    struct mbuf		*m0;
    struct ether_header	*header;
#endif	IE_FDADDR

#ifdef	DEBUGIE
    logMsg ("ie_requeue \n");
#endif DEBUGIE

    if (IF_QFULL(&es->es_if.if_snd)) {
	    printf("output queue full; can't requeue\n");
	    return;
    }

    /* convert the tail buffer back into an mbuf chain */

    if (tfd->ietfd_tlen)
	{
	mtail = copy_to_mbufs((caddr_t)tfd->ietfd_tail->ietbuf_data,
		(int)tfd->ietfd_tlen);
	if (!mtail)
	    {
	    printf("out of mbufs for packet requeue\n");
	    return;
	    }
	}

    /* glue the two mbuf chains together and set m to point to the result */

    if (m = tfd->ietfd_mbuf)
	{
	if (mtail)
	    {
	    while (m->m_next)
		    m = m->m_next;
	    m->m_next = mtail;
	    m = tfd->ietfd_mbuf;
	    }
	}
    else
	m = mtail;

#ifdef	IE_FDADDR
    /*
     * If IE_FDADDR is defined, then the ether header lives in *tfd.
     *
     * Paste the ether header back at the front of the mbuf chain.
     * If there's not enough space at the front, allocate another.
     */
    m0 = m;
    if (m->m_off > MMAXOFF ||
	MMINOFF + sizeof (struct ether_header) > m->m_off)
	{
	m = m_get(M_DONTWAIT, MT_DATA);
	if (!m)
	    {
	    m_freem (m0);
	    printf ("out of mbufs for packet requeue\n");
	    goto out;
	    }
	m->m_next = m0;
	m->m_len = sizeof (struct ether_header);
	}
    else
	{
	m->m_off -= sizeof (struct ether_header);
	m->m_len += sizeof (struct ether_header);
	}
    header = mtod (m, struct  ether_header *);
    header->ether_dhost = *(struct ether_addr *)tfd->ietfd_dhost;
    header->ether_type = tfd->ietfd_type;
#endif	IE_FDADDR

    /* requeue the reconstituted packet */

    IF_PREPEND(&es->es_if.if_snd, m);

    es->es_requeues++;
    es->es_pendreque = 1;

#ifdef	IE_FDADDR
out:
#endif	IE_FDADDR
    /*
     * The incoming mbuf is now either back on the output
     * queue or has been freed, so we don't want our caller
     * to free it.
     */
    tfd->ietfd_mbuf = (struct mbuf *)0;
    }
/*******************************************************************************
*
* iegetmem -
*
* Called to (allocate and re)initialize rmap.
* We need to be careful, since the board may be in use
* (rbufs loaned out).
*/

LOCAL VOID iegetmem (es)
    register struct ie_softc *es;

    {
    int unit = es->es_if.if_unit;

    switch (es->es_type)
	{
	case IE_TE:
	    {
	    /* Map the board's memory.  The page starting at
	     * the board's base address is already mapped from
	     * autoconf time, so we can use this mapping to obtain
	     * the pagetype bits to use for the memory mapping.
	     *
	     * memsize contains the number of bytes in the board's
	     * address space (from the first register to the end of
	     * the buffer?  See ie_alloc.
	     *
	     * es_base is the virtual address allocated the first
	     * time through.
	     *
	     * For the 3E, we don't have to worry about the mies_mbmhi
	     * member (it holds the high 4 bits of the address) since
	     * we can get the full address from mie.
	     *
	     * mapin() takes three arguments: virtual address,
	     * physical address, and size.  The virtual address is
	     * the address previously allocated and stored in es_base.
	     * The physical address is derived from the address set up
	     * by config.  Config is given the address of the csr, and
	     * the address of memory starts 0xfe bytes beyond.
	     * The length is 0x20000, the amount of memory in
	     * the interface.
	     */


	    /* Point to the scp at the top of the interface memory.
	     * Mask the scp address with the memory size to get the
	     * proper offset within the 3E memory.
	     */

	    es->es_scp = (struct iescp *) ((int)es->es_base + 
			       (IESCPADDR & ((IE_TE_MEMSIZE-1))));

	    es->es_memspace = es->es_base;

	    es->es_memsize = IESCPADDR & (IE_TE_MEMSIZE-1);
	    }

#ifdef DEBUGIE
	    printf ("iegetmem: scp address for unit %d is 0x%x\n",
		    unit, es->es_scp);
#endif DEBUGIE
		break;

	default:
		break;
	}
    }
/*******************************************************************************
*
* ieinitmem -
*
* Grab chunks of the memory allocated by iegetmem and
* initialize descriptors, buffers, and CBs in them.
* "Initialization" here refers to setting up these parts
* of the driver's data structures that remain unchanged
* during normal operation, and to setting up the free lists.
* After this routine completes, there's still more to be
* done before turning the chip loose; chip-recognized
* list end markers have to be put in place, and so on.
*
* NOTE:
* The chip references descriptors and control blocks by
* 16-bit offsets from es->es_base, whereas it accesses
* buffers with 24-bit addresses.  Thus, the order of
* the allocations is important.
*/

LOCAL VOID ieinitmem (es)
    register struct ie_softc *es;

    {
    /* Set up dedicated CBs.  The cbsync size must be large enough
     * to hold the largest CB associated with a synchronously-executed
     * command.  Cbsync is cast as required to one of these CB types.
     */

    es->es_cbsync = (struct iecb *) ie_alloc (es, CBSYNC_SIZE);
    es->es_cbnop  = (struct iecb *) ie_alloc (es, sizeof (struct iecb));
    es->es_iscp   = (struct ieiscp *) ie_alloc (es, sizeof (struct ieiscp));
    es->es_scb    = (struct iescb *) ie_alloc (es, sizeof (struct iescb));

#ifdef	DEBUGIE
    /* make sure that all descriptors are accessible */
    ie_chkbounds (es->es_cbsync);
    ie_chkbounds (es->es_cbnop);
    ie_chkbounds (es->es_iscp);
    ie_chkbounds (es->es_scb);
#endif	DEBUGIE

    /* set up descriptor pools (must be within 16 bits of es_base) */
    ie_tbdinit (es, ie_tbds);
    ie_tfdinit (es, ie_tfds);
    ie_rbdinit (es, ie_rbds);
    ie_rfdinit (es, ie_rfds);

    /* set up buffer pools */
    ie_tbufinit (es, ie_tbufs);
    ie_rbufinit (es, ie_rbufs);

#ifdef	DEBUGIE
    printf ("ieinitmem: remaining memory = %d\n", /*(unsigned int)*/es->es_memsize -
	    (((unsigned int)es->es_memspace) - (unsigned int)es->es_base));
#endif	DEBUGIE
    }
/*******************************************************************************
*
* iechipinit - basic 82586 initialization
*/

LOCAL int iechipinit (es)
    register struct ie_softc *es;

    {
    struct ieiscp		*iscp = es->es_iscp;
    struct iescb		*scb = es->es_scb;
    register struct iecb	*cb;
    register struct ieconf	*ic;
    int ok = 0;
    int gotintr;

reset:
    bzero ((caddr_t)es->es_scp, sizeof (struct iescp));	/* 10 bytes */

    es->es_scp->ies_iscp = to_ieaddr (es, (caddr_t)iscp);

    bzero ((caddr_t)iscp, sizeof (struct ieiscp));

    iscp->ieis_busy    = 1;
    iscp->ieis_cbbase = to_ieaddr (es, es->es_base);
    iscp->ieis_scb    = to_ieoff (es, (caddr_t)scb);

    /* Set the scb's ie_magic field and never touch it again.
     * This allows us to check for rogue behavior in the chip's
     * DMA unit by looking for changes in this field.
     */

    bzero ((caddr_t)scb, sizeof (struct iescb));
    scb->ie_magic = IEMAGIC;

    /* Hardware reset the chip.  We make the interval from
     * reset to initial channel attention as small as reasonable
     * to reduce the risk of scribbling chips getting us.
     */
    switch (es->es_type)
	{
#ifdef	IE_MB
	case IE_MB:
	    /* Hardware reset already occurred in iechipreset. */
	    break;
#endif	IE_MB

#ifdef	IE_OB
#endif	IE_OB

	case IE_TE:
	    /* Hardware reset already occurred in iechipreset. */
	    break;
	}

    ieca (es);

    CDELAY(!iscp->ieis_busy, IEDELAY);	/* ensure chip eats iscp */
    CDELAY(scb->ie_cnr, IEDELAY);		/* ensure scb updated too */

    gotintr = iewaitintr (es);		/* wait for interrupt */

    if (iscp->ieis_busy || !scb->ie_cnr || !gotintr)
	{
	printf("init failed:%s%s%s\n",
	    iscp->ieis_busy	? " iscp busy" : "",
	    !scb->ie_cnr	? " no cnr" : "",
	    !gotintr		? " no intr" : "");
	goto exit;
	}

    if (scb->ie_cus != IECUS_IDLE )
	{
	printf ("cus not idle after reset\n");
	iechipreset (es);
	goto reset;
	}

    cb = synccb_res (es, struct iecb *);

#ifdef	DEBUGIE
    if (!cb)
	panic ("iechipinit: nested synccmd");
#endif	DEBUGIE

    bzero ((caddr_t)cb, sizeof (struct iecb));

    cb->iec_cmd = IE_DIAGNOSE;
    iesynccmd (es, cb);

    if (!cb->iec_ok)
	{
	printf ("Intel 82586 failed diagnostics\n");
	synccb_rel (es);
	goto exit;
	}

    synccb_rel (es);

    /* perform TDR test */

#ifdef NOTDEF

    /* Skip, since hardware requires quiet net to work. */

    tdr = synccb_res(es, struct ietdr *);

#ifdef	DEBUGIE
    if (!tdr)
	panic ("iechipinit: nested synccmd 2");
#endif	DEBUGIE

    bzero ((caddr_t)tdr, sizeof (struct ietdr));
    tdr->ietdr_cb.ie_cmd = IE_TDR;
    iesynccmd (es, &tdr->ietdr_cb);
    if (!tdr->ietdr_ok)
	{
#define	TDRCONST	12	/* approx 0.77c/10Mhz */

	int dist = (tdr->ietdr_timhi<<8) + tdr->ietdr_timlo;

	if (dist != 0x7FF)
	    printf("link not OK - distance ~ %dm\n", TDRCONST*dist);
	else
	    printf("link not OK\n");
	synccb_rel (es);
	goto exit;
	}

    if (tdr->ietdr_xcvr)
	printf("transceiver bad\n");
    if (tdr->ietdr_open)
	printf("net not terminated\n");
    if (tdr->ietdr_shrt)
	printf("net shorted\n");
    synccb_rel (es);
#endif NOTDEF

    /* establish default chip operating characteristics */
    ic = synccb_res (es, struct ieconf *);

#ifdef	DEBUGIE
    if (!ic)
	panic ("iechipinit: nested synccmd 3");

#endif	DEBUGIE

    iedefaultconf (ic);
    iesynccmd (es, &ic->ieconf_cb);
    synccb_rel (es);
    ok = 1;

exit:
    return (ok);
    }
/*******************************************************************************
*
* ierustart - initialize and start the Receive Unit
*
* Assumes that descriptor and buffer memory is already allocated.
*/

LOCAL VOID ierustart (es)
    register struct ie_softc *es;

    {
    register struct ierbd *rbd;
    register struct ierfd *rfd;
    register struct iescb *scb;
    register struct ierbuf *rbp;

    /* initialize receive buffers */

    for (rbp = es->es_rbuf_free; rbp; rbp = rbp->ierbuf_next)
	rbp->ierbuf_es = es;	/* set back pointer to softc structure */


    /* initialize rbds */
    for (rbd = es->es_rbdhd; ; )
	{
	/* if the rbd doesn't yet have an rbuf attached to it, hook one on */

	if (!rbd->ierbd_rbuf)
	    {
	    if (!(rbp = es->es_rbuf_free))
		panic ("ierustart: not enough rbufs");
	    es->es_rbuf_free = rbp->ierbuf_next;
	    rbd->ierbd_rbuf = rbp;
	    rbd->ierbd_buf = to_ieaddr(es, (caddr_t)rbp->ierbuf_data);
	    rbd->ierbd_sizehi = IE_BUFSZ >> 8;
	    rbd->ierbd_sizelo = IE_BUFSZ & 0xff;
	    }

	rbd->ierbd_status = 0;

	/* impose chip-recognizable list structure */
	if (rbd == es->es_rbdtl)
	    {
	    rbd->ierbd_el = 1;
	    break;
	    }
	else
	    {
	    rbd->ierbd_el = 0;
	    rbd = rbd->ierbd_link;
	    }
	}

    /* initialize rfds */
    for (rfd = es->es_rfdhd; ; )
	{
	rfd->ierfd_status = 0;
	rfd->ierfd_susp = 0;
	rfd->ierfd_rbd = IENULLOFF;

	/* impose chip-recognizable list structure */
	if (rfd == es->es_rfdtl)
	    {
	    rfd->ierfd_el = 1;
	    break;
	    }
	else
	    {
	    rfd->ierfd_el = 0;
	    rfd = rfd->ierfd_link;
	    }
	}

    /* We must tell the chip where the first frame's first
     * rbd is.  It is responsible for setting this field in
     * subsequent frames.
     */

    rfd = es->es_rfdhd;
    rfd->ierfd_rbd = to_ieoff(es, (caddr_t)es->es_rbdhd);

    /* Kick the chip to get the RU going. */
    scb = es->es_scb;
    if (scb->ie_rus != IERUS_IDLE)
	{
	printf("RU unexpectedly not idle\n");
	iestat(es);
	iechkcca(es);
	scb->ie_cmd = IECMD_RU_ABORT;
	ieca(es);
	}

    iechkcca (es);

    scb->ie_rfa = to_ieoff(es, (caddr_t)rfd);
    scb->ie_cmd = IECMD_RU_START;

    ieca (es);

    CDELAY(scb->ie_rus == IERUS_READY, IEDELAY);

    if (scb->ie_rus != IERUS_READY)
	printf("RU did not become ready\n");
    }
/*******************************************************************************
*
* iedocb -
*
* Put a CB on the CBL.  The old list-end marker is left
* undisturbed, so the chip's CU won't start to process
* the newly-entered cb.  Iecustart worries about moving
* the list end marker to the (new) end of the list.
*/

LOCAL VOID iedocb (es, cb)
    register struct ie_softc *es;
    register struct iecb *cb;

    {
    register int s = splnet();

    cb->iec_status = 0;		/* clear status bits */
    cb->iec_susp   = 0;		/* clear suspend bit */
    cb->iec_el     = 1;		/* will be reset in iecustart */
    cb->iec_intr   = 1;
    cb->iec_next   = 0;

    if (es->es_cbhead)
	{
	es->es_cbtail->iec_next = to_ieoff(es, (caddr_t)cb);
	es->es_cbtail = cb;
	}
    else
	es->es_cbhead = es->es_cbtail = cb;

    (void) splx(s);
    }
/*******************************************************************************
*
* iecustart - start the CU with the current CBL
*/

LOCAL VOID iecustart (es)
    register struct ie_softc *es;

    {
    register struct iecb *cb;
    register struct iescb *scb = es->es_scb;
    int s = splnet();
    int oldLev = intLock ();
	    
    iechkcca (es);

    /* See whether there's anything to do.  The second clause
     * of this test has the effect of squashing out some potential
     * parallelism between chip and driver.  Without it, we would
     * go ahead and add CBs to the list (as the chip sees it) while
     * the chip is still processing it.  Provided that the rest
     * of the driver obeys the proper protocol for accessing CBs,
     * there's no reason for this clause other than paranoia.
     */

    if (es->es_cbhead == NULL || scb->ie_cus == IECUS_READY)
	{
	(void) splx(s);
	intUnlock (oldLev);
	return;
	}

    /* trim and reclaim completed CBs from the front of the list */
    iecuclean (es);

    /* strip list-end markers from the interior of the list */

    if ((cb = es->es_cbhead) == NULL)
	{
	(void) splx(s);
	intUnlock (oldLev);
	return;
	}

    while (cb && cb->iec_next)
	{
	cb->iec_el = 0;
	cb = (struct iecb *)from_ieoff(es, (ieoff_t)cb->iec_next);
	}

    /* start CU */

    scb->ie_cbl = to_ieoff(es, (caddr_t)es->es_cbhead);
    scb->ie_cmd = IECMD_CU_START;

    ieca (es);

    intUnlock (oldLev);
    (void) splx(s);
    }
/*******************************************************************************
*
* iecuclean - process completed CBs, reclaiming specified storage
*
* Allocator is responsible for reclaiming other storage.
* Called by iecustart at splnet and by iecbdone at splnet or hardware level 3.
*/

LOCAL VOID iecuclean (es)
    register struct ie_softc *es;

    {
    register struct iecb *cb;
	    
    while ((cb = es->es_cbhead) && cb->iec_done)
	{
	es->es_cbhead = cb->iec_next
		? (struct iecb *) from_ieoff(es, (ieoff_t)cb->iec_next)
		: (struct iecb *) 0;

	/* Unless the command was run synchronously, it should be
	 * one of the two below.  Anything else indicates trouble.
	 */

	switch (cb->iec_cmd)
	    {
	    case IE_TRANSMIT:
		iexmitdone(es, (struct ietfd *)cb);
		break;

	    case IE_NOP:
		    break;

	    default:
#ifdef	DEBUGIE
		if (es->es_synccmd != 1)
		    printf("unknown cmd %x done\n", cb->iec_cmd);
#endif	DEBUGIE
		    break;
	    }
	}
    }
/*******************************************************************************
* iecbdone - clean up and restart the CBs on the CBL
*
* Called by ieintr at hardware level 3 and by iesynccmd at splnet.
*/

LOCAL VOID iecbdone (es)
    register struct ie_softc *es;

    {
    int s = splnet ();

    iecuclean (es);
    iestartout (es->es_if.if_unit);	/* generate more CBs */

    splx (s);
    }
/*******************************************************************************
*
* iesynccmd -
*
* Do the command contained in cb synchronously, so that we know
* it's complete upon return.  The control block given as argument
* should be obtained with synccb_res and released with synccb_rel.
*/

LOCAL VOID iesynccmd (es, cb)
    register struct ie_softc *es;
    register struct iecb *cb;

    {
    register struct iescb *scb = es->es_scb;
    int s;
    int cmd = 0;

    /* fire it up */
    iedocb (es, cb);
    iecustart (es);

    /* give it a chance to complete */
    CDELAY(cb->iec_done, IEDELAY);

    if (!cb->iec_done)
	{
	iestat (es);
	panic ("iesynccmd");
	}

    /* acknowledge status information from the chip */
    s = splnet();
    iechkcca(es);
    if (scb->ie_cx)
	    cmd |= IECMD_ACK_CX;
    if (scb->ie_cnr)
	    cmd |= IECMD_ACK_CNR;
    scb->ie_cmd = cmd;

    ieca (es);

    /* ??? Is it really appropriate to start the chip up here? */

    if (cmd & (IECMD_ACK_CNR|IECMD_ACK_CX))
	iecbdone (es);

    (void) splx(s);
    }
/*******************************************************************************
*
* ichkcca -
*
* Delay until the 82586 has accepted the current
* command word.  Watch for buggy hardware while at it.
*/

LOCAL VOID iechkcca(es)
    register struct ie_softc *es;

    {
    register struct iescb *scb = es->es_scb;
    register int i;

    for (i = 0; i < IEDELAY; i++)
	{
	if (scb->ie_magic != IEMAGIC)
	    {
	    /* Something's writing where it shouldn't... */
#ifdef DEBUGIE
	    logMsg ("iechkcca: scb->magic address 0x%x\n",&scb->ie_magic);
#endif DEBUGIE
	    panic ("scb overwritten");
	    }

	if (scb->ie_cmd == 0)
		break;
	}

    if (i >= IEDELAY)
	{
	logMsg ("cmd not accepted\n");
	panic ("iechkcca");
	}
    }

int iefifolim = 12;

/*******************************************************************************
*
* iedefaultconf - set default configuration parameters
*/

LOCAL VOID iedefaultconf (ic)
    register struct ieconf *ic;

    {
    bzero((caddr_t)ic, sizeof (struct ieconf));
    ic->ieconf_cb.iec_cmd = IE_CONFIG;
    ic->ieconf_bytes = 12;
    ic->ieconf_fifolim = iefifolim;
    ic->ieconf_pream = 2;		/* 8 byte preamble */
#ifdef	IE_FDADDR
    ic->ieconf_acloc = 0;
#else	IE_FDADDR
    ic->ieconf_acloc = 1;
#endif	IE_FDADDR
    ic->ieconf_alen = 6;
    ic->ieconf_space = 96;
    ic->ieconf_slttmh = 512 >> 8;
    ic->ieconf_minfrm = 64;
    ic->ieconf_retry = 15;
    ic->ieconf_crfilt = 3;
    }
/*******************************************************************************
*
* ieca - activate the channel attention line
*/

LOCAL VOID ieca (es)
    register struct ie_softc *es;

    {
    switch (es->es_type)
	{
#ifdef	IE_MB
	case IE_MB:
	    es->es_mie->mie_ca = 1;
	    es->es_mie->mie_ca = 0;
	    break;
#endif	IE_MB

#ifdef	IE_OB
	case IE_OB:
	    es->es_obie->obie_ca = 1;
	    es->es_obie->obie_ca = 0;
	    break;
#endif	IE_OB

	case IE_TE:
	    es->es_tie->tie_ca = 1;
	    es->es_tie->tie_ca = 0;
	    break;

	default:
	    panic ("ieca");
	}
    }
/*******************************************************************************
*
* iewaitintr - wait for an interrupt and relay results
*/

LOCAL int iewaitintr (es)
    register struct ie_softc *es;

    {
/*
    register struct obie_device *obie = es->es_obie;
    register struct mie_device *mie = es->es_mie;
*/
    register struct tie_device *tie = es->es_tie;
    int ok;

    switch (es->es_type)
	{
#ifdef	IE_OB
	case IE_OB:
	    CDELAY(obie->obie_intr, IEDELAY);
	    ok = obie->obie_intr;
	    break;
#endif	IE_OB

#ifdef	IE_MB
	case IE_MB:
	    CDELAY(mie->mie_intr, IEDELAY);
	    ok = mie->mie_intr;
	    break;
#endif	IE_MB

	case IE_TE:
	    CDELAY(tie->tie_intr, IEDELAY);
	    ok = tie->tie_intr;
	    break;

	default:
	    panic ("iewaitintr");
	}
    return (ok);
    }
/*******************************************************************************
*
* iechipreset -
*
* Cut the chip out of the loop and halt it by starting the reset cycle.
* For IE_MB, we must enable pagemaps, hence we complete the reset cycle.
*
* The 3E has no page maps, but we duplicate the MB code anyway.
*/

LOCAL VOID iechipreset (es)
    register struct ie_softc *es;
    {

    switch (es->es_type)
	{
#ifdef	IE_MB
	case IE_MB:
	    /* see RCS */
#endif	IE_MB

#ifdef	IE_OB
	case IE_OB:
	    /* see RCS */
#endif	IE_OB

	case IE_TE:
	    es->es_tie->tie_reset = 1;
	    taskDelay (5);
	    *(char *)&es->es_tie->tie_status = 0;	/* power on reset */
	    break;

	default:
	    panic ("iechipreset");
	}
    }
/*******************************************************************************
*
* iesplice - splice the chip into the loop
*/

LOCAL VOID iesplice (es)
    register struct ie_softc *es;

    {
    switch (es->es_type)
	{
#ifdef	IE_MB
	case IE_MB:
	    /* see RCS */
#endif	IE_MB

#ifdef	IE_OB
	case IE_OB:
	    /* see RCS */
#endif	IE_OB
	case IE_TE:
	    es->es_tie->tie_ie     = 1;	/* enable chip interrupts */
	    es->es_tie->tie_noloop = 1;	/* enable cable */
	    break;
	default:
	    panic ("iesplice");
	}
    }
/*******************************************************************************
*
* ieget - get packet from interface
*
* Routine to copy from 3E ethernet board into mbufs.
*
* WARNING: makes the fairly safe assumption that mbufs have even lengths
*/

LOCAL struct mbuf *ieget (cp, totlen, off0, ifp)
    FAST char *cp;
    int totlen;
    int off0;
    struct ifnet *ifp;

    {
    FAST struct mbuf *m;
    FAST int off = off0;
    int len;
    struct mbuf *top = 0;
    struct mbuf **mp = &top;
    char *initcp = cp;

    while (totlen > 0)
	{
	u_char *mcp;

	MGET(m, M_DONTWAIT, MT_DATA);
	if (m == 0) 
	    {
	    m_freem(top);
	    top = 0;
	    goto out;
	    }

	if (off)
	    {
	    len = totlen - off;
	    cp += sizeof( struct ether_header ) + off;
	    } 
	else
	    len = totlen;

	m->m_off = MMINOFF;
	if (ifp) 
	    {
	     /* Leave room for ifp. */
	    m->m_len = MIN(MLEN - sizeof(ifp), len);
	    m->m_off += sizeof(ifp);
	    } 
	else 
	    m->m_len = MIN(MLEN, len);

	len = m->m_len;
	mcp = mtod(m, u_char *);
	bcopyBytes (cp, mcp, len);
	cp += len;
	*mp = m;
	mp = &m->m_next;
	if (off) 
	    {
	    /* sort of an ALGOL-W style for statement... */
	    off += m->m_len;
	    if (off == totlen) 
		{
		cp = initcp;
		cp += sizeof( struct ether_header );
		off = 0;
		totlen = off0;
		}
	    } 
	    else
		totlen -= m->m_len;

	if (ifp) 
	    {
	    /* prepend interface pointer to first mbuf */

	    m->m_len += sizeof(ifp);
	    m->m_off -= sizeof(ifp);
	    *(mtod(m, struct ifnet **)) = ifp;
	    ifp = (struct ifnet *)0;
	    }
	}
    return (top);
out:
    return (0);
    }
/*******************************************************************************
*
* ieoutput - 
*/

LOCAL int ieoutput (ifp, m0, dst)
    FAST struct ifnet *ifp;
    FAST struct mbuf *m0;
    struct sockaddr *dst;

    {
    int type;
    int error;
    u_char edst[6];
    struct in_addr idst;
    FAST struct ie_softc *es = ie_softc[ifp->if_unit];
    FAST struct mbuf *m = m0;
    FAST struct ether_header *eh;
    FAST int off;
    int usetrailers;
    int s = splnet ();

    if ((ifp->if_flags & (IFF_UP|IFF_RUNNING)) != (IFF_UP|IFF_RUNNING))
	{
    	error = ENETDOWN;
    	goto bad;
	}

    switch (dst->sa_family)
	{

#ifdef INET
	case AF_INET:
	    idst = ((struct sockaddr_in *)dst)->sin_addr;

	    if (!arpresolve(&es->es_ac, m, &idst, edst, &usetrailers))
		{
		splx (s);
    		return (0);	/* if not yet resolved */
		}

	    off = ntohs((u_short)mtod(m, struct ip *)->ip_len) - m->m_len;

	    if (usetrailers && off > 0 && (off & 0x1ff) == 0 &&
		m->m_off >= MMINOFF + 2 * sizeof (u_short))
		{
    		type = ETHERTYPE_TRAIL + (off>>9);
    		m->m_off -= 2 * sizeof (u_short);
    		m->m_len += 2 * sizeof (u_short);
    		*mtod(m, u_short *) = htons((u_short)ETHERTYPE_IP);
    		*(mtod(m, u_short *) + 1) = htons((u_short)m->m_len);
    		goto gottrailertype;
		}

	    type = ETHERTYPE_IP;
	    off = 0;
	    goto gottype;
#endif

	case AF_UNSPEC:
	    eh = (struct ether_header *)dst->sa_data;
	    bcopy((caddr_t)eh->ether_dhost, (caddr_t)edst, sizeof (edst));
	    type = eh->ether_type;
	    goto gottype;

	default:
	    printf("ex%d: can't handle af%d\n", ifp->if_unit, dst->sa_family);
	    error = EAFNOSUPPORT;
	    goto bad;
	}

gottrailertype:
    /*
     * Packet to be sent as trailer: move first packet
     * (control information) to end of chain.
     */
    while (m->m_next)
    	m = m->m_next;
    m->m_next = m0;
    m = m0->m_next;
    m0->m_next = 0;
    m0 = m;

gottype:
    /*
     * Add local net header.  If no space in first mbuf,
     * allocate another.
     */
    if (m->m_off > MMAXOFF || MMINOFF + sizeof (struct ether_header) > m->m_off)
	{
    	m = m_get(M_DONTWAIT, MT_HEADER);
    	if (m == 0)
	    {
	    error = ENOBUFS;
	    goto bad;
	    }
    	m->m_next = m0;
    	m->m_off = MMINOFF;
    	m->m_len = sizeof (struct ether_header);
	}
    else
	{
    	m->m_off -= sizeof (struct ether_header);
    	m->m_len += sizeof (struct ether_header);
	}
    eh = mtod(m, struct ether_header *);
    eh->ether_type = htons((u_short)type);
    bcopy((char *)edst, (char *)eh->ether_dhost, sizeof (edst));
    bcopy((char *)es->es_enaddr, (char *)eh->ether_shost, 6);

    /*
     * Queue message on interface, and start output if interface
     * not yet active.
     */

/* XXX    s = splimp();   */
    if (IF_QFULL(&ifp->if_snd))
	{
    	IF_DROP(&ifp->if_snd);
/* XXX        splx(s);  */
	splx (s);
    	m_freem(m);
    	return (ENOBUFS);
	}

    IF_ENQUEUE(&ifp->if_snd, m);

    iestartout (ifp->if_unit);

#ifdef DEBUG
    else
    	es->es_wait++;
#endif

/* XXX    splx(s); */
    splx (s);
    return (0);

bad:
    m_freem(m0);
    splx (s);
    return (error);
    }
/*******************************************************************************
*
* copy_from_mbufs -
*/

LOCAL int copy_from_mbufs (buf, m)
    FAST char *buf;
    FAST struct mbuf *m;

    {
    FAST int cc;
    FAST caddr_t cp = buf; 

    while (m) 
	{
	bcopy (mtod (m, caddr_t), cp, (unsigned int) m->m_len);
	cp += m->m_len;
	m = m->m_next;
	}

    cc = (int) cp - (int) buf; 
    return (cc);
    }
/*******************************************************************************
*
* copy_to_mbufs
*/

LOCAL struct mbuf *copy_to_mbufs (buf, totlen)
    char *buf;
    int totlen;

    {
    FAST struct mbuf *m;
    FAST int len;
    FAST caddr_t cp = buf; 
    struct mbuf *top = 0;
    struct mbuf **mp = &top;

    while (totlen > 0)  
	{
	MGET(m, M_DONTWAIT, MT_DATA);
	if (m == 0) 
	    {
	    m_freem(top);
	    return (NULL);
	    }

	len = totlen;

	m->m_off = MMINOFF;
	m->m_len = MIN(MLEN, len);

	bcopy ((char *)cp, (char *)mtod(m, caddr_t), m->m_len);
	cp += m->m_len;

	*mp = m;
	mp = &m->m_next;
	totlen -= m->m_len;
	}
    return (top);
    }

/* routines for interconverting chip and host addresses */

/*******************************************************************************
*
* ieaddr -
*
* Return chip's idea of given address or 0 if not chip accessible.
*/

LOCAL VOID ieaddr (es, cp)
    register struct ie_softc *es;
    register caddr_t cp;

    {
    if ((cp >= es->es_base) && (cp < es->es_base + IE_TE_MEMSIZE))
	return ((int) cp - (int) es->es_base);
    else
	{
	logMsg ("bad ieaddr %#x\n", cp);
	return (0);
	}
    }
/*******************************************************************************
*
* to_ieaddr -
*
* Change 68000 address to Intel 24-bit address.
* We take advantage of the fact that all 82586 blocks with 24-bit
* addresses have an adjacent unused 8-bit field, and store 32 bits.
*/

LOCAL ieaddr_t to_ieaddr(es, cp)
    struct ie_softc *es;
    caddr_t cp;

    {
    union
	{
	int n;
	char c[4];
	} a, b;

    a.n = ieaddr (es, cp);		/* necessary for double mapping */

    b.c[0] = a.c[3];
    b.c[1] = a.c[2];
    b.c[2] = a.c[1];
    b.c[3] = 0;

    return (b.n);
    }
#ifdef	JUNK
/*******************************************************************************
*
* to_ieaddr_opt -
*
* Change 68000 address to Intel 24-bit address -- optimized version.
* This routine can be used when cp has been obtained from an earlier
* call to ieaddr.  The idea is to avoid unnecessary calls to ieaddr,
* whose aggregate cost is large enough to notice.
*/

LOCAL ieaddr_t to_ieaddr_opt(cp)
    int cp;	/* Fast and loose with coercions... */
    {
    union
	{
	int n;
	char c[4];
	} a, b;

    a.n = cp;

    b.c[0] = a.c[3];
    b.c[1] = a.c[2];
    b.c[2] = a.c[1];
    b.c[3] = 0;

    return (b.n);
    }
#endif	JUNK
/*******************************************************************************
*
* from_ieaddr -
*/

LOCAL caddr_t from_ieaddr (es, n)
    struct ie_softc *es;
    ieaddr_t n;
    {
    union
	{
	int n;
	char c[4];
	} a, b;

    a.n = n;

    b.c[0] = 0;
    b.c[1] = a.c[2];
    b.c[2] = a.c[1];
    b.c[3] = a.c[0];

    return (es->es_base + b.n);
    }
/*******************************************************************************
*
* to_ieoff -
*/

LOCAL ieoff_t to_ieoff(es, addr)
    register struct ie_softc *es;
    caddr_t addr;
    {
    union
	{
	ieoff_t s;
	char c[2];
	} a, b;

    a.s = (ieoff_t)(addr - es->es_base);

    b.c[0] = a.c[1];
    b.c[1] = a.c[0];

    return (b.s);
    }
/*******************************************************************************
*
* from_ieoff -
*/

LOCAL caddr_t from_ieoff(es, off)
    register struct ie_softc *es;
    ieoff_t off;
    {
    union
	{
	ieoff_t s;
	char c[2];
	} a, b;

    a.s = off;

    b.c[0] = a.c[1];
    b.c[1] = a.c[0];

    return (es->es_base + b.s);
    }
/*******************************************************************************
*
* to_ieint -
*/
LOCAL ieint_t to_ieint (n)
    ieint_t n;
    {
    union
	{
	ieint_t s;
	char c[2];
	} a, b;

    a.s = n;

    b.c[0] = a.c[1];
    b.c[1] = a.c[0];

    return (b.s);
    }

/* error handling and reporting routines */

LOCAL VOID iestat(es)
    struct ie_softc *es;

    {
    static char *cus[] =
   { "idle", "suspended", "ready", "<3>", "<4>", "<5>", "<6>", "<7>" };
    static char *rus[] =
   { "idle", "suspended", "no resources", "<3>", "ready", "<5>", "<6>", "<7>" };
    register struct iescb *scb = es->es_scb;

    printf ("scb: ");
    if (scb->ie_cx)
	printf ("cx ");
    if (scb->ie_fr)
	printf ("fr ");
    if (scb->ie_cnr)
	printf ("cnr ");
    if (scb->ie_rnr)
	printf ("rnr ");

    printf ("cus=%s ", cus[scb->ie_cus]);
    printf ("rus=%s\n", rus[scb->ie_rus]);

    printf ("cbl=0x%x rfa=0x%x crc=0x%x aln=0x%x rsc=0x%x ovrn=0x%x\n",
	    scb->ie_cbl, scb->ie_rfa,
	    scb->ie_crcerrs, scb->ie_alnerrs, scb->ie_rscerrs,
	    scb->ie_ovrnerrs);

    if (scb->ie_cmd)
	printf ("cmd=0x%x\n", scb->ie_cmd & 0xFFFF);
    }

/* Debugging routines */

#ifdef	DEBUGIE

/*******************************************************************************
*
* ieprt -
*/

VOID ieprt (unit)
    int unit;

    {
    ieprt_softc (unit);
    ieprt_scb (unit);
    ieprt_cbl (unit);
    }
/*******************************************************************************
*
* ieprt_softc -
*/

VOID ieprt_softc (unit)
    int unit;

    {
    struct ie_softc *es = ie_softc[unit];

    printf ("\nsoftc: %x\n", es);
    printf ("mie %x: %x\n", &es->es_mie, es->es_mie);
    printf ("obie %x: %x\n", &es->es_obie, es->es_obie);
    printf ("base %x: %x\n", &es->es_base, es->es_base);
    printf ("scb %x: %x\n", &es->es_scb, es->es_scb);
    }
/*******************************************************************************
*
* ieprt_cbl -
*/

VOID ieprt_cbl (unit)
    int unit;

    {
    struct ie_softc *es = ie_softc[unit];
    struct iecb *cbp;

    printf ("\ncb list: head %x tail %x\n", es->es_cbhead, es->es_cbtail);
    cbp = es->es_cbhead;

    while (cbp)
	{
	printf ("cb %x\n", cbp);

	printf ("done %x, busy %x, ok %x, aborted %x\n",
		cbp->iec_done, cbp->iec_busy, cbp->iec_ok,
		cbp->iec_aborted); 

	printf("el %x, susp %x, intr %x, cmd %x\n",
		cbp->iec_el, cbp->iec_susp, cbp->iec_intr, cbp->iec_cmd); 

	if (cbp->iec_next)
	    {
	    cbp = (struct iecb *)from_ieoff(es,
	    (ieoff_t)cbp->iec_next);
	    }
	else
	    cbp = NULL;
	}
    }
/*******************************************************************************
*
* ieprt_scb -
*/

VOID ieprt_scb (unit)
    int unit;

    {
    static char *cus[] = { "idle", "suspended", "ready" };
    static char *rus[] = { "idle", "suspended", "no resources", "", "ready" };
    register struct iescb *scb = ie_softc[unit]->es_scb;

    printf("\nscb %x: ", scb);
    if (scb->ie_cx)
	printf("cx ");
    if (scb->ie_fr)
	printf("fr ");
    if (scb->ie_cnr)
	printf("cnr ");
    if (scb->ie_rnr)
	printf("rnr ");

    printf("cus=%s ", cus[scb->ie_cus]);
    printf("rus=%s ", rus[scb->ie_rus]);
    printf("cbl %x\n", scb->ie_cbl);

    printf("errors: crc=%d, aln=%d, rsc=%d, ovrn=%d\n",
	    scb->ie_crcerrs, scb->ie_alnerrs, scb->ie_rscerrs,
	    scb->ie_ovrnerrs);

    if (scb->ie_cmd)
	printf("cmd=%x\n", scb->ie_cmd & 0xFFFF);
    }
/*******************************************************************************
*
* ieaddrs -
*/

VOID ieaddrs ()

    {
    printf("ieprt = %x\n", ieprt);
    printf("ieprt_softc = %x\n", ieprt_softc);
    printf("ieprt_scb = %x\n", ieprt_scb);
    printf("ieprt_cbl = %x\n", ieprt_cbl);
    }
/*******************************************************************************
*
* iechkmbuf -
*/

VOID iechkmbuf (m)
    register struct mbuf *m;

    {
    while (m)
	{
	if ((m->m_type != 1 && m->m_type != 2) || m != dtom(m))
	    panic ("iechkmbuf");
	m = m->m_next;
	}
    }
/*******************************************************************************
*
* iedumptbds -
*
* Print out the contents of the transmit buffer descriptor
* chain rooted at tfd->ietfd_tbd.
*/

VOID iedumptbds(es, tfd)
    struct ie_softc *es;
    struct ietfd *tfd;

    {
    register struct ietbd *tbd, *ntbd;
    register caddr_t bufp;

    printf("\nietbd chain starting from tfd at 0x%x\n", tfd);

    tbd = (struct ietbd *) from_ieoff(es, tfd->ietfd_tbd);

    for (; tbd; tbd = ntbd)
	{
	if (tbd->ietbd_next)
	    ntbd = (struct ietbd *) from_ieoff(es, (ieoff_t) tbd->ietbd_next);
	else
	ntbd = (struct ietbd *) 0;
	bufp = from_ieaddr(es, tbd->ietbd_buf);

	printf("\tat: %x, cntlo: 0x%x, cnthi: 0x%x, eof: %d\n",
		tbd, tbd->ietbd_cntlo, tbd->ietbd_cnthi, tbd->ietbd_eof);
	printf("\t  next: 0x%x (0x%x), buf: 0x%x link: %d\n",
		ntbd, tbd->ietbd_next, bufp, tbd->ietbd_link);
	}
    }
#endif	DEBUGIE
