/* if_enp.c - network interface driver for the CMC ENP-10 ethernet controller */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01s,25may89,hjb   enpioctl() SIOCSIFFLAGS, check for IFF_UP before calling 
		 enpinit(). 
01r,24may89,rdc  enpintr no longer does netJobAdd if task level is
                 already processing packets. (fixes netRing overflows)
		 enpoutput now bumps if_opackets.
01q,17apr89,del  fixed comment in enpgetaddr.
01q,13apr89,del  fixed problem introduced by CMC when they changed to
		 version 4.0 Link-10 roms.
01p,05aug88,dnw  fixed oversight in enpreset in 01o.
01o,04aug88,dnw  fixed bug causing occasional initialization failure.
		 simplified reset.
01n,23jun88,rdc  added IFF_BROADCAST to if_flags.
01m,05jun88,dnw  changed enp_softc to array of ptrs to descriptors.
		 increased NENP from 1 to 4.
01l,30may88,dnw  changed to v4 names.
01k,22feb88,jcf  made kernel independent.
01j,20feb88,dnw  cleanup, lint.
01i,10feb88,rdc  re-installed enpreset which i accidently trashed in 01g.
		 optimized packet reception by calling ipintr only once
		 for a bunch of packets.
01h,14jan88,jcf  extended maximum firmware initialization wait to 15 seconds
01g,05jan88,rdc  added include of systm.h 
01f,20nov87,ecs  lint
	   &dnw  changed enpattach to take int vec num instead of vec adrs.
01e,19nov87,dnw  changed to program interrupt vector.
		 changed initial wait loop to use vxMemProbe.
01d,08nov87,dnw  added output hook code.
		 changed calling sequence to input hook.
01c,21oct87,ecs  delinted
01b,28aug87,dnw  removed unnecessary includes.
		 added input hook code.
		 clean up.
01a,21jul87,rdc  adapted from cmc's 4.2 driver. 
*/

#include "vxWorks.h"
#include "param.h"
#include "mbuf.h"
#include "protosw.h"
#include "ioctl.h"
#include "socket.h"
#include "errno.h"
#include "uio.h"
#include "if.h"
#include "route.h"
#include "in.h"
#include "in_systm.h"
#include "ip.h"
#include "ip_var.h"
#include "in_var.h"
#include "if_ether.h"
#include "if_enp.h"
#include "etherLib.h"
#include "vme.h"
#include "iv68k.h"
#include "iosLib.h"
#include "ioLib.h"
#include "memLib.h"
#include "systm.h"

IMPORT VOID ipintr ();


#define NENP	4	/* max number of enp controllers */

/*
 * Ethernet software status per interface.
 *
 * Each interface is referenced by a network interface structure,
 * es_if, which the routing code uses to locate the interface.
 * This structure contains the output queue for the interface, its address, ...
 */

struct enp_softc 
    {
    struct  arpcom es_ac;	/* common ethernet structures */
#define es_if           es_ac.ac_if     /* network-visible interface */
#define es_enaddr       es_ac.ac_enaddr /* hardware ethernet address */
    ENPDEVICE *enpAddr;		/* vme address of board */
    int enpIntLevel;		/* vme interrupt level */
    int enpIntVec;		/* vme interrupt vector */
    BOOL taskLevelActive;		/* netTask is currently processing packets */
    };
    
LOCAL struct enp_softc *enp_softc [NENP];
static int enpverflag;

/* forward declarations */

LOCAL int enpintr();
LOCAL int enpinit();
LOCAL int enpioctl();
LOCAL int enpoutput();
LOCAL int enpreset();
LOCAL VOID enpHandleInt ();
LOCAL struct  mbuf *enpget();

/*********************************************************************
*
* enpattach - attach the enp driver to the network
*
* enpattach connects the enp ISR (enpintr), sets up enpsoftc and attaches
* the interface.  System will initialize the interface when it is ready
* to accept packets. 
*/

STATUS enpattach (unit, addr, ivec, ilevel)
    int unit; 		/* unit number */
    char *addr;		/* address of enp's shared memory */
    int ivec;		/* interrupt vector to connect to */
    int ilevel;		/* interrupt level */

    {
    FAST ENPDEVICE *pEnpDev = (ENPDEVICE *) addr;
    char *enpBase           = addr + 0x1000;
    FAST struct enp_softc *es;
    FAST struct ifnet *ifp;
    FAST int i;
    short status;

    /* wait for kernel reset to finish */

    i = 15 * sysClkRateGet ();
    while ((vxMemProbe ((char *) &pEnpDev->enp_status, READ,
			sizeof (short), (char *)&status) != OK) ||
	   ((status & STS_READY) == 0))
	{
	if (--i <= 0)
	    {
	    errnoSet (S_iosLib_CONTROLLER_NOT_PRESENT);
	    printf ("enpattach: status: 0x%x\n", status);
	    return (ERROR);
	    }

	taskDelay (1);
	}


    /* start link level firmware */

    pEnpDev->enp_intvector = ivec;
    WRITELONG (&pEnpDev->enp_base, enpBase);
    pEnpDev->enp_go = GOVAL;


    /* wait for the link level firmware to finish initialization */

    i = 15 * sysClkRateGet ();
    while ((pEnpDev->enp_state & S_ENPRUN) == 0)
	{
	if (--i <= 0)
	    {
	    printf ("enp%d: link level firmware failed to initialize.\n", unit);
	    errnoSet (S_ioLib_DEVICE_ERROR);
	    return (ERROR);
	    }

	taskDelay (1);
	}


    /* allocate and initialize enp descriptor */

    es = (struct enp_softc *) malloc (sizeof (struct enp_softc));
    if (es == NULL)
	return (ERROR);

    enp_softc[unit] = es;		/* set ptr to enp descriptor in array */

    bzero ((char *) es, sizeof (*es));	/* clear enp descriptor */

    es->enpAddr     = pEnpDev;
    es->enpIntLevel = ilevel;
    es->enpIntVec   = ivec;

    enpgetaddr (unit);

    ifp  = &es->es_if;
    ifp->if_unit   = unit;
    ifp->if_name   = "enp";
    ifp->if_mtu    = ETHERMTU;
    ifp->if_init   = enpinit;
    ifp->if_ioctl  = enpioctl;
    ifp->if_output = enpoutput;
    ifp->if_reset  = enpreset;
    ifp->if_flags  = IFF_BROADCAST;

    intConnect (INUM_TO_IVEC (ivec), enpintr, unit);
    sysIntEnable (ilevel);

    if_attach (ifp);

    return (OK);
    }
/*********************************************************************
*
* enpreset - Reset of interface 
*
* The CMC board is reset by accessing its reset mailbox.
*/

LOCAL VOID enpreset (unit)
    int unit;
    {
    struct enp_softc *es = enp_softc[unit];
    ENPDEVICE *pEnpDev = es->enpAddr;

    RESET_ENP (pEnpDev);

    /* reset the status word that enpattach watches to know when enp is reset;
     * this is necessary because the enp doesn't clear this status immediately
     * on reset.  Re-initializing the enp after a reset (i.e. in VxWorks
     * init after boot rom reset) could get fooled if we rely on enp to clear
     * the READY bit in this status.
     */
    pEnpDev->enp_status = 0;
    }
/*********************************************************************
*
* enpinit - initialize enp.
*
* restart the link level software on the enp board and mark the interface
* as up.
*
*/

LOCAL VOID enpinit (unit)
    int unit;

    {
    struct enp_softc *es = enp_softc[unit];

    es->es_if.if_flags |= IFF_UP|IFF_RUNNING; /* open for business*/
    }


/*********************************************************************
*
* enpintr - Ethernet interface interrupt.
*
* calls netJobAdd to have netTask do the work of reading packets off
* the interface.
*
*/

LOCAL VOID enpintr (unit)
    FAST int unit;

    {
    FAST struct enp_softc *es = enp_softc[unit];

    sysBusIntAck (es->enpIntLevel);

    if (!es->taskLevelActive)
	{
	es->taskLevelActive = TRUE;	/* indicate that enpHandleInt is q'd */
	netJobAdd (enpHandleInt, es);   /* see comment in enpHandleInt */
	}
    }

/*********************************************************************
*
* enpHandleInt - handle interrupt from enp
*/

LOCAL VOID enpHandleInt (es)
    struct enp_softc *es;

    {
    FAST BCB *bcbp;
    FAST ENPDEVICE *addr = es->enpAddr;
    BOOL gotIp = FALSE;

    /* the following loop attempts to pull multiple buffers off of the
     * input queue;  the boolean "taskLevelActive" is used to inform
     * the interrupt level code that we're already processing input
     * buffers, so there's no need to do a netJobAdd, thereby decreasing
     * the possibility of a net task ring buffer overflow.
    */

    do
	{
	es->taskLevelActive = TRUE;

	while ((bcbp = (BCB*) enpringget ((RING*) &addr->enp_tohost)) != 0)
	    {
	    if (enpread (es, bcbp))
		gotIp = TRUE;	/* note that IP packet was received */

	    enpringput ((RING *)&addr->enp_enpfree, bcbp);
	    }

	es->taskLevelActive = FALSE;
	}
    /* we must check once more after resetting taskLevelActive to avoid race */
    while (!enpringempty ((RING*) &addr->enp_tohost));


    /* call IP input routine if IP packet was received */

    if (gotIp)
	ipintr ();
    }

/*********************************************************************
*
* enpread - read a packet off the interface 
*
* enpread copies packets from a bcb into an mbuf and hands it to 
* the next higher layer.
*
* RETURNS: TRUE if IP packet is received, FALSE otherwise.
*/

LOCAL BOOL enpread (es, bcbp)
    struct enp_softc *es;
    FAST BCB *bcbp;

    {
    struct ether_header *eh;
    FAST struct mbuf *m;
    int len, off, resid;
    FAST struct ifqueue *inq;
    unsigned char *pData;
    BOOL isIp = FALSE;		/* TRUE if IP packet received */

    es->es_if.if_ipackets++;

    /*
     * Get input data length.
     * Get pointer to ethernet header (in input buffer).
     */

    len = bcbp->b_msglen;
    READLONG( &bcbp->b_addr, &eh);

    /* call input hook if any */

    if ((etherInputHookRtn != NULL) &&
	(* etherInputHookRtn) (&es->es_if, (char *) eh, len))
	return (FALSE);

    /* Deal with trailer protocol: if type is PUP trailer
     * get true type from first 16-bit word past data.
     * Remember that type was trailer by setting off.
     */

    len -= sizeof (struct ether_header);
    pData = ((unsigned char *) eh) + sizeof (struct ether_header);

    if (eh->ether_type >= ETHERTYPE_TRAIL && 
	eh->ether_type < ETHERTYPE_TRAIL+ETHERTYPE_NTRAILER) 
	{
	off = (eh->ether_type - ETHERTYPE_TRAIL) * 512;
	if (off >= ETHERMTU)
	    return (FALSE);	/* sanity */

	eh->ether_type = *(u_short *) (pData + off);
	resid          = *(u_short *) (pData + off + 2);

	if (off + resid > len)
	    return (FALSE);	/* sanity */

	len = off + resid;
	} 
    else
	off = 0;

    if (len == 0)
	return (FALSE);

    /*
     * Pull packet off interface.  Off is nonzero if packet
     * has trailing header; enpget will then force this header
     * information to be at the front, but we still have to drop
     * the type and length which are at the front of any trailer data.
     */

    m = enpget (pData, len, off, (struct ifnet *) &es->es_if);
    if (m == 0)
	return (FALSE);

    if (off)
	{
	struct ifnet *ifp;

	ifp = *(mtod(m, struct ifnet **));
	m->m_off += 2 * sizeof (u_short);
	m->m_len -= 2 * sizeof (u_short);
	*(mtod(m, struct ifnet **)) = ifp;
	}

    switch (eh->ether_type) 
	{
#ifdef INET
	case ETHERTYPE_IP:
	    inq = &ipintrq;
	    isIp = TRUE;	/* note IP packet received */
	    break;

	case ETHERTYPE_ARP:
	    arpinput(&es->es_ac, m);
	    return (FALSE);
#endif
	default:
	    m_freem(m);
	    return (FALSE);
	}

    if (IF_QFULL(inq)) 
	{
	IF_DROP(inq);
	m_freem(m);
	return (FALSE);
	}

    IF_ENQUEUE(inq, m);

    return (isIp);
    }
/*********************************************************************
*
* enpoutput - Ethernet output routine. (called by user)
*
* Encapsulate a packet of type family for the local net.
* Use trailer local net encapsulation if enough data in first
* packet leaves a multiple of 512 bytes of data in remainder.
* If destination is this address or broadcast, send packet to
* loop device to kludge around the fact that 3com interfaces can't
* talk to themselves.
*
*/

LOCAL int enpoutput (ifp, m0, dst)
    struct ifnet *ifp;
    struct mbuf *m0;
    struct sockaddr *dst;

    {
    int type, s, error;
    u_char edst[6];
    struct in_addr idst;
    FAST struct enp_softc *es = enp_softc[ifp->if_unit];
    FAST struct mbuf *m = m0;
    FAST struct ether_header *eh;
    FAST int off;
    int usetrailers;

    if ((ifp->if_flags & (IFF_UP|IFF_RUNNING)) != (IFF_UP|IFF_RUNNING))
	{
    	error = ENETDOWN;
    	goto bad;
	}

    switch (dst->sa_family)
	{

#ifdef INET
	case AF_INET:
	    idst = ((struct sockaddr_in *)dst)->sin_addr;

	    if (!arpresolve(&es->es_ac, m, &idst, edst, &usetrailers))
    		return (0);	/* if not yet resolved */

	    off = ntohs((u_short)mtod(m, struct ip *)->ip_len) - m->m_len;

	    if (usetrailers && off > 0 && (off & 0x1ff) == 0 &&
		m->m_off >= MMINOFF + 2 * sizeof (u_short))
		{
    		type = ETHERTYPE_TRAIL + (off>>9);
    		m->m_off -= 2 * sizeof (u_short);
    		m->m_len += 2 * sizeof (u_short);
    		*mtod(m, u_short *) = htons((u_short)ETHERTYPE_IP);
    		*(mtod(m, u_short *) + 1) = htons((u_short)m->m_len);
    		goto gottrailertype;
		}

	    type = ETHERTYPE_IP;
	    off = 0;
	    goto gottype;
#endif
#ifdef NS
	case AF_NS:
	    type = ETHERTYPE_NS;
	    bcopy ((caddr_t)&(((struct sockaddr_ns *)dst)->sns_addr.x_host),
		   (caddr_t)edst, sizeof (edst));
	    off = 0;
	    goto gottype;
#endif

	case AF_UNSPEC:
	    eh = (struct ether_header *)dst->sa_data;
	    bcopy ((caddr_t)eh->ether_dhost, (caddr_t)edst, sizeof (edst));
	    type = eh->ether_type;
	    goto gottype;

	default:
	    printf("enp%d: can't handle af%d\n", ifp->if_unit, dst->sa_family);
	    error = EAFNOSUPPORT;
	    goto bad;
	}

gottrailertype:
    /*
     * Packet to be sent as trailer: move first packet
     * (control information) to end of chain.
     */
    while (m->m_next)
    	m = m->m_next;
    m->m_next = m0;
    m = m0->m_next;
    m0->m_next = 0;
    m0 = m;

gottype:
    /*
     * Add local net header.  If no space in first mbuf,
     * allocate another.
     */
    if (m->m_off > MMAXOFF || MMINOFF + sizeof (struct ether_header) > m->m_off)
	{
    	m = m_get(M_DONTWAIT, MT_HEADER);
    	if (m == 0)
	    {
	    error = ENOBUFS;
	    goto bad;
	    }
    	m->m_next = m0;
    	m->m_off = MMINOFF;
    	m->m_len = sizeof (struct ether_header);
	}
    else
	{
    	m->m_off -= sizeof (struct ether_header);
    	m->m_len += sizeof (struct ether_header);
	}
    eh = mtod(m, struct ether_header *);
    eh->ether_type = htons((u_short)type);
    bcopy((caddr_t)edst, (caddr_t)eh->ether_dhost, sizeof (edst));
    bcopy((caddr_t)es->es_enaddr, (caddr_t)eh->ether_shost, 6);

    /* Queue message on interface if possible */

    s = splimp();
    if (enpput (ifp->if_unit, m))
	{
	error = ENOBUFS;
	splx(s);
	return (error);
	}
    splx( s );
    es->es_if.if_opackets++;	/* output packet complete */
    return (0);

bad:
    m_freem(m0);
    return (error);
    }
/*********************************************************************
*
* enpput - copy a packet to the interface.
*
* Routine to copy from mbuf chain to transmitter
* buffer in VMEbus memory.
*/

LOCAL int enpput (unit, m)
    int unit;
    FAST struct mbuf *m;

    {
    struct enp_softc *es = enp_softc [unit];
    FAST ENPDEVICE *addr = (ENPDEVICE *)es->enpAddr;
    FAST BCB *bcbp;
    FAST u_char *buf;
    u_char *bufStart;
    struct mbuf *mp = m;

    if (enpringempty ((RING *) &addr->enp_hostfree))
	return (1);

    bcbp = (BCB *) enpringget ((RING *)&addr->enp_hostfree);

    READLONG (&bcbp->b_addr, &bufStart);

    /* copy mbufs to buffer on enp;
     * note that freeing of mbufs is delayed until after giving
     * the enp the packet, so we can overlap operation */

    for (buf = bufStart; m != NULL; buf += m->m_len, m = m->m_next)
	bcopyBytes (mtod (m, caddr_t), (char *)buf, (int)m->m_len);

    bcbp->b_len = max (MINPKTSIZE, buf - bufStart);


    /* call output hook if any */

    if ((etherOutputHookRtn != NULL) &&
	(* etherOutputHookRtn) (&es->es_if, bufStart, bcbp->b_len))
	{
	m_freem (mp);
	return (0);	/* output hook has already processed this packet */
	}


    if (enpringput ((RING *) &addr->enp_toenp, bcbp) == 1)
	INTR_ENP (addr);

    m_freem (mp);

    return (0);
    }
/*********************************************************************
*
* enpget - get packet from interface.
*
* Routine to copy from VMEbus memory into mbufs.
*
* Warning: This makes the fairly safe assumption that
* mbufs have even lengths.
*/

LOCAL struct mbuf *enpget (buf0, totlen, off0, ifp)
    unsigned char *buf0;
    int totlen, off0;
    struct ifnet *ifp;

    {
    FAST struct mbuf *m;
    int len;
    unsigned char *cp = buf0;
    FAST int off      = off0;
    struct mbuf *top  = 0;
    struct mbuf **mp  = &top;

    while (totlen > 0)
	{
	MGET (m, M_DONTWAIT, MT_DATA);
	if (m == 0) 
	    {
	    m_freem(top);
	    top = 0;
	    goto out;
	    }

	if (off)
	    {
	    len = totlen - off;
	    cp = buf0 + off;
	    } 
	else
	    len = totlen;

	m->m_off = MMINOFF;
	if (ifp) 
	    {
	    /* Leave room for ifp. */
	    m->m_len = MIN (MLEN - sizeof(ifp), len);
	    m->m_off += sizeof(ifp);
	    } 
	else 
	    m->m_len = MIN(MLEN, len);

	bcopyBytes ((char *)cp, mtod (m, char *), m->m_len);
	cp += m->m_len;
	*mp = m;
	mp = &m->m_next;
	if (off) 
	    {
	    /* sort of an ALGOL-W style for statement... */
	    off += m->m_len;
	    if (off == totlen) 
		{
		cp = buf0;
		off = 0;
		totlen = off0;
		}
	    } 
	    else
		totlen -= m->m_len;

	if (ifp) 
	    {
	    /*
	     * Prepend interface pointer to first mbuf.
	     */
	    m->m_len += sizeof(ifp);
	    m->m_off -= sizeof(ifp);
	    *(mtod(m, struct ifnet **)) = ifp;
	    ifp = (struct ifnet *)0;
	    }
	}
    return (top);

out:
    return (0);
    }
/*********************************************************************
*
* enpioctl - ioctl for interface
*
*/

LOCAL int enpioctl (ifp, cmd, data)
    FAST struct ifnet *ifp;
    int cmd;
    caddr_t data;

    {
    FAST struct ifaddr *ifa = (struct ifaddr *) data;
    int error = 0;
    int s;

    s = splimp();

    switch (cmd) 
	{

	case SIOCSIFADDR:
	    ifp->if_flags |= IFF_UP;
	    enpinit (ifp->if_unit);

	    switch (ifa->ifa_addr.sa_family)
		{
#ifdef INET
		case AF_INET:
		    ((struct arpcom *)ifp)->ac_ipaddr = IA_SIN(ifa)->sin_addr;
		    arpwhohas ((struct arpcom *)ifp, &IA_SIN(ifa)->sin_addr);
		    break;
#endif
		}

	case SIOCSIFFLAGS:
	    if (ifp->if_flags & IFF_UP)
		enpinit (ifp->if_unit);
	    break;

	default:
	    error = EINVAL;
	}

    splx (s);
    return (error);
    }

/*********************************************************************
*
* enpgetaddr - get hardwired ethernet address.
*
* Read the ethernet address off the board, one byte at a time.
*	put it in enp_softc
* Because our friends at CMC changed their link-10 roms and did not
* make the change to ENPDEVICE backward compatable with v3.x, we must
* perform a check to try and determine the proper ENPDEVICE structure
* to use. In this case, we look at the ethernet base address and see
* if it is 0x00, if it is we assume that the Link-10 proms are v4.0.
* This assumtion is based on the hope that the new Link-10 roms will
* leave the RING32 buffers that were inserted in the ENPDEVICE structure
* set to 0x00 after a reset. This seems to work. (see if_enp.h for differences
* in the ENPDEVICE structure). If, in the future, this driver is modified
* to reference the fields e_stat, and/or e_netaddr (aka enp_stat and enp_addr)
* that the functions enpverflagset and enpverflag get be used to reference
* the version of Link-10 software is running on the given CMC board.
* (see below).
*/

LOCAL VOID enpgetaddr (unit)
    int unit;

    {
    FAST struct enp_softc *es   = enp_softc[unit];
    FAST ENPDEVICE_OLD *addr    = (ENPDEVICE_OLD *) es->enpAddr;
    FAST ENPDEVICE     *addrnew = es->enpAddr;
    FAST USHORT        *pChk    = (USHORT *) &addr->enp_addr.e_baseaddr;

    if (*pChk == 0x00)	/* check version of Link-10 proms */
	{
	bcopyBytes((caddr_t)&addrnew->enp_addr.e_baseaddr, 
		(caddr_t)es->es_enaddr, sizeof (es->es_enaddr));
	}
    else		/* cmc link-10 v4.0 proms */
	{
	bcopyBytes((caddr_t)&addr->enp_addr.e_baseaddr, 
		(caddr_t)es->es_enaddr, sizeof (es->es_enaddr));
	}
    }

#ifdef FINDVERSION
/******************************************************************************
*
* Note: the following functions are a way of handling the differences 
* between link-19 v3.x and 4.0 roms on the CMC enp10/L when and if we
* can somhow find out the software version number from the board.
* If the version number is available the funtion enpverflagset is
* used to set the version flag , CMCOLD | CMCNEW, for 3.x or 4.0 
* respecfully. enpgetaddr then calls enpverflagget to determine
* which ENPDEVICE structure to use, to reference the ethernet addrs.
*/  

/*********************************************************************
*
* enpgetaddr - get hardwired ethernet address.
*
* Read the ethernet address off the board, one byte at a time.
*	put it in enp_softc
*/

/* LOCAL */ VOID enpgetaddr (unit)
    int unit;

    {
    struct enp_softc *es = enp_softc[unit];
    ENPDEVICE_OLD *addr = (ENPDEVICE_OLD *)es->enpAddr;
    ENPDEVICE *addrnew = es->enpAddr;
    if (enpverflagget () == CMCOLD) 
	{
	printf ("enpgetaddt: old. enp_addr.e_baseaddr: 0x%x\n",
		addr->enp_addr.e_baseaddr);
	bcopyBytes((caddr_t)&addr->enp_addr.e_baseaddr, 
		(caddr_t)es->es_enaddr, sizeof (es->es_enaddr));
		
	}
    else		/* cmc link-10 v4.0 proms */
	{
	printf ("enpgetaddt: new. enp_addr.e_baseaddr: 0x%x\n",
		addrnew->enp_addr.e_baseaddr);
	bcopyBytes((caddr_t)&addrnew->enp_addr.e_baseaddr, 
		(caddr_t)es->es_enaddr, sizeof (es->es_enaddr));
		
	}
    }
/******************************************************************************
*
* enpverflagget -
*
* function to read the version flag as set by enpverflagset.
*
* RETURNS: CMCOLD | CMCNEW  
*/  
int enpverflagget ()
    {
    return (enpverflag);
    }

/******************************************************************************
*
* enpverflagset -
*
* function to set variable enpverflag.
*
* RETURNS OK | ERROR if attempt to set invalid flag value
*/
STATUS enpverflagset (val)
    int val;			/* flag value to set */

    {
    if ((val != CMCOLD) && (val != CMCNEW))
	return (ERROR);
    enpverflag = val;
    return (OK);
    }
#endif FINDVERSION

/********************************************************************** 
 * routines to synchronize enp and host 
 */
/*********************************************************************
*
* enpringempty - check for ring empty.
*
* RETURNS - TRUE | FALSE
*/

LOCAL BOOL enpringempty (rp)
    FAST RING *rp;

    {
    return( rp->r_rdidx == rp->r_wrtidx );
    }
/*********************************************************************
*
* enpringput - write to next ring location
*
*/

LOCAL int enpringput (rp, v)
    FAST RING *rp;
    BCB *v;

    {
    FAST int idx;

    idx = (rp->r_wrtidx + 1) & (rp->r_size-1);
    if (idx != rp->r_rdidx)
	{
	WRITELONG (&rp->r_slot[ rp->r_wrtidx ], v);
	rp->r_wrtidx = idx;
	if ((idx -= rp->r_rdidx) < 0)
	    idx += rp->r_size;
	return (idx);			/* num ring entries */
	}

    return (0);
    }
/*********************************************************************
*
* enpringget - get the next item from a ring
*
*/

LOCAL int enpringget (rp)
    FAST RING *rp;

    {
    int i = 0;

    if (rp->r_rdidx != rp->r_wrtidx)
	{
	READLONG (&rp->r_slot [rp->r_rdidx], &i);
	rp->r_rdidx = (++rp->r_rdidx) & (rp->r_size-1);
	}

    return (i);
    }
