/* if.c - network interface utility routines */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
 * Copyright (c) 1987 Wind River Systems, Inc.
 * Copyright (c) 1980, 1986 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * and the VxWorks Software License Agreement specify the terms and
 * conditions for redistribution.
 *
 *	@(#)if.c	7.1 (Berkeley) 6/4/86
 */

/*
modification history
--------------------
02k,01may89,dnw  changed if_attach to "or" in IFF_NOTRAILERS in if_flags.
02j,19aug88,gae  fixed bug in ifioctl() not handling arpioctl's found by dnw.
02i,30may88,dnw  changed to v4 names.
02h,05jan88,rdc  added include of systm.h 
02g,14dec87,gae  added include of vxWorks.h.
02f,13nov87,jcf  added if_reset which resets all attached network devices.
02e,04nov87,dnw  moved definition of ifnet here from if.h.
02d,02may87,dnw  added init of ifq_maxlen to if_attach() (was in ifinit() only)
		   to relax requirement that all interfaces be attached
		   prior to calling ifinit().
		 removed unnecessary includes.
		 cleaned up slowtimo routine.
		 replaced references to hz with calls to sysClkGetRate().
02c,03apr87,ecs  added header and copyright.
02b,01apr87,jlf  delinted.
02a,02feb87,jlf  Removed CLEAN ifdefs.
*/

/*
DESCRIPTION
*/

#include "vxWorks.h"
#include "param.h"
#include "socket.h"
#include "socketvar.h"
#include "protosw.h"
#include "ioctl.h"
#include "errno.h"

#include "if.h"
#include "af.h"

#include "wdLib.h"

#include "systm.h"

IMPORT VOID netJobAdd ();

int ifqmaxlen = IFQ_MAXLEN;
struct ifnet *ifnet;		/* head of linked list of interfaces */

LOCAL WDOG_ID ifslowtimoWd;	/* watchdog timer for slowtimo routine */

/*
 * Network interface utility routines.
 *
 * Routines with ifa_ifwith* names take sockaddr *'s as
 * parameters.
 */

ifinit()
    {
    register struct ifnet *ifp;

    for (ifp = ifnet; ifp; ifp = ifp->if_next)
	if (ifp->if_snd.ifq_maxlen == 0)
	    ifp->if_snd.ifq_maxlen = ifqmaxlen;

    ifslowtimoWd = wdCreate ();
    if_slowtimo ();
    }


/*
 * Call each interface on a reset.
 */
ifreset ()

    {
    register struct ifnet *ifp;

    for (ifp = ifnet; ifp; ifp = ifp->if_next)
	if (ifp->if_reset)
	    (*ifp->if_reset)(ifp->if_unit);
    }

/*
 * Attach an interface to the list of "active" interfaces.
 */
if_attach(ifp)
    struct ifnet *ifp;
    {
    FAST struct ifnet **p = &ifnet;

    /* make interface not use trailer protocol by default */

    ifp->if_flags |= IFF_NOTRAILERS;

    /* make send queue maxlen be default if none specified;
     * this relaxes the requirement that all interfaces be attached
     * before calling ifinit */

    if (ifp->if_snd.ifq_maxlen == 0)
	ifp->if_snd.ifq_maxlen = ifqmaxlen;

    /* find end of interface list and add the new one */

    while (*p)
	p = &((*p)->if_next);

    *p = ifp;
    }

/*
 * Locate an interface based on a complete address.
 */
/*ARGSUSED*/
struct ifaddr *
ifa_ifwithaddr(addr)
	struct sockaddr *addr;
{
	register struct ifnet *ifp;
	register struct ifaddr *ifa;

#define	equal(a1, a2) \
	(bcmp((caddr_t)((a1)->sa_data), (caddr_t)((a2)->sa_data), 14) == 0)
	for (ifp = ifnet; ifp; ifp = ifp->if_next)
	    for (ifa = ifp->if_addrlist; ifa; ifa = ifa->ifa_next) {
		if (ifa->ifa_addr.sa_family != addr->sa_family)
			continue;
		if (equal(&ifa->ifa_addr, addr))
			return (ifa);
		if ((ifp->if_flags & IFF_BROADCAST) &&
		    equal(&ifa->ifa_broadaddr, addr))
			return (ifa);
	}
	return ((struct ifaddr *)0);
}
/*
 * Locate the point to point interface with a given destination address.
 */
/*ARGSUSED*/
struct ifaddr *
ifa_ifwithdstaddr(addr)
	struct sockaddr *addr;
{
	register struct ifnet *ifp;
	register struct ifaddr *ifa;

	for (ifp = ifnet; ifp; ifp = ifp->if_next) 
	    if (ifp->if_flags & IFF_POINTOPOINT)
		for (ifa = ifp->if_addrlist; ifa; ifa = ifa->ifa_next) {
			if (ifa->ifa_addr.sa_family != addr->sa_family)
				continue;
			if (equal(&ifa->ifa_dstaddr, addr))
				return (ifa);
	}
	return ((struct ifaddr *)0);
}

/*
 * Find an interface on a specific network.  If many, choice
 * is first found.
 */
struct ifaddr *
ifa_ifwithnet(addr)
	register struct sockaddr *addr;
{
	register struct ifnet *ifp;
	register struct ifaddr *ifa;
	register u_int af = addr->sa_family;
	register int (*netmatch)();

	if (af >= AF_MAX)
		return (0);
	netmatch = afswitch[af].af_netmatch;
	for (ifp = ifnet; ifp; ifp = ifp->if_next)
	    for (ifa = ifp->if_addrlist; ifa; ifa = ifa->ifa_next) {
		if (ifa->ifa_addr.sa_family != addr->sa_family)
			continue;
		if ((*netmatch)(&ifa->ifa_addr, addr))
			return (ifa);
	}
	return ((struct ifaddr *)0);
}

#ifdef notdef
/*
 * Find an interface using a specific address family
 */
struct ifaddr *
ifa_ifwithaf(af)
	register int af;
{
	register struct ifnet *ifp;
	register struct ifaddr *ifa;

	for (ifp = ifnet; ifp; ifp = ifp->if_next)
	    for (ifa = ifp->if_addrlist; ifa; ifa = ifa->ifa_next)
		if (ifa->ifa_addr.sa_family == af)
			return (ifa);
	return ((struct ifaddr *)0);
}
#endif

/*
 * Mark an interface down and notify protocols of
 * the transition.
 * NOTE: must be called at splnet or eqivalent.
 */
if_down(ifp)
	register struct ifnet *ifp;
{
	register struct ifaddr *ifa;

	ifp->if_flags &= ~IFF_UP;
	for (ifa = ifp->if_addrlist; ifa; ifa = ifa->ifa_next)
		pfctlinput(PRC_IFDOWN, &ifa->ifa_addr);
}

/*
 * Handle interface watchdog timer routines.  Called
 * from softclock, we decrement timers (if set) and
 * call the appropriate interface routine on expiration.
 */
if_slowtimo ()
    {
    register struct ifnet *ifp;

    for (ifp = ifnet; ifp; ifp = ifp->if_next)
	{
	if (ifp->if_timer == 0 || --ifp->if_timer)
	    continue;
	if (ifp->if_watchdog)
	    (*ifp->if_watchdog)(ifp->if_unit);
	}

    wdStart (ifslowtimoWd, sysClkRateGet()/IFNET_SLOWHZ, netJobAdd,
	     (int) if_slowtimo);
    }

/*
 * Map interface name to
 * interface structure pointer.
 */
struct ifnet *
ifunit(name)
	register char *name;
{
	register char *cp;
	register struct ifnet *ifp;
	int unit;

	for (cp = name; cp < name + IFNAMSIZ && *cp; cp++)
		if (*cp >= '0' && *cp <= '9')
			break;
	if (*cp == '\0' || cp == name + IFNAMSIZ)
		return ((struct ifnet *)0);
	unit = *cp - '0';
	for (ifp = ifnet; ifp; ifp = ifp->if_next) {
		if (bcmp(ifp->if_name, name, (unsigned)(cp - name)))
			continue;
		if (unit == ifp->if_unit)
			break;
	}
	return (ifp);
}

/*
 * Interface ioctls.
 */
ifioctl(so, cmd, data)
	struct socket *so;
	int cmd;
	caddr_t data;
{
	register struct ifnet *ifp;
	register struct ifreq *ifr;

	switch (cmd) {

	case SIOCGIFCONF:
		return (ifconf(cmd, data));

#ifdef	INET
	case SIOCSARP:
	case SIOCDARP:
		/* FALL THROUGH */
	case SIOCGARP:
		return (arpioctl(cmd, data));
#endif	INET
	}
	ifr = (struct ifreq *)data;
	ifp = ifunit(ifr->ifr_name);
	if (ifp == 0)
		return (ENXIO);
	switch (cmd) {

	case SIOCGIFFLAGS:
		ifr->ifr_flags = ifp->if_flags;
		break;

	case SIOCGIFMETRIC:
		ifr->ifr_metric = ifp->if_metric;
		break;

	case SIOCSIFFLAGS:
		if (ifp->if_flags & IFF_UP && (ifr->ifr_flags & IFF_UP) == 0) {
			int s = splimp();
			if_down(ifp);
			splx(s);
		}
		ifp->if_flags = (ifp->if_flags & IFF_CANTCHANGE) |
			(ifr->ifr_flags &~ IFF_CANTCHANGE);
		if (ifp->if_ioctl)
			(void) (*ifp->if_ioctl)(ifp, cmd, data);
		break;

	case SIOCSIFMETRIC:
		ifp->if_metric = ifr->ifr_metric;
		break;

	default:
		if (so->so_proto == 0)
			return (EOPNOTSUPP);
		return ((*so->so_proto->pr_usrreq)(so, PRU_CONTROL,
			cmd, data, ifp));
	}
	return (0);
}

/*
 * Return interface configuration
 * of system.  List may be used
 * in later ioctl's (above) to get
 * other information.
 */
/*ARGSUSED*/
ifconf(cmd, data)
	int cmd;
	caddr_t data;
{
	struct ifconf *ifc = (struct ifconf *)data;
	struct ifnet *ifp = ifnet;
	struct ifaddr *ifa;
	char *cp, *ep;
	struct ifreq ifr, *ifrp;
	int space = ifc->ifc_len, error = 0;

	ifrp = ifc->ifc_req;
	ep = ifr.ifr_name + sizeof (ifr.ifr_name) - 2;
	for (; space > sizeof (ifr) && ifp; ifp = ifp->if_next) {
		bcopy(ifp->if_name, ifr.ifr_name, sizeof (ifr.ifr_name) - 2);
		for (cp = ifr.ifr_name; cp < ep && *cp; cp++)
			;
		*cp++ = '0' + ifp->if_unit; *cp = '\0';
		if ((ifa = ifp->if_addrlist) == 0) {
			bzero((caddr_t)&ifr.ifr_addr, sizeof(ifr.ifr_addr));
			error = copyout((caddr_t)&ifr, (caddr_t)ifrp, sizeof (ifr));
			if (error)
				break;
			space -= sizeof (ifr), ifrp++;
		} else 
		    for ( ; space > sizeof (ifr) && ifa; ifa = ifa->ifa_next) {
			ifr.ifr_addr = ifa->ifa_addr;
			error = copyout((caddr_t)&ifr, (caddr_t)ifrp, sizeof (ifr));
			if (error)
				break;
			space -= sizeof (ifr), ifrp++;
		}
	}
	ifc->ifc_len -= space;
	return (error);
}
