/* smDrv.c - Driver for Interphase V/SM 3200 Disk Controller */

static char *copyright = "Copyright 1988, Integrated Solutions, Inc.";

/* 
modification history
--------------------
*/

/*
DESCRIPTION
*/


#include "vxWorks.h"
#include "types.h"
#include "ioLib.h"
#include "iosLib.h"
#include "rt11Lib.h"
#include "semLib.h"
#include "wdLib.h"
#include "memLib.h"
#include "sysLib.h"
#include "iv68k.h"
#include "dkbad.h"

#include "smreg.h"
#include "sm.h"

/* 
 * SM structures
 */

struct sm_device smdevice[Nsmrt] = {0}; /* Per-device struct (each logical) */

struct sm_drive smdrive[Nsm] = {0};	/* Per-drive struct (each physical) */

struct sm_ctlr smcinfo[NSM] = {0};	/* Per-controller struct */

/*
 * SM addresses
 */

struct addrst {
	struct smctlr  *smaddr;		/* registers, iopb */
	int             vec;		/* vector address */
} smaddrlist[NSM];


/*
 * Buffer used by smbadstrat. Only one transfer can be done at a time per
 * controller.
 */

struct sm_buf smbuf[NSM];			
struct sm_buf smbadbuf[NSM];

/*
 * Transfer struct to hold info for Interrupt code. Only one transfer can
 * be done at a time per controller , so only need one for each copntroller.
 */

struct SM_stat {
	USHORT sm_cmd;			/* command to issue */
	UINT sm_bn;			/* block number */
	UINT sm_addr;			/* physical address of transfer */
	int sm_nblks;			/* number of blocks to transfer */
	int sm_drv_vol;			/* drive/volume to issue */
	char sm_status;			/* set by interrupt code */
}	SM_stat[NSM];

extern char     vme_short[];

USHORT         *SMstd[] = {(USHORT *) & vme_short[0x0000],
			   (USHORT *) & vme_short[0x0200],
			   (USHORT *) & vme_short[0x0400],
			   (USHORT *) & vme_short[0x0600],
			   (USHORT *) & vme_short[0x0800],
			   0};

/*
 * UIB stuff - smuib_dflt used to read in UIB, uib_tmp used to set up smst.
 */

struct smuib    smuib_dflt = {0, 10, 0, 0, 100, 0, 512, 16, 32, 
				1, 3, 1024, 5, 0, 1, 0xFF};
struct smuib 	smuib_tmp;

/*
 * Misc
 */

int             smNum;		/* driver number of SM driver */
static char     devletters[] = {'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 0};
char            smiobuf[SM_SECTOR_SIZE * 2];

/*
 * Macros
 */

#define loword(X)	( (unsigned long)(X) & 0xffff )
#define hiword(X)	(((unsigned long)(X)>>16) & 0xffff)


/* for indexing into drive and device structures */
#define drv_idx(X,Y)	(X * (Nsm/NSM) + Y)
#define dev_idx(X,Y,Z)	(((Nsmrt/Nsm) * ((X * (Nsm/NSM)) + Y)) + Z)

/*
 * Forward Declarations
 */

int             smCreate();
int             smDelete();
int             smOpen();
int             smClose();
int             smRead();
int             smWrite();
int             smReset();
int             smIoctl();
int             smRdSec();
int             smWrtSec();
VOID            smInterrupt();
int		smioStart();


/**************************************************************************
*
* smDrv - install SM driver
*
*/

int 
smDrv()
{
	int             ctlr, drv;	/* indexes */
	static int      result;		/* 0 = not done; 1 = OK; -1 = ERROR */
	char            test = 0xff;	/* used for memory probe */
	struct sm_ctlr *cinfo;		/* pointer to ctlr struct */
	struct sm_device *smdev;	/* pointer to device struct */
	struct sm_diskst *st;		/* pointer to disk geom. struct */

	if (sysProcNumGet() != 0)
		return(ERROR);

	smNum = iosDrvInstall(smCreate, smDelete, smOpen, smClose, smRead, 
		smWrite, smIoctl);

	/* Look for controllers */

	for (ctlr = 0; ctlr < NSM; ctlr++) {
		cinfo = &smcinfo[ctlr];
		if (vxMemProbe(SMstd[ctlr], READ, 1, &test) != ERROR) {
			if (smInit(ctlr) == ERROR)
				printf("    ERROR INITIALIZING CONTROLLER %d\n",ctlr);
			else {
			   result = 1;	/* found a good controller */
			   intConnect(INUM_TO_IVEC(smaddrlist[ctlr].vec), smInterrupt, ctlr);
			   semInit(&cinfo->ctlr_sem);
			   semGive(&cinfo->ctlr_sem);
			   semInit(&cinfo->intr_sem);
			   cinfo->errcnt = 0;

			   /* 
			    * Create devices for each drive and read in
			    * the bad sector map for each using logical
			    * device 'c'.
			    */

			   for (drv = 0; drv < (Nsm / NSM); drv++) {
			     if (smdrive[drv_idx(ctlr, drv)].here) {
			        st = &smdrive[drv_idx(ctlr, drv)].diskst;
#ifdef	PRINTF_M
				printf("	%mM (%d x %d x %d) \n",
    				  st->st_size[2].nblocks, 
				  ST_nspt, ST_ntpc, ST_ncpd);
#else	PRINTF_M
				printf("	sm%d (%d x %d x %d) \n", 
				  drv_idx(ctlr,drv),
				  ST_nspt, ST_ntpc, ST_ncpd);
#endif	PRINTF_M
			        diskpart(&st->st_size[0],ST_nspt,
				  ST_ntpc,ST_ncpd - 1);

			   	smDevCreate(ctlr, drv);
			        semTake(&cinfo->ctlr_sem);
			        smdev = &smdevice[dev_idx(ctlr, drv, 2)];
			        smbadinit(smdev, &smbuf[ctlr],
				  &smdrive[drv_idx(ctlr, drv)].bad, 	
				  smioStart, ST_nspt, ST_ntpc, ST_ncpd);
			        semGive(&cinfo->ctlr_sem);
			     }			
			
			  }
		       }
		}
	}
	return (OK);
}

/**************************************************************************
*
* smInit - Initializes physical drives and set up disk geometry for
*	   both physical and logical drives.
*
*/

int 
smInit(ctlr)
	int             ctlr;
{
	register struct sm_iopb *iopb;	/* iopb pointer */
	struct smuib   *uib;		/* uib pointer */
	struct sm_diskst *st;		/* pointer to disk geometry struct */
	struct sm_ctlr *cinfo;		/* pointer to ctlr struct */
	struct smctlr *smadd;		/* ctlr registers */
	char *bufptr;			/* pointer to buff to read uib into */
	int  retry_cnt;			/* for reading UIB */
	int  drv, tmpvec;
	char *uibwind;			/* vmestd window for uib init xfers */

	cinfo = &smcinfo[ctlr];
	smaddrlist[ctlr].smaddr = (struct smctlr *) SMstd[ctlr];
	bufptr = smiobuf;
	smadd = smaddrlist[ctlr].smaddr;

	/* 
	 * Reset controller
	 */

	smadd->sm_cstatus |= CS_BDCLR;
	smadd->sm_cstatus &= ~CS_BDCLR;
	if (smbusy(smadd)) {
		printf("    CONTROLLER %d WILL NOT RESET\n",ctlr);
		return (ERROR);
	}
		
	/*
	 * Check for board OK 
	 */

	if ((smadd->sm_cstatus & CS_BOK) == 0) {
		printf("    CONTROLLER %d FAILED POWER UP DIAGNOSTICS\n",ctlr);
		return (ERROR);
	}

	smadd->sm_cstatus &= ~CS_SLED;

	/*
	 * Init portion of iopb that doesn't change 
	 */

	iopb = &smadd->sm_iopb[0];
	iopb->iopb_mem_type = MEMT_16BIT;
	iopb->iopb_adr_mod = ADRM_STD_N_D;
	iopb->iopb_int_level = SMINTLEV;
	tmpvec = freevec();
	iopb->iopb_n_vector = iopb->iopb_e_vector = smaddrlist[ctlr].vec = tmpvec;
	iopb->iopb_dma_burst = DMA_BURST;
	iopb->iopb_skew_offset = 0;
	printf("   SM%d at shortio 0x%04x  vector %02d  level %d\n",
	    	ctlr,
	    	(int)SMstd[ctlr]-(int)(vme_short), 
	    	smaddrlist[ctlr].vec,
		SMINTLEV);

	/*
	 * For each drive, clear any faults, supply default UIB and read UIB
	 * written during format 
	 */

	for (drv = 0; drv < (Nsm / NSM); drv+=2) {
		retry_cnt = 0;
		smdrive[drv_idx(ctlr, drv)].here = FALSE;

		/* clear faults */
		iopb->iopb_cmd = CMD_RECALB;
		iopb->iopb_coptions = (drv << 6);
		iopb->iopb_error_code = iopb->iopb_status = 0;
		smadd->sm_cstatus |= CS_GO;
		if (smwait(smadd) || (smadd->sm_cstatus & CS_ERR)) {
			if((iopb->iopb_error_code & ERR_NOT_SEL) == 0) {
			   printf("	sm%d at SM%d    ",
				drv_idx(ctlr,drv)*2,ctlr);
			   printf("    *** DRIVE FAILED RECALIBRATE 0x%x\n",
			     smadd->sm_Dint(drv),iopb->iopb_error_code);
			}
			smadd->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
			goto next;
		}
		smadd->sm_cstatus &= ~CS_DONE;

		/* supply default uib */
		uib = &smuib_tmp;
		bcopy(&smuib_dflt, uib, sizeof(struct smuib));

		/* get vdma window to default uib 
		 */
		uibwind = (char *)sysSetVdma (uib, sizeof (struct smuib));
		if (!uibwind) {
		    printf ("sm: could not allocate vdma for default uib\n");
		    return (ERROR);
		}
		iopb->iopb_buf_addrh = hiword((int)uibwind);
		iopb->iopb_buf_addrl = loword((int)uibwind);

		iopb->iopb_cmd = CMD_INIT;
		iopb->iopb_coptions = (drv << 6);
		iopb->iopb_error_code = iopb->iopb_status = 0;
		smadd->sm_cstatus |= CS_GO;
		if (smwait(smadd) || (smadd->sm_cstatus & CS_ERR)) {
			printf("	sm%d at SM%d    ",
				drv_idx(ctlr,drv)*2,ctlr);
			printf("    *** ERROR ON INIT 0x%x\n",
			    iopb->iopb_error_code);
			smadd->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
			sysGiveVdma (uibwind);
			goto next;
		}
		sysGiveVdma (uibwind);
		smadd->sm_cstatus &= ~CS_DONE;

		/*
		 * read UIB written to disk during format - read into 512 
		 * buffer first, then copy to uib. 
		 */

		/* get vdma window to bufptr
		 */
		uibwind = (char *)sysSetVdma (bufptr, sizeof (smiobuf));
		if (!uibwind) {
		    printf ("sm: could not allocate vdma for bufptr\n");
		    return (ERROR);
		}
retry:		iopb->iopb_cmd = CMD_READ;
		iopb->iopb_coptions = drv << 6;
		iopb->iopb_error_code = iopb->iopb_status = 0;
		iopb->iopb_dsk_addr.log.logh = UIB_DSK_ADDR >> 16;
		iopb->iopb_dsk_addr.log.logl = UIB_DSK_ADDR & 0xFFFF;
		iopb->iopb_blk_count = 1;
		iopb->iopb_buf_addrh = hiword((int)uibwind);
		iopb->iopb_buf_addrl = loword((int)uibwind);
		smadd->sm_cstatus |= CS_GO;
		if (smwait(smadd) || (smadd->sm_cstatus & CS_ERR)) {
			if (retry_cnt++ > SM_RETRYCNT) {
				printf("	sm%d at SM%d    ",
					drv_idx(ctlr,drv)*2,ctlr);
				printf("    *** ERROR READING UIB 0x%x\n",
			    		smadd->sm_Dint(drv),
					iopb->iopb_error_code);
				smadd->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
				sysGiveVdma (uibwind);
				goto next;
			}
			smadd->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
			goto retry;
		}
		/* flush the data cache */
		sysDataCacheFlush ();
		bcopy(bufptr, uib, sizeof(struct smuib));
		smadd->sm_cstatus &= ~CS_DONE;

		/*
		 * send real uib back to controller 
		 */

		iopb->iopb_buf_addrh = hiword((int)uibwind);
		iopb->iopb_buf_addrl = loword((int)uibwind);

		iopb->iopb_cmd = CMD_INIT;
		iopb->iopb_coptions = (drv << 6);
		iopb->iopb_error_code = iopb->iopb_status = 0;
		smadd->sm_cstatus |= CS_GO;
		if (smwait(smadd) || (smadd->sm_cstatus & CS_ERR)) {
			printf("	sm%d at SM%d    ",
				drv_idx(ctlr,drv)*2,ctlr);
			printf("    *** ERROR ON INIT 0x%x\n",
			    smadd->sm_Dint(drv),iopb->iopb_error_code);
			smadd->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
			sysGiveVdma (uibwind);
			goto next;
		}
		smadd->sm_cstatus &= ~CS_DONE;
		sysGiveVdma (uibwind);

		/*
		 * Check UIB values 
		 */

		if((uib->uib_vol_head[0].num_heads < 2) || 
		  (uib->uib_vol_head[0].num_heads > 32) ||
		  ((uib->uib_num_cyl < 50) || (uib->uib_num_cyl > 5000)) ||
		  ((uib->uib_sec_trk < 10) || (uib->uib_sec_trk > 100))) {
			printf("	sm%d at SM%d    ",
				drv_idx(ctlr,drv)*2,ctlr);
			printf("    *** INVALID UIB heads=%d cyl=%d sec/trk=%d\n",
				uib->uib_vol_head[0].num_heads,
				uib->uib_num_cyl, uib->uib_sec_trk);
			goto next;
		}

		/*
		 * Mark drive as here and set up disk geometry struct 
		 */

		smdrive[drv_idx(ctlr, drv)].here = TRUE;
		smGeom(ctlr, drv, 0, uib);
		if(uib->uib_vol_head[1].num_heads != 0) {
			smdrive[drv_idx(ctlr, drv+1)].here = TRUE;
			smGeom(ctlr, drv+1, 1, uib);
		}
next:	}
	return (OK);
}

/***********************************************************************
*
* smGeom - set up disk geometry struct
*
*/

smGeom(ctlr, drive, vol, uib)
	char ctlr, drive, vol;
	struct smuib   *uib;		/* uib pointer */
{
	struct sm_diskst *st;		/* pointer to disk geometry struct */
	st = &smdrive[drv_idx(ctlr, drive)].diskst;

	/*
	 * Set up disk geometry struct 
	 */

	ST_ncpd = uib->uib_num_cyl;
	ST_nspt = uib->uib_sec_trk;
	ST_ntpc = uib->uib_vol_head[vol].num_heads;
	ST_nspc = ST_ncpd * ST_nspt * ST_ntpc;
	ST_spare1 = uib->uib_bytes_sec;
}

/**************************************************************************
*
* smDevCreate - called by smDrv; assumes RT-11; creates as many RT-11 
* logical devices as will fit on the disk.
*
*/

int 
smDevCreate(ctlr, drv)
	char            ctlr, drv;
{
	struct sm_drive *smdrv;		/* pointer to smdrive struct */
	struct sm_device *smdev;	/* pointer to smdevice struct */
	struct sm_diskst  *st;		/* pointer to disk geometry struct */
	char            i;
	int             status = OK;
	int 		tmpsize;

	smdrv = &smdrive[drv_idx(ctlr, drv)];
	st = &smdrv->diskst;

	/*
	 * Create 8 logical devices using the partition info set up by
	 * diskpart, set up the smdevice structure for each device.
	 */ 

	for (i = 0; i < 8; i++) {
		smdev = &smdevice[dev_idx(ctlr, drv, i)];
		if ( st->st_size[i].nblocks == 0 )
			continue;
		smdev->ctlr = ctlr;
		smdev->drive = drv;
		smdev->partition = i;
		if(st->st_size[i].nblocks < 65536)
			tmpsize = st->st_size[i].nblocks;
		else
			tmpsize = 65535;
		status = rt11DevInit(
			   	(RT_VOL_DESC *) smdev,
				ST_spare1,
				ST_nspt,
				tmpsize,
			   	FALSE,
			   	RT_FILES_FOR_2_BLOCK_SEG,
			   	smRdSec,
			   	smWrtSec,
			   	NULL
			);
		if (status != OK) {
			printf("    sm%d at SM%d    ",drv_idx(ctlr,drv),ctlr);
			printf("    *** smDevCreate FAILED rtDevInit\n");
			return (ERROR);
		}
		sprintf(smdev->name, "%s%d%c", 
			SM_DEV_NAME, drv_idx(ctlr,drv), devletters[i]);
		iosDevAdd((DEV_HDR *) smdev, smdev->name, smNum);
	}
}

/**************************************************************************
*
* smCreate - create an RT-11 file
*
* RETURNS: file descriptor number, or ERROR
*
*/

int 
smCreate(smdev, name, mode)
	register struct sm_device *smdev;	/* pointer to SM device
						 * descriptor */
	char           *name;	/* name of file to open */
	int             mode;	/* access mode */

{
	RT_FILE_DESC   *pFd;

	pFd = rt11Create((RT_VOL_DESC *) smdev, name, mode);

	if (pFd == NULL)
		return (ERROR);

	return ((int) pFd);
}

/**************************************************************************
*
* smDelete - delete an RT-11 file
*
* RETURNS OK | ERROR
*
*/

int 
smDelete(smdev, name)
	register struct sm_device *smdev;	/* pointer to SM device
						 * descriptor */
	char           *name;	/* name of file to open */

{
	return (rt11Delete((RT_VOL_DESC *) smdev, name));
}

/**************************************************************************
*
* smOpen - open an RT-11 file
*
* RETURNS: file descriptor number, or ERROR
*
*/

int 
smOpen(smdev, name, mode)
	register struct sm_device *smdev;	/* pointer to SM device
						 * descriptor */
	char           *name;	/* name of file to open */
	int             mode;	/* access mode */

{
	RT_FILE_DESC   *pFd;

	pFd = rt11Open((RT_VOL_DESC *) smdev, name, mode);

	if (pFd == NULL)
		return (ERROR);

	return ((int) pFd);
}

/**************************************************************************
*
* smClose - close an RT-11 file
*
*/

int 
smClose(pFd)
	RT_FILE_DESC   *pFd;	/* file descriptor of file to close */

{
	return (rt11Close(pFd));
}

/**************************************************************************
*
* smRead - read from an RT-11 file
*
* RETURNS: number of bytes read, or 0 if end of file, or ERROR if read error.
*
*/

int 
smRead(pFd, buffer, maxbytes)
	RT_FILE_DESC   *pFd;	/* file descriptor of file to close */
	char           *buffer;	/* buffer to receive data */
	int             maxbytes;	/* max bytes to read in to buffer */

{
	return (rt11Read(pFd, buffer, maxbytes));
}

/**************************************************************************
*
* smWrite - write to an RT-11 file
*
* RETURNS: number of bytes written, or ERROR
*
*/

int 
smWrite(pFd, buffer, nbytes)
	RT_FILE_DESC   *pFd;	/* file descriptor of file to close */
	char           *buffer;	/* buffer to be written */
	int             nbytes;	/* number of bytes to write from buffer */

{
	return (rt11Write(pFd, buffer, nbytes));
}

/**************************************************************************
*
* smReset - neccesary; called by rtLib routines.
*
*/

int 
smReset()
{
	return (OK);
}

/**************************************************************************
*
* smIoctl - do device specific control function
*
*/

int 
smIoctl(pFd, function, arg)
	RT_FILE_DESC   *pFd;		/* file descriptor of file to control */
	int             function;	/* function code */
	int             arg;		/* some argument */

{
	return (rt11Ioctl(pFd, function, arg));
}

/**************************************************************************
*
* smInterrupt
*
*/

VOID 
smInterrupt(ctlr)
	char            ctlr;
{
	struct smctlr  *smadd;		/* pointer to smctlr struct */
	struct sm_iopb *iopb;		/* pointer to iopb struct */
	struct sm_ctlr *cinfo;		/* pointer to ctlr info struct */
	struct SM_stat *stat;		/* transfer struct */
	char            drive;		/* drive that interrupt occured on */
	UINT		i;

	smadd = smaddrlist[ctlr].smaddr;
	iopb = &smadd->sm_iopb[0];
	cinfo = &smcinfo[ctlr];
	stat = &SM_stat[ctlr];
	drive = smadd->sm_cstatus & CS_SCHGSRC;


	/*
	 * If there was a SOFT error - retry up to eight times. If 
	 * retry count is exceeded set cinfo status to ERROR.
	 */

	if (smadd->sm_cstatus & CS_ERR) {
		if (cinfo->errcnt < SM_MAX_ERRORS) {	/* soft error - retry */
			cinfo->errcnt++;
			smadd->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
			smio(ctlr,drive,stat); 
			return;
		} else {	/* hard error */
			printf("    sm%d at SM%d    ",
				drv_idx(ctlr,drive)*2,ctlr);
			i = (iopb->iopb_dsk_addr.log.logh << 16) |
			    (iopb->iopb_dsk_addr.log.logl&0xFFFF);
			printf("    *** esn=%d HARD %s 0x%x\n",
			    i,
			    (iopb->iopb_cmd == CMD_READ) ? "READ" : "WRITE",
			    iopb->iopb_error_code);
			stat->sm_status = ERROR;
			goto opdone;
		}
	}

	/* 
	 * OK - or less than eight retrys 
	 */

	if (cinfo->errcnt) {
		printf("    sm%d at SM%d    ",drv_idx(ctlr,drive)*2,ctlr);
		printf("    *** WARNING SOFT ERROR esn=%d %s\n",
		    stat->sm_bn,
		    (iopb->iopb_cmd == CMD_READ) ? "READ" : "WRITE");
	}
	stat->sm_status = OK;

opdone: cinfo->errcnt = 0;
	smadd->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
	semGive(&cinfo->intr_sem);
	return;
}

/**************************************************************************
* 
* smRdSec - read a sector from an RT-11 volume.
*
*/
int 
smRdSec(smdev, secNum, buffer)
	struct sm_device *smdev;	/* pointer to SM device descriptor */
	int             secNum;		/* number of logical sector to read */
	char           *buffer;		/* buffer to receive block */
{
	struct sm_ctlr *cinfo;		/* pointer to ctlr info struct */
	struct sm_drive *drv;		/* pointer to drive info struct */
	struct sm_buf *bp;			/* pointer to buf struct for smbadstrat */
	struct sm_diskst *st;		/* pointer to diskst struct */

	cinfo = &smcinfo[smdev->ctlr];
	semTake(&cinfo->ctlr_sem);

	drv = &smdrive[drv_idx(smdev->ctlr,smdev->drive)];
	bp = &smbuf[smdev->ctlr];
	st = &drv->diskst;

	/*
	 * Fill in buf struct for smbadstrat, call smbadstrat which will
	 * call smioStart with appropriate sectors.
	 */

	bp->b_flags = B_READ;
	bp->b_bcount = 512 * 1;		/* count in bytes (1 block for now) */
	bp->b_addr = (UINT) buffer;
	bp->b_blkno = secNum + 
	    (ST_nspt * ST_ntpc * (st->st_size[smdev->partition].cyloff + 1));
	
	smbadstrat(smdev, bp, &smbadbuf[smdev->ctlr], &drv->bad, smioStart, 
		ST_nspt, ST_ntpc, ST_ncpd,
		st->st_size[smdev->partition].cyloff);
	semGive(&cinfo->ctlr_sem);
	return (bp->b_flags & B_ERROR);

}

/**************************************************************************
* 
* smWrtSec - write a sector to an RT-11 volume.
*
*/
int 
smWrtSec(smdev, secNum, buffer)
	register struct sm_device *smdev;	/* pointer to SM device
						 * descriptor */
	int             secNum;	/* number of logical sector to read */
	char           *buffer;	/* buffer to write */
{
	struct sm_ctlr *cinfo;		/* pointer to ctlr info struct */
	struct sm_drive *drv;		/* pointer to drive info struct */
	struct sm_buf *bp;		/* pointer to buf struct for smbadstrat */
	struct sm_diskst *st;		/* pointer to diskst struct */

	cinfo = &smcinfo[smdev->ctlr];
	semTake(&cinfo->ctlr_sem);

	drv = &smdrive[drv_idx(smdev->ctlr,smdev->drive)];
	bp = &smbuf[smdev->ctlr];
	st = &drv->diskst;

	/*
	 * Fill in buf struct for smbadstrat, call smbadstrat which will
	 * call smioStart with appropriate sectors.
	 */

	bp->b_flags = B_WRITE;
	bp->b_bcount = 512 * 1;		/* count in bytes (1 block for now) */
	bp->b_addr = (UINT) buffer;
	bp->b_blkno = secNum + 
	    (ST_nspt * ST_ntpc * (st->st_size[smdev->partition].cyloff + 1));
	
	smbadstrat(smdev, bp, &smbadbuf[smdev->ctlr], &drv->bad, smioStart, 
		ST_nspt, ST_ntpc, ST_ncpd,
		st->st_size[smdev->partition].cyloff);
	semGive(&cinfo->ctlr_sem);
	return (bp->b_flags & B_ERROR);

}

/**************************************************************************
*
* smioStart - start I/O ; called by smbadstrat 
*
*/

int 
smioStart(smdev, bp)
	struct sm_device *smdev;	/* pointer to smdevice struct */
	struct sm_buf *bp;		/* I/O transfer info */
{
	struct SM_stat *stat;		/* pointer to transfer struct */
	struct smctlr *smadd;		/* ctlr registers */
	int             track;		/* physical track */

	if (bp->b_blkno < 0) {
		printf("smioStart: Error - funny secNum = %d\n", bp->b_blkno);
		return (ERROR);
	}

	smadd = smaddrlist[smdev->ctlr].smaddr;
	stat = &SM_stat[smdev->ctlr];

	/*
	 * Save data transfer info for interrupt code retrys and 
	 * start I/O
	 */ 

	stat->sm_cmd = (bp->b_flags & B_READ) ? CMD_READ : CMD_WRITE;

	/* get a vdma window into standard vme
	 */
	stat->sm_addr = (UINT) sysSetVdma (bp->b_addr, bp->b_bcount);
	if (!stat->sm_addr) {
	    printf ("sm: could not allocate vdma registers\n");
	    return (ERROR);
	}
	stat->sm_bn = bp->b_blkno; 
	stat->sm_nblks = bp->b_bcount / 512;

	/* flush the data cache */
	sysDataCacheFlush ();

	smio(smdev->ctlr,smdev->drive,stat);

	/*
	 * Wait for interrupt code to complete, then return status.
	 */

	semTake(&smcinfo[smdev->ctlr].intr_sem);
	/* give back vdma registers */
	sysGiveVdma (stat->sm_addr);
	if(stat->sm_status != OK)
		bp->b_flags |= B_ERROR;
}

smio(ctlr,drive,stat)
	char ctlr;
	char drive;
	struct SM_stat *stat;		/* pointer to transfer struct */
{
	struct smctlr *smadd;		/* ctlr registers */
	struct sm_iopb *iopb;		/* pointer to iopb struct */
	struct sm_diskst *st;		/* pointer to disk geometry struct */
	int             track;		/* physical track */
	UINT		i;

	smadd = smaddrlist[ctlr].smaddr;
	iopb = &smadd->sm_iopb[0];
	st = &smdrive[drv_idx(ctlr, drive)].diskst;

	/*
	 * Set up iopb, compute physical sector, track, cylinder.
	 */

	track = stat->sm_bn / ST_nspt;
	iopb->iopb_cmd = stat->sm_cmd;
	iopb->iopb_error_code = iopb->iopb_status = 0;
	iopb->iopb_blk_count = stat->sm_nblks;

	/* sm_addr already contains adjustments to access a vdma window 
	 */
	iopb->iopb_buf_addrh = hiword((int)stat->sm_addr);
	iopb->iopb_buf_addrl = loword((int)stat->sm_addr);

	iopb->iopb_coptions = (drive << 6) | COP_LOG_TRAN | COP_ECC_EN | COP_INT_EN; 
	iopb->iopb_dsk_addr.log.logh = hiword(stat->sm_bn);
	iopb->iopb_dsk_addr.log.logl = loword(stat->sm_bn);
	i = (iopb->iopb_dsk_addr.log.logh << 16) |
	    (iopb->iopb_dsk_addr.log.logl&0xFFFF);
		

	smadd->sm_cstatus |= CS_GO;

}

/**************************************************************************
*
* smwait - wwwwaaaaaiiiiitttttt
*
*/

smwait(smaddr)
	struct smctlr  *smaddr;
{
	long            l = 0;

	SM_DELAY(400);
	while ((smaddr->sm_cstatus & CS_DONE) == 0) {
		if (++l == 50000)
			return (-1);
		SM_DELAY(400);
	}
	return (0);
}

/***********************************************************************
*
* smbusy
*
*/

smbusy(smadd)
struct smctlr *smadd;
{ 	
	long l = 0;

	SM_DELAY(400);
	while (smadd->sm_cstatus & CS_BUSY)  {
		if (++l == 50000)
			return (-1);
		SM_DELAY(400);
	}
	return (0);
}

/**************************************************************************
*
* SM_DELAY - SSSSSMMMMM_____DDDDDEEEEELLLLLAAAAAYYYYY
*
*/

SM_DELAY(n)
{
	while (n--);
}

/**************************************************************************
*
* smMkfs - make a file system on the named logical device
*
*/

STATUS 
smMkfs(name)
	char           *name;	/* Device name */

{
	int             fd;

	fd = open(name, WRITE);

	if (fd == ERROR) {
		printf("smMkfs: error opening sm device %s.\n", name);
		return (ERROR);
	}
	if (ioctl(fd, FIODISKINIT) == ERROR) {
		printf("smMkfs: error initializing sm device %s.\n", name);
		close(fd);
		return (ERROR);
	}
	close(fd);

	return (OK);
}
/************************************************************
*
* testsec - For testing particular sectors. Sector 0
* is protected though.
*
*/

testsec(secNum, cmd, buffer, nblks,ctlr,drive)
	int secNum;
	char cmd;
	char *buffer;
	int nblks;
	char ctlr,drive;
{
	struct sm_device *smdev;	/* pointer to SM device */
	struct sm_ctlr *cinfo;		/* pointer to ctlr info struct */
	struct sm_drive *drv;		/* pointer to drive info struct */
	struct sm_buf *bp;		/* pointer to buf struct for smbadstrat */
	struct sm_diskst *st;		/* pointer to diskst struct */

	smdev = &smdevice[dev_idx(ctlr,drive,2)];
	cinfo = &smcinfo[smdev->ctlr];
	semTake(&cinfo->ctlr_sem);

	drv = &smdrive[drv_idx(smdev->ctlr,smdev->drive)];
	bp = &smbuf[smdev->ctlr];
	st = &drv->diskst;

	/*
	 * Fill in buf struct for smbadstrat
	 */

	if((secNum == 0) && (cmd == B_WRITE)) {
		printf("Don't write over sector 0 bozo !\n");
		return;
	}

	bp->b_flags = cmd;
	bp->b_bcount = 512 * nblks;
	bp->b_addr = (UINT) buffer;
	bp->b_blkno = secNum; 
	
#ifdef	DEBUG2
	printf("testsec, flags = 0x%x, bcount = %d, b_addr = 0x%x, blkno = %d\n",
		bp->b_flags, bp->b_bcount, bp->b_addr, bp->b_blkno);
#endif	DEBUG2

	smbadstrat(smdev, bp, &smbadbuf[smdev->ctlr], &drv->bad, smioStart, 
		ST_nspt, ST_ntpc, ST_ncpd,
		st->st_size[smdev->partition].cyloff);
	semGive(&cinfo->ctlr_sem);
	return (bp->b_flags & B_ERROR);

}

/***********************************************************************
*
* testsec1 - add cylinder offset and call testsec
*
*/

testsec1(secNum, cmd, buffer, nblks, ctlr, drive)
{
	struct sm_diskst *st;
	st = &(smdrive[drv_idx(ctlr,drive)].diskst);
	secNum += secNum + (ST_nspt * ST_ntpc * 1);
	testsec(secNum, cmd, buffer, nblks, ctlr, drive);
}

/*#define BAD144DEBUG	1 /**/

#define DEV_BSIZE	512
#define DEV_BSHIFT	9

/************************************************************************
*
* smbadstrat - Check the bad sector map 
*
*/

VOID smbadstrat (qi, bp, bbp, bad, strat, nspt, ntpc, ncpd, coff)
	struct sm_device *qi;		/* device struct */
	struct sm_buf *bp;			/* buf for I/O transfer stuff */
	struct sm_buf *bbp;		/* buf for I/O transfer stuff */
	struct dkbad *bad;		/* contains bad sector map info */
	int (*strat)();			/* I/O strategy routine */
	USHORT nspt, ntpc, ncpd, coff;
{
	int s, hadbad = 0, len = bp->b_bcount, error;
	UINT blkno;
	int i, hi, lo;
	int sbn = bp->b_blkno;
	int lbn = sbn + ((len+(DEV_BSIZE-1)) >> DEV_BSHIFT) - 1;
	int bbn, nbs;

#define SAVE_BFLAGS	(B_WRITE|B_READ|B_BUSY|B_PHYS|B_AGE|B_DELWRI| \
			B_TAPE|B_UAREA|B_PAGET|B_DIRTY|B_PGIN|B_CACHE| \
			B_INVAL|B_LOCKED|B_HEAD|B_BAD)


	/* 
	 * Binary search the bad sector map for remapped sectors 
	 */

	if (bad->bt_csn == 1 && bad->bt_mbz) {
	    lo = 0;
	    while (sbn <= lbn) {
	    	hi = bad->bt_mbz-1;
		while ((hi - lo) > 1) {
			i = lo + (hi - lo)/2;
			if (bad->bt_bad[i].bt_bn == sbn)
				hi = lo = i;
			else if (bad->bt_bad[i].bt_bn < sbn)
				lo = i;
			else
				hi = i;
		}

		if (bad->bt_bad[lo].bt_bn >= sbn)
			hi = lo;
		bbn = bad->bt_bad[hi].bt_bn;

		if (!hadbad) {
			if (bbn < sbn || bbn > lbn) {
				goto done;		/* no remaps */
			}
			hadbad++;
			*bbp = *bp;
			bbp->b_flags &= SAVE_BFLAGS;
		}

		if (bbn < sbn || bbn > lbn) {	/* transfer from sbn to lbn */
		    bbp->b_blkno = sbn;
		    bbp->b_bcount = len;
		    bbp->b_flags &= SAVE_BFLAGS;
#ifdef BAD144DEBUG
		    printf("smbadstrat, calling strat with blkno=%d\n",
			bbp->b_blkno);
#endif BAD144DEBUG
		    (*strat)(qi, bbp);
		    len -= bbp->b_bcount-bbp->b_resid;
		    sbn = lbn+1;
		} else {
		    if (sbn < bbn) {	/* transfer from sbn to bad block bbn */
			    bbp->b_blkno = sbn;
			    bbp->b_bcount = min(len,(bbn - sbn) * 512);
			    bbp->b_flags &= SAVE_BFLAGS;
#ifdef BAD144DEBUG
		    printf("smbadstrat, calling strat with blkno=%d\n",
			bbp->b_blkno);
#endif BAD144DEBUG
			    (*strat)(qi, bbp);
			    bbp->b_addr += bbp->b_bcount;
			    len -= bbp->b_bcount-bbp->b_resid;
			    if (bbp->b_flags & B_ERROR)
				    break;
		    }

		    /* transfer remapped bbn */
		    bbp->b_blkno = ((ncpd*ntpc*nspt) - nspt - 1) - hi;
		    bbp->b_bcount = min(len, 512);
		    bbp->b_flags &= SAVE_BFLAGS;
#ifdef BAD144DEBUG
		    printf("smbadstrat, calling strat with blkno=%d\n",
			bbp->b_blkno);
#endif BAD144DEBUG
		    (*strat)(qi, bbp);
		    bbp->b_addr += bbp->b_bcount;
		    len -= bbp->b_bcount-bbp->b_resid;
		    sbn = bbn+1;
		}
		if (bbp->b_flags & B_ERROR)
			break;
	    }
	    bp->b_resid = len;
	    if (bbp->b_flags & B_ERROR) {
		    bp->b_flags |= B_ERROR;
	    }
	    bbp->b_flags = 0;
	    return;
	}
#ifdef BAD144DEBUG
    printf("smbadstrat, calling strat with blkno=%d\n",
	bp->b_blkno);
#endif BAD144DEBUG
done:	(*strat)(qi, bp);
}

/************************************************************************
*
* smbadinit - Read in bad sector map
*
*/

VOID smbadinit (qi, bbp, bad, strat, nspt, ntpc, ncpd)
	struct sm_device *qi;		/* pointer to device struct */
	struct sm_buf *bbp;		/* a buffer for I/O transfer */
	struct dkbad *bad;		/* bad struct to put the map into */
	int (*strat) ();
	USHORT nspt, ntpc, ncpd;
{
	int i, error;
	UINT blkno;
	register int nbs;

#ifdef BAD144DEBUG
	printf("smbadinit, &qi=0x%x, &bbp=0x%x, &bad=0x%x, nspt=%d, ntpc=%d, ncpd=%d\n",
		qi, bbp, bad, nspt, ntpc, ncpd);
#endif BAD144DEBUG

	/*
	 * Read in the bad sector map
	 */

	nbs = 0;
	bbp->b_flags = B_READ;
	bbp->b_bcount = sizeof (struct dkbad);
	bbp->b_addr = (UINT) bad;
    	bbp->b_blkno = blkno = ncpd * nspt * ntpc - nspt;
   	(*strat)(qi, bbp);

    	error = bbp->b_flags & B_ERROR;
    	bbp->b_flags = 0;

    	if (error) {
	    	printf("%.3s: cannot read bad sector map\n", &(qi->name)[1]);
	    	bad->bt_csn = -1;
	    	goto initdone;	/* so we'll keep trying */
    	}

#ifdef	BAD144DEBUG
    	printf("%.3s: read bad144 map from block %d\n", &(qi->name)[1], blkno);

	printf("First 10 maps ...\n\n");
	for(i = 0; i < 10; i++) {
		printf("map %d cyl=%d trk=%d sec=%d\n",
			i,
			bad->bt_bad[i].bt_cyl,
			((bad->bt_bad[i].bt_trksec>>8)&0xff),
			(bad->bt_bad[i].bt_trksec&0xff)); 
	}

#endif	BAD144DEBUG

    	if (bad->bt_csn != 0) {
		for (i = 0; i < MAXBADBLK; i++) {
		    if (bad->bt_bad[i].bt_cyl == 0xFFFF) {
			bad->bt_bad[i].bt_bn = -1;
		    }
		    else if ((bad->bt_bad[i].bt_cyl > ncpd) ||
			(((bad->bt_bad[i].bt_trksec>>8)&0xff)>ntpc)||
			((bad->bt_bad[i].bt_trksec&0xff) > nspt) ) {
			    printf("%.3s: bad bad144 map\n", &(qi->name)[1]);
			    bad->bt_csn = -1;
			    break;
		    } 

		    else {

#ifdef	BAD144DEBUG
			    printf("%.3s:(cyl %d,trk %d,sec %d)",
				&(qi->name)[1],
				bad->bt_bad[i].bt_cyl,
				(bad->bt_bad[i].bt_trksec>>8) & 0xff,
				bad->bt_bad[i].bt_trksec&0xff);
#endif	BAD144DEBUG

			    bad->bt_bad[i].bt_bn = 
				(((bad->bt_bad[i].bt_cyl * ntpc) +
				((bad->bt_bad[i].bt_trksec>>8)&0xff))
				*nspt)+
				(bad->bt_bad[i].bt_trksec&0xff);
			    nbs++;

#ifdef	BAD144DEBUG
			    printf("[blk %d] remapped to %d\n",
				bad->bt_bad[i].bt_bn, blkno - nbs);
#endif	BAD144DEBUG
			    
		    }
		    
		}
		if(bad->bt_csn != -1)
			printf("%.3s: %d remapped sectors\n", 
				&(qi->name)[1], nbs);
		bad->bt_csn = 1;
		bad->bt_mbz = nbs;
    	} 
	else
initdone:	bad->bt_csn = -1;
}

