/* devLib.c - miscellaneous ISI driver support routines */

static char *copyright = "Copyright 1989, Integrated Solutions, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION

This module contains miscellaneous driver support routines.
 
*/

#include "vxWorks.h"
#include "types.h"
#include "param.h"
#include "iv68k.h"
#include "rt11Lib.h"
#include "dkbad.h"

extern int excIntStub();

/****************************************************************************
*
* diskpart - calculate standard disk partition sizes.
*
*/

#define	NPARTITIONS	8
#define	PART(x)		(x - 'a')
#define roundup(x, y)   (((unsigned int)(((x)+((y)-1)))/(unsigned int)(y))*(y))

/* Default partition sizes, where they exist.  */
#define	NDEFAULTS	5
#define	SMALLDISK	3
static int	defpart[NDEFAULTS][NPARTITIONS] = {
   { 15884, 66880, 0, 15884, 307200, 0, 0, 291346 },	/* ~ 356+ Mbytes */
   { 15884, 33440, 0, 15884, 55936, 0, 0, 291346 },	/* ~ 206-355 Mbytes */
   { 15884, 33440, 0, 15884, 55936, 0, 0, 0 },		/* ~ 61-205 Mbytes */
   { 15884, 10032, 0, 15884, 0, 0, 0, 0 },		/* ~ 20-60 Mbytes */
   { 0, 0, 0, 0, 0, 0, 0, 0 },				/* < 20 Mbytes */
};

/*
 * Each array defines a layout for a disk; that is, the collection of partitions
 * totally covers the physical space on a disk.
 */
#define	NLAYOUTS	3
static char	layouts[NLAYOUTS][NPARTITIONS] = {
   { 'a', 'b', 'h', 'g' },
   { 'a', 'b', 'h', 'd', 'e', 'f' },
   { 'c' },
};

/*
 * Each disk has some space reserved for a bad sector forwarding table.  DEC 
 * standard 144 uses the first 5 even numbered sectors in the last track of the
 * last cylinder for replicated storage of the bad sector table; another 
 * MAXBADBLK sectors past this is needed as a pool of replacement sectors.
 */
struct size {
	daddr_t	nblocks;
	int	cyloff;
};

diskpart(size, nspt, ntpc, ncpd, nbadblks)
	register struct size	*size;
{
	register int curcyl, def, part, layout;
	int 	threshhold, 
		numcyls[NPARTITIONS], 
		numblks[NPARTITIONS], 
		startcyl[NPARTITIONS];
	char 	*lp;
	int	badsecttable;		/* # sectors */
	int	xxx;
	register struct size	*ssize = size;
	int	nspc = nspt * ntpc;

	/*
	 * Bad sector table contains one track for the replicated copies of the
	 * table and enough full tracks preceding the last track to hold the 
	 * pool of free blocks to which bad sectors are mapped.
	 */
	if (nbadblks)
		badsecttable = nbadblks;
	else
		badsecttable = OMAXBADBLK;
	badsecttable = nspt + roundup(badsecttable, nspt);
	threshhold = howmany(nspc, badsecttable);

	/* 
	 * Figure out if disk is large enough for expanded swap area and 'd', 
	 * 'e', and 'f' partitions.
	 */
	for (def = 0; def < NDEFAULTS; def++) {
		curcyl = 0;
		for (part = PART('a'); part < NPARTITIONS; part++)
			curcyl += howmany(defpart[def][part], nspc);
		if (curcyl < ncpd - threshhold)
			break;
	}

	if (def >= NDEFAULTS)
		return (-1);

	/*
	 * Calculate number of cylinders allocated to each disk partition. We 
	 * may waste a bit of space here, but it's in the interest of 
	 * compatibility (for mixed disk systems).
	 */
	for (curcyl = 0, part = PART('a'); part < NPARTITIONS; part++) {
		numcyls[part] = 0;
		numblks[part] = 0;
		if (defpart[def][part] != 0) {
			numcyls[part] = howmany(defpart[def][part], nspc);
			numblks[part] = defpart[def][part];
			curcyl += numcyls[part];
		}
	}
	numcyls[PART('f')] = ncpd - curcyl;
	numblks[PART('f')] = numcyls[PART('f')] * nspc - badsecttable;
	numcyls[PART('g')] =
		numcyls[PART('d')] + numcyls[PART('e')] + numcyls[PART('f')];
	numblks[PART('g')] = numcyls[PART('g')] * nspc - badsecttable;
	numcyls[PART('c')] = ncpd;
	numblks[PART('c')] = numcyls[PART('c')] * nspc;

	/*
	 * Calculate starting cylinder number for each partition. Note the 'h' 
	 * partition is physically located before the 'g' or 'd' partition.  
	 * This is reflected in the layout arrays defined above.
	 */
	for (layout = 0; layout < NLAYOUTS; layout++) {
		curcyl = 0;
		for (lp = layouts[layout]; *lp != 0; lp++) {
			if (numcyls[PART(*lp)])
				startcyl[PART(*lp)] = curcyl;
			else
				startcyl[PART(*lp)] = 0;
			curcyl += numcyls[PART(*lp)];
		}
	}

	if (defpart[def][PART('a')] == 0) {
	    xxx = numblks[PART('f')];
	    part = PART('a');

	    if (xxx >= defpart[SMALLDISK][part]) {
		startcyl[part] = 0;
		numcyls[part]=howmany( defpart[SMALLDISK][part], nspc);
		numblks[PART('a')] = defpart[SMALLDISK][part];
		xxx -= numcyls[part] * nspc;
		part = PART('b');
		if (xxx >= defpart[SMALLDISK][part]) {
		    startcyl[part] = startcyl[PART('a')]+numcyls[PART('a')];
		    numcyls[part] = howmany( defpart[SMALLDISK][part], nspc);
		    numblks[part] = defpart[SMALLDISK][part];
		    xxx -= numcyls[part] * nspc;
		    part = PART('g');
		    if (xxx >= 0) {
		    	startcyl[part] = startcyl[PART('b')]+numcyls[PART('b')];
		    	numcyls[part] = howmany( xxx, nspc);
		    	numblks[part] = xxx;
		    }
	        } else {
		    startcyl[part] = startcyl[PART('a')]+numcyls[PART('a')];
		    numcyls[part] = howmany( defpart[SMALLDISK][part], nspc);
		    numblks[part] = defpart[SMALLDISK][part];
		}
	    } else {
		startcyl[part] = 0;
		numcyls[part] = howmany( defpart[SMALLDISK][part], nspc);
		numblks[part] = defpart[SMALLDISK][part];
	    }
	}

	for (part = PART('a'); part < NPARTITIONS; part++, size++) {
		size->nblocks = numblks[part];
		size->cyloff = startcyl[part];
	}
#define DO_PRINTPART
#ifdef DO_PRINTPART
	printpart(ssize, nspc);
#endif DO_PRINTPART
}

/****************************************************************************
*
* printpart - print partition info
*
*/

printpart(isize, nspc)
	register struct size	*isize;
{
	int part, ppart;
	int ip, overlap = 0;
	register struct size	*size = isize;
	register struct size	*tsize;
	char pstr[80], tmpstr[80];

	ppart = 0;
	printf("\n		");
	for (part = PART('a'); part < NPARTITIONS; part++, size++) {
		if (size->nblocks == 0)
			continue;
		sprintf(tmpstr,"%c",'a'+part);
		strcpy(pstr,tmpstr);
		overlap = 0;

		for (tsize=isize, ip=PART('a'); ip < NPARTITIONS; ip++, tsize++)
			if (ip != part && tsize->nblocks != 0 &&
			    tsize->cyloff >= size->cyloff && 
			    tsize->cyloff*nspc+tsize->nblocks <=
			    size->cyloff*nspc+size->nblocks) {
				if (!overlap) strcat(pstr,":");
				sprintf(tmpstr,"%c",'a'+ip);
				strcat(pstr,tmpstr);
				overlap++;
			}
		sprintf(tmpstr,":%d", size->nblocks);
		strcat(pstr,tmpstr);
		if((ppart == 4) || ((ppart == 3) && overlap)) {
			printf("\n		");
			printf("%s",pstr);
			ppart = 0;
		}
		else {
			printf("%s",pstr);
			ppart++;
		}
		printf("	");
		if (size->nblocks < 100000)
			printf("	");
	}
	printf("\n");
}

/*************************************************************************
*
* freevec - Find an un-used user vector.
*
* RETURNS - vector NUMBER
*/

int freevec ()
{
	int vec;

	for(vec = 64; vec < 256; vec+=1) {
		if (intVecGet(INUM_TO_IVEC(vec)) == (int)excIntStub) {
			return(vec);
		}
	}
}
