/* picture.c - routines to present target configuration graphically */

/*
modification history
--------------------
02g,29mar89,shl  added bcmp routine since force system V does not have it.
02f,12sep88,gae  documentation.
02e,22jul88,gae  fixed spelling of Excelan, and fixed prompting glitch.
	   +pht  added 0 and 1 to key.
02d,25jun88,gae  changed calls for Excelan board.  Fixed prompting.
		 added interactive parm to "drawboard" invocation.
02c,08apr88,dnw  added 3-pin horizontal and vertical jumper blocks to jumpAt.
02b,04apr88,gae  more cleanups, made getBoard local and now called setBoard.
	   +dnw  fixed bug in printAt of wiping out following character.
		 changed prompt to be more lively, dynamic, and exciting.
02a,16mar88,gae  reworked to minimize trauma of adding new boards.
01f,01mar88, ak  added mv134
01e,22feb88,ecs  changed include of <strings.h> to "strLib.h".
01d,19feb88,jcf  added iv1624a.
                 added enp10.
01c,10dec87,jcf  added mv133a
01b,19nov87,jcf  added key
01a,07nov87,jcf  created
*/

/*
DESCRIPTION
This module contains the primitives required to show
board jumpering with the tool jump (4).
The following routines may be used in drawBoard,
defined in the target specific "pic_cpu.c".
.CS
    VOID printAt (pic, x, y, string)
    VOID jumperAt (pic, x, y, len, orient, type, contents)

.CE
Where
.CS
    len      - the number of jumpers in the block.
    contents - the configuration of the jumpers in the jumper block,
.CE
the order and meaning of the bits defined by orient and type as described below
.CS
    orient   - the orientation of the jumper block
	         HORIZONTAL  : LSB of contents is rightmost jumper.
	         VERTICAL    : LSB of contents is bottom jumper.
    type     - the type of the individual jumpers in the jumper block
	         HORIZONTAL  : 0 = removed, 1 = installed
	         VERTICAL    : 0 = removed, 1 = installed
	         HORIZONTAL_3: 0 = right, 1 = left
	         VERTICAL_3  : 0 = down, 1 = up

.CE
SEE ALSO: jump (4)
*/

#include <stdio.h>

#include "vxWorks.h"
#include "strLib.h"
#include "memLib.h"
#include "picture.h"

IMPORT VOID drawBoard ();
IMPORT VOID drawEx ();
IMPORT VOID drawEnp ();
IMPORT VOID drawNw ();
IMPORT VOID drawIe ();
IMPORT VOID drawMvme215 ();
IMPORT VOID drawDram512 ();

#define NUM_PIC 9
#define PIC_X 	77
#define PIC_Y	23


/*******************************************************************************
*
* main - show the configuration of a board
*
* If a -a flag is included on the command line then all the boards are sent
* to standard out with form-feeds between boards.
*
* NOMANUAL
*/

VOID main (argc,argv)
    int argc;
    char **argv;

    {
    int aFlag = FALSE;
    int choice;
    char command [90];

    if (setOptions (argc, argv, &aFlag) != OK)
	return;				/* return if bad option was given */

    if (aFlag) 
	{
	for (choice = 1; choice <= NUM_PIC; choice ++)
	    {
	    printChoice (choice, FALSE);
	    printf ("\f");		/* print a form-feed */
	    }
	exit (0);
	}

    printOptions ();

    FOREVER
	{
	printf ("jump> ");

	if (gets (command) == NULL ||
	    sscanf (command, "%d", &choice) > 0)
	    {
	    if (feof (stdin))
		{
		printf ("Bye\n");
		exit (0);
		}

	    printChoice (choice, TRUE);
	    }
	else 
	    {
	    switch (command [0])
		{
		case 'q':
		case 'Q':
		case 'e':
		case 'E':
		    printf ("Bye\n");
		    return;

		case '?':
		case 'H':
		case 'h':
		    printOptions ();
		    break;
		}
	    }
	}
    }
/*******************************************************************************
*
* setOptions - parse the argv, argc
*/

LOCAL STATUS setOptions (argc,argv, aFlag)
    int argc;
    char **argv;
    int *aFlag;

    {
    int i = 0;
    char **last_ptr = &argv [argc - 1];

    *aFlag = FALSE;

    while (++argv <= last_ptr && (*argv) [0] == '-')
	{
	for (i = 1; (*argv) [i] != '\0'; i++)
	    switch ((*argv) [i])
	      {
	      case 'a':
		*aFlag = TRUE;
		break;

	      default:  
		printf ("Bad Flag -%c\n", (*argv) [i]);
		printf ("Usage: jump [-a] <directory containing a config.h>\n");
		return (ERROR);
	      }
	}
    return (OK);
    }
/*******************************************************************************
*
* printOptions - print options in interactive mode
*/

LOCAL VOID printOptions ()
    
    {
    static char *print_key [] = {
    "Key: X vertical jumper installed",
    "     : vertical jumper absent",
    "     - horizontal jumper installed",
    "     \" horizontal jumper absent",
    "     0 switch off",
    "     1 switch on",
    "     U three pin vertical jumper, upper jumper installed",
    "     D three pin vertical jumper, lower jumper installed",
    "     L three pin horizontal jumper, left jumper installed",
    "     R three pin horizontal jumper, right jumper installed",
    "\nSelection?"
    };
    static char *print_options [] = {
    "",
    "%2d  : cpu",
    "%2d  : Excelan ethernet controller",
    "%2d  : CMC ethernet controller",
    "%2d  : ISI nw ethernet controller",
    "%2d  : SUN ie ethernet controller",
    "%2d  : MVME 215 shared mem",
    "%2d  : DRAM 512 shared mem",
    " q  : quit",
    " ?  : show me this list"
    };
    int ix;

    for (ix = 0; ix < NELEMENTS(print_key); ix++)
	printf ("%s\n", print_key[ix]);

    for (ix = 0; ix < NELEMENTS(print_options); ix++)
	{
	char buf [90];

	sprintf (buf, print_options[ix], ix);
	printf ("%s\n", buf);
	}
    }
/*******************************************************************************
*
* printChoice - print board choice
*/

LOCAL VOID printChoice (choice, interactive)
    int choice;
    BOOL interactive;

    {
    static char opic [PIC_X * PIC_Y];
    static char pic [PIC_X * PIC_Y];
    static FUNCPTR picRtn [] =
	{
	NULL, drawBoard, drawEx, drawEnp, drawNw,
	drawIe, drawMvme215, drawDram512,
	};

    if (choice > 0 && choice < NELEMENTS (picRtn) && picRtn [choice] != NULL)
	{
	setBoard (pic);
	setBoard (opic);
	(*(picRtn [choice])) (pic, interactive);

	if (bcmp (pic, opic, sizeof (pic)) != 0)
	    printPic (pic);
	}
    }
/*******************************************************************************
*
* setBoard - setup a character array with a generic board
*/

LOCAL VOID setBoard (pic)
    char *pic;

    {
    static char *vme[] = {	/* strings for VME board */
"  ______________________________              _______________________________",
"  |             P1             |              |             P2              |",
"  |                            ----------------                             |",
"  |                                                                         |",
"  |_________________________________________________________________________|"
    };

    int y;
    int ix;
    char *p = pic;

    for (ix = 0; ix < (PIC_X * PIC_Y); ix++)
	*p++ =' ';

    printAt (pic, 0, 0, vme[0]);
    printAt (pic, 0, 1, vme[1]);
    printAt (pic, 0, 2, vme[2]);
    for (y = 3; y < PIC_Y - 1; y++)
	printAt (pic, 0, y, vme[3]);

    printAt (pic, 0, PIC_Y - 1, vme[4]);
    }
/*******************************************************************************
*
* check - validate coordinates
*/

LOCAL STATUS check (x, y, len, orient)
    int x;
    int y;
    int len;
    int orient;

    {
    switch (orient)
	{
	case HORIZONTAL :
	    if ((x + len > PIC_X) || (x < 0))
		{
		printf ("x-coordinate for jumper off board\n");
		return (ERROR);
		}
	    if ((y > PIC_Y) || (y < 0))
		{
		printf ("y-coordinate for jumper off board\n");
		return (ERROR);
		}
	    return (OK);

	case VERTICAL :
	    if ((x > PIC_X) || (x < 0))
		{
		printf ("x-coordinate for jumper off board\n");
		return (ERROR);
		}
	    if ((y + len > PIC_Y) || (y < 0))
		{
		printf ("y-coordinate for jumper off board\n");
		return (ERROR);
		}
	    return (OK);

	default:
	    return (ERROR);
	}
    }
/*******************************************************************************
*
* printPic - print the picture
*/

LOCAL VOID printPic (pic)
    char *pic;		/* picture */

    {
    int y;

    putchar ('\n');
    for (y = 0; y < PIC_Y; y++, pic += PIC_X)
	{
	(void)fwrite (pic, 1, PIC_X, stdout);
	putchar ('\n');
	}
    }

/*******************************************************************************
*
* jumperAt - put jumper into picure
*
* This routine takes x, y cooridinate and the orientation or the block as
* a whole, and the orientation of the jumpers within the block.  The length
* of the block serves as a mask for the contents when filling the jumper
* block.
*
* NOMANUAL
*/

VOID jumperAt (pic, x, y, len, orient, type, contents)
    char *pic;		/* picture */
    int x;		/* coordinates */
    int y;
    int len;		/* jumper block length */
    int orient;		/* orientation of whole jumper block */
    int type;		/* orientation of jumpers within the block */
    int contents;	/* binary contents right to left */

    {
    if (check (x, y, len, orient) != OK)
	return;

    while (--len >= 0)
	{
	switch (orient)
	    {
	    case HORIZONTAL:
		drawJumper (pic, x + len, y, type, contents & 1);
		break;

	    case VERTICAL:
		drawJumper (pic, x, y + len, type, contents & 1);
		break;
	    }

	contents = contents >> 1;
	}
    }
/*******************************************************************************
*
* drawJumper - draw a single jumper at the specified coordinates
*/

LOCAL VOID drawJumper (pic, x, y, type, value)
    char *pic;		/* picture */
    int x;		/* coordinates */
    int y;
    int type;		/* orientation of jumper within the block */
    int value;		/* 1 or 0 */

    {
    char *start = pic + x + (y * PIC_X);

    switch (type)
	{
	case VERTICAL:
	    *start = value ? 'X' : ':';
	    break;

	case HORIZONTAL:
	    *start = value ? '-' : '"';
	    break;

	case VERTICAL_3:
	    *start = value ? 'U' : 'D';
	    break;

	case HORIZONTAL_3:
	    *start = value ? 'L' : 'R';
	    break;
	}
    }
/*******************************************************************************
*
* printAt - label at x, y coordinate
*
* The null is turned into a space.
*
* NOMANUAL
*/

VOID printAt (pic, x, y, string)
    char *pic;		/* picture */
    int x;
    int y;
    char *string;

    {
    strncpy (pic + x + (y * PIC_X), string, strlen (string));
    }
/*******************************************************************************
*
* bcmp - compare one buffer to another
*
* This routine compares the first nbytes characters of `buf1' to `buf2'.
*
* RETURNS
*   0 if first nbytes of `buf1' and `buf2' are identical
*  -1 if buf1 < buf2
*   1 if buf1 > buf2
*/

int bcmp (buf1, buf2, nbytes)
    FAST char *buf1;		/* pointer to first buffer    */
    FAST char *buf2;		/* pointer to second buffer   */
    FAST int nbytes;		/* number of bytes to compare */

    {
    while (--nbytes >= 0)
	{
	if (*buf1 < *buf2)
	    return (-1);
	if (*buf1++ > *buf2++)
	    return (1);
	}

    return (0);				/* equality */
    }
