/* hpToBsd.c - convert HP9000/300 a.out file to BSD format */

/*
modification history
--------------------
01b,12dec88,gae  changed usage to just be stdin and stdout.
01a,12nov87,hin  written
*/

/*
SYNOPSIS
hpToBsd <in.module >out.module

DESCRIPTION
This program convert the HP9000/300 a.out file to the BSD a.out file.
*/

#include <stdio.h>

#include "vxWorks.h"
#include "hpAout.h"
#include "a_out.h"

IMPORT char *malloc ();

#define	S_TEXT	0
#define	S_DATA	1
#define	S_BSS	2
#define	N_SEC	3

/*******************************************************************************
*
* main - read specified file and modify 
*/

VOID main (argc, argv)
    int argc;
    char *argv [];

    {
    struct execHP inExecHP;
    struct nlistHP *inSymHP;
    struct r_infoHP *inRelHP;
    struct exec outExec;
    struct nlist *outSym;
    struct relocation_info *outRel;
    int ix;
    int bytes;
    int symNum;
    int relNum;
    int secSize [N_SEC];
    int strSize;
    int symSize;
    char *buf;
    char *outSymtbl;
    char *inSymtbl;
    char *pOutSymtbl;
    char *pInSymtbl;
    char *strTbl;
    char *pStrTbl;
    char *outRelSec [N_SEC];
    char *inRelSec [N_SEC];
    char *pOutRelSec;
    char *pInRelSec;

    if (argc != 1)
	error ("usage: hpToBsd <in.module >out.module");

    /* read object module header */

    if (fread (&inExecHP, sizeof (inExecHP), 1, stdin) != 1)
	error ("hpToBsd: error reading file header.");


/* header */
    if (inExecHP.a_magic.file_type != 0x106 &&
	inExecHP.a_magic.file_type != 0x107)
	{
    	error ("hpToBsd: error wrong magic number.");
	}
    if ((buf = (char *)malloc (inExecHP.a_text + inExecHP.a_data)) == NULL)
    	error ("hpToBsd: error malloc text & data.");
    if (fread (buf, inExecHP.a_text, 1, stdin) != 1)
    	error ("hpToBsd: error reading text.");
    if (fseek (stdin, DATA_OFFSET(inExecHP), 0) != 0)
    	error ("hpToBsd: error seeking data_offset.");
    if (fread (buf + inExecHP.a_text, inExecHP.a_data, 1, stdin) != 1)
    	error ("hpToBsd: error reading data.");

/* string table */
    if (inExecHP.a_lesyms == 0) 
        {
    	symSize = 0;
    	goto text_rel;
    	}

    if ((inSymtbl = (char *)malloc (inExecHP.a_lesyms)) == NULL)
    	error ("hpToBsd: error malloc inSymtbl.");
    if ((strTbl = (char *)malloc (inExecHP.a_lesyms)) == NULL)
    	error ("hpToBsd: error malloc strTbl.");
    if (fseek (stdin, LESYM_OFFSET(inExecHP), 0) != 0)
    	error ("hpToBsd: error seeking lesym_offset.");
    if (fread (inSymtbl, inExecHP.a_lesyms, 1, stdin) != 1)
    	error ("hpToBsd: error reading symbol table.");

    symNum    = 0;
    pStrTbl   = strTbl + 4;
    pInSymtbl = inSymtbl;

    for (bytes = inExecHP.a_lesyms;
	 bytes > 0;
	 bytes -= sizeof (struct nlistHP) + inSymHP->n_length) 
	{
    	inSymHP = (struct nlistHP *)pInSymtbl;
    	pInSymtbl += sizeof (struct nlistHP);

    	for (ix = 0; ix < inSymHP->n_length; ix++)
    	    *pStrTbl++ = *pInSymtbl++;

    	*pStrTbl++ = '\0';
    	symNum++;
    	}

    strSize = pStrTbl - strTbl;
    *((int *)strTbl) = strSize;

/* symbol table */

    symSize = symNum * sizeof (struct nlist);
    if ((outSymtbl = (char *)malloc (symSize)) == NULL)
    	error ("hpToBsd: error malloc outSymtbl.");

    pInSymtbl  = inSymtbl;
    pOutSymtbl = outSymtbl;
    pStrTbl    = strTbl + 4;

    for (ix = 0; ix < symNum; ix++) 
	{
    	inSymHP = (struct nlistHP *)pInSymtbl;
    	outSym = (struct nlist *)pOutSymtbl;
    	outSym->n_type = (inSymHP->n_type & 0x7) * 2;

    	if (inSymHP->n_type & 0x20) 
    	    outSym->n_type |= 0x1;

    	if (ix == 0)
    	    outSym->n_un.n_strx = 4;
    	else
    	    outSym->n_un.n_strx = pStrTbl - strTbl;

    	outSym->n_other = 0;
    	outSym->n_desc  = 0;
    	outSym->n_value = inSymHP->n_value;

    	pOutSymtbl += sizeof (struct nlist);
    	pInSymtbl += sizeof (struct nlistHP) + inSymHP->n_length;

    	for (; *pStrTbl != '\0'; pStrTbl++);

    	pStrTbl++;
#ifdef	DEBUG
    	symPrint (outSym, strTbl);
#endif	DEBUG
    	}
    
/* text relocation */
text_rel:
    if (inExecHP.a_trsize == 0) 
	{
    	secSize [S_TEXT] = 0;
    	goto data_rel;
    	}

    relNum = inExecHP.a_trsize / sizeof (struct r_infoHP);
    secSize [S_TEXT] = relNum * sizeof (struct relocation_info);

    if ((outRelSec [S_TEXT] = (char *)malloc (secSize [S_TEXT])) == NULL)
    	error ("hpToBsd: error malloc outRelSec [S_TEXT].");
    if ((inRelSec [S_TEXT] = (char *)malloc (inExecHP.a_trsize)) == NULL)
    	error ("hpToBsd: error malloc inRelSec [S_TEXT].");
    if (fseek(stdin, RTEXT_OFFSET(inExecHP), 0) != 0)
    	error ("hpToBsd: error seeking rtext_offset.");
    if (fread (inRelSec [S_TEXT], inExecHP.a_trsize, 1, stdin) != 1)
    	error ("hpToBsd: error reading text relocation.");

    symNum = 0;
    pOutRelSec = outRelSec [S_TEXT];
    pInRelSec = inRelSec [S_TEXT];

    for (ix = 0; ix < relNum; ix ++) 
	{
    	inRelHP             = (struct r_infoHP *)pInRelSec;
    	outRel              = (struct relocation_info *)pOutRelSec;
    	outRel->r_address   = inRelHP->r_address;
    	outRel->r_symbolnum = inRelHP->r_symbolnum;
    	outRel->r_pcrel     = 0;
    	outRel->r_length    = inRelHP->r_length;

    	if (inRelHP->r_segment == 0x3)
    	    outRel->r_extern = 1;
    	else 
	    {
    	    outRel->r_extern = 0;
    	    outRel->r_symbolnum = (inRelHP->r_segment + 2) * 2;
    	    }
    	pOutRelSec += sizeof (struct relocation_info);
    	pInRelSec += sizeof (struct r_infoHP);
#ifdef	DEBUG
    	relPrint (outRel, outSymtbl, strTbl);
#endif	DEBUG
    	}

/* data relocation */
data_rel:
    if (inExecHP.a_drsize == 0) 
	{
    	secSize [S_DATA] = 0;
    	goto header;
    	}
    relNum = inExecHP.a_drsize / sizeof (struct r_infoHP);
    secSize [S_DATA] = relNum * sizeof (struct relocation_info);
    if ((outRelSec [S_DATA] = (char *)malloc (secSize [S_DATA])) == NULL)
    	error ("hpToBsd: error malloc outRelSec [S_DATA].");
    if ((inRelSec [S_DATA] = (char *)malloc (inExecHP.a_drsize)) == NULL)
    	error ("hpToBsd: error malloc inRelSec [S_DATA].");
    if (fseek(stdin, RDATA_OFFSET(inExecHP), 0) != 0)
    	error ("hpToBsd: error seeking rdata_offset.");
    if (fread (inRelSec [S_DATA], inExecHP.a_drsize, 1, stdin) != 1)
    	error ("hpToBsd: error reading data relocation.");

    pOutRelSec = outRelSec [S_DATA];
    pInRelSec = inRelSec [S_DATA];

    for (ix = 0; ix < relNum; ix ++) 
	{
    	inRelHP             = (struct r_infoHP *)pInRelSec;
    	outRel              = (struct relocation_info *)pOutRelSec;
    	outRel->r_address   = inRelHP->r_address;
    	outRel->r_symbolnum = inRelHP->r_symbolnum;
    	outRel->r_pcrel     = 0;
    	outRel->r_length    = inRelHP->r_length;

    	if(inRelHP->r_segment == 0x3)
    	    outRel->r_extern = 1;
    	else 
	    {
    	    outRel->r_extern = 0;
    	    outRel->r_symbolnum = (inRelHP->r_segment + 2) * 2;
    	    }
    	pOutRelSec += sizeof (struct relocation_info);
    	pInRelSec += sizeof (struct r_infoHP);
#ifdef	DEBUG
    	relPrint (outRel, outSymtbl, strTbl);
#endif	DEBUG
    	}

/* header */
header:
    outExec.a_magic  = 0x107;
    outExec.a_text   = inExecHP.a_text;
    outExec.a_data   = inExecHP.a_data;
    outExec.a_bss    = inExecHP.a_bss;
    outExec.a_syms   = symSize;
    outExec.a_entry  = inExecHP.a_entry;
    outExec.a_trsize = secSize [S_TEXT];
    outExec.a_drsize = secSize [S_DATA];

    if (fwrite (&outExec, sizeof (outExec), 1, stdout) != 1)
        error ("hpToBsd: error writing file header.");

    if (fwrite (buf, outExec.a_text + outExec.a_data, 1, stdout) != 1)
        error ("hpToBsd: error writing text & data header.");

    if (secSize [S_TEXT] != 0 &&
        fwrite (outRelSec [S_TEXT], secSize [S_TEXT], 1, stdout) != 1)
	{
	error ("hpToBsd: error writing text relocation.");
	}

    if (secSize [S_DATA] != 0 &&
        fwrite (outRelSec [S_DATA], secSize [S_DATA], 1, stdout) != 1)
	{
	error ("hpToBsd: error writing data relocation.");
        }

    if (symSize != 0) 
	{
    	if (fwrite (outSymtbl, symSize, 1, stdout) != 1)
    	    error ("hpToBsd: error writing symbol table.");
    	if (fwrite (strTbl, strSize, 1, stdout) != 1)
    	    error ("hpToBsd: error writing string table.");
        }

    exit (0);
    }
/*******************************************************************************
*
* error - print error message and die
*
* This routine prints an error message on the standard error output and
* aborts the program with the error status ERROR.  The error message is
* output with the specified format with the single specified argument.
*
* VARARGS1
*/

VOID error (format, arg)
    char *format;	/* format of error message */
    char *arg;		/* argument supplied to format (arbitrary type!) */

    {
    fprintf (stderr, format, arg);
    fprintf (stderr, "\n");
    exit (ERROR);
    }

#ifdef	DEBUG
/*******************************************************************************
*
* symPrint -
*/

VOID symPrint (sym, strTbl)
    struct nlist *sym;
    char *strTbl;

    {
    printf ("<%s>\t",strTbl+sym->n_un.n_strx);
    printf ("value=0x%x\t",sym->n_value);
    printf ("type=0x%x\t",sym->n_type);

    if (sym->n_type & N_STAB)
    	printf ("STAB");
    else if ((sym->n_type & N_FN) == N_FN)
    	printf ("FN");
    else if (sym->n_type & 0x1)
	{
	printf ("EXT ");

    	switch (sym->n_type & 0x1e) 
	    {
    	    case 0x0: printf ("UNDEF"); break;
    	    case 0x2: printf ("ABS"); break;
    	    case 0x4: printf ("TEXT"); break;
    	    case 0x6: printf ("DATA"); break;
    	    case 0x8: printf ("BSS"); break;
    	    case 0x12: printf ("COMM"); break;
    	    }
    	}
    printf ("\t");

    printf ("un.n_strx=%d\t", sym->n_un.n_strx);
    printf ("desc=0x%x\n", sym->n_desc);
    }
/*******************************************************************************
*
* relPrint -
*/

VOID relPrint (rel, symtbl, strTbl)
    struct relocation_info *rel;
    struct nlist *symtbl;
    char *strTbl;

    {
    if (rel->r_extern)
    	printf ("<%s>\n", strTbl+symtbl[rel->r_symbolnum].n_un.n_strx);
    else
    	printf ("<>\n");

    printf ("address=0x%x\t", rel->r_address);
    printf ("symbolnum=%d\t", rel->r_symbolnum);
    printf ("pcrel=%d\t", rel->r_pcrel);
    printf ("length=%d\t", rel->r_length);
    printf ("extern=%d\n", rel->r_extern);
    }
#endif	DEBUG
