/* telnetLib.c - telnet library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
02f,29sep88,gae  documentation.
02e,06jun88,dnw  changed taskSpawn/taskCreate args.
02d,30may88,dnw  changed to v4 names.
02c,28may88,dnw  changed to use shellOrigStdSet (...) instead of shellSetOrig...
		 changed not to use shellTaskId global variable.
02b,01apr88,gae  made it work with I/O system revision.
02a,27jan88,jcf  made kernel independent.
01g,14dec87,dnw  fixed bug in telnetdIn() that caused system crashes.
01f,19nov87,dnw  changed telnetd to wait for shell to exist before accepting
		   remote connections.
01e,17nov87,ecs  lint.
01d,04nov87,ecs  documentation.
	     &   fixed bug in use of setsockopt().
	    dnw  changed to call shellLogoutInstall() instead of logoutInstall.
01c,24oct87,gae  changed setOrig{In,Out,Err}Fd() to shellSetOrig{In,Out,Err}().
		 made telnetdOut() exit on EOF from master pty fd.
		 made telnetInit() not sweat being called more than once.
		 added shellLock() to telnetd to get exclusive use of shell.
01g,20oct87,gae  added logging device for telnet shell; made telnetInit()
		   create pty device.
01f,05oct87,gae  made telnetdExit() from telnetdIn() - used by logout().
		 removed gratuitous standard I/O ioctl's.
		 made "disconnection" cleaner by having shell do restart.
01e,26jul87,dnw  changed default priority of telnet tasks from 100 to 2.
		 changed task priority and ids to be global variables so
		   they can be accessed from outside this module.
01d,04apr87,dnw  de-linted.
01c,27mar87,dnw  documentation
		 fixed bug causing control sequences from remote to be
		   misinterpreted.
		 added flushing of pty in case anything was left before
		   remote login.
01b,27feb87,dnw  changed to spawn telnet tasks UNBREAKABLE.
01a,20oct86,dnw  written.
*/

/*
DESCRIPTION
The telnetLib library provides a remote login facility for VxWorks.
It uses the ARPA telnet protocol to allow
users at remote systems to login to VxWorks.

The telnet daemon (telnetd) allows remote users to log into VxWorks.
The telnet daemon is started by calling telnetInit (typically in usrRoot (2)).
It accepts remote telnet login requests,
and causes the shell's input and output to be redirected to the remote user.

Internally, the telnet daemon provides
a tty-like interface to the remote user through the use of the
VxWorks pseudo terminal driver (ptyDrv (3)).

SEE ALSO:
ptyDrv (3), rlogLib (1),
and the UNIX documentation for telnet, telnetd, and pty
*/

#include "vxWorks.h"
#include "types.h"
#include "socket.h"
#include "in.h"
#include "ioLib.h"
#include "taskLib.h"
#include "telnetLib.h"


/* Defining DEBUG will make this library hook remote telnet clients up to
 * the routine copyStreams() instead of to the shell.  This is useful for
 * debugging telnet without losing the shell. */

/* #define DEBUG */

#define TELNET_SERVICE		23	/* telnet port number */

#define STDIN_BUF_SIZE		512
#define STDOUT_BUF_SIZE		512

#define sendToPty(fd,buf,len)	write(fd,buf,len)
#define sendToRem(fd,buf,len)	write(fd,buf,len)

/* telnet input states */

#define TS_DATA		0
#define TS_IAC		1
#define TS_CR		2
#define TS_BEGINNEG	3
#define TS_ENDNEG	4
#define TS_WILL		5
#define TS_WONT		6
#define TS_DO		7
#define TS_DONT		8

/* global variables */

#ifndef DEBUG
int telnetTaskOptions   = VX_SUPERVISOR_MODE | VX_UNBREAKABLE;
#else
int telnetTaskOptions   = VX_SUPERVISOR_MODE;
#endif
int telnetTaskStackSize = 1500;
int telnetTaskPriority  = 2;	/* priority of telnet tasks */

int telnetdId;			/* task id of telnetd task */
int telnetInTaskId;		/* task id of telnetInTask task */
int telnetOutTaskId;		/* task id of telnetOutTask task */

/* local variables */

LOCAL char *telnetShellName = "shell";

LOCAL char *ptyTelnetName  = "/pty/telnet.";
LOCAL char *ptyTelnetNameM = "/pty/telnet.M";	/* master side */
LOCAL char *ptyTelnetNameS = "/pty/telnet.S";	/* slave side */

LOCAL BOOL activeFlag = FALSE;	/* TRUE if there is an active connection */
LOCAL int telnetdSocket;	/* telnetd socket */
LOCAL int telnetdM;		/* telnetd master pty */
LOCAL int telnetdS;		/* telnetd slave pty */

LOCAL int shellInFd;		/* original console input */
LOCAL int shellOutFd;		/* original console output */
LOCAL int shellErrFd;		/* original console error output */

LOCAL TBOOL myOpts [256];	/* current option settings - this side */
LOCAL TBOOL remOpts [256];	/* current option settings - other side */

LOCAL BOOL raw;			/* TRUE = raw mode enabled */
LOCAL BOOL echo;		/* TRUE = echo enabled */


/* forward declarations */

VOID telnetd ();
VOID telnetOutTask ();
VOID telnetInTask ();
LOCAL VOID telnetdExit ();

/*******************************************************************************
*
* telnetInit - initialize the telnet daemon
*
* This routine initializes the telnet facility, which supports remote login
* to the VxWorks shell via the ARPA telnet protocol.
* It creates a pty device, and spawns the telnet daemon.
* It is called normally from usrRoot(2),
* before any other system tries to login to this VxWorks system via telnet.
*/

VOID telnetInit ()

    {
    static BOOL done;	/* FALSE = not done */

    if (done)
	{
	printErr ("telnetInit: already initialized.\n");
	return;
	}

    if (ptyDrv () == ERROR || ptyDevCreate (ptyTelnetName, 1024, 1024) == ERROR)
	{
	printErr ("telnetInit: unable to create pty device.\n");
	return;
	}

    telnetdId = taskSpawn ("telnetd", telnetTaskPriority,
			   telnetTaskOptions, telnetTaskStackSize,
			   telnetd, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);  

    if (telnetdId == ERROR)
	printErr ("telnetInit: unable to spawn telnetd.\n");
    else
	done = TRUE;
    }
/*******************************************************************************
* 
* telnetd - the VxWorks telnet daemon
*
* telnetd provides a facility for remote users to log into VxWorks over the
* network via the ARPA telnet protocol.  It is spawned by telnetInit (2),
* which should be called by usrRoot(2) at boot time.
*
* Remote telnet requests will cause stdin, stdout, and stderr to be stolen
* away from the console.  When the remote user disconnects, stdin, stdout,
* and stderr are restored, and the shell is restarted.
* 
* telnetd requires the existence of a pseudo terminal device,
* which is created by telnetInit (2) before telnetd is spawned.
* telnetd creates two additional processes, telnetInTask and telnetOutTask
* whenever a remote user is logged in.  These processes exit when
* the remote connection is terminated.
*/

VOID telnetd ()

    {
    int optval;
    struct sockaddr_in myAddress;
    struct sockaddr_in clientAddress;
    int clientAddrLen;
    int client;
    int masterFd;
    int slaveFd;
    int sd;

    /* open a socket and wait for a client */

    sd = socket (AF_INET, SOCK_STREAM, 0);
   
    bzero ((char *) &myAddress, sizeof (myAddress));
    myAddress.sin_family = AF_INET;
    myAddress.sin_port   = TELNET_SERVICE;

    if (bind (sd, (struct sockaddr *) &myAddress, sizeof (myAddress)) == ERROR)
	{
	printErr ("telnetd: bind failed.\n");
	return;
	}

    listen (sd, 1);

    FOREVER
	{
	/* wait for shell to exist */

	while (taskNameToId (telnetShellName) == ERROR)
	    taskDelay (sysClkRateGet ());

	errnoSet (OK);		/* clear errno for pretty i() display */

	/* now accept connection */

	clientAddrLen = sizeof (clientAddress);
	client = accept (sd, (struct sockaddr *)&clientAddress, &clientAddrLen);

	if (client == ERROR)
	    {
	    printErr ("telnetd: accept failed - status = 0x%x\n",
		      errnoGet ());
	    continue;
	    }

	/* check to see if there's already an active connection */

	if (activeFlag)
	    {
	    char *msg = "\r\nSorry, this system is engaged.\r\n";

	    write (client, msg, strlen (msg));
	    close (client);
	    continue;
	    }

	/* create the pseudo terminal:
	 * the master side is connected to the socket to the
	 * remote machine - two processes telnetInTask & telnetOutTask
	 * handle input and output.
	 */

	if ((masterFd = open (ptyTelnetNameM, UPDATE)) == ERROR)
	    {
	    char *msg = "\r\nSorry, trouble wity pty.\r\n";

	    printErr ("telnetd: error opening %s\n", ptyTelnetNameM);
	    write (client, msg, strlen (msg));
	    close (client);
	    continue;
	    }

	if ((slaveFd = open (ptyTelnetNameS, UPDATE)) == ERROR)
	    {
	    char *msg = "\r\nSorry, trouble wity pty.\r\n";

	    printErr ("telnetd: error opening %s\n", ptyTelnetNameS);
	    write (client, msg, strlen (msg));
	    close (client);
	    close (masterFd);
	    continue;
	    }

	if (!shellLock (TRUE))
	    {
	    char *msg = "\r\nSorry, shell is locked.\r\n";

	    printErr ("telnetd: someone tried to login.\n");
	    write (client, msg, strlen (msg));
	    close (client);
	    close (masterFd);
	    close (slaveFd);
	    continue;
	    }

	/* setup the slave device to act like a terminal */

	ioctl (slaveFd, FIOOPTIONS, OPT_TERMINAL);

	printErr ("\ntelnetd: This system *IN USE* via telnet.\n");

	shellLogoutInstall (telnetdExit, TRUE);
	activeFlag    = TRUE;
	telnetdSocket = client;
	telnetdM      = masterFd;
	telnetdS      = slaveFd;

	/* turn on KEEPALIVE so if the client crashes, we'll know about it */

	optval = 1;
	setsockopt (client, SOL_SOCKET, SO_KEEPALIVE,
		    (char *) &optval, sizeof (optval));

	/* flush out pty device */

	ioctl (slaveFd, FIOFLUSH);

	/* get the shell's standard I/O fd's so we can restore them later */

	shellInFd  = ioGlobalStdGet (STD_IN);
	shellOutFd = ioGlobalStdGet (STD_OUT);
	shellErrFd = ioGlobalStdGet (STD_ERR);

#ifndef DEBUG
	/* set shell's standard I/O to pty device; add extra logging device */

	shellOrigStdSet (STD_IN, slaveFd); 
	shellOrigStdSet (STD_OUT, slaveFd);
	shellOrigStdSet (STD_ERR, slaveFd);

	logFdAdd (slaveFd);
#endif	DEBUG

	/* initialize modes and options and offer to do remote echo */

	raw  = FALSE;
	echo = TRUE;
	bzero ((char *) myOpts, sizeof (myOpts));
	bzero ((char *) remOpts, sizeof (remOpts));

	(void)localDoOpt (TELOPT_ECHO, TRUE, client, masterFd, FALSE);

	/* the shell is currently stuck in a read from the console, so we 
	 * restart it */

#ifndef DEBUG
	taskRestart (taskNameToId (telnetShellName));
#else
	sp (copyStreams, slaveFd, slaveFd);
#endif	DEBUG

	/* spawn the output process which transfers data from the master pty 
	 * to the socket;
	 * spawn the input process which transfers data from the client socket
	 * to the master pty */

	if (((telnetOutTaskId = taskSpawn ("telnetOutTask", telnetTaskPriority,
					   telnetTaskOptions,
					   telnetTaskStackSize,
					   telnetOutTask, client,masterFd,
					   0, 0, 0, 0, 0, 0, 0, 0)) == ERROR) ||
	    ((telnetInTaskId =  taskSpawn ("telnetInTask", telnetTaskPriority,
					   telnetTaskOptions,
					   telnetTaskStackSize,
					   telnetInTask, client,masterFd,
					   0, 0, 0, 0, 0, 0, 0, 0)) == ERROR))
	    {
	    printErr ("telnetd: error spawning %s child - status = 0x%x\n", 
		      (telnetOutTaskId != ERROR) ? "output" : "input",
		      errnoGet ());

	    if (telnetOutTaskId != ERROR)
		taskDelete (telnetOutTaskId);

	    telnetdExit (FALSE);	/* try to do a tidy clean-up */
	    }
	}
    }
/*******************************************************************************
* 
* telnetOutTask - stdout to socket process
*
* This routine gets spawned by the telnet daemon to move data between the 
* client socket and the pseudo terminal.  The created task is deleted
* when the client disconnects.
*
* NOMANUAL - but not LOCAL for i(2)
*/

VOID telnetOutTask (sock, ptyMfd)
    FAST int sock;	/* socket to copy output to */
    FAST int ptyMfd;	/* pty Master fd */

    {
    FAST int n;
    char buf [STDOUT_BUF_SIZE];

    while ((n = read (ptyMfd, buf, sizeof (buf))) > 0)
	{
	/* XXX should scan for IAC and double 'em to escape 'em */
	write (sock, buf, n);
	}
    }
/*******************************************************************************
* 
* telnetInTask - socket to stdin process 
*
* This routine gets spawned by the telnet daemon to move
* data between the pseudo terminal and the client socket.
* The task exits, calling telnetdExit(), when the client disconnects.
*
* NOMANUAL - but not LOCAL for i(2)
*/

VOID telnetInTask (sock, ptyMfd)
    FAST int sock;	/* socket to copy input from */
    int ptyMfd;		/* pty Master fd */

    {
    int n;
    int state = TS_DATA;
    char buf [STDIN_BUF_SIZE];

    while ((n = read (sock, buf, sizeof (buf))) > 0)
	state = tnInput (state, sock, ptyMfd, buf, n);

    /* exit and cleanup */

    telnetdExit (FALSE);
    }
/*******************************************************************************
*
* telnetdExit - exit and cleanup routine for telnetd
*
* This is the support routine for logout(2).
* The client socket is closed, shell standard I/O is redirected
* back to the console, and the shell is restarted by sending an EOF.
*/

LOCAL VOID telnetdExit (usedLogout)
    BOOL usedLogout;

    {
    activeFlag = FALSE;
    shellLogoutInstall ((FUNCPTR) NULL, 0);

#ifndef DEBUG
    /* cancel extra logging device; restore shell's standard I/O & unlock */

    logFdDelete (telnetdS);

    shellOrigStdSet (STD_IN,  shellInFd); 
    shellOrigStdSet (STD_OUT, shellOutFd);
    shellOrigStdSet (STD_ERR, shellErrFd);

    shellLock (FALSE);
#endif	DEBUG

    write (telnetdSocket, "\n", 1);
    close (telnetdSocket);

    /* send EOF to shell to cause restart */

    if (!usedLogout)
	write (telnetdM, "\04", 1);	/* control-D */

    close (telnetdM);
    close (telnetdS);
    }
/*******************************************************************************
*
* tnInput - process input from remote
*/

LOCAL VOID tnInput (state, remFd, ptyFd, buf, n)
    FAST int state;	/* state of input stream */
    FAST int remFd;	/* fd of socket to otherside */
    FAST int ptyFd;	/* fd of pty to this side */
    FAST char *buf;	/* ptr to input chars */
    FAST int n;		/* number of chars input */

    {
    char cc;
    int ci;

    while (--n >= 0)
	{
	cc = *buf++;			/* get next character */
	ci = (unsigned char) cc;	/* convert to int since many values
					 * are negative characters */
	switch (state)
	    {
	    case TS_CR:		/* doing crmod; ignore add'l linefeed */
		state = TS_DATA;
		if ((cc != EOS) && (cc != '\n'))
		    sendToPty (ptyFd, &cc, 1);	/* forward char */
		break;

	    case TS_DATA:	/* just pass data */
		if (ci == IAC)
		    state = TS_IAC;
		else
		    {
		    sendToPty (ptyFd, &cc, 1);	/* forward char */
		    if (!myOpts [TELOPT_BINARY] && (cc == '\r'))
			state = TS_CR;
		    }
		break;

	    case TS_IAC:

		switch (ci)
		    {
		    case BREAK:		/* interrupt from remote */
		    case IP:
			/* XXX interrupt (); */
			logMsg ("telnetInTask: interrupt\n");
			state = TS_DATA;
			break;

		    case AYT:		/* Are You There? */
			{
			static char aytAnswer [] = "\r\n[yes]\r\n";

			sendToRem (remFd, aytAnswer, sizeof (aytAnswer) - 1);
			state = TS_DATA;
			break;
			}

		    case EC:		/* erase character */
			sendToPty (ptyFd, "\b", 1);
			state = TS_DATA;
			break;

		    case EL:		/* erase line */
			sendToPty (ptyFd, "\025", 1);
			state = TS_DATA;
			break;

		    case DM:		/* data mark */
			state = TS_DATA;
			break;

		    case SB:		/* sub-option negotiation begin */
			state = TS_BEGINNEG;
			break;

		    case WILL: state = TS_WILL;	break;	/* remote will do opt */
		    case WONT: state = TS_WONT;	break;	/* remote wont do opt */
		    case DO:   state = TS_DO;	break;	/* req we do opt */
		    case DONT: state = TS_DONT;	break;	/* req we dont do opt */

		    case IAC:
			sendToPty (ptyFd, &cc, 1);	/* forward char */
			state = TS_DATA;
			break;
		    }
		break;

	    case TS_BEGINNEG:
		/* ignore sub-option stuff for now */
		if (ci == IAC)
		    state = TS_ENDNEG;
		break;

	    case TS_ENDNEG:
		state = (ci == SE) ? TS_DATA : TS_BEGINNEG;
		break;

	    case TS_WILL:		/* remote side said it will do opt */
		(void)remDoOpt (ci, TRUE, remFd, ptyFd, TRUE);
		state = TS_DATA;
		break;

	    case TS_WONT:		/* remote side said it wont do opt */
		(void)remDoOpt (ci, FALSE, remFd, ptyFd, TRUE);
		state = TS_DATA;
		break;

	    case TS_DO:			/* remote wants us to do opt */
		(void)localDoOpt (ci, TRUE, remFd, ptyFd, TRUE);
		state = TS_DATA;
		break;

	    case TS_DONT:		/* remote wants us to not do opt */
		(void)localDoOpt (ci, FALSE, remFd, ptyFd, TRUE);
		state = TS_DATA;
		break;
	    
	    default:
		printErr ("telnetd: invalid state = %d\n", state);
		break;
	    }
	}

    return (state);
    }
/*******************************************************************************
*
* remDoOpt - request/acknowledge remote enable/disable of option
*
* This routine will try to accept the remote's enable or disable,
* as specified by "will", of the remote's support for the specified option.
* If the request is to disable the option, the option will always be disabled.
* If the request is to enable the option, the option will be enabled IF
* we are capable of supporting it.
* The remote is notified to DO/DONT support the option.
*
* ARGSUSED
*/

LOCAL STATUS remDoOpt (option, enable, remFd, ptyFd, reqFromRem)
    FAST int option;	/* option to be enabled/disabled */
    BOOL enable;	/* TRUE = enable option, FALSE = disable */
    int remFd;		/* fd to remote */
    int ptyFd;		/* fd to local pseudo-terminal */
    BOOL reqFromRem;	/* TRUE = request is from remote */

    {
    BOOL doOpt = enable;

    if (remOpts [option] == enable)
	return (OK);

    switch (option)
	{
	case TELOPT_BINARY:
	case TELOPT_ECHO:
	    setMode (option, enable);
	    break;

	case TELOPT_SGA:
	    break;

	default:
	    doOpt = FALSE;
	    break;
	}

    if ((remOpts [option] != doOpt) || reqFromRem)
	{
	char msg[3];

	msg[0] = IAC;
	msg[1] = doOpt ? DO : DONT;
	msg[2] = option;

	sendToRem (remFd, msg, 3);

	remOpts [option] = doOpt;
	}

    return ((enable == doOpt) ? OK : ERROR);
    }
/*******************************************************************************
*
* localDoOpt - offer/acknowledge local support of option
*
* This routine will try to enable or disable local support for the 
* specified option.
* If local support of the option is already in the desired mode,
* no action is taken.
* If the request is to disable the option, the option will always be disabled.
* If the request is to enable the option, the option will be enabled IF
* we are capable of supporting it.
* The remote is notified that we WILL/WONT support the option.
*
* ARGSUSED
*/

LOCAL STATUS localDoOpt (option, enable, remFd, ptyFd, reqFromRem)
    FAST int option;	/* option to be enabled/disabled */
    BOOL enable;	/* TRUE = enable option, FALSE = disable */
    int remFd;		/* fd to remote */
    int ptyFd;		/* fd to local pseudo-terminal */
    BOOL reqFromRem;	/* TRUE = request is from remote */

    {
    BOOL will = enable;

    if (myOpts [option] == enable)
	return (OK);

    switch (option)
	{
	case TELOPT_BINARY:
	case TELOPT_ECHO:
	    setMode (option, enable);
	    break;

	case TELOPT_SGA:
	    break;

	default:
	    will = FALSE;
	    break;
	}

    if ((myOpts [option] != will) || reqFromRem)
	{
	char msg[3];

	msg[0] = IAC;
	msg[1] = will ? WILL : WONT;
	msg[2] = option;

	sendToRem (remFd, msg, 3);

	myOpts [option] = will;
	}

    return ((will == enable) ? OK : ERROR);
    }
/*******************************************************************************
*
* setMode -
*/

LOCAL VOID setMode (telnetOption, enable)
    int telnetOption;
    BOOL enable;

    {
    FAST int ioOptions;

    switch (telnetOption)
	{
	case TELOPT_BINARY: raw  = enable; break;
	case TELOPT_ECHO:   echo = enable; break;
	}

    if (raw)
	ioOptions = 0;
    else
	{
	ioOptions = OPT_7_BIT | OPT_ABORT | OPT_TANDEM | OPT_LINE;
	if (echo)
	    ioOptions |= OPT_ECHO | OPT_CRMOD;
	}

    ioctl (STD_IN, FIOOPTIONS, ioOptions);
    }
