/* taskHookLib.c - task hook library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01g,08apr89,dnw  changed to store delete hooks in reverse order in the table.
01f,17aug88,gae  documentation.
01e,22jun88,dnw  name tweaks.
01d,05jun88,dnw  changed from hookLib to taskHookLib.
01c,30may88,dnw  changed to v4 names.
01b,28may88,dnw  removed hookAddRebootRtn to rebootLib.c
		 changed some status value names.
01a,25jan88,jcf  written by extracting from vxLib.c.
*/

/*
DESCRIPTION
The taskHookLib library allows for easy extensions to the VxWorks
tasking facility.
Extensions may be placed in three distinct stages of a task's existence:
during task creation, at a task switch, and when a task is deleted.
The routines taskCreateHookAdd and taskCreateHookDelete allow routines
to be dynamically added and deleted from a list of routines which are
called whenever a task is created.
Likewise, taskSwitchHookAdd and taskSwitchHookDelete manipulate
the list of routines to be called at each task switch.
Finally, taskDeleteHookAdd and taskDeleteHookDelete manipulate
the list of routines to be called whenever a task is deleted.

This facility is used by the debugger to provide task specific
breakpoints and single stepping.
It is used by taskVarLib (1) for the "task variable" mechanism.
It is also used by fppLib (1) for floating point coprocessor support.

NOTE
It is possible to have dependencies among task hooks.  For example, a
task delete hook may use facilities that are cleaned up and deleted by
another delete hook.  In these cases, the order that the hooks run is
important.  VxWorks runs the create and switch hooks in the order that
they were added, and runs the delete hooks in the REVERSE of the order
in which they were added.  Thus if the hooks are added in "hierarchical"
order, such that hooks only rely on facilities whose hooks have already
been added, then the required facilities will be initialized before any
other facilities require them, and will be delete after all facilities
are finished with them.

In VxWorks' facilities, this is guaranteed by having each facility have
an initialization routine that first calls any pre-requisite facility's
initialization routine, and then adds its own hooks.  Thus the task hooks
will get added in the correct order.  Each initialization routine
protects itself from multiple invocations, allowing only the first
invocation to have any affect.

SEE ALSO: "Architecture", taskLib (1)
*/

#include "vxWorks.h"
#include "taskLib.h"

/* globals */

FUNCPTR taskCreateTable [VX_MAX_TASK_CREATE_RTNS + 1];
FUNCPTR taskSwitchTable [VX_MAX_TASK_SWITCH_RTNS + 1];
FUNCPTR taskDeleteTable [VX_MAX_TASK_DELETE_RTNS + 1];


/*******************************************************************************
*
* taskCreateHookAdd - add routine to be called at every task create
*
* This routine adds the specified routine to a list of routines
* that get called whenever a task is created.  The routine should be
* declared as follows:
*
* .CS
*	VOID createHook (pNewTcbX)
*	    TCBX *pNewTcbX;		/* pointer to new task's TCB *
* .CE
* RETURNS: OK, or ERROR if table of task switch routines is full.
*
* SEE ALSO: taskCreateHookDelete(2)
*/

STATUS taskCreateHookAdd (createHook)
    FUNCPTR createHook;	/* routine to be called when a task is created */

    {
    return (taskHookAdd (createHook, taskCreateTable, VX_MAX_TASK_CREATE_RTNS));
    }
/*******************************************************************************
*
* taskCreateHookDelete - delete previously added task create routine
*
* This routine removes the specified routine from the list of
* routines to be called at each task create.
*
* RETURNS: OK, or ERROR if routine not in table.
*
* SEE ALSO: taskCreateHookAdd(2)
*/

STATUS taskCreateHookDelete (createHook)
    FUNCPTR createHook;		/* routine to be deleted from list */

    {
    return (taskHookDelete (createHook, taskCreateTable,
			    VX_MAX_TASK_CREATE_RTNS));
    }
/*******************************************************************************
*
* taskSwitchHookAdd - add routine to be called at every task switch
*
* This routine adds the specified routine to a list of routines
* that get called at every task switch.  The routine should be
* declared as follows:
* .CS
*
*	VOID switchHook (pOldTcbX, pNewTcbX)
*	    TCBX *pOldTcbX;	/* pointer to old task's TCBX *
*	    TCBX *pNewTcbX;	/* pointer to new task's TCBX *
*
* .CE
* RETURNS: OK, or ERROR if table of task switch routines is full.
*
* SEE ALSO: taskSwitchHookDelete(2)
*/

STATUS taskSwitchHookAdd (switchHook)
    FUNCPTR switchHook;	/* routine to be called at every task switch */

    {
    return (taskHookAdd (switchHook, taskSwitchTable, VX_MAX_TASK_SWITCH_RTNS));
    }
/*******************************************************************************
*
* taskSwitchHookDelete - delete previously added task switch routine
*
* This routine removes the specified routine from the list of
* routines to be called at each task switch.
*
* RETURNS: OK, or ERROR if routine not in table.
*
* SEE ALSO: taskSwitchHookAdd(2)
*/

STATUS taskSwitchHookDelete (switchHook)
    FUNCPTR switchHook;		/* routine to be deleted from list */

    {
    return (taskHookDelete (switchHook, taskSwitchTable,
			    VX_MAX_TASK_SWITCH_RTNS));
    }
/****************************************************************
*
* taskDeleteHookAdd - add routine to be called at every task delete
*
* This routine adds the specified routine to a list of routines
* that get called whenever a task is deleted.  The routine should be
* declared as follows:
*
* .CS
*	VOID deleteHook (pTcbX)
*	    TCBX *pTcbX;	/* pointer to deleted task's TCBX *
* .CE
* RETURNS: OK, or ERROR if table of task delete routines is full.
*
* SEE ALSO: taskDeleteHookDelete(2)
*
* INTERNAL:
* Unlike the other "hook add" routines, this routine keeps the delete hooks in
* the table in REVERSE of the order in which they are added.  Thus the most
* recently added hook is the FIRST entry in the table.  This causes the delete
* hooks to be run in the reverse order when a task is deleted.  This is
* necessary since a task delete hook may depend upon another facility that
* has a delete hook, e.g. stdio uses task variables.  If we ensure that the
* delete hooks are added in hierarchical order (i.e. any delete hook only
* uses facilities whose delete hooks have already been added), then running
* them in reverse order guarantees that facilities are not cleaned-up
* pre-maturely.
*/

STATUS taskDeleteHookAdd (deleteHook)
    FUNCPTR deleteHook;	/* routine to be called when a task is deleted */

    {
    FAST int ix;
    STATUS status = OK;

    taskLock ();			/* disable task switching */

    if (taskDeleteTable [VX_MAX_TASK_DELETE_RTNS] != NULL)
	{
	/* no free slot found */

	errnoSet (S_taskLib_TASK_HOOK_TABLE_FULL);
	status = ERROR;
	}
    else
	{
	/* move all the hooks down one slot in the table */

	for (ix = VX_MAX_TASK_DELETE_RTNS - 2; ix >= 0; --ix)
	    taskDeleteTable [ix + 1] = taskDeleteTable [ix];

	taskDeleteTable [0] = deleteHook;
	}

    taskUnlock ();			/* re-enable task switching */

    return (status);
    }
/*******************************************************************************
*
* taskDeleteHookDelete - delete previously added task delete routine
*
* This routine removes the specified routine from the list of
* routines to be called at each task delete.
*
* RETURNS: OK, or ERROR if routine not in table.
*
* SEE ALSO: taskDeleteHookAdd(2)
*/

STATUS taskDeleteHookDelete (deleteHook)
    FUNCPTR deleteHook;		/* routine to be deleted from list */

    {
    return (taskHookDelete (deleteHook, taskDeleteTable,
			    VX_MAX_TASK_DELETE_RTNS));
    }

/*******************************************************************************
*
* taskHookAdd - add hook to hook table
*/

LOCAL STATUS taskHookAdd (hook, table, maxEntries)
    FUNCPTR hook;	/* routine to be added to table */
    FUNCPTR table[];	/* table to which to add */
    int maxEntries;	/* max entries in table */

    {
    FAST int ix;

    taskLock ();			/* disable task switching */

    /* find slot after last hook in table */

    for (ix = 0; ix < maxEntries; ++ix)
	{
	if (table[ix] == NULL)
	    {
	    table[ix] = hook;
	    taskUnlock ();	/* re-enable task switching */
	    return (OK);
	    }
	}
    
    /* no free slot found */

    taskUnlock ();		/* re-enable task switching */

    errnoSet (S_taskLib_TASK_HOOK_TABLE_FULL);
    return (ERROR);
    }
/*******************************************************************************
*
* taskHookDelete - delete hook from hook table
*/

LOCAL STATUS taskHookDelete (hook, table, maxEntries)
    FUNCPTR hook;	/* routine to be deleted from table */
    FUNCPTR table[];	/* table from which to delete */
    int maxEntries;	/* max entries in table */

    {
    FAST int ix;

    taskLock ();			/* disable task switching */

    /* find hook in hook table */

    for (ix = 0; ix < maxEntries; ++ix)
	{
	if (table [ix] == hook)
	    {
	    /* move all the remaining hooks up one slot in the table */

	    do
		table [ix] = table [ix + 1];
	    while (table [++ix] != NULL);

	    taskUnlock ();	/* re-enable task switching */
	    return (OK);
	    }
	}

    /* hook not found in table */

    taskUnlock ();		/* re-enable task switching */

    errnoSet (S_taskLib_TASK_HOOK_NOT_FOUND);
    return (ERROR);
    }
