/* spyLib.c - spy CPU activity library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
02i,20aug88,gae  documentation.
02h,22jun88,dnw  name tweaks.
		 clean-up.
02g,07jun88,gae  added call to sysClkEnable ().
02f,06jun88,dnw  changed taskSpawn/taskCreate args.
02e,04jun88,gae  changed spyStart() to spyClkStart().
		 added spyClkStop().
02c,30may88,dnw  changed to v4 names.
02b,24mar88,gae  fixed spyReport() formatting, % calculations, prints
		   out tasks that have no ticks & newly created tasks.
		 fixed bugs allowing multiple spy's & extra delete hooks.
02a,07mar88,gae  made it work with generic kernel -- does not use
		   static array for tasks.
		 Renamed module and all routines to "spy".
		 Added spy{Create,Delete}Task() to correctly handle
		   variable number of tasks between reports.
01i,04nov87,ecs  documentation.
01h,02apr87,ecs  eased lint's mind in taReport.
01g,23mar87,jlf  documentation.
01f,20feb87,jlf  made taStart check the parameter returned by
		   sysAuxClkConnect, to determine whether the board has
		   an aux clock on it.
01e,21dec86,dnw  changed to not get include files from default directories.
01d,19nov86,llk  Added spyTask.
01c,06nov86,jlf  Changed to use new CPU-independent routines, sysAuxClk...
01b,24oct85,dnw  changed taSpy to explicity spawn periodRun since default
		   parameters supplied by period, weren't adequate.
01a,22oct85,jlf  written.
*/
  
/*
DESCRIPTION
This module provides a facility to monitor the CPU usage of all tasks.
The primary interface routine, spy (2), periodically prints a report
on CPU usage with spyReport (2).  The report shows CPU usage taken up by
interrupt level, idle state, and running tasks.
The total usage since the beginning of spy (or the most recent spyClkStart (2)),
and the change in usage (delta) since the last spyReport was displayed.

The facility can also be used "manually", instead of using spy.
In that case, spyClkStart is used to begin monitoring, and spyReport provides
a one-time report of the same information as provided by spy.

This data is gathered by an interrupt level routine that is connected by
spyClkStart to the auxiliary clock.  There is currently no way to use this
facility with CPU's that don't have an auxiliary clock.
Interrupts that are at a higher level than the auxiliary clock's
cannot be monitored.
*/

#include "vxWorks.h"
#include "taskLib.h"
#include "sysSymTbl.h"


#define MAX_SPY_TASKS	200		/* max tasks that can be spy'd */

/* spyTask parameters */

int spyTaskId		= ERROR;	/* ERROR = spy task not active */
int spyTaskOptions	= VX_UNBREAKABLE;
int spyTaskPriority	= 5;
int spyTaskStackSize	= 4000;

/* local variables */

LOCAL UINT spyTotalTicks;		/* all ticks since start */
LOCAL UINT spyIncTicks;			/* all ticks since last report */

LOCAL UINT spyInterruptTicks;		/* int ticks since start */
LOCAL UINT spyInterruptIncTicks;	/* int ticks since last report */

LOCAL UINT spyIdleTicks;		/* idle ticks since start */
LOCAL UINT spyIdleIncTicks;		/* idle ticks since last report*/

LOCAL BOOL spyClkRunning;		/* TRUE = spyClkStart'ed */
LOCAL BOOL spyInitialized;		/* TRUE = hooks installed */
LOCAL int spyCreateCount;		/* num tasks created since report */
LOCAL int spyDeleteCount;		/* num tasks deleted since report */

/* display formats */

LOCAL char *spyFmt1 = "%-12.12s  %8x   %3d%% (%7d)  %3d%% (%7d)\n";
LOCAL char *spyFmt2 = "%-12.12s  %8s   %3d%% (%7d)  %3d%% (%7d)\n";

/*******************************************************************************
*
* spyCreateHook - initialize task tick counters
*
* This routine is installed as a task create hook so that the task
* tick counters can be initialized and spyReport (2)
* can indicate when new tasks appear between reports.
*/

LOCAL VOID spyCreateHook (pTcbX)
    TCBX *pTcbX;	/* TCB extension of new task */

    {
    pTcbX->taskTicks    = 0;
    pTcbX->taskIncTicks = 0;
    spyCreateCount++;
    }
/*******************************************************************************
*
* spyDeleteHook - notify spyLib of task deletion
*
* This routine is installed as a task delete hook so that spyReport (2)
* can indicate when tasks disappear between reports.
*
* ARGSUSED
*/

LOCAL VOID spyDeleteHook (pTcbX)
    TCBX *pTcbX;	/* TCB extension of new task */

    {
    spyDeleteCount++;
    }
/*******************************************************************************
*
* spyClkInt - spyLib interrupt service routine
*
* This routine is called at each tick of the auxiliary clock.
* When at interrupt level the interrupt tick count is incremented.
* If there is no active task then the idle tick count is incremented,
* otherwise increment the active task's tick counter.
*/

LOCAL VOID spyClkInt ()

    {
    FAST TCBX *pTcbX;

    if (intCount () > 1)
	spyInterruptIncTicks++;		/* we interrupted an interrupt */
    else
	{
	pTcbX = taskTcbX (0);		/* get our tcb extension */

	if (taskIsReady (pTcbX->taskId))
	    pTcbX->taskIncTicks++;
	else
	    spyIdleIncTicks++;
	}

    spyIncTicks++;
    }
/*******************************************************************************
*
* spyClkStart - start collecting task activity data
*
* Begin data collection by enabling the auxilary clock interrupts.
* Data from previous collections is cleared.
*
* RETURNS:
*  OK, or
*  ERROR if CPU has no auxiliary clock, or
*        unable to install task create/delete hooks.
*
* SEE ALSO: sysAuxClkConnect (2)
*/

STATUS spyClkStart (intsPerSec)
    int intsPerSec;	/* timer interrupt frequency, */
			/* 0 = use default of 100     */

    {
    FAST int ix;
    FAST int nTasks;
    FAST TCBX *pTcbX;
    int idList [MAX_SPY_TASKS];

    if (spyClkRunning)
	return (ERROR);		/* We're already going */

    if (!spyInitialized &&
	(taskCreateHookAdd (spyCreateHook) == ERROR ||
	 taskDeleteHookAdd (spyDeleteHook) == ERROR))
	{
	printf ("Unable to add create/delete hooks.\n");
	return (ERROR);
	}

    spyInitialized = TRUE;

    if (intsPerSec == 0)
	intsPerSec = 100;

    spyDeleteCount  = 0;
    spyCreateCount  = 0;
    spyTotalTicks = spyIdleTicks    = spyInterruptTicks    = 0;
    spyIncTicks   = spyIdleIncTicks = spyInterruptIncTicks = 0;


    /* initialize tick counters of tasks already running */

    nTasks = taskIdListGet (idList, NELEMENTS (idList));

    for (ix = 0; ix < nTasks; ++ix)
	{
	pTcbX = taskTcbX (idList [ix]);
	pTcbX->taskIncTicks = pTcbX->taskTicks = 0;
	}

    if (sysAuxClkConnect (spyClkInt, 0) != OK)
	{
	printf ("No auxiliary clock on CPU.\n");
	return (ERROR);
	}

    sysAuxClkRateSet (intsPerSec);
    sysAuxClkEnable ();

    spyClkRunning = TRUE;

    return (OK);
    }
/*******************************************************************************
*
* spyClkStop - stop collecting task activity data
*
* This routine disables the auxiliary clock interrupts.
* Data collected remains valid until the next spyClkStart (2).
*/

VOID spyClkStop ()

    {
    sysAuxClkDisconnect ();
    spyClkRunning  = FALSE;
    }
/*******************************************************************************
*
* spyReport - display task activity data
*
* This routine reports on data gathered at interrupt level.  It shows
* the number of ticks used by each task, and at interrupt level, since
* spyClkStart (2), and also since the last spyReport was made.  It
* also shows CPU utilization percentage, and percent idle.  Nothing
* is printed if no interrupts have happened since the previous spyReport.
*/

VOID spyReport ()

    {
    FAST TCBX *pTcbX;
    FAST int ix;
    FUNCPTR symbolAddress;
    char name [MAX_SYS_SYM_LEN + 1];
    TINY type;

    int idList [MAX_SPY_TASKS];		/* task specific statistics */
    int taskIncTicks [MAX_SPY_TASKS];
    int taskTotalTicks [MAX_SPY_TASKS];
    FAST int nTasks;

    int tmpIncTicks;			/* incremental snap shot */
    int tmpIdleIncTicks;
    int tmpInterruptIncTicks;

    int totalPerCent;
    int incPerCent;
    int sumTotalPerCent = 0;
    int sumIncPerCent   = 0;

    /* if there have been no ticks, there is nothing to report */

    if (spyIncTicks == 0)
	return;
	
    /* snap shot and clear task statistics */

    nTasks = taskIdListGet (idList, NELEMENTS (idList));

    for (ix = 0; ix < nTasks; ++ix)
	{
	pTcbX = taskTcbX (idList [ix]);

	/* order is important: save and clear incremental, then update total */

	taskIncTicks [ix]    = pTcbX->taskIncTicks;
	pTcbX->taskIncTicks  = 0;

	pTcbX->taskTicks    += taskIncTicks [ix];
	taskTotalTicks [ix]  = pTcbX->taskTicks;
	}


    /* save and clear incremental counts and accumulate totals */

    tmpIncTicks          = spyIncTicks;
    tmpIdleIncTicks      = spyIdleIncTicks;
    tmpInterruptIncTicks = spyInterruptIncTicks;

    spyIncTicks = spyIdleIncTicks = spyInterruptIncTicks = 0;

    spyTotalTicks       += tmpIncTicks;
    spyInterruptTicks   += tmpInterruptIncTicks;
    spyIdleTicks        += tmpIdleIncTicks;


    /* print info */

    printf ("\n");
    printf (
    "  ENTRY         TID     total %% (ticks)  delta %% (ticks)\n");
    printf (
    "------------  --------  ---------------  ---------------\n");

    for (ix = 0; ix < nTasks; ++ix)
	{
	/* find name in symbol table */

	pTcbX = taskTcbX (idList [ix]);

	symFindByValue (sysSymTbl, (int) pTcbX->entry, name, 
		        (int *) &symbolAddress, (UTINY *) &type);

	if (symbolAddress != pTcbX->entry)
	    name [0] = EOS;	/* no matching symbol */


	/* print line for this task */

	totalPerCent     = (taskTotalTicks [ix] * 100) / spyTotalTicks;
	incPerCent       = (taskIncTicks [ix] * 100) / tmpIncTicks;
	sumTotalPerCent += totalPerCent;
	sumIncPerCent   += incPerCent;

	printf (spyFmt1, name, idList [ix], totalPerCent, taskTotalTicks [ix],
					    incPerCent, taskIncTicks [ix]);
	}

    totalPerCent     = (spyInterruptTicks * 100) / spyTotalTicks;
    incPerCent       = (tmpInterruptIncTicks * 100) / tmpIncTicks;
    sumTotalPerCent += totalPerCent;
    sumIncPerCent   += incPerCent;

    printf (spyFmt2, "INTERRUPT", "", totalPerCent, spyInterruptTicks,
				      incPerCent, tmpInterruptIncTicks);

    totalPerCent     = (spyIdleTicks * 100) / spyTotalTicks;
    incPerCent       = (tmpIdleIncTicks * 100) / tmpIncTicks;
    sumTotalPerCent += totalPerCent;
    sumIncPerCent   += incPerCent;

    printf (spyFmt2, "IDLE", "", totalPerCent, spyIdleTicks,
				 incPerCent, tmpIdleIncTicks);

    printf (spyFmt2, "TOTAL", "", sumTotalPerCent, spyTotalTicks,
				  sumIncPerCent, tmpIncTicks);
    
    printf ("\n");
    
    if (spyCreateCount > 0)
	{
	printf ("%d task%s created.\n", spyCreateCount,
		spyCreateCount == 1 ? " was" : "s were");
	spyCreateCount = 0;
	}

    if (spyDeleteCount > 0)
	{
	printf ("%d task%s deleted.\n", spyDeleteCount,
		spyDeleteCount == 1 ? " was" : "s were");
	spyDeleteCount = 0;
	}
    }
/*******************************************************************************
*
* spyTask - run periodic task activity reports
*
* This routine is spawned as a task by spy (2), in order to provide the
* periodic task activity reports.  It simply loops printing a report,
* then delaying for the specified number of seconds.
*/

VOID spyTask (freq)
    int freq;		/* reporting frequency, in seconds */

    {
    int delay = freq * sysClkRateGet ();

    while (TRUE)
	{
	spyReport ();
	taskDelay (delay);
	}
    }
/*******************************************************************************
*
* spyStop - stop spying and reporting
*
* This routine calls spyClkStop (2) and 
* if periodic reporting by spyTask (2) is active, it is terminated.
*/

VOID spyStop ()

    {
    spyClkStop ();
    if (spyTaskId != ERROR)
	{
	taskDelete (spyTaskId);
	spyTaskId = ERROR;
	}
    }
/*******************************************************************************
*
* spy - begin periodic task activity reports
*
* Collect task activity data, and periodically run spyReport (2).
* Data will be gathered `ticksPerSec' times per second, and a report will
* be made every `freq' seconds.
* Spy spawns spyTask (2) to do the actual reporting.
* 
* It is not necessary to do a spyClkStart (2) before running.
*/

VOID spy (freq, ticksPerSec)
    int freq;			/* reporting frequency, in seconds */
				/* 0 = use default of 5            */
    int ticksPerSec;		/* interrupt clock frequency       */
				/* 0 = use default of 100          */

    {
    if (freq == 0)
	freq = 5;	/* default frequency is 5 secs */

    if (spyClkStart (ticksPerSec) == OK)
	{
	spyTaskId = taskSpawn ("spyTask", spyTaskPriority,
			       spyTaskOptions, spyTaskStackSize,
			       spyTask, freq, 0, 0, 0, 0, 0, 0, 0, 0, 0);
	if (spyTaskId == ERROR)
	    printErr ("Unable to spawn spyTask.\n");
	}
    }
/*******************************************************************************
*
* spyHelp - print a helpful list of the task activity commands
*
* This routine prints the following list of task activity commands:
* .CS
*  spyHelp                       Print this list
*  spyClkStart [ticksPerSec]     Start task activity monitor running
*                                  at ticksPerSec ticks per second
*  spyClkStop                    Stop collecting data
*  spyReport                     Prints display of task activity
*                                  statistics
*  spyStop                       Stop collecting data and reports
*  spy     [freq[,ticksPerSec]]  Start spyClkStart and do a report
*                                  every freq seconds
*  
*  ticksPerSec defaults to 100.  freq defaults to 5 seconds.
* .CE
*/

VOID spyHelp ()

    {
    static char *spy_help[] = {
    "spyHelp                       Print this list",
    "spyClkStart [ticksPerSec]     Start task activity monitor running",
    "                                at ticksPerSec ticks per second",
    "spyClkStop                    Stop collecting data",
    "spyReport                     Prints display of task activity",
    "                                statistics",
    "spyStop                       Stop collecting data and reports",
    "spy     [freq[,ticksPerSec]]  Start spyClkStart and do a report",
    "                                every freq seconds",
    "",
    "ticksPerSec defaults to 100.  freq defaults to 5 seconds.",
    "",
    };
    FAST int ix;

    for (ix = 0; ix < NELEMENTS(spy_help); ix++)
	printf ("%s\n", spy_help [ix]);
    }
