/* shellLib.c - shell execution routines */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/* modification history
-----------------------
02k,23may89,dnw  added VX_FP_TASK to shell task options.
02j,16nov88,gae  made shell be locked when not interactive, i.e. doing scripts.
		 made shell history size be a global variable.
02i,27oct88,jcf  added taskSuspend (0) to shellSigHandler so sigHandlerCleanup
		   is never run on a deleted task.
02h,23sep88,gae  documentation.
02g,06jun88,dnw  changed taskSpawn/taskCreate args.
02f,30may88,dnw  changed to v4 names.
02e,29may88,dnw  added VX_UNBREAKABLE to shell task options.
02d,28may88,dnw  renamed h to shellHistory and logout to shellLogout
		   (h and logout moved to usrLib).
		 removed call to dbgSetTaskBreakable() since shell is spawned
		   unbreakable.
		 replace shellSetOrig...() with shellOrigStdSet().
		 changed call to create() to creat().
02c,26may88,gae  added signal handling to shell task.
02b,02apr88,gae  made it work with I/O system changes.
		 added VX_STDIO option to shell.
		 made shell complain when output can't be closed properly.
02a,27jan88,jcf  made kernel independent.
		   removed unnecessary include of a_out.h
01e,21nov87,gae  fixed double prompt bug with scripts.
01d,02nov87,dnw  moved logout() and shellLogoutInstall() here from usrLib.
	   +gae  changed "korn" references to "led".
01c,24oct87,gae  changed name from shellExec.c.
		 changed setOrig{In,Out,Err}Fd() to shellSetOrig{In,Out,Err}(),
		   setShellPrompt() to shellSetPrompt(), and abortScript() to
		   shellAbortScript().  Added h() and shellLock().
01b,21oct87,gae  added shellInit().
01a,06sep87,gae  split from shell.yacc.
		 made shell restart itself on end of file.
*/

/*
DESCRIPTION
This module contains the execution support routines for the VxWorks shell.
It provides the basic programmer's interface to VxWorks.
It is a C expression interpreter, containing no built-in commands at all.

The nature, use, and syntax of the shell is more fully described in
the "Shell" chapter of the VxWorks Programmer's Manual.

SEE ALSO: ledLib (1)
*/

#include "vxWorks.h"
#include "ctype.h"
#include "ioLib.h"
#include "memLib.h"
#include "strLib.h"
#include "symLib.h"
#include "taskLib.h"

IMPORT int taskRestart ();

#define MAX_PROMPT_LEN	80
#define MAX_SHELL_LINE	128	/* max chars on line typed to shell */

/* global variables */

/* shell task parameters */

int shellTaskId;
char *shellTaskName	= "shell";
int shellTaskPriority	= 1;
int shellTaskOptions	= VX_SUPERVISOR_MODE | VX_STDIO | VX_FP_TASK |
			  VX_UNBREAKABLE;
int shellTaskStackSize	= 5000;		/* default/previous stack size */

int shellHistSize = 20;		/* default shell history size */

int redirInFd;		/* fd of input redirection stream */
int redirOutFd;		/* fd of output redirection stream */

/* local variables */

LOCAL char promptString [MAX_PROMPT_LEN] = "-> ";
LOCAL int ledId;		/* Line EDitor descriptor (-1 = don't use) */
LOCAL BOOL shellAbort;		/* TRUE = someone requested shell to abort */
LOCAL BOOL shellLocked;		/* TRUE = shell is in exclusive use */
LOCAL BOOL shellExecuting;	/* TRUE = shell was already executing */
LOCAL int origFd [3];		/* fd of original interactive streams */
LOCAL FUNCPTR logoutRtn;	/* network logout routine */
LOCAL int logoutVar;		/* network logout variable */

/* forward declarations */

VOID shell ();

/*******************************************************************************
*
* shellInit - start shell
*
* This routine starts the shell task.
* This is normally done by the root task, usrRoot (2) in usrConfig (1).
*/

STATUS shellInit (stackSize, arg)
    int stackSize;	/* shell stack (0 = previous/default value) */
    int arg;		/* argument to shell task */

    {
    if (taskNameToId (shellTaskName) != ERROR)
	return (ERROR);		/* shell task already active */

    if (stackSize != 0)
	shellTaskStackSize = stackSize;

    shellTaskId = taskSpawn (shellTaskName, shellTaskPriority,
			     shellTaskOptions, shellTaskStackSize,
			     shell, arg, 0, 0, 0, 0 ,0 ,0, 0, 0, 0);

    return ((shellTaskId == ERROR) ? ERROR : OK);
    }
/*******************************************************************************
*
* shellRestart - restart the shell task
*
* Used by shellSigHandler; called by excTask(2).
*/

LOCAL VOID shellRestart ()

    {
    if ((taskRestart (shellTaskId)) != ERROR)
	printErr ("shell restarted.\n");
    else
	{
	printErr ("spawning new shell.\n");
	if (shellInit (0, TRUE) == ERROR)
	    printErr ("can't restart shell.\n");
	}
    }
/*******************************************************************************
*
* shellSigHandler - general signal handler for shell task
*
* All signals to the shell task are caught by this handler.
* Any exception info is printed, then the shell's stack is traced.
* Finally the shell is restarted.
*
* ARGSUSED
*/

LOCAL VOID shellSigHandler (sig, code, scp)
    int sig;
    int code;
    SIGCONTEXT *scp;

    {
    extern int dbgPrintCall();

    TCBX *pTcbX = taskTcbX (0);

    excInfoShow (&pTcbX->excInfo);	/* prints any valid exception info */
    printErr ("\007\n");		/* ring bell */
    trcStack ((int *) scp->sc_aregs[6], (int *) scp->sc_sp,
	      (INSTR *) scp->sc_pc, dbgPrintCall);

    excJobAdd (shellRestart);
    taskSuspend (0);			/* wait until excTask restarts us */
    }
/*******************************************************************************
*
* shell - the VxWorks shell
*
* This is the shell task.  It is started with a single parameter, indicating
* whether this is an interactive shell (to be used from a terminal or a
* socket) or not (to execute a script).
*
* Normally, the shell is spawned (in interactive mode) by the root task
* usrRoot (2) when VxWorks starts up.  After that, this routine is called
* only to (by itself) execute scripts, or when the shell is restarted
* after an abort.
*
* The shell gets its input from standard in and sends output to standard
* out.  These normally start out being the console device, but are
* redirected by telnetdTask (2) and rlogindTask (2).
*
* Note that the shell is not reentrant, since yacc will not generate a
* reentrant parser.  Therefore, there can only be one shell executing
* at a time.
*/

VOID shell (interactive)
    BOOL interactive;	/* should be TRUE, except for a script */

    {
    int ix;
    SIGVEC sv;

    /* setup default signal handler */

    sv.sv_handler = shellSigHandler;
    sv.sv_mask    = 0;
    sv.sv_flags   = 0;

    for (ix = 1; ix < 32; ix++)
	sigvec (ix, &sv, (SIGVEC *) NULL);

    /* test for new call of shell or restart of aborted shell */

    if (!shellExecuting)
	{
	/* legitimate call of shell; if interactive save in/out/err fd's */

	if (interactive)
	    {
	    origFd [STD_IN]  = ioGlobalStdGet (STD_IN);
	    origFd [STD_OUT] = ioGlobalStdGet (STD_OUT);
	    origFd [STD_ERR] = ioGlobalStdGet (STD_ERR);
	    shellLocked = FALSE;

	    /* Install Line EDitor interface:
	     * This is done only after the first "interactive"
	     * invocation of the shell so that a startup
	     * script could change some parameters.
	     * Eg. ledId to -1 or shellHistSize to > 20.
	     */

	    if (ledId == NULL &&
		(ledId = ledOpen (STD_IN, STD_OUT, shellHistSize)) == ERROR)
		{
		printErr ("Unable to install Line EDitor interface\n");
		}
	    }
	else
	    {
	    /* do not allow rlogin's while doing a script */
	    shellLocked = TRUE;
	    }

	shellExecuting = TRUE;
	}
    else
	{
	/* this must be a restart, i.e. via ABORT key;
	 * restore original interactive in/out/err fd's */

	if (interactive)
	    {
	    ioGlobalStdSet (STD_IN,  origFd [STD_IN]);
	    ioGlobalStdSet (STD_OUT, origFd [STD_OUT]);
	    ioGlobalStdSet (STD_ERR, origFd [STD_ERR]);
	    printf ("\n");
	    }
	else
	    return;
	}

    ioctl (STD_IN, FIOOPTIONS, OPT_TERMINAL);

    (void)execShell (interactive);

    shellExecuting = FALSE;
    }
/*******************************************************************************
*
* execShell - execute stream of shell commands
*/

LOCAL STATUS execShell (interactive)
    FAST BOOL interactive;

    {
    char inLine [MAX_SHELL_LINE + 1];
    FAST int i;
    STATUS status = OK;

    shellAbort = FALSE;

    while (TRUE)
	{
	/* read next line */

	if (interactive)
	    {
	    int nchars;

	    printf ("%s", promptString);

	    if (ledId > NULL)
		nchars = ledRead (ledId, inLine, MAX_SHELL_LINE);
	    else
		nchars = fioRdString (STD_IN, inLine, MAX_SHELL_LINE);

	    if (nchars == EOF)
		{
		/* start shell over again */
		excJobAdd (taskRestart, shellTaskId);
		}
	    }
	else if (fioRdString (STD_IN, inLine, MAX_SHELL_LINE) != EOF)
	    printf ("%s\n", inLine);
	else
	    break;	/* end of script - bye */

	/* got a line of input:
	 *   ignore comments, blank lines, and null lines */

	inLine [MAX_SHELL_LINE] = EOS;	/* make sure inLine has EOS */

	for (i = 0; inLine [i] == ' '; i++)	/* skip leading blanks */
	    ;

	if (inLine [i] != '#' && inLine [i] != EOS)
	    {
	    /* parse & execute command */

	    status = execute (&inLine [i]);

	    if (status != OK && !interactive)
		{
		status = ERROR;
		break;
		}

	    /* check for script aborted;
	     *   note that "shellAbort" is a static variable and
	     *   is not reset until we get back up to an interactive
	     *   shell.  Thus all levels of nested scripts are aborted. */

	    if (shellAbort)
		{
		if (!interactive)
		    {
		    status = ERROR;
		    break;
		    }
		else
		    shellAbort = FALSE;
		}
	    }
	}

    return (status);
    }
/*******************************************************************************
*
* shellScriptAbort - signal shell to stop processing script
*
* This routine signals the shell to abort processing a script file.
* It can be called from within a script, if an error is detected.
*/

VOID shellScriptAbort ()

    {
    shellAbort = TRUE;
    }
/*******************************************************************************
*
* shellHistory - display (or set) shell history
*
* The shell maintains a commands history (actually the
* line editing libary ledLib(1) does).  The history size
* is by default 20 commands.
*
* SEE ALSO: ledLib(1), h(2)
*/

VOID shellHistory (size)
    int size;	/* 0 = display, >0 = set history to new size */

    {
    ledControl (ledId, NONE, NONE, size);
    }
/*******************************************************************************
*
* execute - interpret and execute a source line
*
* This routine parses and executes the specified source line.
* First any I/O redirection is cracked, then if any text remains,
* that text is parsed and executed via the yacc-based interpreter.
* If no text remains after the I/O redirection, then the shell itself
* is invoked (recursively) with the appropriate redirection.
* Note that blank lines, null lines, and comment lines should NOT
* be passed to this routine.  Initial blanks should be stripped too!
*
* RETURNS: OK | ERROR
*/

LOCAL STATUS execute (line)
    FAST char *line;

    {
    int newInFd;
    int newOutFd;
    int oldInFd  = ioGlobalStdGet (STD_IN);
    int oldOutFd = ioGlobalStdGet (STD_OUT);
    STATUS status;

    /* get any redirection specs */

    if (getRedir (line, &newInFd, &newOutFd) != OK)
	return (ERROR);

    if (*line == EOS)
	{
        /* set any redirections specified, call shell, and restore streams */

	if (newInFd >= 0)
	    ioGlobalStdSet (STD_IN, newInFd);
	if (newOutFd >= 0)
	    ioGlobalStdSet (STD_OUT, newOutFd);

	status = execShell (FALSE);

	ioGlobalStdSet (STD_IN, oldInFd);
	ioGlobalStdSet (STD_OUT, oldOutFd);
	}
    else
	{
	/* set global stream fds for redirection of function calls;
	 * a -1 means no redirection */

	redirInFd = newInFd;
	redirOutFd = newOutFd;

	/* initialize parse variables and parse and execute line */

	yystart (line);
	status = (yyparse () == 0) ? OK : ERROR;
	}

    /* close redirection files */

    if (newInFd >= 0)
	close (newInFd);

    if (newOutFd >= 0 && close (newOutFd) == ERROR)
	printf ("can't close output.\n");

    return (status);
    }
/*******************************************************************************
*
* getRedir - establish redirection specified on input line
*
* This routines picks the redirection specs off the end of the input line.
* The values pointed to by pInFd and pOutFd are set to -1 if the 
* input and output respectively are not redirected, and to the fd of the
* redirection stream if they are redirected.
* Note that this routine also trucates from the end of the input line
* any successfully cracked redirection specs, i.e. an EOS is inserted
* in the input line at the point where the redirection specs began.
*
* RETURNS: ERROR if error in redirection spec or opening stream,
*	   OK if successful redirection found, or no redirection found.
*/

LOCAL STATUS getRedir (line, pInFd, pOutFd)
    char *line;
    FAST int *pInFd;	/* -1, or fd of of input redirection */
    FAST int *pOutFd;	/* -1, or fd of of output redirection */

    {
    *pInFd = *pOutFd = -1;

    if (get1Redir (line, pInFd, pOutFd) != OK ||
        get1Redir (line, pInFd, pOutFd) != OK)
	{
	if (*pInFd >= 0)
	    close (*pInFd);

	if (*pOutFd >= 0)
	    close (*pOutFd);

	return (ERROR);
	}

    return (OK);
    }
/*******************************************************************************
*
* get1Redir - get a redirection from a line
*
* This routine picks a single redirection specification off the end
* of the specified line.  
* 
* RETURNS: ERROR if error in redirection spec or opening stream,
*	   OK if successful redirection found, or no redirection found.
*/

LOCAL STATUS get1Redir (line, pInFd, pOutFd)
    char *line;		/* line to scan */
    int *pInFd;		/* if '<' found then fd is assigned here */
    int *pOutFd;	/* if '>' found then fd is assigned here */

    {
    FAST char *p;	/* position in line */
    char *name;		/* name of redirection file if found */

    if (strlen (line) == 0)
	return (OK);

    /* Set p == end of line */

    p = line + strlen (line) - 1;

    /* pick off last word and back up to previous non-blank before that */

    while (p > line && *p == ' ')
	{
	p--;		/* skip trailing blanks */
	}

    *(p + 1) = EOS;

    /* stop searching if:
     *   reached beginning of line,
     *   reached a blank,
     *   reached a redirection token,
     *   hit a semicolon or quote.
     */

    while (p > line  && *p != ' ' &&
	   *p != '<' && *p != '>' &&
	   *p != ')' && *p != ';' &&
	   *p != '"')
	{
	p--;		/* skip back to start of word */
	}

    name = p + 1;	/* name must begin here */

    while (p > line && *p == ' ')
	p--;		/* skip back to previous non-blank */


    /* is this a redirection? */ 

    if (*p == '>' && *(p -1) != '>')
	{
	if (*pOutFd >= 0)
	    {
	    printf ("ambiguous output redirect.\n");
	    return (ERROR);
	    }

	if ((*pOutFd = creat (name, WRITE)) < 0)
	    {
	    printf ("can't create output stream '%s'.\n", name);
	    return (ERROR);
	    }

	*p = EOS; /* truncate to exclude the redirection stuff just used */
	}
    else if (*p == '<' && *(p -1) != '<')
	{
	if (*pInFd >= 0)
	    {
	    printf ("ambiguous input redirect.\n");
	    return (ERROR);
	    }

	if ((*pInFd = open (name, READ)) < 0)
	    {
	    printf ("can't open input '%s'.\n", name);
	    return (ERROR);
	    }

	*p = EOS; /* truncate to exclude the redirection stuff just used */
	}

    return (OK);
    }
/*******************************************************************************
*
* shellPromptSet - change the shell's prompt
*
* This routine changes the shell's prompt string to `newPrompt'.
*/

VOID shellPromptSet (newPrompt)
    char *newPrompt;	/* string to become new shell prompt */

    {
    strncpy (promptString, newPrompt, MAX_PROMPT_LEN);
    }
/*******************************************************************************
*
* shellOrigStdSet - set the shell's default input/output/error fds
*
* This routine is called to change the shell's default standard-in/out/error fd.
* It is normally used only by the shell itself, and by rlogindTask (2) and
* telnetdTask (2).
*/

VOID shellOrigStdSet (which, fd)
    int which;	/* STD_IN, STD_OUT, STD_ERR */
    int fd;	/* fd to be default */

    {
    origFd [which] = fd;
    }
/*******************************************************************************
*
* shellLock - lock access to shell
*
* The shell may be in one of two modes: locked or unlocked.
* When locked, cooperating tasks, such as telnetdTask (2) and rlogindTask (2),
* will not take the shell.
*
* RETURNS:
*  when request == lock
*    TRUE if able to lock, or
*    FALSE if unable to lock
*  when request == unlock
*    TRUE if able to unlock, or
*    FALSE if wasn't already locked
*/

BOOL shellLock (request)
    BOOL request;	/* TRUE = lock, FALSE = unlock */

    {
    if (request == shellLocked)
	return (FALSE);

    shellLocked = request;
    return (TRUE);
    }
/*******************************************************************************
* 
* shellLogoutInstall - logout hook for telnetdTask and rlogindTask
*
* NOMANUAL
*/ 

VOID shellLogoutInstall (logRtn, logVar)
    FUNCPTR logRtn;
    int logVar;

    {
    logoutRtn = logRtn;
    logoutVar = logVar;
    }
/*******************************************************************************
* 
* shellLogout - logout of VxWorks shell
*
* This routine logs out from VxWorks' shell.
* The client's socket is closed and standard I/O is restored to the console.
*
* SEE ALSO: rlogindTask (2), telnetdTask (2), logout(2)
*/ 

VOID shellLogout ()

    {
    if (logoutRtn != NULL)
	(*logoutRtn)(logoutVar);
    }
