/* rt11Lib.c - RT-11 file I/O library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
05f,07jul88,jcf  fixed mallocs to match new declaration.
05e,05jun88,dnw  changed from rtLib to rt11Lib.
05d,30may88,dnw  changed to v4 names.
05c,04may88,jcf  changed semaphores and sem calls for new semLib.
05b,09nov87,ecs  removed intemperate printErr from rtCreate.
05a,03nov87,ecs  documentation.
04z,20oct87,gae  made rtRename() cancel device portion of name, if present.
		 made rtDirEntry() return name for DES_TENTATIVE.
		 fixed bug in create() by flushing directory;
		 made create fail if create already active on device.
		 made rtSqueeze local.
04y,14sep87,dpg  moved squeeze to usrLib.c. changed FIOSQUEEZE call in rtIoctl
		 to use a fd rather than some aesoteric device pointer (i.e.
		 keep it looking the same).
04x,10sep87,dpg  deleted rtMaxRoom & maxRoom - we don't really need them now
		 renamed rtPack & pack to rtSqueeze & squeeze
04w,26aug87,dpg  added rtMaxRoom to find largest piece of free space on
		 a disk device, and a FIOMAXROOM to ioctl at it; together
		 with maxRoom - a user callable routine
04v,26jun87,dpg  added rtPack to compress free space on an RT-11 device into
		 one big block so that hard disks don't get too fragmented.
		 also added ioctl entry to provide uniform interface to
		 rtPack & pack (user callable routine)
04u,05may87,gae  fixed "the other" fatal bug introduced in 04t in rtRead().
		 changed constant N_RETRIES to rtMaxRetries to allow variation.
04u,28apr87,gae  fixed fatal bug introduced in 04t in rtVolMount().
04t,26mar87,gae  Made rtVolMount() fail on bad (unlikely) directory.
		 Got rid of rtVolFlush(), then renamed rtVFlush() to
		   rtVolFlush().
04s,07mar87,gae  Made the following LOCAL: rtGetEntry(), rtAbsSector(),
		   rtDirEntry(), rtRename(), rtFlush(), rtWhere(), rtSeek(),
		   rtVolFlush(), and rtVolInit().  Fixed documentation.
		 Made rtDevInit() return ERROR if out of memory.
04r,14feb87,dnw  Changed to accept null filename as special name for "raw disk"
		   instead of "-", now that ioLib protects against accidental
		   null filenames.
04q,03feb87,llk  Modified directory segments to handle any number of file
		   entries.  Directories are 1 segment long, with at least
		   2 blocks per segment.  2 block segments are still RT-11
		   compatible.  rtDevInit has a new parameter, the maximum
		   number of file entries that will go on the device.
		   See documentation below under RT-11 COMPATIBILITY.
		 Added FIODISKCHANGE to ioctl function codes.  Used for
		   manually issuing a ready change.
		 Fixed rtDirEntry so that it checks if the entry number passed
		   to it is less than the maximum possible entry number.
04p,14jan87,llk  changed to handle file size up to RT_MAX_BLOCKS_PER_FILE.
		 Changed rtVolInit to initialize disk with as many maximum
		 sized files as needed to cover all the disk space.
		 Changed rtCoalesce to coalesce files together iff the new
		 empty file would have less than RT_MAX_BLOCKS_PER_FILE blocks.
...deleted pre 87 history - see RCS
*/

/*
This library provides services for file oriented device drivers which use
the RT-11 file standard.  This module takes care of all the buffering,
directory maintenance, and RT-11-specific details necessary.
The routines in this library are NOT to be called directly by users, but
rather by device drivers.

To use this library for a particular device, the device structure
must contain, as the very first item, an RT_VOL_DESC.  This is
initialized by calling rt11DevInit.  On that call, the driver gives the
addresses of three routines which it must supply, one that reads a sector,
one that writes a sector, and one that resets the device.
Those routines are described below. This call also specifies the maximum
number of files that can be contained in the device's directory, and
sets up the disk format (bytes/sector, sectors/track, etc.).

Thereafter, when the driver receives a request from the I/O system, it 
simply calls the corresponding routine in this module to fulfill that
request.  If the rt11Lib routine needs to actually access the device, it
does so via the rdSec and wrtSec routines which are in the RT_VOL_DESC.

When a file is opened with rt11Open or rt11Create, rt11Lib sets up an
RT_FILE_DESC that contains the information for that particular file on the disk.
This includes a pointer to the device's RT_VOL_DESC, the file pointers, 
starting block, extent, and block buffer.
A pointer to the file's RT_FILE_DESC is returned by rt11Open and rt11Close.
This pointer is then used to identify the file in subsequent calls to
the other file oriented functions, rt11Read, rt11Write, rt11Close, and several
control functions accessed through rt11Ioctl.

RDSEC, WRTSEC, AND RESET
As noted above, the driver must supply routines to read and write single
sectors from the disk, and reset the device.  This is done in the call to
rt11DevInit.  For instance:

.CS
    rt11DevInit (vdptr,..., myRdSec, myWrtSec, myReset);
.CE

The read and write routines must be declared as follows:

.CS
    STATUS myWrtSec (devPtr, secNum, buffer)
	MY_DEVICE *devPtr;	/* pointer to device descriptor	*
	int secNum;		/* number of sector to write	*
	char *buffer;		/* data to write to sector	*
.CE

The rdSec routine is identical, except the name must be different.
These routines must read (write) the specified sector from (to) disk, then
return a value of OK (if all went well) or ERROR (otherwise).

The reset routine should be declared as follows:

.CS
    STATUS myReset (devPtr)
	MY_DEVICE *devPtr;	/* pointer to device descriptor	*
.CE

When called, the routine should reset the disk and controller.  This routine
is called each time a new volume is mounted, and during the error retry loop
for reading and writing blocks.  If the reset was successful, the routine should
return OK.  If not, it should return ERROR.  Volumes will be available only 
if the reset is successful.

The routine names may be anything desired; rt11Lib accesses these routines not
by name, but by the address passed in the rt11DevInit call.

MULTIPLE LOGICAL DEVICES AND RT-11 COMPATIBILITY

The sector number passed to the sector read and write routines is an absolute
number, starting from sector 0 at the beginning of the device.  If desired,
the driver may add an offset from the beginning of the physical device
before the start of the logical device.  This would normally be done by
keeping a trackOffset parameter in the MY_DEVICE structure, and adding the
proper number of tracks to the sector passed to the read and write routines.

The RT-11 standard defines the disk to start on track 1.  Track 0 is set
aside for boot information, etc.  Therefore, in order to retain true
compatibility with RT-11 systems, a one track offset needs to be added
to the sector numbers passed to the sector read and write routines,
and the device size needs to be declared as one track smaller than it
actually is.  This must be done by the driver using this library.
rt11Lib does not add such an offset for you.

In the VxWorks implementation, the directory is a fixed size,
able to contain at least as many files as specified in the call to rt11DevInit.
If the maximum number of files is specified to be RT_FILES_FOR_2_BLOCK_SEG,
then strict RT-11 compatibility is maintained because this is the initial
allocation in the RT-11 standard.

DIRECTORY ENTRIES
An ioctl call with function == FIODIRENTRY code returns information about a
particular directory entry.
A pointer to a REQ_DIR_ENTRY structure is passed as
the parameter.  The field "entryNum" in the REQ_DIR_ENTRY structure
must be set to the desired entry number.
The name of the file, its size (in bytes), and its creation date are
returned in the structure.
If the specified entry is empty, the name will
be an empty string, the size will be the space in this slot,
and the date will be meaningless.
Typically, the entries are accessed sequentially, starting
with entryNum = 0, until the terminating entry is reached,
indicated by a return code of ERROR.

DIRECTORIES IN MEMORY
A copy of the directory for each volume is kept in memory (in the RT_VOL_DESC
structure).  This speeds up directory accesses, but requires that rt11Lib
be notified when disks are changed (i.e. floppies are swapped).
If the driver can find this out (by interrogating controller status or by
receiving an interrupt) the driver simply calls rt11ReadyChange when a disk
is inserted or removed.  rt11Lib will automatically try to mount the device 
next time it needs it.

If the driver does not have access to the information that disk volumes
have been changed, rt11ReadyChange must be called at somewhat arbitrary times.
It should probably be called before each open, create, delete,
and directory listing.

rt11ReadyChange can also be called by user tasks by issuing an ioctl call
with FIODISKCHANGE as the function code.  

RTFMT
The RT-11 standard defines a peculiar software interleave and
track-to-track skew as part of the format.  The rtFmt parameter passed to
rt11DevInit should be TRUE if this formatting is desired.  This
should be the case if strict RT-11 compatibility is desired, or if files
must be transferred between the development and target machines using
the VxWorks-supplied RT-11 tools.
Software interleave and skew will automatically be dealt with by rt11Lib.

When rtFmt has been passed as TRUE and the maximum number of files is
specified RT_FILES_FOR_2_BLOCK_SEG, the driver does not need to do
anything else (except to add the track offset as described above) to
maintain RT-11 compatibility.

Note that if the number of files under either a VxWorks system,
or an RT-11 system, is specified other than RT_FILES_FOR_2_BLOCK_SEG
compatibility is lost because VxWorks allocates a contiguous directory,
whereas RT-11 systems create chained directories.

ACCESSING THE RAW DISK
rt11Lib recognizes a null filename as a special case in the rt11Open and
rt11Create calls, indicating access to the entire "raw" disk is desired, as
opposed to a file on the disk.
This is how a stream is obtained to a disk that doesn't have a file system.
For example, when one wants to initialize a new file system on the disk
use an ioctl call with FIODISKINIT.
To read the directory of a disk on which no file names are known,
open the "raw" disk and use an ioctl call with FIODIRENTRY.

INITIALIZING RTLIB
Before any other routines in rt11Lib can be used, the routine rt11Init must
be called to initialize this library.
In this call, one specifies the maximum number of RT-11 files
that can be open simultaneously.  rt11Init allocates memory for that many
RT_FILE_DESCs.  Attempts to open more RT-11 files than the specified maximum
will result in errors from rt11Open and rt11Create.

RT-11 HINTS
The RT-11 file system is much simpler than the more common
UNIX or MS-DOS file systems.  The advantage of RT-11 is its
speed; file access is made in at most one seek because files
are contiguous.  Some of the most common errors
for users with a UNIX background are:

    + only one create at a time may be active per device.
    + file size is set by the first create and close sequence;
	to ensure a specific file size use lseek;
	there is no append function to expand a file.
    + files are strictly block oriented, unused portions
	of a block are filled with NULL's -- there is no
	end-of-file marker other than the last block.

INCLUDE FILE
rt11Lib.h

SEE ALSO
"I/O System", ioLib (1), iosLib (1), ramDrv (3)
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "ctype.h"
#include "semLib.h"
#include "rt11Lib.h"
#include "memLib.h"
#include "strLib.h"

#define RT_NAME_LEN	11

int rt11MaxRetries = 5;		/* number of retries for read/write sectors */

LOCAL char rad50 [] = " abcdefghijklmnopqrstuvwxyz$.\3770123456789";

LOCAL RT_FILE_DESC *rt11Fd;	/* pointer to list of file descriptors	    */
LOCAL int rt11MaxFiles;		/* max RT-11 files that can be open at once */
LOCAL SEM_ID rt11FdSemId;	/* interlock file descriptor list access    */
LOCAL int rt11Day = 0;		/* day of month, default is 0		    */
LOCAL int rt11Month = 0;	/* month of year, default is 0		    */
LOCAL int rt11Year = 72;	/* year, default is '72 (stored as 0)	    */

/* forward declarations */

LOCAL RT_FILE_DESC *rt11GetFd ();


/*******************************************************************************
*
* rt11Close - close an RT-11 file
*
* This routine closes the specified RT-11 file.
* The end of the buffer beyond the end of file is cleared out,
* the buffer is flushed, and the directory is updated if necessary.
*
* RETURNS:
*    OK, or
*    ERROR if directory couldn't be flushed or
*             entry couldn't be found.
*/

STATUS rt11Close (pfd)
    FAST RT_FILE_DESC *pfd;	/* file descriptor pointer */

    {
    FAST RT_DIR_ENTRY *pEntry = &pfd->rfd_dir_entry;
    FAST int remaining_blocks;
    FAST int nblocks;
    int buf_index;
    STATUS status;
    char name[RT_NAME_LEN];	/* file name gets unpacked into here */
    int entryNum;		/* file's entry number in directory */
    int start;			/* for receiving side-effect of rt11FindEntry */

    /* if current buffer has been written to and contains end of file,
     *   clear out end of buffer that lies beyond end of file. */

    if (pfd->rfd_modified && ((pfd->rfd_curptr / RT_BYTES_PER_BLOCK) == 
			      (pfd->rfd_endptr / RT_BYTES_PER_BLOCK)))
	{
	buf_index = pfd->rfd_endptr - pfd->rfd_curptr;

	bzero (&pfd->rfd_buffer[buf_index], RT_BYTES_PER_BLOCK - buf_index);
	}

    status = rt11Flush (pfd);

    /* if file is new, update directory */

    if (pEntry->de_status == DES_TENTATIVE)
	{
	/* update directory entry to be permanent with actual size */

	nblocks = ((pfd->rfd_endptr - 1) / RT_BYTES_PER_BLOCK) + 1;
	remaining_blocks = pEntry->de_nblocks - nblocks;

	pEntry->de_status  = DES_PERMANENT;
	pEntry->de_nblocks = nblocks;

	semTake (pfd->rfd_vdptr->vd_semId);	/* get ownership of volume */

	/* unpack name and find entry number */

	rt11NameString (pfd->rfd_dir_entry.de_name, name);
	entryNum = rt11FindEntry (pfd->rfd_vdptr, name, &start);

	if (entryNum == ERROR)
	    status = ERROR;
	else
	    {
	    rt11PutEntry (pfd->rfd_vdptr, entryNum, pEntry);

	    /* if unused blocks are left over, insert EMPTY entry in directory*/

	    if (remaining_blocks != 0)
		{
		pEntry->de_status  = DES_EMPTY;
		pEntry->de_nblocks = remaining_blocks;

		rt11InsEntry (pfd->rfd_vdptr, entryNum + 1, pEntry);
		rt11Coalesce (pfd->rfd_vdptr, entryNum + 1);
		}

	    /* make sure directory is written out */

	    if (rt11VolFlush (pfd->rfd_vdptr) != OK)
		status = ERROR;
	    }

	semGive (pfd->rfd_vdptr->vd_semId);		/* release volume */
	}

    rt11FreeFd (pfd);				/* mark fd not in use */
    return (status);
    }
/*******************************************************************************
*
* rt11Coalesce - merge empty directory entries
*
* This routine merges a directory entry with its empty neighbors, if any.
* It stops when there are no more empty neighbors, or when adding 
* another empty file would cause it to have more than RT_MAX_BLOCKS_PER_FILE
* blocks in an empty file.
* The directory will be updated.
*
* Possession of the volume descriptor's semaphore must have been secured
* before this routine is called.
*/

LOCAL VOID rt11Coalesce (vdptr, entryNum)
    RT_VOL_DESC *vdptr;		/* pointer to volume descriptor		*/
    int entryNum;		/* number of empty entry to coalesce	*/

    {
    RT_DIR_ENTRY entry;		/* 1st of 1-3 empty directory entries	*/
    RT_DIR_ENTRY nentry;	/* next entry				*/
    int n = 0;			/* number of entries to merge into first*/

    rt11GetEntry (vdptr, --entryNum, &entry);

    if ((entryNum < 0) || (entry.de_status != DES_EMPTY))
	rt11GetEntry (vdptr, ++entryNum, &entry);

    rt11GetEntry (vdptr, entryNum + 1, &nentry);	/* get next entry */

    /* start coalescing -- don't coalesce 2 files if the new file would
       have more than RT_MAX_BLOCKS_PER_FILE */

    while (nentry.de_status == DES_EMPTY)
	{
    	if ((entry.de_nblocks + nentry.de_nblocks) > RT_MAX_BLOCKS_PER_FILE)
	    {
	    /* These two files can't be merged. The new file would be too big */

	    if (n > 0)		/* don't change entry ptr w.o. coalescing */
		break;
	    else
		{
		/* Can't coalesce current empty entry.  Point to next one */

		entryNum++;
		entry = nentry;
		}
	    }
	else
	    {
	    entry.de_nblocks += nentry.de_nblocks;	/* seize empty blocks */
	    ++n;
	    }
	rt11GetEntry (vdptr, entryNum + 1 + n, &nentry);/* get next entry */
	}

    if (n > 0)			/* any entries coalesced? */
	{
	rt11PutEntry (vdptr, entryNum, &entry);		/* put empty entry */
	rt11PutEntry (vdptr, ++entryNum, &nentry);	/* put nonempty entry */

	/* move rest of entries up by n places */

	while (nentry.de_status != DES_END)
	    {
	    rt11GetEntry (vdptr, ++entryNum + n, &nentry);
	    rt11PutEntry (vdptr, entryNum, &nentry);
	    }
	}
    }
/*******************************************************************************
*
* rt11Create - create an RT-11 file
*
* This routine creates the file `name' with the specified mode.
* If the file already exists, it is first deleted then recreated.
* The largest empty space on the device is allocated to the new file.
* Excess space will be recovered when the file is closed.
* An RT-11 file descriptor is initialized for the file.
*
* A file name of zero length, i.e. "" is used to open an entire "raw" disk.
* In this case, no attempt is made to access the disk's directory, so that
* even un-initialized disks may be accessed.
*
* RETURNS
*    Pointer to RT-11 fd, or
*    NULL if error in create.
*/

RT_FILE_DESC *rt11Create (vdptr, name, mode)
    RT_VOL_DESC *vdptr;	/* pointer to volume descriptor		*/
    char *name;		/* RT-11 string (ffffff.ttt)		*/
    int mode;		/* file mode (READ/WRITE/UPDATE)	*/

    {
    RT_DIR_ENTRY entry;
    int start;	
    FAST int e_num;
    RT_DIR_ENTRY max_entry;
    int max_start;
    int max_e_num;		/* maximum entry number 	    */
    FAST int nEntries;		/* number of entries that will fit in segment */
    FAST RT_FILE_DESC *pfd = rt11GetFd (); /* file descriptor pointer */

    if (pfd == NULL)
	return (NULL);		/* no free file descriptors */

    nEntries = (vdptr->vd_nSegBlocks * RT_BYTES_PER_BLOCK
		- (sizeof (RT_DIR_SEG) - sizeof (RT_DIR_ENTRY)))
		/ sizeof (RT_DIR_ENTRY);

    /* check for create of raw device (null filename) */

    if (name [0] == EOS)
	{
	/* check that volume is available */

	semTake (vdptr->vd_semId);		/* get ownership of volume */

	if (rt11CheckVol (vdptr, FALSE) < OK)
	    {
	    semGive (vdptr->vd_semId);		
	    rt11FreeFd (pfd);
	    errnoSet (S_rt11Lib_VOLUME_NOT_AVAILABLE);
	    return (NULL);
	    }

	/* disk must be writable to create file */

	if (rt11VolMode (vdptr) == READ)
	    {
	    semGive (vdptr->vd_semId);		
	    rt11FreeFd (pfd);
	    errnoSet (S_ioLib_WRITE_PROTECTED);
	    return (NULL);
	    }

	semGive (vdptr->vd_semId);		

	/* null name is special indicator for "raw" disk;
	 *   fabricate a bogus directory entry covering entire volume */

	pfd->rfd_dir_entry.de_status  = DES_BOGUS;
	pfd->rfd_dir_entry.de_date    = rt11Date (rt11Year, rt11Month, rt11Day);
	pfd->rfd_dir_entry.de_nblocks = vdptr->vd_nblocks;
	rt11NameR50 ("device.raw", &pfd->rfd_dir_entry.de_name);

	pfd->rfd_start	= 0;
	pfd->rfd_endptr	= vdptr->vd_nblocks * RT_BYTES_PER_BLOCK;
	}
    else
	{
	if(rt11SlashRemove (name) == ERROR) {
	    rt11FreeFd (pfd);
	    return(NULL);
	}

	/* first delete any existing file with the specified name */

	rt11Delete (vdptr, name);

	/* check that volume is available */

	semTake (vdptr->vd_semId);		/* get ownership of volume */

	if ((rt11CheckVol (vdptr, TRUE) < OK) || (vdptr->vd_status < OK))
	    {
	    semGive (vdptr->vd_semId);			/* release volume */
	    rt11FreeFd (pfd);
	    errnoSet (S_rt11Lib_VOLUME_NOT_AVAILABLE);
	    return (NULL);
	    }

	/* disk must be writable to create file */

	if (rt11VolMode (vdptr) == READ)
	    {
	    semGive (vdptr->vd_semId);			/* release volume */
	    rt11FreeFd (pfd);
	    errnoSet (S_ioLib_WRITE_PROTECTED);
	    return (NULL);
	    }

	/* search entire directory for largest empty entry.
	 * keep track of start block by accumulating entry lengths.
	 */

	start = vdptr->vd_dir_seg->ds_start;
	max_e_num = NONE;

	for (e_num = 0; e_num < nEntries; e_num++)
	    {
	    rt11GetEntry (vdptr, e_num, &entry);
	
	    /* if this is the end of the directory, then quit. */

	    if (entry.de_status == DES_END)
		break;

	    /* check if create already in progress */

	    if (entry.de_status == DES_TENTATIVE)
		{
		semGive (vdptr->vd_semId);	/* release volume */
		rt11FreeFd (pfd);
		errnoSet (S_rt11Lib_NO_MORE_FILES_ALLOWED_ON_DISK);
		return (NULL);
		}

	    /* check if this is largest empty file so far */

	    if ((entry.de_status == DES_EMPTY) &&
		((max_e_num == NONE) || 
		 (entry.de_nblocks > max_entry.de_nblocks)))
		{
		max_e_num = e_num;
		max_entry = entry;
		max_start = start;
		}

	    /* add file length to get start of next file */

	    start += entry.de_nblocks;
	    }

	/* check for running out of room for file entries 
	 * (the last entry is a terminating entry) */

	if (e_num >= nEntries - 1)
	    {
	    semGive (vdptr->vd_semId);			/* release volume */
	    rt11FreeFd (pfd);
	    errnoSet (S_rt11Lib_NO_MORE_FILES_ALLOWED_ON_DISK);
	    return (NULL);
	    }

	/* check for no empty holes found */

	if (max_e_num == NONE)
	    {
	    semGive (vdptr->vd_semId);			/* release volume */
	    rt11FreeFd (pfd);
	    errnoSet (S_rt11Lib_DISK_FULL);
	    return (NULL);
	    }

	/* found an empty hole; initialize entry */

	max_entry.de_status = DES_TENTATIVE;
	max_entry.de_date   = rt11Date (rt11Year, rt11Month, rt11Day);
	rt11NameR50 (name, &max_entry.de_name);

	rt11PutEntry (vdptr, max_e_num, &max_entry);

	rt11VolFlush (vdptr);

	semGive (vdptr->vd_semId);			/* release volume */

	pfd->rfd_dir_entry = max_entry;
	pfd->rfd_start	   = max_start;
	pfd->rfd_endptr	   = 0;
	}

    /* initialize rest of file descriptor */

    pfd->rfd_mode	= mode;
    pfd->rfd_vdptr	= vdptr;
    pfd->rfd_curptr	= NONE;
    pfd->rfd_newptr	= 0;
    pfd->rfd_modified	= FALSE;

    return (pfd);
    }
/*******************************************************************************
*
* rt11Date - generate RT-11 encoded date
*
* This routine encodes the specified date into RT-11's format.
*
* RETURNS: encoded date
*/

LOCAL int rt11Date (year, month, day)
    int year;		/* 72, or 72...03	*/
    int month;		/* 0, or 1...12		*/
    int day;		/* 0, or 1...31		*/

    {
    if ((year -= 72) < 0)
	year += 100;

    return ((month << 10) | (day << 5) | (year & 0x1f));
    }
/*******************************************************************************
*
* rt11Delete - delete RT-11 file
*
* This routine deletes the file `name' from the specified RT-11 volume.
*
* RETURNS:
*    OK, or
*    ERROR if file not found or volume not available.
*/

STATUS rt11Delete (vdptr, name)
    RT_VOL_DESC *vdptr;	/* pointer to volume descriptor		*/
    char *name;		/* RT-11 filename (ffffff.ttt)		*/

    {
    int entryNum;
    int start;
    RT_DIR_ENTRY entry;

    semTake (vdptr->vd_semId);		/* get ownership of volume */

    if(rt11SlashRemove (name) == ERROR) {
	semGive (vdptr->vd_semId);		
	return(NULL);
    }

    /* check that volume is available */

    if ((rt11CheckVol (vdptr, TRUE) < OK) || (vdptr->vd_status < OK))
	{
	semGive (vdptr->vd_semId);		
	errnoSet (S_rt11Lib_VOLUME_NOT_AVAILABLE);
	return (ERROR);
	}

    if (rt11VolMode (vdptr) == READ)
	{
	semGive (vdptr->vd_semId);		
	errnoSet (S_ioLib_WRITE_PROTECTED);
	return (ERROR);
	}

    /* search for entry with specified name */

    if ((entryNum = rt11FindEntry (vdptr, name, &start)) == ERROR)
	{
	semGive (vdptr->vd_semId);		/* release volume */
	return (ERROR);
	}

    rt11GetEntry (vdptr, entryNum, &entry);
    entry.de_status = DES_EMPTY;
    entry.de_date = 0;
    rt11PutEntry (vdptr, entryNum, &entry);

    rt11Coalesce (vdptr, entryNum);		/* merge with empty neighbors */

    /* make sure directory is written out */

    if (rt11VolFlush (vdptr) != OK)
	{
	semGive (vdptr->vd_semId);		/* release volume */
	return (ERROR);
	}

    semGive (vdptr->vd_semId);			/* release volume */

    return (OK);
    }
/*******************************************************************************
*
* rt11DevInit - initialize RT-11 device descriptor
*
* This routine initializes the device descriptor.  The parameters bytesPerSec,
* secPerTrack, and nSectors describe the physical format of the device.
* rt11Fmt is TRUE iff the device is to be accessed using standard RT-11 skew
* and interleave.
* The function passed as rdSec_func will be called whenever sectors
* need to be read from the device, wrtSec_func will be called
* whenever sectors need to be written, and reset_func will be called when the
* device needs to be reset (after a volume change, for instance).  None of
* these functions will be called from rt11DevInit.
*
* The device's directory will consist of one segment able to contain at
* least as many files as specified by nEntries.
* If nEntries is RT_FILES_FOR_2_BLOCK_SEG, then strict RT-11 
* compatibility is maintained.
*
* NOTE
* An ERROR will be returned if rt11Fmt is TRUE and sectors/track is odd.
* This is because an odd number of sectors/track is incompatible with the
* RT-11 interleaving algorithm.  
*
* INTERNAL
* The semaphore in the device is given, so the device is available for
* use immediately.
*
* RETURNS:
*    OK, or
*    ERROR if invalid device parameters were specified, or out of memory.
*/

STATUS rt11DevInit (vdptr, bytesPerSec, secPerTrack, nSectors, 
		    rt11Fmt, nEntries, rdSec_func, wrtSec_func, reset_func)
    RT_VOL_DESC *vdptr;		/* pointer to volume descriptor		*/
    int bytesPerSec;		/* number of bytes per physical sector	*/
    int secPerTrack;		/* number of sectors per track		*/
    int nSectors;		/* number of sectors on device		*/
    BOOL rt11Fmt;		/* TRUE iff device is to be accessed	*/
				/* using RT-11 track skew & interleave	*/
    FAST int nEntries;		/* number of directory entries for this */
				/* volume, including terminating entry	*/
    FUNCPTR rdSec_func;		/* function to read a sector from	*/
				/* this device				*/
    FUNCPTR wrtSec_func;	/* function to write a sector to	*/
				/* this device				*/
    FUNCPTR reset_func;		/* function to reset device		*/

    {
    FAST int i;
    FAST int segmentLen;	/* segment length in bytes */

    /* don't allow odd number of sectors/track if RT-11 interleave specified */

    if (rt11Fmt && (secPerTrack & 1))
	{
	errnoSet (S_rt11Lib_INVALID_DEVICE_PARAMETERS);
	return (ERROR);
	}

    /* initialize volume descriptor */

    vdptr->vd_status		= OK;
    vdptr->vd_rdSec		= rdSec_func;
    vdptr->vd_wrtSec		= wrtSec_func;
    vdptr->vd_reset		= reset_func;
    vdptr->vd_rtFmt		= rt11Fmt;
    vdptr->vd_bytesPerSec	= bytesPerSec;
    vdptr->vd_secTrack		= secPerTrack;
    vdptr->vd_nSectors		= nSectors;
    vdptr->vd_secBlock		= RT_BYTES_PER_BLOCK / bytesPerSec;
    vdptr->vd_nblocks		= nSectors / vdptr->vd_secBlock;
    vdptr->vd_state		= RT_VD_READY_CHANGED;
    vdptr->vd_mode		= UPDATE;

    /* how many bytes of space are needed for the segment? */

    segmentLen = sizeof (RT_DIR_SEG) + (nEntries - 1) * sizeof (RT_DIR_ENTRY);

    /* Allocate blocks for the segment.
     * The smallest segment is 2 blocks long */

    if  ((i = 1 + (segmentLen / RT_BYTES_PER_BLOCK)) < 2)
	vdptr->vd_nSegBlocks = 2;
    else
        vdptr->vd_nSegBlocks = i;
 
    /* allocate segment blocks */

    vdptr->vd_dir_seg = (RT_DIR_SEG *)
			malloc ((unsigned)
				(vdptr->vd_nSegBlocks * RT_BYTES_PER_BLOCK));
    if (vdptr->vd_dir_seg == NULL)
	return (ERROR);

    vdptr->vd_semId = semCreate ();
    semGive (vdptr->vd_semId);

    return (OK);
    }
/*******************************************************************************
*
* rt11DirEntry - get info from a directory entry
*
* This routine returns information about a particular directory entry into
* the REQ_DIR_ENTRY structure whose pointer is passed as a parameter.
* The information put there is the name of the file, its size (in bytes),
* and its creation date.  If the specified entry is an empty (keeps
* track of empty space on the disk), the name will be an empty string,
* the size will be correct, and the date will be meaningless.
*
* Before this routine is called, the field entryNum must set in 
* the structure pointed to by rdeptr.  That is the entry whose information
* will be returned.  Typically, the entries are accessed sequentially starting
* with 0 until the terminating entry is reached (indicated by a return code
* of ERROR).
*
* RETURNS: 
*    OK, or
*    ERROR if no such entry.
*/

LOCAL STATUS rt11DirEntry (vdptr, rdeptr)
    RT_VOL_DESC *vdptr;		/* pointer to volume descriptor		   */
    REQ_DIR_ENTRY *rdeptr;	/* ptr to structure into which to put info */

    {
    RT_DIR_ENTRY *deptr;	/* pointer to directory entry in question     */
    FAST int maxEntries;	/* max number of entries allowed in directory */

    semTake (vdptr->vd_semId);	/* get ownership of volume */

    if (rt11CheckVol (vdptr, TRUE) < OK)
	{
	semGive (vdptr->vd_semId);		
	errnoSet (S_rt11Lib_VOLUME_NOT_AVAILABLE);
	return (ERROR);
	}

    /* what is the maximum number of entries allowed in this directory? */

    maxEntries = (vdptr->vd_nSegBlocks * RT_BYTES_PER_BLOCK
		- (sizeof (RT_DIR_SEG) - sizeof (RT_DIR_ENTRY)))
		/ sizeof (RT_DIR_ENTRY);

    if (rdeptr->entryNum >= maxEntries)
	{
	semGive (vdptr->vd_semId);		
	errnoSet (S_rt11Lib_ENTRY_NUMBER_TOO_BIG);
	return (ERROR);
	}
	
    deptr = &(vdptr->vd_dir_seg->ds_entries[rdeptr->entryNum]);

    switch (deptr->de_status)
	{
	case DES_TENTATIVE:	/* - should indicate tentative somehow? */
	case DES_PERMANENT:
	    rt11NameString (deptr->de_name, rdeptr->name);
	    break;

	case DES_EMPTY:
	    rdeptr->name[0] = EOS;			/* empty, no name */
	    break;

	default:
	    semGive (vdptr->vd_semId);			/* release volume */
	    errnoSet (S_rt11Lib_FILE_NOT_FOUND);
	    return (ERROR);				/* no such entry */
	}

    rdeptr->nChars = deptr->de_nblocks * RT_BYTES_PER_BLOCK;
    rdeptr->day = (deptr->de_date >> 5) & 0x001f;
    rdeptr->month = (deptr->de_date >> 10) & 0x000f;
    rdeptr->year = (deptr->de_date & 0x001f) + 1972;

    semGive (vdptr->vd_semId);			/* release volume */

    return (OK);
    }
/*******************************************************************************
*
* rt11FindEntry - find a directory entry by name on an RT-11 volume
*
* This routines searches for the directory entry of the specified file
* from the specified volume, if it exists.
*
* Possession of the volume descriptor's semaphore must have been secured
* before this routine is called.
*
* RETURNS:
*    entry number if found, or
*    ERROR if file not found
*/

LOCAL int rt11FindEntry (vdptr, name, pstart)
    FAST RT_VOL_DESC *vdptr;	/* pointer to volume descriptor */
    char *name;			/* RT-11 filename (ffffff.ttt)	*/
    int *pstart;		/* pointer where to return start block number */

    {
    RT_DIR_ENTRY entry;
    RT_NAME name_r50;
    FAST int i;

    /* search entire directory for active entry matching specified name.
     * keep track of start block by accumulating entry lengths.
     */

    rt11NameR50 (name, &name_r50);
    *pstart = vdptr->vd_dir_seg->ds_start;

    for (i = 0; ; i++)
	{
	rt11GetEntry (vdptr, i, &entry);
	
	/* if this is the end of the directory, then file not found. */

	if (entry.de_status == DES_END)
	    {
	    errnoSet (S_rt11Lib_FILE_NOT_FOUND);
	    return (ERROR);
	    }


	/* check if this is the desired file */

        if (((entry.de_status == DES_PERMANENT)		|| 
	     (entry.de_status == DES_TENTATIVE))	&&
	    (bcmp ((char *) &entry.de_name, (char *) &name_r50, 
							sizeof (RT_NAME)) == 0))
	    {
	    return (i);
	    }

	/* entry doesn't match;
	 * add file length to get start of next file; bump to next entry */

	*pstart += entry.de_nblocks;
	}
    }
/*******************************************************************************
*
* rt11Flush - flush RT-11 file output buffer
*
* This routine guarantees that any output that has been requested
* is actually written to the disk.  In particular, writes that have
* been buffered are written to the disk.
*
* RETURNS:
*    OK, or
*    ERROR if something needed to be written but couldn't.
*/

LOCAL STATUS rt11Flush (pfd)
    FAST RT_FILE_DESC *pfd;	/* pointer to file descriptor */

    {
    int block;

    if (pfd->rfd_modified)
	{
	/* write out the current (dirty) block and reset 'modified' indicator */

	block = pfd->rfd_start + (pfd->rfd_curptr / RT_BYTES_PER_BLOCK);

	if (rt11WrtBlock (pfd->rfd_vdptr, block, pfd->rfd_buffer) < OK)
	    return (ERROR);

	pfd->rfd_modified = FALSE;
	}

    return (OK);
    }
/*******************************************************************************
*
* rt11FreeFd - free a file descriptor
*/

LOCAL VOID rt11FreeFd (pfd)
    RT_FILE_DESC *pfd;			/* pointer to file descriptor to free */

    {
    semTake (rt11FdSemId);
    pfd->rfd_status = NOT_IN_USE;
    semGive (rt11FdSemId);
    }
/*******************************************************************************
*
* rt11GetFd - get an available file descriptor
*
* RETURNS: pointer to file descriptor, or NULL if none available.
*/

LOCAL RT_FILE_DESC *rt11GetFd ()
    {
    FAST RT_FILE_DESC *pRtFd;
    FAST int i;

    semTake (rt11FdSemId);

    for (i = 0, pRtFd = &rt11Fd [0]; i < rt11MaxFiles; i++, pRtFd++)
	{
	if (pRtFd->rfd_status == NOT_IN_USE)
	    {
	    pRtFd->rfd_status = OK;
	    semGive (rt11FdSemId);
	    return (pRtFd);
	    }
	}

    semGive (rt11FdSemId);
    errnoSet (S_rt11Lib_NO_FREE_FILE_DESCRIPTORS);

    return (NULL);
    }
/*******************************************************************************
*
* rt11GetEntry - get a directory entry from an RT-11 volume
*
* This routines gets the specified directory entry from the specified
* volume.
*
* Possession of the volume descriptor's semaphore must have been secured
* before this routine is called.
*
* RETURNS: a copy of the requested entry.
*/

LOCAL VOID rt11GetEntry (vdptr, entryNum, pEntry)
    RT_VOL_DESC *vdptr;	   /* pointer to volume descriptor	       */
    int entryNum;	   /* number of entry to get (first entry = 0) */
    RT_DIR_ENTRY *pEntry;  /* pointer where to return directory entry  */

    {
    bcopy ((char *) &(vdptr->vd_dir_seg->ds_entries[entryNum]),
	   (char *) pEntry, sizeof (RT_DIR_ENTRY));
    }
/*******************************************************************************
*
* rt11Init - prepare to use RT-11 library
*
* This routine initializes the RT-11 library.  It must be called exactly once,
* before any other routine in the library.  The argument specifies the number
* of RT-11 files that may be open at once.  This routine allocates and sets 
* up the necessary memory structures, initializes semaphores, etc.
*
* Normally this routine will be called from the root task, usrRoot (2),
* in usrConfig (1).
*
* RETURNS:
*    OK, or
*    ERROR if out of memory.
*/

STATUS rt11Init (maxFiles)
    int maxFiles;	/* maximum number of simultaneously	*/
                        /* open RT-11 files			*/

    {
    int i;

    /* initialize interlocking of file descriptor list access */

    rt11FdSemId = semCreate ();
    semGive (rt11FdSemId);

    rt11MaxFiles = maxFiles;
    rt11Fd = (RT_FILE_DESC *) malloc ((unsigned)
				      (rt11MaxFiles * sizeof (RT_FILE_DESC)));

    if (rt11Fd == (RT_FILE_DESC *) NULL)
	return (ERROR);

    for (i = 0; i < rt11MaxFiles; i++)
	rt11FreeFd (&rt11Fd[i]);

    return (OK);
    }
/*******************************************************************************
*
* rt11InsEntry - insert a new directory entry on an RT-11 volume
*
* This routines inserts the specified directory entry into the specified
* volume at the specified entry index.
*
* Possession of the volume descriptor's semaphore must have been secured
* before this routine is called.
*
* NOTE: Currently this routine does not handle the case of attempting
*	to add an entry to a full directory.
*/

LOCAL VOID rt11InsEntry (vdptr, entryNum, pEntry)
    RT_VOL_DESC *vdptr;	  /* pointer to volume descriptor		 */
    int entryNum;	  /* number of entry to insert (first entry = 0) */
    RT_DIR_ENTRY *pEntry; /* pointer to entry to insert			 */

    {
    RT_DIR_ENTRY previous_entry;
    RT_DIR_ENTRY entry;

    /* replace current entry at entryNum with new */

    rt11GetEntry (vdptr, entryNum, &entry);
    rt11PutEntry (vdptr, entryNum, pEntry);


    /* replace each entry with the previous */

    while (entry.de_status != DES_END)
	{
	previous_entry = entry;
	entryNum++;

	rt11GetEntry (vdptr, entryNum, &entry);
	rt11PutEntry (vdptr, entryNum, &previous_entry);
	}


    /* put end-marker entry in next slot */

    rt11PutEntry (vdptr, entryNum + 1, &entry);
    }
/*******************************************************************************
*
* rt11Ioctl - do device specific control function
*
* This routine performs the following ioctl (2) functions:
*
* .CS
*    FIODISKINIT:   Initialize the disk volume.  This routine does not
*                   format the disk, this must be done by the driver.
*    FIODIRENTRY:   Returns pointer to the next dir entry, into
*                   the REQ_DIR_ENTRY pointed to by arg.
*    FIORENAME:     Rename the file to the string pointed to by arg.
*    FIOSEEK:       Sets the file's current byte position to
*                   the position specified by arg.
*    FIOWHERE:      Returns the current byte position in the file.
*    FIOFLUSH:      Flush file output buffer.
*                   Guarantees that any output that has been requested
*                   is actually written to the device.
*    FIONREAD:      Return in arg the number of unread bytes.
*    FIODISKCHANGE: Indicate media change.  See rt11ReadyChange(2).
*    FIOSQUEEZE:    Reclaim fragmented free space on RT-11 volume
* .CE
*
* If an ioctl fails, the task's status (see errnoGet (2)) indicates
* the nature of the error.
*
* RETURNS:
*    OK, or
*    ERROR if function failed or unknown function, or
*    current byte pointer for FIOWHERE.
*/

STATUS rt11Ioctl (pFd, function, arg)
    RT_FILE_DESC *pFd;	/* file descriptor of file to control	*/
    int function;	/* function code			*/
    int	arg;		/* some argument			*/

    {
    int where;

    switch (function)
	{
	case FIODISKINIT:
	    return (rt11VolInit (pFd->rfd_vdptr));

	case FIODIRENTRY:
	    return (rt11DirEntry (pFd->rfd_vdptr, (REQ_DIR_ENTRY *)arg));

	case FIORENAME:
	    return (rt11Rename (pFd, (char *)arg));

	case FIOSEEK:
	    return (rt11Seek (pFd, arg));

	case FIOWHERE:
	    return (rt11Where (pFd));

	case FIOFLUSH:
	    return (rt11Flush (pFd));

	case FIONREAD:
	    if ((where = rt11Where (pFd)) == ERROR)
		return (ERROR);
	    *((int *) arg) = pFd->rfd_endptr - where;
	    return (OK);

	case FIODISKCHANGE:
	    rt11ReadyChange (pFd->rfd_vdptr);
	    return (OK);

	case FIOSQUEEZE:
	    return (rt11Squeeze (pFd->rfd_vdptr));

	default:
	    errnoSet (S_ioLib_UNKNOWN_REQUEST);
	    return (ERROR);
	}
    }
/*******************************************************************************
*
* rt11NameR50 - convert ASCII string to radix-50 RT-11 name
*
* This routine converts the specified ASCII string to an RT-11 name.
* The string is must be of the form "ffffff.ttt" and null terminated.
*
* If characters are in string that don't exist in the radix-50 world,
* undefined things will happen.
*/

LOCAL VOID rt11NameR50 (string, pName)
    char *string;	/* pointer to ASCII string to convert */
    RT_NAME *pName;	/* pointer where to return RT-11 radix-50 name struct */

    {
    char tstring [RT_NAME_LEN];
    FAST char *dot;

    /* get local copy of string and make sure it's terminated */

    strncpy (tstring, string, RT_NAME_LEN - 1);
    tstring [RT_NAME_LEN - 1] = EOS;


    /* find the dot, if any; convert extension; and replace dot w/EOS */

    dot = index (tstring, '.');

    if (dot == 0)
	pName->nm_type = 0;
    else
	{
	*dot = EOS;
	pName->nm_type = rt11R50out (dot + 1);
	}

    /* convert name part 1 then part 2 if more */

    pName->nm_name1 = rt11R50out (tstring);

    if (strlen (tstring) <= 3)
	pName->nm_name2 = 0;
    else
	pName->nm_name2 = rt11R50out (tstring + 3);
    }
/*******************************************************************************
*
* rt11R50out - convert up to 3 ASCII characters to radix-50
*
* RETURNS: radix-50 equivalent of first 3 chars of string.
*/

LOCAL int rt11R50out (string)
    FAST char *string;	/* string to convert */

    {
    unsigned int r50 = 0;
    int r;
    int i;
    char ch;

    for (i = 0; i < 3; i++)
	{
	if (*string == EOS)
	    r = 0;
	else
	    {
	    ch = *string;

	    if (isupper (ch))
		ch = tolower (ch);

	    r = (char *) index (rad50, ch) - rad50;
	    string++;
	    }

	r50 = (r50 * 050) + r;
	}

    return (r50);
    }
/*******************************************************************************
*
* rt11NameString - convert radix-50 RT-11 name to ASCII string
*
* This routine converts the specified RT-11 name into an ASCII string
* of the form "ffffff.ttt" and null terminated.
*/

LOCAL VOID rt11NameString (name, string)
    RT_NAME name;	/* RT-11 radix-50 name structure	*/
    FAST char *string;	/* pointer to receive ASCII string	*/

    {
    FAST char *pStr;	/* index into string */

    rt11R50in (name.nm_name1, string);
    rt11R50in (name.nm_name2, string + 3);

    for (pStr = string; (pStr < (string + 6)) && (*pStr != ' '); ++pStr)
	;		/* prepare to overwrite trailing blanks */

    *pStr++ = '.';

    rt11R50in (name.nm_type, pStr);

    *(pStr + 3) = EOS;
    }
/*******************************************************************************
*
* rt11R50in - convert radix-50 to 3 ASCII characters
*/

LOCAL VOID rt11R50in (r50, string)
    unsigned int r50;	/* radix-50 number to convert	 */
    char *string;	/* where to put resulting string */

    {
    FAST int i;

    for (i = 2; i >= 0; i--)
	{
	string [i] = rad50 [r50 % 050];

	r50 /= 050;
	}
    }
/*******************************************************************************
*
* rt11Open - open a file on an RT-11 volume
*
* This routine opens the file `name' with the specified mode.
* The volume directory is searched, and if the file is found
* an RT-11 file descriptor is initialized for it.
*
* The directory currently in memory for the volume is used unless
* there has been a ready change (rt11ReadyChange (2)) or this is the very first
* open. If that is the case, the directory will be read from the disk
* automatically.
*
* A null file name is treated specially, to open an entire disk.
* In this case, no attempt is made to access the disk's directory, so that
* even un-initialized disks may be accessed.
*
* RETURNS:
*    Pointer to RT-11 file descriptor, or
*    NULL if volume not available or
*	     no more RT-11 fd's available or
*            no such file.
*/

RT_FILE_DESC *rt11Open (vdptr, name, mode)
    RT_VOL_DESC *vdptr;	/* pointer to volume descriptor		*/
    char *name;		/* RT-11 filename (ffffff.ttt)		*/
    int mode;		/* file mode (READ/WRITE/UPDATE)	*/

    {
    int start;	
    int entryNum;
    FAST RT_FILE_DESC *pFd = rt11GetFd ();	/* file descriptor pointer */

    /* get a free file descriptor */

    if (pFd == NULL)
	return (NULL);

    /* check for open of raw device (null filename) */

    if (name [0] == EOS)
	{
	/* check that volume is available */

	semTake (vdptr->vd_semId);		/* get ownership of volume */

	if (rt11CheckVol (vdptr, FALSE) < OK)
	    {
	    semGive (vdptr->vd_semId);		
	    rt11FreeFd (pFd);
	    errnoSet (S_rt11Lib_VOLUME_NOT_AVAILABLE);
	    return (NULL);
	    }

	if (rt11VolMode (vdptr) < mode)
	    {
	    semGive (vdptr->vd_semId);		
	    rt11FreeFd (pFd);
	    errnoSet (S_ioLib_WRITE_PROTECTED);
	    return (NULL);
	    }

	semGive (vdptr->vd_semId);		

	/* null name is special indicator for "raw" disk;
	 * fabricate a bogus directory entry covering entire volume */

	start = 0;

	pFd->rfd_dir_entry.de_status  = DES_BOGUS;
	pFd->rfd_dir_entry.de_date    = rt11Date (rt11Year, rt11Month, rt11Day);
	pFd->rfd_dir_entry.de_nblocks = vdptr->vd_nblocks;
	rt11NameR50 ("device.raw", &pFd->rfd_dir_entry.de_name);
	}
    else
	{

	semTake (vdptr->vd_semId);		/* get ownership of volume */

	if(rt11SlashRemove (name) == ERROR) {
	    semGive (vdptr->vd_semId);		
	    rt11FreeFd (pFd);
	    return(NULL);
	}

	/* check that volume is available */

	if ((rt11CheckVol (vdptr, TRUE) < OK) || (vdptr->vd_status < OK))
	    {
	    semGive (vdptr->vd_semId);		
	    rt11FreeFd (pFd);
	    errnoSet (S_rt11Lib_VOLUME_NOT_AVAILABLE);
	    return (NULL);
	    }

	if (rt11VolMode (vdptr) < mode)
	    {
	    semGive (vdptr->vd_semId);		
	    rt11FreeFd (pFd);
	    errnoSet (S_ioLib_WRITE_PROTECTED);
	    return (NULL);
	    }

	/* find specified file name in directory */

	if ((entryNum = rt11FindEntry (vdptr, name, &start)) == ERROR)
	    {
	    semGive (vdptr->vd_semId);			/* release volume */
	    rt11FreeFd (pFd);
	    return (NULL);
	    }

	rt11GetEntry (vdptr, entryNum, &pFd->rfd_dir_entry);
	semGive (vdptr->vd_semId);			/* release volume */
	}

    /* initialize file descriptor */

    pFd->rfd_mode	= mode;
    pFd->rfd_vdptr	= vdptr;
    pFd->rfd_start	= start;
    pFd->rfd_curptr	= NONE;
    pFd->rfd_newptr	= 0;
    pFd->rfd_endptr	= pFd->rfd_dir_entry.de_nblocks * RT_BYTES_PER_BLOCK;
    pFd->rfd_modified	= FALSE;

    return (pFd);
    }
/*******************************************************************************
*
* rt11PutEntry - put directory entry of RT-11 file
*
* This routines puts the specified directory entry of the specified
* volume.
*
* Posession of the volume descriptor's semaphore must have been secured
* before this routine is called.
*/

LOCAL VOID rt11PutEntry (vdptr, entryNum, pEntry)
    RT_VOL_DESC *vdptr;		/* pointer to volume descriptor		    */
    int entryNum;		/* number of entry to get (first entry = 0) */
    RT_DIR_ENTRY *pEntry;	/* pointer to entry to put		    */

    {
    vdptr->vd_dir_seg->ds_entries [entryNum] = *pEntry;
    }
/*******************************************************************************
*
* rt11Read - read from an RT-11 file
*
* This routine reads from the file specified by the file descriptor
* (returned by rt11Open (2)) into the specified buffer.
* Maxbytes bytes will be read, if there is that much data in the file.
*
* RETURNS:
*    Number of bytes read, or
*    0 if end of file, or
*    ERROR if maxbytes <= 0 or unable to get next block.
*
*/

int rt11Read (pfd, buffer, maxbytes)
    FAST RT_FILE_DESC *pfd;	/* file descriptor pointer		*/
    char *buffer;		/* address to receive sector		*/
    int maxbytes;		/* maximum bytes to read into buffer	*/

    {
    int buf_index;
    STATUS status;
    FAST int nbytes;

    /* check for valid maxbytes */

    if (maxbytes <= 0)
	{
	errnoSet (S_rt11Lib_INVALID_NUMBER_OF_BYTES);
	return (ERROR);
	}

    /* get new pointer in buffer */

    if ((status = rt11NewBlock (pfd)) <= 0)
	return (status);	/* 0 is end-of-file */

    /* fill all of caller's buffer or rest of buffer, whichever is less. */

    buf_index = pfd->rfd_newptr % RT_BYTES_PER_BLOCK;

    nbytes = min (maxbytes, RT_BYTES_PER_BLOCK - buf_index);

    bcopy (&pfd->rfd_buffer [buf_index], buffer, nbytes);

    /* update new pointer and return number of bytes read */

    pfd->rfd_newptr += nbytes;

    return (nbytes);
    }
/*******************************************************************************
*
* rt11Rename - change name of RT-11 file
*
* This routine changes the name of the specified stream to the specified
* new name.
*
* RETURNS: 
*    OK, or
*    ERROR if newName already in use, or
*          unable to write out new directory info.
*/

LOCAL STATUS rt11Rename (pfd, newName)
    RT_FILE_DESC *pfd;		/* pointer to file descriptor	*/
    char *newName;		/* change name of file to this	*/

    {
    char oldName [RT_NAME_LEN];	/* current file name gets unpacked into here */
    char *tail;			/* tail of newName */
    int entryNum;		/* gets index into directory of oldname      */
    int start;			/* for receiving side-effect of rt11FindEntry*/
    RT_DIR_ENTRY entry;		/* directory entry of oldname goes here      */

    /* ensure newName doesn't include device name */

    if (iosDevFind (newName, &tail) != NULL)
	strcpy (newName, tail);

    semTake (pfd->rfd_vdptr->vd_semId);		/* get ownership of volume   */

    /* be sure new name isn't already in use */

    if (rt11FindEntry (pfd->rfd_vdptr, newName, &start) != ERROR)
	{
	semGive (pfd->rfd_vdptr->vd_semId);	/* release volume */
	errnoSet (S_rt11Lib_FILE_ALREADY_EXISTS);
	return (ERROR);				/* entry already exists */
	}

    /* unpack name and find entry number */

    rt11NameString (pfd->rfd_dir_entry.de_name, oldName);
    if ((entryNum = rt11FindEntry (pfd->rfd_vdptr, oldName, &start)) == ERROR)
	{
	semGive (pfd->rfd_vdptr->vd_semId);		/* release volume */
	return (ERROR);					/* entry not found */
	}

    rt11GetEntry (pfd->rfd_vdptr, entryNum, &entry);
    rt11NameR50 (newName, &entry.de_name);		/* put in new name */
    rt11NameR50 (newName, &pfd->rfd_dir_entry.de_name);	/* rename fd also */
    rt11PutEntry (pfd->rfd_vdptr, entryNum, &entry);

    /* make sure directory is written out */

    if (rt11VolFlush (pfd->rfd_vdptr) < OK)
	{
	semGive (pfd->rfd_vdptr->vd_semId);		/* release volume */
	return (ERROR);					/* entry not found */
	}

    semGive (pfd->rfd_vdptr->vd_semId);		/* release volume */
    return (OK);
    }
/*******************************************************************************
*
* rt11Seek - change file's current character position
*
* This routine sets the specified file's current character position to
* the specified position. This only changes the pointer, doesn't affect
* the hardware at all.
*
* RETURNS: OK | ERROR
*/

LOCAL STATUS rt11Seek (pfd, position)
    FAST RT_FILE_DESC *pfd;	/* file descriptor pointer		*/
    int position;		/* desired character position in file	*/

    {
    if (pfd->rfd_status == NOT_IN_USE)
	{
	errnoSet (S_rt11Lib_FILE_NOT_FOUND);
	return (ERROR);
	}

    if (position > (pfd->rfd_dir_entry.de_nblocks * RT_BYTES_PER_BLOCK))
	{
	errnoSet (S_rt11Lib_BEYOND_FILE_LIMIT);
	return (ERROR);
	}

    /* update new file byte pointer; also end pointer if file just grew */

    pfd->rfd_newptr = position;

    if (pfd->rfd_endptr < position)
	pfd->rfd_endptr = position;

    return (OK);
    }
/*******************************************************************************
*
* rt11DateSet - set current date
*
* This routine sets the RT-11 file system's current date.  
* All files created will have this creation date.
*
* To set a blank date, do:
* .CS
*	rt11DateSet (72, 0, 0);	/* a date outside RT-11's epoch *
* .CE
*/

VOID rt11DateSet (year, month, day)
    int year;		/* year (72...03 (RT-11's days are numbered)) */
    int month;		/* month (0, or 1...12) */
    int day;		/* day (0, or 1...31) */

    {
    rt11Year  = year % 100;
    rt11Month = month;
    rt11Day   = day;
    }
/*******************************************************************************
*
* rt11VolFlush - flush RT-11 volume directory
*
* This routine guarantees that any changes to the volume directory entry
* are actually written to disk.  
*
* Posession of the volume descriptor's semaphore must have been secured
* before this routine is called.
*
* RETURNS: OK | ERROR
*/

LOCAL STATUS rt11VolFlush (vdptr)
    FAST RT_VOL_DESC *vdptr;	/* pointer to volume descriptor */

    {
    FAST int i;
    char *psegTop = (char *) vdptr->vd_dir_seg;	/* top of directory segment */
    FAST char *pseg = psegTop;
    FAST int nSegBytes = vdptr->vd_nSegBlocks * RT_BYTES_PER_BLOCK;

    vdptr->vd_status = OK;

    swab (psegTop, psegTop, nSegBytes);	/* swap bytes in entire segment */

    /* flush each block */

    for (i = 0; i < vdptr->vd_nSegBlocks; i++)
	{
    	if (rt11WrtBlock (vdptr, RT_DIR_BLOCK + i, pseg) != OK)
	    {
	    vdptr->vd_status = ERROR;
	    break;
	    }

	pseg += RT_BYTES_PER_BLOCK;
	}

    swab (psegTop, psegTop, nSegBytes);	/* swap bytes in segment back */

    return (vdptr->vd_status);
    }
/*******************************************************************************
*
* rt11VolInit - initialize an RT-11 volume
*
* This routine initializes a disk volume to be an RT-11 disk.  The volume
* structures in memory are initialized, the volume's in-memory directory
* is initialized, and then the empty directory is flushed to the disk
* itself, by calling rt11VolFlush.
*
* The volume descriptor itself should have already been initialized with
* rt11DevInit.
*
* This routine does not format the disk; that must have already been done.
*
* RETURNS:
*    OK, or
*    ERROR if can't flush to device.
*/

LOCAL STATUS rt11VolInit (vdptr)
    RT_VOL_DESC *vdptr;		/* pointer to volume descriptor */

    {
    STATUS status;
    FAST int freeBlocks;	/* number of free file blocks left */
    FAST RT_DIR_SEG *pseg = vdptr->vd_dir_seg;
    FAST int i = 0;

    semTake (vdptr->vd_semId);		/* get ownership of volume */

    /* initialize directory segment */

    pseg->ds_nsegs	= 1;
    pseg->ds_next_seg	= 0;
    pseg->ds_last_seg	= 1;
    pseg->ds_extra	= 0;
    
    /* first data block starts after the directory segment */

    pseg->ds_start	= RT_DIR_BLOCK + vdptr->vd_nSegBlocks;

    freeBlocks = vdptr->vd_nblocks - pseg->ds_start;

    /* make first directory entries be EMPTY covering entire remaining volume */
 
    /* first, make all files of size RT_MAX_BLOCKS_PER_FILE */

    while (freeBlocks > RT_MAX_BLOCKS_PER_FILE)
	{
	pseg->ds_entries[i].de_status = DES_EMPTY;
	pseg->ds_entries[i++].de_nblocks = RT_MAX_BLOCKS_PER_FILE;
	freeBlocks -= RT_MAX_BLOCKS_PER_FILE;
	}

    /* make last empty directory entry */

    if (freeBlocks > 0)
	{
	pseg->ds_entries[i].de_status = DES_EMPTY;
	pseg->ds_entries[i++].de_nblocks = freeBlocks;
	}
    
    /* make terminating entry */

    pseg->ds_entries[i].de_status = DES_END;

    status = rt11VolFlush (vdptr);

    if (status != OK)
	vdptr->vd_status = ERROR;

    semGive (vdptr->vd_semId);			/* release volume */
    return (status);
    }
/*******************************************************************************
*
* rt11VolMount - prepare to use RT-11 volume
*
* This routine prepares the library to use the RT-11 volume on the device
* specified.  The volume directory segment is read from the disk.
* This routine should be called every time a disk is changed (i.e. a floppy
* is swapped, or whatever), or before every open and create, if the driver
* can't tell when disks are changed.
*
* Eventually this routine could be taught to read the home 
* block to get more initialization info.
*
* Note:  If an error occurs while reading, it's possible that the
*	 bytes in the directory segment will be swapped.
*
* RETURNS: OK | ERROR
*/

LOCAL STATUS rt11VolMount (vdptr)
    FAST RT_VOL_DESC *vdptr;		/* pointer to volume descriptor */

    {
    FAST int  i;
    FAST char *pseg = (char *) vdptr->vd_dir_seg;

    vdptr->vd_status = ERROR;

    /* read in all blocks of directory segment */

    for (i = 0; i < vdptr->vd_nSegBlocks; i++)
	{
	if (rt11RdBlock (vdptr, RT_DIR_BLOCK + i, pseg) != OK)
	    return (vdptr->vd_status);

	pseg += RT_BYTES_PER_BLOCK;
	}

    pseg = (char *) vdptr->vd_dir_seg;

    swab (pseg, pseg, vdptr->vd_nSegBlocks * RT_BYTES_PER_BLOCK);

    /* check reasonableness of directory segment */

    if (vdptr->vd_dir_seg->ds_nsegs    == 1	&&
	vdptr->vd_dir_seg->ds_next_seg == 0	&&
	vdptr->vd_dir_seg->ds_last_seg == 1	&&
	vdptr->vd_dir_seg->ds_extra    == 0)
	{
	vdptr->vd_status = OK;
	}
    
    return (vdptr->vd_status);
    }
/*******************************************************************************
*
* rt11Where - tell where file is (current character position)
*
* This routine tells you where the file pointer is for a file.
*
* RETURNS:
*    Current character position of file, or
*    ERROR if invalid file descriptor.
*/

LOCAL int rt11Where (pfd)
    FAST RT_FILE_DESC *pfd;	/* file descriptor pointer */

    {
    if (pfd->rfd_status == NOT_IN_USE)
	{
	errnoSet (S_rt11Lib_FILE_NOT_FOUND);
	return (ERROR);
	}
    else
	return (pfd->rfd_newptr);
    }
/*******************************************************************************
*
* rt11Write - write to an RT-11 file
*
* This routine writes to the file specified by the file descriptor
* (returned by rt11Open (2)) from the specified buffer.  If the block
* containing the disk locations to be written is already in memory
* the block won't be flushed.  If another in-memory block is needed,
* any block already in memory will be flushed.
*
* RETURNS:
*    Number of bytes written (error if != nbytes), or
*    ERROR if nbytes < 0, or no more space for the file,
*		or can't write block.
*/

int rt11Write (pfd, buffer, nbytes)
    FAST RT_FILE_DESC *pfd;	/* file descriptor pointer		*/
    char *buffer;		/* data to be written			*/
    int nbytes;			/* number of bytes to write from buffer */

    {
    FAST int bytes_copied;
    FAST int buf_index;
    int bytes_left = nbytes;

    /* check for valid nbytes */

    if (nbytes < 0)
	{
	errnoSet (S_rt11Lib_INVALID_NUMBER_OF_BYTES);
	return (ERROR);
	}

    /* read into successive blocks until all of caller's buffer written */

    while (bytes_left > 0)
	{
	/* get new pointer in buffer */

	if (rt11NewBlock (pfd) <= 0)
	    return (ERROR);

	/* copy length = rest of caller's buffer or rest of fd buffer,
	 *   whichever is less.
	 */

	buf_index = pfd->rfd_newptr % RT_BYTES_PER_BLOCK;

	bytes_copied = min (bytes_left, RT_BYTES_PER_BLOCK - buf_index);

	bcopy (buffer, &pfd->rfd_buffer [buf_index], bytes_copied);

	pfd->rfd_modified = TRUE;

	/* update pointers and bytes remaining */

	pfd->rfd_newptr += bytes_copied;

	if (pfd->rfd_endptr < pfd->rfd_newptr)
	    pfd->rfd_endptr = pfd->rfd_newptr;

	buffer += bytes_copied;
	bytes_left -= bytes_copied;
	}

    return (nbytes);
    }

/*******************************************************************************
*
* rt11NewBlock - make file descriptor block buffer contain new pointer
*
* This routine does whatever is necessary to make the block buffer
* in the specified file descriptor contain the byte addressed by
* the current pointer in the descriptor.  In particular, if on entry
* the buffer already contains the desired byte, then no action is taken.
* Otherwise if the buffer is modified (contains data written by user
* but not yet on disk) then the block is written.  Then the correct
* block is read if the mode is READ or UPDATE.
*
* RETURNS:
*    number of bytes in buffer if successful, or
*    0 if end of file, or
*    ERROR if unable to read/write block
*/

LOCAL int rt11NewBlock (pfd)
    FAST RT_FILE_DESC *pfd;	/* file descriptor pointer */

    {
    FAST int new_block;
    int cur_block;

    /* calculate block num of desired new block and of current block if any */

    new_block = pfd->rfd_newptr / RT_BYTES_PER_BLOCK;

    cur_block = (pfd->rfd_curptr == NONE) ?
    			NONE : (pfd->rfd_curptr / RT_BYTES_PER_BLOCK);


    /* check for new block already in current buffer */

    if (new_block == cur_block)
	return (RT_BYTES_PER_BLOCK);


    /* check for end of file */

    if (new_block >= pfd->rfd_dir_entry.de_nblocks)
	return (0);


    /* flush current block, read in new block, update current pointer */

    if (rt11Flush (pfd) < OK)
	return (ERROR);

    if (pfd->rfd_mode != WRITE)
	{
	if (rt11RdBlock (pfd->rfd_vdptr, new_block + pfd->rfd_start,
		         pfd->rfd_buffer) < OK)
	    return (ERROR);
	}

    pfd->rfd_curptr = new_block * RT_BYTES_PER_BLOCK;

    return (RT_BYTES_PER_BLOCK);
    }
/*******************************************************************************
*
* rt11RdBlock - read a block from an RT-11 volume
*
* This routine reads the specified block from the specified volume.
*
* RETURNS: OK | ERROR
*/

LOCAL STATUS rt11RdBlock (vdptr, block_num, buffer)
    FAST RT_VOL_DESC *vdptr;	/* pointer to volume descriptor */
    int block_num;		/* number of block to read */
    FAST char *buffer;		/* buffer to receive block */

    {
    FAST int i;
    int phys_sector;
    FAST int sector_num = block_num * vdptr->vd_secBlock;

    /* read each sector in block */

    for (i = 0; i < vdptr->vd_secBlock; ++i)
	{
	phys_sector = vdptr->vd_rtFmt ?
			    rt11AbsSector (vdptr->vd_secTrack, sector_num) :
			    sector_num;

	vdptr->vd_retry = 0;

	while (((*vdptr->vd_rdSec) (vdptr, phys_sector, buffer)) != OK) 
	    {
	    /* read error: reset volume and retry, up to retry limit */

	    if (rt11Reset (vdptr) != OK)
		return (ERROR);		/* drive won't event reset! */

	    if (++(vdptr->vd_retry) >= rt11MaxRetries)
		return (ERROR);
	    }

	sector_num++;
	buffer += vdptr->vd_bytesPerSec;
	}

    return (OK);
    }
/*******************************************************************************
*
* rt11WrtBlock - write a block to an RT-11 volume
*
* This routine writes the specified block to the specified volume.
*
* RETURNS: OK | ERROR
*/

LOCAL STATUS rt11WrtBlock (vdptr, block_num, buffer)
    FAST RT_VOL_DESC *vdptr;	/* pointer to volume descriptor */
    int block_num;		/* number of block to write */
    char *buffer;		/* buffer to write to block */

    {
    FAST int i;
    int phys_sector;
    FAST int sector_num = block_num * vdptr->vd_secBlock;

    /* write each sector in block */

    for (i = 0; i < vdptr->vd_secBlock; ++i)
	{
	phys_sector = vdptr->vd_rtFmt ?
			    rt11AbsSector (vdptr->vd_secTrack, sector_num) :
			    sector_num;

	vdptr->vd_retry = 0;

	while (((* vdptr->vd_wrtSec) (vdptr, phys_sector, buffer)) != OK)
	    {
	    /* write error: reset volume and retry, up to retry limit */

	    if (rt11Reset (vdptr) != OK)
		return (ERROR);		/* drive won't event reset! */
		
	    if (++(vdptr->vd_retry) >= rt11MaxRetries)
		return (ERROR);		/* retry limit reached */
	    }

	sector_num++;
	buffer += vdptr->vd_bytesPerSec;
	}

    return (OK);
    }
/*******************************************************************************
*
* rt11AbsSector - determine absolute sector of logical sector
*
* This routine calculates the absolute sector number from the beginning of a
* medium using the RT-11 2:1 software interleave and 6 sector
* track-to-track skew.
*
* RETURNS: absolute sector number of specified logical sector number.
*/

LOCAL int rt11AbsSector (secPerTrack, sector)
    FAST int secPerTrack;	/* sectors per track */
    FAST int sector;		/* logical sector number
				 * to calculate */

    {
    int physical_sector;
    FAST int track = sector / secPerTrack;
    FAST int trk_sector	= sector % secPerTrack;

    /* convert to actual offset by interleave and skew factors */

    physical_sector = ((trk_sector * 2) + (track * 6)) % secPerTrack;

    if (trk_sector >= (secPerTrack / 2))
	physical_sector++;


    /* calculate absolute sector from start of disk */

    return (track * secPerTrack + physical_sector);
    }
/*******************************************************************************
*
* rt11CheckVol - verify that volume descriptor is current
*
* RETURNS: OK | ERROR
*/

LOCAL STATUS rt11CheckVol (vdptr, doMount)
    FAST RT_VOL_DESC *vdptr;	/* pointer to device descriptor   */
    BOOL doMount;		/* mount the device, if necessary */

    {
    switch (vdptr->vd_state)
	{
	case RT_VD_CANT_RESET:
	    /* this state means we tried to reset and failed so we
	     * don't try again until a ready change occurs */

	    return (ERROR);	/* can't reset */

	case RT_VD_CANT_MOUNT:
	    /* this state means we tried to mount and failed so we
	     * don't try again until a ready change occurs */

	    if (doMount)
		return (ERROR);	/* can't mount */

	    break;		/* caller didn't want a mount anyway */
	    

	case RT_VD_READY_CHANGED:
	    /* ready change occurred; try to reset volume */

	    if (rt11Reset (vdptr) != OK)
		return (ERROR);				/* can't reset */

	    vdptr->vd_state = RT_VD_RESET;		/* note volume reset */

	    /* fall-through to try to mount */

	case RT_VD_RESET:
	    /* volume reset but not mounted; if caller requested, try mount */

	    if (doMount)
		{
		if (rt11VolMount (vdptr) < OK)
		    {
		    vdptr->vd_state = RT_VD_CANT_MOUNT;	/* don't try again */
		    return (ERROR);			/* can't mount */
		    }

		vdptr->vd_state = RT_VD_MOUNTED;	/* note vol mounted */
		}
	    
	    break;


	case RT_VD_MOUNTED:
	    break;
	}

    return (OK);	/* caller got what he wanted */
    }
/*******************************************************************************
*
* rt11Reset - reset a volume
*
* This routine calls the specified volume's reset routine.
* If the reset fails, vd_state is set to (RT_VD_CANT_RESET)
* to indicate that disk won't reset.
*/

LOCAL STATUS rt11Reset (vdptr)
    FAST RT_VOL_DESC *vdptr;	/* pointer to device descriptor */

    {
    if (vdptr->vd_reset != NULL && (vdptr->vd_reset) (vdptr) != OK)
	{
	vdptr->vd_state = RT_VD_CANT_RESET;
	return (ERROR);
	}

    return (OK);
    }
/*******************************************************************************
*
* rt11VolMode - inquire the current volume mode (READ/WRITE/UPDATE)
*/

LOCAL STATUS rt11VolMode (vdptr)
    FAST RT_VOL_DESC *vdptr;	/* pointer to device descriptor */

    {
    return (vdptr->vd_mode);
    }
/*******************************************************************************
*
* rt11ReadyChange - notify rt11Lib of a change in ready status
*
* This routine sets the volume descriptor's state to RT_VD_READY_CHANGED.
* It should be called whenever a driver senses that a device has come on-line
* or gone off-line, e.g. a disk has been inserted or removed.
*/

VOID rt11ReadyChange (vdptr)
    RT_VOL_DESC *vdptr;	/* pointer to device descriptor */

    {
    vdptr->vd_state = RT_VD_READY_CHANGED;
    }
/*******************************************************************************
*
* rt11ModeChange - modify mode of RT-11 volume
*
* This routine sets the volume descriptor's mode to newMode.
* It should be called whenever the read and write capabilities are determined,
* usually after a ready change.  See rt11ReadyChange(2).
*
* The rt11DevInit(2) routine initially sets the mode to UPDATE,
* ie. both READ and WRITE.
*/

VOID rt11ModeChange (vdptr, newMode)
    RT_VOL_DESC *vdptr;	/* pointer to volume descriptor */
    int newMode;	/* READ/WRITE/UPDATE (both)	*/ 

    {
    vdptr->vd_mode = newMode;
    }
/*******************************************************************************
*
* rt11Squeeze - reclaim fragmented free space on RT-11 volume
*
* This routine moves data around on an RT-11 volume so that all the
* little bits of free space left on the device are joined together.
*
* CAVEAT:
* There MUST NOT be any files open on the device when this procedure
* is called - if there are then their condition after the call will
* be unknown - almost certainly damaged, and writing to a file may
* destroy the entire disk.
*/

LOCAL STATUS rt11Squeeze (vdptr)
    FAST RT_VOL_DESC *vdptr;	/* pointer to volume descriptor */

    {
#define	NUM_BLOCKS	16

    FAST int rdptr;		/* number of next directory entry to read */
    FAST int wdptr;		/* number of next directory entry to write */
    FAST int rbptr;		/* number of next block to read */
    FAST int wbptr;		/* number of next block to write */
    FAST int i;			/* ever present!! */
    int nblk;			/* number of blocks in file */
    int numToRead;		/* number of blocks to transfer in this pass */
    int mtcount;		/* number of empty blocks on device */
    RT_DIR_ENTRY dirEntry;	/* directory entry for current file */
    FAST char *buffer = malloc (RT_BYTES_PER_BLOCK * NUM_BLOCKS);
    
    if (buffer == NULL)
	return (ERROR);		/* oops - couldn't do it */
    
    semTake (vdptr->vd_semId);		/* get ownership of volume */

    rdptr = wdptr = 0;		/* start at first directory entry */
    rbptr = wbptr = vdptr->vd_dir_seg->ds_start;
				/* blocks start at first one after directory */

    if ((rt11CheckVol (vdptr, TRUE) < OK) || (vdptr->vd_status < OK))
	{
	semGive (vdptr->vd_semId);
	errnoSet (S_rt11Lib_VOLUME_NOT_AVAILABLE);
	free (buffer);
	return (ERROR);
	}

    if (rt11VolMode (vdptr) == READ)	/* check it's not write protected */
	{
	semGive (vdptr->vd_semId);		
	errnoSet (S_ioLib_WRITE_PROTECTED);
	free (buffer);
	return (ERROR);
	}

    mtcount = 0;			/* initially no empty blocks */

    /* now loop through the entire directory till we hit the end */

    while (rt11GetEntry (vdptr, rdptr++, &dirEntry),
	   dirEntry.de_status != DES_END)
	{
	if (dirEntry.de_status == DES_PERMANENT)
	    {				/* we got an active file */
	    rt11PutEntry (vdptr, wdptr++, &dirEntry);
					/* write it back */
	    nblk = dirEntry.de_nblocks;

	    while (nblk)		/* loop till all blocks transferred */
		{
		numToRead = min (nblk, NUM_BLOCKS);
		nblk -= numToRead;	/* read full buffer or file size */

		if (rbptr != wbptr)
		    {
		    /* only move if it's at a different place */

		    for (i = 0; i < numToRead; i++)
			{
			/* read another buffers worth */
			rt11RdBlock (vdptr, rbptr++,
				     &buffer [i * RT_BYTES_PER_BLOCK]);
			}

		    for (i = 0; i < numToRead; i++)
			{
			/* write it back */
			rt11WrtBlock (vdptr, wbptr++,
				      &buffer [i * RT_BYTES_PER_BLOCK]);
			}
		    }
		else
		    {
		    /* same place on disk, don't move it */

		    rbptr += numToRead;
		    wbptr += numToRead;	/* just update the pointers */
		    }
		}
	    }
	else
	    {
	    /* not active: just update block read pointer & empty block count */

	    rbptr += dirEntry.de_nblocks;
	    mtcount += dirEntry.de_nblocks;
	    }
	}

    dirEntry.de_status = DES_EMPTY;
    dirEntry.de_nblocks = mtcount;		/* set count of empty blocks */
    rt11PutEntry (vdptr, wdptr++, &dirEntry);	/* write DES_EMPTY entry */

    dirEntry.de_status = DES_END;
    rt11PutEntry(vdptr, wdptr, &dirEntry);	/* write DES_END to terminate */
    rt11VolFlush(vdptr);			/* flush volume */

    semGive (vdptr->vd_semId);			/* and release it */
    free (buffer);				/* and release the memory */

    return (OK);
    }

int rt11SlashRemove(name)
char *name;
{
    char *tmpnameptr;

    tmpnameptr = malloc(strlen(name));
    if(name[0] == '/') {
        strcpy(tmpnameptr,&(name[1]));
        strcpy(name, tmpnameptr);
    }
    if(index(name,'/') != 0){
	free(tmpnameptr);
	return(ERROR);
    }
    free(tmpnameptr);
    return(OK);
}
