/* logLib.c - message logging library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
02g,07jul88,jcf  made logShow global but NOMANUAL, so lint would let up.
02f,22jun88,dnw  name tweaks.
02e,06jun88,dnw  changed taskSpawn/taskCreate args.
02d,30may88,dnw  changed to v4 names.
02c,23apr88,jcf  changed semaphores for new semLib.
02b,05apr88,gae  changed fprintf() to fdprintf().  Added debug rtn logShow().
02a,26jan88,jcf  made kernel independent.
01q,14dec87,gae  changed logMsg() to not have multiple argument decl's for
		   parsing by the C interface generator.
01p,16nov87,ecs  documentation.
01o,03nov87,dnw  changed to let system pick logTask id;
		   now available in logTaskId.
01n,14oct87,gae  added log{Add,Del}Fd(), allowing multiple logging fds.
		 removed NOT_GENERIC stuff.
01m,24mar87,jlf  documentation
01l,21dec86,dnw  changed to not get include files from default directories.
01k,31jul86,llk  uses new spawn
01j,09jul86,dnw  restored NOT_GENERIC stuff that got lost in v01i.
01i,01jul86,jlf  minor documentation
01h,09apr86,dnw  added call to vxSetTaskBreakable to make logTask unbreakable.
		 fixed documentation of logMsg().
01g,07apr86,dnw  increased logTsk stack from 1000 to 2000.
		 corrected errors in documentation.
		 added logSetFd().
01f,11mar86,jlf  changed GENERIC to NOT_GENERIC
01e,20jul85,jlf  documentation.
01d,04jan85,ecs  added an ARGSUSED to logMsg, got rid of 'housetime'.
01c,19sep84,jlf  cleaned up comments a little
01b,07sep84,jlf  added copyright notice and comments.
01a,28aug84,dnw  culled from fioLib and enhanced.
*/

/*
DESCRIPTION
This module handles message logging.  Its normal use is to log error
messages on the system console, though the messages could just as well
go to a disk file, a printer, or whatever.

There are two basic components to the logging system.  The first is the
logging task (logTask).  This task reads a pipe for messages containing
a printf-style format string and arguments.  When it gets a message,
it formats it, adds some information to it, and writes it on the file
descriptor that was
specified on the logInit (or subsequently set by logFdSet or logFdAdd).
The second main component is logMsg.  This routine has exactly the same calling
sequence as printf, but sends the format string and arguments in a message
to logTask.

There are two important things to note about logging.
One is that logMsg can be called from interrupt level,
as well as from task level, whereas normal I/O to a non-pipe device
(printf to a serial port, for example)
can't be done from interrupt level.  
The other is that the print formatting is done in logTask's context, 
rather than the context of the task calling logMsg.  
Since formatting can require considerable stack space, 
this can reduce stack sizes for tasks that only need to do I/O for error output.

SEE ALSO: "I/O System", fioLib (1), pipeDrv (3)
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "ioLib.h"
#include "taskLib.h"
#include "semLib.h"

#define MAX_ARGS	6	/* max args to log message */
#define MAX_LOGFDS	5	/* max log fds */

typedef struct		/* LOGMSG */
    {
    int id;
    char *fmt;
    int args [MAX_ARGS];
    } LOGMSG;

/* logTask parameters */

int logTaskId;
int logTaskPriority	= 0;
int logTaskOptions	= VX_SUPERVISOR_MODE | VX_UNBREAKABLE;
int logTaskStackSize	= 2000;

/* local variables */

LOCAL SEM_ID logFdSemId;	/* semaphore for accessing logFd's */
LOCAL int logFd [MAX_LOGFDS];	/* output fd's used for logging */
LOCAL int logPipe = -1;		/* fd of pipe to log task */
LOCAL int numLogFds = 0;	/* number of active logging fd's */
LOCAL int logMsgsLost = 0;	/* count of number of log messages lost */

/* forward declarations */

VOID logTask ();


/*******************************************************************************
*
* logInit - initialize message logging library
*
* This routine must be called before any other routine in logLib (1).
* Normally this is done from the root task, in usrConfig.c (1).
*
* With this call, the file descriptor to the logging device is specified,
* and the number of messages that may be in the logging pipe.
* If more than this number of messages are in the pipe, they will be discarded.
* A message is printed to indicate lost messages.
*
* This routine spawns logTask (2), the task level portion of error logging.
*
* RETURNS
*  OK,   or
*  ERROR if couldn't create pipe, or couldn't spawn logTask (2).
*/

STATUS logInit (fd, maxMsgs)
    int fd;		/* file descriptor to use as logging device */
    int maxMsgs;	/* maximum number of messages allowed in *
			 * the log pipe */

    {
    if ((pipeDevCreate ("/pipe/log", maxMsgs, sizeof (LOGMSG)) != OK) ||
        ((logPipe = open ("/pipe/log", UPDATE)) < 0))

	return (ERROR);

    logTaskId = taskSpawn ("logTask",logTaskPriority,
			   logTaskOptions, logTaskStackSize,
			   logTask, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    if (logTaskId == ERROR)
	return (ERROR);

    logFdSemId = semCreate ();
    semGive (logFdSemId);

    logFdSet (fd);

    return (OK);
    }
/*******************************************************************************
*
* logMsg - log formatted error message
*
* This routine logs the specified message via the logging task.
* The arguments to the routine are identical to printf (2):
* a format string followed by arguments to be filled in the format.
* However, logMsg (2) is restricted to a maximum of 6 arguments following
* the format string.
*
* The task id of the caller will be prepended to the specified message.
*
* EXAMPLE
*
* If the following code were executed by task 20:
*
*   {
*   name = "GRONK";
*   num = 123;
*
*   logMsg ("ERROR - name = %s, num = %d.\\n", name, num);
*   }
*
* then the following error message would appear on the system log:
*
* t20: ERROR - name = GRONK, num = 123.
*
* SEE ALSO: printf(2)
* VARARGS1
* ARGSUSED
*/

VOID logMsg (fmt, arg1, arg2, arg3, arg4, arg5, arg6)
    char *fmt;	/* format string for print */
    int arg1;	/* optional arguments for fmt */
    int arg2;
    int arg3;
    int arg4;
    int arg5;
    int arg6;

    {
    LOGMSG logmsg;

    /* get calling task's id */

    if (intContext ())
	logmsg.id = -1;
    else
	logmsg.id = taskIdSelf ();

    bcopy ((char *)&fmt, (char *)&logmsg.fmt,
	   sizeof (fmt) + MAX_ARGS * sizeof (arg1));

    if (write (logPipe, (char *)&logmsg, sizeof (logmsg)) != sizeof (logmsg))
	logMsgsLost++;
    }
/*******************************************************************************
*
* logFdSet - set the primary logging file descriptor
*
* This routine sets the file descriptor to which logMsg (2) messages are
* written.  This allows the logging device to be changed from the file
* descriptor specified in the call to logInit (2).
*
* The old logging file descriptor is not closed or otherwise affected by this
* call.  It simply won't be used any more by the logging facilities.
*
* This routine first deletes the old file descriptor (if one had been
* previously set) then adds the new file descriptor.
*
* SEE ALSO: logFdAdd (2), logFdDelete (2)
*/

VOID logFdSet (fd)
    int fd;	/* file descriptor to use as logging device */

    {
    static int oldLogFd = NONE;

    if (oldLogFd != NONE)
	logFdDelete (oldLogFd);

    if (logFdAdd (fd) == OK)
	oldLogFd = fd;
    else
	oldLogFd = NONE;
    }
/*******************************************************************************
*
* logFdAdd - add another logging file descriptor
*
* This routine will cause logging to also appear on the additional file
* descriptor.  The file descriptor should be a valid open file descriptor.
*
* RETURNS:
* OK, or
* ERROR if number of additional logging file descriptors is exceeded.
*/

STATUS logFdAdd (fd)
    int fd;	/* file descriptor for additional logging device */

    {
    if ((numLogFds + 1) > MAX_LOGFDS)
	{
	/* XXX should errnoSet (S_logLib_TOO_MANY_LOGGING_FDS); */
	return (ERROR);
	}

    semTake (logFdSemId);

    logFd [numLogFds++] = fd;

    semGive (logFdSemId);

    return (OK);
    }
/*******************************************************************************
*
* logFdDelete - delete logging file descriptor
*
* This routine removes a logging file descriptor added by logFdAdd (2).
* The file descriptor is not closed, it is just not used by the logging
* facilities anymore.
*
* RETURNS:
* OK, or
* ERROR if file descriptor not previously assigned using logFdAdd (2).
*
* SEE ALSO: logFdAdd (2)
*/

STATUS logFdDelete (fd)
    int fd;	/* file descriptor to stop using as logging device */

    {
    FAST int i;
    FAST int j;

    semTake (logFdSemId);

    for (i = j = 0; i < numLogFds; i++, j++)
	{
	/* shift array of logFd's after deleting unwanted fd */

	if (((logFd [j] = logFd [i]) == fd) && i == j)
	    j--;
	}

    if (i == j)
	{
	semGive (logFdSemId);
	return (ERROR);		/* didn't find specified fd */
	}

    numLogFds--;

    semGive (logFdSemId);

    return (OK);
    }
/*******************************************************************************
*
* logTask - message logging support task
*
* This is the task that actually prints messages logged with the logMsg (2)
* routine.  It continually reads a pipe and prints any 
* messages that come through the pipe in log format on the file descriptor 
* that was specified on the logInit (2) call
* (or subsequent logFdSet (2) or logFdAdd (2) calls).
*
* This task is spawned by logInit (2).
*/

VOID logTask ()

    {
    static int oldMsgsLost;
    LOGMSG logmsg;
    int newMsgsLost;	/* used in case logMsgsLost is changed during use */

    FOREVER
	{
	if (read (logPipe, (char *)&logmsg, sizeof (logmsg)) <= 0)
	    lprintf ("logTask: error reading log messages.\n");

	/* print task id */

	lprintf ((logmsg.id < 0) ? "int: " : "t%x: ", logmsg.id);

	/* print caller's message */

	lprintf (logmsg.fmt, 
		 logmsg.args[0], logmsg.args[1], logmsg.args[2], 
		 logmsg.args[3], logmsg.args[4], logmsg.args[5]);

	/* check for any more messages lost */

	newMsgsLost = logMsgsLost;

	if (newMsgsLost != oldMsgsLost)
	    {
	    lprintf ("logTask: %d log messages lost.\n",
		     newMsgsLost - oldMsgsLost);

	    oldMsgsLost = newMsgsLost;
	    }
	}
    }
/*******************************************************************************
*
* lprintf - log printf
*
* Performs an fdprintf on all logFds.
*
* VARARGS1
*/

LOCAL VOID lprintf (fmt, arg1, arg2, arg3, arg4, arg5, arg6)
    char *fmt;	/* format string for print */
    int arg1;	/* optional arguments to fmt */
    int arg2;
    int arg3;
    int arg4;
    int arg5;
    int arg6;

    {
    FAST int ix;

    semTake (logFdSemId);

    for (ix = 0; ix < numLogFds; ix++)
	fdprintf (logFd [ix], fmt, arg1, arg2, arg3, arg4, arg5, arg6);

    semGive (logFdSemId);
    }
/*******************************************************************************
*
* logShow - show active logging fd's (debug only)
* NOMANUAL
*/

VOID logShow ()

    {
    FAST int ix;

    printf ("%3s %3s\n", "num", "fd");

    for (ix = 0; ix < numLogFds; ix++)
	printf ("%3d %3d\n", ix, logFd [ix]);
    }
