/* ioLib.c - I/O interface library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
03n,02may89,dab  documentation touchup in lseek().
03m,23mar89,dab  made L_XTND option work in lseek(); fixed L_INCR bug in
		 lseek(); documentation touchup in creat().
03l,23nov88,llk  fixed bug in handling multiple links.
	    dnw  changed ioTaskStd{Set,Get} to take taskId arg.
03k,08oct88,gae  made global standard fd array local.
03j,07sep88,gae  documentation.
03i,29aug88,gae  documentation.
03h,15aug88,jcf  lint.
03g,15jul88,llk  changed to allocate an fd name before calling iosFdSet or
		   iosFdNew.
03f,30jun88,llk  changed open and create so that they handle links.
03e,04jun88,llk  uses a default path instead of a default device.
		 changed ioDefDev to ioDefPath.
		 added ioDefPathPrompt, ioDefPathShow, ioDefPathSet,
		    ioDefPathGet, ioDefDirGet.
		 rewrote ioDefDevGet.
		 removed ioSetDefDev.
		 added ioFullFileNameGet.
03d,30may88,dnw  changed to v4 names.
03c,28may88,dnw  deleted obsolete create().
03b,27may88,llk  see change 03e above.
03a,30mar88,gae  made I/O system, i.e. iosLib, responsible for low-level
		   creat,open,read,write,close calls to drivers as
		   fdTable & drvTable are now hidden.
                 added parameters to open() for UNIX compatibility.
		 made L_INCR option work in lseek().
		 added io{G,S}et{Global,Task}Std().
02u,19apr88,llk  added a parameter to open.  Inspired by NFS and aspirations
		   toward UNIX compatibility.
02t,31dec87,jlf  made creat and open follow links.
02s,05nov87,jlf  documentation
02r,30sep87,gae  removed usage of fdTable by using new iosFdSet().
		 added FIOGETNAME to ioctl.
02q,28aug87,dnw  changed creat() and open() to only treat ERROR (-1) as error
		   indication from driver routines instead of < 0 to allow
		   driver values (i.e. structure pointers) to be in memory
		   with high bit set.
02p,28apr87,gae  close() now returns status; added UNIX compatible creat().
02o,24mar87,jlf  documentation
02n,25feb87,ecs  minor documentation.
		 added include of strLib.h.
02m,14feb87,dnw  changed create(), open(), delete() to return ERROR if null
		   filename is specified.  However, filename "." is turned
		   into null filename before being passed to driver routine.
02l,20dec86,dnw  fixed ioGetDefDevTail() to require complete match of dev name.
		 changed to not get include files from default directories.
02k,07oct86,gae	 added ioGetDefDevTail().
02j,04sep86,jlf  minor documentation
02i,01jul86,jlf  minor documentation
02h,03apr86,llk	 fixed syntax error in lseek.
02g,03mar86,jlf  changed ioctrl calls to ioctl.
		 added lseek.
...deleted pre 86 history - see RCS
*/

/*
DESCRIPTION
This library contains the interface to the I/O system.  Calls to the
routines in this module cause calls to the corresponding routines in the
driver for a particular device.  Those driver routines do all the actual
work, so much of what actually transpires during a read, write, etc.  is
device dependent.  If the drivers adhere to convention, however, the
results should be consistent from device to device.

GLOBAL FD'S
Just as VxWorks has a single address space for all tasks, there is
single "file descriptor space" for all fd's.  Like UNIX, an fd
is a small integer returned by a creat, open or socket call.
When VxWorks starts up in usrConfig (1), a console device is opened
and this file descriptor is assigned to the global standard input, output,
and error fd's with ioGlobalStdSet.  The term global conveys that
all tasks use these file descriptors when using standard input (0),
standard output (1), and standard error (2).  A task may reassign
its default I/O with ioTaskStdSet.

INCLUDE FILE: ioLib.h

SEE ALSO: "I/O System", iosLib (1)
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "memLib.h"
#include "ioLib.h"
#include "iosLib.h"
#include "strLib.h"
#include "taskLib.h"

IMPORT char ioDefPath [MAX_FILENAME_LENGTH];	/* current default i/o prefix */

#define	STD_VALID(fd)	(((fd) >= 0) && ((fd) < 3))

LOCAL int ioStdFd [3];		/* global standard input/output/error */


/*******************************************************************************
*
* creat - create a file
*
* Create a new file named `name', and open it with the specified
* `mode'.  This routine figures out which device the file is to be
* created on, then calls that device driver's create routine to do
* most of the actual work.  Therefore, much of what really transpires
* here is device/driver dependent.
*
* The `mode' parameter is used for the file descriptor, and is set as either
* READ, WRITE, or UPDATE for the duration that the file is opened.
* To create NFS files with a particular file mode of the UNIX chmod (2) type,
* call open() with the file mode specified in the third argument.
*
* INTERNAL
* A driver's create routine will return FOLLOW_LINK if any part of the file name
* contains a link (in the directory path).  In this case, it will also
* have changed the name of the file being opened to incorporate the name of
* the link.  The new file name is then repeatedly resubmitted to the driver's
* open routine until all links are resolved.
*
* RETURNS:
*    File descriptor number, or
*    ERROR if no filename specified,
*	      no such device,
*             no more fd's available (see iosInit),
*             or driver returns ERROR.
*
* SEE ALSO: "I/O System", open (2)
*/

int creat (name, mode)
    char *name;		/* name of the file to create */
    int mode;		/* file creation mode         */
			/* (UNIX chmod style)         */

    {
    DEV_HDR *pDevHdr1;
    DEV_HDR *pDevHdr2;
    int value;
    int fd;
    char fullFileName [MAX_FILENAME_LENGTH];
    char *pPartFileName;
    char *pName;

    /* don't allow null filename (for user protection) but change
     * "." into what null filename would be if we allowed it */

    if (name[0] == EOS)
	{
	errnoSet (S_ioLib_NO_FILENAME);
	return (ERROR);
	}

    if (strcmp (name, ".") == 0)
	++name;		/* point to EOS (turn it into null filename) */

    if (ioFullFileNameGet (name, &pDevHdr1, fullFileName) == ERROR)
	return (ERROR);

    if ((fd = iosFdNew (pDevHdr1, (char *) NULL, 0)) == ERROR)
	return (ERROR);

    value = iosCreate (pDevHdr1, fullFileName, mode);

    if (value == ERROR)
	{
	iosFdFree (fd);
	return (ERROR);
	}

    while (value == FOLLOW_LINK)
	{
	/* if a file name contains a link, the driver's create routine changed
	 * fullFileName to incorporate the link's name. Try to create the file
	 * that the driver's create routine handed back.
	 */

	if ((pDevHdr2 = iosDevFind (fullFileName, &pPartFileName)) == NULL)
	    return (ERROR);

	if (fullFileName != pPartFileName)
	    {
	    /* link file name starts with a vxWorks device name,
	     * possibly a different device from the current one.
	     */

	    strcpy (fullFileName, pPartFileName);
	    }
	else
	    {
	    /* link file name does not start with a vxWorks device name.
	     * create the file on the current device.
	     */

	    pDevHdr2 = pDevHdr1;
	    }

	if ((value = iosCreate (pDevHdr2, fullFileName, mode)) == ERROR)
	    {
	    iosFdFree (fd);
	    return (ERROR);
	    }
	} /* while */

    pName = malloc ((unsigned) (strlen (name) + 1));
    strcpy (pName, name);

    iosFdSet (fd, pDevHdr1, pName, value);

    return (fd);
    }
/*******************************************************************************
*
* delete - delete a file
* 
* Delete the specified file.  Delete calls the device driver's delete
* routine for the particular device on which the file is located to do
* all the actual work.
*
* RETURNS:
*    OK if there is no delete routine for the device, or driver return OK,
*    ERROR if there is no such device, or driver returns ERROR.
*
* SEE ALSO: "I/O System"
*/

STATUS delete (name)
    char *name;		/* name of the file to delete */

    {
    DEV_HDR *pDevHdr;
    char fullFileName [MAX_FILENAME_LENGTH];

    /* don't allow null filename (for user protection) */

    if (name[0] == EOS)
	{
	errnoSet (S_ioLib_NO_FILENAME);
	return (ERROR);
	}

    if (ioFullFileNameGet (name, &pDevHdr, fullFileName) == ERROR)
	return (ERROR);

    return (iosDelete (pDevHdr, fullFileName));
    }
/*******************************************************************************
*
* open - open a file
*
* Open the file for reading, writing, or updating (reading and writing)
* as specified by the flags argument, and returns a descriptor for that file.
* The flags argument may indicate that the file is to be created if it does not 
* already exist (by specifying the O_CREAT flag), in which case the file is
* created with mode `mode' as described in UNIX chmod (2).  However only the
* NFS driver makes use of the `mode' argument.
*
* The `flags' value is constructed by or'ing the following:
* (only one of the first three below may be used):
*
*	READ    (or O_RDONLY)	open for reading only
*	WRITE   (or O_WRONLY)	open for writing only
*	UPDATE  (or O_RDWR)	open for reading and writing
*	O_CREAT	                create a new file
*
* INTERNAL
* A driver's open routine will return FOLLOW_LINK if any part of the file name
* contains a link (directory path or file name).  In this case, it will also
* have changed the name of the file being opened to incorporate the name of
* the link.  The new file name is then repeatedly resubmitted to the driver's
* open routine until all links are resolved.
*
* RETURNS:
*    File descriptor number, or
*    ERROR if no filename specified,
*	      no such device,
*	      no more fd's (see iosInit (2)),
*             or driver returns ERROR.
*
* SEE ALSO: "I/O System"
*
* VARARGS2
*/

int open (name, flags, mode)
    char *name;		/* name of the file to open        */
    int flags;		/* READ, WRITE, UPDATE, or O_CREAT */
    int mode;		/* mode of file to be created      */
			/* (UNIX chmod style)              */

    {
    DEV_HDR *pDevHdr1;
    DEV_HDR *pDevHdr2;
    int value;
    int fd;
    char fullFileName [MAX_FILENAME_LENGTH];
    char *pPartFileName;
    char *pName;

    /* don't allow null filename (for user protection) but change
     * "." into what null filename would be if we allowed it */

    if (name[0] == EOS)
	{
	errnoSet (S_ioLib_NO_FILENAME);
	return (ERROR);
	}

    if (strcmp (".", name) == 0)
	++name;		/* point to EOS (turn it into null filename) */

    if (ioFullFileNameGet (name, &pDevHdr1, fullFileName) == ERROR)
	return (ERROR);

    if ((fd = iosFdNew (pDevHdr1, (char *) NULL, 0)) == ERROR)
	return (ERROR);

    value = iosOpen (pDevHdr1, fullFileName, flags, mode);

    if (value == ERROR)
	{
	iosFdFree (fd);
	return (ERROR);
	}

    while (value == FOLLOW_LINK)
	{
	/* if a file name contains a link, the driver's open routine changed
	 * fullFileName to incorporate the link's name. Try to open the file
	 * that the driver's open routine handed back.
	 */

	if ((pDevHdr2 = iosDevFind (fullFileName, &pPartFileName)) == NULL)
	    return (ERROR);

	if (fullFileName != pPartFileName)
	    {
	    /* link file name starts with a vxWorks device name,
	     * possibly a different device from the current one.
	     */

	    strcpy (fullFileName, pPartFileName);
	    }
	else
	    {
	    /* link file name does not start with a vxWorks device name.
	     * open the file on the current device.
	     */

	    pDevHdr2 = pDevHdr1;
	    }

	if ((value = iosOpen (pDevHdr2, fullFileName, flags, mode)) == ERROR)
	    {
	    iosFdFree (fd);
	    return (ERROR);
	    }
	} /* while */

    pName = malloc ((unsigned) (strlen (name) + 1));
    strcpy (pName, name);

    iosFdSet (fd, pDevHdr1, pName, value);

    return (fd);
    }
/*******************************************************************************
*
* close - close a file
*
* Close the specified file.
* Calls the device driver's close routine to do the actual work,
* therefore, what really transpires here is device dependent.
* The file descriptor is free'd.
*
* RETURNS:
*  status of driver's close routine, or
*  ERROR if invalid fd
*
* SEE ALSO: "I/O System"
*/

STATUS close (fd)
    int fd;		/* file descriptor to close */

    {
    return (iosClose (fd));
    }
/*******************************************************************************
*
* read - read bytes from a file or device
*
* Reads some number of bytes (less than or equal to maxbytes) from the
* specified stream.  Calls the device driver to do all the actual work.
* 
* RETURNS:
*  number of bytes actually read;
*  (between 1 and maxbytes, 0 if end of file), or
*  ERROR if no such open fd, or driver returns ERROR
*
* SEE ALSO: "I/O System"
*/

int read (fd, buffer, maxbytes)
    int fd;		/* file descriptor from which to read      */
    char *buffer;	/* pointer to buffer to receive bytes      */
    int maxbytes;	/* max number of bytes to read into buffer */

    {
    return (iosRead (fd, buffer, maxbytes));
    }
/*******************************************************************************
*
* write - write bytes to a file
*
* Writes nbytes from buffer to fd.
* Calls the driver write routine to do all the actual work.
* Actual operation is driver dependent.
*
* RETURNS:
*  number of bytes actually written
*  (if not equal to nbytes, some error has occurred), or
*  ERROR if no such open fd, or driver returns ERROR
*
* SEE ALSO: "I/O System"
*/

int write (fd, buffer, nbytes)
    int fd;		/* file descriptor on which to write     */
    char *buffer;	/* buffer containing bytes to be written */
    int nbytes;		/* number of bytes to write              */

    {
    return (iosWrite (fd, buffer, nbytes));
    }
/*******************************************************************************
*
* ioctl - do file specific control function
*
* Performs a device-specific special function on a device.  Most
* requests are passed on to the driver, which actually handles them
* as it wishes.  One request is handle directly at the I/O interface level:
*
*   ioctl (fd, FIOGETNAME, &nameBuf)
*	return in buffer `nameBuf' the file
*	name of the file descriptor
*
* RETURNS:
*    ERROR if no such open fd, or
*    whatever the driver returns.
*
* SEE ALSO: "I/O System"
*
* VARARGS2
*/

int ioctl (fd, function, arg)
    int fd;		/* file descriptr     */
    int function;	/* function code      */
    int	arg;		/* arbitrary argument */

    {
    return (iosIoctl (fd, function, arg));
    }
/*******************************************************************************
*
* lseek - set file read/write pointer
*
* Sets the file read/write pointer of file 'fd' to 'offset'.
* The `whence' has three values which affect the file position pointer:
*
*      L_SET  - set to `offset',
*      L_INCR - set to current position plus `offset',
*      L_XTND - set to the size of the file plus `offset'.
*
* This routine uses ioctl (2) with functions FIOWHERE and FIOSEEK.
*
* RETURNS:
*    the new offset from the beginning of the file, or ERROR
*
* ARGSUSED
*/

int lseek (fd, offset, whence)
    int fd;		/* file descriptor            */
    long offset;	/* new byte offset to seek to */
    int whence;		/* relative file position     */
    
    {
    int where;
    long nBytes;

    switch (whence)
	{
	case L_SET:
	    return ((ioctl (fd, FIOSEEK, offset) == OK) ? offset : ERROR);

	case L_INCR:
	    if ((where = ioctl (fd, FIOWHERE)) == ERROR)
		return (ERROR);

	    offset += where;
	    return ((ioctl (fd, FIOSEEK, offset) == OK) ? offset : ERROR);

	case L_XTND:
	    if ((where = ioctl (fd, FIOWHERE)) == ERROR)
		return (ERROR);
	    if (ioctl (fd, FIONREAD, &nBytes) == ERROR)
		return (ERROR);

	    offset += where + nBytes; 
	    return ((ioctl (fd, FIOSEEK, offset) == OK) ? offset : ERROR);

	default:
	    return (ERROR);
	}
    }

/*******************************************************************************
*
* ioFullFileNameGet - get the full file name using the current directory
*
* The device header gets returned, along with the complete name of the
* file relative to that device (i.e. the device name is not in the returned
* file name).
*
* RETURNS: OK or ERROR if first component is not a device
*/

STATUS ioFullFileNameGet (partName, ppDevHdr, fullFileName)
    char *partName;	/* original name */
    DEV_HDR **ppDevHdr;	/* pointer to device header that gets returned */
    char *fullFileName;	/* resulting complete file name */

    {
    char *pPathName;
    char defDir [MAX_FILENAME_LENGTH];

    if ((*ppDevHdr = iosDevFind (partName, &pPathName)) == NULL)
	return (ERROR);

    if (partName != pPathName)
	{
	/* partName starts with a device name, pass on the directory */

	strcpy (fullFileName, pPathName);
	}
    else
	{
	/* use default path */

	ioDefDirGet (defDir);
	pathCat (defDir, partName, fullFileName);
	}
    return (OK);
    }
/*******************************************************************************
*
* ioDefPathSet - set current default path
*
* This routine sets up the default I/O path.
* Path `name' must begin with an existing device's name.
*
* RETURNS: OK or ERROR if first component is not a device
*
* SEE ALSO: ioDefPathGet (2)
*/

STATUS ioDefPathSet (name)
    char *name;		/* name of the new default device and path */

    {
    char *pTail = name;

    if (iosDevFind (name, &pTail) == NULL)
	return (ERROR);

    if (pTail == name)
	{
	/* name does not begin with an existing device's name */

	errnoSet (S_ioLib_NO_DEVICE_NAME_IN_PATH);
	return (ERROR);
	}

    strncpy (ioDefPath, name, MAX_FILENAME_LENGTH);
    return (OK);
    }
/*******************************************************************************
*
* ioDefPathGet - get current default directory
*
* Copies the name of the current default path to the character array
* passed as a parameter.
*
* SEE ALSO: ioDefPathSet (2)
*/

VOID ioDefPathGet (name)
    char *name;		/* where to return the name */

    {
    strncpy (name, ioDefPath, MAX_FILENAME_LENGTH);
    }
/*******************************************************************************
*
* ioDefDevGet - get current default device
*
* Copies the name of the current default device to the character array
* passed as a parameter.
*/

VOID ioDefDevGet (devName)
    char *devName;		/* where to return the device name */

    {
    char *pTail = devName;
    FAST int devNameLen;			/* device name length */

    /* find the device name in the default path name */

    if (iosDevFind (ioDefPath, &pTail) == NULL)
	{
	/* no default device, return null device name */
	*devName = EOS;
	}
    else
	{
	/* extract device name from path name */

	devNameLen = (int) (pTail - ioDefPath);
	bcopy (ioDefPath, devName, devNameLen);
	devName [devNameLen] = EOS;
	}
    }
/*******************************************************************************
*
* ioDefDirGet - get current default directory
*
* Copies the default path into `dirName'.  The default device is removed.
*/

VOID ioDefDirGet (dirName)
    char *dirName;		/* where to return the directory name */

    {
    char *pTail = dirName;

    /* find the directory name in the default path name */

    if (iosDevFind (ioDefPath, &pTail) == NULL)
	{
	/* no default device, return null directory name */

	*dirName = EOS;
	}
    else
	strcpy (dirName, pTail);
    }

/*******************************************************************************
*
* ioGlobalStdSet - set underlying global fd for standard input/output/error
*
* The console file descriptor is ordinarily the input, output and
* error fd used by all tasks.  If, for instance, it is desirable to
* send error output to a different file use this routine.
* If `stdFd' is not 0, 1, or 2 then this routine has no effect.
*
* SEE ALSO: ioGlobalStdGet (2), ioTaskStdSet (2)
*/

VOID ioGlobalStdSet (stdFd, newFd)
    int stdFd;	/* standard input (0), output (1), or error (2) */
    int newFd;	/* new underlying fd                            */

    {
    if (STD_VALID (stdFd))
	ioStdFd [stdFd] = newFd;
    }
/*******************************************************************************
*
* ioGlobalStdGet - get underlying global fd for standard input/output/error
*
* Usually this will return the console file descriptor unless changed
* with ioGlobalStdSet (2).
*
* RETURNS:
*    underlying fd, or
*    ERROR if `stdFd' is not 0, 1, or 2
*
* SEE ALSO: ioGlobalStdSet (2), ioTaskStdGet (2)
*/

int ioGlobalStdGet (stdFd)
    int stdFd;	/* standard input (0), output (1), or error (2) */

    {
    return (STD_VALID (stdFd) ? ioStdFd [stdFd] : ERROR);
    }
/*******************************************************************************
*
* ioTaskStdSet - set underlying task fd for standard input/output/error
*
* The console file descriptor as set by ioGlobalStdSet, is ordinarily
* the input, output and error fd used by a task.
* If a particular task should use different input, output, and error fd's
* use this routine to change them.
* If `stdFd' is not 0, 1, or 2 then this routine has no effect.
*
* This routine has no effect if called at interrupt level.
*
* SEE ALSO: ioGlobalStdGet (2), ioTaskStdGet (2)
*/

VOID ioTaskStdSet (taskId, stdFd, newFd)
    int taskId;	/* id of task whose std fd is to be set (0 = self) */
    int stdFd;	/* standard input (0), output (1), or error (2) */
    int newFd;	/* new underlying fd                            */

    {
    TCBX *pTcbX;

    if (STD_VALID (stdFd) && (pTcbX = taskTcbX (taskId)) != NULL)
	pTcbX->taskStd [stdFd] = newFd;
    }
/*******************************************************************************
*
* ioTaskStdGet - get underlying task fd for standard input/output/error
*
* Usually this will return the console file descriptor
* as set by ioGlobalStdSet (2) unless changed for this particular task
* with ioTaskStdSet (2).
*
* RETURNS:
*  underlying fd, or
*  ERROR if `stdFd' is not 0, 1, or 2, or
*  if called at interrupt level
*
* SEE ALSO: ioGlobalStdGet (2), ioTaskStdSet (2)
*/

int ioTaskStdGet (taskId, stdFd)
    int taskId;	/* id of desired task (0 = self) */
    int stdFd;	/* standard input (0), output (1), or error (2) */

    {
    TCBX *pTcbX;
    int taskFd;

    if (STD_VALID (stdFd) && (pTcbX = taskTcbX (taskId)) != NULL)
	{
	taskFd = pTcbX->taskStd [stdFd];
	if (STD_VALID (taskFd))
	    return (ioStdFd [taskFd]);
	else
	    return (taskFd);
	}

    return (ERROR);
    }
