/* intLib.c - interrupt subroutine library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
02x,24may89,jcf  made v02w solution optional.  clean up.
02w,21apr89,jcf  fixed intUnlock () to solve psos priority inversion.
02v,23sep88,gae  documentation touchup.
02u,30aug88,gae  more documentation tweaks.
02t,18aug88,gae  documentation.
02s,12aug88,jcf  changed interrupt wrapper to better accomodate '010 wind.
02r,30may88,dnw  changed to v4 names.
02q,29oct87,gae  fixed intConnect() to use sysKernelTrap() and split
		   routine construction into intHandlerCreate().
		 made intSetVecBase() truly a NOP for 68000's.
02p,10may87,gae  changed intConnect to not rely on structure layout.
02o,27may87,llk  changed intConnect so that UI_ENTER is not included for 68020s.
02n,16mar87,dnw  fixed documentation.
		 added vxLockLevel.
02m,21dec86,dnw  changed to not get include files from default directories.
02l,29nov86,dnw  removed conditional assembly for 68020/68000.
02k,28oct86,llk  conditionally compile out vector base address code
		 for non 68020 based systems.
02j,27oct86,llk  added facilities for using the vector base register.
		 added intSetVec, intGetVec, intSetVecBase, intGetVecBase.
02i,01jul86,jlf  minor documentation
02h,14apr86,rdc  changed memAllocates to mallocs. 
02g,18jan86,dnw  added intCount routine to return interrupt nesting depth.
		 included interrupt termination code, formerly in intEnd,
		   directly in driver built by intConnect.
02f,23oct85,jlf  changed stub made by intConnect to not call intEnt, but
		   rather put the same code as intEnt in-line.  This is because
		   of ISR stack switching.
02e,11oct85,dnw  de-linted.
02d,06aug85,jlf  removed include of intLib.h
02c,20jul85,jlf  documentation.
02b,31may85,jlf  added intContext and intCnt.  Made intConnect call
		 intEnt (in intALib.s) instead of trapping to UI_ENTER.
02a,05apr85,rdc  installed modifications for vrtx version 3.
01e,06sep84,ecs  removed intInit, intTask, intTlCall, and intPrintf to dbgLib.
		 got rid of includes of memLib.h, rngLib.h, and semLib.h.
01d,04sep84,dnw  changed call to fioLog to printf.
01c,22aug84,ecs  changed intPrintf to return status.
01b,17aug84,dnw  added interrupt to task-level support routines
01a,02aug84,dnw  extracted from old ubarLib
*/

/*
This library provides routines to manipulate and connect to hardware
interrupts and exceptions.
Most importantly, any C language routine can be connected to any exception,
interrupt, or trap by calling the routine intConnect.
Interrupt vectors can be accessed directly by the routines
intVecSet and intVecGet.
On the 68010 and 68020, the vector base register can be accessed by the routines
intVecBaseSet and intVecBaseGet.
Tasks can lock and unlock interrupts by calling the routines
intLock and intUnlock.
The routines intCount and intContext can be used to determine whether
the CPU is running in an interrupt context or in a normal task context.

INTERRUPT VECTORS AND NUMBERS
Most of the routines in this library take an interrupt vector as a
parameter, which is the byte offset into the vector table.
Macros are provided to convert these interrupt vectors to interrupt
numbers and vice versa: IVEC_TO_INUM(intVector) changes a vector
to a number.  INUM_TO_IVEC(intNumber) turns a number into a vector.
TRAPNUM_TO_IVEC(trapNumber) converts a trap number to a vector.

EXAMPLE
There are instances where it is desired to switch between one of
several routines for a particular interrupt.  The following
code fragment is one alternative.
.CS
    vector  = INUM_TO_IVEC(some_int_vec_num);
    oldfunc = intVecGet (vector);
    newfunc = intHandlerCreate (routine, parameter);
    intVecSet (vector, newfunc);
    ...
    intVecSet (vector, oldfunc);    /* use original routine *
    ...
    intVecSet (vector, newfunc);    /* reconnect new routine *
.CE

INCLUDE FILE: iv68k.h
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "memLib.h"
#include "sysLib.h"
#include "taskLib.h"

#define	HI_WORD(w)		(short)(((int)(w) & 0xffff0000) >> 16)
#define	LO_WORD(w)		(short)((int)(w) & 0x0000ffff)


/* imports */

IMPORT int kernelType;	/* kernel type */
IMPORT VOID intEnt ();	/* interrupt entrance stub */
IMPORT VOID intExit ();	/* interrupt exit stub */

/* globals */

int intLockLevel = 7;	/* interrupt level to set to "lock-out" interrupts */

int intCnt = 0;		/* Counter to tell if we are at interrupt (non-task) */
			/* level.  Bumped and de-bumped by intALib */

BOOL intPsosUnlockFix = FALSE;
			/* if intPsosUnlockFix is set to TRUE, the intUnlock
			 * routine will call taskDelay (0) when using the
			 * PSOS kernel.  This solves a PSOS priority
			 * inversion problem.  It is not the default because 
			 * it slows PSOS semaphores down 2.5 times.
			 */

/* locals */

LOCAL FUNCPTR *intVecBase = 0;		/* vector base address */

LOCAL USHORT intConnectCode []	=	/* intConnect stub */
    {
/*
*0x00  4EB9 kkkk kkkk	jsr	_intEnt  	  * tell kernel
*0x06  48E7 E0C0	movem.l	d0-d2/a0-a1,-(a7) * save regs
*0x0a  2F3C pppp pppp	move.l	#parameter,-(a7)  * push param
*0x10  4EB9 rrrr rrrr	jsr	routine		  * call C routine
*0x16  588F		addq.l  #4,a7             * pop param
*0x18  4CDF 0307	movem.l (a7)+,d0-d2/a0-a1 * restore regs
*0x1c  4EF9 kkkk kkkk	jmp     _intExit          * exit via kernel
*/
     0x4eb9, 0x0000, 0x0000,	/* _intEnt filled in at runtime */
     0x48e7, 0xe0c0,
     0x2f3c, 0x0000, 0x0000,	/* parameter filled in at runtime */
     0x4eb9, 0x0000, 0x0000,	/* routine to be called filled in at runtime */
     0x588f,
     0x4cdf, 0x0307,
     0x4ef9, 0x0000, 0x0000,	/* _intExit filled in at runtime */
    };


/* forward declarations */

FUNCPTR	intHandlerCreate ();
FUNCPTR	*intVecBaseGet ();
FUNCPTR	intVecGet ();


/*******************************************************************************
*
* intConnect - connect C routine to hardware interrupt
*
* This routine connects the specified C routine to the specified
* interrupt vector.  When an interrupt occurs that vectors through
* the specified address, the routine will be called with the specified
* parameter.  The routine will be invoked in supervisor mode at interrupt
* level.  A proper C environment will have been established,
* the necessary registers saved, and the stack set up.
*
* The routine can be any normal C code, except that it must not
* invoke certain operating system functions.
*
* This routine simply calls intHandlerCreate (2) and intVecSet (2).
* It is the address of the handler returned by intHandlerCreate (2)
* that actually gets put in the interrupt vector.
*
* RETURNS:
*    OK or
*    ERROR if unable to build interrupt handler
*
* SEE ALSO: intHandlerCreate (2), intVecSet (2)
*/

STATUS intConnect (vector, routine, parameter)
    FUNCPTR *vector;		/* interrupt vector to attach to */
    FUNCPTR routine;		/* routine to be called */
    int parameter;		/* parameter to be passed to routine */

    {
    FUNCPTR intDrvRtn = intHandlerCreate (routine, parameter);

    if (intDrvRtn == NULL)
	return (ERROR);

    /* make vector point to synthesized code */

    intVecSet (vector, intDrvRtn);

    return (OK);
    }
/*******************************************************************************
*
* intHandlerCreate - construct interrupt handler for C routine
*
* This routine builds an interrupt handler around the specified C routine.
* This interrupt handler is then suitable for connecting to a
* specific vector address with intVecSet (2).
* The routine will be invoked in supervisor mode at interrupt level.
* A proper C environment will have been established,
* the necessary registers saved, and the stack set up.
*
* The routine can be any normal C code, except that it must not
* invoke certain operating system functions.
*
* IMPLEMENTATION:
* This routine builds an interrupt handler of the following form in
* allocated memory.
*
* .CS
*  0x00  4EB9 kkkk kkkk	jsr	_intEnt  	  * tell kernel
*  0x06  48E7 E0C0	movem.l	d0-d2/a0-a1,-(a7) * save regs
*  0x0a  2F3C pppp pppp	move.l	#parameter,-(a7)  * push param
*  0x10  4EB9 rrrr rrrr	jsr	routine		  * call C routine
*  0x16  588F		addq.l  #4,a7             * pop param
*  0x18  4CDF 0307	movem.l (a7)+,d0-d2/a0-a1 * restore regs
*  0x1c  4EF9 kkkk kkkk	jmp     _intExit          * exit via kernel
* .CE
*
* RETURNS:
*    pointer to new interrupt handler, or
*    NULL if out of memory
*/

FUNCPTR intHandlerCreate (routine, parameter)
    FUNCPTR routine;		/* routine to be called */
    int parameter;		/* parameter to be passed to routine */

    {
    FAST USHORT *pCode;		/* pointer to newly synthesized code */

    pCode = (USHORT *)malloc (sizeof (intConnectCode));

    if (pCode != NULL)
	{
	/* copy intConnectCode into new code area */

	bcopy ((char *)intConnectCode, (char *)pCode, sizeof (intConnectCode));

	/* set the addresses & instructions */

	pCode [1]  = HI_WORD (intEnt);
	pCode [2]  = LO_WORD (intEnt);
	pCode [6]  = HI_WORD (parameter);
	pCode [7]  = LO_WORD (parameter);
	pCode [9]  = HI_WORD (routine);
	pCode [10] = LO_WORD (routine);
	pCode [15] = HI_WORD (intExit);
	pCode [16] = LO_WORD (intExit);
	}

    return ((FUNCPTR)pCode);
    }
/*******************************************************************************
*
* intContext - determine if we are in interrupt context or task context
*
* This routine returns true if and only if we are executing in
* interrupt context and NOT in a meaningful task context.
*
* RETURNS: TRUE or FALSE
*/

BOOL intContext ()

    {
    return (intCnt > 0);
    }
/*******************************************************************************
*
* intCount - get current interrupt nesting depth
*
* This routine returns the number of interrupts that are currently nested.
*
* CAVEAT
* While it may appear that intCount will never return a value greater
* than 7, remember that ISR's may modify the interrupt mask to allow
* lower priority interrupts.
*
* RETURNS: number of nested interrupts
*/

int intCount ()

    {
    return (intCnt);
    }
/*******************************************************************************
*
* intLock - lock out interrupts
*
* This routine is used to disable interrupts.  The interrupt level is
* set to the lock-out level (level 7 by default).  The routine returns
* the previous interrupt level, and this should be restored by a call
* to intUnlock (2).
*
* IMPORTANT CAVEAT
* intLock can be called both from interrupt level and from task level.
* When called from a task context, the interrupt lock level is part
* of the task's context.  Locking out interrupts does not prevent rescheduling.
* Thus, if a task locks out interrupts and then invokes kernel services that
* cause the task to block (e.g. taskSuspend (2) or taskDelay (2)) or
* causes a higher priority task to be ready
* (e.g. semGive (2) or taskResume (2)), then rescheduling will
* occur and interrupts will be unlocked while other tasks run.
* Rescheduling may be explicitly disabled with taskLock (2).
*
* EXAMPLE
* .CS
*  oldLevel = intLock ();
*  /* ...work with interrupts locked out... *
*  intUnlock (oldLevel);
* .CE
*
* RETURNS: previous interrupt level
*
* SEE ALSO: intUnlock (2), taskLock (2)
*/

int intLock ()
    {
    return (intLevelSet (intLockLevel));
    }
/*******************************************************************************
*
* intUnlock - cancel effect of intLock
*
* This routine is used to re-enable interrupts that have been disabled by
* intLock (2).  Use the level obtained from the preceding intLock (2) call.
*
* SEE ALSO: intLock (2)
*/

VOID intUnlock (level)
    int level;			/* level to which to restore interrupts */

    {
    intLevelSet (level);

    /* When the interrupt level on the processor is greater than zero,
     * the pSOS kernel assumes that kernel calls were made from an interrupt
     * service routine.  Thus psos does not affect the changes, defering
     * their operation until the RET_I call at the end of the ISR.  The
     * assumption that a non-zero interrupt level corresponds only to the
     * execution of an interrupt service routine, trips us up here.  When we
     * unlock interrupts to zero, and we are not in an interrupt service
     * routine, and we are using pSOS, we must force the entry into the kernel
     * so that pSOS will have chance to finish any queued kernel work.  (such
     * as semGive ())
     */

    if ((kernelType == KERNEL_PSOS) && (intPsosUnlockFix) &&
	(intCnt == 0) && (level == 0))
	taskDelay (0);
    }
/*******************************************************************************
*
* intVecBaseSet - set the vector base address
*
* This routine sets the vector base address.  The CPU's vector base
* register is set to the specified value, and subsequent calls to 
* intVecGet/intVecSet will use this base address.
* The vector base address is initially 0, until changed by calls to this
* routine.
*
* NOTE:
* The 68000 does not have a vector base register
* so this routine is a NOP for 68000 systems.
*
* SEE ALSO: intVecBaseGet (2), intVecGet (2), intVecSet (2)
*/

VOID intVecBaseSet (baseAddr)
    FUNCPTR *baseAddr;	    /* new vector base address */

    {
    if (sysCpu == MC68000)
	return;

    intVecBase = baseAddr;	/* keep the base address in a static variable */

    intVBRSet (baseAddr);	/* set the actual vector base register */
    }
/*******************************************************************************
*
* intVecBaseGet - get the vector base address
*
* This routine returns the current vector base address that has been set
* via the intVecBaseSet (2) routine.
*
* RETURNS: current vector base address
*
* SEE ALSO: intVecBaseSet (2)
*/

FUNCPTR *intVecBaseGet ()

    {
    return (intVecBase);
    }
/******************************************************************************
*
* intVecSet - set CPU vector
*
* This routine sets the specified exception/interrupt vector to the
* specified address.  The vector is specified as an offset into the
* CPU's vector table.  This vector table starts, by default, at address
* 0, but, on 68010 and 68020 CPU's, may be set to start at any address
* via the intVecBaseSet (2) routine.  The vector table is setup in
* usrInit (2) and starts at the available lowest memory address.
*
* SEE ALSO: intVecBaseSet (2), intVecGet (2)
*/

VOID intVecSet (vector, function)
    FUNCPTR *vector;		/* vector offset */
    FUNCPTR function;		/* address to place in vector */

    {
    FUNCPTR *newVector;

    /* vector is offset by the vector base address */

    newVector = (FUNCPTR *) ((int) vector + (int) intVecBaseGet ());
    *newVector = function;
    }
/*******************************************************************************
*
* intVecGet - get vector
*
* This routine returns the current value of the specified
* exception/interrupt vector.  The vector is specified as an offset into the
* CPU's vector table.  This vector table starts, by default, at address
* 0, but, on 68010 and 68020 CPU's, may be set to start at any address
* via the intVecBaseSet (2) routine.
*
* RETURNS: current value of specified vector
*
* SEE ALSO: intVecSet (2), intVecBaseSet (2)
*/

FUNCPTR intVecGet (vector)
    FUNCPTR *vector;	/* vector offset */

    {
    /* vector is offset by vector base address */

    return (* (FUNCPTR *) ((int) vector + (int) intVecBaseGet ()));
    }
