/* inetLib.c - Internet address manipulation routines */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
 * Copyright (c) 1987,1988, Wind River Systems, Inc.
 * Copyright (c) 1983 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * and the VxWorks Software License Agreement specify the terms and
 * conditions for redistribution.
 */

/*
modification history
--------------------
01j,29sep88,gae  documentation.
01i,18aug88,gae  documentation.
01h,30may88,dnw  changed to v4 names.
01g,18feb88,dnw  changed inet_netof_string() to be work with subnet masks
		   by changing to call in_{makeaddr_b,netof} instead of
		   inet_{...}.
		 lint.
01f,15dec87,gae  appeased lint; checked malloc's for NULL.
01e,17nov87,ecs  lint.
01d,16nov87,jlf  documentation.
01c,16nov87,llk  documentation.
		 changed to use more descriptive variable names.
01b,06nov87,dnw  cleanup.
01a,01nov87,llk  written.
		 modified routines for VxWorks style (setStatus(), etc.)
		 added inet_netof_string(), inet_makeaddr_b(), inet_ntoa_b().
		 changed inet_ntoa() and inet_makeaddr() so that they malloc
		   the structures that they return.
		 NOTE: inet_addr returns u_long as specified in SUN documenta-
		   tion, NOT in ISI documentation and arpa/inet.h header file.
*/

/*
DESCRIPTION
InetLib provides routines for manipulating Internet addresses,
including the UNIX BSD 4.3 "inet_" routines.
There are routines for converting
between character addresses in Internet standard "." (dot)
notation and integer addresses, routines for extracting the
network and host portions out of an Internet address, and routines
for constructing Internet addresses given the network and host
address parts.

All Internet addresses are returned in network order (bytes
ordered from left to right).  All network numbers and local
address parts are returned as machine format integer values.

INTERNET ADDRESSES

Internet addresses are typically specified in dot notation or
as a 4 byte number.  Values specified using the dot notation
take one of the following forms:

		a.b.c.d
		a.b.c
		a.b
		a

When four parts are specified, each is interpreted as a byte
of data and assigned, from left to right, to the four bytes
of an Internet address.  Note that when an Internet address
is viewed as a 32-bit integer quantity on any MC68000 family
machine, the bytes referred to above appear as ``a.b.c.d''.
That is, MC68000-based bytes are ordered from left to right.

When a three-part address is specified, the last part is
interpreted as a 16-bit quantity and placed in the right-most
two bytes of the network address.  This makes the three-part
address format convenient for specifying Class B network
addresses as ``128.net.host''.

When a two-part address is supplied, the last part is interpreted
as a 24-bit quantity and placed in the right-most
three bytes of the network address.  This makes the two-part
address format convenient for specifying Class A network
addresses as ``net.host''.

When only one part is given, the value is stored directly in
the network address without any byte rearrangement.

All numbers supplied as parts in a dot notation may be
decimal, octal, or hexadecimal, as specified in the C
language.  That is, a leading 0x or 0X implies hexadecimal;
otherwise, a leading 0 implies octal.  With no leading 0,
the number is interpreted as decimal.

INCLUDE FILES:
inetLib.h, in.h

SEE ALSO:
"Network", UNIX BSD 4.3 documentation for inet (3N)
*/


#include "vxWorks.h"
#include "types.h"
#include "ctype.h"
#include "in.h"
#include "memLib.h"
#include "strLib.h"
#include "inetLib.h"


/*******************************************************************************
*
* inet_addr - convert a dot notation Internet address to a long integer
*
* Internet address interpretation routine.  All the network library
* routines call this routine to interpret entries in the data bases
* which are expected to be an address.  The value returned is in network order.
*
* EXAMPLE
* .CS
*  inet_addr ("90.0.0.2");
* .CE
* returns 0x5a000002.
*
* RETURNS: Internet address, or ERROR
*/

u_long inet_addr (inetString)
    register char *inetString;

    {
    register u_long val, base, n;
    register char c;
    u_long parts[4], *pp = parts;

again:

    /* Collect number up to ``.''.  Values are specified as for C:
     * 0x=hex, 0=octal, other=decimal. */

    val = 0; base = 10;
    if (*inetString == '0')
	base = 8, inetString++;
    if (*inetString == 'x' || *inetString == 'X')
	base = 16, inetString++;
    while (c = *inetString)
	{
	if (isdigit (c)) 
	    {
	    val = (val * base) + (c - '0');
	    inetString++;
	    continue;
	    }
	if (base == 16 && isxdigit (c))
	    {
	    val = (val << 4) + (c + 10 - (islower (c) ? 'a' : 'A'));
		inetString++;
		continue;
	    }
	    break;
	} /* while */

    if (*inetString == '.')
	{
	/*
	 * Internet format:
	 *	a.b.c.d
	 *	a.b.c	(with c treated as 16-bits)
	 *	a.b	(with b treated as 24 bits)
	 */

	if (pp >= parts + 4)
	    {
	    errnoSet (S_inetLib_ILLEGAL_INTERNET_ADDRESS);
	    return (ERROR);
	    }
	*pp++ = val, inetString++;
	goto again;
	}

    /* Check for trailing characters. */

    if (*inetString && !isspace (*inetString))
	{
	errnoSet (S_inetLib_ILLEGAL_INTERNET_ADDRESS);
	return (ERROR);
	}
    *pp++ = val;

    /* Concoct the address according to the number of parts specified. */
    
    n = pp - parts;
    switch ((int) n)
	{
	case 1:				/* a -- 32 bits */
	    val = parts[0];
	    break;

	case 2:				/* a.b -- 8.24 bits */
	    val = (parts[0] << 24) | (parts[1] & 0xffffff);
	    break;

	case 3:				/* a.b.c -- 8.8.16 bits */
	    val = (parts[0] << 24) | ((parts[1] & 0xff) << 16)
		   | (parts[2] & 0xffff);
	    break;

	case 4:				/* a.b.c.d -- 8.8.8.8 bits */
	    val = (parts[0] << 24) | ((parts[1] & 0xff) << 16) |
		  ((parts[2] & 0xff) << 8) | (parts[3] & 0xff);
	    break;

	default:
	    errnoSet (S_inetLib_ILLEGAL_INTERNET_ADDRESS);
	    return (ERROR);
	}

    return (htonl (val));
    }
/*******************************************************************************
*
* inet_lnaof - get local net address (host number) from Internet address
*
* Return the local network address portion of an Internet address;
* handles class A, B and C network number formats.
*
* EXAMPLE
* .CS
*  inet_lnaof (0x5a0000002);
* .CE
* returns 2.
*
* RETURNS: local address portion of `inetAddress'
*/

int inet_lnaof (inetAddress)
    int inetAddress;    /* inet address from which to extract local portion */

    {
    register u_long i = ntohl ((u_long) inetAddress);

    if (IN_CLASSA (i))
	return ((i) &IN_CLASSA_HOST);
    else if (IN_CLASSB (i))
	return ((i) &IN_CLASSB_HOST);
    else
	return ((i) &IN_CLASSC_HOST);
    }
/*******************************************************************************
*
* inet_makeaddr_b - construct an Internet address from network + host numbers
*
* Given the network number and local host address the Internet address is
* formed.
* This routine is identical to the UNIX inet_makeaddr (2) routine
* except that a buffer for the resulting value must provided.
*
* EXAMPLE
* .CS
*  inet_makeaddr_b (0x5a, 2, pInetAddr);
* .CE
* returns the address 0x5a000002 in the location pointed to by pInetAddr.
*/

VOID inet_makeaddr_b (netAddr, hostAddr, pInetAddr)
    int netAddr;		/* network part of the inet address */
    int hostAddr;		/* host part of the inet address */
    struct in_addr *pInetAddr;	/* where to return the inet address */

    {
    register u_long addr;

    if (netAddr < 128)
	addr = (netAddr << IN_CLASSA_NSHIFT) | (hostAddr & IN_CLASSA_HOST);
    else if (netAddr < 65536)
	addr = (netAddr << IN_CLASSB_NSHIFT) | (hostAddr & IN_CLASSB_HOST);
    else
	addr = (netAddr << IN_CLASSC_NSHIFT) | (hostAddr & IN_CLASSC_HOST);

    pInetAddr->s_addr = htonl (addr);
    }
/*******************************************************************************
*
* inet_makeaddr - construct an Internet address from network + host numbers
*
* Given the network number and local host address the Internet address is
* formed.
*
* WARNING
* Supplied for UNIX compatibility only.
* Each time this routine is called 4 bytes are malloc'd.
* Use inet_makeaddr_b (2) instead.
*
* EXAMPLE
* .CS
*  inet_makeaddr (0x5a, 2);
* .CE
* returns the address 0x5a000002 in the newly malloc'd in_addr structure. 
*
* RETURNS: in_addr containing network address
*
* SEE ALSO: inet_makeaddr_b (2)
*/

struct in_addr inet_makeaddr (netAddr, hostAddr)
    int netAddr;
    int hostAddr;

    {
    struct in_addr *pAddr = (struct in_addr *) malloc (sizeof (struct in_addr));

    if (pAddr != NULL)
	inet_makeaddr_b (netAddr, hostAddr, pAddr);

    return (*pAddr);
    }
/*******************************************************************************
*
* inet_netof - return the network number from an Internet address
*
* This routine extracts the network portion of an Internet address.
*
* EXAMPLE
* .CS
*  inet_netof (0x5a000002);
* .CE
* returns 0x5a.
*
* RETURNS: network portion of `inetAddress'
*/

int inet_netof (inetAddress)
    struct in_addr inetAddress;	/* the inet address */

    {
    register u_long i = ntohl ((u_long) inetAddress.s_addr);

    if (IN_CLASSA (i))
	return (((i)&IN_CLASSA_NET) >> IN_CLASSA_NSHIFT);
    else if (IN_CLASSB (i))
	return (((i)&IN_CLASSB_NET) >> IN_CLASSB_NSHIFT);
    else
	return (((i)&IN_CLASSC_NET) >> IN_CLASSC_NSHIFT);
    }
/*******************************************************************************
*
* inet_netof_string - local network extractor in dot notation
*
* Calculate the dot notation network Internet address from a host's 
* dot notation Internet address.  Handles class A, B and C network addresses.
* The buffer `netString' should be INET_ADDR_LEN bytes long.
*
* NOTE
* This routine handles subnet masks correctly, whereas the
* other routines in this library, do not.
*
* EXAMPLE
* .CS
*  inet_netof_string ("90.0.0.2", netString);
* .CE
* returns "90.0.0.0" in `netString'.
*/

VOID inet_netof_string (inetString, netString)
    char *inetString;	/* Internet address to extract local portion from */
    char *netString;	/* net Internet address to return */

    {
    struct in_addr inaddrHost;
    struct in_addr inaddrNet;
    
    /* convert string to u_long */

    inaddrHost.s_addr = inet_addr (inetString);

    /* extract network portion from host, put it in inaddrNet;
     * note that in_netof and in_makeaddr_b are used (instead of inet_...)
     * because they know about subnet masks. */

    in_makeaddr_b (in_netof (inaddrHost), (int) INADDR_ANY, &inaddrNet);

    /* convert network portion to dot notation */

    inet_ntoa_b (inaddrNet, netString);
    }
/*******************************************************************************
*
* inet_network - convert Internet network number from string to address
*
* Given an ASCII Internet network number, an network address is constructed.
*
* EXAMPLE
* .CS
*  inet_network ("90");
* .CE
* returns 0x5a.
*
* RETURNS: Internet address version of address
*/

u_long inet_network (inetString)
    register char *inetString;		/* string version of inet addrs */

    {
    register u_long val, base, n;
    register char c;
    u_long parts[4], *pp = parts;
    register int i;

again:
    val = 0;
    base = 10;

    if (*inetString == '0')
	base = 8, inetString++;
    if (*inetString == 'x' || *inetString == 'X')
	base = 16, inetString++;

    while (c = *inetString)
	{
	if (isdigit (c))
	    {
	    val = (val * base) + (c - '0');
	    inetString++;
	    continue;
	    }
	if (base == 16 && isxdigit (c))
	    {
	    val = (val << 4) + (c + 10 - (islower (c) ? 'a' : 'A'));
	    inetString++;
	    continue;
	    }
	break;
	}

    if (*inetString == '.')
	{
	if (pp >= parts + 4)
	    {
	    errnoSet (S_inetLib_ILLEGAL_NETWORK_NUMBER);
	    return (ERROR);
	    }
	*pp++ = val, inetString++;
	goto again;
	}

    if (*inetString && !isspace (*inetString))
	{
	errnoSet (S_inetLib_ILLEGAL_NETWORK_NUMBER);
	return (ERROR);
	}

    *pp++ = val;
    n = pp - parts;

    if (n > 4)
	{
	errnoSet (S_inetLib_ILLEGAL_NETWORK_NUMBER);
	return (ERROR);
	}

    for (val = 0, i = 0; i < n; i++)
	{
	val <<= 8;
	val |= parts[i] & 0xff;
	}

    return (val);
    }
/*******************************************************************************
*
* inet_ntoa_b - convert network dot notation address to ASCII
*
* Convert network-format Internet address to base 256 d.d.d.d representation.
*
* This routine is identical to the UNIX inet_ntoa (2) routine
* except that a buffer of size INET_ADDR_LEN must be provided.
*
* EXAMPLE
* .CS
*  inet_ntoa_b (0x5a0000002, pString);
* .CE
* returns the string "90.0.0.2" in `pString'.
*/

VOID inet_ntoa_b (inetAddress, pString)
    struct in_addr inetAddress;
    char *pString;		/* where to return ASCII string */

    {
    register char *p = (char *)&inetAddress;

#define	UC(b)	(((int)b)&0xff)
    sprintf (pString, "%d.%d.%d.%d", UC(p[0]), UC(p[1]), UC(p[2]), UC(p[3]));
    }
/*******************************************************************************
*
* inet_ntoa - convert network dot notation address to ASCII
*
* Convert network-format Internet address to base 256 d.d.d.d representation.
*
* WARNING
* Supplied for UNIX compatibility only.
* Each time this routine is called 18 bytes are malloc'd.
* Use inet_ntoa_b (2) instead.
*
* EXAMPLE
* .CS
*  inet_ntoa (0x5a0000002);
* .CE
* returns a pointer to the newly malloc'd string "90.0.0.2".
*
* RETURNS: pointer to string version of Internet address
*
* SEE ALSO: inet_ntoa_b (2)
*/

char *inet_ntoa (inetAddress)
    struct in_addr inetAddress;

    {
    FAST char *buf = malloc (INET_ADDR_LEN);

    if (buf != NULL)
	inet_ntoa_b (inetAddress, buf);

    return (buf);
    }
