/* hostLib.c - host table subroutine library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/* 
modification history
--------------------
01g,23May89,shl  changed addr parameter in hostGetByAddr() from char* to int.
01f,21apr89,shl  added hostGetByAddr().
		 added hostDelete ().
01e,18mar88,gae  documentation.
01d,07jul88,jcf  fixed malloc to match new declaration.
01c,04jun88,gae  changed S_remLib_* to S_hostLib_*.
01b,30may88,dnw  changed to v4 names.
01a,28may88,dnw  extracted from remLib.
*/

/*
DESCRIPTION
This module provides routines to store and access the network host database.
The host table contains information regarding the known hosts on the
local network.  The host table (displayed with hostShow (2)) contains
the internet address, the official host name, and aliases.

By convention network addresses are specified in a dot (`.') notation.
The library inetLib (1) contains internet address manipulation routines.
Host names, and aliases, may contain any printable character.

SEE ALSO
inetLib (1), "Network"
*/

#include "vxWorks.h"
#include "types.h"
#include "in.h"
#include "memLib.h"
#include "semLib.h"
#include "strLib.h"
#include "inetLib.h"
#include "lstLib.h"
#include "hostLib.h"

typedef struct hostname_struct
    {
    struct hostname_struct *link;
    char *name;
    } HOSTNAME;

typedef struct
    {
    NODE node;
    HOSTNAME hostName;
    struct in_addr netAddr;
    } HOSTENTRY;

LOCAL LIST hostList;
LOCAL SEM_ID hostListSemId;


/*******************************************************************************
*
* hostTblInit - initialize network host table
*
* This routine must be called to initialize the host list data structure
* used by routines throughout this module.  It should be called by
* usrRoot (2) in usrConfig (1), before any other routines in this module.
*/

VOID hostTblInit ()

    {
    static BOOL done;	/* FALSE = not done, TRUE = already initialized */

    if (!done)
	{
	done = TRUE;
	lstInit (&hostList);
	hostListSemId = semCreate ();
	semGive (hostListSemId);
	}
    }
/*******************************************************************************
*
* hostAdd - add a host to the host table
*
* This command is used to add a hostname to the local host table.
* This must be called before sockets on the remote host are opened,
* or files on the remote host are accessed via netDrv (3) or nfsDrv (3).
*
* The host table has one entry per internet address.
* More than one name may be used for an address.
* Additional host names are added as aliases.
*
* EXAMPLE
* .CS
*    -> hostAdd "wrs", "90.2"
*    -> hostShow
*    hostname         inet address       aliases
*    --------         ------------       -------
*    localhost        127.0.0.1         
*    yuba             90.0.0.3          
*    wrs              90.0.0.2          
*    value = 12288 = 0x3000 = _bzero + 0x18
*    -> 
* .CE
* RETURNS:
*  OK, or
*  ERROR if host table full,
*  host name already entered,
*  invalid inet address, or
*  out of memory
*/

STATUS hostAdd (hostName, hostAddr)
    char *hostName;	/* host name */
    char *hostAddr;	/* host address in standard internet format */

    {
    HOSTNAME *pHostNamePrev;	/* pointer to previous host name entry */
    FAST HOSTNAME *pHostName;	/* pointer to host name entry */
    FAST HOSTENTRY *pHostEntry;
    struct in_addr netAddr;	/* network address */

    if ((netAddr.s_addr = inet_addr (hostAddr)) == ERROR)
	return (ERROR);
    
    semTake (hostListSemId);

    for (pHostEntry = (HOSTENTRY *)lstFirst (&hostList);
	 pHostEntry != NULL;
	 pHostEntry = (HOSTENTRY *)lstNext (&pHostEntry->node))
	{
	if (pHostEntry->netAddr.s_addr == netAddr.s_addr)
	    {
	    /* host internet address already in table, add name as an alias */

	    for (pHostName = &pHostEntry->hostName;
		 pHostName != NULL;
		 pHostName = pHostName->link)
		{
		/* make sure name is not already used for this address */

		if (strcmp (pHostName->name, hostName) == 0)
		    {
		    semGive (hostListSemId);
		    errnoSet (S_hostLib_HOST_ALREADY_ENTERED);
		    return (ERROR);
		    }
	
		pHostNamePrev = pHostName;
		}
	    
	    /* name not used for this address, add it as an alias */

	    pHostNamePrev->link = (HOSTNAME *) malloc (sizeof (HOSTNAME));

	    if (pHostNamePrev->link == NULL ||
		hostNameFill (pHostNamePrev->link, hostName) == ERROR)
		{
		semGive (hostListSemId);
		return (ERROR);
		}

	    semGive (hostListSemId);
	    return (OK);
	    }
	}

    /* host name and internet address not in host table, add new host */

    if ((pHostEntry = (HOSTENTRY *)malloc (sizeof (HOSTENTRY))) == NULL ||
	hostNameFill (&pHostEntry->hostName, hostName) == ERROR)
	{
	semGive (hostListSemId);
	return (ERROR);
	}

    pHostEntry->netAddr = netAddr;

    lstAdd (&hostList, &pHostEntry->node);

    semGive (hostListSemId);
    return (OK);
    }
/*******************************************************************************
*
* hostDelete - delete a host from host table.
*
* This command is used to delete a hostname from the local host table.
* The host entry is deleted if hostname is used as argument. 
* The alias of the hostname is deleted if alias is used as argument.
*
* RETURNS
*    OK, or
*    ERROR if host unknown
*
*/

STATUS hostDelete (name, addr)
    char *name;	/* host name or alias */
    char *addr;	/* host address in standard internet format */

    {
    HOSTNAME *pHostNamePrev;	/* pointer to previous host name entry */
    HOSTNAME *pHostNameNext;	/* pointer to next host name entry */
    FAST HOSTNAME *pHostName;
    FAST HOSTENTRY *pHostEntry;
    struct in_addr netAddr;

    /* convert from string to int format */
    if ((netAddr.s_addr = inet_addr (addr)) == ERROR)
	return ERROR;

    semTake (hostListSemId);

    /* search inet address */
    for (pHostEntry = (HOSTENTRY *)lstFirst (&hostList);
	 pHostEntry != NULL;
	 pHostEntry = (HOSTENTRY *)lstNext (&pHostEntry->node))
        {
	if ((pHostEntry->netAddr.s_addr == netAddr.s_addr) &&
            (strcmp (pHostEntry->hostName.name, name) == 0))
		{
		lstDelete (&hostList, &pHostEntry->node);
		semGive (hostListSemId);
		free ((char *) pHostEntry);
		return (OK);
		 }
	else
	    {
	    pHostNamePrev = &pHostEntry->hostName;

	    if (pHostEntry->netAddr.s_addr == netAddr.s_addr)
		{
		for (pHostName = &pHostEntry->hostName;	/*search aliases */
		     pHostName != NULL;
		     pHostName = pHostName->link)
		    {	
		    pHostNameNext = pHostName->link;

		    if (strcmp (pHostName->name, name) == 0) /* compare alias */
			{
			pHostNamePrev->link = pHostNameNext;
			semGive (hostListSemId);
			free ((char *) pHostName);
			return (OK);
			}
	            pHostNamePrev = pHostName;			
		    }
                }
	    }
	}

    errnoSet (S_hostLib_UNKNOWN_HOST);

    semGive (hostListSemId);
    return (ERROR);
    }
/*******************************************************************************
*
* hostGetByName - look up a host in the host table by its name.
*
* This routine returns the internet address of a host that has
* been added to the host table by hostAdd (2).
*
* RETURNS
*  internet address (as an int), or
*  ERROR if host unknown
*/

int hostGetByName (name)
    char *name;		/* name of host */

    {
    FAST HOSTNAME *pHostName;
    FAST HOSTENTRY *pHostEntry;
    int retAddr = ERROR;

    semTake (hostListSemId);

    for (pHostEntry = (HOSTENTRY *)lstFirst (&hostList);
	 pHostEntry != NULL;
	 pHostEntry = (HOSTENTRY *)lstNext (&pHostEntry->node))
	{
	/* check official name */

	if (strcmp (pHostEntry->hostName.name, name) == 0)
	    {
	    retAddr = (int)pHostEntry->netAddr.s_addr;
	    break;
	    }

	/* check aliases */

	for (pHostName = pHostEntry->hostName.link;
	     pHostName != NULL;
	     pHostName = pHostName->link)
	    {
	    if (strcmp (pHostName->name, name) == 0)
		{
		retAddr = (int)pHostEntry->netAddr.s_addr;

		/* force termination of outer loop */
		pHostEntry = (HOSTENTRY *)lstLast (&hostList);
		break;
		}
	    }
	}

    if (retAddr == ERROR)
	errnoSet (S_hostLib_UNKNOWN_HOST);

    semGive (hostListSemId);
    return (retAddr);
    }
/*******************************************************************************
*
* hostGetByAddr - look up a host in the host table by its internet address
*
* This routine finds the host name by its internet address and put it in the
* buffer name which should be preallocated with (MAXHOSTNAMELEN + 1) bytes of
* memory. The returned name is null-terminated unless insufficient space is 
* provided.
*
* RETURNS
*  OK, or
*  ERROR if host unknown 
*
* WARNING 
*  hostGetByAddr does not look for aliases. 
*  host names are limited to MAXHOSTNAMELEN (from hostLib.h) characters,
*  currently 64.
*
* SEE ALSO
*  hostGetByName (2)
*/

STATUS hostGetByAddr (addr, name)
    int addr;		/* inet address of host */
    char *name; 	/* buffer to hold the name */

    {
    FAST HOSTENTRY *pHostEntry;
    struct in_addr netAddr; 
    STATUS status = ERROR;
    int n;

    netAddr.s_addr = addr;

    semTake (hostListSemId);

    /* search for internet address */
    for (pHostEntry = (HOSTENTRY *)lstFirst (&hostList);
	 pHostEntry != NULL;
	 pHostEntry = (HOSTENTRY *)lstNext (&pHostEntry->node))
	{
	if (pHostEntry->netAddr.s_addr == netAddr.s_addr)
	    {
	    n = strlen (pHostEntry->hostName.name);
	    
	    strncpy (name, pHostEntry->hostName.name,	
			min (n + 1 , MAXHOSTNAMELEN +1));
	    status = OK;
	    break;
	    }
  	}
    semGive (hostListSemId);

    if (status != OK)
	errnoSet (S_hostLib_UNKNOWN_HOST);

    return (status);
    }
/*******************************************************************************
*
* hostShow - show the host table
*
* Prints a list of remote hosts along with their internet addresses and aliases.
*
* SEE ALSO: hostAdd (2)
*/

VOID hostShow ()

    {
    char inetBuf [INET_ADDR_LEN];
    FAST HOSTNAME *pHostName;
    FAST HOSTENTRY *pHostEntry;

    printf ("hostname         inet address       aliases\n");
    printf ("--------         ------------       -------\n");

    semTake (hostListSemId);

    for (pHostEntry = (HOSTENTRY *)lstFirst (&hostList);
	 pHostEntry != NULL;
	 pHostEntry = (HOSTENTRY *)lstNext (&pHostEntry->node))
	{
	inet_ntoa_b (pHostEntry->netAddr, inetBuf);
	printf ("%-16s %-18s", pHostEntry->hostName.name, inetBuf);

	for (pHostName = pHostEntry->hostName.link;
	     pHostName != NULL;
	     pHostName = pHostName->link)
	    {
	    printf ("%s ", pHostName->name);
	    }

	printf ("\n");
	}

    semGive (hostListSemId);
    }
/*******************************************************************************
*
* hostNameFill - fill in host name
*
* RETURNS: OK or ERROR if out of memory
*/

LOCAL STATUS hostNameFill (pHostName, newHostName)
    FAST HOSTNAME *pHostName;
    char *newHostName;

    {
    FAST char *pName = malloc ((unsigned) (strlen (newHostName) + 1));

    if (pName == NULL)
	return (ERROR);

    strcpy (pName, newHostName);
    pHostName->name = pName;
    pHostName->link = NULL;

    return (OK);
    }
