/* floatLib.c - floating point formatting & conversion subroutine library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01g,20apr89,dab  fixed precision bug in cvtb(). 
		 documentation touchup in fcvtb() and cvtb().
01f,05jun88,dnw  changed name from fltLib to floatLib.
01e,30may88,dnw  changed to v4 names.
01d,28may88,dnw  changed atof() to fltAtoF().
01c,17mar88,gae  added 'E' and 'G' format specifiers.
01b,05nov87,jlf  documentation
01a,01aug87,gae  written/extracted from fioLib.c.
*/

/*
DESCRIPTION
This library provides all the floating-point formatting and conversion
routines to be used in fioLib(1), and some related floating-point routines.
To allow re-entrancy ecvtb and fcvtb are provided in place of
UNIX ecvt and fcvt.

This library is optional and may be omitted by removing the call to
floatInit in usrRoot(2), in usrConfig(1).  If omitted, format
specifications in printf(2) and scanf(2) will not be supported.
*/

#include "vxWorks.h"
#include "ctype.h"
#include "strLib.h"
#include "vwModNum.h"
#include "errno.h"

IMPORT char *nindex();

#define	IEEE		/* support IEEE */
#define	NDIG	350	/* 350 to handle IEEE stuff */

typedef union	/* DOUBLE */
    {
    double ddat;
    struct
        {
        long l1;
        long l2;
        } ldat;
    } DOUBLE;

/* forward declarations */

char *ecvtb ();
char *fcvtb ();
char *gcvt ();
LOCAL int floatAtoF ();
LOCAL char *cvtb ();
LOCAL int floatFormat ();


/*******************************************************************************
*
* floatInit - initialize floating-point I/O support
*
* This routine must be called if floating-point format specifications are
* to be supported in printf(2) and scanf(2).
* It is usually called by the root task, usrRoot(2), in usrConfig(1).
*/

VOID floatInit ()
    {
    fioFltInstall (floatFormat, floatAtoF);
    }
/*******************************************************************************
*
* floatFormat - format arg for output
*
* This routine converts from a floating point value to an ASCII representation.
* The `spec' parameter indicates the format type desired; the actual work
* is done ecvtb, fcvtb, and gcvt.
*
* RETURNS: number of characters placed in buffer
*/

LOCAL int floatFormat (spec, arg, precision, buffer, bufsize)
    char spec;		/* format specification: 'e', 'f', 'g'	*/
    double arg;		/* value to be formatted		*/
    int precision;	/* number of digits to format		*/
    char buffer [];	/* where to put output			*/
    int bufsize;	/* maximum size to format		*/

    {
    char *pBuf;
    int length = 0;

    switch (spec)
	{
	case 'e':
	case 'E':
	    length = etob (buffer, arg, bufsize, precision);
	    break;

	case 'f':
	    length = ftob (buffer, arg, bufsize, precision);
	    break;

	case 'g':
	case 'G':
	    length = strlen (gcvt (arg, precision, buffer));
	    break;
	}

    if ((spec == 'E' || spec == 'G') &&
	(pBuf = nindex (buffer, 'e', length)) != NULL)
	*pBuf = 'E';

    return (length);
    }
/*******************************************************************************
*
* floatAtoF - convert an ASCII string to a float
*
* This routine converts an ASCII string, in character buffer `s' to a double.
* The double is returned in the location pointed to by `fptr'.
* A maximum of `width' characters is used, with a zero implying no limit.
* Atof recognizes an optional string of spaces, then an optional sign,
* then a string of digits optionally containing a decimal point,
* then an optional `e' or `E' followed by an optionally signed integer.
*
* RETURNS: The number of characters consumed.
*/

LOCAL int floatAtoF (s, fptr, width)
    char s [];		/* ASCII string containing number	*/
    double *fptr;	/* where to put converted float		*/
    int width;		/* max. number of characters to consume	*/

    {
    /* Any integer <= maxd can be represented exactly as a double,
     * where maxd = *(double *)imaxd
     * Note: maxd = 9007199254740991 in decimal
     */
    static int imaxd[]	= {0x433fffff, 0xffffffff};
    static double pow10[]={10., 100., 1e4, 1e8, 1e16, 1e32, 1e64, 1e128, 1e256};
    static double digits[10] = {0., 1., 2., 3., 4., 5., 6., 7., 8., 9.};
    static double ten	= 10.;
    BOOL intok	= TRUE;
    int needinv;
    int eexp;
    int neexp;
    int exp	= 0;
    int needneg = 0;
    int maxi	= (0x7fffffff - 9)/10;
    double exp10;
    double d	= 0;
    double maxd	= *(double *)imaxd;
    char *sb = s;
    FAST char c;
    FAST int id	= 0;

    if (width == 0)
	width = maxi;

    while (isspace (c = *s))
	s++;

    switch (c)
	{
	case '-':
	    needneg = 1;
	case '+':
	    s++;
	    break;
	}

    while ((c = *s++) && (s - sb < width))
	{
	if (c >= '0' && c <= '9')
	    {
	    if (intok)
		{
		if (id < maxi)
		    id = id * 10 + (c - '0');
		else
		    {
		    intok = FALSE;
		    d = id;
		    d = ten * d + digits [c - '0'];
		    }
		}
	    else
		{
		if (d < maxd)
		    d = ten * d + digits [c - '0'];
		else
		    exp++;
		}
	    }
	else
	    break;
	}

    if (c == '.')
	{
	while ((c = *s++) && (s - sb < width))
	    {
	    if (c >= '0' && c <= '9')
		{
		if (intok)
		    {
		    if (id < maxi)
			{
			id = id * 10 + (c - '0');
			exp--;
			}
		    else
			{
			intok = FALSE;
			d = id;
			d = ten * d + digits [c - '0'];
			exp--;
			}
		    }
		else
		    {
		    if (d < maxd)
			{
			d = ten * d + digits [c - '0'];
			exp--;
			}
		    }
		}
	    else
		break;
	    }
	}

    if (intok)
	d = id;

    eexp = 0;
    neexp = 0;

    if (c == 'e' || c == 'E')
	{
	switch (c = *s)
	    {
	    case '-':
		neexp = 1;
	    case '+':
		s++;
	    }

	while ((c = *s++) >= '0' && c <= '9' && (s - sb < width))
	    eexp = 10 * eexp + (c - '0');
	}

    if (neexp)
	exp -= eexp;
    else
	exp += eexp;

    needinv = 0;

    if (exp < 0)
	{
	needinv = 1;
	exp = -exp;
	}

    exp10 = 1.;

    for (id = 0; exp && id <= 8; exp >>= 1, id++)
	{
	if (exp & 1)
	    exp10 *= pow10 [id];
	}

    if (needinv)
	d /= exp10;
    else
	d *= exp10;

    if (needneg)
	d = -d;

    *fptr = d;

    return (s - sb);
    }
/*******************************************************************************
*
* ftob - convert float to buffer
*
* This routine converts from a floating point value to an ASCII representation.
* The value will have nDecimal digits to the right of the decimal point.
* The value will be prepended with a `-' if necessary.
* The value will not be terminated with an EOS.
*
* EXAMPLE
*     nchars = ftob (myBuf, -50.3, 10, 2);
*
* leaves the string "-50.30" (with no EOS) in myBuf, and returns the value 2.
*
* RETURNS: The number of characters put in the caller's buffer.
*/

LOCAL int ftob (buffer, value, maxChars, nDecimal)
    FAST char *buffer;	/* buffer to receive converted ASCII		*/
    double value;	/* value to convert				*/
    int maxChars;	/* max. number of characters to put in buffer	*/
    FAST nDecimal;	/* number of digits to right of decimal point	*/

    {
    char fltbuf [NDIG];
    int decpt;
    int sign;
    int charsNeeded;
    FAST int nchars = 0;

    ecvtb (value, sizeof (fltbuf), &decpt, &sign, fltbuf);

    charsNeeded = nDecimal + 2 + ((sign == 0) ? 0 : 1);

    if ((decpt > maxChars	&& sign == 0)	||
	((decpt > maxChars - 1) && sign != 0)	||
	(decpt < 0 && charsNeeded > maxChars))
	{
	/* won't fit! */

	bfill (buffer, maxChars, '*');
	return (maxChars);
	}

    /* Fill the buffer */

    if (sign != 0)
	buffer [nchars++] = '-';	/* negative number */
	
    if (decpt < 0)
	{
	/* it's a fractional number */

	buffer [nchars++] = '0';
	buffer [nchars++] = '.';
	decpt = -decpt;
	bfill (&buffer [nchars], min (decpt, nDecimal), '0');
	nchars += min (decpt, nDecimal);

	if (nDecimal > decpt)
	    {
	    bcopy (fltbuf, &buffer [nchars], nDecimal - decpt);
	    nchars += (nDecimal - decpt);
	    }
	}
    else
	{
	/* number > 1 */

	bcopy (fltbuf, &buffer [nchars], decpt);
	nchars += decpt;

	if (nchars < maxChars && nDecimal > 0)
	    {
	    buffer [nchars++] = '.';
	    bcopy (&fltbuf [decpt], &buffer [nchars],
		   min (nDecimal, maxChars - nchars));
	    nchars += min (nDecimal, maxChars - nchars);
	    }
	}

    return (nchars);
    }
/*******************************************************************************
*
* etob - convert float to buffer in mantissa/exponent format
*
* This routine converts from a floating point value to an ASCII representation.
* The value will have nDecimal digits to the right of the decimal point.
* The value will be prepended with a `-' if necessary.
* The value will not be terminated with an EOS.
*
* EXAMPLE
*     nchars = etob (myBuf, -50.3, 10, 3);
*
* leaves the string "-5.030E1" (with no EOS) in myBuf, and returns the value 2.
*
* RETURNS: The number of chars put in the caller's buffer.
*/

LOCAL int etob (buffer, value, maxChars, nDecimal)
    FAST char *buffer;	/* buffer to receive converted ASCII value	*/
    double value;	/* value to convert				*/
    int maxChars;	/* max. number of characters to put in buffer	*/
    FAST nDecimal;	/* number of digits to right of decimal point	*/

    {
    char fltbuf [NDIG];
    char expbuf [30];
    int explength;
    int decpt;
    int sign;
    int charsNeeded;
    FAST int nchars = 0;

    ecvtb (value, sizeof (fltbuf), &decpt, &sign, fltbuf);

    /* See if the buffer is big enough */

    charsNeeded = nDecimal + 2 + ((sign == 0) ? 0 : 1);
    explength = itob (expbuf, decpt-1, -10);
    charsNeeded += explength + 1;

    if (charsNeeded > maxChars)
	{
	/* won't fit! */

	bfill (buffer, maxChars, '*');
	return (maxChars);
	}

    /* Fill the buffer */

    if (sign != 0)
	buffer [nchars++] = '-';	/* negative number */
	
    buffer [nchars++] = fltbuf [0];

    if (nDecimal != 0)
	{
	buffer [nchars++] = '.';
	bcopy (&fltbuf [1], &buffer [nchars], nDecimal);
	nchars += nDecimal;
	}

    buffer [nchars++] = 'e';
    bcopy (expbuf, &buffer [nchars], explength);
    nchars += explength;

    return (nchars);
    }
/*******************************************************************************
*
* frexp - split into mantissa and exponent
*
* frexp returns the mantissa of the double `value' as a double
* quantity, x, of magnitude less than 1.  It stores an integer
* n such that value = x*2**n indirectly through `eptr'.
*/

double frexp (value, eptr)
    double value;	/* value to split	*/
    int *eptr;		/* returned n		*/

    {
    DOUBLE dat;

    dat.ddat = value;
    *eptr = ((dat.ldat.l1 & 0x7ff00000) >> 20) - 1022;
    dat.ldat.l1 = (dat.ldat.l1 & ~0x7ff00000) | 0x3fe00000;

    return (dat.ddat);
    }
/*******************************************************************************
*
* ldexp - split into mantissa and exponent
* 
* ldexp returns the quantity value*2**exp.
*/

double ldexp (value, exp)
    double value;	/* value to split	*/
    int exp;		/* desired exponent	*/

    {
    int vexp;
    DOUBLE dat;

    dat.ddat = value;
    vexp = (dat.ldat.l1 & 0x7ff00000) >> 20;
    vexp += exp;

    if (vexp > 0x7ff) 
        {
	/* deal with too large an exponent */

        errnoSet (M_errno | EDOM);

        /* Figure out later what to do with denormalized */

        if (dat.ldat.l1 & 0x80000000)
            dat.ldat.l1 = 0xffffffff;
        else
            dat.ldat.l1 = 0x7fffffff;
        dat.ldat.l2 = 0xffffffff;
        return (dat.ddat);
        }
    else if (vexp < 0) 
        {
	/* deal with too small an exponent */

        errnoSet (M_errno | EDOM);

        if (dat.ldat.l1 & 0x80000000)
            return (-0.);
        else
            return (0.);
        }
    else 
        {
        dat.ldat.l1 &= ~0x7ff00000;
        dat.ldat.l1 |= (vexp << 20);
        return (dat.ddat);
        }
    }
/*******************************************************************************
*
* modf - split into mantissa and exponent
* 
* modf returns the positive fractional part of value and
* stores the integer part indirectly through `iptr'.
*/

double modf (value, iptr)
    double value;		/* value to split		*/
    double *iptr;		/* returned integer part	*/

    {
    DOUBLE dat;
    int exp;
    double fracPart;
    double intPart;
    int negflag = (value < 0);

    if (negflag) 
        value = -value;		/* make it positive */

    dat.ddat = value;

    /* Separate the exponent, and change it from biased to 2's comp. */

    exp = ((dat.ldat.l1 & 0x7ff00000) >> 20) - 1023;

    if (exp <= -1) 
        {
	/* If exponent is negative, fracPart == |value|, and intPart == 0. */

        fracPart = value;
        intPart = 0.;
        }
    /* clear the fractional part in dat */

    else if (exp <= 20) 
        {
        dat.ldat.l1 &= (-1 << (20 - exp));
        dat.ldat.l2 = 0;
        intPart = dat.ddat;
        fracPart = value - intPart;
        }

    else if (exp <= 52) 
        {
        dat.ldat.l2 &= (-1 << (52 - exp));
        intPart = dat.ddat;
        fracPart = value - intPart;
        }

    else 
        {
        fracPart = 0.;
        intPart = value;
        }

    *iptr = (negflag ? -intPart : intPart);
    return (negflag ? -fracPart : fracPart);
    }

/*******************************************************************************
*
* ecvtb - convert floating point to ASCII representation
*
* ecvtb converts a floating point number to an ASCII decimal representation.
* The number of digits is specified by ndigit.
* The position of the decimal point relative to the beginning of the
* string is stored indirectly through `decpt'.
* If `decpt' is negative then the decimal point is to the left of
* the returned digits.
* If the sign of `value' is negative, the word pointed to by sign
* is non-zero.  The low-order digit is rounded.
*
* This routine is identical to the UNIX ecvt routine, except that the
* buffer must be passed.  Therefore, this routine is re-entrant, while
* UNIX ecvt is not.
*
* RETURNS: Pointer to buffer, for UNIX similarity.
*/

char *ecvtb (value, ndigit, decpt, sign, buffer)
    double value;	/* value to be converted			*/
    int ndigit;		/* number of digits to place into buffer	*/
    int *decpt;		/* position of decimal point in buffer		*/
    int *sign;		/* 0 == positive, 1 == negative			*/
    char *buffer;	/* where to put ASCII conversion of arg	+ EOS	*/

    {
    return (cvtb (value, ndigit, decpt, sign, TRUE, buffer));
    }
/*******************************************************************************
*
* fcvtb - convert floating point to ASCII representation (Fortran-style)
*
* fcvtb is identical to ecvtb (2), except that the correct digit has
* been rounded for Fortran F-format style output with `ndigit' digits
* to the right of the decimal point.  `ndigit' can be negative to
* indicate rounding to the left of the decimal point.
*
* RETURNS: Pointer to buffer, for UNIX similarity.  `buffer' should
* contain at least 350 (for IEEE support) + max (0, `ndigit') characters
* to accomodate any double-precision value.
*/

char *fcvtb (value, ndigit, decpt, sign, buffer)
    double value;	/* value to be converted			*/
    int ndigit;		/* number of digits before or after decimal point */
    int *decpt;		/* position of decimal point in buffer		*/
    int *sign;		/* 0 == positive, 1 == negative			*/
    char *buffer;	/* where to put ASCII conversion of arg	+ EOS	*/

    {
    return (cvtb (value, ndigit, decpt, sign, FALSE, buffer));
    }
/*******************************************************************************
*
* gcvt - convert floating point to ASCII representation (e- or f-style)
*
* gcvt converts the value to a null-terminated ASCII string in buffer.
* It attempts to produce `ndigit' significant digits in Fortran F-format
* style output if possible; otherwise Fortran E-format, ready for printing.
* Trailing zeros may be suppressed.
*
* This routine is UNIX compatible.
*
* RETURNS: Pointer to buffer.
*/

char *gcvt (value, ndigit, buffer)
    double value;	/* value to be converted			*/
    int ndigit;		/* number of digits to place into buffer	*/
    char *buffer;	/* where to put ASCII conversion of arg	+ EOS	*/

    {
    char fltbuf [NDIG];
    int sign;
    int decpt;
    FAST i;
    FAST char *p1 = fltbuf;
    FAST char *p2 = buffer;

    ecvtb (value, ndigit, &decpt, &sign, fltbuf);

    if (sign)
	*p2++ = '-';

    for (i = ndigit-1; i > 0 && p1[i] == '0'; i--)
	ndigit--;

    if (decpt >= 0 && (decpt - ndigit) > 4 || decpt < 0 && decpt < -3)
	{
	/* use E-style */

	decpt--;

	*p2++ = *p1++;
	*p2++ = '.';

	for (i = 1; i < ndigit; i++)
	    *p2++ = *p1++;

	*p2++ = 'e';
	if (decpt < 0)
	    {
	    decpt = -decpt;
	    *p2++ = '-';
	    }
	else
	    *p2++ = '+';

#ifdef IEEE
	if (decpt >= 100)
	    {
	    *p2++ = decpt / 100 + '0';
	    decpt = decpt % 100;
	    }
#endif IEEE

	*p2++ = decpt / 10 + '0';
	*p2++ = decpt % 10 + '0';
	}
    else 
	{
	if (decpt <= 0)
	    {
	    if (*p1 != '0')
		*p2++ = '.';

	    while (decpt < 0)
		{
		decpt++;
		*p2++ = '0';
		}
	    }

	for (i = 1; i <= ndigit; i++)
	    {
	    *p2++ = *p1++;

	    if (i == decpt)
		*p2++ = '.';
	    }

	if (ndigit < decpt)
	    {
	    while (ndigit++ < decpt)
		*p2++ = '0';

	    *p2++ = '.';
	    }
	}

    if (p2[-1] == '.')
	p2--;

    *p2 = '\0';

    return (buffer);
    }
/*******************************************************************************
*
* cvtb - convert floating point to ASCII representation
*
* This is the support routine for ecvt and fcvt.
* cvtb converts a floating point number to an ASCII decimal representation.
* For ecvt `ndigits' specifies the total number of digits returned, while
* for fcvt `ndigits' specifies the number of digits to the right of the
* decimal point returned.
* The position of the decimal point relative to the beginning
* of the string is stored indirectly through `decpt'.
* If `decpt' is negative then the decimal point is to the left of
* the returned digits.
* If the sign of value is negative, the word pointed to by `sign' is non-zero.
* The low-order digit is rounded.
* If `eflag' is FALSE then use Fortran F-format style output.
*
* NOTE: Buffer size should be at least ndigit + 1 for terminating EOS if
* eflag is TRUE.  Otherwise buffer size should be 350 (IEEE) + ndigit + 1 for
* terminating EOS.
*
* RETURNS: Pointer to buffer.
*/

LOCAL char *cvtb (value, ndigit, decpt, sign, eflag, buffer)
    double value;	/* value to be converted			*/
    int ndigit;		/* number of digits to place into buffer	*/
    int *decpt;		/* position of decimal point in buffer		*/
    int *sign;		/* 0 == positive, 1 == negative			*/
    BOOL eflag;		/* FALSE == Fortran style F-format		*/
    char *buffer;	/* where to put ASCII conversion of value + EOS	*/

    {
    char fltbuf [NDIG];
    double intPart;
    double fracPart;
    FAST int r2	  = 0;
    FAST char *p  = &fltbuf [0];
    FAST char *p1 = &fltbuf [NDIG];

#ifdef	IEEE
    if ((*(int *) &value & 0x7ff00000) == 0x7ff00000) 
        {
        if (((*(int *) &value & 0xfffff) == 0) && (*((int *) &value + 1) == 0)) 
            {
            if (*(int *) &value > 0)
                strcpy (fltbuf, "Inf");		/* Infinity */
            else
                strcpy (fltbuf, "-Inf");	/* Negative infinity */
            }
        else
            strcpy (fltbuf, "NaN");		/* Not a Number */

	goto cvt_return;
        }
#endif	IEEE

    if (ndigit < 0)
        ndigit = 0;
    else if (ndigit >= NDIG-1)
	ndigit = NDIG - 2;

    *sign = (value < 0);

    if (*sign)
        value = -value;	/* make it positive */

    value = modf (value, &intPart);

    /* Do integer part.  Start from the end of fltbuf. */

    if (intPart != 0) 
        {
        while (intPart != 0) 
            {
	    /* Strip off the ones digit, stick it in the fltbuf, and divide the
	     * number by 10 */

            fracPart = modf (intPart / 10, &intPart);
            *--p1 = (int)((fracPart + .03) * 10) + '0';
            r2++;
            }

	/* move the int part to the beginning of the string */

        while (p1 < &fltbuf [NDIG])
            *p++ = *p1++;
        } 
    else if (value > 0) 
        {
	/* 0 < value < 1.  Normalize it.  */

        while ((fracPart = value * 10) < 1) 
            {
            value = fracPart;
            r2--;
            }
        }

    p1 = &fltbuf [ndigit];

    if (eflag == 0)
        p1 += r2;

    *decpt = r2;

    if (p1 < &fltbuf[0])
	{
	fltbuf[0] = EOS;
	goto cvt_return;
	}

    while (p <= p1 && p < &fltbuf[NDIG])
        {
        value *= 10;
        value = modf (value, &fracPart);
        *p++ = (int) fracPart + '0';
        }

    if (p1 >= &fltbuf [NDIG]) 
        {
        fltbuf [NDIG - 1] = EOS;
	goto cvt_return;
        }

    p = p1;
    *p1 += 5;

    while (*p1 > '9') 
        {
        *p1 = '0';
        if (p1 > fltbuf)
            ++*--p1;
        else 
            {
            *p1 = '1';
            (*decpt)++;
            if (eflag == 0) 
                {
                if (p > fltbuf)
                    *p = '0';
                p++;
                }
            }
        }

    *p = EOS;

    cvt_return:


    if (!eflag)
	ndigit += r2;

    /* only copy as much as user's buffer can hold */
    strncpy (buffer, fltbuf, ndigit+1);
    buffer [ndigit+1] = EOS;

    return (buffer);
    }
