/* fioLib.c - formatted I/O library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
04g,19aug88,gae  documentation.
04f,13aug88,gae  fixed bug in convert() for E and F.
04e,30jun88,gae  fixed bug in sscanf () and convert() for number specifier.
04d,30may88,dnw  changed to v4 names.
04c,28may88,dnw  removed NOT_GENERIC stuff.
		 removed obsolete rtns: bcdtoi,itobcd,atoi,atow,atos,bprintf
		 removed obsolete: fioSet{In,Out,Err},
		 removed skipHexSpecifier to bootConfig.
		 removed skipSpace to bootConfig and bootLib.
		 made itob NOMANUAL.
		 made digtobin LOCAL.
04b,21apr88,gae  added conversion of 'g' for scanf().
04a,02apr88,gae  rewrote convert and support routines to work with stdio.
		 changed fprintf() to fdprintf().
		 changed fio{Std,Set}{In,Out,Err}() to use new 0,1,2 I/O scheme.
		 made putbuf() local.
03i,16mar88,gae  fixed bug in convert() to return FALSE for invalid specifier.
	         fixed fatal bug in atos() of not handling unterminated table.
		 made atos() return ERROR when given invalid table.
		 sscanf can now do flag:       "<*>".
		 sscanf can now do specifiers: "E|F|c|u|o".
		 printf can now do flags:      "<+><sp><#>".
		 printf can now do specifiers: "u|E|G|X".
		 added nindex() and fioToUpper().
03h,16feb88,llk  fixed sscanf so that it handles double precision floats.
03g,10dec87,gae  changed some VARARGS routines to have standard "args"
		   parameter for parsing by C interface generator.
03f,16nov87,jcf  added skipHexSpecifier to aid in hex parsing
03e,05nov87,jlf  documentation
03d,12oct87,jlf  added [...] strings to scanf.
03c,01aug87,gae  moved floating-point support to fltLib.c.
		 added fioFltInstall().
03b,31jul87,gae  replaced homebrew atof() with the real thing.
		 renamed gcvtb() to gcvt().  fixed bug in ftob().
03a,07may87,jlf  added floating point support, with #ifdef's.
	   +gae	 This involved: %e and %f to format() and convert().
		 New routines: ecvtb, fcvtb, gcvt, atof, atoe, ftob, etob,
		 frexp, ldexp, and modf.
02x,04apr87,dnw  fixed bug in format() that limited fields to 128 characters.
02w,23mar87,jlf  documentation.
02v,17jan87,gae	 made timeConvert and format handle 12 hour clock time code.
02u,20dec86,dnw  changed to not get include files from default directories.
02t,10oct86,dnw  documentation.
02s,04sep86,jlf  documentation.
02r,29jul86,llk  documentation changes
02q,27jul86,llk  added standard error file descriptor (std_err)
		 added fioSetErr, fioStdErr, printErr
02p,01jul86,jlf  minor documentation
02o,27mar86,ecs  changed timeConvert to check for legality of time entered.
		 changed timeDecode to return integer rather than bcd.
02n,11mar86,jlf  changed GENERIC stuff to NOT_GENERIC.
*/

/*
DESCRIPTION
This library provides a higher level interface to the I/O system,
including formatted I/O.  This library implements some of the
facilities of stdioLib (1) the UNIX compatible "standard I/O" library.

Additional, and optional, floating-point formatting and conversion
routines are found in floatLib (1).  The routine floatInit (2) must
be called before format specifications '%e', '%f', or '%g' are
understood.

SEE ALSO: stdioLib (1), floatLib (1)
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "ctype.h"
#include "fioLib.h"
#include "ioLib.h"
#include "strLib.h"

/* floating-point support */

LOCAL FUNCPTR fioFormatRtn;
LOCAL FUNCPTR fioConvertRtn;

/* forward declarations */

char *nindex ();
LOCAL VOID putbuf ();
LOCAL VOID printb ();
LOCAL int getbuf ();
LOCAL int digtobin ();


/*******************************************************************************
*
* fioFltInstall - install routines to print and scan floating-point numbers
*
* This routine is a hook for fltLib to install its special floating-point
* formatting & conversion routines.  It should only be called by floatInit(2).
*
* NOMANUAL
*/

VOID fioFltInstall (formatRtn, convertRtn)
    FUNCPTR formatRtn;		/* routine to format floats for output	*/
    FUNCPTR convertRtn;		/* routine to convert floats to args	*/

    {
    fioFormatRtn = formatRtn;
    fioConvertRtn = convertRtn;
    }
/*******************************************************************************
*
* printf - print formatted string to standard output
*
* This routine prints formatted strings to standard output.  Fmt is a string
* containing two types of objects; plain characters, which are simply copied
* to standard output, and conversion specifications, each of which causes
* conversion and printing of the next argument from the list of arguments
* which follow fmt.  There may be an arbitrary number of args, but there
* must be at least as many as are needed by the format string.
*
* This routine is compatible with the UNIX/K&R standard printf, except
* for minor differences.  Tutorial info can be found in the
* Kernighan & Ritchie C manual, or in the UNIX documentation.
*
* FORMAT STRING
* The format string contains normal ASCII text, which will be passed
* through as is, except for special conversion specifications of the
* following form:
*
* .CS
*	 %<-><+><sp><#><n><.n><l>[d|e|f|g|o|x|u|c|s|b]
*
*    -	- left adjust converted argument in its field.
*    +	- result of a signed conversion will always begin
*         with a sign.
*   sp	- result of a signed conversion will always begin
*         with a sign or blank if positive.
*    #	- result is printed in alternate form.
*    n	- minimum field width.
*	  If converted argument is shorter than this, it will
*	  be padded on the left (or right if left adjustment
*         is indicated). Padding character is blank normally,
*         and zero if field width was specified with a leading
*	  zero.  If '*', get fieldwidth from arg, main arg
*         follows.  '0*' is permissible.  A '*' eats an arg
*         in addition the argument being converted.
*
*    .n	- maximum field width.
*	  If '*', get fieldwidth from arg, main arg follows.
*	  A '*.*' eats 2 args in addition the argument being
*	  converted.
*
*    l	- indicates that corresponding data item is a long
*	  rather than an int.
*
*    [d|e|f|o|x|u|c|s] - conversion character.
*	d	- argument is converted to signed decimal
*		  notation.
*	e	- argument is a float; printed in the style
*		  [-]d.ddde+dd where there is one digit before
*		  the decimal point and the number after is
*		  equal to the precision specification for the
*		  argument.  When the precision is missing,
*		  six digits are produced. If preceded by an l
*		  then argument is a double.
*	f	- argument is a float; printed in the style
*		  [-]ddd.ddd where the number of ds after the
*		  decimal point is equal to the precision
*		  specification for the argument.  If the
*		  precision is missing, six digits are given;
*		  if the precision is explicitly 0, no digits
*		  and no decimal point are printed. If
*                 preceded by an l then argument is a double.
*       g       - uses either e or f whichever is more compact.
*	o	- argument is converted to unsigned octal
*		  notation.
*	x	- argument is converted to unsigned
*		  hexadecimal notation.
*	u	- argument is converted to unsigned decimal
*		  notation.
*	c	- argument is a single character.
*	s	- argument is a string; characters printed
*		  until a null, or maximum field width is
*		  reached.
*	b	- argument is a buffer; characters printed
*		  until minimum field width is reached.
*
* .CE
* DIFFERENCES FROM UNIX PRINTF
*
* This printf has the `b' format specification.
* Otherwise the two are source compatible.
*
* SEE ALSO: UNIX printf manual entry, Kernighan & Ritchie C manual
*
* VARARGS1
*/

VOID printf (fmt, args)
    char *fmt;		/* format string for print	*/
    int args;		/* optional arguments to format	*/

    {
    fioFormat (fmt, &args, printb, 1);
    }
/*******************************************************************************
*
* fdprintf - print formatted string to specified fd
*
* This routine is exactly like printf (2), except that it sends its output
* to the specified fd, rather than to standard output.
*
* SEE ALSO: printf (2), UNIX printf manual entry, Kernighan & Ritchie C manual
*
* VARARGS2
*/

VOID fdprintf (fd, fmt, args)
    int fd;		/* fd to which to print		*/
    char *fmt;		/* format string for print	*/
    int args;		/* optional arguments to format	*/

    {
    fioFormat (fmt, &args, printb, fd);
    }
/*******************************************************************************
*
* sprintf - convert arguments to text under format control
*
* This routine is exactly like printf (2), except that it puts its output
* in the buffer passed as an argument, rather than on standard output.
* The buffer is null terminated.
*
* DIFFERENCES FROM UNIX SPRINTF
*
* Format string differences are exactly the same as printf (2).
* This sprintf returns something useful.
*
* RETURNS: Number of characters decoded into buffer, including EOS.
*
* SEE ALSO: printf (2), UNIX printf manual entry, Kernighan & Ritchie C manual
*
* VARARGS2
*/

int sprintf (buffer, fmt, args)
    char *buffer;		/* buffer to receive decoded text	*/
    char *fmt;			/* format string			*/
    int args;			/* optional arguments to sprintf	*/

    {
    char *start = buffer;

    fioFormat (fmt, &args, putbuf, (int) &buffer);
    *(buffer++) = EOS;
    return (buffer - start);
    }

/*******************************************************************************
*
* fioFormat - format processor
*
* This routine is used by printf (2), sprintf (2), bprintf (2), and fdprintf (2)
* to handle the actual conversion of a format string.  The
* first argument is a format, as described in the entry for printf (2).
* The second argument
* is a pointer to the first argument to be used in processing the
* format string.  Other arguments should follow this argument directly
* in memory, as would be the case in the normal C calling sequence on
* the 680x0.
* 
* As the format string is processed, it will be passed to the routine whose
* address is passed as the third  parameter, which may output it to a
* device, put it in a buffer, or whatever.  The routine should be declared
* as follows:
*
* .CS
*	outRoutine (buffer, nchars, outarg)
*	    char *buffer;      /* buffer passed to routine	*
*	    int nchars;        /* length of buffer		*
*	    int outarg;        /* arbitrary argument passed to	*
*                              /* format routine		*
* .CE
* outarg may be an fd, a buffer address, or whatever (but not a structure).
*
* INTERNAL
* The maximum format width of any one value is 129 bytes. 
*
* VARARGS3
*/

VOID fioFormat (fmt, arg_list, routine, outarg)
    FAST char *fmt;	/* format string				*/
    int *arg_list;	/* list of arguments to be converted to ASCII	*/
    FUNCPTR routine;	/* routine to handle args as they're decoded	*/
    int outarg;		/* argument to routine				*/

    {
    char *buf_ptr;
    FAST int *arg_ptr;
    FAST char *next;
    BOOL r_justify;
    BOOL do_sign;
    BOOL do_blank;
    BOOL do_altern;
    BOOL doFlags;
    FAST int length;
    FAST int minwidth;
    FAST int maxwidth;
    int prefix;
    int fillwidth;
    char fill_char;
    char buffer [129];
    char filBuffer [129];
    int arg;

    for (; ; fmt = next + 1)
	{
	buf_ptr   = buffer;
	doFlags   = TRUE;
	r_justify = TRUE;
	do_sign   = FALSE;
	do_blank  = FALSE;
	do_altern = FALSE;
	fill_char = ' ';
	minwidth  = 0;
	maxwidth  = NONE;
	prefix    = 0;

	/* output plain text until '%' encountered */

	for (next = fmt; (*next != EOS) && (*next != '%'); next++)
	    ;

	if (fmt < next)
	    (*routine) (fmt, next - fmt, outarg);

	if (*next++ == EOS)
	    return;

	while (doFlags)
	    {
	    switch (*next)
		{
		case '-':	/* left adjustment flag */
		    r_justify = FALSE;
		    break;

		case '+':	/* show leading plus or minus */
		    do_sign = TRUE;
		    break;

		case ' ':	/* prepend blank if not negative */
		    do_blank = TRUE;
		    break;

		case '#':	/* use alternate format */
		    do_altern = TRUE;
		    break;

		default:
		    doFlags = FALSE;
		    break;
		}

	    if (doFlags)
		next++;
	    }

	if (*next == '0')	/* minimum field width */
	    {
	    fill_char = '0';
	    next++;
	    }

	if (*next == '*')	/* get fieldwidth from next arg */
	    {
	    minwidth = *(arg_list++);
	    next++;
	    }
	else
	    while (isdigit (*next))
		minwidth = minwidth * 10 + *next++ - '0';

	/* check for maximum field width */

	if (*next == '.')
	    {
	    next++;
	    if (*next == '*')
		{
		maxwidth = *(arg_list++);	/* get maxwidth from next arg */
		next++;
		}
	    else
		{
		maxwidth = 0;
		while (isdigit (*next))
		    maxwidth = maxwidth * 10 + *next++ - '0';
		}
	    }


	if (*next == 'l')
	    next++;			/* pass over length modifier */

	arg_ptr = arg_list;

	/* check for type and convert */

	switch (*next)
	    {
	    case 'd':			/* signed decimal */
		if (*arg_ptr >= 0)
		    {
		    if (do_sign || do_blank)
			buffer [prefix++] = do_sign ? '+' : ' ';

		    length = itob (&buffer [prefix], *arg_ptr, 10) + prefix;
		    }
		else
		    length = itob (buffer, *arg_ptr, -10);
		break;

	    case 'u':			/* unsigned decimal */
		length = itob (buffer, *arg_ptr, 10);
		break;

	    case 'o':			/* unsigned octal */
		if (do_altern)
		    buffer [prefix++] = '0';
		length = itob (&buffer [prefix], *arg_ptr, 8) + prefix;
		break;

	    case 'x':			/* unsigned hexadecimal */
	    case 'X':			/* unsigned hexadecimal, in CAPS */
		if (do_altern)
		    {
		    buffer [prefix++] = '0';
		    buffer [prefix++] = 'x';
		    }

		length = itob (&buffer [prefix], *arg_ptr, 16) + prefix;

		if (*next == 'X')
		    fioToUpper (buffer, length);
		break;

	    case 'c':			/* character */
		arg = *arg_ptr;		    /* ptr to widened char */
		buffer [0] = arg;
		length = 1;
		break;

	    case 's':			/* string */
		buf_ptr = (char *) *arg_ptr;
		length = strlen (buf_ptr);
		break;

	    case 'b':			/* buffer */
		buf_ptr = (char *) *arg_ptr;
		length = minwidth;
		break;

	    case 'e':		/* exponential float */
	    case 'E':		/* exponential float, with big E */
	    case 'f':		/* float */
	    case 'g':		/* general */
	    case 'G':		/* general, with big E */
		if (fioFormatRtn != NULL)
		    {
		    double num = *(double *)arg_ptr;
		    int floatPrecision = 6;	/* default */

		    if (maxwidth != NONE)
			{
			floatPrecision = maxwidth;
			maxwidth = NONE;
			}

		    if (do_sign || do_blank)
			buffer [prefix++] = do_sign ? '+' : ' ';

		    length = (*fioFormatRtn) (*next, num, floatPrecision,
					   &buffer [prefix], sizeof (buffer));

		    /* in alternate mode add decimal point if necessary */

		    if (do_altern && nindex (buffer, '.', length) == NULL)
			buffer [length++] = '.';

		    /* increment arg_list an extra time, since floats are
		     * passed as 4 byte quantities.
		     */

		    arg_list++;
		    break;
		    }

		/* drop thru, if don't know how to do floating-point */

	    default:			/* unrecognized char */
		buffer [0] = *next;		/* allows "%%" to output '%' */
		length = 1;
		arg_list--;			/* don't eat an argument */
		break;
	    }


	arg_list++;	/* step to next arg */


	/* limit field to specified max if any */

	if ((maxwidth != NONE) && (length > maxwidth))
	    length = maxwidth;		/* don't exceed maxwidth */


	/* check if field needs to be filled, and output left fill if any */

	if (minwidth > length)
	    {
	    fillwidth = minwidth - length;
	    bfill (filBuffer, fillwidth, fill_char);

	    if (r_justify)
		(*routine) (filBuffer, fillwidth, outarg);
	    }


	/* output field contents */

	if (length != 0)
	    (*routine) (buf_ptr, length, outarg);


	/* output right fill if any */

	if ((minwidth > length) && !r_justify)
	    (*routine) (filBuffer, fillwidth, outarg);
	}
    }

/*******************************************************************************
*
* sscanf - obtain values for arguments from ASCII string
*
* Reads characters from string str, interprets them according to formats
* in string fmt, and assigns values to arguments starting with that
* pointed at by arg.
*
* This routine is similar to the UNIX/K&R standard.  Tutorial info may be
* found in the UNIX documentation, and in Kernighan & Ritchie's C
* manual.
*
* FORMAT STRING
*
* The format string may contain:
*
*     White-space characters (space, tab, carriage return, newline,
*     or formfeed) which are ignored.
*
*     Ordinary characters (not %) which are expected to match the
*     next non-white-space character in the string str.
*
*     Conversion specifications are of the following form:
*.CS
*	 [%<*><n><h|l>[c|d|e|f|E|F|g|o|u|x|s|[...]]
*
*	<*>   - optional suppress assignment.
*	<n>   - an optional number specifying a maximum
*		fieldwidth.
*	<h|l> - an optional character indicating that the
*		value converted is to be assigned to a
*		short (h) or long (l) integer or a double (l)
*		precision float.
*
*    [c|d|e|f|E|F|g|o|u|x|s|[...]] - conversion character.
*		c - a character is expected in the input,
*                   the corresponding argument should be
*                   a character pointer.  If a field width
*                   is given the argument should be an array.
*		d - a decimal integer is expected in the 
*		    input; the corresponding argument should
*		    be a pointer to a (possibly short or long)
*		    integer.
*	e|f|E|F|g - a floating point number is expected;
*		    corresponding argument should be a pointer
*                   to a float, or double if E, F, or preceding l.
*		o - an octal integer is expected in the input;
*		    the corresponding argument should be a
*		    pointer to a (possibly short of long)
*                   integer.
*		u - a decimal integer is expected in the input;
*		    the corresponding argument should be a
*		    pointer to an unsigned (possibly short
*                   or long) integer.
*		x - a hexadecimal integer is expected in the
*		    input; the corresponding argument should be
*		    be a pointer to a (possibly short or long)
*                   integer.
*		s - a character string is expected, delimited
*		    by white space, a fieldwidth specification,
*		    or an EOS; the corresponding argument should
*		    be a character pointer, pointing to a char-
*		    acter array large enough to accept both the
*		    string and a terminating NULL, which will
*		    be added.
*		[...] - a character string is expected,
*		    delimited by any character not contained
*		    in the bracketed string.  The left bracket
*		    is followed by a set of characters and a right
*		    bracket.  The characters between the brackets
*		    define a set of characters making up the string.
*		    If the first character is not circumflex (^),
*		    the input field is all characters until the
*		    first character not in the set between the
*		    brackets.  If the first character after the
*		    left bracket is  ^ , the input field is all
*		    characters until the first character which
*		    is in the remaining set of characters between
*		    the brackets.  Ranges of the form "a-z" are
*		    allowed in the bracketed set.  The special
*		    characters '^' and ']' can be included in the
*		    set by making them the first characters in it.
*		    The corresponding argument should be a
*		    character pointer, pointing to a character
*		    array large enough to accept both the string
*		    and a terminating NULL, which will be added.
* .CE
* White space characters in the string to be decoded are ignored, except
* insofar as they serve to delimit substrings to be decoded. Conversion
* proceeds until complete, or something cannot be decoded according to
* the format specification. Arguments for which a value can't be decoded
* are left unchanged.
*
* RETURNS:
* number of arguments to which values have been successfully assigned
*
* DIFFERENCES FROM UNIX
* This sscanf will stop reading a %s string when the maximum field
* width is reached.
*
* SEE ALSO: UNIX scanf, Kernighan & Ritchie C manual
* 
* VARARGS2
*/

int sscanf (str, fmt, args)
    char *str;		/* string to decode			*/
    char *fmt;		/* the format string			*/
    int args;		/* optional arguments to sscanf		*/

    {
    BOOL suppress;		/* assignment was suppressed */
    int unget;			/* bit bucket */
    int args_assigned = 0;
    int **argptr = (int **)&args;

    while (*str != EOS)
	{
	unget = -1;

	switch (*fmt)
	    {
	    case EOS:			/* format string exhausted */
		return (args_assigned);

	    case '%':			/* conversion specification */

		fmt++;
		if (convert(&fmt, argptr, &suppress, getbuf, (int)&str, &unget))
		    {
		    if (!suppress)
			{
			argptr++;
			args_assigned++;		/* successful */
			}
		    }
		else
		    return (args_assigned);	/* unsuccessful */
		break;

	    case ' ':
	    case '\n':
	    case '\t':

		fmt++;
		while (isspace (*str))
		    str++;

		if (*str == EOS)
		    return (args_assigned);	/* finished */

		break;

	    default:			/* character to match */

		if (*str != *fmt)
		    return (args_assigned);	/* match failed */

		++str;			/* advance past matched char */
		++fmt;			/* advance past matched char */
		break;
	    }

	if (unget != -1)
	    --str;
	}

    return (args_assigned);
    }

/*******************************************************************************
*
* digtobin - convert an ASCII digit to it's binary equivalent
*
* This routine converts a single ASCII digit to an integer.  Digits may
* be [0-9a-zA-Z].  It works like normal hex conversion,
* but doesn't stop at base 16.  Thus:
*
*	digtobin ('5') == 5
*	digtobin ('d') == 0x0d == 13
*	digtobin ('Z') == 35
*
* RETURNS:
*   The binary equivalent of the ASCII digit, or
*   -1 if digit is not in the set [0-9a-zA-Z].
*/

LOCAL int digtobin (digit)
    char digit;		/* ASCII digit to convert */

    {
    if (isdigit (digit))
	return (digit - '0');

    else if (islower (digit))
	return (digit - 'a' + 10);

    else if (isupper (digit))
	return (digit - 'A' + 10);

    else
	return (-1);
    }
/*******************************************************************************
*
* itob - convert integer to buffer
*
* This routine converts from a value to an ASCII representation of
* the value, in the specified base.  The representation is copied
* into the buffer.  The buffer will not be terminated with an EOS.
*
* If base is positive, the value is treated as unsigned.  If base is
* negative, the value is treated as signed, and will be prepended with
* a `-' if necessary.
*
* EXAMPLE
*     nchars = itob (myBuf, 50, 10);
*
* leaves the string "50" (with no EOS) in myBuf, and returns the value 2.
*
* RETURNS: The number of chars put in the caller's buffer.
*
* NOMANUAL
*/

int itob (buffer, value, base)
    FAST char *buffer;	/* buffer to receive converted ASCII	*/
    int value;		/* value to convert			*/
    int base;		/* base of conversion			*/

    {
    static char digits [] = "0123456789abcdefghijklmnopqrstuvwxyz";
    FAST int nchars  = 0;
    FAST int absbase = abs (base);
    FAST unsigned int absval;

    /* if base is positive treat value as unsigned */

    if (base > 0)
	absval = (unsigned int) value;
    else
	absval = abs (value);


    /* generate the digits backwards in caller's buffer */

    do
	{
	buffer [nchars++] = digits [absval % absbase];

	absval /= absbase;
	}
    while (absval != 0);


    /* add minus sign if base indicates signed and value is negative */

    if ((base < 0) && (value < 0))
	buffer [nchars++] = '-';


    /* invert the characters in the caller's buffer */

    binvert (buffer, nchars);

    return (nchars);
    }
/*******************************************************************************
*
* nindex - find character `c' in `buffer'
*
* RETURNS:
*    address of the first occurrence character `c' in `buffer', or
*    NULL if not found
*
* NOMANUAL
*/

char *nindex (buffer, c, n)
    FAST char *buffer;	/* where to search */
    FAST char c;	/* target character */
    FAST int n;		/* maximum number of characters to inspect */

    {
    FAST int ix;

    for (ix = 0; ix < n; ix++)
	{
	if (*buffer == c)
	    return (buffer);
	buffer++;
	}

    /* didn't find ch */
    return (NULL);
    }
/*******************************************************************************
*
* fioToUpper - convert buffer to upper case
*
* Converts contents of buffer to upper case.
* Stops at either `length' characters or EOS.
*/

LOCAL VOID fioToUpper (buffer, length)
    FAST char *buffer;		/* string to be convert to upper case */
    FAST int length;		/* length of buffer; NONE = unlimited */

    {
    FAST int ix;

    for (ix = 0; *buffer != EOS && (ix < length || length == NONE); ix++)
	{
	if (islower (*buffer))
	    *buffer = toupper (*buffer);
	buffer++;
	}
    }

/*******************************************************************************
*
* fioStdIn - get current assignment of standard input fd (obsolete)
*
* This routine is obsolete, use ioGlobalStdGet (2).
*
* Returns the fd which is currently the standard input.
*
* RETURNS: 0
*/

int fioStdIn ()

    {
    return (STD_IN);
    }
/*******************************************************************************
*
* fioStdOut - get current assignment of standard output fd (obsolete)
*
* This routine is obsolete, use ioGlobalStdGet (2).
*
* Returns the fd that is currently the standard output.
*
* RETURNS: 1
*/

int fioStdOut ()

    {
    return (STD_OUT);
    }
/*******************************************************************************
*
* fioStdErr - get current assignment of standard error fd (obsolete)
*
* This routine is obsolete, use ioGlobalStdGet (2).
*
* Returns the fd which is currently the standard error.
*
* RETURNS: 2
*/

int fioStdErr ()

    {
    return (STD_ERR);
    }
/*******************************************************************************
*
* printErr - print formatted string to standard error fd
*
* This routine is exactly like printf(2), but sends it's output to
* standard error.
*
* SEE ALSO: printf(2)
*
* VARARGS1
*/

VOID printErr (fmt, args)
    char *fmt;		/* format string for print	*/
    int args;		/* optional arguments to format	*/

    {
    fioFormat (fmt, &args, printb, 2);
    }
/*******************************************************************************
*
* fioRead - read a full buffer or to end-of-file
*
* This routine repeatedly calls the routine `read' until maxbytes have
* been read into buffer.  If EOF is reached then the number of bytes read
* will be less than maxbytes.
*
* RETURNS:
*   Number of bytes read, or
*   ERROR if error in read.
*/

int fioRead (fd, buffer, maxbytes)
    int fd;		/* file descriptor of file to read	*/
    char *buffer;	/* buffer to receive input		*/
    int maxbytes;	/* maximum number of bytes to read	*/

    {
    int original_maxbytes = maxbytes;
    int nbytes;

    while (maxbytes > 0)
	{
	nbytes = read (fd, buffer, maxbytes);

	if (nbytes < 0)
	    return (ERROR);

	if (nbytes == 0)
	    {
	    errnoSet (S_fioLib_UNEXPECTED_EOF);
	    return (original_maxbytes - maxbytes);
	    }

	maxbytes -= nbytes;
	buffer   += nbytes;
	}

    return (original_maxbytes);
    }
/*******************************************************************************
*
* fioRdString - read a string from a file
*
* This routine gets a line of input into string.
* The specified input fd is read until either the specified line
* limit is reached, an end of file is reached, an EOS character
* is read, or a newline is read.
* The newline or EOF is stripped off, and an end-of-string marker (EOS)
* is put on the end unless maxbytes characters have been read.
*
* RETURNS:
*   The length of the string read, including terminating EOS, or
*   EOF if end-of-file before any characters are read.
*/

int fioRdString (fd, string, maxbytes)
    int fd;		/* file descriptor of device to read	*/
    char string [];	/* buffer to receive input		*/
    int maxbytes;	/* maximum number of characters to read	*/

    {
    char c;
    int nchars;
    int i = 0;

    /* read characters until 
     *    1) specified line limit is reached,
     *    2) end-of-file is reached,
     *    3) EOS character is read,
     * or 4) newline character is read.
     */

    while (i < maxbytes)
	{
	nchars = read (fd, &c, 1);

	/* check for newline terminator */

	if ((nchars != 1) || (c == '\n') || (c == EOS))
	    {
	    string [i++] = EOS;
	    break;
	    }

	string [i++] = c;
	}


    if ((nchars == 0) && (i == 1))
	i = EOF;

    return (i);
    }

/*******************************************************************************
*
* putbuf - put chars in buffer
*
* This routine copies length bytes from source to destination, leaving the
* destination buffer pointer pointing at byte following block copied.
* Its primary use is as a support routine for sprintf(2).
*/

LOCAL VOID putbuf (inbuf, length, outptr)
    char *inbuf;		/* pointer to source buffer	*/
    int length;			/* number of bytes to copy	*/
    char **outptr;		/* pointer to destination buffer */

    {
    bcopy (inbuf, *outptr, length);
    *outptr += length;
    }
/*******************************************************************************
*
* printb - printf support routine: print chars in buffer
*/

LOCAL VOID printb (buf, nbytes, fd)
    char *buf;
    int nbytes;
    int fd;

    {
    write (fd, buf, nbytes);
    }
/*******************************************************************************
*
* getbuf - get next character in string
*
* RETURNS: next character or EOF if empty
*/

LOCAL int getbuf (str)
    char **str;

    {
    int c = (int)(**str);

    if (c == EOS)
	return (EOF);
    ++*str;
    return (c);
    }
/*******************************************************************************
*
* convert - assign value to argument according to format
*
* RETURNS:
*    TRUE if conversion successful, otherwise FALSE
*    pSuppress is set TRUE if assignment was suppressed    
*
* NOMANUAL
*/

BOOL convert (fmtptr, argptr, pSuppress, get, getarg, pUnget)
    char **fmtptr;	/* conversion specification string		*/
    int **argptr;	/* to whom to assign the value			*/
    BOOL *pSuppress;	/* set to TRUE if argptr was not assigned       */
    FUNCPTR get;
    int getarg;
    int *pUnget;

    {
    int fieldwidth = 0;		/* maximum number of chars to convert.
				   default 0 implies no limit */

				/* default int or float is 4 bytes long. */
    BOOL isShort = FALSE;	/*    size of arg pointed at by argptr. */
    BOOL isLong = FALSE;	/*    size of arg pointed at by argptr. */
    int nchars = 0;		/* number of chars eaten from string */
    double fnum;		/* decoded float number */
    int num;			/* decoded number */
    char fch = **fmtptr;	/* character from format string */

    /* check for assignment suppression */

    if (*pSuppress = (fch == '*'))
	fch = *++*fmtptr;

    /* check for specification of maximum fieldwidth */

    while (isdigit (fch = **fmtptr))
	{
	fieldwidth = 10 * fieldwidth + (fch - '0');
	++*fmtptr;
	}

    /* check for specification of size of returned value */

    switch (fch)
	{
	case 'h':		/* 2 byte short integer */
	    isShort = TRUE;
	    fch = *++*fmtptr;		/* get past size spec */
	    break;

	case 'l':		/* long integer or double precision float */
	    isLong = TRUE;
	    fch = *++*fmtptr;		/* get past size spec */
	    break;
	}

    switch (fch)
	{
	case 'c':		/* character, or character array */
	    if (fieldwidth == 0)
		fieldwidth = 1;
	    nchars = dochar ((char *)(*pSuppress ? NULL: *argptr),
				fieldwidth, get, getarg, pUnget);
	    if (nchars == 0)
		return (FALSE);		/* unsuccessful conversion */
	    break;

	case 'x':		/* hex integer */
	case 'o':		/* octal integer */
	case 'd':		/* decimal integer */
	case 'u':		/* unsigned decimal integer */
	    nchars = donum (fch, &num, fieldwidth, get, getarg, pUnget);
	    if (nchars == 0)
		return (FALSE);		/* unsuccessful conversion */

	    if (!*pSuppress)
		{
		if (fch == 'u')
		    {
		    if (isShort)
			*((unsigned short *) *argptr) = num;
		    else
			*((unsigned long *) *argptr) = num;
		    }
		else
		    {
		    if (isShort)
			*((short *) *argptr) = num;
		    else
			*((long *) *argptr) = num;
		    }
		}
	    break;

	case 'E':		/* exponential double */
	case 'F':		/* double */
	    isLong = TRUE;
	    fch = tolower (fch);
	    /* drop through */

	case 'e':		/* exponential float */
	case 'f':		/* float */
	case 'g':		/* float */
	    nchars = donum (fch, (int *)&fnum, fieldwidth, get, getarg,
			    pUnget);
	    if (nchars == 0)
		return (FALSE);		/* unsuccessful conversion */

	    if (!*pSuppress)
		{
		if (isLong)
		    *((double *) *argptr) = fnum;
		else
		    *((float *) *argptr) = fnum;
		}
	    break;

	case 's':		/* string */
	    nchars = doword ((char *)(*pSuppress ? NULL: *argptr),
			      fieldwidth, get, getarg, pUnget);
	    if (nchars == 0)
		return (FALSE);		/* unsuccessful conversion */
	    break;

	case '[':		/* non-space-delimited string */
	    nchars = dostring ((char *)(*pSuppress ? NULL: *argptr),
			        fieldwidth, *fmtptr, get, getarg, pUnget);
	    if (nchars <= 0)
		return (FALSE);

	    /* get past format specifier string */

	    while (**fmtptr != ']')
		++*fmtptr;
	    break;


	default:
	    return (FALSE);
	}

    ++*fmtptr;			/* point past format specifier */

    return (TRUE);		/* successful conversion */
    }


/* support routines for convert */

#define	NEXT()	(ix++, *pUnget = ch = get(getarg))

/*******************************************************************************
*
* dochar - perform character conversion
*
* RETURNS: number of characters converted
*/

LOCAL int dochar (result, fieldwidth, get, getarg, pUnget)
    char *result;
    int fieldwidth;
    FUNCPTR get;
    int getarg;
    int *pUnget;

    {
    int ch;
    int ix = 0;

    NEXT();

    while (ch != EOF && (ix <= fieldwidth || fieldwidth == 0))
	{
	if (result != NULL)
	    {
	    if (ix > 1)
		result++;
	    *result = (char)ch;
	    }
	NEXT();
	}

    if (ch == EOF || (ix <= fieldwidth && fieldwidth != 0))
	*pUnget = -1;

    return (ix - 1);
    }
/*******************************************************************************
*
* donum - perform number conversion
*
* RETURNS: number of characters converted
*/

LOCAL int donum (type, num, fieldwidth, get, getarg, pUnget)
    char type;	/* 'd','u','o','x','e','f' */
    int *num;
    int fieldwidth;
    FUNCPTR get;
    int getarg;
    int *pUnget;

    {
    char numbuf[100];	/* buffer for hex number */
    char *np = numbuf;	/* current position in above buffer */
    char numdigs = 0;	/* number of digits seen */
    int base = 10;	/* default is base 10 conversion */
    BOOL expdone = FALSE; /* exponent done? */
    BOOL scalar = type != 'e' && type != 'f';	/* integer or float? */
    BOOL neg;		/* negative or positive? */
    int dig;		/* digit at i'th position */
    int n = 0;		/* buffer for building number */
    int ix = 0;		/* present used width */
    int ch;

    if (type == 'o') 
	base = 8;
    else if (type == 'x') 
	base = 16;

    /* skip white stuff */

    while (isspace (NEXT()))
	;

    ix = 0;	/* don't count spaces */

    /* skip over sign */

    if (neg = ((char)ch == '-'))
	{
	*np++ = ch;
	NEXT();
	}
    else if ((char)ch == '+')
	NEXT();

    while (ix < fieldwidth || fieldwidth == 0)
	{
	dig = digtobin ((char)ch);

	if (dig >= 0 && dig < base)
	    {
	    numdigs++;
	    n = (n * base) + dig;
	    }
	else if ((char)ch == '.')
	    {
	    if (scalar || base != 10)
		break;
	    numdigs++;
	    }
	else if (!expdone && ((char)ch == 'e' || (char)ch == 'E'))
	    {
	    if (scalar || base != 10 || numdigs == 0)
		break;
	    expdone = TRUE;
	    *np++ = (char)ch;
	    NEXT();
	    if ((char)ch != '+' && (char)ch != '-' && !isdigit((char)ch))
		break;
	    }
	else
	    break;

	*np++ = (char)ch;
	NEXT();
	}

    if (ch == EOF || (ix < fieldwidth && fieldwidth != 0))
	*pUnget = -1;

    if (np == numbuf)
	return (0);	/* didn't do anything */

    if (scalar)
	*num = neg ? -n : n;
    else
	{
	if (fioConvertRtn == NULL)
	    return (0);	/* don't know how to scan floating-point */

	*np = EOS;	/* let's not forget to null terminate! */

	/* convert routine atof() returns number of characters
	 * used, actually one more than ix, ...strange but true. */

	(void)(*fioConvertRtn) (numbuf, (double *)num, 0);
	}

    return (ix);
    }
/*******************************************************************************
*
* doword - perform conversion of space delineated string
*
* RETURNS: number of characters converted
*/

LOCAL int doword (result, fieldwidth, get, getarg, pUnget)
    char *result;
    int fieldwidth;
    FUNCPTR get;
    int getarg;
    int *pUnget;

    {
    int ix = 0;
    int ch;

    while (isspace (NEXT()))
	;

    ix = 0; /* don't count spaces */

    while (ch != EOF && (ix < fieldwidth || fieldwidth == 0))
	{
	if (isspace (ch))
	    break;

	if (result != NULL)
	    *result++ = (char)ch;

	NEXT();
	}

    if (ch == EOF || (ix < fieldwidth && fieldwidth != 0))
	*pUnget = -1;

    if (result != NULL)
	*result = EOS;

    return (ix);
    }
/*******************************************************************************
*
* dostring - perform user specified string conversion
*
* RETURNS: number of characters converted
*/

LOCAL int dostring (result, fieldwidth, charsAllowed, get, getarg, pUnget)
    char *result;
    int fieldwidth;
    FAST char *charsAllowed;	/* The chars (not) allowed in the string */
    FUNCPTR get;
    int getarg;
    int *pUnget;

    {
#define	NUM_ASCII_CHARS	128
    char lastChar = 0;
    char charsAllowedTbl [NUM_ASCII_CHARS];
    FAST BOOL allowed;
    int nx;
    int ix;
    int ch;

    if (*charsAllowed++ != '[')		/* skip past the [ */
	return (ERROR);

    /* chars in table will (not) be allowed */

    allowed = (*charsAllowed != '^');

    if (!allowed)
	charsAllowed++;			/* skip past ^ */

    /* initialize table to (not) allow any chars */

    for (ix = 0; ix < NUM_ASCII_CHARS; ix++)
	charsAllowedTbl [ix] = !allowed;

    /* Check for the first char of the set being '-' or ']', in which case
       they are not special chars. */

    if (*charsAllowed == '-' || *charsAllowed == ']')
	charsAllowedTbl [*charsAllowed++] = allowed;

    /* Examine the rest of the set, and put it into the table. */

    for (nx = 0; nx < NUM_ASCII_CHARS; nx++)
	{
	if (*charsAllowed == ']' || *charsAllowed == EOS)
	    break;	/* all done */

	/* Check if its of the form x-y.
	 * If x<y, then allow all chars in the range.
	 * If x>=y, its not a legal range.  Just allow '-'. 
	 * Also, if the '-' is the last char before the final ']',
	 * just allow '-'.
	 */
	if (*charsAllowed == '-' &&
	    (lastChar < *(charsAllowed+1) && *(charsAllowed+1) != ']'))
	    {
	    charsAllowed++;	/* skip - */
	    for (ix = lastChar; ix <= *charsAllowed; ix++)
		charsAllowedTbl [ix] = allowed;
	    }
	else
	    {
	    charsAllowedTbl [*charsAllowed] = allowed;
	    lastChar = *charsAllowed;
	    }
	charsAllowed++;
	}

    /* make sure that ']' appeared */

    if (*charsAllowed != ']')
	return (ERROR);

    /* Copy the appropriate portion of the string */

    ix = 0;

    NEXT();

    while (ch != EOF && (ix < fieldwidth || fieldwidth == 0))
	{
	if (! charsAllowedTbl [(char)ch])
	    break;

	if (result != NULL)
	    *result++ = (char)ch;
	NEXT();
	}

    if (ch == EOF || (ix < fieldwidth && fieldwidth != 0))
	*pUnget = -1;

    if (result != NULL)
	*result = EOS;

    return (ix);
    }
