/* dsmLib.c - a 68K disassembler */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
02m,30may88,dnw  changed to v4 names.
02l,26oct87,hin  Fixed bug, when base displacement was used without base 
		   register, instruction length was wrong.
                 Added 68881 instructions.
02k,14sep87,llk  Fixed bug.  itShift instructions weren't handled correctly.
		   Split itShift into itRegShift and itMemShift.
02j,30jul87,llk  Fixed bug.  "PC indirect with index" was not handled correctly
		   for cases where there was no memory indirection.
02i,23mar87,jlf  documentation.
02h,09feb87,llk  Fixed bug.  ASL (and ASR) was displaying "ASL 0" for "ASL 8".
02g,21dec86,dnw  changed to not get include files from default directories.
02f,19dec86,llk  Fixed bug.  Collision occurred between AND and EXG.
02e,03dec86,ecs  Fixed bug in dsmPrint that caused 5-word instructions to
		    take up 2 lines.
02d,26may86,llk	 Got rid of lint introduced in 02a.
02c,23may86,llk	 Fixed bug.  CMPA was disassembled as CMPM.
02b,08apr86,llk	 Fixed bug introduced in 02a.  Negative displacements
		 of branch instructions not printed correctly.
02a,03apr86,llk	 Added 68020 instructions, reordered inst [] table by the
		 number of bits in an instruction's mask.  Added two word
		 instruction handling.  Added the routines mode6And7Words, 
		 prOffWid, prIndirectIndex, prDisplace, findTwoWord.
		 Added the extension parameter to modeNwords.  Enable 
		 printing of unlimited number of words per instruction.
		 Corrected spelling of "dissasembler".
		 Still needs better handling of two word instructions.
01q,20jul85,jlf  documentation.
01p,12jun85,rdc  added 68010 instructions.
01o,11nov84,jlf  Fixed shift instructions to work right.
01n,11nov84,jlf  Made EXT be its own type.
01m,20sep84,ecs&jlf  Added dsmData.
		 Changed dsmPrint to print unknown instructions as data.
		 Fixed bug in printing cmpa (introduced in 01k).
		 Made CMPM be its own type.
01l,18sep84,jlf  Removed l from the description.
01k,17sep84,jlf  Separated out adda and suba from add and sub, to fix a bug
		 with printing adda's sometimes.
01j,10sep84,ecs  Removed l, lPrtAddress, and nxtToDsm to dbgLib.c.
01g,09sep84,jlf  Added dsmNbytes, got rid of GLOBAL.
01h,06aug84,jlf  Added copyright notice, and some comments.
01g,30aug84,jlf  Fixed bug that sometimes caused ADDX and ADD.L to get 
		 confused (and same for SUBX).
01f,23aug84,ecs  Changed nxtToDsm to GLOBAL.
01e,16aug84,jlf  Fixed bug in printing DBcc and Scc instructions, by adding
		 new instruction types itDb and itScc.  Changed some
		 routines to accept the address of the instruction they
		 are disassembling as a parameter, rather than just looking
		 at the pointer to the instruction.  This will make it
		 easier to make this stuff work under unix.
01d,07aug84,ecs  Added call to setStatus to dsmFind.
		 Split instruction type definitions and definition of structure
		   INST off to dsmLib.h.
01c,24jul84,ecs&jlf  Appeased lint and fixed subq bug (didn't print dest.).
01b,16jul84,jlf  Minor bug fixes.  Made unlk instruction use A reg (bug
		 had D reg) and made long immediate operands print right, 
		 even if low order word is 0.
01a,29jun84,jlf	 written
*/

/*
This library has all the stuff necessary to print 680x0 object code in
assembly language format. The disassembly is done into Motorola format.

The programming interface is via dsmInst, which prints a single disassembled
instruction, and dsmNbytes, which will tell you how big an instruction is.

To disassemble from the shell, use the 'l' instruction, which calls this
library to do the actual work.  See dbgLib for details.

ADDRESS PRINTING ROUTINE
Many of the operands to assembly language instructions are addresses.  In
order to allow the ability to print these symbolically, the call to dsmInst
has, as a parameter, the address of a routine to call to print an address.
When dsmLib needs to print an address as part of an operand field, it will call
the supplied routine to do the actual printing.  The routine should be declared
as:

    VOID prtAddress (address)
	int address;	* address to print *

When called, the routine should print the address on standard out (with printf
or whatever) in whatever form, numeric or symbolic, that it desires.  The
routine used by 'l', for instance, looks up the address in the system symbol
table and prints the symbol associated with it, if there is one.  If not, it
just prints the address as a hex number.

If the prtAddress argument to dsmPrint is NULL, a default print routine is used,
which just prints the address as a hex number.

DEFICIENCIES
Movem's are not disassembled very nicely.  Some instructions and 
effective addressing modes that are actually illegal are not recognized 
as such, and are happily printed as whatever instruction they would 
probably be, if they weren't illegal.

DIVUL and DIVSL never get printed out.  DIVU and DIVS, respectively, are printed
instead.

Instructions should really be compared by their first two words since the
68020 is a 32 bit processor.  This would lessen the chance of bugs where
a word can be disassembled into more than one instruction.

INCLUDE FILE: dsmLib.h
*/

#include "vxWorks.h"
#include "dsmLib.h"
#include "symLib.h"

/*
This table is ordered by the number of bits in an instruction's mask,
beginning with the greatest number of bits in a mask.  This scheme is 
used for avoiding conflicts between instructions when matching bit 
patterns.  The instruction ops are arranged sequentially within each 
group of instructions for a particular mask so that uniqueness can 
be easily spotted.  When adding a new instruction to the inst [] table, 
if it has the same op as another instruction, this means it is a two word 
instruction.  Its type will be itTwoWord and an entry must be made for 
it at the beginning of instTwo [].
*/

LOCAL INST inst [] =
    {
    /* Two word instructions */
    {"CAS2",	0x0cfc,		0xffff,	itTwoWord},
    {"CAS2",	0x0efc,		0xffff,	itTwoWord},
    {"CALLM",	0x06d0,		0xfff8,	itTwoWord},
    {"CALLM",	0x06e8,		0xfff8,	itTwoWord},
    {"CALLM",	0x06f0,		0xfff8,	itTwoWord},
    {"CALLM",	0x06f8,		0xfff8,	itTwoWord},
    {"CHK2",	0x00c0,		0xffc0,	itTwoWord},
    {"CMP2",	0x00c0,		0xffc0,	itTwoWord},
    {"CHK2",	0x02c0,		0xffc0,	itTwoWord},
    {"CMP2",	0x02c0,		0xffc0,	itTwoWord},
    {"CHK2",	0x04c0,		0xffc0,	itTwoWord},
    {"CMP2",	0x04c0,		0xffc0,	itTwoWord},
    {"CAS",	0x0ac0,		0xffc0,	itTwoWord},
    {"CAS",	0x0cc0,		0xffc0,	itTwoWord},
    {"CAS",	0x0ec0,		0xffc0,	itTwoWord},
    {"MULS",	0x4c00,		0xffc0,	itTwoWord},
    {"MULU",	0x4c00,		0xffc0,	itTwoWord},
    {"DIVS",	0x4c40,		0xffc0,	itTwoWord},
    {"DIVU",	0x4c40,		0xffc0,	itTwoWord},

    /*   Fpp instructions */
    {"FABS",	0xf200,		0xffc0,	itFpp},
    {"FB",	0xf280,		0xff80,	itFb},
    {"FDB",	0xf248,		0xfff8,	itFpp},
    {"FMOVECR",	0xf200,		0xffff,	itFpp},
    {"FNOP",	0xf280,		0xff80,	itFpp},
    {"FRESTORE",	0xf340,		0xffc0,	itFrestore},
    {"FSAVE",	0xf300,		0xffc0,	itFsave},
    {"FS",	0xf240,		0xffc0,	itFpp},
    {"FTRAP",	0xf278,		0xfff8,	itFpp},

    /*   16 bit mask */
    {"ORI",	0x003c,		0xffff,	itImmCCR},	/* to CCR */
    {"ORI",	0x007c,		0xffff,	itImmTSR},	/* to SR */
    {"ANDI",	0x023c,		0xffff,	itImmCCR},	/* to CCR */
    {"ANDI",	0x027c,		0xffff,	itImmTSR},	/* to SR */
    {"EORI",	0x0a3c,		0xffff,	itImmCCR},	/* to CCR */
    {"EORI",	0x0a7c,		0xffff,	itImmTSR},	/* to SR */
    {"ILL",	0x4afc,		0xffff,	itComplete},
    {"RESET",	0x4e70,		0xffff,	itComplete},
    {"NOP",	0x4e71,		0xffff,	itComplete},
    {"STOP",	0x4e72,		0xffff,	itStop},
    {"RTE",	0x4e73,		0xffff,	itComplete},
    {"RTD",	0x4e74,		0xffff,	itRTD},
    {"RTS",	0x4e75,		0xffff,	itComplete},
    {"TRAPV",	0x4e76,		0xffff,	itComplete},
    {"RTR",	0x4e77,		0xffff,	itComplete},

    /*     15 bit mask	*/
    {"MOVEC",	0x4e7a,		0xfffe,	itMovec},

    /*     13 bit mask	*/
    {"LINK",	0x4808,		0xfff8,	itLinkL},
    {"SWAP",	0x4840,		0xfff8,	itSwap},
    {"BKPT",	0x4848,		0xfff8,	itBkpt},
    {"EXT",	0x4880,		0xfff8,	itExt},
    {"EXT",	0x48c0,		0xfff8,	itExt},
    {"EXTB",	0x49c0,		0xfff8,	itExt},
    {"LINK",	0x4e50,		0xfff8,	itLink},
    {"UNLK",	0x4e58,		0xfff8,	itUnlk},
    {"DBT",	0x50c8,		0xfff8,	itDb},
    {"TRAPT",	0x50f8,		0xfff8,	itTrapcc},
    {"DBF",	0x51c8,		0xfff8,	itDb},
    {"TRAPF",	0x51f8,		0xfff8,	itTrapcc},
    {"DBHI",	0x52c8,		0xfff8,	itDb},
    {"TRAPHI",	0x52f8,		0xfff8,	itTrapcc},
    {"DBLS",	0x53c8,		0xfff8,	itDb},
    {"TRAPLS",	0x53f8,		0xfff8,	itTrapcc},
    {"DBCC",	0x54c8,		0xfff8,	itDb},
    {"TRAPCC",	0x54f8,		0xfff8,	itTrapcc},
    {"DBCS",	0x55c8,		0xfff8,	itDb},
    {"TRAPCS",	0x55f8,		0xfff8,	itTrapcc},
    {"DBNE",	0x56c8,		0xfff8,	itDb},
    {"TRAPNE",	0x56f8,		0xfff8,	itTrapcc},
    {"DBEQ",	0x57c8,		0xfff8,	itDb},
    {"TRAPEQ",	0x57f8,		0xfff8,	itTrapcc},
    {"DBVC",	0x58c8,		0xfff8,	itDb},
    {"TRAPVC",	0x58f8,		0xfff8,	itTrapcc},
    {"DBVS",	0x59c8,		0xfff8,	itDb},
    {"TRAPVS",	0x59f8,		0xfff8,	itTrapcc},
    {"DBPL",	0x5ac8,		0xfff8,	itDb},
    {"TRAPPL",	0x5af8,		0xfff8,	itTrapcc},
    {"DBMI",	0x5bc8,		0xfff8,	itDb},
    {"TRAPMI",	0x5bf8,		0xfff8,	itTrapcc},
    {"DBGE",	0x5cc8,		0xfff8,	itDb},
    {"TRAPGE",	0x5cf8,		0xfff8,	itTrapcc},
    {"DBLT",	0x5dc8,		0xfff8,	itDb},
    {"TRAPLT",	0x5df8,		0xfff8,	itTrapcc},
    {"DBGT",	0x5ec8,		0xfff8,	itDb},
    {"TRAPGT",	0x5ef8,		0xfff8,	itTrapcc},
    {"DBLE",	0x5fc8,		0xfff8,	itDb},
    {"TRAPLE",	0x5ff8,		0xfff8,	itTrapcc},

    /*     12 bit mask	*/
    {"RTM",	0x06c0,		0xfff0,	itRtm},
    {"TRAP",	0x4e40,		0xfff0,	itTrap},
    {"MOVE",	0x4e60,		0xfff0,	itMoveUSP},	/* USP */

    /*     10 bit mask	*/
    {"ORI",	0x0000,		0xffc0,	itImm},
    {"ORI",	0x0040,		0xffc0,	itImm},
    {"ORI",	0x0080,		0xffc0,	itImm},
    {"ANDI",	0x0200,		0xffc0,	itImm},
    {"ANDI",	0x0240,		0xffc0,	itImm},
    {"ANDI",	0x0280,		0xffc0,	itImm},
    {"SUBI",	0x0400,		0xffc0,	itImm},
    {"SUBI",	0x0440,		0xffc0,	itImm},
    {"SUBI",	0x0480,		0xffc0,	itImm},
    {"ADDI",	0x0600,		0xffc0,	itImm},
    {"ADDI",	0x0640,		0xffc0,	itImm},
    {"ADDI",	0x0680,		0xffc0,	itImm},
    {"BTST",	0x0800,		0xffc0,	itStatBit},
    {"BCHG",	0x0840,		0xffc0,	itStatBit},
    {"BCLR",	0x0880,		0xffc0,	itStatBit},
    {"BSET",	0x08c0,		0xffc0,	itStatBit},
    {"EORI",	0x0a00,		0xffc0,	itImm},
    {"EORI",	0x0a40,		0xffc0,	itImm},
    {"EORI",	0x0a80,		0xffc0,	itImm},
    {"CMPI",	0x0c00,		0xffc0,	itImm},
    {"CMPI",	0x0c40,		0xffc0,	itImm},
    {"CMPI",	0x0c80,		0xffc0,	itImm},
    {"MOVES",	0x0e00,		0xffc0,	itMoves},
    {"MOVES",	0x0e40,		0xffc0,	itMoves},
    {"MOVES",	0x0e80,		0xffc0,	itMoves},
    {"MOVE",	0x40c0,		0xffc0,	itMoveFSR},	/* from SR */
    {"MOVE",	0x42c0,		0xffc0,	itMoveFCCR},	/* from CCR */
    {"MOVE",	0x44c0,		0xffc0,	itMoveCCR},	/* to CCR */
    {"MOVE",	0x46c0,		0xffc0,	itMoveTSR},	/* to SR */
    {"NBCD",	0x4800,		0xffc0,	itNbcd},
    {"PEA",	0x4840,		0xffc0,	itNbcd},
    {"TAS",	0x4ac0,		0xffc0,	itNbcd},
    {"JSR",	0x4e80,		0xffc0,	itNbcd},
    {"JMP",	0x4ec0,		0xffc0,	itNbcd},
    {"ST",	0x50c0,		0xffc0,	itScc},
    {"SF",	0x51c0,		0xffc0,	itScc},
    {"SHI",	0x52c0,		0xffc0,	itScc},
    {"SLS",	0x53c0,		0xffc0,	itScc},
    {"SCC",	0x54c0,		0xffc0,	itScc},
    {"SCS",	0x55c0,		0xffc0,	itScc},
    {"SNE",	0x56c0,		0xffc0,	itScc},
    {"SEQ",	0x57c0,		0xffc0,	itScc},
    {"SVC",	0x58c0,		0xffc0,	itScc},
    {"SVS",	0x59c0,		0xffc0,	itScc},
    {"SPL",	0x5ac0,		0xffc0,	itScc},
    {"SMI",	0x5bc0,		0xffc0,	itScc},
    {"SGE",	0x5cc0,		0xffc0,	itScc},
    {"SLT",	0x5dc0,		0xffc0,	itScc},
    {"SGT",	0x5ec0,		0xffc0,	itScc},
    {"SLE",	0x5fc0,		0xffc0,	itScc},
    {"ASR",	0xe0c0,		0xffc0,	itMemShift},
    {"ASL",	0xe1c0,		0xffc0,	itMemShift},
    {"LSR",	0xe2c0,		0xffc0,	itMemShift},
    {"LSL",	0xe3c0,		0xffc0,	itMemShift},
    {"ROXR",	0xe4c0,		0xffc0,	itMemShift},
    {"ROXL",	0xe5c0,		0xffc0,	itMemShift},
    {"ROR",	0xe6c0,		0xffc0,	itMemShift},
    {"ROL",	0xe7c0,		0xffc0,	itMemShift},
    {"BFTST",	0xe8c0,		0xffc0,	itBfchg},
    {"BFEXTU",	0xe9c0,		0xffc0,	itBfext},
    {"BFCHG",	0xeac0,		0xffc0,	itBfchg},
    {"BFEXTS",	0xebc0,		0xffc0,	itBfext},
    {"BFCLR",	0xecc0,		0xffc0,	itBfchg},
    {"BFFFO",	0xedc0,		0xffc0,	itBfext},
    {"BFSET",	0xeec0,		0xffc0,	itBfchg},
    {"BFINS",	0xefc0,		0xffc0,	itBfins},

    /*     10 bit mask	*/
    {"CMPM",	0xb108,		0xf1f8,	itCmpm},
    {"CMPM",	0xb148,		0xf1f8,	itCmpm},
    {"CMPM",	0xb188,		0xf1f8,	itCmpm},
    {"EXG",	0xc140,		0xf1f8,	itExg},
    {"EXG",	0xc148,		0xf1f8,	itExg},
    {"EXG",	0xc188,		0xf1f8,	itExg},
    {"cpDBcc",	0xf048,		0xf1f8,	itCpDbcc},
    {"cpTRAP",	0xf078,		0xf1f8,	itCpTrapcc},

    /*     9 bit mask	*/
    {"SBCD",	0x8100,		0xf1f0,	itBcd},
    {"PACK",	0x8140,		0xf1f0,	itPack},
    {"UNPK",	0x8180,		0xf1f0,	itPack},
    {"ABCD",	0xc100,		0xf1f0,	itBcd},

    /*     8 bit mask	*/
    {"NEGX",	0x4000,		0xff00,	itNegx},
    {"CLR",	0x4200,		0xff00,	itNegx},
    {"NEG",	0x4400,		0xff00,	itNegx},
    {"NOT",	0x4600,		0xff00,	itNegx},
    {"TST",	0x4a00,		0xff00,	itNegx},
    {"BRA",	0x6000,		0xff00,	itBra},
    {"BSR",	0x6100,		0xff00,	itBra},
    {"BHI",	0x6200,		0xff00,	itBra},
    {"BLS",	0x6300,		0xff00,	itBra},
    {"BCC",	0x6400,		0xff00,	itBra},
    {"BCS",	0x6500,		0xff00,	itBra},
    {"BNE",	0x6600,		0xff00,	itBra},
    {"BEQ",	0x6700,		0xff00,	itBra},
    {"BVC",	0x6800,		0xff00,	itBra},
    {"BVS",	0x6900,		0xff00,	itBra},
    {"BPL",	0x6a00,		0xff00,	itBra},
    {"BMI",	0x6b00,		0xff00,	itBra},
    {"BGE",	0x6c00,		0xff00,	itBra},
    {"BLT",	0x6d00,		0xff00,	itBra},
    {"BGT",	0x6e00,		0xff00,	itBra},
    {"BLE",	0x6f00,		0xff00,	itBra},

    /*     8 bit mask	*/
    {"MOVEM",	0x4880,		0xfb80,	itMovem},

    /*     7 bit mask	*/
    {"BTST",	0x0100,		0xf1c0,	itDynBit},
    {"BCHG",	0x0140,		0xf1c0,	itDynBit},
    {"BCLR",	0x0180,		0xf1c0,	itDynBit},
    {"BSET",	0x01c0,		0xf1c0,	itDynBit},
    {"CHK",	0x4100,		0xf1c0,	itChk},
    {"CHK",	0x4180,		0xf1c0,	itChk},
    {"LEA",	0x41c0,		0xf1c0,	itLea},
    {"DIVU",	0x80c0,		0xf1c0,	itDivW},
    {"DIVS",	0x81c0,		0xf1c0,	itDivW},
    {"SUB",	0x9000,		0xf1c0,	itOr},
    {"SUB",	0x9040,		0xf1c0,	itOr},
    {"SUB",	0x9080,		0xf1c0,	itOr},
    {"SUBA",	0x90c0,		0xf1c0,	itAdda},
    {"SUB",	0x9100,		0xf1c0,	itOr},
    {"SUB",	0x9140,		0xf1c0,	itOr},
    {"SUB",	0x9180,		0xf1c0,	itOr},
    {"SUBA",	0x91c0,		0xf1c0,	itAdda},
    {"CMP",	0xb000,		0xf1c0,	itOr},
    {"CMP",	0xb040,		0xf1c0,	itOr},
    {"CMP",	0xb080,		0xf1c0,	itOr},
    {"CMPA",	0xb0c0,		0xf1c0,	itAdda},
    {"CMPA",	0xb1c0,		0xf1c0,	itAdda},
    {"MULU",	0xc0c0,		0xf1c0,	itDivW},
    {"MULS",	0xc1c0,		0xf1c0,	itDivW},
    {"ADD",	0xd000,		0xf1c0,	itOr},
    {"ADD",	0xd040,		0xf1c0,	itOr},
    {"ADD",	0xd080,		0xf1c0,	itOr},
    {"ADDA",	0xd0c0,		0xf1c0,	itAdda},
    {"ADD",	0xd100,		0xf1c0,	itOr},
    {"ADD",	0xd140,		0xf1c0,	itOr},
    {"ADD",	0xd180,		0xf1c0,	itOr},
    {"ADDA",	0xd1c0,		0xf1c0,	itAdda},

    /*     7 bit mask	*/
    {"SUBX",	0x9100,		0xf130,	itX},
    {"ADDX",	0xd100,		0xf130,	itX},

    /*     7 bit mask	*/
    {"ASR",	0xe000,		0xf118,	itRegShift},
    {"LSR",	0xe008,		0xf118,	itRegShift},
    {"ROXR",	0xe010,		0xf118,	itRegShift},
    {"ROR",	0xe018,		0xf118,	itRegShift},
    {"ASL",	0xe100,		0xf118,	itRegShift},
    {"LSL",	0xe108,		0xf118,	itRegShift},
    {"ROXL",	0xe110,		0xf118,	itRegShift},
    {"ROL",	0xe118,		0xf118,	itRegShift},

    /*     7 bit mask	*/
    {"MOVEP",	0x0080,		0xf038,	itMovep},

    /*     6 bit mask	*/
    {"cpBcc",	0xf080,		0xf180,	itCpBcc},

    /*     5 bit mask	*/
    {"MOVEA",	0x0040,		0xc1c0,	itMoveA},

    /*     5 bit mask	*/
    {"ADDQ",	0x5000,		0xf100,	itQuick},
    {"SUBQ",	0x5100,		0xf100,	itQuick},
    {"MOVEQ",	0x7000,		0xf100,	itMoveq},

    /*     4 bit mask	*/
    {"MOVE",	0x1000,		0xf000,	itMoveB},
    {"MOVE",	0x2000,		0xf000,	itMoveL},
    {"MOVE",	0x3000,		0xf000,	itMoveW},
    {"OR",	0x8000,		0xf000,	itOr},
    {"EOR",	0xb000,		0xf000,	itOr},
    {"AND",	0xc000,		0xf000,	itOr},
    {"",	NULL,		NULL,	0}
    };



/* 
Table of two word instructions.
The entire mask is comprised of instTwo.mask and instTwo.inst.mask 
*/

LOCAL INSTTWO instTwo [] =
    {
    /*     22 bit mask	*/
    {0x0cfc,	0xffff,	{"CAS2",	0x0000,	0x0e38,	itCas2}},
    {0x0efc,	0xffff,	{"CAS2",	0x0000,	0x0e38,	itCas2}},
    {0x00c0,	0xffc0,	{"CHK2",	0x0800,	0x0fff,	itChk2}},
    {0x00c0,	0xffc0,	{"CMP2",	0x0000,	0x0fff,	itChk2}},
    {0x02c0,	0xffc0,	{"CHK2",	0x0800,	0x0fff,	itChk2}},
    {0x02c0,	0xffc0,	{"CMP2",	0x0000,	0x0fff,	itChk2}},
    {0x04c0,	0xffc0,	{"CHK2",	0x0800,	0x0fff,	itChk2}},
    {0x04c0,	0xffc0,	{"CMP2",	0x0000,	0x0fff,	itChk2}},

    /*     21 bit mask	*/
    {0x06d0,	0xfff8,	{"CALLM",	0x0000,	0xff00,	itCallm}},
    {0x06e8,	0xfff8,	{"CALLM",	0x0000,	0xff00,	itCallm}},
    {0x06f0,	0xfff8,	{"CALLM",	0x0000,	0xff00,	itCallm}},
    {0x06f8,	0xfff8,	{"CALLM",	0x0000,	0xff00,	itCallm}},

    /*     20 bit mask	*/
    {0x0ac0,	0xffc0,	{"CAS",		0x0000,	0xfe38,	itCas}},
    {0x0cc0,	0xffc0,	{"CAS",		0x0000,	0xfe38,	itCas}},
    {0x0ec0,	0xffc0,	{"CAS",		0x0000,	0xfe38,	itCas}},

    /*     19 bit mask	*/
    {0x4c00,	0xffc0,	{"MULU",	0x0000,	0x8bf8,	itDivL}},
    {0x4c00,	0xffc0,	{"MULS",	0x0800,	0x8bf8,	itDivL}},
    {0x4c40,	0xffc0,	{"DIVU",	0x0000,	0x8bf8,	itDivL}},
    {0x4c40,	0xffc0,	{"DIVS",	0x0800,	0x8bf8,	itDivL}},

    {NULL,	NULL,	{"",		NULL,	NULL,	0}}
    };

/* 
Table of fpp instructions.
The entire mask is comprised of instFpp.mask and instFpp.inst.mask 
*/

LOCAL INSTTWO instFpp [] =
    {
    {0xf200, 0xffc0, {"FABS",		0x0018, 0xa07f, itFabs}},
    {0xf200, 0xffc0, {"FACOS",		0x001c, 0xa07f, itFacos}},
    {0xf200, 0xffc0, {"FADD",		0x0022, 0xa07f, itFadd}},
    {0xf200, 0xffc0, {"FASIN",		0x000c, 0xa07f, itFasin}},
    {0xf200, 0xffc0, {"FATAN",		0x000a, 0xa07f, itFatan}},
    {0xf200, 0xffc0, {"FATANH",		0x000b, 0xa07f, itFatanh}},
    {0xf200, 0xffc0, {"FCMP",		0x0038, 0xa07f, itFcmp}},
    {0xf200, 0xffc0, {"FCOS",		0x001d, 0xa07f, itFcos}},
    {0xf200, 0xffc0, {"FCOSH",		0x0019, 0xa07f, itFcosh}},
    {0xf200, 0xffc0, {"FDB",		0x0000, 0xffc0, itFdb}},
    {0xf200, 0xffc0, {"FDIV",		0x0020, 0xa07f, itFdiv}},
    {0xf200, 0xffc0, {"FETOX",		0x0010, 0xa07f, itFetox}},
    {0xf200, 0xffc0, {"FETOXM1",	0x0008, 0xa07f, itFetoxm1}},
    {0xf200, 0xffc0, {"FGETEXP",	0x001e, 0xa07f, itFgetexp}},
    {0xf200, 0xffc0, {"FGETMAN",	0x001f, 0xa07f, itFgetman}},
    {0xf200, 0xffc0, {"FINT",		0x0001, 0xa07f, itFint}},
    {0xf200, 0xffc0, {"FINTRZ",		0x0003, 0xa07f, itFintrz}},
    {0xf200, 0xffc0, {"FLOG10",		0x0015, 0xa07f, itFlog10}},
    {0xf200, 0xffc0, {"FLOG2",		0x0016, 0xa07f, itFlog2}},
    {0xf200, 0xffc0, {"FLOGN",		0x0014, 0xa07f, itFlogn}},
    {0xf200, 0xffc0, {"FLOGNP1",	0x0006, 0xa07f, itFlognp1}},
    {0xf200, 0xffc0, {"FMOD",		0x0021, 0xa07f, itFmod}},
    {0xf200, 0xffc0, {"FMOVE",		0x0000, 0xa07f, itFmove}},
    {0xf200, 0xffc0, {"FMOVE",		0x6000, 0xe000, itFmovek}},
    {0xf200, 0xffc0, {"FMOVE",		0x8400, 0xdfff, itFmovel}},
    {0xf200, 0xffc0, {"FMOVE",		0x8800, 0xdfff, itFmovel}},
    {0xf200, 0xffc0, {"FMOVE",		0x9000, 0xdfff, itFmovel}},
    {0xf200, 0xffc0, {"FMOVECR",	0x5c00, 0xfc00, itFmovecr}},
    {0xf200, 0xffc0, {"FMOVEM",		0xc000, 0xc700, itFmovem}},
    {0xf200, 0xffc0, {"FMOVEM",		0x8000, 0xc3ff, itFmovemc}},
    {0xf200, 0xffc0, {"FMUL",		0x0023, 0xa07f, itFmul}},
    {0xf200, 0xffc0, {"FNEG",		0x001a, 0xa07f, itFneg}},
    {0xf200, 0xffc0, {"FNOP",		0x0000, 0xffff, itFnop}},
    {0xf200, 0xffc0, {"FREM",		0x0025, 0xa07f, itFrem}},
    {0xf200, 0xffc0, {"FSCALE",		0x0026, 0xa07f, itFscale}},
    {0xf200, 0xffc0, {"FS",		0x0000, 0xffc0, itFs}},
    {0xf200, 0xffc0, {"FSGLDIV",	0x0024, 0xa07f, itFsgldiv}},
    {0xf200, 0xffc0, {"FSGLMUL",	0x0027, 0xa07f, itFsglmul}},
    {0xf200, 0xffc0, {"FSIN",		0x000e, 0xa07f, itFsin}},
    {0xf200, 0xffc0, {"FSINCOS",	0x0030, 0xa078, itFsincos}},
    {0xf200, 0xffc0, {"FSINH",		0x0002, 0xa07f, itFsinh}},
    {0xf200, 0xffc0, {"FSQRT",		0x0004, 0xa07f, itFsqrt}},
    {0xf200, 0xffc0, {"FSUB",		0x0028, 0xa07f, itFsub}},
    {0xf200, 0xffc0, {"FTAN",		0x000f, 0xa07f, itFtan}},
    {0xf200, 0xffc0, {"FTANH",		0x0009, 0xa07f, itFtanh}},
    {0xf200, 0xffc0, {"FTENTOX",	0x0012, 0xa07f, itFtentox}},
    {0xf200, 0xffc0, {"FTRAP",		0x0000, 0xffc0, itFtrap}},
    {0xf200, 0xffc0, {"FTST",		0x003a, 0xa07f, itFtst}},
    {0xf200, 0xffc0, {"FTWOTOX",	0x0011, 0xa07f, itFtwotox}},
    {NULL,	NULL,	{"",		NULL,	NULL,	0}}
    };

/* Forward declarations */

LOCAL INST *findTwoWord ();
LOCAL INST *findFpp ();


/*******************************************************************************
*
* dsmFind - disassemble one instruction
*
* This routine figures out which instruction is pointed to by binInst,
* and returns a pointer to the INST which describes it.  If no INST is
* found, returns NULL.
*/

LOCAL INST *dsmFind (binInst)
    USHORT binInst [];

    {
    FAST INST *iPtr;
    FAST INST *iPtr2;

    /* Find out which instruction it is */

    for (iPtr = &inst [0]; iPtr->mask != NULL; iPtr++)

	if ((binInst [0] & iPtr->mask) == iPtr->op)
	    {
	    if (iPtr->type == itTwoWord)
		{
	    	/* if binInst is a two word instruction, find out which one */

		if ((iPtr2 = findTwoWord (&binInst [0])) != NULL)
			return (iPtr2);
		}
	    else if (iPtr->type == itFpp)
		{
	    	/* if binInst is a fpp instruction, find out which one */

		if ((iPtr2 = findFpp (&binInst [0])) != NULL)
			return (iPtr2);
		}
	    else
	    	return (iPtr);
	    }

    /* If we're here, we couldn't find it */

    errnoSet (S_dsmLib_UNKNOWN_INSTRUCTION);
    return (NULL);
    }
/*******************************************************************************
*
* findTwoWord - find a two word instruction
*
* This routine figures out which (if any) of the two word instructions 
* matches the instruction pointed to by binInst, and returns a pointer
* to the INST which describes it.  Returns NULL if there is no match.
*/

LOCAL INST *findTwoWord (binInst)
    USHORT binInst [];

    {
    INSTTWO *iPtrTwo;

    for (iPtrTwo = &instTwo [0]; iPtrTwo->mask != NULL; iPtrTwo++)
	{
	if (((binInst [0] & iPtrTwo->mask) == iPtrTwo->op)
	    && ((binInst [1] & iPtrTwo->inst.mask) == iPtrTwo->inst.op))
		return (&iPtrTwo->inst);
	}
    return (NULL);
    }
/*******************************************************************************
*
* findFpp - find a fpp instruction
*
* This routine figures out which (if any) of the fpp instructions 
* matches the instruction pointed to by binInst, and returns a pointer
* to the INST which describes it.  Returns NULL if there is no match.
*/

LOCAL INST *findFpp (binInst)
    USHORT binInst [];

    {
    INSTTWO *iPtrFpp;

    for (iPtrFpp = &instFpp [0]; iPtrFpp->mask != NULL; iPtrFpp++)
	{
	if (((binInst [0] & iPtrFpp->mask) == iPtrFpp->op)
	    && ((binInst [1] & iPtrFpp->inst.mask) == iPtrFpp->inst.op))
		return (&iPtrFpp->inst);
	}
    return (NULL);
    }
/*******************************************************************************
*
* dsmPrint - print a disassembled instruction
*
* This routine prints an instruction in disassembled form.  It takes
* as input a pointer to the instruction, a pointer to the INST
* that describes it (as found by dsmFind), and an address with which to
* prepend the instruction.
*/

LOCAL VOID dsmPrint (binInst, iPtr, address, nwords, prtAddress)
    USHORT binInst [];	/* Pointer to instructin */
    FAST INST *iPtr;		/* Pointer to INST returned by dsmFind */
    int address;		/* Address with which to prepend instructin */
    int nwords;			/* Instruction length, in words */
    FUNCPTR prtAddress;		/* Address printing function */

    {
    FAST int ix;		/* index into binInst */
    FAST int wordsToPrint;	/* # of 5-char areas to reserve for printing
				   of hex version of instruction */
    wordsToPrint = (((nwords - 1) / 5) + 1) * 5;

    /* Print the address and the instruction, in hex */

    printf ("%06x  ", address);
    for (ix = 0; ix < wordsToPrint; ++ix)
	/* print lines in multiples of 5 words */
	{
	if ((ix > 0) && (ix % 5) == 0)		/* print words on next line */
	    printf ("\n        ");
	printf ((ix < nwords) ? "%04x " : "     ", binInst [ix]);
	}

    if (iPtr == NULL)
	{
	printf ("DC.W       $%04x\n", binInst[0]);
	return;
	}

    /* Print the instruction mnemonic, the size code (.w, or whatever), and
       the arguments */

    printf ("%-6s", iPtr->name);
    prtSizeField (binInst, iPtr);
    printf ("    ");
    prtArgs (binInst, iPtr, address, prtAddress);
    }
/*******************************************************************************
*
* dsmNwords - return the length (in words) of an instruction
*/

LOCAL int dsmNwords (binInst, iPtr)
    USHORT binInst [];
    FAST INST *iPtr;

    {
    int frstArg;	/* length of first argument */

    if (iPtr == NULL)
	return (1);			/* not an instruction */

    switch (iPtr->type)
	{
	case itLea:
	case itBcd:
	case itNbcd:
	case itDynBit:
	case itMemShift:
	case itMoveTSR:
	case itMoveFSR:
	case itMoveCCR:
	case itQuick:
	case itScc:
	case itMoveFCCR:
	case itDivW:
	case itCpSave:
	    return (1 + modeNwords ((binInst [0] & 0x0038) >> 3,
				    binInst [0] & 7, 1, &binInst [1]));

	case itMoves:
	case itStatBit:
	case itMovem:
	case itCallm:
	case itBfchg:
	case itBfext:
	case itBfins:
	case itCpGen:
	case itCpScc:
	    return (2 + modeNwords ((binInst [0] & 0x0038) >> 3,
				    binInst [0] & 7, 0, &binInst [2]));

	case itDivL:
	    return (2 + modeNwords ((binInst [0] & 0x0038) >> 3,
				    binInst [0] & 7, 2, &binInst [2]));

	case itAdda:
	    return (1 + modeNwords ((binInst [0] & 0x0038) >> 3,
				    binInst [0] & 7,
				    ((binInst [0] & 0x0100) == 0) ? 1 : 2, 
				    &binInst [1]));
	case itMoveA:
	    return (1 + modeNwords ((binInst [0] & 0x0038) >> 3,
				    binInst [0] & 7,
				    ((binInst [0] & 0x1000) == 0) ? 2 : 1,
				    &binInst [1]));
	case itNegx:
	case itOr:
	    return (1 + modeNwords ((binInst [0] & 0x0038) >> 3,
				    binInst [0] & 7,
				    ((binInst [0] & 0x0080) == 0) ? 1 : 2,
				    &binInst [1]));
	case itChk:
	    return (1 + modeNwords ((binInst [0] & 0x0038) >> 3,
				    binInst [0] & 7,
				    ((binInst [0] & 0x0080) == 0) ? 2 : 1,
				    &binInst [1]));

	case itChk2:
	    return (2 + modeNwords ((binInst [0] & 0x0038) >> 3,
				    binInst [0] & 7,
				    ((binInst [0] & 0x0400) == 0) ? 1 : 2,
				    &binInst [2]));

	case itCas:
	    return (2 + modeNwords ((binInst [0] & 0x0038) >> 3,
				    binInst [0] & 7,
				    ((binInst [0] & 0x0600) == 3) ? 2 : 1,
				    &binInst [2]));

	case itComplete:
	case itMoveq:
	case itExg:
	case itSwap:
	case itTrap:
	case itX:
	case itMoveUSP:
	case itUnlk:
	case itCmpm:
	case itExt:
	case itBkpt:
	case itRtm:
	case itRegShift:
	    return (1);

	case itMovep:
	case itStop:
	case itLink:
	case itDb:
	case itRTD:
	case itMovec:
	case itPack:
	case itImmCCR:
	case itImmTSR:
	    return (2);
	
	case itLinkL:
	case itCas2:
	case itCpDbcc:
	    return (3);

	case itBra:
	    switch (binInst [0] & 0xff)
		{
		case 0: 	return (2);
		case 0xff:	return (3);
		default:	return (1);
		}

	case itTrapcc:
	    return (((binInst [0] & 7) == 4) ? 1 : binInst [0] & 7);

	case itImm:
	    return ((((binInst [0] & 0x0080) == 0) ? 2 : 3) + 
		modeNwords ((binInst [0] & 0x0038) >> 3, binInst [0] & 7, 0,
			((binInst [0] & 0x0080) == 0) ? &binInst [2] 
			: &binInst [3]));

	case itMoveB:
	case itMoveW:
	    {
	    frstArg = modeNwords ((binInst [0] & 0x0038) >> 3,
				   binInst [0] & 7, 1, &binInst [1]);
	    return (1 + frstArg + modeNwords ((binInst [0] & 0x01c0) >> 6,
			    		      (binInst [0] & 0x0e00) >> 9, 1, 
			    		      &binInst [1 + frstArg]));
	    }
	case itMoveL:
	    {
	    frstArg = modeNwords ((binInst [0] & 0x0038) >> 3,
				    binInst [0] & 7, 2, &binInst [1]);
	    return (1 + frstArg + modeNwords ((binInst [0] & 0x01c0) >> 6,
				    	      (binInst [0] & 0x0e00) >> 9, 2, 
					      &binInst [1 + frstArg]));
	    }
	case itCpBcc:
	    return ((binInst [0] & 0x40) ? 3 : 2);

	case itCpTrapcc:
	    switch (binInst [0] & 7)
		{
		case 2:		return (3);
		case 3:		return (4);
		case 4:		return (2);
		}

	/* fpp instructions */
	case itFb:
	    if (binInst[0] & 0x0040)
		return(3);
	    else
		return(2);

	case itFrestore:
	case itFsave:
	    return(1 + modeNwords((binInst[0] & 0x0038) >> 3, 
			    binInst[0] & 0x7, 0, &binInst[1]));

	case itFmovek:
	case itFmovel:
	case itFmovem:
	case itFmovemc:
	    return(2 + fppNwords((binInst[0] & 0x0038) >> 3,
			    binInst[0] & 0x7, 1, 0, 0, &binInst[2]));

	case itFmovecr:
		return(2);

	case itFabs:
	case itFacos:
	case itFadd:
	case itFasin:
	case itFatan:
	case itFatanh:
	case itFcmp:
	case itFcos:
	case itFcosh:
	case itFdb:
	case itFdiv:
	case itFetox:
	case itFetoxm1:
	case itFgetexp:
	case itFgetman:
	case itFint:
	case itFintrz:
	case itFlog10:
	case itFlog2:
	case itFlogn:
	case itFlognp1:
	case itFmod:
	case itFmove:
	case itFmul:
	case itFneg:
	case itFnop:
	case itFrem:
	case itFscale:
	case itFs:
	case itFsgldiv:
	case itFsglmul:
	case itFsin:
	case itFsincos:
	case itFsinh:
	case itFsqrt:
	case itFsub:
	case itFtan:
	case itFtanh:
	case itFtentox:
	case itFtrap:
	case itFtst:
	case itFtwotox:
	    return(2 + fppNwords((binInst[0] & 0x0038) >> 3,
		    binInst[0] & 0x7, (binInst[1] & 0x4000) >> 14,
		    (binInst[1] & 0x1c00) >> 10,
		    (binInst[1] & 0x0380) >> 7, &binInst[2]));
	}
    /* We never get here, but just for lint ... */
    return (0);
    }
/*******************************************************************************
*
* fppNwords - number of words of extension used by the mode for fpp instrtns
*
* ARGSUSED
*/

LOCAL int fppNwords (mode, reg, rm, src, dest, extension)
    FAST USHORT mode;
    USHORT reg;
    USHORT rm;
    USHORT src;
    USHORT dest;
    USHORT extension [];    /* extension array for effective address */

    {
    if (rm == 0)
	return (0);

    else
	{
	if (mode == 0x7 && reg == 0x4)
	    {
	    switch (src)
		{
		case 0:
		case 1:
		    return (2);
		case 2:
		case 3:
		    return (6);
		case 4:
		    return (1);
		case 5:
		    return (4);
		case 6:
		    return (1);
		}
	    }
	else
	    return (modeNwords (mode, reg, 0, extension));
	}
    /* We never get here, but just for lint ... */
    return (0);
    }
/*******************************************************************************
*
* modeNwords - Figure out the number of words of extension needed for a mode
*/

LOCAL int modeNwords (mode, reg, size, extension)
    FAST USHORT mode;
    USHORT reg;
    int size;		/* the size of an immediate operand in words */
    USHORT extension [];/* extension array for effective address */

    {
    switch (mode)
	{
	case 0x00:    				/* Dn */
	case 0x01:				/* An */
	case 0x02:				/* (An) */
	case 0x03:				/* (An)+ */
	case 0x04:				/* -(An) */

	    return (0);

	case 0x05:				/* (d16,An) */

	    return (1);
	
	case 0x06:				/* reg indirect w.index modes */

	    return (mode6And7Words (&extension [0]));

	case 0x07:				/* memory indirect */
	    switch (reg)
		{
		/* With mode 7, sub-modes are determined by the
		   register number */

		case 0x0:			/* abs.short */
		case 0x2:			/* PC + off */
		     
		     return (1);

		case 0x3:			/* PC + ind + off */

		    return (mode6And7Words (&extension [0]));

		case 0x1:				/* abs.long */

		    return (2);

		case 0x4:				/* imm. */
		    return (size);
		}
	}
    /* We never get here, but just for lint ... */
    return (0);
    }
/*******************************************************************************
*
* mode6And7Words - number of words of extension needed for modes 6 and 7
*/

LOCAL int mode6And7Words (extension)
    FAST USHORT extension [];

    {
    int count = 1;	/* number of words in extension */

    if ((extension [0] & 0x0100) == 0)		/* (An) + (Xn) + d8 */
	return (count);

    /* 02m,26oct87,hin */
    /* if ((extension [0] & 0x80) == 0) */ /* base register added */
	switch ((extension [0] & 0x30) >> 4)	/* base displacement size */
	    {
	    case 0:		/* reserved or NULL displacement */
	    case 1:
		break;

	    case 2:		/* word displacement */
		count += 1;
		break;

	    case 3:		/* long displacement */
		count += 2;
		break;
	    }

    if ((extension [0] & 0x40) == 0)		/* index operand added */
	{
	switch (extension [0] & 3)
	    {
	    case 0:		/* reserved or NULL displacement */
	    case 1:
		break;

	    case 2:		/* word displacement */
		count += 1;
		break;

	    case 3:		/* long displacement */
		count += 2;
		break;
	    }
	}

    return (count);
    }
/*******************************************************************************
*
* prtArgs - Print the arguments for an instruction
*/

LOCAL VOID prtArgs (binInst, iPtr, address, prtAddress)
    USHORT binInst [];	/* Pointer to the binary instruction */
    FAST INST *iPtr;	/* Pointer to the INST describing binInst */
    int address;	/* Address at which the instruction resides */
    FUNCPTR prtAddress;	/* routine to print addresses. */

    {
    int frstArg;
    int displacement;

    switch (iPtr->type)
	{
	case itBra:
	    switch (binInst [0] & 0xff)
		{
		case 0:
		    displacement = (short) binInst[1];
		    break;
		case 0xff:
		    displacement = (((short) binInst [1] << 16) | binInst [2]);
		    break;
		default:
		    displacement = (char) (binInst [0] & 0xff);
		    break;
		}

	    (*prtAddress) (address + 2 + displacement);
	    printf ("\n");
	    break;

	case itDivW:
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 7,
			&binInst [1], 2, prtAddress);
	    printf (",D%x\n", (binInst [0] & 0x0e00) >> 9);
	    break;

	case itChk:
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 7,
			&binInst [1], (binInst [0] & 0x80) ? 2 : 4, prtAddress);
	    printf (",D%x\n", (binInst [0] & 0x0e00) >> 9);
	    break;

	case itChk2:
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 7,
			&binInst [2], 0, prtAddress);
	    printf ((binInst [1] & 0x8000) ? ",A%x\n" : ",D%x\n", 
		    (binInst [1] & 0x7000) >> 12);
	    break;

	case itLea:
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
			&binInst [1], 2, prtAddress);
	    printf (",A%x\n", (binInst [0] & 0x0e00) >> 9);
	    break;

	case itComplete:
	    /* No arguments */
	    printf ("\n");
	    break;

	case itStatBit:
	case itCallm:
	    printf ("#$%x,", binInst [1]);
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
			&binInst [2], 0, prtAddress);
	    printf ("\n");
	    break;

	case itDynBit:
	    printf ("D%x,", (binInst [0] & 0x0e00) >> 9);
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
			&binInst [1], 0, prtAddress);
	    printf ("\n");
	    break;

	case itExg:
	    switch ((binInst [0] & 0x00f8) >> 3)
		{
		case 0x08:
		    printf ("D%x,D%x\n", (binInst [0] & 0x0e00) >> 9,
					 binInst [0] & 0x0007);
		    break;
		case 0x09:
		    printf ("A%x,A%x\n", (binInst [0] & 0x0e00) >> 9,
					 binInst [0] & 0x0007);
		    break;
		case 0x11:
		    printf ("D%x,A%x\n", (binInst [0] & 0x0e00) >> 9,
					 binInst [0] & 0x0007);
		    break;
		}
	    break;

	case itImm:
	    switch ((binInst [0] & 0x00c0) >> 6)	/* size */
		{
		case 0:					/* byte */
		    printf ("#$%x,", binInst [1] & 0x00ff);
		    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
				&binInst [2], 0, prtAddress);
		    break;
		case 1:					/* word */
		    printf ("#$%x,", binInst [1]);
		    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
				&binInst [2], 0, prtAddress);
		    break;
		case 2:					/* long */
		    printf ("#$%x%04x,", binInst [1], binInst [2]);
		    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
				&binInst [3], 0, prtAddress);
		    break;
		}
	    printf ("\n");
	    break;

	case itMoveB:
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
			&binInst [1], 1, prtAddress);
	    printf (",");
	    frstArg = modeNwords ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
				  1, &binInst [1]);
	    prEffAddr ((binInst [0] & 0x01c0) >> 6, (binInst [0] & 0x0e00) >> 9,
			&binInst [1 + frstArg], 1, prtAddress);
	    printf ("\n");
	    break;

	case itMoveW:
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
			&binInst [1], 2, prtAddress);
	    printf (",");
	    frstArg = modeNwords ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
				  1, &binInst [1]);
	    prEffAddr ((binInst [0] & 0x01c0) >> 6, (binInst [0] & 0x0e00) >> 9,
			&binInst [1 + frstArg], 2, prtAddress);
	    printf ("\n");
	    break;

	case itMoveL:
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
			&binInst [1], 4, prtAddress);
	    printf (",");
	    frstArg = modeNwords ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
				  2, &binInst [1]);
	    prEffAddr ((binInst [0] & 0x01c0) >> 6, (binInst [0] & 0x0e00) >> 9,
			&binInst [1 + frstArg], 4, prtAddress);
	    printf ("\n");
	    break;

	case itImmCCR:
	    printf ("#$%x,CCR\n", binInst [1] & 0xff);
	    break;

	case itImmTSR:
	    printf ("#$%x,SR\n", binInst [1]);
	    break;

	case itMoveCCR:
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
			&binInst [1], 2, prtAddress);
	    printf (",CCR\n");
	    break;

	case itMoveTSR:
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
			&binInst [1], 2, prtAddress);
	    printf (",SR\n");
	    break;

	case itMoveFSR:
	    printf ("SR,");
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
			&binInst [1], 0, prtAddress);
	    printf ("\n");
	    break;

	case itMoveUSP:

	    printf (((binInst [0] & 0x8) == 0) ? "A%x,USP\n" : "USP,A%x\n",
		    binInst [0] & 0x07);
	    break;

	case itMovep:
	    if ((binInst [0] & 0x0040) == 0)

		printf ("%x(A%x),D%x\n", binInst [1], binInst [0] & 0x07,
			(binInst [0] & 0x0c00) >> 9);

	    else

		printf ("D%x,%x(A%x)\n", (binInst [0] & 0x0c00) >> 9,
			binInst [1], binInst [0] & 0x07);

	    break;

	case itMovem:
	    if ((binInst [0] & 0x0400) == 0)
		{
		prMovemRegs (binInst [1], (binInst [0] & 0x38) >> 3);
		printf (",");
		prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
			    &binInst [2], 0, prtAddress);
		printf ("\n");
		}
	    else
		{
		prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
			    &binInst [2], 0, prtAddress);
		printf (",");
		prMovemRegs (binInst [1], (binInst [0] & 0x38) >> 3);
		printf ("\n");
		}
	    break;

	case itMoveq:
	    printf ("#$%x,D%x\n", binInst [0] & 0x00ff,
		    (binInst [0] & 0x0e00) >> 9);
	    break;

	case itNbcd:
	case itNegx:
	case itCpSave:
	case itScc:
	case itMemShift:
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
			&binInst [1], 0, prtAddress);
	    printf ("\n");
	    break;

	case itCpGen:
	case itCpScc:
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
			&binInst [2], 0, prtAddress);
	    printf ("\n");
	    break;

	case itMoveA:
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
		&binInst [1], ((binInst [0] & 0x1000) == 0) ? 4 : 2,
		prtAddress);
	    printf (",A%x\n", (binInst [0] & 0x0e00) >> 9);
	    break;

	case itAdda:
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
		&binInst [1], ((binInst [0] & 0x0100) == 0) ? 2 : 4,
		prtAddress);
	    printf (",A%x\n", (binInst [0] & 0x0e00) >> 9);
	    break;

	case itCmpm:
	    printf ("(A%x)+,(A%x)+\n", (binInst [0] & 0x0007),
		(binInst [0] & 0x0e00) >> 9);
	    break;

	case itOr:
	    switch ((binInst [0] & 0x01c0) >> 6)	/* op-mode */
		{
		case 0:				/* byte, to D reg */
		    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
			&binInst [1], 1, prtAddress);
		    printf (",D%x\n", (binInst [0] & 0x0e00) >> 9);
		    break;

		case 1:				/* word to D reg */
		    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
			&binInst [1], 2, prtAddress);
		    printf (",D%x\n", (binInst [0] & 0x0e00) >> 9);
		    break;

		case 2:				/* long, to D reg */
		    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
			&binInst [1], 4, prtAddress);
		    printf (",D%x\n", (binInst [0] & 0x0e00) >> 9);
		    break;

		case 4:				/* byte, to eff address */
		case 5:				/* word, to eff address */
		case 6:				/* long, to eff address */
		    printf ("D%x,", (binInst [0] & 0x0e00) >> 9);
		    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
			&binInst [1], 0, prtAddress);
		    printf ("\n");
		    break;

		}
	    break;

	case itQuick:

	    /* If the data field is 0, it really means 8 */

	    if ((binInst [0] & 0x0e00) == 0)
		printf ("#8,");
	    else
		printf ("#%x,", (binInst [0] & 0x0e00) >> 9);

	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
			&binInst [1], 0, prtAddress);
	    printf ("\n");
	    break;

	case itBcd:
	    if ((binInst [0] & 0x0004) == 0)
		printf ("D%x,D%x\n", (binInst [0] & 0x0e00) >> 9,
				    binInst [0] & 0x0007);
	    else
		printf ("-(A%x),-(A%x)\n", (binInst [0] & 0x0e00) >> 9,
				    binInst [0] & 0x0007);
	    break;

	case itRegShift:
	    if ((binInst [0] & 0x0e20) == 0)
		printf ("#8,");
	    else if ((binInst [0] & 0x0020) == 0)
		printf ("#%x,", (binInst [0] & 0x0e00) >> 9);
	    else
		printf ("D%x,", (binInst [0] & 0x0e00) >> 9);
		
	    printf ("D%x\n", binInst [0] & 0x07);
	    break;

	case itTrapcc:
	    if ((binInst [0] & 7) == 2)
		printf ("#$%x", binInst [1]);
	    else if ((binInst [0] & 7) == 3)
		printf ("#$%x%04x", binInst [1], binInst [2]);
	    printf ("\n");
	    break;

	case itStop:
	    printf ("$%x\n", binInst [1]);
	    break;

	case itSwap:
	case itExt:
	    printf ("D%x\n", binInst [0] & 0x07);
	    break;

	case itUnlk:
	    printf ("A%x\n", binInst [0] & 0x07);
	    break;

	case itLink:
	    printf ("A%x,#%x\n", binInst [0] & 0x07, binInst [1]);
	    break;

	case itLinkL:
	    printf ("A%x,#%x%04x\n", binInst [0] & 7, binInst [1], binInst [2]);
	    break;

	case itRtm:
	    printf ((binInst [0] & 8) ? "A%x\n" : "D%x\n", binInst [0] & 7);
	    break;

	case itTrap:
	    printf ("#%x\n", binInst [0] & 0x0f);
	    break;

	case itBkpt:
	    printf ("#%x\n", binInst [0] & 7);
	    break;

	case itX:
	    printf (((binInst [0] & 0x08) == 0) ?
		    "D%x,D%x\n" : "-(A%x),-(A%x)\n",
		    binInst [0] & 0x07, (binInst [0] & 0x0e00) >> 9);
	    break;
	
	case itPack:
	    printf (((binInst [0] & 0x08) == 0) ?
		    "D%x,D%x," : "-(A%x),-(A%x),",
		    binInst [0] & 0x07, (binInst [0] & 0x0e00) >> 9);
	    printf ("#$%x\n", binInst [1]);
	    break;

	case itDb:
	    printf ("D%x,", binInst [0] & 0x07);
	    (*prtAddress) (address + 2 + (short) binInst [1]);
	    printf ("\n");
	    break;

	case itRTD:
	    printf ("#$%x\n", binInst [1]);
	    break;

	case itBfchg:
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 7, &binInst [2],
	              0, prtAddress);
	    prOffWid ((binInst [1] & 0x0800) >> 11, (binInst [1] & 0x07c0) >> 6,
		      (binInst [1] & 0x20) >> 5, binInst [1] & 0x1f);
	    printf ("\n");
	    break;

	case itBfext:
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 7, &binInst [2],
	              0, prtAddress);
	    prOffWid ((binInst [1] & 0x0800) >> 11, (binInst [1] & 0x07c0) >> 6,
		      (binInst [1] & 0x20) >> 5, binInst [1] & 0x1f);
	    printf (",D%x\n", (binInst [1] & 0x7000) >> 12);
	    break;

	case itBfins:
	    printf ("D%x,", (binInst [1] & 0x7000) >> 12);
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 7, &binInst [2],
	              0, prtAddress);
	    prOffWid ((binInst [1] & 0x0800) >> 11, (binInst [1] & 0x07c0) >> 6,
		      (binInst [1] & 0x20) >> 5, binInst [1] & 0x1f);
	    printf ("\n");
	    break;

	case itDivL:
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 7, &binInst [2],
			4, prtAddress);
	    printf (",");
	    if (binInst [1] & 0x0400)
		printf ("D%x:", binInst [1] & 7);
	    printf ("D%x\n", (binInst [1] & 0x7000) >> 12);
	    break;

	case itCas:
	    printf ("D%x,D%x,", binInst [1] & 7, (binInst [1] & 0x01c0) >> 6);
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 7, &binInst [2],
			0, prtAddress);
	    printf ("\n");
	    break;

	case itCas2:
	    printf ("D%x:D%x,D%x:D%x,", binInst [1] & 7, binInst [2] & 7,
		     (binInst [1] & 0x01c0) >> 6, (binInst [2] & 0x01c0) >> 6 );
	    printf (binInst [1] & 0x8000 ? "(A%x):" : "(D%x):", 
		     (binInst [1] & 0x7000) >> 12);
	    printf (binInst [2] & 0x8000 ? "(A%x)\n" : "(D%x)\n", 
		     (binInst [2] & 0x7000) >> 12);
	    break;

	case itMoves:
	    if ((binInst[1] & 0x0800) == 0)	    
		{
		/* from <ea> to general reg */

		prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
			    &binInst [2], 0, prtAddress);
		if ((binInst[1] & 0x8000) == 0)
		    printf (",D%x\n",(binInst[1] & 0x7000) >> 12);
		else 
		    printf (",A%x\n",(binInst[1] & 0x7000) >> 12);
		}
	    else 
		{
		/* from general reg to <ea> */
		if ( (binInst[1] & 0x8000) == 0)
		    printf ("D%x, ",(binInst[1] & 0x7000) >> 12);
		else 
		    printf ("A%x, ",(binInst[1] & 0x7000) >> 12);
		prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
			    &binInst [2], 0, prtAddress);
		printf ("\n");
		}
	    break;

	case itMovec:
	    if ((binInst[0] & 1) == 0)
		{
		/* control reg to general reg */
		prContReg (binInst[1] & 0x0fff);
		if ((binInst[1] & 0x8000) == 0)
		    printf (", D%x\n",(binInst[1] & 0x7000) >> 12);
		else
		    printf (", A%x\n",(binInst[1] & 0x7000) >> 12);
		}
	    else
		{
		/* general reg to control reg */
		if ((binInst[1] & 0x8000) == 0)
		    printf ("D%x,",(binInst[1] & 0x7000) >> 12);
		else
		    printf ("A%x,",(binInst[1] & 0x7000) >> 12);
		prContReg (binInst[1] & 0x0fff);
		printf ("\n");
		}
	    break;
	
	case itMoveFCCR:
	    printf ("CCR, ");
	    prEffAddr ((binInst [0] & 0x38) >> 3, binInst [0] & 0x7,
			&binInst [1], 0, prtAddress);
	    printf ("\n");
	    break;

	case itCpBcc:
	    if ((binInst [0] & 0x40) == 0)
		printf ("#$%x%04x\n", binInst [1], binInst [2]);
	    else
		printf ("#$%x\n", binInst [1]);
	    break;
	
	case itCpDbcc:
	    printf("D%x,#$%x\n", binInst [0] & 7, binInst [2]);
	    break;

	case itCpTrapcc:
	    switch (binInst [0] & 7)
		{
		case 2:	
		    printf ("#$%x\n", binInst [2]);
		    break;
		case 3:		
		    printf ("#$%x%04x\n", binInst [2], binInst [3]);
		    break;
		case 4:	
		    printf ("#0\n");
		    break;
		}
	    break;

	/* fpp instructions */
	case itFb:
		if(binInst[0] & 0x40)
			printf ("#$%x\n", binInst[1]);
		else
			printf ("#$%x%04x\n", binInst[1], binInst[2]);
		break;
	
	case itFrestore:
	case itFsave:
	case itFs:
		prEffAddr((binInst[0] & 0x38)>>3, binInst[0] & 0x7,
			&binInst[1], 0, prtAddress);
		printf ("\n");
		break;

	case itFdb:
		printf ("D%x,#$%x\n",binInst[0] & 0x7, binInst[2]);
		break;
	
	case itFtrap:
		switch(binInst[0] & 7)
		    {
		    case 2:
			    printf ("#$%x\n",binInst[2]); break;
		    case 3:
			    printf ("#$%x%04x\n", binInst[2], binInst[3]);
			    break;
		    case 4:
			    printf ("#0\n"); break;
		    }
		break;
	case itFtst:
		if (binInst[1] & 0x4000)
			prEffAddr((binInst[0] & 0x38)>>3, binInst[0] & 0x7,
				&binInst[2], 0, prtAddress);
		else
			printf ("F%x", (binInst[1] & 0x0380)>>7);
		printf ("\n");
		break;

	case itFmovek:
		printf ("F%x,", (binInst[1] & 0x0380)>>7);
		prEffAddr((binInst[0] & 0x38) >> 3, binInst[0] & 0x7,
			&binInst[2], 0, prtAddress);
		printf ("\n");
		break;

	case itFmovel:
		if (binInst[1] & 0x2000)
		    {
		    prFmovemcr ((binInst[1] & 0x1c00) >> 10);
		    printf (",");
		    prEffAddr((binInst[0] & 0x38) >> 3, binInst[0] & 0x7,
			    &binInst[2], 0, prtAddress);
		    }
		else
		    {
		    prEffAddr((binInst[0] & 0x38) >> 3, binInst[0] & 0x7,
			    &binInst[2], 0, prtAddress);
		    printf (",");
		    prFmovemcr ((binInst[1] & 0x1c00) >> 10);
		    }
		printf ("\n");
		break;
	
	case itFmovecr:
		printf ("#$%x,", binInst[1] & 0x7f);
		printf ("F%x\n",(binInst[1] & 0x0380) >> 7);
		break;

	case itFmovem:
		if (binInst[1] & 0x2000)
		    {
		    prFmovemr((binInst[1] & 0x1800) >> 11, binInst[1] & 0xff);
		    printf (",");
		    prEffAddr((binInst[0] & 0x38) >> 3, binInst[0] & 0x7,
			    &binInst[2], 0, prtAddress);
		    }
		else
		    {
		    prEffAddr((binInst[0] & 0x38) >> 3, binInst[0] & 0x7,
			    &binInst[2], 0, prtAddress);
		    printf (",");
		    prFmovemr ((binInst[1] & 0x1800) >> 11, binInst[1] & 0xff);
		    }
		printf ("\n");
		break;

	case itFmovemc:
		if (binInst[1] & 0x2000)
		    {
		    prFmovemcr ((binInst[1] & 0x1c00) >> 10);
		    printf (",");
		    prEffAddr((binInst[0] & 0x38) >> 3, binInst[0] & 0x7,
			    &binInst[2], 0, prtAddress);
		    }
		else
		    {
		    prEffAddr ((binInst[0] & 0x38) >> 3, binInst[0] & 0x7,
			    &binInst[2], 0, prtAddress);
		    printf (",");
		    prFmovemcr ((binInst[1] & 0x1c00) >> 10);
		    }
		printf ("\n");
		break;
	
	case itFsincos:
		if (binInst[1] & 0x4000)
			prEffAddr((binInst[0] & 0x38) >> 3, binInst[0] & 0x7,
				&binInst[2], 0, prtAddress);
		else
			printf ("F%x,", (binInst[1] & 0x1c00) >> 10);
		printf (",F%x:F%x\n", (binInst[1] & 0x7),
				(binInst[1] & 0x380) >> 7);
		break;

	case itFabs:
	case itFacos:
	case itFadd:
	case itFasin:
	case itFatan:
	case itFatanh:
	case itFcmp:
	case itFcos:
	case itFcosh:
	case itFdiv:
	case itFetox:
	case itFetoxm1:
	case itFgetexp:
	case itFgetman:
	case itFint:
	case itFintrz:
	case itFlog10:
	case itFlog2:
	case itFlogn:
	case itFlognp1:
	case itFmod:
	case itFmove:
	case itFmul:
	case itFneg:
	case itFnop:
	case itFrem:
	case itFscale:
	case itFsgldiv:
	case itFsglmul:
	case itFsin:
	case itFsinh:
	case itFsqrt:
	case itFsub:
	case itFtan:
	case itFtanh:
	case itFtentox:
	case itFtwotox:
		if (binInst[1] & 0x4000)
		    {
		    prEffAddr((binInst[0] & 0x38) >> 3, binInst[0] & 0x7,
			    &binInst[2], 0, prtAddress);
		    printf (",F%x", (binInst[1] & 0x0380) >> 7);
		    }
		else
		    printf ("F%x,F%x", (binInst[1] & 0x1c00) >> 10,
			    (binInst[1] & 0x0380) >> 7);
		printf ("\n");
		break;
	}
    }
/*******************************************************************************
*
* prContReg - print the name of a control register
*/

LOCAL VOID prContReg (contReg)
    USHORT contReg;

    {
    switch (contReg)
	{
	case SFC:
	    printf ("SFC");
	    break;
	case DFC:
	    printf ("DFC");
	    break;
	case CACR:
	    printf ("CACR");
	    break;
	case USP:
	    printf ("USP");
	    break;
	case VBR:
	    printf ("VBR");
	    break;
	case CAAR:
	    printf ("CAAR");
	    break;
	case MSP:
	    printf ("MSP");
	    break;
	case ISP:
	    printf ("ISP");
	    break;
	}
    }
/*******************************************************************************
*
* prEffAddr - Print the argument for an effective address
*/

LOCAL VOID prEffAddr (mode, reg, extension, size, prtAddress)
    FAST USHORT mode;	/* mode indicator */
    FAST USHORT reg;	/* register number (or special if mode = 7) */
    USHORT extension [];/* extension data, if required */
    int size;		/* size of extension, in bytes, for immediate */
    FUNCPTR prtAddress;	/* Function to print addresses */

    {
    switch (mode)	/* Effective mode */
	{
	case 0x00:    				/* Dn */
	    printf ("D%x", reg);
	    break;

	case 0x01:				/* An */
	    printf ("A%x", reg);
	    break;

	case 0x02:				/* (An) */
	    printf ("(A%x)", reg);
	    break;

	case 0x03:				/* (An)+ */
	    printf ("(A%x)+", reg);
	    break;

	case 0x04:				/* -(An) */
	    printf ("-(A%x)", reg);
	    break;

	case 0x05:				/* (d16,An) */
	    printf ("($%x,A%x)", extension [0], reg);
	    break;

	case 0x06:				/* addr reg + index + offset */
	    prIndirectIndex (&extension [0], mode, reg);
	    break;

	case 0x07:
	    switch (reg)
		{
		/* With mode 7, sub-modes are determined by the
		   register number */

		case 0x0:				/* abs.short */
		    (*prtAddress) (extension [0]);
		    break;

		case 0x1:				/* abs.long */
		    (*prtAddress) (extension [1] + (extension [0] << 16));
		    break;

		case 0x2:				/* PC + displacement */
		    printf ("($%x,PC)", extension [0]);
		    break;

		case 0x3:				/* rel + ind + off */
		    prIndirectIndex (&extension [0], mode, reg);
		    break;

		case 0x4:				/* imm. */
		    switch (size)
			{
			case 1:			/* 1 byte */
			    printf ("#$%x", extension [0] & 0xff);
			    break;
			case 2:			/* 2 bytes */
			    printf ("#$%x", extension [0]);
			    break;
			case 4:
			    printf ("#$%x%04x", extension [0], extension [1]);
			    break;
			}
		    break;
		}
	}
    }
/*******************************************************************************
*
* prMovemRegs - print the regs for a movem instruction
*/

LOCAL VOID prMovemRegs (extension, mode)
    FAST USHORT extension;
    FAST USHORT mode;

    {
    FAST int i;
    BOOL forward = mode == 4;

    /* If the mode is predecrement, the extension word has the bits in
       the opposite order, which explains all the weird conditionals below. */

    printf ("#<");
    for (i = 0; i <= 7; i++)
	{
	if ((extension & (forward ? 0x8000 : 1)) != 0)
	    printf ("D%x,", i);

	extension = forward ? extension << 1 : extension >> 1;
	}
    for (i = 0; i <= 7; i++)
	{
	if ((extension & (forward ? 0x8000 : 1)) != 0)
	    printf ("A%x,", i);

	extension = forward ? extension << 1 : extension >> 1;
	}
    printf (">");
    }
/*******************************************************************************
*
* prFmovemr - print the registers for a fmovemr instruction
*/

LOCAL VOID prFmovemr (mode, rlist)
    FAST USHORT mode;
    FAST USHORT rlist;

    {
    FAST int i;
    BOOL postincr = mode == 2 || mode == 3;

    printf ("#<");

    for (i = 0; i <= 7; i++)
	{
	if (rlist & (postincr ? 0x80 : 1))
	    printf ("F%x,", postincr ? i : 7 - i);
	rlist = postincr ? rlist << 1 : rlist >> 1;
	}

    printf (">");
    }
/*******************************************************************************
*
* prFmovemcr - print the regs for a fmovemcr instruction
*/

LOCAL VOID prFmovemcr (rlist)
    FAST USHORT rlist;

    {
    printf ("#<");
    if (rlist & 1)
	printf ("FPIAR,");
    if (rlist & 2)
	printf ("FPSR,");
    if (rlist & 4)
	printf ("FPCR");
    printf (">");
    }
/*******************************************************************************
*
* prtSizeField - print the size field of an instruction (.S, .W, or .L)
*/

LOCAL VOID prtSizeField (binInst, iPtr)
    USHORT binInst [];
    FAST INST *iPtr;

    {
    switch (iPtr->type)
	{
	case itLea:
	case itBcd:
	case itNbcd:
	case itMoveTSR:
	case itMoveFSR:
	case itMoveCCR:
	case itComplete:
	case itMoveq:
	case itExg:
	case itSwap:
	case itUnlk:
	case itTrap:
	case itMoveUSP:
	case itStop:
	case itBra:
	case itBfchg:
	case itBfext:
	case itBfins:
	case itDb:
	case itScc:
	case itRTD:
	case itMovec:
	case itMoveFCCR:
	case itBkpt:
	case itCallm:
	case itPack:
	case itRtm:
	case itImmCCR:
	case itImmTSR:
	case itCpGen:
	case itCpSave:
	case itMemShift:
	    printf ("  ");
	    break;

	case itDynBit:
	case itStatBit:
	    printf (((binInst [0] & 0x0038) == 0) ? ".L" : ".B");
	    break;

	case itNegx:
	case itQuick:
	case itX:
	case itImm:
	case itMoves:
	    switch ((binInst [0] & 0x00c0) >> 6)
		{
		case 0:
		    printf (".B");
		    break;
		case 1:
		case 3:
		    printf (".W");
		    break;
		case 2:
		    printf (".L");
		    break;
		}
	    break;

	case itOr:
	case itCmpm:
	case itRegShift:
	    switch ((binInst [0] & 0x00c0) >> 6)
		{
		case 0:
		    printf (".B");
		    break;
		case 1:
		    printf (".W");
		    break;
		case 2:
		    printf (".L");
		    break;
		}
	    break;

	case itCas:
	case itCas2:
	    switch ((binInst [0] & 0x0600) >> 9)
		{
		case 1:
		    printf (".B");
		    break;
		case 2:
		    printf (".W");
		    break;
		case 3:
		    printf (".L");
		    break;
		}
	    break;
	
	case itChk2:
	    switch ((binInst [0] & 0x0600) >> 9)
		{
		case 0:
		    printf (".B");
		    break;
		case 1:
		    printf (".W");
		    break;
		case 2:
		    printf (".L");
		    break;
		}
	    break;

	case itTrapcc:
	    switch (binInst [0] & 7)
		{
		case 2:
		    printf (".W");
		    break;

		case 3:
		    printf (".L");
		    break;

		case 4:
		    printf ("  ");
		    break;
		}
	    break;

	case itChk:
	    printf (((binInst [0] & 0x0080) == 0) ? ".W" : ".L");
	    break;

	case itMoveA:
	    printf (((binInst [0] & 0x1000) == 0) ? ".L" : ".W");
	    break;
	    
	case itAdda:
	    printf (((binInst [0] & 0x0100) == 0) ? ".W" : ".L");
	    break;
	    
	case itMovem:
	case itMovep:
	case itExt:
	case itCpBcc:
	    printf (((binInst [0] & 0x0040) == 0) ?  ".W" : ".L");
	    break;

	case itCpTrapcc:
	    switch (binInst [0] & 7)
		{
		case 2:	
		    printf (".W");
		    break;
		case 3:		
		    printf (".L");
		    break;
		case 4:
		    printf ("  ");
		    break;
		}
	    break;

	case itMoveB:
	case itCpScc:
	    printf (".B");
	    break;

	case itMoveW:
	case itDivW:
	case itLink:
	case itCpDbcc:
	    printf (".W");
	    break;

	case itMoveL:
	case itDivL:
	case itLinkL:
	    printf (".L");
	    break;

	/* fpp instructions */
	case itFb:
		if (binInst[0] & 0x0040)
			printf (".L");
		else
			printf (".W");
		break;

	case itFrestore:
	case itFsave:
	case itFdb:
	case itFnop:
		printf ("  ");
		break;
	
	case itFtrap:
		if ((binInst[0] & 0x7) == 0x2)
			printf (".W");
		else if ((binInst[0] & 0x7) == 0x3)
			printf (".L");
		else ;
		break;

	case itFmovek:
		switch ((binInst[1] & 0x1c00) >> 10)
		    {
		    case 0:		printf (".L"); break;
		    case 1:		printf (".S"); break;
		    case 2:		printf (".X"); break;
		    case 3:		printf (".P"); break;
		    case 4:		printf (".W"); break;
		    case 5:		printf (".D"); break;
		    case 6:		printf (".B"); break;
		    case 7:		printf (".P"); break;
		    }
		break;

	case itFmovel:
	case itFmovemc:
		printf (".L");
		break;

	case itFmovecr:
	case itFmovem:
		printf (".X");
		break;

	case itFabs:
	case itFacos:
	case itFadd:
	case itFasin:
	case itFatan:
	case itFatanh:
	case itFcmp:
	case itFcos:
	case itFcosh:
	case itFdiv:
	case itFetox:
	case itFetoxm1:
	case itFgetexp:
	case itFgetman:
	case itFint:
	case itFintrz:
	case itFlog10:
	case itFlog2:
	case itFlogn:
	case itFlognp1:
	case itFmod:
	case itFmove:
	case itFmul:
	case itFneg:
	case itFrem:
	case itFscale:
	case itFs:
	case itFsgldiv:
	case itFsglmul:
	case itFsin:
	case itFsincos:
	case itFsinh:
	case itFsqrt:
	case itFsub:
	case itFtan:
	case itFtanh:
	case itFtentox:
	case itFtst:
	case itFtwotox:
	    if (binInst[1] & 0x4000)
		{
		switch ((binInst[1] & 0x1c00)>>10)
		    {
		    case 0:
			    printf (".L"); break;
		    case 1:
			    printf (".S"); break;
		    case 2:
			    printf (".X"); break;
		    case 3:
			    printf (".P"); break;
		    case 4:
			    printf (".W"); break;
		    case 5:
			    printf (".D"); break;
		    case 6:
			    printf (".B"); break;
		    }
		}
		break;
	}
    }
/*******************************************************************************
*
* nPrtAddress - A dumb routine to print addresses as numbers.
*/

LOCAL VOID nPrtAddress (address)
    int address;

    {
    printf ("$%x", address);
    }
/*******************************************************************************
*
* prOffWid - print the offset and width fields of an instruction
*/

LOCAL VOID prOffWid (dO, offset, dW, width)
    FAST USHORT dO;		/* if dO is true, offset in data reg */
    FAST USHORT offset;		/* field offset */
    FAST USHORT dW;		/* if dW is true, width in data reg */
    FAST USHORT width;		/* field width */

    {
    printf ( dO ? "{D%x:" : "{#%d:", offset);
    printf ( dW ? "D%x}" : "#%d}", width);
    }
/*******************************************************************************
*
* prIndirectIndex - print memory indirect with index, address register or 
* 	program counter
*
* Assumes only modes 6 and 7 of effective addressing
*/

LOCAL VOID prIndirectIndex ( extension, mode, reg)
    USHORT extension [];
    USHORT mode;
    USHORT reg;

    {
    USHORT scale;		/* scaling factor */
    USHORT bdSize;		/* base displacement size */
    USHORT iS;			/* index suppress bit */
    USHORT indexReg;		/* index register number */
    USHORT indexIndirect;	/* index/indirect selection */

    scale 	= (extension [0] & 0x0600) >> 9;
    bdSize 	= (extension [0] & 0x0030) >> 4;
    iS		= (extension [0] & 0x0040) >> 6;
    indexReg	= (extension [0] & 0x7000) >> 12;
    indexIndirect = extension [0] & 7;

    if ((extension [0] & 0x0100) == 0)
	{
    	/* address register indirect with index (8-bit displacement) */
	
	printf ("($%x,", extension [0] & 0xff); /* print displacement */

	/* print address register */

	if (mode == 7)
	    printf ("PC,");
	else
	    printf ("A%x,", reg);

	/* print the index register.  The high order bit of the
	   extension word determines whether it's a D or A reg */

	printf (extension [0] & 0x8000 ? "A%x" : "D%x", indexReg);

	/* determine whether the index register is a word or a long,
	   also determined by a bit in the extension word */

	printf (extension [0] & 0x0800 ? ".L" : ".W");

	/* print the scaling factor */

	printf ("*%d)", 1 << scale);
	}

    else if ((iS == 1) || ((iS == 0) && (indexIndirect == 0)))
	{
    	/* address register indirect with index (base displacement) 
	 * or PC indirect with index (base displacement)
	 */

	printf ("(");

	/* print the base displacement, address register, index register, 
	   length, and scaling factor */

	prDisplace (bdSize, &extension [1]);
	if (mode == 7)
	    printf (",PC,");
	else if ((extension[0] & 0x80) == 0)
	    printf (",A%x,", reg);
	else
	    printf (",");
	printf (extension [0] & 0x8000 ? "A%x" : "D%x", indexReg);
	printf (extension [0] & 0x0800 ? ".L" : ".W");
	printf ("*%d)", 1 << scale);
	}
    
    else if ((iS == 0) && ((indexIndirect > 4) && (indexIndirect < 8)))
    	/* memory indirect post-indexed */
	{
	printf ("([");

	/* print the base displacement, address register, index register, 
	   length, and scaling factor */

	prDisplace (bdSize, &extension [1]);
	if (mode == 7)
	    printf (",PC],");
	else
	    printf (",A%x],", reg);
	printf (extension [0] & 0x8000 ? "A%x" : "D%x", indexReg);
	printf (extension [0] & 0x0800 ? ".L" : ".W");
	printf ("*%d,", 1 << scale);

	/* print the outer displacement */

	prDisplace (extension [0] & 7, &extension [0] + bdSize);
	printf (")");
	}

    else if ((iS == 0) && (indexIndirect >= 1) && (indexIndirect <= 3))
    	/* memory indirect pre-indexed */
	{
	printf ("([");

	/* print the base displacement, address register, index register, 
	   length, and scaling factor */

	prDisplace (bdSize, &extension [1]);
	if (mode == 7)
	    printf (",PC,");
	else
	    printf (",A%x,", reg);
	printf (extension [0] & 0x8000 ? "A%x" : "D%x", indexReg);
	printf (extension [0] & 0x0800 ? ".L" : ".W");
	printf ("*%d],", 1 << scale);

	/* print the outer displacement */

	prDisplace (extension [0] & 7, &extension [0] + bdSize);
	printf (")");
	}
    }
/*******************************************************************************
*
* prDisplace - print displacement
*
* Used for printing base and outer displacements.  Only looks at two least
* significant bits of the size.
*/

LOCAL VOID prDisplace (size, pDisp)
    USHORT size;
    USHORT pDisp [];

    {
    switch (size & 3)
	{
	case 1:				/* NULL displacement */
	    printf ("0");
	    break;

	case 2:				/* word displacement */
	    printf ("$%x", pDisp [0]);
	    break;

	case 3:				/* long displacement */
	    printf ("$%x%04x", pDisp [0], pDisp [1]);
	    break;
	}
    }
/**************************************************************************
*
* dsmData - disassemble and print a word as data
*
* This routine is called to disassemble and print (on standard out) a
* single 16 bit word as data (that is, as a DC.W assembler directive).
* The disassembled data will be prepended with the address passed as
* a parameter.
*
* RETURNS : The number of words occupied by the data (always 1).
*/

int dsmData (binInst, address)
    USHORT *binInst;	/* Pointer to the data */
    int address;	/* Address prepended to data */
    
    {
    dsmPrint (binInst, (INST *)NULL, address, 1, nPrtAddress);
    return (1);
    }
/*******************************************************************************
*
* dsmInst - disassemble and print a single instruction
*
* This routine is called to disassemble and print (on standard out) a
* single instruction.  The function passed as parameter prtAddress will
* be used to print any operands which might be construed as addresses.
* This could be either a simple routine that just prints a number,
* or one that looks up the address in a symbol table.
* The disassembled instruction will be prepended with the address
* passed as a parameter.
*
* If prtAddress == 0, a default routine will be used that prints 
* addresses as hex numbers.
*
* RETURNS : The number of 16-bit words occupied by the instruction.
*/

int dsmInst (binInst, address, prtAddress)
    FAST USHORT *binInst;	/* Pointer to the instruction */
    int address;		/* Address prepended to instruction */
    FUNCPTR prtAddress;		/* Address printing function */
    
    {
    FAST INST *iPtr;
    FAST int size;

    if (prtAddress == NULL)
	prtAddress = nPrtAddress;

    iPtr = dsmFind (binInst);
    size = dsmNwords (binInst, iPtr);
    dsmPrint (binInst, iPtr, address, size, prtAddress);
    return (size);
    }
/*******************************************************************************
*
* dsmNbytes - find out how big an instruction is
*
* This routine tells how big an instruction is, in bytes. 
*
* RETURNS:
*    size of instruction, or
*    0 if instruction is unrecognized
*/

int dsmNbytes (binInst)
    FAST USHORT *binInst;	/* Pointer to the instruction */
    
    {
    FAST INST *iPtr = dsmFind (binInst);

    return ((iPtr == NULL) ? 0 : 2 * dsmNwords (binInst, iPtr));
    }
