/* bootLib.c - boot rom subroutine library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01p,23may89,dnw  made promptRead be LOCAL.
01o,20aug88,gae  documentation
01n,30may88,dnw  changed to v4 names.
01m,29may88,dnw  changed calls to atoi() to sscanf().
01l,28may88,dnw  moved skipSpace here from fioLib.
		 changed calls to fioStdIn to STD_IN.
		 made promptParam...() LOCAL again (copies in nfsLib now).
01k,19apr88,llk  made promptParamNum and promptParamString global routines.
01j,19feb88,dnw  added bootExtractNetmask().
01i,18nov87,ecs  lint.
		 added include of strLib.h.
		 documentation.
01h,15nov87,dnw  changed to print '?' for unprintable chars in parameters.
01g,06nov87,jlf  documentation
01f,13oct87,dnw  added flags field to boot line encoding and decoding.
01e,09oct87,jlf  changed to deal with new ISI boot format with ethernet adrs.
01d,14may87,rdc  procnum can now be defined by a line VBNUM=n or VB=n for
	   +dnw    compatability with isi bootproms.
		 changed prompt to indicate password is for ftp only.
		 fixed bootEncodeParams() to add EOS.
01c,23mar87,jlf  documentation.
		 changed routine names to meet naming conventions.
01b,20dec86,dnw  added promptBootParams().
		 changed to not get include files from default directories.
01a,18dec86,llk  created.
*/

/*
DESCRIPTION
This library contains routines for manipulating a boot line.
Routines are provided to interpret, construct, print, and prompt for
a boot line.

When VxWorks is first booted, certain parameters are specified pertaining
to network addresses, boot device, host, file, etc.
This information is encoded into a single ASCII string known as the boot line.
The boot line is placed at a known address (specified in config.h)
by the boot roms so that the system being booted can discover the parameters
that were used to boot the system.
The boot line is the ONLY means of communication from the boot roms to
the booted system.

The boot line is of the form:

bootdev(0,procnum)hostname:filename e=# b=# h=# g=# u=userid pw=passwd f=#

EXAMPLE
.CS
 enp(0,0)host:/usr/vw/mz7122/config/vxWorks e=90.0.0.2 b=91.0.0.2
              h=100.0.0.4 g=90.0.0.3 u=vxWorks pw=realtime f=2


 where:
   bootdev  = "ex" for excelan ethernet, "bp" for backplane, etc.
		 - for "bp" this field can have an optional anchor
		   address specification of the form "bp=<adrs>"
		   (see bootBpAnchorExtract(1))
   procnum  = the processor number on the backplane (0..n).
   hostname = the name of the host to boot from.
   filename = the file to be booted.
   e        = the internet address of the ethernet interface.
		 - this field can have an optional subnet mask
		   of the form <inet_adrs>:<subnet_mask>
		   (see bootNetmaskExtract(1))
   b        = the internet address of the backplane interface.
		 - this field can have an optional subnet mask
		   as above
   h        = the internet address of the host to boot from.
   g        = the internet address of the gateway to the host.
		 - leave blank if host is on same network
   u        = a valid user name on the host.
   pw       = password for user on the host.
	         - usually left blank;
                 - if specified, then "ftp" is used for file
		   transfers
   f        = system dependent configuration flags.
		 - contains an `or' of option bits defined in
		   sysLib.h
.CE
 Internet addresses are specified in standard "dot" notation, e.g. 90.0.0.2.
 The order of assigned values is arbitrary.

SEE ALSO: bootConfig (1)
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "ctype.h"
#include "strLib.h"
#include "sysLib.h"

#define PARAM_PRINT_WIDTH	21

LOCAL char strBootDevice []		= "boot device";
LOCAL char strHostName []		= "host name";
LOCAL char strFileName []		= "file name";
LOCAL char strInetOnEthernet []		= "inet on ethernet (e)";
LOCAL char strInetOnBackplane []	= "inet on backplane (b)";
LOCAL char strHostInet []		= "host inet (h)";
LOCAL char strGatewayInet []		= "gateway inet (g)";
LOCAL char strUser []			= "user (u)";
LOCAL char strFtpPassword []		= "ftp password (pw)";
LOCAL char strFtpPasswordLong []	= "ftp password (pw) (blank = use rsh)";
LOCAL char strProcNum []		= "processor number";
LOCAL char strFlags []			= "flags (f)";


/*******************************************************************************
*
* bootStringToParams - interpret the boot parameters from the boot line
*
* This routine parses the ASCII string and returns the values into
* the provided parameters.
*
* See the manual entry for bootLib(1) for a description of the format
* of the boot line.
*
* RETURNS:
* pointer to the last character successfully parsed plus one
* (points to EOS if OK)
*/

char *bootStringToParams (bootParams, bootDev, hostName, bootFile, 
		          ead, bad, had, gad, usr, passwd, pProcNum, pFlags)
    char *bootParams;	/* boot line to be parsed              */
    char *bootDev;	/* name of boot device                 */
    char *hostName;	/* name of host                        */
    char *bootFile;	/* name of file to boot                */
    char *ead;		/* ethernet internet address           */
    char *bad;		/* backplane internet address          */
    char *had;		/* host internet address               */
    char *gad;		/* gateway internet address            */
    char *usr;		/* user id                             */
    char *passwd;	/* password                            */
    int *pProcNum;	/* where to return processor number    */
    int *pFlags;	/* where to return configuration flags */

    {
    char *p1 = bootParams;
    char dummy [BOOT_FIELD_LEN];
    char procNum [BOOT_FIELD_LEN];
    char flags [BOOT_FIELD_LEN];

    /* initialize boot parameters */

    bootDev [0]  = EOS;
    hostName [0] = EOS;
    bootFile [0] = EOS;
    ead [0]      = EOS;
    bad [0]      = EOS;
    had [0]      = EOS;
    gad [0]      = EOS;
    usr [0]      = EOS;
    passwd [0]   = EOS;
    procNum [0]  = 0;
    flags [0]    = 0;

    *pProcNum    = NONE;
    *pFlags	 = 0;


    /* get boot device and proc num */

    if (!getWord  (&p1, bootDev, BOOT_FIELD_LEN, " \t(")	||
	!getConst (&p1, "(")					||
	!getNum   (&p1, (int *) dummy)				||
	!getConst (&p1, ",")					||
	!getNum   (&p1, pProcNum)				||
	!getConst (&p1, ")"))
	{
	return (p1);
	}


    /* skip over the ethernet address stuck in by ISI, if its there. */

    if (*p1 == '(')
	{
	getWord (&p1, dummy, BOOT_FIELD_LEN, ">");
	++p1;
	}

    /* get host name and boot file */

    if (!getWord  (&p1, hostName, BOOT_FIELD_LEN, " \t:")	||
	!getConst (&p1, ":")					||
	!getWord  (&p1, bootFile, BOOT_FIELD_LEN, " \t"))
	{
	return (p1);
	}

    /* get optional assignments */

    while (skipSpace (&p1), (*p1 != EOS))
	{
	if (!getAssign (&p1, "ead", ead) &&
	    !getAssign (&p1, "bad", bad) &&
	    !getAssign (&p1, "had", had) &&
	    !getAssign (&p1, "gad", gad) &&
	    !getAssign (&p1, "usr", usr) &&
	    !getAssign (&p1, "pw", passwd) &&
	    !getAssignNum (&p1, "vbnum", pProcNum) &&
	    !getAssign (&p1, "e", ead) &&
	    !getAssign (&p1, "b", bad) &&
	    !getAssign (&p1, "h", had) &&
	    !getAssign (&p1, "g", gad) &&
	    !getAssign (&p1, "u", usr) &&
	    !getAssignNum (&p1, "vb", pProcNum) &&
	    !getAssignNum (&p1, "n", pProcNum) &&
	    !getAssignNum (&p1, "f", pFlags))
	    {
	    break;
	    }
	}

    return (p1);
    }
/*******************************************************************************
*
* bootParamsToString - construct a boot line
*
* This routine encodes a boot line using the specified boot parameters.
*
* See the manual entry for bootLib(1) for a description of the format
* of the boot line.
*
* RETURNS: OK
*/

STATUS bootParamsToString (paramString, bootDev, hostName, bootFile,
			   ead, bad, had, gad, usr, passwd, procNum, flags)
    char *paramString;	/* where to return the encoded boot line */
    char *bootDev;	/* name of boot device                   */
    char *hostName;	/* name of host                          */
    char *bootFile;	/* name of file to boot                  */
    char *ead;		/* ethernet internet address             */
    char *bad;		/* backplane internet address            */
    char *had;		/* host internet address                 */
    char *gad;		/* gateway internet address              */
    char *usr;		/* user id                               */
    char *passwd;	/* password                              */
    int procNum;	/* processor number                      */
    int flags;		/* configuration flags                   */

    {
    sprintf (paramString, "%s(0,%d)%s:%s", bootDev, procNum, hostName,
	     bootFile);
    
    addAssignString (paramString, "e", ead);
    addAssignString (paramString, "b", bad);
    addAssignString (paramString, "h", had);
    addAssignString (paramString, "g", gad);
    addAssignString (paramString, "u", usr);
    addAssignString (paramString, "pw", passwd);

    addAssignNum (paramString, "f", flags);

    return (OK);
    }
/*******************************************************************************
*
* bootParamsShow - print boot line parameters
*
* This routine prints the boot parameters in the specified boot string
* in a verbose multi-line display.
*/

VOID bootParamsShow (paramString)
    char *paramString;		/* boot parameter string */

    {
    char bootDev [BOOT_FIELD_LEN];	/* boot device code */
    char hostName [BOOT_FIELD_LEN];	/* name of host */
    char bootFile [BOOT_FIELD_LEN];	/* file Name */
    char ead [BOOT_FIELD_LEN];		/* ethernet internet addr */
    char bad [BOOT_FIELD_LEN];		/* backplane internet addr */
    char had [BOOT_FIELD_LEN];		/* host internet addr */
    char gad [BOOT_FIELD_LEN];		/* gateway internet addr */
    char usr [BOOT_FIELD_LEN];		/* user name */
    char passwd [BOOT_FIELD_LEN];	/* password */
    int  procNum;			/* processor number */
    int  flags;				/* configuration flags */

    bootStringToParams (paramString, bootDev, hostName, bootFile, ead, bad,
		        had, gad, usr, passwd, &procNum, &flags);

    printf ("\n");

    printParamString (strBootDevice, bootDev);
    printParamString (strHostName, hostName);
    printParamString (strFileName, bootFile);
    printParamString (strInetOnEthernet, ead);
    printParamString (strInetOnBackplane, bad);
    printParamString (strHostInet, had);
    printParamString (strGatewayInet, gad);
    printParamString (strUser, usr);
    printParamString (strFtpPassword, passwd);

    if (procNum != NONE)
	printf ("%-*s: %d\n", PARAM_PRINT_WIDTH, strProcNum, procNum);

    if (flags != 0)
	printf ("%-*s: 0x%x\n", PARAM_PRINT_WIDTH, strFlags, flags);

    printf ("\n");
    }
/*******************************************************************************
*
* bootParamsPrompt - prompt for boot line parameters
*
* This routine prompts the user for each of the boot paramters.
* For each parameter, the current value is displayed and
* a new value is prompted for.
*
*   Typing a <CR> leaves the parameter unchanged.
*   Typing a `.' clears the parameter.
*
* The string passed as a parameter is used for the initial values.
* A new boot line will be produced, changed as the user specifies.
* That line will be copied over the string passed as a parameter.
* If there are no initial values, the string should be empty on entry.
*/

VOID bootParamsPrompt (string)
    char *string;	/* default boot line;  may be changed */

    {
    char bootDev [BOOT_FIELD_LEN];	/* boot device code */
    char hostName [BOOT_FIELD_LEN];	/* name of host */
    char bootFile [BOOT_FIELD_LEN];	/* file Name */
    char ead [BOOT_FIELD_LEN];		/* ethernet internet addr */
    char bad [BOOT_FIELD_LEN];		/* backplane internet addr */
    char had [BOOT_FIELD_LEN];		/* host internet addr */
    char gad [BOOT_FIELD_LEN];		/* gateway internet addr */
    char usr [BOOT_FIELD_LEN];		/* user name */
    char passwd [BOOT_FIELD_LEN];	/* password */
    int  procNum;			/* processor number */
    int  flags;				/* configuration flags */
    FAST int n = 0;
    FAST int i;

    /* interpret the boot parameters */

    bootStringToParams (string, bootDev, hostName, bootFile, ead, bad,
		        had, gad, usr, passwd, &procNum, &flags);

    printf ("\n'.' = clear field;  '-' = go to previous field;  ^D = quit\n\n");

    /* prompt the user for each item;
     *   i:  0 = same field, 1 = next field, -1 = previous field,
     *     -98 = error, -99 = quit
     */

    FOREVER
	{
	switch (n)
	    {
	    case 0:  i = promptParamString (strBootDevice, bootDev);	break;
	    case 1:  i = promptParamString (strHostName, hostName);	break;
	    case 2:  i = promptParamString (strFileName, bootFile);	break;
	    case 3:  i = promptParamString (strInetOnEthernet, ead);	break;
	    case 4:  i = promptParamString (strInetOnBackplane, bad);	break;
	    case 5:  i = promptParamString (strHostInet, had);		break;
	    case 6:  i = promptParamString (strGatewayInet, gad);	break;
	    case 7:  i = promptParamString (strUser, usr);		break;
	    case 8:  i = promptParamString (strFtpPasswordLong, passwd);break;
	    case 9:  i = promptParamNum (strProcNum, &procNum, FALSE);	break;
	    case 10: i = promptParamNum (strFlags, &flags, TRUE);	break;
	    default: i = -99; break;
	    }

	/* check for QUIT */

	if (i == -99)
	    {
	    printf ("\n");
	    break;
	    }
	
	/* move to new field */

	if (i != -98)
	    n += i;
	}

    bootParamsToString (string, bootDev, hostName, bootFile, ead, bad, had,
		        gad, usr, passwd, procNum, flags);
    }
/*******************************************************************************
*
* bootNetmaskExtract - extract netmask field from adrs:mask field
*
* This routine extracts the optional subnet mask field from an internet address
* field.  Subnet masks can be specified for an internet interface by appending
* to the internet address a colon and the net mask in hex.  For example, the
* "inet on ethernet" field of the boot parameters could be specified as:
*
*    inet on ethernet: 90.1.0.1:ffff0000
*
* In this case, the network portion of the address (normally just 90)
* is extended by the subnet mask (to 90.1).  This routine picks off the
* optional trailing subnet mask by replacing the colon in the specified
* string with an EOS and then scanning the remainder as a hex number.
* This number, the net mask, is returned via the ``pNetmask'' pointer.
*
* RETURNS:
*    1 if netmask specified correctly
*    0 if netmask not specified
*   -1 if invalid netmask specified
*/

STATUS bootNetmaskExtract (string, pNetmask)
    char *string;	/* string containing adrs field */
    int *pNetmask;	/* pointer where to return netmask */

    {
    return (bootSubfieldExtract (string, pNetmask, ':'));
    }
/******************************************************************************
*
* bootBpAnchorExtract - extract backplane anchor field from bp=addr field
*
* This routine extracts the optional backplane anchor address field from a
* boot device field.  The anchor can be specified for the backplane
* driver by appending to the device name (i.e. "bp") a "=" and the
* address in hex.  For example, the "boot device" field of the boot
* parameters could be specified as:
*
*    boot device: bp=800000
*
* In this case, the backplane anchor address would be at address 0x800000,
* instead of the default specified in config.h.
*
* This routine picks off the optional trailing anchor address by replacing
* the "=" in the specified string with an EOS and then scanning the
* remainder as a hex number.
* This number, the anchor address, is returned via the ``pAnchorAdrs'' pointer.
*
* RETURNS:
*    1 if anchor address specified correctly
*    0 if anchor address not specified
*   -1 if invalid anchor address specified
*/

STATUS bootBpAnchorExtract (string, pAnchorAdrs)
    char *string;	/* string containing adrs field */
    char **pAnchorAdrs;	/* pointer where to return anchor address */

    {
    return (bootSubfieldExtract (string, (int *) pAnchorAdrs, '='));
    }


/******************************************************************************
*
* bootSubfieldExtract - extract a numeric subfield from a boot field
*
* Extracts subfields in fields of the form "<field><delimeter><subfield>".
* i.e. <inet>:<netmask> and bp=<anchor>
*/  

LOCAL STATUS bootSubfieldExtract (string, pValue, delimeter)
    char *string;	/* string containing field to be extracted */
    int *pValue;	/* pointer where to return value */
    char delimeter;	/* character delimeter */

    {
    FAST char *pDelim;
    int value;

    /* find delimeter */

    pDelim = index (string, delimeter); 
    if (pDelim == NULL)
	return (0);		/* no subfield specified */

    /* scan remainder for numeric subfield */

    string = pDelim + 1;

    if ((scanNum (&string, &value, TRUE) != OK) || (*string != EOS))
	return (-1);		/* invalid subfield specified */

    *pDelim = EOS;		/* terminate string at the delimeter */
    *pValue = value;		/* return value */
    return (1);			/* valid subfield specified */
    }
/*******************************************************************************
*
* addAssignNum - add a numeric value assignment to a string
*/

LOCAL VOID addAssignNum (string, code, value)
    FAST char *string;
    char *code;
    int value;

    {
    if (value != 0)
	{
	string += strlen (string);
	sprintf (string, (value <= 7) ? " %s=%d" : " %s=0x%x", code, value);
	}
    }
/*******************************************************************************
*
* addAssignString - add a string assignment to a string
*/

LOCAL VOID addAssignString (string, code, value)
    FAST char *string;
    char *code;
    char *value;

    {
    if (value[0] != EOS)
	{
	string += strlen (string);
	sprintf (string, " %s=%s", code, value);
	}
    }
/*******************************************************************************
*
* getWord - get a word out of a string
*
* Words longer than the specified max length are truncated.
*
* RETURNS: TRUE if word is successfully extracted from string, FALSE otherwise;
* Also updates ppString to point to next character following extracted word.
*/

LOCAL BOOL getWord  (ppString, pWord, length, delim)
    char **ppString;	/* ptr to ptr to string from which to get word */
    FAST char *pWord;	/* where to return word */
    int length;		/* max length of word to get including EOS */
    char *delim;	/* string of delimiters that can terminate word */

    {
    FAST char *pStr;

    skipSpace (ppString);

    /* copy up to any specified delimeter, EOS, or max length */

    pStr = *ppString;
    while ((--length > 0) && (*pStr != EOS) && (index (delim, *pStr) == 0))
	*(pWord++) = *(pStr++);

    *pWord = EOS;


    /* if we copied anything at all, update pointer and return TRUE */

    if (pStr != *ppString)
	{
	*ppString = pStr;
	return (TRUE);
	}


    /* no word to get */

    return (FALSE);
    }
/*******************************************************************************
*
* getConst - get a constant string out of a string
*
* case insensitive compare for identical strings
*/

LOCAL BOOL getConst (ppString, pConst)
    char **ppString;
    FAST char *pConst;

    {
    FAST int ch1;
    FAST int ch2;
    FAST char *pString;

    skipSpace (ppString);

    for (pString = *ppString; *pConst != EOS; ++pString, ++pConst)
	{
	ch1 = *pString;
	ch1 = (isascii (ch1) && isupper (ch1)) ? tolower (ch1) : ch1;
	ch2 = *pConst;
	ch2 = (isascii (ch2) && isupper (ch2)) ? tolower (ch2) : ch2;

	if (ch1 != ch2)
	    return (FALSE);
	}

    /* strings match */

    *ppString = pString;
    return (TRUE);
    }
/*******************************************************************************
*
* getNum - get a numeric value from a string
*/

LOCAL BOOL getNum (ppString, pNum)
    FAST char **ppString;
    int *pNum;

    {
    skipSpace (ppString);

    if (sscanf (*ppString, "%d", pNum) != 1)
	return (FALSE);

    /* skip over number */

    while (isdigit (**ppString))
	(*ppString)++;

    return (TRUE);
    }
/*******************************************************************************
*
* getAssign - get an assignment out of a string
*/

LOCAL BOOL getAssign (ppString, valName, pVal)
    FAST char **ppString;
    char *valName;
    char *pVal;

    {
    skipSpace (ppString);
    if (!getConst (ppString, valName))
	return (FALSE);

    skipSpace (ppString);
    if (!getConst (ppString, "="))
	return (FALSE);

    return (getWord (ppString, pVal, BOOT_FIELD_LEN, " \t"));
    }
/*******************************************************************************
*
* getAssignNum - get a numeric assignment out of a string
*/

LOCAL BOOL getAssignNum (ppString, valName, pVal)
    FAST char **ppString;
    char *valName;
    int *pVal;

    {
    char buf [BOOT_FIELD_LEN];
    char *pBuf = buf;
    char *pTempString;
    int tempVal;

    /* we don't modify *ppString or *pVal until we know we have a good scan */

    /* first pick up next field into buf */

    pTempString = *ppString;
    if (!getAssign (&pTempString, valName, buf))
	return (FALSE);

    /* now scan buf for a valid number */

    if ((scanNum (&pBuf, &tempVal, FALSE) != OK) || (*pBuf != EOS))
	return (FALSE);

    /* update string pointer and return value */

    *ppString = pTempString;
    *pVal = tempVal;

    return (TRUE);
    }

/*******************************************************************************
*
* printClear - print string with '?' for unprintable characters
*/

LOCAL VOID printClear (param)
    FAST char *param;

    {
    FAST char ch;

    while ((ch = *(param++)) != EOS)
	printf ("%c", (isascii (ch) && isprint (ch)) ? ch : '?');
    }
/*******************************************************************************
*
* printParamString - print string parameter
*/

LOCAL VOID printParamString (paramName, param)
    char *paramName;
    char *param;

    {
    if (param [0] != EOS)
	{
	printf ("%-*s: ", PARAM_PRINT_WIDTH, paramName);
	printClear (param);
	printf ("\n");
	}
    }
/*******************************************************************************
*
* promptParamString - prompt the user for a string parameter
*
* - carriage return leaves the parameter unmodified;
* - "." clears the parameter (null string).
*/

LOCAL VOID promptParamString (paramName, param)
    char *paramName;
    FAST char *param;

    {
    FAST int i;
    char buf [100];

    printf ("%-*s: ", PARAM_PRINT_WIDTH, paramName);
    printClear (param);
    if (*param != EOS)
	printf (" ");

    if ((i = promptRead (buf, sizeof (buf))) != 0)
	return (i);

    if (buf[0] == '.')
	{
	param [0] = EOS;	/* just '.'; make empty field */
	return (1);
	}

    strcpy (param, buf);	/* update parameter */
    return (1);
    }
/*******************************************************************************
*
* promptParamNum - prompt the user for a parameter
*
* carriage return leaves the parameter unmodified
*/

LOCAL VOID promptParamNum (paramName, pValue, hex)
    char *paramName;
    int *pValue;
    BOOL hex;

    {
    char buf [100];
    char *pBuf;
    int value;
    int i;

    printf (hex ? "%-*s: 0x%x " : "%-*s: %d ", PARAM_PRINT_WIDTH, paramName,
	    *pValue);

    if ((i = promptRead (buf, sizeof (buf))) != 0)
	return (i);

    
    /* scan for number */

    pBuf = buf;
    if ((scanNum (&pBuf, &value, FALSE) != OK) || (*pBuf != EOS))
	{
	printf ("invalid number.\n");
	return (-98);
	}

    *pValue = value;
    return (1);
    }
/*******************************************************************************
*
* promptRead - read the result of a prompt and check for special cases
*
* Special cases:
*    '-'  = previous field
*    '^D' = quit
*    CR   = leave field unchanged
*    too many characters = error
*
* RETURNS:
*   0 = OK
*   1 = skip this field
*  -1 = go to previous field
* -98 = ERROR
* -99 = quit
*/

LOCAL int promptRead (buf, bufLen)
    char *buf;
    int bufLen;

    {
    FAST int i;

    i = fioRdString (STD_IN, buf, bufLen);

    if (i == EOF)
	return (-99);			/* EOF; quit */

    if (i == 1)
	return (1);			/* just CR; leave field unchanged */
    
    if ((i == 2) && (buf[0] == '-'))
	return (-1);			/* '-'; go back up */

    if (i >= bufLen)
	{
	printf ("too big - maximum field width = %d.\n", bufLen);
	return (-98);
	}

    return (0);
    }
/*******************************************************************************
*
* skipSpace - advance pointer past white space
*
* Increments the string pointer passed as a parameter to the next
* non-white-space character in the string.
*/

LOCAL VOID skipSpace (strptr)
    FAST char **strptr;	/* pointer to pointer to string */

    {
    while (isspace (**strptr))
	++*strptr;
    }
/******************************************************************************
*
* scanNum - scan string for numeric value
*
* This routine scans the specified string for a numeric value.  The string 
* pointer is updated to reflect where the scan stopped, and the value is
* returned via pValue.  The value will be scanned by default in decimal unless
* hex==TRUE.  In either case, preceding the value with "0x" or "$" forces
* hex.  Spaces before and after number are skipped.
*
* RETURNS: OK if value scanned successfully, ERROR otherwise
*
* EXAMPLES:
*	scanNum (&string, &value, FALSE);
*    with:
*	string = "   0xf34  ";
*	    returns OK, string points to EOS, value=0xf34
*	string = "   0xf34r  ";
*	    returns OK, string points to 'r', value=0xf34
*	string = "   r0xf34  ";
*	    returns ERROR, string points to 'r', value unchanged
*
* NO_MANUAL
*/  

STATUS scanNum (ppString, pValue, hex)
    FAST char **ppString;
    int *pValue;
    FAST BOOL hex;

    {
    FAST char *pStr;
    FAST char ch;
    FAST int n;
    FAST int value = 0;

    skipSpace (ppString);

    /* pick base */

    if (**ppString == '$')
	{
	++*ppString;
	hex = TRUE;
	}
    else if (strncmp (*ppString, "0x", 2) == 0)
	{
	*ppString += 2;
	hex = TRUE;
	}


    /* scan string */

    pStr = *ppString;

    FOREVER
	{
	ch = *pStr;

	if (!isascii (ch))
	    break;

	if (isdigit (ch))
	    n = ch - '0';
	else
	    {
	    if (!hex)
		break;

	    if (isupper (ch))
		ch = tolower (ch);

	    if ((ch < 'a') || (ch > 'f'))
		break;

	    n = ch - 'a' + 10;
	    }

	value = (value * (hex ? 16 : 10)) + n;
	++pStr;
	}

    
    /* error if we didn't scan any characters */

    if (pStr == *ppString)
	return (ERROR);

    /* update users string ptr to point to character that stopped the scan */

    *ppString = pStr;
    skipSpace (ppString);

    *pValue = value;

    return (OK);
    }
