/* if_enp.h - header file for enp10 driver */ 

/*
modification history
--------------------
01c,13apr89,del  added ENPDEVICE structure to be compatible with CMC
		 Link-10 v4.0 roms.
01c,04aug88,dnw  added enp_status and associated values.
01b,19nov87,dnw  added e_intvector
01a,27jul87,rdc  adapted from cmc's 4.2 driver. 
*/


/* since the enp10 only supports 16 bit data accesses, use
   the following macros to access 32 bit values */

#define WRITELONG(addr, value) *(short *) addr = (unsigned) value >> 16; \
			      *((short *) addr + 1) = (unsigned) value & 0xffff;
#define READLONG(source, dest) *(short *) dest = *(short *)source; \
				*((short *) dest + 1) = *((short *) source + 1);

#define CMCOLD		1		/* values for enpverflag */
#define CMCNEW		2

#define ENPSTART	0xf02000	/* standard enp start addr */
#define K		*1024

#define ENP_BSS_SIZE	0x102

#define ENPSIZE		(124 K)		/* VME bus space allocated to enp */
#define MINPKTSIZE	60		/* minimum ethernet packet size */

/* Note: paged window (4 K) is identity mapped by ENP kernel to provide
 * 124 K contiguous RAM (as reflected in RAM_SIZE
 */

#define RAM_WINDOW	(128 K)
#define IOACCESS_WINDOW (4 K)
#define FIXED_WINDOW	(RAM_WINDOW - IOACCESS_WINDOW)
#define RAMROM_SWAP	(4 K)
#define RAM_SIZE	(FIXED_WINDOW - RAMROM_SWAP)

#define HOST_RAMSIZE	(48 K)
#define ENP_RAMSIZE	(20 K)

/* ...top of 4K local i/o space for ENP */

typedef struct iow10 {
	char	pad1[0x81];
/* written to: causes an interrupt on the host at the vector written
   read from : returns the most significant 8 bits of the slave address */
	char	vector;
	char	pad2[0x1F];
	char	csrarr[0x1E];
	char	pad3[2];
	char	ier;		/* intr. enable reg., 0x80 == enable,0 == off*/
	char	pad4[1];
	char	tir;		/* transmit intr. (Level 4 INP autovector) */
	char	pad5[1];
	char	rir;		/* receive intr. (Level 5 INP autovector) */
	char	pad6[1];
	char	uir;		/* utility intr. (Level 1 INP autovector) */
	char	pad7[7];
	char	mapfirst4k;	/* bit 7 set means ram, clear means rom */
	char	pad8[0x11];
	char	exr;		/* exception register, see bit defines above */
	char	pad9[0xD1F];
	char	hst2enp_interrupt;	/* R or W interrupts ENP */
	char	pad10[0xFF];
	char	hst2enp_reset;	/* R or W resets ENP */
} iow10;

typedef struct
    {
    u_char enaddr[6];
    } ETHADDR;

typedef
struct ethlist
{
	int	e_listsize;		/* active addr entries */
	ETHADDR	e_baseaddr;		/* addr lance is working with */
	ETHADDR e_addrs[ 16 ];		/* possible addresses */
} ETHLIST;

typedef
struct enpstat
{
	unsigned long e_xmit_successful;	/* Successful transmissions */
	unsigned long e_mult_retry;		/* multiple retries on xmit */
	unsigned long e_one_retry;		/* single retries */
	unsigned long e_fail_retry;		/* too many retries */
	unsigned long e_deferrals;		/* transmission delayed due
						   to active medium */
	unsigned long e_xmit_buff_err;		/* xmit data chaining failed --
						   "can't happen" */
	unsigned long e_silo_underrun;		/* transmit data fetch failed */
	unsigned long e_late_coll;		/* collision after xmit */
	unsigned long e_lost_carrier;
	unsigned long e_babble;			/* xmit length > 1518 */
	unsigned long e_no_heartbeat;		/* transceiver mismatch --
						   not an error		*/
	unsigned long e_xmit_mem_err;
	unsigned long e_rcv_successful;		/* good receptions */
	unsigned long e_rcv_missed;		/* no recv buff available */
	unsigned long e_crc_err;		/* checksum failed */
	unsigned long e_frame_err;		/* crc error AND
						   data length != 0 mod 8 */
	unsigned long e_rcv_buff_err;		/* rcv data chain failure --
						   "can't happen" */
	unsigned long e_silo_overrun;		/* receive data store failed */
	unsigned long e_rcv_mem_err;
} ENPSTAT;

typedef struct RING
{
	short	r_rdidx;
	short	r_wrtidx;
	short	r_size;
	short	r_pad;
	int	r_slot[1];
} RING;

typedef struct RING32
{
	short	r_rdidx;
	short	r_wrtidx;
	short	r_size;
	short	r_pad;			/* to make VAXen happy */
	int	r_slot[ 32 ];
} RING32;

/*
 * 	ENP Ram data layout
 *
 *	If you don't put it here - it isn't there
 *
 */
/* the two structures that follow are almost identical, the only difference
 * is that the second, ENPDEVICE, contians two extra RING32 fields that
 * some propeller head at CMC decided to add to the middle of the old ENPDEVICE
 * structure (ENPDEVICE_OLD) instead of at the end of the structure, thus
 * making the new structure incompatible with the old structure. (see if_enp
 * for details on how this problem is being handled.
 */
typedef struct {
	char	enp_ram_rom[4 K];
	union {
		char	all_ram[RAM_SIZE];
		struct {
			unsigned short	t_go;
			unsigned short	t_status;
			unsigned int	t_pstart;
		} t;
		struct {
			char	nram[RAM_SIZE - (HOST_RAMSIZE + ENP_RAMSIZE)];
			char	hram[HOST_RAMSIZE];
			char	kram[ENP_RAMSIZE];
		} u_ram;
		struct
		{
			char	pad7[ 0x100 ];	/* starts 0x1100 - 0x2000 */
			short	e_enpstate;		
			short	e_enpmode;		
			int	e_enpbase;		
			short	e_enprun;
			short	e_intvector;

			RING32	h_toenp;		
			RING32	h_hostfree;		
			RING32	e_tohost;		
			RING32 	e_enpfree;		

			ENPSTAT	e_stat;
			ETHLIST	e_netaddr;		
		} iface;
	} enp_u;
	iow10	enp_iow;
} ENPDEVICE_OLD;		/* cmc link-10 v3.x */

typedef struct 
    {
    char	enp_ram_rom[4 K];
    union 
	{
	char	all_ram[RAM_SIZE];
	struct 
	    {
	    USHORT	t_go;
	    USHORT	t_status;   
	    UINT	t_pstart;
	    } t;
	struct 
	    {
	    char	nram[RAM_SIZE - (HOST_RAMSIZE + ENP_RAMSIZE)];
	    char	hram[HOST_RAMSIZE];
	    char	kram[ENP_RAMSIZE];
	    } u_ram;
	struct 
	    {
	    char	pad7[ 0x100 ];	/* starts 0x1100 - 0x2000 */
	    short	e_enpstate;		
	    short	e_enpmode;		
	    UINT	e_enpbase;		
	    short	e_enprun;
	    short	e_intvector;

	    RING32	h_toenp;		
	    RING32	h_hostfree;		
	    RING32	e_tohost;		
	    RING32 	e_enpfree;		
	    RING32	e_rcvdma;
	    RING32	h_rcv_d;

	    ENPSTAT	e_stat;
	    ETHLIST	e_netaddr;		
	    } iface;
	} enp_u;
    struct iow10	enp_iow;
    } ENPDEVICE;			/* cmc link-10 v4.0 */ 

#define	enp_ram		enp_u.all_ram
#define	enp_nram	enp_u.u_ram.nram
#define	enp_hram	enp_u.u_ram.hram
#define	enp_kram	enp_u.u_ram.kram
#define	enp_go		enp_u.t.t_go
#define	enp_status	enp_u.t.t_status 
#define	enp_prog_start	enp_u.t.t_pstart
#define enp_state	enp_u.iface.e_enpstate
#define enp_mode	enp_u.iface.e_enpmode
#define enp_base	enp_u.iface.e_enpbase
#define enp_enprun	enp_u.iface.e_enprun
#define enp_intvector	enp_u.iface.e_intvector
#define enp_toenp	enp_u.iface.h_toenp
#define enp_hostfree	enp_u.iface.h_hostfree
#define enp_tohost	enp_u.iface.e_tohost
#define enp_enpfree	enp_u.iface.e_enpfree
#define enp_stat	enp_u.iface.e_stat
#define enp_addr	enp_u.iface.e_netaddr

#define ENPVAL		0xff	/* value to poke in enp_iow.hst2enp_interrupt */
#define RESETVAL	0x00	/* value to poke in enp_iow.enp2hst_clear_intr*/
#define GOVAL		0x8080	/* value to poke in enp_go */

#define INTR_ENP(addr)		addr->enp_iow.hst2enp_interrupt = ENPVAL

#define ACK_ENP_INTR(addr)

#define IS_ENP_INTR(addr)	( 1 )

#define RESET_ENP(addr)		addr->enp_iow.hst2enp_reset = 01

#define ENP_GO( addr,start )	{  \
				  int goval = 0x80800000; \
				  WRITELONG(&addr->enp_prog_start, start) \
				  WRITELONG(&addr->enp_go, goval) \
				}

#define SPL_ENP			spl4


/*
 * status bits - in t_status for 3.1 t_csr for 4.0
 */

#define STS_ONLINE	0x02	/* set when GO command received */
#define STS_READY	0x04	/* set when ready for GO command */
#define STS_NO_PROM	0x08	/* don't go to PROM address */
#define STS_INT_ENABLE	0x40	/* enable enp interrupts */

/*
 * state bits
 */

#define S_ENPRESET	01		/* enp is in reset state */
#define S_ENPRUN	02		/* enp is in run state */

/*
 * mode bits
 */

#define E_SWAP16		0x1		/* swap two octets within 16 */
#define E_SWAP32		0x2		/* swap 16s within 32 */
#define E_SWAPRD		0x4		/* swap on read */
#define E_SWAPWRT		0x8		/* swap on write */
#define E_DMA			0x10		/* enp does data moving */

#define E_EXAM_LIST		0x80000000	/* enp should examine addr list */

/*
 * 	Download ioctl definitions
 */

#define mkioctl(type,value) (0x20000000|('type'<<8)|value)

#define ENPIOGO		mkioctl( S,1 )		/* start the enp */
#define ENPIORESET	mkioctl( S,2 )		/* reset the enp */

/*
 * 	The ENP Data Buffer Structure
 */

typedef struct BCB
{
	struct BCB *b_link;
	short	 b_stat;
	short	 b_len;
	char	*b_addr;
	short	 b_msglen;
	short	 b_reserved;
}BCB;

