/* sysLib.c - Microbar system dependent library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01s,08jun88,gae  Revised code/documentation for uniformity across targets.
		 made sysMemTop actually size memory.
		 sysClkConnect now returns STATUS.
		 added sysClk{Disable,Enable}, sysClkRateSet() doesn't enable.
		 added sysAuxClk{Disable,Enable}, and MADE IT WORK!
		 fixed sysIntAck to return NULL.
		 added sysProcNumGet.
		 made sysTmrRateSet() local.
01i,30may88,dnw  changed to v4 names.
01h,27apr88,ecs  added sysExcMsg.
01r,30may88,dnw  changed to v4 names.
01q,29apr88,gae  fixed sysMemTop(); cleanup.
01p,27apr88,ecs  added sysExcMsg.
01o,20nov87,dnw  changed sysIntGen to take interrupt vec num instead of vec adrs
		 changed to use new defines in config.h
01n,19nov87,jcf  documentation.
01m,17nov87,jcf  changed sysToMonitor to take and pass an argument to bootrom
01l,26oct87,jcf  added sysMailboxConnect
01k,03jun87,dnw  added sysClkConnect(), sysIntGen(), sysMailboxEnable(),
		   sysBusToLocalAdrs().
		 added more sys... global variables.
		 added sysToMonitor().
		 changed sysLocalToBusAdrs() to take adrs space arg.
		 deleted sysMailBoxAdrs().
		 deleted sysGetProcNum().
01k,01apr87,jlf  documentation.
		 added sysIntEnable() and sysIntDisable().
		 changed call to tas() to vxTas().
01j,26mar87,rdc  added sysGetProcNum(), sysSetProcNum(), sysMailBoxAddr(),
		 sysBusTas().
01i,25mar87,jlf  documentation
01h,24feb87,jlf  added (null) sysAuxClk routines.
01g,14feb87,dnw  added sysSetLowAdrs(), sysLocalToBusAdrs(), sysIntAck()
01f,21dec86,dnw  changed to not get include files from default directories.
01e,18dec86,llk  added sysModel.
01d,13jun85,rdc  reset timer 0 in sysHwInit, to eliminate spurious interrupts.
01c,13mar85,jlf  reset timer 1 in sysHwInit, to eliminate spurious interrupts.
01b,28feb85,rdc  sysTmrSetRate now uses different clock rates for counters 1&2.
01a,02aug84,dnw  created from old ubarLib.c
*/

/*
DESCRIPTION
This library contains a set of routines to manipulate the primary functions
of the CPU board.  The goal is to provide a board-independant interface on
which VxWorks and application code can be built in a system-independant way.
Not every feature of every board is supported by this library; a particular
board may have various extensions to the capabilities described here.
Also not every board will support all the functions provided by this library.
And some boards provide some of the functions of this library with hardware
switches, jumpers, or PALs, instead of software controllable registers.

The funtions addressed here include:

    initialization functions:
	- initialize hardware to known state
	- identify the system

    memory/address space functions:
	- get on-board memory limit
	- map from local to bus and bus to local address spaces
	- enable/disable cache memory
	- set/get non-volatile RAM

    bus interrupt functions:
	- enable/disable bus interrupt levels
	- generate bus interrupts

    serial channel functions (see tyCoDrv):
	- enable/disable serial channel interrupts
	- set serial channel baud rates
	- get/put bytes from a serial channel

    clock/timer functions:
       - enable/disable timer interrupts
       - set timer periodic rate

    mailbox/location monitor functions:
       - enable mailbox/location monitor interrupts
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "ioLib.h"
#include "sysLib.h"
#include "config.h"
#include "dbc68k.h"

IMPORT char end;		/* end of system, created automatically by ld */

IMPORT VOID logMsg ();

/* globals */

int   sysBus      = BUS;		/* system bus type (VME_BUS, etc) */
int   sysCpu      = CPU;		/* system cpu type (MC680x0) */
char *sysBootLine = BOOT_LINE_ADRS;	/* address of boot line */
char *sysExcMsg   = EXC_MSG_ADRS;	/* catastrophic message area */
int   sysProcNum;			/* processor number of this cpu */
int   sysFlags;				/* boot flags */
char  sysBootHost[BOOT_FIELD_LEN];	/* name of host from which we booted */
char  sysBootFile[BOOT_FIELD_LEN];	/* name of file from which we booted */


/* locals */

LOCAL int sysClkTicksPerSecond	= 60;
LOCAL int auxClkTicksPerSecond	= 60;
LOCAL int sysClkRunning		= FALSE;
LOCAL int auxClkRunning		= FALSE;


/*******************************************************************************
*
* sysModel - return model name of the system CPU
*
* This routine returns a pointer to a string with the model name of
* this CPU.
*
* RETURNS: pointer to string "Microbar DBC68K2"
*/

char *sysModel ()

    {
    return ("Microbar DBC68K2");
    }
/*******************************************************************************
*
* sysHwInit - initialize hardware
*
* This routine initializes various features of the board.
* It is normally called from usrInit (2) in usrConfig (1).
* This routine must be called in supervisor mode (joke).
*
* The timers are initialized and turned off.
* The interrupt controller chip is set up.
*/

VOID sysHwInit ()

    {
    /* turn off timers 0 & 1 */

    *DBC_PIT_C = 0x70;
    *DBC_PIT_C = 0x30;

    /* initialize the 8259 interrupt controller:
     *    vector base is 0x100, i.e. user vector 0 */

    *DBC_PIC_0 = 0;	/* sync up */
    *DBC_PIC_0 = 0;
    *DBC_PIC_0 = 0;
    *DBC_PIC_0 = 0;

    *DBC_PIC_0 = 0x13;	/* ICW1 (needs ICW4, no ICW3) */
    *DBC_PIC_1 = 0x40;	/* ICW2 vector */
    *DBC_PIC_1 = 0x0f;	/* ICW4 */
    }
/*******************************************************************************
*
* sysMemTop - get top of memory address
*
* This routine returns the address of the first missing byte of memory.
* It starts probing at the end of bss, then tries to read a byte
* at every 4K boundary until it finds one that can't be read.
*
* This routine must be called in system mode, the first time only.  After
* that, it can be called in user mode as well.
*
* RETURNS: address of the first missing byte of memory
*/

char *sysMemTop ()

    {
    static char *memTop = NULL;	/* top of memory address */
    char bitBucket;

#define	PAGE_SIZE	0x1000

    if (memTop == NULL)
	{
	/* Look for the top of memory starting at the first even page
	 * boundary after _end. */

	memTop = (char *) ((((int) &end) + (PAGE_SIZE - 1)) &
			    (~ (PAGE_SIZE - 1)));

	for (; vxMemProbe (memTop, READ, 1, &bitBucket) == OK;
	     memTop += PAGE_SIZE)
	    ;

	/* memTop is now pointing to the first non-existent address */
	}

    return (memTop);
    }
/*******************************************************************************
*
* sysToMonitor - transfers to rom monitor
*
* This routine transfers control to the rom monitor.  It is usually called
* only by the routine reboot, which services control-x, and bus errors at
* interrupt level.  In special circumstances, however, the user may wish
* to introduce a new startType such that a special bootrom facility would be
* enabled.
*
* RETURNS: OK (if we ever continue from the rom monitor)
*
* INTERNAL
* Note that the restart address is at (ROM_BASE_ADRS + 8) bytes.
*/

STATUS sysToMonitor (startType)
    int startType;	/* parameter is passed to ROM to tell it how to boot *
			   The possible type are defined in h/sysLib.h */
    {
    (* ((FUNCPTR) (ROM_BASE_ADRS + 8))) (startType);

    return (OK);	/* in case we ever continue from rom monitor */
    }

/*******************************************************************************
*
* sysClkConnect - connect routine to system clock interrupt
*
* This routine connects the given function to the system clock interrupt.
* System clock interrupts are not enabled.
* It is normally called from usrRoot (2) in usrConfig (1) to connect
* usrClock (2) to the system clock interrupt.
*
* RETURN: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), usrClock (2)
*/

STATUS sysClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each system clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    return (intConnect (INUM_TO_IVEC (INT_VEC_CLOCK0), routine, arg));
    }
/*******************************************************************************
*
* sysClkDisable - turn off clock interrupts
*/

VOID sysClkDisable ()

    {
    /* this generates 1 unwanted/extra interrupt... hmmm */

    sysTmrRateSet (TMR0_CLOCK, 0, TMD_TERMINAL_COUNT);

    sysClkRunning = FALSE;
    }
/*******************************************************************************
*
* sysClkEnable - turn clock interrupts on
*/

VOID sysClkEnable ()

    {
    sysTmrRateSet (TMR0_CLOCK, sysClkTicksPerSecond, TMD_SQUARE_WAVE);
    sysClkRunning = TRUE;
    }
/*******************************************************************************
*
* sysClkRateGet - get rate of system clock
*
* RETURNS: number of ticks per second of the system clock
*/

int sysClkRateGet ()
    
    {
    return (sysClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysClkRateSet - set rate of system clock
*
* Sets the clock rate of the system clock, and enables the clock interrupt
* if necessary.
*/

VOID sysClkRateSet (ticksPerSecond)
    int ticksPerSecond;		/* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
	sysClkTicksPerSecond = ticksPerSecond;

    if (sysClkRunning)
	{
	sysClkDisable ();
	sysClkEnable ();
	}
    }

/*******************************************************************************
*
* sysAuxClkConnect - connect a routine to the aux clock interrupt
*
* This routine connects a user routine to the auxiliary clock interrupt.
* Auxiliary clock interrupts are not enabled.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), sysAuxClkDisconnect (2)
*/

STATUS sysAuxClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each auxiliary clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    /* connect the routine to the interrupt vector */

    return (intConnect (INUM_TO_IVEC (INT_VEC_CLOCK1), routine, arg));
    }
/*******************************************************************************
*
* sysAuxClkDisconnect - disconnect a routine from the aux clock interrupt
*
* This routine disables the auxiliary clock interrupt and disconnects
* the routine currently connected to the auxiliary clock interrupt.
*
* SEE ALSO: sysAuxClkConnect (2)
*/

VOID sysAuxClkDisconnect ()

    {
    /* disable the auxiliary clock interrupt */

    sysAuxClkDisable ();

    /* connect the dummy routine, just in case */

    sysAuxClkConnect (logMsg, (int) "auxiliary clock interrupt\n");
    }
/*******************************************************************************
*
* sysAuxClkDisable - turn off auxiliary clock interrupts
*/

VOID sysAuxClkDisable ()

    {
    /* this generates 1 unwanted/extra interrupt... hmmm */

    sysTmrRateSet (TMR1_CLOCK, 0, TMD_TERMINAL_COUNT);

    auxClkRunning = FALSE;
    }
/*******************************************************************************
*
* sysAuxClkEnable - turn auxiliary clock interrupts on
*/

VOID sysAuxClkEnable ()

    {
    sysTmrRateSet (TMR1_CLOCK, auxClkTicksPerSecond, TMD_SQUARE_WAVE);
    auxClkRunning = TRUE;
    }
/*******************************************************************************
*
* sysAuxClkRateGet - get rate of auxiliary clock
*
* This routine finds out the auxiliary clock speed.
*
* RETURNS: number of ticks per second of the auxiliary clock
*
* SEE ALSO: sysAuxClkRateSet (2)
*/

int sysAuxClkRateGet ()
    
    {
    return (auxClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysAuxClkRateSet - set rate of auxiliary clock
*
* This routine sets the clock rate of the auxiliary clock.
* Auxiliary clock interrupts are not enabled.
*
* SEE ALSO: sysAuxClkConnect (2), sysAuxClkRateGet (2)
*/

VOID sysAuxClkRateSet (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
	auxClkTicksPerSecond = ticksPerSecond;

    if (auxClkRunning)
	{
	sysAuxClkDisable ();
	sysAuxClkEnable ();
	}
    }

/*******************************************************************************
*
* sysLocalToBusAdrs - convert local address to bus address
*
* Given a local memory address, this routine returns the MULTIBUS address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK (always)
*
* SEE ALSO: sysBusToLocalAdrs (2)
*
* ARGSUSED
*/

STATUS sysLocalToBusAdrs (adrsSpace, localAdrs, pBusAdrs)
    int adrsSpace;	/* bus address space in which busAdrs resides */
    char *localAdrs;	/* local address to convert */
    char **pBusAdrs;	/* where to return bus address */

    {
    *pBusAdrs = localAdrs;
    return (OK);
    }
/*******************************************************************************
*
* sysBusToLocalAdrs - convert bus address to local address
*
* Given a MULTIBUS memory address, this routine returns the local address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK (always)
*
* SEE ALSO: sysLocalToBusAdrs (2)
*
* ARGSUSED
*/

char *sysBusToLocalAdrs (adrsSpace, busAdrs, pLocalAdrs)
    int adrsSpace;	/* bus address space in which busAdrs resides */
    char *busAdrs;	/* bus address to convert */
    char **pLocalAdrs;	/* where to return local address */

    {
    *pLocalAdrs = busAdrs;
    return (OK);
    }
/*******************************************************************************
*
* sysIntEnable - enable interrupt level
*
* This routine enables the specified interrupt level.
*
* NOTE UBAR:
* Interrupts are jumperable but not software controllable.
* This routine has no effect.
*
* RETURNS: OK
*
* ARGSUSED
*/

int sysIntEnable (intLevel)
    int intLevel;	/* interrupt level to enable */

    {
    return (OK);
    }
/*******************************************************************************
*
* sysIntDisable - disable interrupt level
*
* This routine disables the specified interrupt level.
*
* NOTE UBAR:
* Interrupt levels are jumperable but not software controllable.
* This routine has no effect.
*
* RETURNS: OK
*
* ARGSUSED
*/

int sysIntDisable (intLevel)
    int intLevel;	/* interrupt level to disable */

    {
    return (OK);
    }
/*******************************************************************************
*
* sysBusIntAck - acknowledge interrupt
*
* This routine acknowledges the specified interrupt.
*
* NOTE UBAR:
* If jumpered correctly, MULTIBUS interrupts will be acknowledged automatically.
* This routine has no effect.
* On some boards this routine returns the vector put on
* bus by interrupting device.
*
* RETURNS: NULL
*
* ARGSUSED
*/

int sysBusIntAck (intLevel)
    int intLevel;

    {
    return (NULL);
    }
/*******************************************************************************
*
* sysBusIntGen - generate interrupt
*
* This routine generates a MULTIBUS bus interrupt.
*
* NOTE UBAR:
* Unable to generate MULTIBUS interrupts.
* This routine has no effect.
*
* RETURNS: ERROR unable to generate interrupt
*
* ARGSUSED
*/

STATUS sysBusIntGen (level, vector)
    int level;
    int vector;

    {
    return (ERROR);
    }

/*******************************************************************************
*
* sysMailboxConnect - connect routine to the mailbox interrupt
*
* This routine connects the given function to the mailbox interrupt.
*
* RETURNS: ERROR, since the mailbox facility is not implemented
*
* SEE ALSO: intConnect (2)
*
* ARGSUSED
*/

STATUS sysMailboxConnect (routine, arg)
    FUNCPTR routine;
    int arg;

    {
    return (ERROR);
    }
/*******************************************************************************
*
* sysMailboxEnable - enable mailbox interrupt
*
* This routine enables mailbox interrupt.
*
* RETURNS: ERROR, since the mailbox facility is not implemented
*
* ARGSUSED
*/

STATUS sysMailboxEnable (mailboxAdrs)
    char *mailboxAdrs;

    {
    return (ERROR);
    }
/*******************************************************************************
*
* sysProcNumGet - set processor number
*
* This routine returns the processor number previously set with 
* sysProcNumSet (2).
*
* RETURNS: processor number
*/

int sysProcNumGet ()

    {
    return (sysProcNum);
    }
/*******************************************************************************
*
* sysProcNumSet - set processor number
*
* Sets the processor number of this CPU.  Should be unique on the
* backplane.
*/

VOID sysProcNumSet (procNum)
    int procNum;

    {
    sysProcNum = procNum;
    }
/*******************************************************************************
*
* sysBusTas - test and set across backplane
*
* This routine does a 680x0 test-and-set instruction across the backplane.
*
* RETURNS: TRUE (successful set), or FALSE (failure)
*/

int sysBusTas (addr)
    char *addr;

    {
    return (vxTas (addr));
    }

/* miscellaneous support routines */

/*******************************************************************************
*
* sysTmrSetRate - set rate of a hardware timer/counter
*
* This routine sets the specified timer to the specified rate.
*/

LOCAL VOID sysTmrRateSet (timerNum, ticksPerSecond, mode)
    int timerNum;	/* number of timer to be set (0-2) */
    int ticksPerSecond;	/* number of interrupts per second */
    int mode;		/* timer mode */

    {
    short count = 0;
    
    if (ticksPerSecond > 0)
	count = (timerNum == 0 ? TMR0_FREQ : TMR1_FREQ) / ticksPerSecond;

    *DBC_PIT_C = (timerNum << 6) | 0x30 | (mode << 1);

    *(DBC_PIT_0 + (timerNum << 1)) = LSB (count);
    *(DBC_PIT_0 + (timerNum << 1)) = MSB (count);
    }
