/* tyCoDrv.c - The SUN-3/e on-board USART tty handler */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01c,08jul88,gae  changed to use proper z8530.h definitions. Junked tyImrSet().
		   moved sysUsartInt stuff to here and called tyCoInt --
		   does have to multiplex auxiliary clock interrups.
01b,13nov87,ecs  documentation.
01a,03jun87,jcf  written, by modifying v01f of the 68020 version.
*/

/*
DESCRIPTION
This is the driver for the SUN-3/e on-board serial ports.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, tyCoDrv to
initialize the driver, and tyCoDevCreate to create devices.

TYCODRV
Before using the driver, it must be initialized by calling the routine:
.CS
    tyCoDrv ()
.CE
This routine should be called exactly once, before any reads, writes, or
tyCoDevCreates.  Normally, it is called from usrRoot.

CREATING TERMINAL DEVICES
Before a terminal can be used, it must be created.  This is done
with the tyCoDevCreate call.
Each port to be used should have exactly one device associated with it,
by calling this routine.

.CS
    STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
	char *name;	/* Name to use for this device *
	int channel;	/* Physical channel for this device, 0 or 1 *
	int rdBufSize;	/* Read buffer size, in bytes *
	int wrtBufSize;	/* Write buffer size, in bytes *
.CE

For instance, to create the device "/tyCo/0", with buffer sizes of 512 bytes,
the proper call would be:
.CS
   tyCoDevCreate ("/tyCo/0", 0, 512, 512);
.CE
IOCTL
This driver responds to all the same ioctl codes as a normal ty driver.
The DARTs have countdown clocks from a base frequency of 4.9152 MHz to
set the baud rates, so all baud rates from 50 to 38400 are selectable.
*/

#include "vxWorks.h"
#include "ioLib.h"
#include "tyLib.h"
#include "sun3e.h"
#include "config.h"

IMPORT FUNCPTR sysAuxClkRoutine;
IMPORT int sysAuxClkArg;

typedef struct			/* TY_CO_DEV */
    {
    TY_DEV tyDev;
    BOOL created;	/* true if this device has really been created */
    char *cp;		/* control port */
    char *dp;		/* data port */
    } TY_CO_DEV;

LOCAL TY_CO_DEV tyCoDv [N_USART_CHANNELS] =	/* device descriptors */
    {
    {{{{NULL}}}, FALSE, SUN_SCCA, SUN_SCDA},
    {{{{NULL}}}, FALSE, SUN_SCCB, SUN_SCDB},
    {{{{NULL}}}, FALSE, SUN_KYBC, SUN_KYBD},
    };

LOCAL int tyCoDrvNum;		/* driver number assigned to this driver */

/* forward declarations */

LOCAL int tyCoOpen ();
LOCAL int tyCoRead ();
LOCAL int tyCoWrite ();
LOCAL STATUS tyCoIoctl ();
LOCAL VOID tyCoInt ();
LOCAL VOID tyCoStartup ();


/*******************************************************************************
*
* tyCoDrv - ty driver initialization routine
*
* This routine initializes the serial driver, sets up interrupt vectors,
* and performs hardware initialization of the serial ports.
*
* This routine must be called in supervisor state, since it does
* physical I/O directly.
*/

STATUS tyCoDrv ()

    {
    /* check if driver already installed */

    if (tyCoDrvNum > 0)
	return (OK);

    tyCoHrdInit ();

    (void) intConnect (INUM_TO_IVEC (INT_VEC_DUART), tyCoInt, NULL);

    tyCoDrvNum = iosDrvInstall (tyCoOpen, (FUNCPTR) NULL, tyCoOpen,
				(FUNCPTR) NULL, tyCoRead, tyCoWrite, tyCoIoctl);

    return (tyCoDrvNum == ERROR ? ERROR : OK);
    }
/******************************************************************
*
* tyCoDevCreate - create a device for the onboard ports
*
* This routine creates a device on one of the serial ports.  Each port
* to be used should have exactly one device associated with it, by calling
* this routine.
*/

STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
    char *name;		/* Name to use for this device */
    int channel;	/* Physical channel for this device (0 or 1) */
    int rdBufSize;	/* Read buffer size, in bytes */
    int wrtBufSize;	/* Write buffer size, in bytes */

    {
    if (tyCoDrvNum <= 0)
	{
	errnoSet (S_ioLib_NO_DRIVER);
	return (ERROR);
	}

    /* if this device already exists, don't create it */

    if (tyCoDv [channel].created)
	return (ERROR);

    /* initialize the ty descriptor, and turn on the bit for this
     * receiver and transmitter in the interrupt mask */

    if (tyDevInit ((TY_DEV_ID) &tyCoDv [channel], 
		    rdBufSize, wrtBufSize, tyCoStartup) != OK)
	{
	return (ERROR);
	}

    /* Mark the device as having been created, and add the device to
     * the I/O system */

    tyCoDv [channel].created = TRUE;
    return (iosDevAdd ((DEV_HDR *) &tyCoDv [channel], name, tyCoDrvNum));
    }
/**********************************************************************
*
* tyCoHrdInit - initialize the usarts.
*
* This routine initializes the on-board usarts for the VxWorks environment. 
*
* This routine must be called in supervisor mode, since it accesses I/O
* space. (This depends on how sun set up there page lookup table!)
*
*/

LOCAL VOID tyCoHrdInit ()

    {
    int ix;
    int delay;		/* delay for reset */
    char *cp;
    int oldlevel;	/* current interrupt level mask */
    char zero = 0;

    oldlevel = intLock ();	/* disable interrupts during init */

    for (ix = 0; ix < 2; ix++)	/* should go to N_USARTS */
	{
	cp = tyCoDv [ix].cp;

	*cp = SCC_WR0_SEL_WR9;		/* master interupt control */
	*cp = ((ix % 2) == 0) ?
	    SCC_WR9_CH_A_RST :		/* reset channel A */
	    SCC_WR9_CH_B_RST;		/* reset channel B */

	for (delay = 0; delay < 5000; delay++);	/* Wait for reset */

	*cp = SCC_WR0_SEL_WR4;		/* misc parms and modes */
	*cp = SCC_WR4_1_STOP | SCC_WR4_32_CLOCK;

	*cp = SCC_WR0_SEL_WR3;		/* rx parms */
	*cp = SCC_WR3_RX_8_BITS;	/* 8 bit word */

	*cp = SCC_WR0_SEL_WR5;		/* tx parms */
	*cp = SCC_WR5_TX_8_BITS;	/* 8 bit word */

	*cp = SCC_WR0_SEL_WR6;
	*cp = zero;			/* clear sync data */

	*cp = SCC_WR0_SEL_WR7;
	*cp = zero;			/* transmit */

	*cp = SCC_WR0_SEL_WR9;		/* master interupt control */
	*cp = SCC_WR9_MIE;		/* enable interrups */

	*cp = SCC_WR0_SEL_WR10;
	*cp = zero;				/* clear sync, loop, poll */

	*cp = SCC_WR0_SEL_WR11;
	*cp = SCC_WR11_RX_BR_GEN | SCC_WR11_TX_BR_GEN | SCC_WR11_TRXC_OI |
	      SCC_WR11_OUT_BR_GEN;

	*cp = SCC_WR0_SEL_WR12;		/* LSB of baud constant */
	*cp = 6;			/* 0x0006 = 9600 baud */
	*cp = SCC_WR0_SEL_WR13;		/* MSB of baud constant */
	*cp = zero;

	*cp = SCC_WR0_SEL_WR14;
	*cp = SCC_WR14_BR_SRC;

	/* enable the receivers and transmitters on both channels */

	*cp = SCC_WR0_SEL_WR3;		/* rx parms */
	*cp = SCC_WR3_RX_8_BITS | SCC_WR3_RX_EN;	/* Turn RX on */

	*cp = SCC_WR0_SEL_WR5;		/* tx parms */
	*cp = SCC_WR5_TX_8_BITS | SCC_WR5_TX_EN;	/* Turn TX on */
	
	*cp = SCC_WR0_RST_TX_CRC;		/* reset tx crc */

	*cp = SCC_WR0_SEL_WR14;			/* misc control bits */
	*cp = SCC_WR14_BR_SRC | SCC_WR14_BR_EN;

	*cp = SCC_WR0_SEL_WR15;
	*cp = zero;				/* No external ints */

	*cp = SCC_WR0_RST_INT;		/* reset extenal interrupts */
	*cp = SCC_WR0_RST_INT;		/* twice */

	*cp = SCC_WR0_SEL_WR1;		/* enable rx and tx */
        *cp = SCC_WR1_INT_ALL_RX | SCC_WR1_TX_INT_EN;
	} 

    /* enable system interupt mask reg for UART interupts */

    *SUN_IR = *SUN_IR | IR_SCC;

    /* all interrupts are masked out: the receiver interrupt will be
     * enabled in tyCoDevCreate
     */

    intUnlock (oldlevel);
    } 

/* routines provided to I/O system */

/*******************************************************************************
*
* tyCoOpen - open file to SUN 3/e usart
*
* ARGSUSED
*/

LOCAL int tyCoOpen (pTyCoDv, name, mode)
    TY_CO_DEV *pTyCoDv;
    char *name;
    int mode;

    {
    return ((int) pTyCoDv);
    }
/*******************************************************************************
*
* tyCoRead - task level read routine for SUN 3/e usart
*
* This routine fields all read calls to the SUN 3/e usart.
*/

LOCAL int tyCoRead (pTyCoDv, buffer, maxbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int maxbytes;

    {
    return (tyRead ((TY_DEV_ID) pTyCoDv, buffer, maxbytes));
    }
/*******************************************************************************
*
* tyCoWrite - task level write routine for SUN 3/e usart
*
* This routine fields all write calls to the SUN 3/e usart.
*/

LOCAL int tyCoWrite (pTyCoDv, buffer, nbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int nbytes;

    {
    return (tyWrite ((TY_DEV_ID) pTyCoDv, buffer, nbytes));
    }
/*******************************************************************************
*
* tyCoIoctl - special device control
*
* This routine handles baud rate requests, and passes all other requests
* to tyIoctl.
*/

LOCAL STATUS tyCoIoctl (pTyCoDv, request, arg)
    TY_CO_DEV *pTyCoDv;	/* device to control */
    int request;	/* request code */
    int arg;		/* some argument */

    {
    STATUS status;

#define	MAX_BAUD	38400

    switch (request)
	{
	case FIOBAUDRATE:

	    if (arg <= MAX_BAUD && arg > 50)
		{
		/* Assumes that clock prescaler is set to x32 */

		*pTyCoDv->cp = SCC_WR0_SEL_WR12; /* LSB of baud constant */
		*pTyCoDv->cp = LSB((MAX_BAUD*2/arg) - 2);
		*pTyCoDv->cp = SCC_WR0_SEL_WR13; /* MSB of baud constant */
		*pTyCoDv->cp = MSB((MAX_BAUD*2/arg) - 2);

		status = OK;
		}
	    else
		status = ERROR;

	    break;

	default:
	    status = tyIoctl ((TY_DEV_ID) pTyCoDv, request, arg);
	    break;
	}

    return (status);
    }

/*******************************************************************************
*
* tyCoInt - interrupt level processing for SUN3E usart
*
* This routine handles an interrupt from the on-board usart.
* There are two SCC's each with two channels.  One channel is
* dedicated to the auxillary clock.  All of them interrupt at level 6 and
* are autovectored to here.  This routine demultiplexes the interrupts.
* There could be multiple interrupts pending so as long as we are here we
* field them all.
*/

LOCAL VOID tyCoInt ()

    {
    static char *channelCont [4] = {SUN_SCCA, SUN_SCCB, SUN_KYBC, SUN_MSC};
    FAST int ix;		/* channel index */
    UINT ip;			/* interrupts pending */

    for (ix = 0; ix < 4 ; ix += 2)
	{
	for (*channelCont [ix] = SCC_WR0_SEL_WR3;
	    (ip = *channelCont [ix]) != NULL;
	    *channelCont [ix] = SCC_WR0_SEL_WR3)
	    {
	    if (ip & SCC_RR3_A_RX_IP)
		rxInt (ix);
	    if (ip & SCC_RR3_B_RX_IP)
		rxInt (ix + 1);
	    if (ip & SCC_RR3_A_TX_IP)
		txInt (ix);
	    if (ip & SCC_RR3_B_TX_IP)
		txInt (ix + 1);
	    if (ip & SCC_RR3_A_EXT_IP)
		;
	    if (ip & SCC_RR3_B_EXT_IP)
		{
		if (sysAuxClkRoutine != NULL)
		    sysAuxClkRoutine (sysAuxClkArg);

		*channelCont [ix + 1] = SCC_WR0_RST_INT;
		}
	    }

	*channelCont [ix]     = SCC_WR0_RST_HI_IUS; /* reset the interrupt */
	*channelCont [ix + 1] = SCC_WR0_RST_HI_IUS; /* reset the interrupt */
	}
    }
/*******************************************************************************
*
* rxInt - handle a receiver interrupt
*
* This routine gets called by tyCoInt to handle a receiver interrupt.
*/

LOCAL VOID rxInt (channel)
    FAST int channel;

    {
    (void)tyIRd ((TY_DEV_ID) &tyCoDv [channel], *tyCoDv [channel].dp);
    }
/*******************************************************************************
*
* txInt - handle a transmitter interrupt
*
* This routine gets called by tyCoInt to handle a xmitter interrupt.
* If there is another character to be transmitted, it sends it.  If
* not, or if a device has never been created for this channel, we just
* disable the interrupt.
*/

LOCAL VOID txInt (channel)
    FAST int channel;

    {
    char outChar;

    if ((tyCoDv [channel].created) &&
	(tyITx ((TY_DEV_ID) &tyCoDv [channel], &outChar) == OK))
	{
	*(tyCoDv [channel].dp) = outChar;
	}
    else
	{
	/* send EOI to transmitter */

	*(tyCoDv [channel].cp) = SCC_WR0_RST_TX_INT;
	}
    }
/*******************************************************************************
*
* tyCoStartup - transmitter startup routine
*
* Call interrupt level character output routine for SUN 3/e usart.
*/

LOCAL VOID tyCoStartup (pTyCoDv)
    TY_CO_DEV *pTyCoDv;		/* ty device to start up */

    {
    char outChar;

    if ((pTyCoDv->created) &&
	(tyITx ((TY_DEV_ID) pTyCoDv, &outChar) == OK))
	{
        *(pTyCoDv->dp) = outChar;
	}
    }
