/* romInit.s - SUN 3/E initialization module for code in ROM */

/*
modification history
--------------------
01b,20dec88,gae  cleared first megabyte of memory.
01a,01aug88,jcf  written based on mv133 version.
	   +gae
*/

/*
DESCRIPTION
This module is the first code executed in VxWorks boot ROMs.

As the SUN 3/E has only 64k of EPROM space and VxWorks bootrom code
requires nearly 128k VxWorks bootroms must be located on a
separate ROM board.  In addition, the "magic" of the
SUN ROM is undocumented.  Therefore, one must rely on
the SUN ROM to set up the DVMA / Memory Management.

INTERNAL
The following are equal because there is no initial 8 bytes for sp + pc.
    ROM_BASE_ADRS  0x00300000 -- config.h
    ROM_START_ADRS 0x00300000 -- MakeSkel
*/

#define ASMLANGUAGE
#include "vxWorks.h"
#include "sysLib.h"
#include "config.h"

	/* internals */

	.globl	_romInit	/* start of system code */

	/* externals */

	.globl	_usrInit	/* system initialization routine */


	.text
	.even

/*******************************************************************************
*
* romInit - cold or warm boot
*/

_romInit:
	movew	#0x3700,sr	/* disable interrupts, turn on M bit */
	bra	romInit1	/* initial start in ROM */

	movew	#0x3700,sr	/* disable interrupts, turn on M bit */
	bra	romInit2	/* subsequent starts in ROM */


/*******************************************************************************
*
* romInit1 - cold boot
*
* Enable cache, clear all memory, copy ROM to RAM, set stack, go to usrInit
* This routine is position independent.

* romInit1 ()

*/

romInit1:
	clrb	0x0fe0a000		/* turn off clock */
	moveb	#0x0c,0x0fe06011
	clrb	0x0fe0a000
	moveb	#0x1c,0x0fe06011
	orb	#0x01,0x0fe0a000

	/* clear 1 megabyte  of memory;
         * outer loop counter in d2 = 0x3 =
         *
         *   1M byte to clear
         *   ------------------  - 1
         *   64K * 4               ^
         *    ^    ^               |
         *    |    |               for dbra at end of outer loop
         *    |    bytes per inner loop
         *    inner loops per outer loop
         */
 
        movel   #0x3,d2                /* outer loop counter */
        movel   #LOCAL_MEM_LOCAL_ADRS,a0 /* get start of local memory */
        clrl    d0                      /* value to fill */
 
 
fb1:    movel   #0xffff,d1              /* set to fill 64K */
fb2:    movel   d0,a0@+                 /* move 4 bytes */
        dbra    d1,fb2                  /* inner loop test */
        dbra    d2,fb1                  /* outer loop test */
 
 
	/* copy system from ROM to RAM */

	movel	#_romInit,a0		/* where system should be in RAM */
	movel	#ROM_BASE_ADRS,a1	/* where system is in ROM */
	movel	#0x7ffd,d1		/* 32K*4 = 128K bytes (no sp+pc)*/
loop1:
	movel	a1@+,a0@+		/* copy 4 bytes of code */
	dbf	d1,loop1


	/* set stack to grow down from start of code and call system init */

	movel	#_romInit,a7		/* set stack to start of code */
	movel	#BOOT_COLD,a7@-		/* push arg = start type */
	jsr	_usrInit		/* never returns - starts up kernel */

/*******************************************************************************
*
* romInit2 - warm boot
*
* Same as romInit but doesn't clear memory.
* This routine is position independent.

* romInit2 ()

*/

romInit2:
	link	a6,#0		/* no unlk necessary; we change stacks */

	/* Enable the cache by writing to the cache control register.
	 * Since the assemblers differ on the syntax for this one
	 * (if they even have it), it's done with two .word's. */

	movel	#0x09,d0	/* clear and enable cache */
	.word	0x4e7b,0x0002	/* movec d0,cacr */

	/* copy system from ROM to RAM */

	movel	#_romInit,a0		/* where system should be in RAM */
	movel	#ROM_BASE_ADRS,a1	/* where system is in ROM */
	movel	#0x7ffd,d1		/* 32K*4 = 128K bytes (no sp+pc) */
loop2:
	movel	a1@+,a0@+		/* copy 4 bytes of code */
	dbf	d1,loop2


	/* set stack to grow down from start of code and call system init */

	movel	#_romInit,a7		/* set stack to start of code */
	movel	a6@(0x08),a7@-		/* push arg = start type */
	jsr	_usrInit		/* never returns - starts up kernel */

