/* tyCoDrv.c - The Mizar mz7122 and mz7124 DUSCC tty handler */

static char *copyright = "Copyright 1988, Mizar Digital Systems, Inc.";

/*
modification history
--------------------
01c,26jul88,gae  updated to 4.0.
01b,22jun88,mcl  cleanup and release.
01a,23may88,miz  written by modifying mz7120/tyCoDrv.c.
*/

/*
DESCRIPTION
This is the driver for the mz7122 and mz7124 DUSCC chip.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, tyCoDrv to
initialize the driver, and tyCoDevCreate to create devices.

TYCODRV
Before using the driver, it must be initialized by calling the routine:
.CS
    STATUS tyCoDrv ()
.CE
The status returned is OK.

This routine should be called exactly once, before any reads, writes, or
tyCoDevCreates.  Normally, it is called from usrRoot.

CREATING TERMINAL DEVICES
Before a terminal can be used, it must be created.  This is done
with the tyCoDevCreate call.
Each port to be used should have exactly one device associated with it,
by calling this routine.

.CS
    STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
	char *name;	/* Name to use for this device
	int channel;	/* Physical channel for this device, always 0
	int rdBufSize;	/* Read buffer size, in bytes
	int wrtBufSize;	/* Write buffer size, in bytes
.CE

For instance, to create the device "/tyCo/0", with buffer sizes of 512 bytes,
the proper call would be:
.CS
   tyCoDevCreate ("/tyCo/0", 0, 512, 512);
.CE
IOCTL
This driver responds to all the same ioctl codes as a normal ty driver.

*/

#include "vxWorks.h"
#include "iv68k.h"
#include "ioLib.h"
#include "iosLib.h"
#include "memLib.h"
#include "tyLib.h"
#include "config.h"

IMPORT int safeLoc;


typedef struct			/* BAUD */
    {
    int rate;		/* a baud rate */
    char xmtVal;	/* rate to write to the Transmitter Timing Register */
    char rcvVal;	/* rate to write to the Reciever Timing Register */
    } BAUD;

typedef struct			/* TY_CO_DEV */
    {
    TY_DEV tyDev;
    BOOL created;	/* true if this device has really been created */
    unsigned char *rdr;	/* receiver data port */
    unsigned char *xdr;	/* transmitter data port */
    unsigned char *ttr;	/* transmitter timer register */
    unsigned char *rtr;	/* receiver timer register */
    unsigned char *ier;	/* interrupt enable register */
    unsigned char rrdy; /* receiver ready bit */
    unsigned char xrdy; /* transmitter ready bit */
    } TY_CO_DEV;

LOCAL TY_CO_DEV tyCoDv [N_CHANNELS] =	/* device descriptors */
    {
    {{{{NULL}}}, FALSE,
	M562_RXFIFOA, M562_TXFIFOA, M562_TTRA, M562_RTRA, M562_IERA,
	M562_GSR_A_RXRDY, M562_GSR_A_TXRDY },
    {{{{NULL}}}, FALSE,
	M562_RXFIFOB, M562_TXFIFOB, M562_TTRB, M562_RTRB, M562_IERB,
	M562_GSR_B_RXRDY, M562_GSR_B_TXRDY },
    };

/* baudTable is a table of the available baud rates, and the values to write
   to the csr reg to get those rates */

LOCAL BAUD baudTable [] =
    {
    {50,
	M562_TTR_CLK_BRG | M562_TTR_BAUD_50,
	M562_RTR_CLK_BRG | M562_RTR_BAUD_50 },
    {75,
	M562_TTR_CLK_BRG | M562_TTR_BAUD_75,
	M562_RTR_CLK_BRG | M562_RTR_BAUD_75 },
    {110,
	M562_TTR_CLK_BRG | M562_TTR_BAUD_110,
	M562_RTR_CLK_BRG | M562_RTR_BAUD_110},
    {134,
	M562_TTR_CLK_BRG | M562_TTR_BAUD_134_5,
	M562_RTR_CLK_BRG | M562_RTR_BAUD_134_5},
    {150,
	M562_TTR_CLK_BRG | M562_TTR_BAUD_150,
	M562_RTR_CLK_BRG | M562_RTR_BAUD_150},
    {200,
	M562_TTR_CLK_BRG | M562_TTR_BAUD_200,
	M562_RTR_CLK_BRG | M562_RTR_BAUD_200},
    {300,
	M562_TTR_CLK_BRG | M562_TTR_BAUD_300,
	M562_RTR_CLK_BRG | M562_RTR_BAUD_300},
    {600,
	M562_TTR_CLK_BRG | M562_TTR_BAUD_600,
	M562_RTR_CLK_BRG | M562_RTR_BAUD_600},
    {1050,
	M562_TTR_CLK_BRG | M562_TTR_BAUD_1050,
	M562_RTR_CLK_BRG | M562_RTR_BAUD_1050},
    {1200,
	M562_TTR_CLK_BRG | M562_TTR_BAUD_1200,
	M562_RTR_CLK_BRG | M562_RTR_BAUD_1200},
    {2000,
	M562_TTR_CLK_BRG | M562_TTR_BAUD_2000,
	M562_RTR_CLK_BRG | M562_RTR_BAUD_2000},
    {2400,
	M562_TTR_CLK_BRG | M562_TTR_BAUD_2400,
	M562_RTR_CLK_BRG | M562_RTR_BAUD_2400},
    {4800,
	M562_TTR_CLK_BRG | M562_TTR_BAUD_4800,
	M562_RTR_CLK_BRG | M562_RTR_BAUD_4800},
    {9600,
	M562_TTR_CLK_BRG | M562_TTR_BAUD_9600,
	M562_RTR_CLK_BRG | M562_RTR_BAUD_9600},
    {19200,
	M562_TTR_CLK_BRG | M562_TTR_BAUD_19200,
	M562_RTR_CLK_BRG | M562_RTR_BAUD_19200},
    {38400,
	M562_TTR_CLK_BRG | M562_TTR_BAUD_38400,
	M562_RTR_CLK_BRG | M562_RTR_BAUD_38400} };

LOCAL int tyCoDrvNum;		/* driver number assigned to this driver */

/* forward declarations */

LOCAL VOID tyCoStartup ();
LOCAL int tyCoOpen ();
LOCAL int tyCoRead ();
LOCAL int tyCoWrite ();
LOCAL STATUS tyCoIoctl ();
LOCAL VOID tyCoInt ();
LOCAL VOID txInt ();
LOCAL VOID rxInt ();

/*******************************************************************************
*
* tyCoDrv - ty driver initialization routine
*
* This routine initializes the driver, sets up interrupt vectors,
* and performs hardware initialization of the ports.
*
* This routine must be called in supervisor state, since it does
* physical I/O directly.
*/

STATUS tyCoDrv ()

    {
    /* check if driver already installed */
     
    if (tyCoDrvNum > 0)
	return (OK);
		    
    /* The DUSCC interrupts on autovector 6. */

    sysM562Connect (rxInt, txInt);

    tyCoHrdInit ();

    tyCoDrvNum = iosDrvInstall (tyCoOpen, (FUNCPTR) NULL, tyCoOpen,
				(FUNCPTR) NULL, tyCoRead, tyCoWrite, tyCoIoctl);

    return (tyCoDrvNum == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* tyCoDevCreate - create a device for the onboard ports
*
* This routine creates a device on one of the ports.  Each port
* to be used should have exactly one device associated with it, by calling
* this routine.
*/

STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
    char *name;		/* Name to use for this device */
    int channel;	/* Physical channel for this device (0 or 1) */
    int rdBufSize;	/* Read buffer size, in bytes */
    int wrtBufSize;	/* Write buffer size, in bytes */

    {
    if (tyCoDrvNum <= 0)
	{
	errnoSet (S_ioLib_NO_DRIVER);
	return (ERROR);
	}
				     
    /* if this device already exists, don't create it */

    if (tyCoDv [channel].created)
	return (ERROR);

    /* initialize the ty descriptor, and turn on the bit for this
     * receiver in the interrupt mask */

    if (tyDevInit ((TY_DEV_ID) &tyCoDv [channel], 
		    rdBufSize, wrtBufSize, tyCoStartup) != OK)

	return (ERROR);

    *(tyCoDv [channel].ier) = M562_IER_RXRDY;
    safeLoc++;

    /* Mark the device as having been created, and add the device to
     * the io sys */

    tyCoDv [channel].created = TRUE;
    safeLoc++;
    return (iosDevAdd ((DEV_HDR *) &tyCoDv [channel], name, tyCoDrvNum));
    }
/*******************************************************************************
*
* tyCoHrdInit - initialize the DUSCC
*
* This routine initializes the DUSCC for the VxWorks environment. 
*
* This routine must be called in supervisor mode, since it accesses I/O space.
*/

LOCAL VOID tyCoHrdInit ()

    {
    int oldlevel;		/* current interrupt level mask */

    oldlevel = intLock ();

    /* 8 data bits, 1 stop bit, no parity, set for 9600 baud */

    *M562_CMR1A	= M562_CMR1_ASYNC;
    safeLoc++;
    *M562_CMR2A	= M562_CMR2_DTI_POLL_OR_INT;
    safeLoc++;
    *M562_TPRA	= M562_TPR_ASYNC_1 | M562_TPR_8BITS;
    safeLoc++;
    *M562_TTRA	= M562_TTR_CLK_BRG | M562_TTR_BAUD_9600;
    safeLoc++;
    *M562_RPRA	= M562_RPR_8BITS;
    safeLoc++;
    *M562_RTRA	= M562_RTR_CLK_BRG | M562_RTR_BAUD_9600;
    safeLoc++;
    *M562_OMRA	= M562_OMR_TXRDY_FIFO_EMPTY | M562_OMR_TX_RES_CHAR_LENGTH_TPR;
    safeLoc++;
    *M562_CCRA	= M562_CCR_TX_ENABLE_TX;
    safeLoc++;
    *M562_CCRA	= M562_CCR_RX_ENABLE_RX;
    safeLoc++;

    *M562_CMR1B	= M562_CMR1_ASYNC;
    safeLoc++;
    *M562_CMR2B	= M562_CMR2_DTI_POLL_OR_INT;
    safeLoc++;
    *M562_TPRB	= M562_TPR_ASYNC_1 | M562_TPR_8BITS;
    safeLoc++;
    *M562_TTRB	= M562_TTR_CLK_BRG | M562_TTR_BAUD_9600;
    safeLoc++;
    *M562_RPRB	= M562_RPR_8BITS;
    safeLoc++;
    *M562_RTRB	= M562_RTR_CLK_BRG | M562_RTR_BAUD_9600;
    safeLoc++;
    *M562_OMRB	= M562_OMR_TXRDY_FIFO_EMPTY | M562_OMR_TX_RES_CHAR_LENGTH_TPR;
    safeLoc++;
    *M562_CCRA	= M562_CCR_TX_ENABLE_TX;
    safeLoc++;
    *M562_CCRA	= M562_CCR_RX_ENABLE_RX;
    safeLoc++;

    /* all interrupts are masked out: the receiver interrupt will be enabled
       in the tyCoDevCreate */

    intUnlock (oldlevel);
    } 
/*******************************************************************************
*
* tyCoOpen - open file to DUSCC
*
* ARGSUSED
*/

LOCAL int tyCoOpen (pTyCoDv, name, mode)
    TY_CO_DEV *pTyCoDv;
    char *name;
    int mode;

    {
    return ((int) pTyCoDv);
    }
/*******************************************************************************
*
* tyCoRead - task level read routine for DUSCC
*
* This routine fields all read calls to the DUSCC.
*/

LOCAL int tyCoRead (pTyCoDv, buffer, maxbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int maxbytes;

    {
    return (tyRead ((TY_DEV_ID) pTyCoDv, buffer, maxbytes));
    }
/*******************************************************************************
*
* tyCoWrite - task level write routine for DUSCC
*
* This routine fields all write calls to the DUSCC.
*/

LOCAL int tyCoWrite (pTyCoDv, buffer, nbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int nbytes;

    {
    return (tyWrite ((TY_DEV_ID) pTyCoDv, buffer, nbytes));
    }
/*******************************************************************************
*
* tyCoIoctl - special device control
*
* This routine handles baud rate requests, and passes all other requests
* to tyIoctl.
*/

LOCAL STATUS tyCoIoctl (pTyCoDv, request, arg)
    TY_CO_DEV *pTyCoDv;	/* device to control */
    int request;	/* request code */
    int arg;		/* some argument */

    {
    int i;
    STATUS status;

    switch (request)
	{
	case FIOBAUDRATE:
	    status = ERROR;
	    for (i = 0; i < NELEMENTS (baudTable); i++)
		{
		if (baudTable [i].rate == arg)
		    {
		    *pTyCoDv->ttr = baudTable [i].xmtVal;
		    safeLoc++;
		    *pTyCoDv->rtr = baudTable [i].rcvVal;
		    status = OK;
		    break;
		    }
		}
	    break;

	default:
	    status = tyIoctl ((TY_DEV_ID) pTyCoDv, request, arg);
	    break;
	}
    return (status);
    }

/*******************************************************************************
*
* rxInt - handle a receiver interrupt
*
* This routine gets called by sysM562Int to handle a receiver interrupt.
*/

LOCAL VOID rxInt (channel)
    FAST int channel;

    {
    tyIRd ((TY_DEV_ID) &tyCoDv [channel], (char) *tyCoDv [channel].rdr);
    safeLoc++;
    *M562_GSR = tyCoDv [channel].rrdy;
    }
/*******************************************************************************
*
* txInt - handle a transmitter interrupt
*
* This routine gets called by sysM562Int to handle a xmitter interrupt.
* If there is another character to be transmitted, it sends it.  If
* not, or if a device has never been created for this channel, we just
* disable the interrupt.
*/

LOCAL VOID txInt (channel)
    FAST int channel;

    {
    char outChar;

    if ((tyCoDv [channel].created) &&
	(tyITx ((TY_DEV_ID) &tyCoDv [channel], &outChar) == OK))
	{
	*(tyCoDv [channel].xdr) = outChar;
	safeLoc++;
	*M562_GSR = tyCoDv [channel].xrdy;
	}
    else
	{
	/* turn off the transmitter */

	*(tyCoDv [channel].ier) &= ~M562_IER_TXRDY;
	safeLoc++;
	}
    }
/*******************************************************************************
*
* tyCoStartup - transmitter startup routine
*
* Call interrupt level character output routine for mz7122/24 DUSCC.
*/

LOCAL VOID tyCoStartup (pTyCoDv)
    TY_CO_DEV *pTyCoDv;		/* ty device to start up */

    {
    char outChar;

    if ((pTyCoDv->created) &&
	(tyITx ((TY_DEV_ID) pTyCoDv, &outChar) == OK))
	{
	*(pTyCoDv->ier) |= M562_IER_TXRDY;
	safeLoc++;
	*(pTyCoDv->xdr) = outChar;
	safeLoc++;
	*M562_GSR = pTyCoDv->xrdy;
	safeLoc++;
	}
    }
