/* sysLib.c - Motorola MVME-147 system dependent library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01f,15oct88,dnw  changed hw init to turn of SYSFAIL
01e,09sep88,gae  documentation.
01d,07sep88,gae  documentation.
01c,26aug88,gae  documentation.
01b,09aug88,gae  fixed auxiliary clock and mailbox interrupts.
		 Memory now maps on bus at 8 M (was 4M).
01a,03may88,dfm  created by modifying version 01a of mv133a/sysLib.c.
	   +rdc  beefed up for 4.0.
*/
/*
DESCRIPTION
This library contains a set of routines to manipulate the primary functions
of the CPU board.  The goal is to provide a board-independant interface on
which VxWorks and application code can be built in a system-independant way.
Not every feature of every board is supported by this library; a particular
board may have various extensions to the capabilities described here.
Also not every board will support all the functions provided by this library.
And some boards provide some of the functions of this library with hardware
switches, jumpers, or PALs, instead of software controllable registers.

The funtions addressed here include:

    initialization functions:
	- initialize hardware to known state
	- identify the system

    memory/address space functions:
	- get on-board memory limit
	- map from local to bus and bus to local address spaces
	- enable/disable cache memory
	- set/get non-volatile RAM

    bus interrupt functions:
	- enable/disable bus interrupt levels
	- generate bus interrupts

    serial channel functions (see tyCoDrv):
	- enable/disable serial channel interrupts
	- set serial channel baud rates
	- get/put bytes from a serial channel

    clock/timer functions:
       - enable/disable timer interrupts
       - set timer periodic rate

    mailbox/location monitor functions:
       - enable mailbox/location monitor interrupts

AUTHOR
Original port of the MVME-147 by
Dennis McGhie of De la Rue Giori, Sunnyvale, CA
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "vme.h"
#include "memLib.h"
#include "sysLib.h"
#include "config.h"

IMPORT char end;		/* end of system, created automatically by ld */
IMPORT VOID logMsg ();



/* globals */

int   sysBus      = BUS;                /* system bus type (VME_BUS, etc) */
int   sysCpu      = CPU;                /* system CPU type (MC680x0) */
char *sysBootLine = BOOT_LINE_ADRS;	/* address of boot line */
char *sysExcMsg   = EXC_MSG_ADRS;	/* catastrophic message area */
int   sysProcNum;			/* processor number of this CPU */
int   sysFlags;			/* boot flags */
char  sysBootHost [BOOT_FIELD_LEN];	/* name of host from which we booted */
char  sysBootFile [BOOT_FIELD_LEN];	/* name of file from which we booted */

char lnEnetAddr [6] = { 0x08, 0x00, 0x3e, 0x00, 0x00, 0x00 };
			/* the latter gets initialized in sysHwInit */


/* locals */

LOCAL int sysClkTicksPerSecond   = 60;
LOCAL BOOL sysClkRunning         = FALSE;
LOCAL FUNCPTR sysClkRoutine      = NULL;
LOCAL int sysClkArg;
LOCAL BOOL sysClkIsConnected     = FALSE;
LOCAL int auxClkTicksPerSecond   = 60;
LOCAL BOOL sysAuxClkRunning      = FALSE;
LOCAL FUNCPTR sysAuxClkRoutine   = NULL;
LOCAL int sysAuxClkArg;
LOCAL BOOL auxClkIsConnected     = FALSE;
LOCAL FUNCPTR sysMailboxRoutine  = NULL;
LOCAL int sysMailboxArg;
LOCAL BOOL sysMailboxIsConnected = FALSE;

LOCAL VOID sysClkInt ();
LOCAL VOID sysMailboxInt ();

LOCAL VOID sysAcfailInt ();
LOCAL VOID sysAbortInt ();


/*******************************************************************************
*
* sysModel - return model name of the system CPU
*
* Use this routine to find the model name of the system CPU.
*
* RETURNS: pointer to string "Motorola MVME-147"
*/

char *sysModel ()
    {
    return ("Motorola MVME-147");
    }
/*******************************************************************************
*
* sysHwInit - initialize hardware
*
* This routine initializes various features of the board.
* It is normally called from usrInit (2) in usrConfig (1).
*
* Control registers are set up.
* The Peripheral Channel Controller (PCC) is initialized.
* Timers are disabled.  The VMEchip is set up.
*/

VOID sysHwInit ()

    {
    /* base for 16 PCC generated interrupters */
    *INT_BASE_CTL = PCC_INT_VEC_BASE;

    /* base for 8 VMEchip utility interrupts */
    *LCSR_UTIL_VEC_BASE = UTIL_INT_VEC_BASE;

    /* no access to VMEchip GCSR until processor number is set later on */
    *LCSR_GCSR_BASE_ADRS = 0x0f;

    intVecSet (INUM_TO_IVEC (INT_VEC_ACFAIL), sysAcfailInt);
    *ACFAIL_INT_CTL = 0x80;		/* reset & disable ACFAIL interrrupts */
					/* MVME-147 rev A boards screw up */

    *WDOG_CSR = 0x0a;			/* reset & disable watch dog timer */

    /* MVME-147 documentation is incomplete regarding the function of the
     * PCC generated BERR interrupt.  I tried to use it because I thought
     * that it was necessary, but totally failed.  Then when it was disabled,
     * suddenly the 68030 BERR interrupts worked just fine.  Unless a) the
     * documentation is made clearer, and b) there is some obvious advantage
     * to it, I intend to leave it disabled. */

/**#    intVecSet (INUM_TO_IVEC (INT_VEC_BERR), sysBerrInt);
/**#    *BERR_INT_CTL = 0x88;		/* reset & enable BERR interrrupts */
    *BERR_INT_CTL = 0x80;		/* reset & disable BERR interrrupts */

    intVecSet (INUM_TO_IVEC (INT_VEC_ABORT), sysAbortInt);

    *ABORT_INT_CTL = 0x88;	/* reset & enable the abort button interrupt */

    *TIC_1_CSR = 0;			/* disable tic 1 counter */
    *TIC_1_INT_CTL = 0x80;		/* reset & disable tic 1 interrupts */
    *TIC_2_CSR = 0;			/* disable tic 2 counter */
    *TIC_2_INT_CTL = 0x80;		/* reset & disable tic 2 interrupts */

    *SERIAL_INT_CTL = 0;		/* serial line interrupts */

    *PRINTER_INT_CTL = 0;		/* disable printer interrupts */

    *SCSI_DMA_TABLE_ADRS = 0;		/* clear to be safe */
    *SCSI_PCC_DMA_ADRS = 0;
    *SCSI_DMA_BYTE_COUNT = 0;

    *SCSI_DMA_INT_CTL = 0x80;		/* reset&disable SCSI dma interrupts */
    *SCSI_DMA_CSR = 0;			/* disable SCSI dma */
    *SCSI_INT_CTL = 0x40;		/* disable SCSI interrupts (reset) */

    *LAN_INT_CTL = 0;			/* disable LAN interrupts */

/* XXX map at 8 M.
    *SLAVE_BASE_CTL = 5;		/* VME slave access would be at */
					/* (1 * DRAMSIZE) if allowed */
    *SLAVE_BASE_CTL = 0xa;		/* VME slave access would be at */
					/* (2 * DRAMSIZE) if allowed */

    *SOFT_1_INT_CTL = 0;		/* disable software interrupt 1 */
    *SOFT_2_INT_CTL = 0;		/* disable software interrupt 2 */

    *LCSR_SYS_CONF    = 0;		/* priority based arbiter &
					 * turn off SYSFAIL */
    *LCSR_RQST_CONF   = 3;		/* VME level 3 requester, ROR mode */
    *LCSR_MASTER_CONF = 0;		/* simple A32 D32 master */
    *LCSR_TMO_CONF    = 0x40;		/* access & arbitration timer setup */

    *LCSR_RQST_CONF |= 0x80;		/* request VME bus mastership */
    while ((*LCSR_RQST_CONF & 0x40) == 0)
        ;				/* hang until master granted */
    *LCSR_SLAVE_CONF = 0;		/* no VME access for now */
			    /* note VMEchip slave write posting doesn't work */

    *LCSR_SLAVE_AM = 0xfb;		/* allow all but block types of
					 * VME access if enabled */

    *LCSR_RQST_CONF &= 0x7f;		/* release VME bus mastership */

    *LCSR_MASTER_AM = 0;		/* master dynamically calculates
					 * AM codes */

    *LCSR_INT_MASK = 0;			/* mask all VME interrupt levels */
    *LCSR_UTIL_INT_MASK = 0;		/* disable signals & location monitors*/

    *GP_STATUS = 3;			/*reset power-up & parity error detect*/
    *GP_CTL = 0x10;			/* master interrupt enable */
					/* no parity checking */
					/* reset button enabled */

    /* extract the ethernet address out of nv ram */

    /* NOTE the Motorola convention for the ethernet address is that they only
     * save the low 3 bytes in BBRAM.  The high three bytes are the
     * manufacturers code, and Motorola software knows its own.  
     * The Motorola code is 0x08003E.
     */

    lnEnetAddr [3] = *BB_ENET;
    lnEnetAddr [4] = *(BB_ENET + 1);
    lnEnetAddr [5] = *(BB_ENET + 2);
    }
/*******************************************************************************
*
* sysMemTop - get top of memory address
*
* This routine finds the size of system RAM.
*
* NOTE MVME-147:
* Assumed size is as specified in config.h.
* Can't probe for missing address because it might be a second CPU.
*
* RETURNS: address of the first missing byte of memory
*/

char *sysMemTop ()

    {
    static char *memTop = (char *)LOCAL_MEM_SIZE;

    return (memTop);
    }
/*******************************************************************************
*
* sysToMonitor - transfers to rom monitor
*
* This routine transfers control to the rom monitor.  It is usually called
* only by the routine reboot, which services control-x, and bus errors at
* interrupt level.  In special circumstances, however, the user may wish
* to introduce a new startType such that a special bootrom facility could be
* enabled.
*
* RETURNS: OK (if we ever continue from the rom monitor)
*
* INTERNAL
* Note that the "WARM" restart address is at (ROM_BASE_ADRS + 16) bytes.
*/

STATUS sysToMonitor (startType)
    int startType; /* parameter is passed to ROM to tell it how to boot, */
                   /* possible types are defined in sysLib.h             */
    {
    (* ((FUNCPTR) (ROM_BASE_ADRS + 16))) (startType);

    return (OK);	/* in case we ever continue from rom monitor */
    }

/*******************************************************************************
*
* sysClkConnect - connect routine to system clock interrupt
*
* This routine connects the given function to the system clock interrupt.
* It is normally called from usrRoot (2) in usrConfig (1) to connect
* usrClock (2) to the system clock interrupt.
*
* RETURN: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), usrClock (2)
*/

STATUS sysClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each system clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    if (!sysClkIsConnected &&
	intConnect (INUM_TO_IVEC (INT_VEC_CLOCK), sysClkInt, TRUE))
	{
	return (ERROR);
	}

    sysClkIsConnected = TRUE;

    sysClkRoutine = routine;
    sysClkArg     = arg;

    return (OK);
    }
/*******************************************************************************
*
* sysClkDisable - turn off system clock interrupts
*/

VOID sysClkDisable ()

    {
    if (sysClkRunning)
	{
	/* disable interrupts */

	*TIC_1_INT_CTL = 0; 
	sysClkRunning = FALSE;
	}
    }
/********************************************************************************
* sysClkEnable - turn system clock interrupts on
*/

VOID sysClkEnable ()

    {
    int temp;

    if (!sysClkRunning)
	{
	/* preload the preload register */

	temp = 65536 - (160000 / sysClkTicksPerSecond);
	*TIC_1_PRELOAD = (short) temp;

	/* enable the clock interrupt. */

	/* load counter from preload register */
	*TIC_1_CSR = 0;		

	/* clr overflow counter, enable & start counter */
	*TIC_1_CSR = 7;			

	/* reset int, enable int, set level */
	*TIC_1_INT_CTL = INT_LVL_TIC_1 | 0x88; 

	sysClkRunning = TRUE;
	}
    }
/*******************************************************************************
*
* sysClkRateGet - get rate of system clock
*
* This routine is used to find out the system clock speed.
*
* RETURNS: number of ticks per second of the system clock
*
* SEE ALSO: sysClkRateSet (2)
*/

int sysClkRateGet ()
    
    {
    return (sysClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysClkRateSet - set rate of system clock
*
* This routine sets the clock rate of the system clock.
* It is normally called by usrRoot (2) in usrConfig (1).
*
* SEE ALSO: sysClkGetRate (2)
*/

VOID sysClkRateSet (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
	sysClkTicksPerSecond = ticksPerSecond;
	     
    if (sysClkRunning)
	{
	sysClkDisable ();
	sysClkEnable ();
	}
    }
/*******************************************************************************
*
* sysClkInt - handle system or auxiliary clock interrupts
*
* This routine handles system or auxiliary clock interrupts.
* It is attached to both the system clock interrupt vector and
* the auxiliary clock vector.
*/

LOCAL VOID sysClkInt (sysClock)
    BOOL sysClock;		/* TRUE=system, FALSE=auxiliary */

    {
    if (sysClock)
	{
	if (sysClkRoutine != NULL)
	    (* sysClkRoutine) (sysClkArg);

	/* reset clock interrupt */

	/* clear overflow counter, enable & start counter */
	*TIC_1_CSR = 7;			

	/* reset int, enable int, set level */
	*TIC_1_INT_CTL = INT_LVL_TIC_1 | 0x88; 
	}
    else
	{
	if (sysAuxClkRoutine != NULL)
	    (* sysAuxClkRoutine) (sysAuxClkArg);

	/* reset clock interrupt */

	/* clear overflow counter, enable & start counter */
	*TIC_2_CSR = 7;

	/* reset int, enable int, set level */
	*TIC_2_INT_CTL = INT_LVL_TIC_2 | 0x88;
	}
    }

/*******************************************************************************
*
* sysAuxClkConnect - connect a routine to the auxiliary clock interrupt
*
* This routine connects a user routine to the auxiliary clock interrupt.
* Auxiliary clock interrupts are not enabled.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), sysAuxClkDisconnect (2)
*/

STATUS sysAuxClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each auxiliary clock interrupt */
    int arg;		/* argument to auxiliary clock interrupt routine */

    {
    if (!auxClkIsConnected &&
	intConnect (INUM_TO_IVEC(INT_VEC_AUX_CLOCK), sysClkInt, FALSE) == ERROR)
	{
	return (ERROR);
	}

    auxClkIsConnected = TRUE;

    sysAuxClkRoutine = routine;
    sysAuxClkArg     = arg;

    return (OK);
    }
/*******************************************************************************
*
* sysAuxClkDisconnect - disconnect a routine from the auxiliary clock interrupt
*
* This routine disables the auxiliary clock interrupt and disconnects
* the routine currently connected to the auxiliary clock interrupt.
*
* SEE ALSO: sysAuxClkConnect (2)
*/

VOID sysAuxClkDisconnect ()

    {
    /* disable auxiliary clock */

    sysAuxClkDisable ();

    /* connect dummy routine, just in case */

    sysAuxClkConnect (logMsg, (int) "auxiliary clock interrupt\n");
    }
/*******************************************************************************
*
* sysAuxClkDisable - turn off auxiliary clock interrupts
*/

VOID sysAuxClkDisable ()

    {
    if (sysAuxClkRunning)
	{
	/* clear overflow counter & stop counter */
	*TIC_2_CSR = 0x04;
	/* reset int, disable int, set level to 0 */
	*TIC_2_INT_CTL = 0x80;

	/*
	*TIC_2_INT_CTL = 0; 
	*/
	sysAuxClkRunning = FALSE;
	}
    }
/*******************************************************************************
*
* sysAuxClkEnable - turn auxiliary clock interrupts on
*/

VOID sysAuxClkEnable ()

    {
    int temp;

    if (!sysAuxClkRunning)
	{
	/* preload the preload register */

	temp = 65536 - (160000 / auxClkTicksPerSecond);

	*TIC_2_PRELOAD = (short) temp;

	/* enable the clock interrupt. */

	*TIC_2_CSR = 0;	/* load counter from preload register */

	/* clr overflow counter, enable & start counter */
	*TIC_2_CSR = 7;

	/* reset int, enable int, set level */
	*TIC_2_INT_CTL = INT_LVL_TIC_2 | 0x88;

	sysAuxClkRunning = TRUE;
	}
    }
/*******************************************************************************
*
* sysAuxClkRateGet - get rate of auxiliary clock
*
* This routine finds out the auxiliary clock speed.
*
* RETURNS: number of ticks per second of the auxiliary clock
*
* SEE ALSO: sysAuxClkRateSet (2)
*/

int sysAuxClkRateGet ()
    
    {
    return (auxClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysAuxClkRateSet - set rate of auxiliary clock
*
* This routine sets the clock rate of the auxiliary clock.
* Auxiliary clock interrupts are not enabled.
*
* SEE ALSO: sysAuxClkConnect (2), sysAuxClkRateGet (2)
*/

VOID sysAuxClkRateSet (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
	auxClkTicksPerSecond = ticksPerSecond;
	     
    if (sysAuxClkRunning)
	{
	sysAuxClkDisable ();
	sysAuxClkEnable ();
	}
    }

/*******************************************************************************
*
* sysLocalToBusAdrs - convert local address to bus address
*
* Given a local memory address, this routine returns the VME address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unable to get to that local address from the bus
*
* SEE ALSO: sysBusToLocalAdrs (2)
*/

STATUS sysLocalToBusAdrs (adrsSpace, localAdrs, pBusAdrs)
    int adrsSpace;      /* bus address space in which busAdrs resides,    */
                        /* use address modifier codes as defined in vme.h */
                        /* such as VME_AM_STD_SUP_DATA                    */
    char *localAdrs;    /* local address to convert                       */
    char **pBusAdrs;    /* where to return bus address                    */

    {
    if ((localAdrs < LOCAL_MEM_LOCAL_ADRS) || (localAdrs >= sysMemTop ()))
	{
	/* this is off-board memory - just return local address */

	*pBusAdrs = localAdrs;
	return (OK);
	}
    else
	{
	/* this is on-board memory - map to bus address space;
	 *   the following memory mapping is established in sysProcNumSet():
	 *   - only processor 0 has memory on bus,
	 *   - the memory is placed in STD & EXT space at
	 *      address LOCAL_MEM_BUS_ADRS.
	 *   - short I/O is placed in SHORT at address 0.
	 */

	switch (adrsSpace)
	    {
	    case VME_AM_SUP_SHORT_IO: /* just make sure 16 bits */
	    case VME_AM_USR_SHORT_IO: /* local map is 0xffff0000..0xffffffff */
		*pBusAdrs = (char *) ((int)localAdrs & 0xffff);
		return (OK);

	    case VME_AM_STD_SUP_PGM:
	    case VME_AM_STD_SUP_DATA:
	    case VME_AM_STD_USR_PGM:
	    case VME_AM_STD_USR_DATA:
	    case VME_AM_EXT_SUP_PGM:
	    case VME_AM_EXT_SUP_DATA:
	    case VME_AM_EXT_USR_PGM:
	    case VME_AM_EXT_USR_DATA:
		*pBusAdrs = localAdrs +
			    LOCAL_MEM_BUS_ADRS - LOCAL_MEM_LOCAL_ADRS;
		return (OK);

	    default:
		return (ERROR);
	    }
	}
    }
/*******************************************************************************
*
* sysBusToLocalAdrs - convert bus address to local address
*
* Given a VME memory address, this routine returns the local address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unknown address space
*
* SEE ALSO: sysLocalToBusAdrs (2)
*/

STATUS sysBusToLocalAdrs (adrsSpace, busAdrs, pLocalAdrs)
    int adrsSpace;      /* bus address space in which busAdrs resides,     */
                        /* use address modifier codes as defined in vme.h, */
                        /* such as VME_AM_STD_SUP_DATA                     */
    char *busAdrs;      /* bus address to convert                          */
    char **pLocalAdrs;  /* where to return local address                   */

    {
    switch (adrsSpace)
	{
	case VME_AM_SUP_SHORT_IO:
	case VME_AM_USR_SHORT_IO:
	    *pLocalAdrs = (char *) (0xffff0000 | (int)busAdrs);
	    return (OK);

	case VME_AM_STD_SUP_PGM:
	case VME_AM_STD_SUP_DATA:
	case VME_AM_STD_USR_PGM:
	case VME_AM_STD_USR_DATA:
	    if (busAdrs < sysMemTop () || busAdrs > (char *)0x00ffffff)
		return (ERROR);

	    *pLocalAdrs = (char *) busAdrs;
	    return (OK);

	case VME_AM_EXT_SUP_PGM:
	case VME_AM_EXT_SUP_DATA:
	case VME_AM_EXT_USR_PGM:
	case VME_AM_EXT_USR_DATA:
	    if ((busAdrs < sysMemTop ()) || (busAdrs > (char *)0xff800000))
		return (ERROR);
	    *pLocalAdrs = (char *) busAdrs;
	    return (OK);

	default:
	    return (ERROR);
	}
    }
/*******************************************************************************
*
* sysIntDisable - disable interrupt level
*
* This routine disables the specified interrupt level.
*
* RETURNS: OK
*/

STATUS sysIntDisable (intLevel)
    int intLevel;	/* interrupt level to disable */

    {
    *LCSR_INT_MASK &= ~(1 << intLevel);
    return (OK);
    }
/*******************************************************************************
*
* sysIntEnable - enable interrupt level
*
* This routine enables the specified VME interrupt level.
*
* RETURNS: OK
*
* ARGSUSED
*/

STATUS sysIntEnable (intLevel)
    int intLevel;	/* interrupt level to enable */

    {
    *LCSR_INT_MASK |= (1 << intLevel);
    return (OK);
    }
/*******************************************************************************
*
* sysBusIntAck - acknowledge interrupt
*
* This routine acknowledges the specified interrupt.
*
* NOTE:
* VME interrupts will be acknowledged automatically by hardware.
* This routine has no effect.
*
* RETURNS: NULL
*
* ARGSUSED
*/

int sysBusIntAck (intLevel)
    int intLevel;	/* interrupt level to acknowledge */

    {
    return (NULL);
    }
/*******************************************************************************
*
* sysBusIntGen - generate interrupt
*
* This routine generates a VME bus interrupt.
*
* RETURNS: OK
*/

STATUS sysBusIntGen (level, vector)
    int level;
    int vector;

    {
    *LCSR_INT_STATUS_ID = (char) vector;
    *LCSR_INT_REQ = (char) level;

    return (OK);
    }

/*******************************************************************************
*
* sysMailboxConnect - connect routine to the mailbox interrupt
*
* This routine connects the given function to the mailbox interrupt.
*
* NOTE:
* The mailbox interrupt is signal high priority (SIGHP).
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2)
*/

STATUS sysMailboxConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each mailbox interrupt */
    int arg;		/* argument with which to call routine */

    {
    if (!sysMailboxIsConnected &&
	intConnect (INUM_TO_IVEC (UTIL_INT_VEC_BASE + VME_INT_SIGHP),
			sysMailboxInt, NULL) == ERROR)
	{
	return (ERROR);
	}

    sysMailboxIsConnected = TRUE;
    sysMailboxRoutine     = routine;
    sysMailboxArg         = arg;

    return (OK);
    }
/*******************************************************************************
* sysMailboxInt - handle mailbox interrupt
*
* Mailbox interrupts must be acknowledged.
*/

LOCAL VOID sysMailboxInt ()
    {
    if (sysMailboxRoutine != NULL)
	sysMailboxRoutine (sysMailboxArg);
    *GCSR_GLOBAL_1 |= 2;	/* acknowledge SIGHP interrupt */
    }
/*******************************************************************************
*
* sysMailboxEnable - enable mailbox interrupt
*
* This routine enables the mailbox interrupt.
*
* NOTE:
* The mailbox interrupt is signal high priority (SIGHP).
* The address of the register used to set SIGHP is
* configured as a function of the processor number and is
* set in sysProcNumSet, and so mailboxAdrs is ignored.
*
* RETURNS: OK
*
* ARGSUSED
*/

STATUS sysMailboxEnable (mailboxAdrs)
    char *mailboxAdrs;		/* address of mailbox (ignored) */

    {
    *LCSR_UTIL_INT_MASK |= 0x20;	/* unmask SIGHP interrupt */
    sysIntEnable (5);
    return (OK);
    }
/******************************************************************************
*
* sysProcNumGet - get processor number
*
* This routine returns the processor number previously set with 
* sysProcNumSet (2).
*
* RETURNS: processor number
*/

int sysProcNumGet ()

    {
    return (sysProcNum);
    }
/******************************************************************************
*
* sysProcNumSet - set processor number
*
* Set the processor number for this CPU.  Processor numbers should be
* unique on a single backplane.
*
* NOTE:
* The MVME-147 supports four different mappings of local memory to
* VME address space.  There are separate specifications for EXT
* and STD address spaces, so in theory up to 8 CPUs could be
* mapped simultaneously.  A more practical limit is 3.  Since
* the mapping is somewhat complicated, and vxWorks only requires
* mapping for processor 0, and then only for backplane networks,
* the implementation of multiple CPU mapping is left as an
* exercise for the reader.  On the other hand the VMEchip GCSR
* address can be specified anywhere from 0x00 to 0xe0 in increments
* of 0x10 in the short I/O space.  This can allow up to 15 CPUs
* in the same cage to form a backplane network using signal
* interrupts to coordinate their activities.
*/

VOID sysProcNumSet (procNum)
    int procNum;		/* processor number */

    {
    /* Note that in sysHwInit:
     *     *SLAVE_BASE_CTL = 5;
     *     *LCSR_SLAVE_AM = 0xfb;
     * This sets the VME address mapping of local memory to start at DRAMSIZE
     * for both STD and EXT modifiers.  The following code allows
     * processor 0 to respond to externally initiated VME cycles.
     */

    if (procNum == 0)
        {
	*LCSR_RQST_CONF |= 0x80;	/* request VME bus mastership */
	while ((*LCSR_RQST_CONF & 0x40) == 0)
	    ;				/* hang until master granted */
	*LCSR_SLAVE_CONF = 0x80;	/* enable VME access from now on */
			    /* note VMEchip slave write posting doesn't work */
	*LCSR_RQST_CONF &= 0x7f;	/* release VME bus mastership */
	}

    sysProcNum = procNum;

    if (sysProcNum < 15)	/* VME address of GCSR is func (sysProcNum) */
        *LCSR_GCSR_BASE_ADRS = (char)sysProcNum;
    }
/******************************************************************************
*
* sysBusTas - test and set across VME bus
*
* This routine does a 680x0 test-and-set instruction across the backplane.
*
* RETURNS: TRUE (successful set) or FALSE (failure)
*/

BOOL sysBusTas (adrs)
    char *adrs;		/* address to be tested and set */

    {
    return (vxTas (adrs));
    }


/* miscellaneous support routines */

/******************************************************************************
*
* sysNvRamGet - get contents out of non-volatile RAM
*
* Copies non-volatile memory into string.
* The string will be terminated with an EOS.
*/

VOID sysNvRamGet (string, strLen, offset)
    char *string;    /* where to copy non-volatile RAM           */
    int strLen;      /* maximum number of bytes to copy          */
    int offset;      /* (even) byte offset into non-volatile RAM */

    {
    bcopyBytes (NV_BOOT_LINE + offset, string, strLen);
    string [strLen] = EOS;

    if (string [0] == EOS)
	strcpy (string, DEFAULT_BOOT_LINE);
    }
/*******************************************************************************
*
* sysNvRamSet - write to non-volatile RAM
*
* Copy 'string' into non-volatile RAM.
*
* INTERNAL
* The non-volatile RAM in the MVME-147 is very simple.
* It is simply organized as an array of static RAM 2040 bytes
* long starting at address BBRAM.  Note that although the
* NVRAM looks just like normal memory, it is a single byte wide
* chip and will break long words into four transfers.  Thus
* there is no significant speed advantage to longs and bcopyBytes (2)
* is used instead just to be safe.  config.h defines the base
* of the boot line space in the NVRAM.
*/

VOID sysNvRamSet (string, strLen, offset)
    char *string;     /* string to be copied into non-volatile RAM */
    int strLen;       /* maximum number of bytes to copy           */
    int offset;       /* (even) byte offset into non-volatile RAM  */
    {
    bcopyBytes (string, NV_BOOT_LINE + offset, strLen);
    }
/*******************************************************************************
*
* sysMemParityError - memory parity error interrupt routine
*
* This routine handles the memory parity error on the MVME-147 CPU.
* It clears the error bit in the GPSR in the PCC and then logs a message.
*/

VOID sysMemParityError ()

    {
    *GP_STATUS = 1;			/* reset parity error detected */
    logMsg ("Memory parity error\n");
    }
/*******************************************************************************
*
* sysLanIntEnable - enable LAN interrupt level
*
* This routine enables interrupts at the specified level for on board LAN chip.
* LAN interrupts are controlled by the PCC chip.
*
* RETURNS: OK
*/

STATUS sysLanIntEnable (intLevel)
    int intLevel;	/* interrupt level to enable */

    {
    *LAN_INT_CTL = (intLevel & 7) | 8;		/* PCC LAN interrupt enable */

    return (OK);
    }
/*******************************************************************************
*
* sysLanIntDisable - disable LAN interrupt
*
* This routine disables interrupts for the on board LAN chip.
*
* LAN interrupts are controlled by the PCC chip.
*
* RETURNS: OK
*/

STATUS sysLanIntDisable ()

    {
    *LAN_INT_CTL = 0;

    return (OK);
    }
