/* tyP2Drv.c - The MVME-133 P2 tty handler */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01b,22sep88,gae  fixed channel enables so that characters echo.
01a,25jun88,gae  based on work by Jim Foris which was derived the iv24 port.
*/

/*
DESCRIPTION
This is the driver for the Z8530 Serial Communications Controller chip.
Only the ASYNC modes are supported on the are 2 channels on the P2 bus.
Channel B is the RS-232C port, channel A is an RS-485 port which is untested.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, tyP2Drv to
initialize the driver, and tyP2DevCreate to create devices.

TYP2DRV
Before using the driver, it must be initialized by calling the routine:
.CS
    tyP2Drv ()
.CE
This routine should be called exactly once, before any reads, writes, or
tyP2DevCreates.  Normally, it is called from usrRoot.

CREATING TERMINAL DEVICES
Before a terminal can be used, it must be created.
This is done with the tyP2DevCreate call.
Each port to be used should have exactly one device associated with it,
by calling this routine.

.CS
    STATUS tyP2DevCreate (name, channel, rdBufSize, wrtBufSize)
	char *name;	/* Name to use for this device
	int channel;	/* Physical channel for this device, always 0
	int rdBufSize;	/* Read buffer size, in bytes
	int wrtBufSize;	/* Write buffer size, in bytes
.CE

For instance, to create the device "/tyP2/0", with buffer sizes of 512 bytes,
the proper call would be:
.CS
   tyP2DevCreate ("/tyP2/0", 0, 512, 512);
.CE
Port B, or channel 1, is the RS-232C port.  Port A, or channel 0, is RS-485
and untested.

IOCTL
This driver responds to all the same ioctl codes as a normal ty driver.
The baud rates available are 110 through 19.2 baud.
The default baud rate is 9600.
*/


#include "vxWorks.h"
#include "ioLib.h"
#include "iosLib.h"
#include "tyLib.h"
#include "mv133.h"
#include "z8530.h"
#include "config.h"

#define	HZ	1230769		/* external clock rate */

#define DEFAULT_BAUD	9600

typedef struct			/* TY_P2_DEV */
    {
    TY_DEV tyDev;
    BOOL created;	/* true if this device has really been created */
    char *contReg;	/* control register I/O address */
    char *dataPort;	/* data port I/O address */
    } TY_P2_DEV;

LOCAL TY_P2_DEV tyP2Dv [N_SIO_CHANNELS]=	/* device descriptors */
    {
    {{{{NULL}}}, FALSE, SCCA, SCCA_D},		/* RS-485 */
    {{{{NULL}}}, FALSE, SCCB, SCCB_D},		/* RS-232C */
    };

LOCAL int tyP2DrvNum;		/* driver number assigned to this driver */


/* forward declarations */

LOCAL VOID tyP2Startup ();
LOCAL int tyP2Open ();
LOCAL int tyP2Read ();
LOCAL int tyP2Write ();
LOCAL STATUS tyP2Ioctl ();
LOCAL VOID tyP2Int ();


/*******************************************************************************
*
* tyP2Drv - ty driver initialization routine
*
* This routine initializes the serial driver, sets up interrupt vectors,
* and performs hardware initialization of the serial ports.
*
* This routine must be called in supervisor state, since it does
* physical I/O directly.
*/

STATUS tyP2Drv ()

    {
    /* check if driver already installed */

    if (tyP2DrvNum > 0)
	return (OK);

    (void) intConnect (INUM_TO_IVEC(SCC_INT_VECT_NUM), tyP2Int, NULL);

    tyP2HrdInit ();

    tyP2DrvNum = iosDrvInstall (tyP2Open, (FUNCPTR)NULL, tyP2Open,
				(FUNCPTR)NULL, tyP2Read, tyP2Write, tyP2Ioctl);

    return (tyP2DrvNum == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* tyP2DevCreate - create a device for the onboard ports
*
* This routine creates a device on one of the serial ports.  Each port
* to be used should have exactly one device associated with it, by calling
* this routine.
*
* RETURNS: OK or ERROR
*/

STATUS tyP2DevCreate (name, channel, rdBufSize, wrtBufSize)
    char *name;		/* Name to use for this device */
    int channel;	/* Physical channel for this device (0=A, 1=B) */
    int rdBufSize;	/* Read buffer size, in bytes */
    int wrtBufSize;	/* Write buffer size, in bytes */

    {
    if (tyP2DrvNum <= 0)
	{
	errnoSet (S_ioLib_NO_DRIVER);
	return (ERROR);
	}

    /* if this device already exists, don't create it */

    if (tyP2Dv[channel].created)
	return (ERROR);

    /* initialize the ty descriptor, and turn on the bit for this
     * receiver in the interrupt mask */

    if (tyDevInit (&tyP2Dv[channel].tyDev, 
		    rdBufSize, wrtBufSize, tyP2Startup) != OK)
	{
	return (ERROR);
	}

    tyP2InitChannel (channel);	    /* initialize the channel hardware */

    /* Mark the device as having been created, and add the device to
     * the I/O system.
     */

    tyP2Dv[channel].created = TRUE;

    return (iosDevAdd ((DEV_HDR *) &tyP2Dv[channel], name, tyP2DrvNum));
    }
/*******************************************************************************
*
* tyP2HrdInit - initialize the USART
*
* This routine initializes the USART for the VxWorks environment. 
* This routine must be called in supervisor mode.
*/

LOCAL VOID tyP2HrdInit ()

    {
    int oldlevel;		/* current interrupt level mask */
    int ix;

    oldlevel = intLock ();

    *SCCA = SCC_WR0_NULL_CODE;	/* sync the state machine */
    *SCCA = SCC_WR0_NULL_CODE;

    *SCCA = SCC_WR0_SEL_WR9;	/* write register 9 - master int control */
    *SCCA = SCC_WR9_HDWR_RST;	/* reset */

    for (ix = 0; ix < N_SIO_CHANNELS; ix++)
	tyP2ResetChannel (ix);

    intUnlock (oldlevel);
    } 
/*******************************************************************************
*
* tyP2ResetChannel - reset a single channel
*/

LOCAL VOID tyP2ResetChannel (channel)
    int channel;

    {
    int delay;
    FAST char *cr = tyP2Dv [channel].contReg;	/* SCC control reg adr */

    *cr = SCC_WR0_NULL_CODE;
    *cr = SCC_WR0_NULL_CODE;

    *cr = SCC_WR0_ERR_RST;
    *cr = SCC_WR0_RST_INT;

    *cr = SCC_WR0_SEL_WR9;	/* write register 9 - master int control */

    if (channel == 0)
	*cr = SCC_WR9_CH_A_RST;	/* reset channel A */
    else
	*cr = SCC_WR9_CH_B_RST;	/* reset channel B */


    for (delay = 0; delay < 1000; delay++); /* spin wheels for a moment */

    *cr = SCC_WR0_NULL_CODE;
    }
/*******************************************************************************
*
* tyP2InitChannel - initialize a single channel
*/

LOCAL VOID tyP2InitChannel (channel)
    int channel;

    {
    int baudConstant;
    int oldlevel;			/* current interrupt level mask */
    FAST char *cr = tyP2Dv [channel].contReg;/* SCC control reg adr */

    oldlevel = intLock ();		/* disable interrupts during init */

    tyP2ResetChannel (channel);		/* reset the channel */

    /* MODES */

    /* initialize registers */

    *cr = SCC_WR0_SEL_WR4;	/* write reg 4 - misc parms and modes */
    *cr = SCC_WR4_1_STOP | SCC_WR4_16_CLOCK;

    *cr = SCC_WR0_SEL_WR2;		/* interrupt vector */
    *cr = SCC_INT_VECT_NUM;

    *cr = SCC_WR0_SEL_WR3;		/* write reg 3 - receive parms */
    *cr = SCC_WR3_RX_8_BITS;		/* 8 bits, Rx enabled */

    *cr = SCC_WR0_SEL_WR5;		/* tx parms */
    *cr = SCC_WR5_TX_8_BITS | SCC_WR5_RTS | SCC_WR5_DTR;
		/* 8 bits, RTS/DTR must be enabled but unused by MVME-133 */

    *cr = SCC_WR0_SEL_WR9;		/* master interrupt control */
    *cr = SCC_WR9_MIE;			/* enable interrupts */

    *cr = SCC_WR0_SEL_WR10;
    *cr = SCC_WR10_NRZ;             /* make sure in NRZ format */

    *cr = SCC_WR0_SEL_WR11;		/* clock mode */
    *cr = SCC_WR11_RX_BR_GEN | SCC_WR11_TX_BR_GEN | SCC_WR11_TRXC_OI |
	    SCC_WR11_OUT_BR_GEN;	/* Rx & Tx clocks = BRG output, */
					/* TXRC = BRG output */
    /* calculate the baud rate constant, from the input clock freq.
     * assumes that the input clock is the 1.230769 MHz external clock source,
     * and the divide by 16 bit is set (done in tyP2HrdInit).
     */

    baudConstant = (HZ / (DEFAULT_BAUD * 32)) - 2;

    *cr = SCC_WR0_SEL_WR12;		/* LSB of baud constant */
    *cr = LSB(baudConstant);		/* write LSB */
    *cr = SCC_WR0_SEL_WR13;		/* MSB of baud constant */
    *cr = MSB(baudConstant);		/* write MSB */

    *cr = SCC_WR0_SEL_WR14;
    *cr = SCC_WR14_SRC_BR | SCC_WR14_BR_SRC;  /* disable BRG */

    *cr = SCC_WR0_SEL_WR15;		/* external/status interrupt cntrl */
    *cr = SCC_WR15_BREAK_IE;		/* interrupt on BREAK */


    /* ENABLES */

    *cr = SCC_WR0_SEL_WR14;		/* misc control bits */
    *cr = SCC_WR14_SRC_BR | SCC_WR14_BR_EN;	/* enable BRG */

    *cr = SCC_WR0_SEL_WR3;		/* write reg 3 - receive parms */
    *cr = SCC_WR3_RX_8_BITS | SCC_WR3_RX_EN;	/* 8 bits, Rx enabled */

    *cr = SCC_WR0_SEL_WR5;		/* tx parms */
    *cr = SCC_WR5_TX_8_BITS | SCC_WR5_RTS | SCC_WR5_DTR | SCC_WR5_TX_EN;
					/* 8 bits, Tx enabled */

    *cr = SCC_WR0_RST_INT;		/* reset external interrupts */
    *cr = SCC_WR0_RST_INT;		/* twice */

    *cr = SCC_WR0_SEL_WR1;		/* int and xfer mode */
    *cr = SCC_WR1_INT_ALL_RX | SCC_WR1_TX_INT_EN;
					/* interrupt on Rx & Tx */

    intUnlock (oldlevel);
    }

/*******************************************************************************
*
* tyP2Open - open file to USART
*
* ARGSUSED
*/

LOCAL int tyP2Open (pTyP2Dv, name, mode)
    TY_P2_DEV *pTyP2Dv;
    char *name;
    int mode;

    {
    return ((int) pTyP2Dv);
    }
/*******************************************************************************
*
* tyP2Read - task level read routine
*
* This routine fields all read calls to the P2 USART.
*/

LOCAL int tyP2Read (pTyP2Dv, buffer, maxbytes)
    TY_P2_DEV *pTyP2Dv;
    char *buffer;
    int maxbytes;

    {
    return (tyRead (&pTyP2Dv->tyDev, buffer, maxbytes));
    }
/*******************************************************************************
*
* tyP2Write - task level write routine
*
* This routine fields all write calls to the P2 USART.
*/

LOCAL int tyP2Write (pTyP2Dv, buffer, nbytes)
    TY_P2_DEV *pTyP2Dv;
    char *buffer;
    int nbytes;

    {
    return (tyWrite (&pTyP2Dv->tyDev, buffer, nbytes));
    }
/*******************************************************************************
*
* tyP2Ioctl - special device control
*
* This routine handles baud rate requests.
* All other requests are passed to tyIoctl.
*/

LOCAL STATUS tyP2Ioctl (pTyP2Dv, request, arg)
    TY_P2_DEV *pTyP2Dv;	/* device to control */
    int request;	/* request code */
    int arg;		/* some argument */

    {
    int baudConstant;
    char *cr;
    STATUS status;
    int oldLevel;

    switch (request)
	{
	case FIOBAUDRATE:

	    /* calculate the baud rate constant, from the input clock freq.
	     * assumes that the input clock is the 1.230769 MHz external clock
	     * source, and divide by 16 bit is set (done in tyP2HrdInit).
	     */

	    baudConstant = (HZ / (arg * 32)) - 2;

	    oldLevel = intLock ();

	    cr = pTyP2Dv->contReg;
	    *cr = SCC_WR0_NULL_CODE;	/* sync the state machine */
	    *cr = SCC_WR0_SEL_WR14;	/* disable BRG */
	    *cr = SCC_WR14_SRC_BR | SCC_WR14_BR_SRC;

	    *cr = SCC_WR0_SEL_WR12;	/* LSB of baud constant */
	    *cr = LSB(baudConstant);	/* write LSB */
	    *cr = SCC_WR0_SEL_WR13;	/* MSB of baud constant */
	    *cr = MSB(baudConstant);	/* write MSB */

	    *cr = SCC_WR0_SEL_WR14;	/* enable BRG */
	    *cr = SCC_WR14_SRC_BR | SCC_WR14_BR_EN;

	    intUnlock (oldLevel);

	    status = OK;
	    break;

	default:
	    status = tyIoctl (&pTyP2Dv->tyDev, request, arg);
	    break;
	}
    return (status);
    }

/*******************************************************************************
*
* tyP2Int - interrupt level processing
*
* This routine handles an interrupt from the one of the SCC's.
*/

LOCAL VOID tyP2Int ()

    {
    FAST char *cr;
    FAST char intStatus;
    FAST TY_P2_DEV *pTyP2Dv;
    int channel = 1;	/* B */
    char outChar;

    /* Find out which channel interrupted.
     * Channel B of the interrupting SCC indicates
     * which channel really interrupted.
     */

    pTyP2Dv = &tyP2Dv [channel];	/* point to B */
    cr = pTyP2Dv->contReg;

    *cr = SCC_WR0_SEL_WR2;		/* read reg 2 */
    intStatus = *cr;

    if ((intStatus & 0x08) != 0)
	{
	channel = 0;			/* channel A interrupted */
	pTyP2Dv = &tyP2Dv [channel];
	cr = pTyP2Dv->contReg;
	}

    switch (intStatus & 0x06)
	{
	case 0x00:			/* Tx Buffer Empty */
	    if (pTyP2Dv->created && tyITx (&pTyP2Dv->tyDev, &outChar) == OK)
		*pTyP2Dv->dataPort = outChar;
	    else
		{
		/* no more chars to transmit, reset the tx int,
		 * so the SCC doesn't keep interrupting
		 */

		*cr = SCC_WR0_RST_TX_INT;
		}
	    break;

	case 0x02:			/* External Status Change */
	    *cr = SCC_WR0_ERR_RST;
	    outChar = *pTyP2Dv->dataPort;   /* throw away the receive buffer */
	    /* handle 'break' character */
	    break;

	case 0x04:			/* RxChar Avail */
	    if (pTyP2Dv->created)
		(void) tyIRd (&pTyP2Dv->tyDev, *pTyP2Dv->dataPort);
	    break;

	case 0x06:			/* Special receive condition */
	    *cr = SCC_WR0_ERR_RST;
	    break;			/* ignore */
	}

    *cr = SCC_WR0_RST_HI_IUS;	/* reset the interrupt */
    }
/*******************************************************************************
*
* tyP2Startup - transmitter startup routine
*
* Call interrupt level character output routine.
*/

LOCAL VOID tyP2Startup (pTyP2Dv)
    TY_P2_DEV *pTyP2Dv;		/* ty device to start up */

    {
    char outChar;

    if (tyITx (&pTyP2Dv->tyDev, &outChar) == OK)
	*pTyP2Dv->dataPort = outChar;
    }
