/* sysALib.s - system dependent assembly language routines - ISI version */

/*
DESCRIPTION
This module contains system-dependent routines which must be written
in assembly language.  They perform functions such as locking and
unlocking interrupts, trapping to the ROM monitor, etc.  

INTERNAL: Many routines in this module "link" and "unlk" the "c" frame pointer
a6@ although they don't use it in any way!  This only for the benefit of
the stacktrace facility to allow it to properly trace tasks executing within
these routines.

*/

#define ASMLANGUAGE
#include "vxWorks.h"
#include "sysLib.h"
#include "vme68k30.h"
#include "config.h"

	/* internals */

	.globl	_sysInit	/* start of system code */
	.globl  _sysKernelTrap
	.globl  _sysVwTrap
	.globl  _sysVwTrapRtn
	.globl	_sysCacheEnable
	.globl	_sysMMUDisable

	/* externals */

	.globl	_usrInit	/* system initialization routine */

	.globl  _vme_short
        .globl  _vme_std
        .globl  _vme_ext
        .globl  _vme_window_std
        .globl  _vme_window_ext
	.globl	_vbmatm_reg

        .set    _vme_short,VME_SHORTIO
        .set    _vme_std,  VME_STANDARD
        .set    _vme_ext,  VME_EXTENDED
        .set    _vme_window_std,VME_WINDOW_STD
        .set    _vme_window_ext,VME_WINDOW_EXT
	.set	_vbmatm_reg,VBMATM_REG

	.globl	_sysTarget
	.set	_sysTarget,TARGET_68K30

	.text
	.even

/***********************************************************************
*
* sysInit - start after boot
*
* This is the system start-up entry point for VxWorks in ram.
* It is the first code executed after booting.
* It disables the interrupts, sets up the stack,
* and jumps to the C routine usrInit in usrConfig.c.
*
* The initial stack is set to grow down from sysInit.
* Note that this initial stack is used only by usrInit,
* then is never used again.
* Memory for the stack needs to be accounted for when determining the load
* address of the system.
*
* THIS IS NOT A CALLABLE ROUTINE.

* sysInit ()	/* THIS IS NOT A CALLABLE ROUTINE *

*/

_sysInit:
	movew	#0x3700,sr		/* disable interrupts, turn on M bit */
	movl	#3,d0			/* set function code registers */
	movl	d0,sfc
	movl	d0,dfc
	movel	#_sysInit,a7		/* set stack to grow down from code */
	movel	#BOOT_WARM_AUTOBOOT,a7@- /* push start type arg = WARM_BOOT */
	jsr	_usrInit		/* never returns - starts up kernel */

/***********************************************************************
*
* sysKernelTrap - trap to kernel function
*
* This routine traps to the kernel. It can only be called from assembly
* language.
* The trap number for the kernel is defined in config.h.
*
* NOMANUAL

* sysKernelTrap ()

*/
_sysKernelTrap:
	trap	#TRAP_KERNEL
	rts

/***********************************************************************
*
* sysVwTrap - trap to vxWorks function
*
* This routine traps to VxWorks. It can only be called from assembly
* language.
* The trap number for VxWorks is defined in config.h.
*
* NOMANUAL

* sysVwTrap ()

*/
_sysVwTrap:
	trap	#TRAP_VXWORKS
	rts

/*
 * This routine calls the routine whose address is in a0.  Since the system
 * is now in supervisor state (since we got here from a trap) this can be
 * used to call a routine in supervisor state.
 * THIS IS NOT A C CALLABLE ROUTINE!!
 */

_sysVwTrapRtn:
	jsr	a0@		/* vector to proper routine. */
	rte			/* return from trap */


/***********************************************************************
*
* sysCacheEnable - enable cache memory
*
* Enable or disable the 68030 on-chip cache.
*
* SEE ALSO: "MC68030 32-Bit Microprocessor User's Manual"

* VOID sysCacheEnable (enable)
*     BOOL enable;		/* enable if TRUE, otherwise disable 
*
* This modifies the cacr as well as mmutt0 on the mmu
*/
_sysCacheEnable:

	link	a6,#0
	movl	cac,d0
	movel	a6@(0x08),d1		/* put enable in d1 */
	cmpl	#1,d1			/* does enable == TRUE? */
	bne	sce0			/* if not, disable cache */
	orl	#(ICACHE_ENABLE|ICACHE_CLEAR),d0
	movel 	#0x00ff8147,_mmu_spot
	bra	sce1

sce0:	andl	#~ICACHE_ENABLE,d0
	movel 	#0x00ff8547,_mmu_spot

	/* Enable or disable the cache by writing to the cache control
	   register and the transparent translation register 0. */
sce1:
	movel	#_mmu_spot,a0
	pmove	a0@,mmutt0
	movl	d0,cac
	unlk	a6
	rts

/***********************************************************************
*
* sysMMUDisable - disable memory management
*
* Disable the 68030 on-chip memory management.
*
* This consists of setting the transparent translation
* register for the mmu:
*	enable tt0
*	cache on
*	R/W masked
*	FC masked
*	logical address masked
*/

/* pmove instruction cannot use direct or immediate addressing
 * so we need a memory location through which we can transfer
 * mmu data. */

_mmu_spot:
	.long	0

_sysMMUDisable:

	movel	#_mmu_spot,a0

	movel 	#0x00ff8147,a0@
	pmove	a0@,mmutt0
	rts

/***********************************************************************
 *
 * afcrb, afcwb, afcrw, afcww, afcrl, afcwl
 *
 * read/write byte/word/long to alternate function code space
 * these must be used to access I/O space on the 68k30 board
 */
	.globl	_afcrb,_afcwb,_afcrw,_afcww,_afcrl,_afcwl
_afcrb:
	movl	sp@(4),a0
	clrl	d0
	movsb	a0@,d0
	rts

_afcrw:
	movl	sp@(4),a0
	clrl	d0
	movsw	a0@,d0
	rts

_afcrl:
	movl	sp@(4),a0
	movsl	a0@,d0
	rts

_afcwb:
	movl	sp@(4),a0
	movl	sp@(8),d0
	movsb	d0,a0@
	rts

_afcww:
	movl	sp@(4),a0
	movl	sp@(8),d0
	movsw	d0,a0@
	rts

_afcwl:
	movl	sp@(4),a0
	movl	sp@(8),d0
	movsl	d0,a0@
	rts

/*
 * kernel putc -- uses delay
 */
	.globl	_k_putc
_k_putc:
	movel	d0,sp@-
	movb	sp@(11),d0
	movsb	d0,0xc0000003
	movl	#0x1800,d0
1:	dbeq	d0,1b
	movel	sp@+,d0
	rts

/*
 * trap to monitor
 */
	.globl  _monTrap
_monTrap:
	jsr	_sysGetReboCount
	addql	#1,d0
	movel	d0,a7@-
	jsr	_sysSetReboCount
	addql	#4,a7
	lea     pc@(1f),a0
	addl    #MEM_DUP_MAP,a0                 /* run in dup space */
	jmp     a0@
1:
	reset
	clrl	d6
	clrl	d7
	jmp     PROM_RESET_REBOOT

/*
 * connect interprocessor interrupt vector
 */
	.globl	_ipi_connect
_ipi_connect:
	movel	#ipi_hnd,INT_VEC_IPI
	rts
ipi_hnd:
	rte
	
/* 
 * flush the onboard data cache
 */
	 .globl	_flush_dcache
_flush_dcache:
	 movl    cac,d0
	 orl     #DCACHE_CLEAR,d0
	 movl    d0,cac
	 rts

        .globl  _flush_dcache_entry
_flush_dcache_entry:
	movl    sp@(4),a0               /* get address */
	movl    sp@(8),d0               /* get count */
	jle     2f
	cmpl    #100,d0
	jle     1f
	movl    cac,d0
	orl     #DCACHE_CLEAR,d0
	movl    d0,cac
	jra     2f

1:  	movl    a0,caa
	movl    cac,d1
	orl     #DCACHE_CE,d1
	movl    d1,cac
	addl    #4,a0
	subl    #4,d0
	jgt     1b
2:  	rts

	 .globl	_dcache_enable
_dcache_enable:
	 movel	a7@(4),d1
	 movl    cac,d0
	 cmpl	#1,d1
	 beq	1f
	 andl	 #~DCACHE_ENABLE,d0
	 bra	 2f
1:	 orl     #(DCACHE_ENABLE|DCACHE_CLEAR|DCACHE_WA),d0
2:	 movl    d0,cac
	 rts

	.globl  _flush_ext_cache_entry
_flush_ext_cache_entry:
	movw    sr,d1
	orw     #7,sr
	movw    _scr,d0
	andl    #SCR_CACHE_ENAB,d0
	movw    d1,sr
	tstl    d0
	jeq     3f
	movl    sp@(4),a0               /* get address */
	movl    sp@(8),d1               /* get count */
	jle     3f
1:  	movsw   d0,a0@
	addl    #16,a0
	subl    #16,d1
	jgt     1b
3:  	rts

        .globl  _flush_ext_cache
_flush_ext_cache:
	movw    _scr,d0
	andl    #SCR_CACHE_ENAB,d0
	tstl    d0			/* don't flush if cache is off */
	jeq     1f
	movw    sr,d1
	movw    d1,a0
	orw     #7,sr
	movw    _scr,d0
	movl    d0,d1
	andl    #~SCR_CACHE_ENAB,d0
	orw     #SCR_CACHE_CLEAR,d0
	movsw   d0,SCR_ADDR
	movsw   d1,SCR_ADDR
	movw    a0,d1
	movw    d1,sr
1:	rts

/* 
 * ts-11 command buffer
 */
	.globl	_ts_cmd_buf
	.globl	_ts_cmd_buf_end
_ts_cmd_buf:
	.space	64
_ts_cmd_buf_end:
