/* sysLib.c - Integrated Solutions 68k30 system dependent library */

/*
DESCRIPTION
This library contains system-dependent routines to set and get the
system clock rate, initialize system hardware, find the top of memory,
set and connect to an auxiliary clock, deal with interrupts, and deal
with the interrupt portion of the USART.
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "memLib.h"
#include "semLib.h"
#include "sysLib.h"
#include "config.h"

/*#define SYSDEBUG	/**/

IMPORT char end;	/* end of system, inserted automatically by ld */
IMPORT int logMsg();

/* globals */

int   sysBus	  = BUS;		/* system bus type (VME_BUS, etc) */
int   sysCpu	  = CPU;		/* system cpu type (MC680x0) */
int   sysCommType = USE_IPI;		/* system inter-processor comm. type */
char  *sysExcMsg  = EXC_MSG_ADRS;	/* catastrophic msg area */
char  *sysBootLine = BOOT_LINE_ADRS;	/* address of boot line */
int   sysProcNum;			/* processor number of this cpu */
int   sysFlags;				/* boot flags */
char  sysBootHost[BOOT_FIELD_LEN];	/* name of host from which we booted */
char  sysBootFile[BOOT_FIELD_LEN];	/* name of file from which we booted */


/* locals */

LOCAL int clkTicksPerSecond;
LOCAL int sysClkRunning		= FALSE;

LOCAL int auxClkTicksPerSecond	= 60;
LOCAL int auxClkRunning		= FALSE;
LOCAL FUNCPTR sysAuxClkRoutine 	= NULL;
LOCAL int sysAuxClkArg		= NULL;

char *memTop 		= NULL;
LOCAL FUNCPTR usartTxInt 	= NULL;
LOCAL FUNCPTR usartRxInt 	= NULL;
LOCAL char imr[4];		/* our copy of the interrupt mask regs */
unsigned short scr;		/* our copy of the control register    */

/*
 * vdma resource allocator for the ISI 68030 board 
 * under vxWorksPlus
 */


#define N_VDMA_REGS		256
#define VDMA_PSIZE		0x1000
#define VDMA_PMASK		(~(long)(VDMA_PSIZE-1))
#define VDMA_REG_BASE		0xc000

#define VDMA_REG_ADDR(i)  	\
		(&vme_short[VDMA_REG_BASE|(((i)&0xff)<<6)|(vdma_clu_no<<1)])
#define SET_VDMA_REG(r,a)	\
		{*((short *)(VDMA_REG_ADDR(r))) =(short)((a)>>12);}
#define VDMA_TO_VME(i)		((((i) & 0xff) << 12) | (vdma_clu_no << 20))
#define VME_TO_VDMA(i)		(((i) >> 12) & 0xff)

/* data about each vdmareg */

struct vdmareg {
    short *addr;		/* address of vdmareg */
    int bsize;			/* size of this vdma register block */
				/* number of vdmaregs allocated in this */
				/* block, or number unallocated if this */
				/* has not yet been allocated */
    int allocated;		/* has this reg been allocated ? */
    int next;			/* next vdmareg in chain */
    int prev;			/* previous vdmareg in chain */
} vdma[N_VDMA_REGS];

SEMAPHORE vdma_sem;
char vdma_clu_no;

/* counter/timer data */

int CTMode[NCTS] = {MODE_NONE, MODE_NONE};
int CTRunning[NCTS] = {0, 0};
int CTRate[NCTS] = {10, 10};
FUNCPTR CTRoutine[NCTS] = {NULL, NULL};
int CTArg[NCTS] = {0, 0};
int CTBlock[NCTS] = {2, 3};

struct CTRegs {
    long acr;
    long opcr;
    long ctu;
    long ctl;
    long start;
    long stop;
} CTRegs[NCTS] = {
    {0xc0000024, 0xc000002d, 0xc0000026, 0xc0000027, 0xc000002e, 0xc000002f},
    {0xc0000034, 0xc000003d, 0xc0000036, 0xc0000037, 0xc000003e, 0xc000003f},
};

extern char vme_short[];
extern short vbmatm_reg[];

/************************************************************************
*
* sysModel - return model name of the system CPU
*
* Use this routine to find the model name of the system CPU.
*
* RETURNS: pointer to string "ISI 68030".
*/
char *sysModel ()
{
    return ("ISI 68030");
}

/**********************************************************************
*
* sysHwInit - initialize hardware
*
* This routine initializes on-board devices
* It is normally called from usrInit (2) in usrConfig (1).
*
* IS30: This routine must be called in supervisor mode on the VME68k30.
*/
VOID sysHwInit ()
{
    unsigned short tmp_scr;

    /* be sure interrupts from the OCTART are OFF */
    afcwb (VME_U_IMRA, 0);
    afcwb (VME_U_IMRB, 0);
    afcwb (VME_U_IMRC, 0);
    afcwb (VME_U_IMRD, 0);


    /* init the vmebus master address translation map */
    sysInitVbmatm ();

    /* set up the  SCR reg: LED's off, interrupts on, flush cache
     */
    tmp_scr = SCR_CACHE_CLEAR | SCR_PROM_DISAB | SCR_VME_SYSFAIL;
    sysScrSet (tmp_scr, ~tmp_scr);

    tmp_scr = SCR_PROM_DISAB | SCR_VME_SYSFAIL | SCR_LED_OFF |
	SCR_INTR_ENAB | SCR_FPU_USE;
    sysScrSet (tmp_scr, ~tmp_scr);

    /* turn off the MMU */
    sysInitMMU ();

    /* test for floating point, if it exists, then enable FPP in SCR 
     */
    if (fppProbe () != OK)
	tmp_scr &= ~SCR_FPU_USE;

    /* if we're cluster 0, turn on vme interrupts, otherwise, leave
     * them off (this must be changed for special configurations
     * that use interrupts on clusters)
     */
    if (!sysProcNumGet ())
	tmp_scr |= SCR_VME_INT_ENAB;

    sysScrSet (tmp_scr, ~tmp_scr);

    sysDataCacheEnab (1); 	/* enable the on-board 64k data cache */

    ipi_connect ();
}

/**********************************************************************
*
* sysNmiTrap
*
* catch nmi interrupts and reboot if we are not processor 0
*/
sysNmiTrap ()
{
    logMsg ("NMI: VME SYSFAIL\n");
    if (sysProcNumGet())
	monTrap ();
    return OK;
}

/**********************************************************************
*
* sysMemTop - get top of memory address
*
* This routine returns the address of the first missing byte of memory.
* It starts probing at 0, then tries to read a byte at every 4K boundary
* until it finds one that can't be read.
*
* This routine must be called in system mode, the first time only.  After
* that, it can be called in user mode as well.
*/

char *sysMemTop ()
{
    char orig;

    if (memTop == NULL) {

	/* Look for the top of memory.  Starting at the first even page
	   boundary after _end, probe each page, then write it and
	   read it back, just to be sure. */

	memTop = (char *) ((((int) &end) + (VME_PAGE_SIZE - 1)) &
			    (~ (VME_PAGE_SIZE - 1)));

	while (1) {
	    char zero = 0, one = 0;

	    if (memProbe (memTop, READ, 1, &one) != OK)
		break;

	    /* the 68030 is readable above local memory, so
	     * we have to check it for writeability also
	     */
	    if (memProbe (memTop, WRITE, 1, &zero) != OK)
		break;

	    orig = *memTop;
	    *memTop = 0x55;
	    if (*memTop != 0x55)
		break;
	    *memTop = orig;

	    if(((int)memTop + VME_PAGE_SIZE) > (int)0x4000000)
		break;

	    memTop += VME_PAGE_SIZE;
	}
    }
    return (memTop);
}

/*******************************************************************************
*
* sysToMonitor - transfer to rom monitor
*
* This routine transfers control to the rom monitor.  It is usually called
* only by the routine reboot, which services control-x, and bus errors at
* interrupt level.  In special circumstances, however, the user may wish
* to introduce a new startType such that a special bootrom facility would be
* enabled.  This calls the assembly routine monTrap () to do all the work.
*
* RETURNS: OK (if we ever continue from the rom monitor).
*/
STATUS sysToMonitor (startType)
int startType;	/* parameter is passed to ROM to tell it how to boot *
		   The possible type are defined in h/sysLib.h */
{
    monTrap ();
    return (OK);	/* in case we ever continue from rom monitor */
}

/**********************************************************************
*
* sysClkConnect - connect routine to system clock interrupt
*
* This routine connects the given function to the system clock interrupt.
* It is normally called from usrRoot (2) in usrConfig (1) to connect
* usrClock (2) to the system clock interrupt.
*/
VOID sysClkConnect (routine, arg)
FUNCPTR routine;	/* routine to be called at each sysclk interrupt */
int arg;		/* argument with which to call routine */
{
    intConnect (INUM_TO_IVEC (INT_VEC_CLOCK), routine, arg);
}

/*******************************************************************************
*
* sysClkDisable - turn off system clock interrupts
*/
VOID sysClkDisable ()
{
    afcwb (SYS_CLK_OPCR, 0);
    tyCoReset (0);

    sysClkRunning = FALSE;
    return(OK);
}

/**********************************************************************
*
* sysClkEnable - enable system clock
*
*/
sysClkEnable()
{
    afcwb (SYS_CLK_ACR, VME_U_ACR_TMR_XTAL);
    afcwb (SYS_CLK_OPCR, VME_U_OPCR_MOPA_CTO);
    afcwb (SYS_CLK_CTU, ((3686400 / (2 * clkTicksPerSecond)) >> 8) & 0xff);
    afcwb (SYS_CLK_CTL, (3686400 / (2 * clkTicksPerSecond)) & 0xff);
    afcrb (SYS_CLK_START);

    /* because of bug in chip, we have to reinitialize serial channel */
    tyCoReset (0);

    sysClkRunning = TRUE;
    return(OK);
}

/**********************************************************************
*
* sysClkRateSet - set rate of system clock
*
* This routine sets the clock rate of the system clock.
* It is normally called by usrRoot (2) in usrConfig (1).
*/
VOID sysClkRateSet (ticksPerSecond)
int ticksPerSecond;	    /* number of clock interrupts per second */
{
    clkTicksPerSecond = ticksPerSecond;

    if (sysClkRunning == TRUE) {	/* restart clock with new rate */
	sysClkDisable ();
	sysClkEnable ();
    }
}

/**********************************************************************
*
* sysClkRateGet - get rate of system clock
*
* This routine is used to find out the system clock speed.
*
* RETURNS: The number of ticks of the system clock per second.
*/
int sysClkRateGet ()
{
    return (clkTicksPerSecond);
}

/**********************************************************************
*
* sysInitMMU - Initialize the 68030 memory management
* 
* This makes sure the top of memory has been calculated,
* then disables the 68030's on-chip MMU, even though the boot rom
* should have already done that.
*/
LOCAL VOID sysInitMMU ()
{
    sysMMUDisable ();	/* Disable  on-chip MMU */
    sysMemTop ();	/**/
}

/**********************************************************************
*
* sysInitVbmatm - Initialize the 68k30 vmebus address modifier map
* 
* This makes sure the vmebus master address translation map is 
* properly initialized for vmebus transfers 
*/
LOCAL VOID sysInitVbmatm ()
{
   afcww(&vbmatm_reg[VBM_SHORT_SLOT], (VBM_AM_SHORT << 6)|VBM_PORT_16);
   afcww(&vbmatm_reg[VBM_STD16_SLOT], (VBM_AM_STD << 6)|VBM_PORT_16);
   afcww(&vbmatm_reg[VBM_EXT_SLOT], (VBM_AM_EXT<<6)|VBM_PORT_16);
   afcww(&vbmatm_reg[VBM_STD32_SLOT], (VBM_AM_STD << 6)|VBM_PORT_32);
   afcww(&vbmatm_reg[4], (VBM_AM_EXT << 6)|VBM_PORT_32|4); /* access to 0x1 */

   return OK;
}

/**************************************************************************
*
* sysAuxClkEnable - enable the aux clock
*
* This sets the counter on chan B of the octart to act as the
* aux clock.  It also enables the aux clock interrupt.  If 
* auxClkTicksPerSecond is 0, then it disables the aux clock.
* This routine does not connect any function to the aux clock.
*
* RETURNS: OK, or ERROR if auxClkTicksPerSecond has a bad value
*/
sysAuxClkEnable()
{
    int ctr;

    /* calculate the divide ratio, and write it to the timer chip.  If
       auxClkTicksPerSecond == 0, write a vl of 0 to stop the clock. */

    ctr = (auxClkTicksPerSecond == 0) ?
	0 : (3686400 / (16 * auxClkTicksPerSecond));

    if ((ctr & 0xffff0000) == 0) {
	/* set up octart to use internal xtal 
	 */
	afcwb (AUX_CLK_ACR ,VME_U_ACR_CTR_XTAL);
	afcwb (AUX_CLK_CTU, (ctr & 0xff00) >> 8);
	afcwb (AUX_CLK_CTL, (ctr & 0x00ff));
	sysImrSet (AUX_CLK_BLK, VME_U_INT_TIMER, 0);
	afcrb (AUX_CLK_START);
	auxClkRunning = TRUE;
    }
    else if (!ctr) {
	/* if ctr == 0, then turn off clock and interrupts 
	 */
	sysImrSet (AUX_CLK_BLK, 0, VME_U_INT_TIMER);
	afcrb (AUX_CLK_STOP);
	auxClkRunning = FALSE;
    }
    else {
	printErr ("Invalid timer rate.  Frequency too small.\n");
	auxClkRunning = FALSE;
	return(ERROR);
    }

    return(OK);
}

/**************************************************************************
*
* sysAuxClkRateSet - set the auxiliary timer frequency
*
* This routine changes the clock rate of the auxiliary clock.
* A rate of 0 stops the clock.  
*
* RETURNS: OK, or ERROR if ticksPerSecond has a bad value
* NOTE: This routine must be called in supervisor mode.
*/
int sysAuxClkRateSet (ticksPerSecond)
int ticksPerSecond;	    /* number of clock interrupts per second;
			     * 0 stops the clock */
{
    int ctr = (3686400 / (16 * ticksPerSecond));

    if ((ctr & 0xffff0000) != 0) {
	printErr ("Invalid timer rate.  Frequency too small.\n");
	return(ERROR);
    }

    auxClkTicksPerSecond = ticksPerSecond;

    if (auxClkRunning) {
        sysAuxClkDisable ();
        sysAuxClkEnable ();
    }
    return (OK);
}

/**************************************************************************
*
* sysAuxClkRateGet - get the auxiliary timer frequency
*
* RETURNS: The number of ticks of the auxiliary clock per second.
*/
int sysAuxClkRateGet ()
{
    return (auxClkTicksPerSecond);
}

/**************************************************************************
*
* sysAuxClkConnect - connect a routine to the auxiliary clock interrupt
*
* This routine connects a user routine to the auxiliary clock interrupt
*
* RETURNS: OK, to indicate that there is an auxclk on this CPU board.
*/
STATUS sysAuxClkConnect (routine, parm)
FUNCPTR routine;	/* routine to be called at each auxclk interrupt */
int parm;		/* argument with which to call routine */
{
    sysAuxClkRoutine = routine;
    sysAuxClkArg = parm;

    return (OK);
}

/**************************************************************************
*
* sysAuxClkDisconnect - clear the auxiliary clock routine.
* 
* This routine disables the auxiliary clock interrupt, stops the timer,
* and disconnects the routine currently connected to the auxiliary clock
* interrupt.
*/
VOID sysAuxClkDisconnect ()
{
    sysAuxClkDisable();

    /* connect dummy routine, just in case */

    sysAuxClkConnect (logMsg, (int) "auxiliary clock interrupt\n");
}

/*******************************************************************************
*
* sysAuxClkDisable - turn off auxiliary clock interrupts
*/
VOID sysAuxClkDisable ()
{
    /* turn off interrupt and stop clock 
     */
    sysImrSet (AUX_CLK_BLK, 0, VME_U_INT_TIMER);
    afcrb (AUX_CLK_STOP);
    auxClkRunning = FALSE;
}

/*******************************************************************************
*
* sysIpiConnect - connect routine to the interprocessor interrupt
*
* This routine connects the given function to the interprocessor interrupt.
* For the 68K30, this just uses intConnect
* It is expected that this will be used only with the vb driver, so
* there is no corresponding disconnect function.
*
* SEE ALSO: intConnect (2)
*/
STATUS sysIpiConnect(routine, arg)
FUNCPTR routine;
int arg;
{
    intConnect ((FUNCPTR *) 0x6C, routine, arg);
    return;
}

/************************************************************************
*
* sysLocalToBusAdrs - convert local address to bus address
*
* Given a local memory address, this routine returns the VME address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unable to get to that local address from the bus.
*
* ARGSUSED
*/
STATUS sysLocalToBusAdrs (adrsSpace, localAdrs, pBusAdrs)
int adrsSpace;		/* bus address space in which busAdrs resides;
			 * use address modifier codes as defined in vme.h,
			 * such as VME_AM_STD_SUP_DATA */
char *localAdrs;	/* local address to convert */
char **pBusAdrs;	/* where to return bus address */
{
    switch (adrsSpace) {
    case VME_AM_SUP_SHORT_IO:
    case VME_AM_USR_SHORT_IO:
	*pBusAdrs = (char *) (NULL);
	return (ERROR);

    /* the address returned for standard address space is the
     * one that would be used if the 68k30 used direct dma.
     * Since virtual dma is always used in standard address
     * space, vdma registers must be allocated to access any
     * addresses using setvdma ()
     */
    case VME_AM_STD_SUP_ASCENDING:
    case VME_AM_STD_SUP_PGM:
    case VME_AM_STD_SUP_DATA:
    case VME_AM_STD_USR_ASCENDING:
    case VME_AM_STD_USR_PGM:
    case VME_AM_STD_USR_DATA:
	*pBusAdrs = (char *) (sysProcNumGet()<<20 | (int) localAdrs);
	return (OK);

    case VME_AM_EXT_SUP_ASCENDING:
    case VME_AM_EXT_SUP_PGM:
    case VME_AM_EXT_SUP_DATA:
    case VME_AM_EXT_USR_ASCENDING:
    case VME_AM_EXT_USR_PGM:
    case VME_AM_EXT_USR_DATA:
	*pBusAdrs = (char *) (sysProcNumGet()<<28 | (int) localAdrs);
	return (OK);

    default:
	return (ERROR);
    }
}

/*************************************************************************
*
* sysBusToLocalAdrs - convert bus address to local address
*
* Given a VME memory address, this routine returns the local address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unknown address space.
*/
STATUS sysBusToLocalAdrs (adrsSpace, busAdrs, pLocalAdrs)
int adrsSpace;		/* bus address space in which busAdrs resides;
			 * use address modifier codes as defined in vme.h,
			 * such as VME_AM_STD_SUP_DATA */
char *busAdrs;		/* bus address to convert */
char **pLocalAdrs;	/* where to return local address */

{
    switch (adrsSpace) {
    case VME_AM_SUP_SHORT_IO:
    case VME_AM_USR_SHORT_IO:
	*pLocalAdrs = (char *) (VME_SHORTIO | (int) busAdrs);
	return (OK);

    case VME_AM_STD_SUP_ASCENDING:
    case VME_AM_STD_SUP_PGM:
    case VME_AM_STD_SUP_DATA:
    case VME_AM_STD_USR_ASCENDING:
    case VME_AM_STD_USR_PGM:
    case VME_AM_STD_USR_DATA:
	*pLocalAdrs = (char *) (VME_STANDARD | (int) busAdrs);
	return (OK);

    case VME_AM_EXT_SUP_ASCENDING:
    case VME_AM_EXT_SUP_PGM:
    case VME_AM_EXT_SUP_DATA:
    case VME_AM_EXT_USR_ASCENDING:
    case VME_AM_EXT_USR_PGM:
    case VME_AM_EXT_USR_DATA:
	*pLocalAdrs = (char *) (VME_EXTENDED | (int) busAdrs);
	return (OK);

    default:
	return (ERROR);
    }
}

/*************************************************************************
*
* sysIntEnable - enable specified interrupt level
*
* IS30: Null routine for this CPU.
*/
STATUS sysIntEnable (intLevel)
int intLevel;	/* interrupt level to enable */
{
    return (OK);
}

/*************************************************************************
*
* sysIntDisable - disable specified interrupt level
*
* IS30: Null routine for this CPU.
*/
STATUS sysIntDisable (intLevel)
int intLevel;	/* interrupt level to disable */
{
    return (OK);
}

/*************************************************************************
*
* sysBusIntAck - acknowledge specified interrupt
*
* IS30: Null routine for this CPU.
*/
int sysBusIntAck (intLevel)
int intLevel;		/* interrupt level to acknowledge */
{
    return (NULL);
}

/**********************************************************************
*
* sysImrSet - set and clear bits in the USART's interrupt mask register
*
* This routine sets and clears bits in the usart's IMR.  It may be called
* either in user or supervisor state.
*
* This routine sets and clears bits in a local copy of the IMR, then
* writes that local copy to the usart.  This means that all changes to
* the IMR must go through this routine.  Otherwise, any direct changes
* to the IMR would be lost the next time this routine is called.
*
* NOTE: This routine must be called in supervisor mode.
*/
sysImrSet (block, setBits, clearBits)
int block;		/* which register block the IMR is in */
char setBits;	/* which bits to set in the IMR */
char clearBits;	/* which bits to clear in the IMR */
{
    unsigned long addr;

    imr[block] = (imr[block] | setBits) & (~clearBits);
    addr = (unsigned long)(OCTART_ADDR + (block << 4) + 5);
    afcwb (addr, imr[block]);
}

/**********************************************************************
*
* sysScrSet - set and clear bits in the 68k30's control register
*
* This routine sets and clears bits in a local copy of the scr, then
* makes the modifications to the real scr.  The scr can't be read, so 
* this is used to keep track of any changes to it.
*/
sysScrSet (setBits, clearBits)
unsigned short setBits;		/* which bits to set in the IMR */
unsigned short clearBits;	/* which bits to clear in the IMR */
{
    scr = (scr | setBits) & ~clearBits;
    afcww (SCR_ADDR, scr);
}

/**********************************************************************
*
* sysUsartInt - interrupt level processing for VME68k20 usart
*
* This routine handles an interrupt from the on-board usart.  The interrupt
* is decoded by checking the uart's interrupt status reg, and the appropriate 
* routine invoked.  
*
* This routine is located here because we use the extra timer on the usart
* chip as the auxiliary timer, which is handled here.
*/
LOCAL VOID sysUsartInt ()
{
    register int block;
    register char blk_stat;

    for (block = 0; block < N_USART_BLOCKS; block++) {
	blk_stat = afcrb (OCTART_ADDR + (block << 4) + 5);

	/* transmitter channel A interrupt.  call usart routine, if
	   there is one, else issue an EOI 
	 */
	if (blk_stat & VME_U_INT_TXA) {
	    if (usartTxInt == NULL)
		sysImrSet (block, 0, VME_U_INT_TXA); 	/* EOI */
	    else
		(*usartTxInt) (2 * block);
	}

	/* transmitter channel B interrupt.  call usart routine, if
	   there is one, else issue an EOI 
	 */
	if (blk_stat & VME_U_INT_TXB) {
	    if (usartTxInt == NULL)
		sysImrSet (block, 0, VME_U_INT_TXB); 	/* EOI */
	    else
		(*usartTxInt) (2 * block + 1);
	}

	if ((blk_stat & VME_U_INT_RXA) && (usartRxInt != NULL))
	    (*usartRxInt) (2 * block);		/* receiver channel A */

	if ((blk_stat & VME_U_INT_RXB) && (usartRxInt != NULL))
	    (*usartRxInt) (2 * block + 1);	/* receiver channel B */

	/* aux-clk interrupt, reset the timer count and restart
	 * with interrupts enabled 
	 */
	if (blk_stat & VME_U_INT_TIMER)	{	/* on-chip timer */
	    if (block == AUX_CLK_BLK) {
		unsigned short cnt = 3686400 / (16 * auxClkTicksPerSecond);

		afcrb (AUX_CLK_STOP);	/* EOI on the timer */
		afcwb (AUX_CLK_CTU, (cnt & 0xff00) >> 8);
		afcwb (AUX_CLK_CTL, (cnt & 0x00ff));
		afcrb (AUX_CLK_START);	

		if (sysAuxClkRoutine != NULL)
		    (*sysAuxClkRoutine) (sysAuxClkArg);
	    }
	    else if (block == CTBlock[CT1] || block == CTBlock[CT2]) {
		int ctno;
		unsigned short cnt;

		ctno = (block == CTBlock[CT1] ? CT1 : CT2);
		cnt = 3686400 / (16 * CTRate[ctno]);

		afcrb (CTRegs[ctno].stop);	/* EOI on the timer */
		afcwb (CTRegs[ctno].ctu, (cnt & 0xff00) >> 8);
		afcwb (CTRegs[ctno].ctl, (cnt & 0x00ff));
		afcrb (CTRegs[ctno].start);	

		if (CTRoutine[ctno] != NULL && CTRunning[ctno] 
		    && CTMode[ctno] == MODE_COUNTER)
		    (*CTRoutine[ctno]) (CTArg[ctno]);
	    }
	}
    }
}

/************************************************************************
*
* sysUsartConnect - set interrupt routines for the usart
*
* This routine connects the serial driver interrupt routines to the
* interrupt routine.  It should be called once, from the serial driver.
*/
VOID sysUsartConnect (recvRoutine, xmitRoutine)
FUNCPTR recvRoutine;	/* receive routine */
FUNCPTR xmitRoutine;	/* transmit routine */
{
    intConnect (INUM_TO_IVEC (INT_VEC_CONSOLE), sysUsartInt, NULL);
    usartTxInt = xmitRoutine;
    usartRxInt = recvRoutine;
}

/*************************************************************************
*
* sysProcNumSet - set processor number
*
* Set the processor number for this CPU.  Processor numbers should be
* unique on a single backplane.
* For the is68k30 board, processors must have a unique cluster number,
* so just set the processor number to the cluster number. This routine
* ignores the input parameter.
*/
VOID sysProcNumSet (procNum)
int procNum;		/* processor number */
{
    procNum = afcrb (VME_U_IP0);
    sysProcNum = (procNum >> 4) & 0x3;
}

/*************************************************************************
*
* sysProcNumGet - set processor number
*
* Get the processor number for this CPU.  Reads the input port and
* returns the CPU cluster number.
*/
sysProcNumGet ()
{
    int procNum;
	
    procNum = afcrb (VME_U_IP0);
    return ((procNum >> 4) & 0x3);
}

/************************************************************************
*
* sysBusTas - test and set across VME bus
*
* This routine does a 680x0 test-and-set instruction across the backplane.
*
* RETURNS: TRUE (successful set) or FALSE (failure).
*/
BOOL sysBusTas (addr)
char *addr;		/* address to be tested and set */
{
    return (vxTas (addr));
}

/************************************************************************
*
* sysDataCacheEnab - enable or disable the on-board external cache
*
* This routine enables (disables) the external cache on the 68k30 board.
* It also enables (disables) the 68030's on-chip data cache.
*
* Returns 1 if cache was previously enabled, 0 if cache was previously
* disabled.
*/

sysDataCacheEnab (onoff)
int onoff;
{
    int s=intLock();
    unsigned short oldscr = scr;

    if (onoff) {
	dcache_enable (1);
	sysScrSet (SCR_CACHE_CLEAR, SCR_CACHE_ENAB);
	sysScrSet (SCR_CACHE_ENAB, SCR_CACHE_CLEAR);
    }
    else {
	dcache_enable (0);
	sysScrSet (0, SCR_CACHE_ENAB);
    }

    intUnlock (s);
    return (oldscr & SCR_CACHE_ENAB ? 1 : 0);
}

/************************************************************************
*
* sysDataCacheFlush - flush both of the 68k30 data caches
*
* This routine flushes both the 68030 on-chip data cache
* and the on-board 64K data cache.
*/

sysDataCacheFlush ()
{
    flush_dcache ();		/* flush the on-chip dcache of the 68030 */

    sysExtCacheFlush ();	/* flush the on-board 64K data cache */
    return (OK);
}

/************************************************************************
*
* sysExtCacheFlush - flush the on-board external cache
*
* This routine flushes the external cache on the 68k30 board.
*/

sysExtCacheFlush ()
{
    flush_ext_cache ();
    return (OK);
}

/************************************************************************
*
* sysCacheFlushEntry - flush a cache entry 
*
* This routine flushes the requested cache entries on the 68k30 board
* and the 68030 on-chip data cache.
*
* addr is the address of the entry to flush
* num is the number of bytes to flush
*/

sysDataCacheFlushEntry (addr, num)
char *addr;
int num;
{
    if (num) {
	flush_dcache ();
	if (num > CACHE_FLUSH_THRESH)
	    flush_ext_cache_entry (addr, num /16 + (num % 16 ? 1:0));
	else
	    flush_ext_cache ();
    }
    return (OK);
}


/*
 * vdma resource allocator for the ISI 68030 board 
 * under uniworks
 */

/************************************************************************/
/* 
 * sysSetVdma
 *
 * set the vdma registers for a buffer at
 * location buff of size nbytes
 * if there are enough vdma registers to service
 * this request, then the caller will get a vdma
 * window into the buffer that was asked for.
 *
 * returns the upper 20 bits of the address which
 * must be accessed to get to the block
 */
char *sysSetVdma (buff, nbytes)
char *buff;
int nbytes;
{
    register char *lub;
    register int regno, needed, cnt;
    register struct vdmareg *vr;

/*    logMsg ("Set vdma, buff = 0x%x\n", buff); /**/
    lub = buff + nbytes - 1;
    needed = ((((int)lub & VDMA_PMASK) - ((int)buff & VDMA_PMASK)) >> 12) + 1;

    semTake (&vdma_sem);

    /* look for a register block that is large enough 
     */
    if ((regno = getreg (needed)) < 0) {
	logMsg ("vdma: out of vdma registers: needed = %d, regno = %d\n", 
	    needed, regno);
    } 

    /* we found a large enough block, we have to allocate
     * the registers.
     */
    else {
	vr = &vdma[regno];
	if (needed > vr->bsize) {		/* sanity check */
	    logMsg ("vdma: allocated bad blocksize, regno = %d\n", regno);
	    regno = -1;
	} 
	else if (needed == vr->bsize) {		/* requested = bsize */
	    setblock (regno, VDMA_PMASK & (int)buff); /* set the regs */
	}
	else {					/* requested < bsize */
	    breakblock (regno, needed);		/* split the reg block */
	    setblock (regno, VDMA_PMASK & (int)buff); /* set the regs */
	}
    }
    semGive (&vdma_sem);

    return ((char *)(VDMA_TO_VME(regno) | ((int)buff & ~VDMA_PMASK)));
}

/************************************************************************/
/* 
 * sysInitVdma
 *
 * initialize vdma register structure
 */
sysInitVdma ()
{
    register int i;

    semInit (&vdma_sem);
    semGive (&vdma_sem);

    /* initialize addr to point to each vdma register in shortio */
    /* the only 'first register' is 0, set after the loop	 */
    /* no registers are yet allocated				 */

    vdma_clu_no = sysProcNumGet () & 0x7;

    for (i = 0; i < N_VDMA_REGS; i++) {
	vdma[i].addr  = (short *)VDMA_REG_ADDR(i);
	vdma[i].bsize  = 0;
	vdma[i].allocated  = 0;
	vdma[i].next  = 0;
	vdma[i].prev  = 0;

	/* set each vdma register above the top of local memory
	 */
	SET_VDMA_REG(i, 0x0ffff000);
    }

    vdma[0].bsize  = N_VDMA_REGS;

    /* If this is cluster 0
     * open a window from VME standard address 0 to 
     * CPU 0's physical memory address 0 so vme_std
     * can be used to probe for vb memory
     */
    if (!sysProcNumGet()) 
	if (sysSetVdma(0, 1) != 0)
	    logMsg ("Warning: VME standard address window not at phys[0]\n");

    return (0);
}

/************************************************************************/
/* 
 * sysGiveVdma
 *
 * give up vdma registers taken earlier that begin
 * with register at address vmeoff. coalesces register blocks
 * that are free and contiguous
 *
 * returns 0 or -1 for error
 */
sysGiveVdma (vmeoff)
int vmeoff;
{
    register int cnt, reg;
    register int regno;

/*    logMsg ("Give vdma, vmeoff = 0x%x\n", vmeoff); /**/
    regno = VME_TO_VDMA(vmeoff);
    /* see if the register block has really been allocated */
    if (!vdma[regno].allocated) {
	logMsg ("vdma: can't deallocate %d\n", regno);
	return (-1);
    }
    semTake (&vdma_sem);

    /* put the block at regno back into free space */
    for (cnt = 0; cnt < vdma[regno].bsize; cnt++)
	vdma[regno+cnt].allocated = 0;

    /* coalesce the block with the next block if the
     * next one is not allocated
     */
    reg = vdma[regno].next;
    if (!vdma[reg].allocated) {
	vdma[regno].next = vdma[reg].next;
	vdma[regno].bsize += vdma[reg].bsize;
	vdma[vdma[regno].next].prev = regno;
	vdma[reg].next = 0;		/* just for safety */
	vdma[reg].prev = 0;
	vdma[reg].bsize = 0;
    }

    /* coalesce the block with the previous block if the
     * previous one is not allocated
     */
    reg = vdma[regno].prev;
    if (!vdma[reg].allocated) {
	vdma[reg].next = vdma[regno].next;
	vdma[reg].bsize += vdma[regno].bsize;
	vdma[vdma[reg].next].prev = reg;
	vdma[regno].next = 0;		/* just for safety */
	vdma[regno].prev = 0;
	vdma[regno].bsize = 0;
    }

    semGive (&vdma_sem);
    return (0);
}

/************************************************************************/
/* 
 * getreg
 *
 * returns the number of the first register of the block
 * containing enough registers
 * or -1 if an error occurred or there are not enough regs
 */
getreg (nregs)
int nregs;
{
    register int regno;
    register struct vdmareg *vr;

    regno = 0;
    do {
	vr = &vdma[regno];
	if ((!vr->allocated) && (vr->bsize >= nregs))
	    return (regno);
	regno = vdma[regno].next;
    } while (regno != 0);

    return (-1);
}

/************************************************************************/
/* 
 * breakblock
 *
 * break up a register block at regno into two
 * register blocks, the first of size 'needed'
 *
 * returns 0 or -1 for error
 */
breakblock (regno, needed)
int regno, needed;
{
    register int bhalf;

    /* check for needed > bsize, or needed = bsize */
    if (needed > vdma[regno].bsize) {
	logMsg ("vdma: cannot break vdma register block\n");
	return (-1);
    } 
    else if (needed == vdma[regno].bsize)
	return (0);

    /* if we've reached this point, some real work now needs to 
     * be done
     */

    /* set up the lower half of the register block as
     * its own register block
     */
    bhalf = regno + needed;		/* bhalf is next block now */
    vdma[bhalf].bsize = vdma[regno].bsize - needed;
    vdma[bhalf].allocated = 0;		/* just for safety */
    vdma[bhalf].next = vdma[regno].next;
    vdma[bhalf].prev = regno;

    /* reset the upper half of the register block with
     * its new size
     */
    vdma[regno].bsize = needed;
    vdma[regno].next = bhalf;
    return (0);
}

/************************************************************************/
/* 
 * setblock
 *
 * set the standard vme window for register block
 * with starting register 'regno' to the address 
 * specified address is on a page boundary
 *
 * returns 0 or -1 for error
 */
setblock (regno, addr)
int regno;
char *addr;
{
    register struct vdmareg *vr;
    register int cnt, maxval;

    /* check for register too high */
    if ((maxval = vdma[regno].bsize) >= N_VDMA_REGS) {
	logMsg ("vdma: attempt to set register %d\n", maxval);
	return (-1);
    }

    /* mark every register in the block as allocated
     * and set the register to the proper page
     */
    for (cnt = 0; cnt < maxval; cnt++) {
	vdma[regno+cnt].allocated = 1;
	SET_VDMA_REG((regno+cnt), ((cnt * VDMA_PSIZE) + ((int)addr & VDMA_PMASK)));
    }

    return (0);
}


/*
 * routines for the manipulation of the extra two timer/counters on
 * the 68k30 board
 */


/* setup counter/timer mode before starting C/T */
/* the C/T must not be running to change its mode */

sysCTModeSet (ctno, mode)
int ctno, mode;
{
    if ((ctno >= NCTS) || CTRunning[ctno] || mode > MAX_MODES)
	return ERROR;

    CTMode[ctno] = mode;
    return OK;
}

/* return current mode of the counter/timer 
 */
sysCTModeGet (ctno)
int ctno;
{
    if (ctno >= NCTS)
	return ERROR;

    return (CTMode[ctno]);
}


/* routines to use C/T as MODE_COUNTER 
 * These routines operate like the aux clock 
 */

/* set rate of counter
 */
sysCounterRateSet (ctno, ticsPerSec)
int ctno, ticsPerSec;
{
    int ctr = (3686400 / (16 * ticsPerSec));

    if ((ctno >= NCTS) || CTMode[ctno] != MODE_COUNTER)
	return ERROR;

    if ((ctr & 0xffff0000) != 0) {
	printErr ("Invalid timer rate.  Frequency too small.\n");
	return(ERROR);
    }

    CTRate[ctno] = ticsPerSec;

    if (CTRunning[ctno]) {
        sysCounterDisable (ctno);
        sysCounterEnable (ctno);
    }
    return (OK);
}

/* returns rate of Counter 
 */
sysCounterRateGet (ctno)
int ctno;
{
    if ((ctno >= NCTS) || CTMode[ctno] != MODE_COUNTER)
	return ERROR;

    return (CTRate[ctno]);
}

/* connect a routine and argument to the counter interrupt 
 */
sysCounterConnect (ctno, routine, arg)
int ctno, arg;
FUNCPTR routine;
{
    if (((ctno >= NCTS) || CTRunning[ctno]) || CTMode[ctno] != MODE_COUNTER)
	return ERROR;

    CTRoutine[ctno] = routine;
    CTArg[ctno] = arg;

    return (OK);
}

/* disconnect a routine and argument from the counter interrupt 
 */
sysCounterDisconnect (ctno)
int ctno;
{
    if ((ctno >= NCTS) || CTMode[ctno] != MODE_COUNTER)
	return ERROR;

    sysCounterDisable(ctno);

    /* connect dummy routine, just in case */

    sysCounterConnect (logMsg, (int) "C/T interrupt\n");
    return (OK);
}

sysCounterEnable (ctno)
int ctno;
{
    int ctr;

    if ((ctno >= NCTS) || CTMode[ctno] != MODE_COUNTER)
	return ERROR;

    /* calculate the divide ratio, and write it to the timer chip.  If
       CTRate == 0, write a vl of 0 to stop the clock. */

    ctr = (CTRate[ctno] == 0) ?
	0 : (3686400 / (16 * CTRate[ctno]));

    if ((ctr & 0xffff0000) == 0) {
	/* set up octart to use internal xtal 
	 */
	afcwb (CTRegs[ctno].acr ,VME_U_ACR_CTR_XTAL);
	afcwb (CTRegs[ctno].ctu, (ctr & 0xff00) >> 8);
	afcwb (CTRegs[ctno].ctl, (ctr & 0x00ff));
	sysImrSet (CTBlock[ctno], VME_U_INT_TIMER, 0);
	afcrb (CTRegs[ctno].start);
	CTRunning[ctno] = TRUE;
    }
    else if (!ctr) {
	/* if ctr == 0, then turn off clock and interrupts 
	 */
	sysImrSet (CTBlock[ctno], 0, VME_U_INT_TIMER);
	afcrb (CTRegs[ctno].stop);
	CTRunning[ctno] = FALSE;
    }
    else {
	printErr ("Invalid timer rate.  Frequency too small.\n");
	CTRunning[ctno] = FALSE;
	return(ERROR);
    }

    return(OK);
}

/* disable the counter 
 */
sysCounterDisable (ctno)
int ctno;
{
    if ((ctno >= NCTS) || CTMode[ctno] != MODE_COUNTER)
	return ERROR;

    /* turn off interrupt and stop clock 
     */
    sysImrSet (CTBlock[ctno], 0, VME_U_INT_TIMER);
    afcrb (CTRegs[ctno].stop);
    CTRunning[ctno] = FALSE;
    return OK;
}

/* read the current value of the counter 
 * before it times out
 *
 * The data sheet suggests stopping the counter before reading
 * it in case a carry is propagating between the two byte-wide registers.
 * However, at this high clock rate, stopping and restarting the counter
 * causes the counter to lose accuracy, so it is not stopped for a read.
 */
sysCounterRead (ctno)
int ctno;
{
    int value;

    if ((ctno >= NCTS) || CTMode[ctno] != MODE_COUNTER)
	return ERROR;

/*  afcrb (CTRegs[ctno].stop); /**/
    value = afcrb (CTRegs[ctno].ctu) << 8;
    value += afcrb (CTRegs[ctno].ctl) & 0xff;
/*  if (CTRunning[ctno] == TRUE)
	afcrb (CTRegs[ctno].start); /**/
    return (value & 0xffff);
}


/* 
 * routines to use C/T as MODE_TIMER 
 * 
 * These functions generate a square wave output on output
 * pin 2 or 3 of the OCTART chip (MPO).  As they stand, they don't have 
 * much use, but with a few customizations to some of the other C/T 
 * routines, these timer functions can be used to change clock rates 
 * or cascade counters.
 */


/* set the rate of the timer
 */
sysTimerRateSet (ctno, cyclesPerSec)
int ctno, cyclesPerSec;
{
    if ((ctno >= NCTS) || CTMode[ctno] != MODE_TIMER)
	return ERROR;

    CTRate[ctno] = cyclesPerSec;

    if (CTRunning[ctno] == TRUE) {	/* restart clock with new rate */
	sysTimerStop (ctno);
	sysTimerStart (ctno);
    }
}

/* get the rate of the timer
 */
sysTimerRateGet (ctno)
int ctno;
{
    if ((ctno >= NCTS) || CTMode[ctno] != MODE_TIMER)
	return ERROR;

    return (CTRate[ctno]);
}

/* start timer
 */
sysTimerStart (ctno)
int ctno;
{
    if ((ctno >= NCTS) || CTMode[ctno] != MODE_TIMER)
	return ERROR;

    afcwb (CTRegs[ctno].acr, VME_U_ACR_TMR_XTAL);
    afcwb (CTRegs[ctno].opcr, VME_U_OPCR_MOPA_CTO);
    afcwb (CTRegs[ctno].ctu, ((3686400 / (2 * CTRate[ctno])) >> 8) & 0xff);
    afcwb (CTRegs[ctno].ctl, (3686400 / (2 * CTRate[ctno])) & 0xff);
    afcrb (CTRegs[ctno].start);

    /* because of bug in chip, we have to reinitialize serial channel */
    tyCoReset (ctno == CT1 ? 2 : 3);

    CTRunning[ctno] = TRUE;
    return(OK);
}

/* stop timer
 */
sysTimerStop (ctno)
int ctno;
{
    if ((ctno >= NCTS) || CTMode[ctno] != MODE_TIMER)
	return ERROR;

    afcwb (CTRegs[ctno].opcr, 0);
    tyCoReset (ctno == CT1 ? 2 : 3);

    CTRunning[ctno] = FALSE;
    return(OK);
}


/* routines to use C/T as MODE_FREERUN */

/* start the free running counter
 */
sysFreerunStart (ctno)
int ctno;
{
    if ((ctno >= NCTS) || CTMode[ctno] != MODE_FREERUN)
	return ERROR;

    afcwb (CTRegs[ctno].acr ,VME_U_ACR_CTR_XTAL);
    afcwb (CTRegs[ctno].ctu,  0xff);
    afcwb (CTRegs[ctno].ctl,  0xff);
    afcrb (CTRegs[ctno].start);
    CTRunning[ctno] = TRUE;

    return(OK);
}

/* stop the free running counter
 */
sysFreerunStop (ctno)
int ctno;
{
    if ((ctno >= NCTS) || CTMode[ctno] != MODE_FREERUN)
	return ERROR;

    /* turn off interrupt and stop clock 
     */
    afcrb (CTRegs[ctno].stop);
    CTRunning[ctno] = FALSE;
    return OK;
}

/* read the free running counter
 *
 * The data sheet suggests stopping the counter before reading
 * it in case a carry is propagating between the two byte-wide registers.
 * However, at this high clock rate, stopping and restarting the counter
 * causes the counter to lose accuracy, so it is not stopped for a read.
 */
sysFreerunRead (ctno)
int ctno;
{
    int value;

    if ((ctno >= NCTS) || CTMode[ctno] != MODE_FREERUN)
	return ERROR;

/*  afcrb (CTRegs[ctno].stop); /**/
    value = afcrb (CTRegs[ctno].ctu) << 8;
    value += afcrb (CTRegs[ctno].ctl) & 0xff;
/*  if (CTRunning[ctno] == TRUE)
	afcrb (CTRegs[ctno].start); /**/
    return (value & 0xffff);
}

/* 
 * non-volatile ram manipulation routines
 * access real-time clock (battery backup), bootstring,
 * and reset counter.
 */

char *day_string[] =  {
	"",
	"Sunday",
	"Monday",
	"Tuesday",
	"Wednesday",
	"Thursday",
	"Friday",
	"Saturday",
};

char *month_string[] = {
	"",
	"January",
	"Feburary",
	"March",
	"April",
	"May",
	"June",
	"July",
	"August",
	"September",
	"October",
	"November",
	"December",
};

sysSetDate(syear, smonth, sdate, sday, shour, sminute, ssecond)
int syear, smonth, sdate, sday, shour, sminute, ssecond;
{
    unsigned char year, month, date, day, hour, minute, second;

    if ((syear > 99) || (smonth < 1 || smonth > 12)
	|| (sdate < 1 || sdate > 31) || (sday < 1 || sday > 7)
	|| (shour > 23) || (sminute > 59) || (ssecond > 59))
	    return ERROR;

    year = binary_to_bcd(syear);
    month = binary_to_bcd(smonth);
    date = binary_to_bcd(sdate);
    day = binary_to_bcd(sday);
    hour = binary_to_bcd(shour);
    minute = binary_to_bcd(sminute);
    second = binary_to_bcd(ssecond);

    afcwb(NVRAM_CONTROL, NVRAM_WRITE);
    afcwb(NVRAM_SECOND, 0x00);
    afcwb(NVRAM_HOUR, 0x80);
    afcwb(NVRAM_CONTROL, 0);
    {int i = 300000; while (i--);}
    afcwb(NVRAM_CONTROL, NVRAM_WRITE);
    afcwb(NVRAM_HOUR, 0x00);
    afcwb(NVRAM_YEAR, year);
    afcwb(NVRAM_MONTH, month);
    afcwb(NVRAM_DATE, date);
    afcwb(NVRAM_DAY, day);
    afcwb(NVRAM_HOUR, hour);
    afcwb(NVRAM_MINUTE, minute);
    afcwb(NVRAM_SECOND, second);
    afcwb(NVRAM_CONTROL, 0);
    return OK;
}

sysGetDate (year, month, date, day, hour, minute, second)
int *year, *month, *date, *day, *hour, *minute, *second;
{

    afcwb(NVRAM_CONTROL, NVRAM_READ);
    *year = bcd_to_binary(afcrb(NVRAM_YEAR));
    *month = bcd_to_binary(afcrb(NVRAM_MONTH));
    *date = bcd_to_binary(afcrb(NVRAM_DATE));
    *day = bcd_to_binary(afcrb(NVRAM_DAY));
    *hour = bcd_to_binary(afcrb(NVRAM_HOUR));
    *minute = bcd_to_binary(afcrb(NVRAM_MINUTE));
    *second = bcd_to_binary(afcrb(NVRAM_SECOND));
    afcwb(NVRAM_CONTROL, 0);
    return OK;
}

sysDispDate()
{
    unsigned char year, month, date, day, hour, minute, second;

    afcwb(NVRAM_CONTROL, NVRAM_READ);
    year = bcd_to_binary(afcrb(NVRAM_YEAR));
    month = bcd_to_binary(afcrb(NVRAM_MONTH));
    date = bcd_to_binary(afcrb(NVRAM_DATE));
    day = bcd_to_binary(afcrb(NVRAM_DAY));
    hour = bcd_to_binary(afcrb(NVRAM_HOUR));
    minute = bcd_to_binary(afcrb(NVRAM_MINUTE));
    second = bcd_to_binary(afcrb(NVRAM_SECOND));
    afcwb(NVRAM_CONTROL, 0);
    printf("  %s %s %d %2d   %2d:%2d:%2d\n",
	day_string[day],
	month_string[month],
	date, year, hour, minute, second);
    return OK;
}

sysGetReboCount()
{
    return (afcrl(NVRAM_RESET_COUNTER));
}

sysSetReboCount(num)
long num;
{
    afcwl(NVRAM_RESET_COUNTER, num);
    return OK;
}

sysDispReboCount()
{
    long num;

    num = afcrl(NVRAM_RESET_COUNTER);
    printf("  Reboot counter = %d\n", num);
    return OK;
}

sysSetBootArg(buf)
char *buf;
{
    register i;

    for (i = 0 ; i < (NVRAM_BOOT_ARGSIZE - 1) && buf[i] ; i++)
	afcwb(NVRAM_BOOT_ARG + i, buf[i]);
	    
    afcwb(NVRAM_BOOT_ARG + i, 0);
    return OK;
}

sysDispBootArg()
{
    register i;
    char c;

    printf("  Boot arg = ");
    i = NVRAM_BOOT_ARG;
    do {
	c = afcrb(i++);
	printf("%c", c);
    } while(c);
    printf("\n");
    return OK;
}
	
/* 
 * stop the real-time clock oscillator
 */
sysOscStop()
{
    afcwb(NVRAM_SECOND, 0x80);
    return OK;
}

/* 
 * read a buffer from non-volatile ram
 */
sysReadNvram (offset, buf, nbytes)
int offset;
char *buf;
int nbytes;
{
    register i;

    if ((offset >= NVRAM_USER_MAX)||((offset + nbytes - 1) >= NVRAM_USER_MAX))
	return ERROR;

    for (i = 0; i < nbytes; i++)
	buf[i] = afcrb(NVRAM_ADDR + offset + i);

    return OK;
}


/* 
 * write a buffer to non-volatile ram
 */
sysWriteNvram (offset, buf, nbytes)
int offset;
char *buf;
int nbytes;
{
    register i;

    if ((offset >= NVRAM_USER_MAX)||((offset + nbytes - 1) >= NVRAM_USER_MAX))
	return ERROR;

    for (i = 0; i < nbytes; i++)
	afcwb(NVRAM_ADDR + offset + i, buf[i]);

    return OK;
}


binary_to_bcd(i)
{
    return((i / 10) << 4 | (i % 10));
}
	
bcd_to_binary(i)
unsigned char i;
{
    return((i >> 4) * 10 + (i & 0x0f));
}

/* 
 * substitute for vxMemProbe because of the write buffers on the
 * 68K30 board 
 */

int
memProbe (adrs, mode, length, pVal)
register char *adrs;	/* address to be probed          */
int mode;		/* READ or WRITE                 */
int length;		/* 1, 2, or 4                    */
register char *pVal;	/* where to return value,        */
			/* or ptr to value to be written */
{
    int status, cacheOn;
    int oldLevel = intLock ();			/* lock out CPU */
    FUNCPTR oldVec = (FUNCPTR) intVecGet (IV_BUS_ERROR);	
    int vxMemProbeTrap ();

    intVecSet (IV_BUS_ERROR, vxMemProbeTrap);	/* replace bus error vector */
    cacheOn = sysDataCacheEnab (0);

    /* do probe */

    if (mode == READ)
	status = memProbeSup (length, adrs, pVal);
    else
	status = memProbeSup (length, pVal, adrs);


    /* replace old bus error vector and unlock */

    intVecSet (IV_BUS_ERROR, oldVec);

    if (cacheOn)
	sysDataCacheEnab (1);

    intUnlock (oldLevel);

    return (status);
}

#ifdef SYSDEBUG
/* pre-kernel startup debugging tools 
 */

k_putdata (string, datum, type)
char *string;
int datum;
char type;
{
    k_puts (string);
    if (type == 'h') {
	k_puts ("0x");
	k_puthex (datum);
    } else
	k_putint (datum);
    k_puts ("\r\n");
}

k_puts (string)
char *string;
{
    int i;
    char *p;

    for (p = string, i = 0; i < 256; p++, i++) {
	if (!*p) return 0;
	k_putc (*p);
    }
    return (-1);
}


k_putint (num)
int num;
{
    char digs[15];
    int i;

    if (num == 0)  {
	k_putc ('0');
	return;
    }

    bzero (digs, 15);
    for (i = 0; i < 15; i++) {
	digs [i] = num % 10 + 0x30;
	num /= 10;
    }
    for (i = 14; i >= 0; i--) {
	if (digs[i] != 0x30)
	    break;
    }
    for (; i >= 0; i--)
	k_putc (digs[i]);
}

k_puthex (num)
unsigned int num;
{
    char digs[15];
    int i;
    static char hdigs [] = "0123456789ABCDEF";

    if (num == 0)  {
	k_putc ('0');
	return;
    }

    bzero (digs, 15);

    for (i = 0; i < 15; i++) {
	digs [i] = hdigs[num & 0xf];
	num >>= 4;
    }
    for (i = 14; i >= 0; i--) {
	if (digs[i] != 0x30)
	    break;
    }
    for (; i >= 0; i--)
	k_putc (digs[i]);
}

/************************************************************************/
/* 
 * setwindow
 *
 * move the standard 1-Mbyte vme window to
 * address specified 
 */
setwindow (addr)
unsigned addr;
{
    register int count;

    /* mask the address to upper 20 bits */
    addr &= VDMA_PMASK;
    for (count = 0; count < N_VDMA_REGS; count++) {
	SET_VDMA_REG(count, (addr + (count * VDMA_PSIZE)));
    }
}
#else SYSDEBUG
    k_puts(){}
#endif SYSDEBUG
