/* tyCoDrv.c - The Integrated Solutions VME68225 on-board USART tty handler */

static char *copyright = "Copyright 1986-1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
This is the driver for the Integrated Solutions 68020 on-board serial ports.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, tyCoDrv to
initialize the driver, and tyCoDevCreate to create devices.

TYCODRV
Before using the driver, it must be initialized by calling the routine:
.CS
    tyCoDrv ()
.CE
This routine should be called exactly once, before any reads, writes, or
tyCoDevCreate's.  Normally, it is called from usrRoot(2) in usrConfig(1).

CREATING TERMINAL DEVICES
Before a terminal can be used, it must be created.  This is done
with the tyCoDevCreate call.
Each port to be used should have exactly one device associated with it,
by calling this routine.

.CS
    STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
	char *name;	/* Name to use for this device *
	int channel;	/* Physical channel for this device (0 or 1) *
	int rdBufSize;	/* Read buffer size, in bytes *
	int wrtBufSize;	/* Write buffer size, in bytes *
.CE

For instance, to create the device "/tyCo/0", with buffer sizes of 512 bytes,
the proper call would be:
.CS
   tyCoDevCreate ("/tyCo/0", 0, 512, 512);
.CE
IOCTL
This driver responds to all the same ioctl codes as a normal ty driver.
The baud rates available are 50, 110, 134, 200, 300, 600, 1200, 1050,
2400, 4800, 7200, 9600, 38400.

*/

#include "vxWorks.h"
#include "ioLib.h"
#include "iosLib.h"
#include "memLib.h"
#include "wdLib.h"
#include "tyLib.h"
#include "vme68225.h"

typedef struct			/* BAUD */
    {
    int rate;		/* a baud rate */
    char csrVal;	/* rate to write to the csr reg to get that baud rate */
    } BAUD;

typedef struct			/* TY_CO_DEV */
    {
    TY_DEV tyDev;
    BOOL created;	/* true if this device has really been created */
    char *dr;		/* data port */
    char *csr;		/* clock select register */
    char *cr;		/* control reg */
    char rem;		/* bit for receive-enable mask */
    char tem;		/* bit for transmit-enable mask */
    } TY_CO_DEV;

LOCAL TY_CO_DEV tyCoDv [N_USART_CHANNELS] =	/* device descriptors */
    {
    {{{{NULL}}}, FALSE, VME_U_RHRA, VME_U_CSRA, VME_U_CRA,
	       VME_U_INT_RXA, VME_U_INT_TXA},
    {{{{NULL}}}, FALSE, VME_U_RHRB, VME_U_CSRB, VME_U_CRB,
	       VME_U_INT_RXB, VME_U_INT_TXB}
    };

/* baudTable is a table of the available baud rates, and the values to write
   to the csr reg to get those rates */

LOCAL BAUD baudTable [] =
    {
    {50,	0x00}, {110,	0x11}, {134,	0x22}, {200,	0x33},
    {300,	0x44}, {600,	0x55}, {1200,	0x66}, {1050,	0x77},
    {2400,	0x88}, {4800,	0x99}, {7200,	0xaa}, {9600,	0xbb},
    {38400,	0xcc}
    };

LOCAL int tyCoDrvNum;		/* driver number assigned to this driver */

/* forward declarations */

LOCAL VOID tyCoStartup ();
LOCAL int tyCoOpen ();
LOCAL int tyCoRead ();
LOCAL int tyCoWrite ();
LOCAL STATUS tyCoIoctl ();
LOCAL VOID txInt ();
LOCAL VOID rxInt ();

/*******************************************************************************
*
* tyCoDrv - ty driver initialization routine
*
* This routine initializes the serial driver, sets up interrupt vectors,
* and performs hardware initialization of the serial ports.
*
* This routine must be called in supervisor state, since it does
* physical I/O directly.
*
* RETURNS: OK or ERROR if unable to add device
*/

STATUS tyCoDrv ()

    {
    if (tyCoDrvNum > 0)
	return (OK);		/* driver already installed */

    /* the 68020 board uses a single auto-vectored (level 5) interrupt
       for all DUART exception processing; you have to poll the chip's
       interrupt status register to find the cause of the interrupt. The
       actual interrupt handling routine is in sysLib.c.  All we need
       to do here is connect to it using sysUsartConnect. */

    sysUsartConnect (rxInt, txInt);

    tyCoHrdInit ();

    tyCoDrvNum = iosDrvInstall (tyCoOpen, (FUNCPTR) NULL, tyCoOpen,
				(FUNCPTR) NULL, tyCoRead, tyCoWrite, tyCoIoctl);

    return (tyCoDrvNum == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* tyCoDevCreate - create a device for the onboard ports
*
* This routine creates a device on one of the serial ports.  Each port
* to be used should have exactly one device associated with it, by calling
* this routine.
*
* RETURNS:
*    OK, or
*    ERROR if no driver,
*    device already created, or
*    trouble initializing device.
*/

STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
    char *name;		/* Name to use for this device */
    int channel;	/* Physical channel for this device (0 or 1) */
    int rdBufSize;	/* Read buffer size, in bytes */
    int wrtBufSize;	/* Write buffer size, in bytes */

    {
    if (tyCoDrvNum <= 0)
	{
	errnoSet (S_ioLib_NO_DRIVER);
	return (ERROR);
	}

    /* if there is a device already on this channel, don't do it */

    if (tyCoDv [channel].created)
	return (ERROR);

    /* initialize the ty descriptor, and turn on the bit for this
     * receiver in the interrupt mask */

    if (tyDevInit ((TY_DEV_ID) &tyCoDv [channel], 
		    rdBufSize, wrtBufSize, tyCoStartup) != OK)
	{
	return (ERROR);
	}

    sysImrSet (tyCoDv [channel].rem, 0);

    /* Mark the device as having been created,
     * and add the device to the I/O system.
     */

    tyCoDv [channel].created = TRUE;

    return (iosDevAdd ((DEV_HDR *) &tyCoDv [channel], name, tyCoDrvNum));
    }
/*******************************************************************************
*
* tyCoHrdInit - initialize the USART
*
* This routine initializes the ISI on-board USART.
*
* It must be called in supervisor mode, since it accesses I/O space.
*/

LOCAL VOID tyCoHrdInit ()

    {
    int oldlevel;		/* current interrupt level mask */

    oldlevel = intLock ();	 /* disable interrupts during init */

    *VME_U_CRA = VME_U_MR_RST;
    *VME_U_CRB = VME_U_MR_RST;
    *VME_U_CRA = VME_U_RX_RST;
    *VME_U_CRB = VME_U_RX_RST;
    *VME_U_CRA = VME_U_TX_RST;
    *VME_U_CRB = VME_U_TX_RST;
    *VME_U_CRA = VME_U_ERR_RST;
    *VME_U_CRB = VME_U_ERR_RST;
    *VME_U_CRA = VME_U_BRK_RST;	
    *VME_U_CRB = VME_U_BRK_RST;

    *VME_U_MR1A = VME_U_NO_PAR | VME_U_8_BITS;
    *VME_U_MR1B = VME_U_NO_PAR | VME_U_8_BITS;

    *VME_U_MR2A = VME_U_1_STOP; 
    *VME_U_MR2B = VME_U_1_STOP; 

    *VME_U_CSRA = VME_U_RX_9600 | VME_U_TX_9600;
    *VME_U_CSRB = VME_U_RX_9600 | VME_U_TX_9600;

    /* enable the receivers and transmitters on both channels*/

    *VME_U_CRA = VME_U_TX_EN | VME_U_RX_EN;
    *VME_U_CRB = VME_U_TX_EN | VME_U_RX_EN;
    
    /* all interrupts are masked out: the receiver interrupt will be
       enabled in tyCoDevCreate */

    intUnlock (oldlevel);
    } 

/*******************************************************************************
*
* tyCoOpen - open file to ISI USART
*
* ARGSUSED
*/

LOCAL int tyCoOpen (pTyCoDv, name, mode)
    TY_CO_DEV *pTyCoDv;
    char *name;
    int mode;

    {
    return ((int) pTyCoDv);
    }
/*******************************************************************************
*
* tyCoRead - task level read routine for ISI USART
*
* This routine fields all read calls to the ISI USART.  It
* calls tyRead with a pointer to the appropriate element of ty_dev.
*/

LOCAL int tyCoRead (pTyCoDv, buffer, maxbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int maxbytes;

    {
    return (tyRead ((TY_DEV_ID) pTyCoDv, buffer, maxbytes));
    }
/*******************************************************************************
*
* tyCoWrite - task level write routine for ISI USART
*
* This routine fields all write calls to the ISI USART.  It
* calls tyWrite with a pointer to the appropriate element of ty_dev.
*/

LOCAL int tyCoWrite (pTyCoDv, buffer, nbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int nbytes;

    {
    return (tyWrite ((TY_DEV_ID) pTyCoDv, buffer, nbytes));
    }
/*******************************************************************************
*
* tyCoIoctl - special device control
*
* This routine handles baud rate requests, and passes all other requests
* to tyIoctl.
*/

LOCAL STATUS tyCoIoctl (pTyCoDv, request, arg)
    TY_CO_DEV *pTyCoDv;	/* device to control */
    int request;	/* request code */
    int arg;		/* some argument */

    {
    int i;
    STATUS status;

    switch (request)
	{
	case FIOBAUDRATE:
	    status = ERROR;
	    for (i = 0; i < NELEMENTS (baudTable); i++)
		{
		if (baudTable [i].rate == arg)
		    {
		    *pTyCoDv->csr = baudTable [i].csrVal;
		    status = OK;
		    break;
		    }
		}
	    break;

	default:
	    status = tyIoctl ((TY_DEV_ID) pTyCoDv, request, arg);
	    break;
	}
    return (status);
    }

/*******************************************************************************
*
* rxInt - handle a receiver interrupt
*
* This routine gets called by sysUsartInt to handle a receiver interrupt.
*/

LOCAL VOID rxInt (channel)
    FAST int channel;

    {
    tyIRd ((TY_DEV_ID) &tyCoDv [channel], *tyCoDv [channel].dr);
    }
/*******************************************************************************
*
* txInt - handle a transmitter interrupt
*
* This routine gets called by sysUsartInt to handle a xmitter interrupt.
* If there is another character to be transmitted, it sends it.  If
* not, or if a device has never been created for this channel, we just
* disable the interrupt.
*/

LOCAL VOID txInt (channel)
    FAST int channel;

    {
    char outChar;

    if ((tyCoDv [channel].created) &&
	(tyITx ((TY_DEV_ID) &tyCoDv [channel], &outChar) == OK))

	*(tyCoDv [channel].dr) = outChar;

    else

	/* turn off the transmitter */

	sysImrSet (0, tyCoDv [channel].tem);
    }
/*******************************************************************************
*
* tyCoStartup - Transmitter startup routine
*
* Call interrupt level character output routine for ISI USART.
*/

LOCAL VOID tyCoStartup (pTyCoDv)
    TY_CO_DEV *pTyCoDv;		/* ty device to start up */

    {
    /* all we gotta do is enable the transmitter and it should interrupt
     * to write the next char out. */

    sysImrSet (pTyCoDv->tem, 0);
    }
