/* tyCoDrv.c - Heurikon HK68/V20,V2F,V2FA USART tty handler */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01l,24feb89,jcc  changed misc. names for compatibility with revised m68901.h.
       +dnw+rml  misc cleanups and optimizations
01k,30may88,dnw  changed to v4 names.
01j,30apr88,gae  changed MFP usage.
01i,15nov87,dnw  changed to use INUM_TO_IVEC.
01h,12nov87,ecs  documentation.
01g,24oct87,dnw  removed erroneous set of {} in initialization of tyCoDv.
01f,23oct87,ecs  delinted.
01e,09sep87,dnw  fixed a bad comment
01d,14jul87,jcf  fixed a bad comment
01c,21dec86,dnw  changed to not get include files from default directories.
01b,02dec86,dnw  changed tyCoDevCreate to have standard calling sequence.
01a,24sep86,llk  written by modifying 01c of isi 68020 version.
*/

/*
DESCRIPTION
This is the driver for the Heurikon HK68/V20 Multi-Function Peripheral Chip.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, tyCoDrv to
initialize the driver, and tyCoDevCreate to create devices.

TYCODRV
Before using the driver, it must be initialized by calling the routine:
.CS
    tyCoDrv ()
.CE
This routine should be called exactly once, before any reads, writes, or
tyCoDevCreates.  Normally, it is called from usrRoot.

CREATING TERMINAL DEVICES
Before a terminal can be used, it must be created.  This is done
with the tyCoDevCreate call.
Each port to be used should have exactly one device associated with it,
by calling this routine.

.CS
    STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
	char *name;	/* Name to use for this device
	int channel;	/* Physical channel for this device, always 0
	int rdBufSize;	/* Read buffer size, in bytes
	int wrtBufSize;	/* Write buffer size, in bytes
.CE

For instance, to create the device "/tyCo/0", with buffer sizes of 512 bytes,
the proper call would be:
.CS
   tyCoDevCreate ("/tyCo/0", 0, 512, 512);
.CE
IOCTL
This driver responds to all the same ioctl codes as a normal ty driver.
The baud rates available are 50, 110, 134, 200, 300, 600, 1200, 1050,
2400, 4800, 7200, 9600, 38400.

*/

#include "vxWorks.h"
#include "iv68k.h"
#include "ioLib.h"
#include "iosLib.h"
#include "memLib.h"
#include "wdLib.h"
#include "tyLib.h"
#include "hkv2f.h"

IMPORT int tyWrite ();
IMPORT int tyRead ();

typedef struct			/* TY_CO_DEV */
    {
    TY_DEV tyDev;
    BOOL created;	/* true if this device has really been created */
    char *ucr;		/* USART control register */
    char *rsr;		/* receiver status register */
    char *tsr;		/* transmit status register */
    char *udr;		/* USART data register */
    } TY_CO_DEV;

LOCAL TY_CO_DEV tyCoDv =	/* device descriptors */
    {
    {{{NULL}}}, FALSE,
    MFP_UCR (MFP_BASE_ADR), MFP_RSR (MFP_BASE_ADR),
    MFP_TSR (MFP_BASE_ADR), MFP_UDR (MFP_BASE_ADR)
    };

LOCAL int tyCoDrvNum;		/* driver number assigned to this driver */

typedef struct			/* BAUD */
    {
    int rate;		/* a baud rate */
    char timeConstant;	/* time constant to write to the timers C and D data
			   registers (TCDR & TDDR) to get that baud rate */
    } BAUD;

LOCAL BAUD baudTable [] =
    {
    {150,	0x128}, {300,	0x64}, {600,	0x32}, {1200,	0x16},
    {2400,	0x08}, {4800,	0x04}, {9600,	0x02}, {19200,	0x01}
    };

/* forward declarations */

LOCAL VOID tyCoStartup ();
LOCAL int tyCoOpen ();
LOCAL STATUS tyCoIoctl ();
LOCAL VOID tyCoTxInt ();
LOCAL VOID tyCoRxInt ();

/*******************************************************************************
*
* tyCoDrv - ty driver initialization routine
*
* This routine initializes the serial driver, sets up interrupt vectors,
* and performs hardware initialization of the serial ports.
*
* This routine must be called in supervisor state, since it does
* physical I/O directly.
*/

STATUS tyCoDrv ()

    {
    /* check if driver already installed */

    if (tyCoDrvNum > 0)
	return (OK);

    /* The MFP interrupts on lvl 4 and vectors to the appropriate routines */

    intConnect (INUM_TO_IVEC (MFP_INT_VECT_NUM+MFP_INT_TRANS), tyCoTxInt, NULL);
    intConnect (INUM_TO_IVEC (MFP_INT_VECT_NUM+MFP_INT_RECV), tyCoRxInt, NULL);

    tyCoHrdInit ();

    tyCoDrvNum = iosDrvInstall (tyCoOpen, (FUNCPTR) NULL, tyCoOpen,
				(FUNCPTR) NULL, tyRead, tyWrite, tyCoIoctl);

    return (tyCoDrvNum == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* tyCoDevCreate - create a device for the onboard ports
*
* This routine creates a device on one of the serial ports.  Each port
* to be used should have exactly one device associated with it, by calling
* this routine.
* NOTE: The channel parameter is ignored since the HK68/V20 has only one
* on-board serial port.  The parameter is included only to make the calling
* sequence compatible with other versions. Thus ARGSUSED.
*/

STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
    char *name;		/* Name to use for this device */
    int channel;	/* Physical channel for this device (0 only!) */
    int rdBufSize;	/* Read buffer size, in bytes */
    int wrtBufSize;	/* Write buffer size, in bytes */

    {
    FAST int temp;

    if (tyCoDrvNum <= 0)
	{
	errnoSet (S_ioLib_NO_DRIVER);
	return (ERROR);
	}

    /* if this device already exists, don't create it */

    if (tyCoDv.created)
	return (ERROR);

    /* initialize the ty descriptor */

    if (tyDevInit (&tyCoDv.tyDev, rdBufSize, wrtBufSize, tyCoStartup) != OK)
	return (ERROR);

    /* enable receiver by turning on the bit for this recvr in the int mask */

    temp = *MFP_IMRA (MFP_BASE_ADR);
    temp |= MFP_A_RX_FULL;		/* unmask recv interrupts */
    *MFP_IMRA (MFP_BASE_ADR) = temp;

    /* Mark the device as having been created, and add the device to
     * the I/O system */

    tyCoDv.created = TRUE;
    return (iosDevAdd ((DEV_HDR *) &tyCoDv, name, tyCoDrvNum));
    }
/*******************************************************************************
*
* tyCoHrdInit - initialize the USART
*
* This routine initializes the HK68/V20 USART for the VxWorks environment. 
*
* This routine must be called in supervisor mode, since it accesses I/O space.
*/

LOCAL VOID tyCoHrdInit ()

    {
    int oldlevel;		/* current interrupt level mask */
    FAST int temp;

    oldlevel = intLock ();

    /* 8 data bits, 1 stop bit, no parity */

    *MFP_UCR (MFP_BASE_ADR) = MFP_UCR_16X | 
			      MFP_UCR_8BIT |
			      MFP_UCR_1STOP |
			      MFP_UCR_NO_PARITY;
  
    /* set timer C,D for delay mode, /4 prescale */

    *MFP_TCDCR (MFP_BASE_ADR) = MFP_TCDCR_C_DELAY_4 | 
			       MFP_TCDCR_D_DELAY_4;	

    *MFP_TCDR (MFP_BASE_ADR) = 0x02;	/* set timer C for 9600 baud */
    *MFP_TDDR (MFP_BASE_ADR) = 0x02;	/* set timer D for 9600 baud */

    /* enable the receivers and transmitters on both channels */

    *MFP_RSR (MFP_BASE_ADR) = MFP_RSR_RX_ENABLE;	/* enable rcvr status */
    *MFP_TSR (MFP_BASE_ADR) = MFP_TSR_TX_ENABLE;	/* enable xmit status */

    temp = *MFP_IERA (MFP_BASE_ADR);
    temp |= MFP_A_RX_FULL;		/* enable recv interrupts */
    *MFP_IERA (MFP_BASE_ADR) = temp;

    temp = *MFP_IERA (MFP_BASE_ADR);
    temp |= MFP_A_TX_EMPTY;		/* enable transmit interrupts */
    *MFP_IERA (MFP_BASE_ADR) = temp;

    /* all interrupts are still masked out via the interrupt mask registers: 
     * turn the receiver on and off by twiddling the interrupt mask reg (IMRA),
     * the receiver interrupt will be enabled in tyCoDevCreate,
     * the transmitter interrupt will be enabled in tyCoStartup
     */

    intUnlock (oldlevel);
    } 
/*******************************************************************************
*
* tyCoOpen - open file to HK68/V20 USART
*
* ARGSUSED
*/

LOCAL int tyCoOpen (pTyCoDv, name, mode)
    TY_CO_DEV *pTyCoDv;
    char *name;
    int mode;

    {
    return ((int) pTyCoDv);
    }
/*******************************************************************************
*
* tyCoIoctl - special device control
*
* This routine handles baud rate requests, and passes all other requests
* to tyIoctl.
*
* RETURNS:
*    ERROR if unsupported baud rate, or
*    whatever tyIoctl returns for other requests.
*/

LOCAL STATUS tyCoIoctl (pTyCoDv, request, arg)
    TY_CO_DEV *pTyCoDv;	/* device to control */
    int request;	/* request code */
    int arg;		/* some argument */

    {
    int i;
    FAST STATUS status;

    switch (request)
	{
	case FIOBAUDRATE:

	    /* set timers C,D for delay mode, /4 prescale */

	    *MFP_TCDCR (MFP_BASE_ADR) = MFP_TCDCR_C_DELAY_4 | 
				        MFP_TCDCR_D_DELAY_4;	
	    status = ERROR;
	    for (i = 0; i < NELEMENTS (baudTable); i++)
		{
		if (baudTable [i].rate == arg)
		    {
		    *MFP_TCDR (MFP_BASE_ADR) = baudTable [i].timeConstant;
		    *MFP_TDDR (MFP_BASE_ADR) = baudTable [i].timeConstant;
		    status = OK;
		    break;
		    }
		}
	    break;

	default:
	    status = tyIoctl (&pTyCoDv->tyDev, request, arg);
	    break;
	}

    return (status);
    }
/*******************************************************************************
*
* tyCoTxInt - handle a transmitter interrupt
*
* This routine gets called to handle transmitter interrupts.
* If there is another character to be transmitted, it sends it.  If
* not, or if a device has never been created for this channel, we just
* disable the interrupt.
*/

LOCAL VOID tyCoTxInt ()

    {
    char outChar;
    FAST int temp;

    if ((tyCoDv.created) && (tyITx (&tyCoDv.tyDev, &outChar) == OK))
	*(tyCoDv.udr) = outChar;
    else
	{
	/* turn off the transmitter */

	temp = *MFP_IMRA (MFP_BASE_ADR);
	temp &= ~MFP_A_TX_EMPTY;
	*MFP_IMRA (MFP_BASE_ADR) = temp;
	}
    }
/*******************************************************************************
*
* tyCoRxInt - handle a receiver interrupt
*
* This routine gets called to handle receiver interrupts.
* 
*/

LOCAL VOID tyCoRxInt ()

    {
    tyIRd (&tyCoDv.tyDev, *tyCoDv.udr);
    }
/*******************************************************************************
*
* tyCoStartup - Transmitter startup routine
*
* Call interrupt level character output routine for HK68/V20 USART.
*/

LOCAL VOID tyCoStartup (pTyCoDv)
    TY_CO_DEV *pTyCoDv;		/* ty device to start up */

    {
    char outchar;
    FAST int temp;

    /* write the next char out */

    if (tyITx (&pTyCoDv->tyDev, &outchar) == OK)
	*pTyCoDv->udr = outchar;

    /* turn the transmitter on */

    temp = *MFP_IMRA (MFP_BASE_ADR);
    temp |= MFP_A_TX_EMPTY;
    *MFP_IMRA (MFP_BASE_ADR) = temp;
    }
