/* sysLib.c - Heurikon HK68/V20,V2F,V2FA system dependent library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
02g,19may89,dnw  fixed error in turning on sys ctrlr in sysProcNumSet().
02f,24feb89,jcc  changed misc. names for compatibility with revised m68901.h.
	dnw+rml  changed strategy for dealing with BCL: now BCL value is stored
		   at 0x500 in local memory so that booted system will know what
		   boot ROMs set BCL to, so that the system controller won't get
		   turned off momentarily during boot
		 added enabling of VSB arbiter with SYSFLG_VENDER_0 flag
		 miscellaneous documentation cleanups and optimizations
02e,09sep88,gae  documentation.
02d,09aug88,gae  fixed sysIntGen() to use 8 bit vectors.  Added sysLED().
           +cwp  fixed sysBusTas() to release bus.
02c,04jul88,jcf  made sysBCLSet global, so bootroms can alter sys controller.
		  reincarnated sysGetBCL, renamed it to sysBCLGet.
02b,22jun88,dnw  changed sysHwInit() to not alter system controller bit in BCL.
02a,08jun88,gae  Revised code/documentation for uniformity across targets.
		 made sysMemTop actually size memory.
		 sysClkConnect now returns STATUS.
		 fixed sysClk{Disable,Enable}, sysClkRateSet() doesn't enable.
		 fixed sysAuxClk{Disable,Enable}.
		 fixed sysIntAck to return NULL.
		 changed sys[GS]etBCL() to sysBCL[GS]et() for naming continuity.
		 added sysProcNumGet().
01i,30may88,dnw  changed to v4 names.
01h,27apr88,ecs  added sysExcMsg.
01v,30may88,dnw  changed to v4 names.
01u,30apr88,gae  fixed MFP usage.  Added sys{Aux}Clk{Dis,En}able().
01t,11apr88,ecs  added sysExcMsg.
01s,14mar88,gae  added parameters to sysNv{G,S}et().
01r,20nov87,dnw  changed sysIntGen to take interrupt vec num instead of vec adrs
		 changed to use new defines in config.h
01q,19nov87,jcf  documentation.
01p,16nov87,jcf  changed sysToMonitor to take and pass an argument to bootrom
01o,12nov87,ecs  documentation.
01n,26oct87,jcf  added sysMailboxConnect 
01m,23oct87,dnw  fixed bug in sysMailboxEnable().
		 changed to not enable bus controller if the
		   SYSFLG_NO_SYS_CONTROLLER flag is set in sysFlags.
01l,03jun87,dnw  added sysClkConnect(), sysIntGen(), sysMailboxEnable(),
		   sysBusToLocalAdrs().
		 added more sys... global variables.
		 added sysToMonitor().
		 deleted sysMailBoxAdrs().
		 deleted sysGetProcNum().
		 changed sysIntAck() to only do software ack for levels 2,4,7.
		 changed sysLocalToBusAdrs() to take adrs space arg.
		 changed aux clock to log msg if int occurs while unconnected.
01k,08apr87,llk  wrote sysIntEnable() and sysIntDisable().  They return STATUS.
		 removed unused param from sysHwInit().
01j,02apr87,jlf  documentation.
		 changed calls to tas() to vxTas ().
		 added sysIntEnable() and sysIntDisable().
01i,26mar87,rdc  added sysGetProcNum(), sysSetProcNum(), sysMailBoxAddr(),
		 sysBusTas().
01h,23feb87,jlf  made sysAuxClkConnect return OK, to show there is an
		   auxiliary clock on-board.
		 added sysSetBCL and sysGetBCL.
01g,14feb87,dnw  added sysSetLowAdrs(), sysLocalToBusAdrs(), sysIntAck()
01f,20dec86,dnw  simplified non-volatile ram routines.
		 changed to not get include files from default directories.
01e,08jan87,rdc  enabled level 2 interrupts in sysHwInit.
01d,18dec86,llk  changed non-volatile RAM usage.
		 Now, the whole boot command is stored in NV-RAM.
		 Arguments to crackBootParams must be initialized with EOS.
		 Added sysModel.
01c,30nov86,dnw  changed initialization of bus ctrl latch.
01b,03nov86,rdc  added non-volatile RAM routines.
01a,07oct86,llk  written by modifying 01b of isi 68020 version.
*/

/*
DESCRIPTION
This library contains a set of routines to manipulate the primary functions
of the CPU board.  The goal is to provide a board-independent interface on
which VxWorks and application code can be built in a system-independant way.
Not every feature of every board is supported by this library; a particular
board may have various extensions to the capabilities described here.
Also not every board will support all the functions provided by this library.
And some boards provide some of the functions of this library with hardware
switches, jumpers, or PALs, instead of software controllable registers.

The functions addressed here include:

    initialization functions:
	- initialize hardware to known state
	- identify the system

    memory/address space functions:
	- get on-board memory limit
	- map from local to bus and bus to local address spaces
	- enable/disable cache memory
	- set/get non-volatile RAM

    bus interrupt functions:
	- enable/disable bus interrupt levels
	- generate bus interrupts

    serial channel functions (see tyCoDrv):
	- enable/disable serial channel interrupts
	- set serial channel baud rates
	- get/put bytes from a serial channel

    clock/timer functions:
       - enable/disable timer interrupts
       - set timer periodic rate

    mailbox/location monitor functions:
       - enable mailbox/location monitor interrupts
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "vme.h"
#include "memLib.h"
#include "sysLib.h"
#include "config.h"
#include "hkv2f.h"

IMPORT VOID logMsg ();

#define	HZ	2457600		/* clock rate */

/* globals */

int   sysBus      = BUS;		/* system bus type (VME_BUS, etc) */
int   sysCpu      = CPU;		/* system CPU type (MC680x0) */
char *sysBootLine = BOOT_LINE_ADRS;	/* address of boot line */
char *sysExcMsg   = EXC_MSG_ADRS;	/* catastrophic message area */
int   sysProcNum;			/* processor number of this CPU */
int   sysFlags;				/* boot flags */
char  sysBootHost[BOOT_FIELD_LEN];	/* name of host from which we booted */
char  sysBootFile[BOOT_FIELD_LEN];	/* name of file from which we booted */


/* locals */

LOCAL int sysClkTicksPerSecond = 60;
LOCAL int auxClkTicksPerSecond = 60;
LOCAL int sysClkRunning = FALSE;
LOCAL int auxClkRunning = FALSE;

/* pointer to current value of bus control latch;
 * the BCL is not readable so we have to keep a copy of it so we know what
 * its current value is when we want to change specific bits.
 * Furthermore, we have to know what the boot ROMs set it to so that we
 * don't turn off the system controller, even momentarily, between the
 * ROM initialization and the VxWorks system initialization.
 * To do this, we store the current setting of the BCL at 0x500 in local
 * memory.  Note that this memory gets cleared on reset.
 */
LOCAL ULONG *sysBclPtr = ((ULONG *) (LOCAL_MEM_LOCAL_ADRS + 0x500));


/* forward declarations */

ULONG sysBCLSet ();


/*******************************************************************************
*
* sysModel - return model name of this processor board
*
* Use this routine to find the model name of the system CPU.
*
* RETURNS: pointer to string "Heurikon HK68/V20,V2F,V2FA"
*/

char *sysModel ()
    {
    return ("Heurikon HK68/V20,V2F,V2FA");
    }
/*******************************************************************************
*
* sysHwInit - initialize the hardware on this board
*
* This routine initializes various features of the board.
* It is normally called from usrInit (2) in usrConfig (1).
* This routine must be called in supervisor mode.
*
* The timers are initialized and turned off.
* The Multi-Function Peripheral chip is initialized.
* The bus control register is set for "normal" addressing.
*
* SEE ALSO:
* Heurikon HK68/V20 User's Manual
*/

VOID sysHwInit ()

    {
    /* disable interrupts during init */

    *MFP_IERA(MFP_BASE_ADR) = 0;	/* clear interrupt enable register A */
    *MFP_IERB(MFP_BASE_ADR) = 0;	/* clear interrupt enable register B */

    *MFP_IMRA(MFP_BASE_ADR) = 0;	/* clear interrupt mask register A */
    *MFP_IMRB(MFP_BASE_ADR) = 0;	/* clear interrupt mask register B */

    /* clear interrupt pending and in-service registers */

    *MFP_IPRA(MFP_BASE_ADR) = 0;	/* interrupt pending register A */
    *MFP_IPRB(MFP_BASE_ADR) = 0;	/* interrupt pending register B */
    *MFP_ISRA(MFP_BASE_ADR) = 0;	/* interrupt in-service register A */
    *MFP_ISRB(MFP_BASE_ADR) = 0;	/* interrupt in-service register B */

    /* initialize vector register on MFP chip */

    *MFP_VR(MFP_BASE_ADR) = MFP_INT_VECT_NUM;

    /* initialize timers */

    *MFP_TACR(MFP_BASE_ADR)  = MFP_TCR_DELAY_200;	/* timer A delay mode,
							 * 200 prescale */

    *MFP_TBCR(MFP_BASE_ADR)  = 0;	/* clear timer B control register */
    *MFP_TCDCR(MFP_BASE_ADR) = 0;	/* clear timer C and D control regs. */

    *MFP_TADR(MFP_BASE_ADR)  = 205;	/* timer A data register */
					/* - 205==60Hz w/200 prescale */
    *MFP_TBDR(MFP_BASE_ADR)  = 0;	/* clear timer B data register */
    *MFP_TCDR(MFP_BASE_ADR)  = 0;	/* clear timer C data register */
    *MFP_TDDR(MFP_BASE_ADR)  = 0;	/* clear timer D data register */

    /* set up Heurikon's VME bus control register:
     *   release bus on request, std addressing;
     * NOTE:
     *   We explicitly DON'T affect the system controller bit
     *   in the BCL here because it may have been turned on by the bootroms
     *   during booting, and turning it off here could cause the backplane
     *   to hang.
     */

    (void) sysBCLSet ((ULONG) ~HK_BCL_SYS_CONTROLLER,
	              (ULONG) (HK_BCL_BC_ROR | HK_BCL_RELEASE_HOLD));

    *HK_RELEASE_SYSFAIL = 1;	/* turn off SYSFAIL */

    sysLED (FALSE);		/* turn off user LED */
    }
/*******************************************************************************
*
* sysMemTop - get the address of the top of memory
*
* This routine returns the address of the first missing byte of memory.
*
* NOTE HK68/V2F:
* Memory reappears throughout the the address range 0x200000 - 0x300000.
* The algorithm to determine memory size is described in the HK68/V20 Manual.
*
* RETURNS: address of the first missing byte of memory
*/

char *sysMemTop ()

    {
    LOCAL char memDummy;		/* used for calculating memTop */
    LOCAL char *memTop = NULL;		/* top of memory address */
    FAST char *m0, *m1, *m2, *m3;	/* pointers in memory */
    FAST int  memSize;
    FAST char testValue = 55;

    if (memTop == NULL)
	{
        /* m0 points to a dummy byte in RAM */

        m0 = &memDummy;

        /* get addresses of memory locations instrumental 
	   in calculating RAM size */

        m1 = m0 + 0x40000;		/* get address 256 K after m0 */
        m2 = m0 + 0x100000;		/* get address 1 M after m0 */
        m3 = m0 + 0x400000;		/* get address 4 M after m0 */

        *m1 = *m2 = *m3 = 0;		/* clear out key memory locations */

        *m0 = testValue;

        /* we can determine the size of memory by checking the 
           key memory locations */

        if (*m1 == testValue)
	    memSize = 0x40000;
        else if (*m2 == testValue)
	    memSize = 0x100000;
        else if (*m3 == testValue)
	    memSize = 0x400000;
        else
	    memSize = 0x1000000;

        memTop = (char *) LOCAL_MEM_LOCAL_ADRS + memSize;

	/* memTop is now pointing to the first non-existent address */
        }

    return (memTop);
    }
/*******************************************************************************
*
* sysToMonitor - transfer execution to the ROM monitor
*
* This routine transfers control to the ROM monitor.  It is usually called
* only by the routine reboot, which services control-X, and bus errors at
* interrupt level.  In special circumstances, however, the user may wish
* to introduce a new startType such that a special bootrom facility would be
* enabled.
*
* RETURNS: OK (if we ever continue from the ROM monitor)
*
* INTERNAL
* Note that the "WARM" restart address is at (ROM_BASE_ADRS + 16) bytes.
*/

STATUS sysToMonitor (startType)
    int startType;  /* tells how to boot, values in sysLib.h */

    {
    (* ((FUNCPTR) (ROM_BASE_ADRS + 16))) (startType);

    return (OK);	/* in case we ever continue from rom monitor */
    }

/*******************************************************************************
*
* sysClkConnect - connect a routine to the system clock interrupt
*
* This routine connects the given function to the system clock interrupt.
* It is normally called from usrRoot (2) in usrConfig (1) to connect
* usrClock (2) to the system clock interrupt.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect(2), usrClock(2)
*/

STATUS sysClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each system clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    return (intConnect (INUM_TO_IVEC (INT_VEC_CLOCK), routine, arg));
    }
/*******************************************************************************
*
* sysClkDisable - turn off system clock interrupts
*
* This routine disables the system clock interrupt.
*
* SEE ALSO: sysClkEnable(2)
*/

VOID sysClkDisable ()

    {
    FAST int temp;

    /* disable interrupts */

    temp = *MFP_IERA(MFP_BASE_ADR);
    temp &= ~MFP_A_TIMER_A;
    *MFP_IERA(MFP_BASE_ADR) = temp;

    sysClkRunning = FALSE;
    }
/*******************************************************************************
*
* sysClkEnable - turn system clock interrupts on
*
* This routine enables the system clock interrupt.
*
* SEE ALSO: sysClkDisable(2)
*/

VOID sysClkEnable ()

    {
    FAST int temp;

    /* write the timer value (using a 200 prescale) */

    *MFP_TACR(MFP_BASE_ADR) = MFP_TCR_DELAY_200;  /* delay mode, 200 prescale */
    *MFP_TADR(MFP_BASE_ADR) = HZ / (200 * sysClkTicksPerSecond);

    /* set mask */

    temp = *MFP_IMRA(MFP_BASE_ADR);
    temp |= MFP_A_TIMER_A;
    *MFP_IMRA(MFP_BASE_ADR) = temp;

    /* enable interrupts */

    temp = *MFP_IERA(MFP_BASE_ADR);
    temp |= MFP_A_TIMER_A;
    *MFP_IERA(MFP_BASE_ADR) = temp;

    sysClkRunning = TRUE;
    }
/*******************************************************************************
*
* sysClkRateGet - get the interrupt rate of the system clock
*
* This routine is used to find out the system clock speed.
*
* RETURNS: number of ticks per second of the system clock
*
* SEE ALSO: sysClkRateSet(2)
*/

int sysClkRateGet ()
    
    {
    return (sysClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysClkRateSet - set the interrupt rate of the system clock
*
* This routine sets the interrupt rate of the system clock.
* If the system clock is currently enabled, the clock is disabled, then
* re-enabled with the new rate.
* If the system clock is not enabled when this routine is called, the
* new rate is simply saved, and will be effective when the clock is enabled.
*
* This routine is usually called by usrRoot(2) in usrConfig(1).
*
* SEE ALSO: sysClkRateGet(2), sysClkEnable(2)
*/

VOID sysClkRateSet (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
	sysClkTicksPerSecond = ticksPerSecond;

    if (sysClkRunning)
	{
	sysClkDisable ();
	sysClkEnable ();
	}
    }

/*******************************************************************************
*
* sysAuxClkConnect - connect a routine to the auxiliary clock interrupt
*
* This routine connects the specified routine to the auxiliary clock interrupt.
* Auxiliary clock interrupts are not enabled.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect(2), sysAuxClkDisconnect(2)
*/

STATUS sysAuxClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each auxiliary clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    /* connect the routine to the interrupt vector */

    return (intConnect (INUM_TO_IVEC (MFP_INT_VECT_NUM + MFP_INT_TIMER_B),
			routine, arg));
    }
/*******************************************************************************
*
* sysAuxClkDisconnect - disconnect a routine from the auxiliary clock interrupt
*
* This routine disables the auxiliary clock interrupt and disconnects
* the routine currently connected to the auxiliary clock interrupt.
*
* SEE ALSO: sysAuxClkConnect (2), sysAuxClkDisable (2)
*/

VOID sysAuxClkDisconnect ()

    {
    /* disable the auxiliary clock interrupt */

    sysAuxClkDisable ();

    /* connect dummy routine, just in case */

    sysAuxClkConnect (logMsg, (int) "auxiliary clock interrupt\n");
    }
/*******************************************************************************
*
* sysAuxClkDisable - turn off auxiliary clock interrupts
*
* This routine disables auxiliary clock interrupts.
*/

VOID sysAuxClkDisable ()

    {
    FAST int temp;

    *MFP_TBCR(MFP_BASE_ADR) = MFP_TCR_STOP;	/* timer B stopped */

    /* disable interrupts */

    temp = *MFP_IERA(MFP_BASE_ADR);
    temp &= ~MFP_A_TIMER_B;
    *MFP_IERA(MFP_BASE_ADR) = temp;

    /* mask out interrupts  */

    temp = *MFP_IMRA(MFP_BASE_ADR);
    temp &= ~MFP_A_TIMER_B;
    *MFP_IMRA(MFP_BASE_ADR) = temp;

    auxClkRunning = FALSE;
    }
/*******************************************************************************
*
* sysAuxClkEnable - turn on auxiliary clock interrupts
*
* This routine enables auxiliary clock interrupts.
*/

VOID sysAuxClkEnable ()

    {
    int temp;

    /* Write the timer value (assuming a 200 prescale) */

    *MFP_TBDR(MFP_BASE_ADR) = HZ / (200 * auxClkTicksPerSecond);
    *MFP_TBCR(MFP_BASE_ADR) = MFP_TCR_DELAY_200;	/* timer B delay mode,
							 * 200 prescale */

    /* set mask */

    temp = *MFP_IMRA(MFP_BASE_ADR);
    temp |= MFP_A_TIMER_B;
    *MFP_IMRA(MFP_BASE_ADR) = temp;

    /* enable interrupt */

    temp = *MFP_IERA(MFP_BASE_ADR);
    temp |= MFP_A_TIMER_B;
    *MFP_IERA(MFP_BASE_ADR) = temp;

    auxClkRunning = TRUE;
    }
/*******************************************************************************
*
* sysAuxClkRateGet - get the interrupt rate of the auxiliary clock
*
* This routine returns the interrupt rate of the auxiliary clock.
*
* RETURNS: number of ticks per second of the auxiliary clock
*
* SEE ALSO: sysAuxClkRateSet(2)
*/

int sysAuxClkRateGet ()
    
    {
    return (auxClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysAuxClkRateSet - set the interrupt rate of the auxiliary clock
*
* This routine sets the interrupt rate of the auxiliary clock.
* If the auxiliary clock is currently enabled, the clock is disabled, then
* re-enabled with the new rate.
* If the auxiliary clock is not enabled when this routine is called, the
* new rate is simply saved, and will be effective when the clock is enabled.
*
* SEE ALSO: sysAuxClkRateGet (2), sysAuxClkEnable (2)
*/

VOID sysAuxClkRateSet (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
	auxClkTicksPerSecond = ticksPerSecond;

    if (auxClkRunning)
	{
	sysAuxClkDisable ();
	sysAuxClkEnable ();
	}
    }

/*******************************************************************************
*
* sysLocalToBusAdrs - convert a local address to a bus address
*
* Given a local memory address, this routine returns the VMEbus address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unable to get to that local address from the bus
*
* SEE ALSO: sysBusToLocalAdrs(2)
*/

STATUS sysLocalToBusAdrs (adrsSpace, localAdrs, pBusAdrs)
    int adrsSpace;      /* address space of busAdrs, as defined in vme.h */
    char *localAdrs;    /* local address to convert */
    char **pBusAdrs;    /* where to return bus address */

    {
    if ((localAdrs < (char *) LOCAL_MEM_LOCAL_ADRS) ||
	(localAdrs >= sysMemTop ()))
	{
	/* this is off-board memory - just return local address */

	*pBusAdrs = localAdrs;
	}
    else
	{
	/* this is on-board memory - map to bus address space;
	 *   the following memory mapping is established in sysProcNumSet():
	 *   - only processor 0 has memory on bus,
	 *   - the memory is placed in STD space at address LOCAL_MEM_BUS_ADRS.
	 */

	if ((sysProcNum != 0) ||
	    ((adrsSpace != VME_AM_STD_SUP_PGM) &&
	     (adrsSpace != VME_AM_STD_SUP_DATA) &&
	     (adrsSpace != VME_AM_STD_USR_PGM) &&
	     (adrsSpace != VME_AM_STD_USR_DATA)))
	    return (ERROR);

	*pBusAdrs = localAdrs + (LOCAL_MEM_BUS_ADRS - LOCAL_MEM_LOCAL_ADRS);
	}

    return (OK);
    }
/*******************************************************************************
*
* sysBusToLocalAdrs - convert a bus address to a local address
*
* Given a VMEbus address, this routine returns the local address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unknown address space
*
* SEE ALSO: sysLocalToBusAdrs (2)
*/

STATUS sysBusToLocalAdrs (adrsSpace, busAdrs, pLocalAdrs)
    int adrsSpace;      /* address space of busAdrs, as defined in vme.h */
    char *busAdrs;      /* bus address to convert */
    char **pLocalAdrs;  /* where to return local address */

    {
    switch (adrsSpace)
	{
	case VME_AM_SUP_SHORT_IO:
	case VME_AM_USR_SHORT_IO:
	    *pLocalAdrs = (char *) (0x00ff0000 | (int) busAdrs);
	    return (OK);

	case VME_AM_STD_SUP_ASCENDING:
	case VME_AM_STD_SUP_PGM:
	case VME_AM_STD_SUP_DATA:
	case VME_AM_STD_USR_ASCENDING:
	case VME_AM_STD_USR_PGM:
	case VME_AM_STD_USR_DATA:
	    *pLocalAdrs = (char *) (0x01000000 | (int) busAdrs);
	    return (OK);

	case VME_AM_EXT_SUP_ASCENDING:
	case VME_AM_EXT_SUP_PGM:
	case VME_AM_EXT_SUP_DATA:
	case VME_AM_EXT_USR_ASCENDING:
	case VME_AM_EXT_USR_PGM:
	case VME_AM_EXT_USR_DATA:
	    if (busAdrs < (char *) 0x03000000)
		return (ERROR);

	    *pLocalAdrs = busAdrs;
	    return (OK);

	default:
	    return (ERROR);
	}
    }
/*******************************************************************************
*
* sysIntDisable - disable a VMEbus interrupt level
*
* This routine disables the specified interrupt level.
*
* NOTE HK68/V20:
* Interrupt levels 2, 4, and 7 are handled by the Multi-Function Peripheral
* chip.
* Interrupt levels 1, 3, 5, and 6 are handled by the bus control latch.
*
* RETURNS: OK, or ERROR if intLevel not in range 1-7
*/

STATUS sysIntDisable (intLevel)
    int intLevel;	/* bus interrupt level to disable */

    {
    switch (intLevel)
	{
	case 1:
	    /* disable level 1 interrupts with the bus control latch */
	    (void) sysBCLSet ((ULONG)HK_BCL_IRQ1_ENABLE, (ULONG) 0);
	    break;

	case 2:
	    /* disable level 2 interrupts on the MFP */
	    *MFP_IERB(MFP_BASE_ADR) &= ~MFP_INT_LEVEL_2;	/* disable */
	    *MFP_IMRB(MFP_BASE_ADR) &= ~MFP_INT_LEVEL_2;	/* mask */
	    break;

	case 3:
	    /* disable level 3 interrupts with the bus control latch */
	    (void) sysBCLSet ((ULONG)HK_BCL_IRQ3_ENABLE, (ULONG) 0);
	    break;

	case 4:
	    /* disable level 4 interrupts on the MFP */
	    *MFP_IERB(MFP_BASE_ADR) &= ~MFP_INT_LEVEL_4;	/* disable */
	    *MFP_IMRB(MFP_BASE_ADR) &= ~MFP_INT_LEVEL_4;	/* mask */
	    break;

	case 5:
	    /* disable level 5 interrupts with the bus control latch */
	    (void) sysBCLSet ((ULONG)HK_BCL_IRQ5_ENABLE, (ULONG) 0);
	    break;

	case 6:
	    /* disable level 6 interrupts with the bus control latch */
	    (void) sysBCLSet ((ULONG)HK_BCL_IRQ6_ENABLE, (ULONG) 0);
	    break;

	case 7:
	    /* disable level 7 interrupts on the MFP */
	    *MFP_IERA(MFP_BASE_ADR) &= ~MFP_INT_LEVEL_7;	/* disable */
	    *MFP_IMRA(MFP_BASE_ADR) &= ~MFP_INT_LEVEL_7;	/* mask */
	    break;

	default:
	    return (ERROR);
	}

    return (OK);
    }
/*******************************************************************************
*
* sysIntEnable - enable a bus interrupt level
*
* This routine enables the specified VMEbus interrupt level.
*
* NOTE HK68/V20:
* Interrupt levels 2, 4, and 7 are handled by the Multi-Function Peripheral
* chip.
* Interrupt levels 1, 3, 5, and 6 are handled by the bus control latch.
*
* RETURNS: OK, or ERROR if intLevel not in range 1-7
*/

STATUS sysIntEnable (intLevel)
    int intLevel;	/* interrupt level to enable */

    {
    switch (intLevel)
	{
	case 1:
	    /* enable level 1 interrupts with the bus control latch */
	    (void) sysBCLSet ((ULONG)HK_BCL_IRQ1_ENABLE,
			      (ULONG)HK_BCL_IRQ1_ENABLE);
	    break;

	case 2:
	    /* enable level 2 interrupts on the MFP */
	    *MFP_IERB(MFP_BASE_ADR) |= MFP_INT_LEVEL_2;	/* enable */
	    *MFP_IMRB(MFP_BASE_ADR) |= MFP_INT_LEVEL_2;	/* mask */
	    break;

	case 3:
	    /* enable level 3 interrupts with the bus control latch */
	    (void) sysBCLSet ((ULONG)HK_BCL_IRQ3_ENABLE,
			      (ULONG)HK_BCL_IRQ3_ENABLE);
	    break;

	case 4:
	    /* enable level 4 interrupts on the MFP */
	    *MFP_IERB(MFP_BASE_ADR) |= MFP_INT_LEVEL_4;	/* enable */
	    *MFP_IMRB(MFP_BASE_ADR) |= MFP_INT_LEVEL_4;	/* mask */
	    break;

	case 5:
	    /* enable level 5 interrupts with the bus control latch */
	    (void) sysBCLSet ((ULONG)HK_BCL_IRQ5_ENABLE,
			      (ULONG)HK_BCL_IRQ5_ENABLE);
	    break;

	case 6:
	    /* enable level 6 interrupts with the bus control latch */
	    (void) sysBCLSet ((ULONG)HK_BCL_IRQ6_ENABLE,
			      (ULONG)HK_BCL_IRQ6_ENABLE);
	    break;

	case 7:
	    /* enable level 7 interrupts on the MFP */
	    *MFP_IERA(MFP_BASE_ADR) |= MFP_INT_LEVEL_7;	/* enable */
	    *MFP_IMRA(MFP_BASE_ADR) |= MFP_INT_LEVEL_7;	/* mask */
	    break;

	default:
	    return (ERROR);
	}

    return (OK);
    }
/*******************************************************************************
*
* sysBusIntAck - acknowledge a bus interrupt
*
* This routine acknowledges the specified interrupt.
*
* NOTE HK68/V2F:
* Certain interrupts (IRQ2, IRQ4, and IRQ7) require that particular memory
* locations be read to acknowledge the interrupt and to retrieve the vector
* number put on the bus by the interrupting device.  For other interrupts,
* this routine performs no action since they are acknowledged automatically
* by the hardware.
*
* RETURNS:
*   For 2,4,7:  the vector number put on the bus
*		by the interrupting device,
*   For others: ERROR
*
* SEE ALSO:
* Heurikon HK68/V20 User's Manual
*/

int sysBusIntAck (intLevel)
    int intLevel;	/* bus interrupt level to acknowledge */

    {
    switch (intLevel)
	{
	case 2:  return (*HK_VEC_IRQ2);
	case 4:  return (*HK_VEC_IRQ4);
	case 7:  return (*HK_VEC_IRQ7);
	default: return (ERROR);
	}
    }
/*******************************************************************************
*
* sysBusIntGen - generate a bus interrupt
*
* This routine generates a VMEbus interrupt of the specified level.
*
* RETURNS: OK or ERROR if intLevel not in range 1-7
*/

STATUS sysBusIntGen (intLevel, vector)
    int intLevel;	/* bus interrupt level to generate (1-7) */
    int vector;		/* interrupt vector to generate (0-255) */

    {
    if ((intLevel < 1) || (intLevel > 7))
	return (ERROR);

    *(char *) (0xfea000 + (intLevel << 2)) = vector;

    return (OK);
    }

/*******************************************************************************
*
* sysMailboxConnect - connect a routine to the mailbox interrupt
*
* This routine connects the given routine to the mailbox interrupt.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect(2), sysMailboxEnable(2)
*/

STATUS sysMailboxConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each mailbox interrupt */
    int arg;		/* argument with which to call routine */

    {
    *MFP_IERB(MFP_BASE_ADR) &= ~MFP_MBOX_INT;	/* disable interrupts */
    *MFP_IMRB(MFP_BASE_ADR) &= ~MFP_MBOX_INT;	/* mask interrupts */

    return (intConnect (INUM_TO_IVEC (INT_VEC_BP), routine, arg));
    }
/*******************************************************************************
*
* sysMailboxEnable - enable the mailbox interrupt
*
* This routine enables the mailbox interrupt.
*
* RETURNS: OK
*/

STATUS sysMailboxEnable (mailboxAdrs)
    char *mailboxAdrs;		/* mailbox address */

    {
    FAST int mboxBaseCode;

    /* set mailbox address */

    mboxBaseCode = ((((int) mailboxAdrs & 0xE000) >> 8) |
		    (((int) mailboxAdrs & 0x00F8) >> 3))
			<< HK_BCL_MBOX_BASE_SHIFT;

    (void) sysBCLSet ((ULONG)(HK_BCL_MBOX_BASE_MASK | HK_BCL_MBOX_ENABLE),
    	              (ULONG) (mboxBaseCode | HK_BCL_MBOX_ENABLE));

    /* enable mailbox interrupt */

    *MFP_IERB(MFP_BASE_ADR) |= MFP_MBOX_INT;	/* enable */
    *MFP_IMRB(MFP_BASE_ADR) |= MFP_MBOX_INT;	/* mask */

    return (OK);
    }
/*******************************************************************************
*
* sysProcNumGet - get the processor number
*
* This routine returns the processor number previously set with 
* sysProcNumSet (2).
*
* RETURNS: processor number
*/

int sysProcNumGet ()

    {
    return (sysProcNum);
    }
/*******************************************************************************
*
* sysProcNumSet - set the processor number
*
* Set the processor number for this CPU.  Processor numbers should be
* unique on a single backplane.
*
* NOTE HK68/V20:
* This routine performs several functions:
*   - turns on the VMEbus arbitor if we are processor 0
*     (unless SYSFLG_NO_SYS_CONTROLLER flag is set in sysFlags).
*   - turns on the VSB arbitor if SYSFLG_VENDOR_0 flag is set in sysFlags.
*   - maps local RAM onto the VMEbus if we are processor 0.
*/

VOID sysProcNumSet (procNum)
    int procNum;		/* processor number */

    {
    FAST int memSize;
    FAST ULONG value = 0;
    FAST ULONG mask = HK_BCL_SYS_CONTROLLER |
		      HK_BCL_VSB_ARBE |
		      HK_BCL_VSMAP_MASK;

    if (procNum == 0)
	{
	/* enable system controller, unless sysFlags specifically says not to */

	if ((sysFlags & SYSFLG_NO_SYS_CONTROLLER) == 0)
	    value |= HK_BCL_SYS_CONTROLLER;


	/* set on-board memory in Standard 24 bit address space */

	value |= HK_BCL_MP1_STD | HK_BCL_MP2_24A;


	/* enable bus access to all local memory */

	memSize = sysMemTop () - (char *) LOCAL_MEM_LOCAL_ADRS;

	if (memSize == 0x1000000)
	    value |= HK_BCL_MP3_16M_0V_24A_00;
	else if (memSize == 0x400000)
	    value |= HK_BCL_MP3_4M_0V_24A_00;
	else
	    value |= HK_BCL_MP3_1M_0V_24A_00;
	}


    /* enable VSB controller if SYSFLG_VENDOR_0 is set */

    if ((sysFlags & SYSFLG_VENDOR_0) != 0)
	value |= HK_BCL_VSB_ARBE;


    (void) sysBCLSet (mask, value);

    sysProcNum = procNum;
    }
/*******************************************************************************
*
* sysBusTas - test and set a location across the VMEbus
*
* This routine does a 680x0 test-and-set instruction across the backplane.
*
* NOTE HK68/V20:
* On most CPU's this is equivalent to the vxTas(2) routine, but
* this board has problems doing TAS instructions on the bus - apparently
* the on-board arbiter has problems with read-modify-write cycles.
* To get around the problem we get exclusive access to the
* bus for the duration of the TAS instruction.
*
* RETURNS: TRUE (successful set) or FALSE (failure)
*/

BOOL sysBusTas (addr)
    char *addr;		/* address to be tested and set */

    {
    FAST BOOL retVal;
    FAST ULONG oldBCL;

    /* get exclusive access to bus */

    oldBCL = sysBCLSet ((ULONG) HK_BCL_BC_NO_RELEASE,
			(ULONG) HK_BCL_BC_NO_RELEASE);

    if (*addr)		/* access location to acquire bus */
	;

    retVal = vxTas (addr);

    /* release bus */

    (void) sysBCLSet ((ULONG) HK_BCL_BC_NO_RELEASE, oldBCL);

    if (*addr)		/* access location to release hold */
	;

    return (retVal);
    }


/* miscellaneous support routines */

/*******************************************************************************
*
* sysBCLSet - set the bus control latch
*
* This routine sets or clears bits in the bus control latch.
* The parameter "mask" determines which bits will be changed, and "value"
* determines the value to which those bits will be set.
* In  other words, newBCL = (oldBCL & ~mask) | (value & mask).
*
* NOTE
* Bus arbitration on the HK68/V20 is enabled or disabled by a bit in the
* bus control latch.  Once a value has been chosen for this bit, it must
* not be changed.  If the arbitration bit is changed, bus errors may result.
* In order to maintain this bit, from execution in bootConfig(1) through
* execution in usrConfig(1), an image of the bus control latch is stored
* on the HK68/V20 in low RAM at RAM base plus 0x500 bytes.
*
* RETURNS: previous value of bus control latch
*
* SEE ALSO:
* sysBCLGet(2),
* Heurikon HK68/V20 User's Manual
*/

ULONG sysBCLSet (mask, value)
    ULONG mask;		/* which bits to change */
    ULONG value;	/* what to change bits to */

    {
    FAST ULONG oldBCL;

    oldBCL = *sysBclPtr;
    *sysBclPtr = (oldBCL & ~mask) | (value & mask);
    *HK_BUS_CONTROL_LATCH = *sysBclPtr;

    return (oldBCL);
    }
/*******************************************************************************
*
* sysBCLGet - return the current value of the bus control latch
*
* This routine returns the last value to which the bus control latch
* was set.  It is only effective if all changes to the bus control
* latch are done through sysBCLSet (2).
*/

ULONG sysBCLGet ()

    {
    return (*sysBclPtr);
    }
/*******************************************************************************
*
* sysMemParityError - handle a memory parity error interrupt routine
*
* This routine handles a memory parity error.  It simply logs a message.
*
* NOTE HK68/V2F:
* Memory parity errors generally indicate that uninitialized memory has been
* read.  VxWorks zeroes all of RAM when the system is initialized.  Therefore,
* memory parity errors should not occur.
*/

VOID sysMemParityError ()

    {
    logMsg ("memory parity error\n");
    }
/*******************************************************************************
*
* sysNvRamGet - get the contents of non-volatile RAM
*
* Copies non-volatile memory into string.
* The string will be terminated with an EOS.
*
* RETURNS: OK
*/

STATUS sysNvRamGet (string, strLen, offset)
    FAST char *string;	/* where to copy non-volatile RAM */
    int strLen;		/* maximum number of bytes to copy */
    int offset;		/* (even) byte offset into non-volatile RAM */

    {
    FAST char *pNvRam;
    FAST int ix;

    /* read from NVRAM into shadow RAM */

    pNvRam = (char *) HK_NV_RECALL;
    pNvRam = (char *) (HK_NVRAM_ADDR + ((offset + 1) & ~1));

    /* construct the character string from NV nibbles; nibbles are ordered as:
     *
     * <lower addresses> (high nibble)(low nibble)... <higher addresses>
     *
     * Odd addresses contain low order nibbles, even addresses contain
     * high order nibbles. */

    for (ix = 0; ix < HK_NVRAM_SIZE && ix < strLen; ix++)
	{
	*(string++) = (*pNvRam << 4) | (*(pNvRam + 1) & 0xf);
	pNvRam += 2;
	}

    *string = EOS;

    return (OK);
    }
/*******************************************************************************
*
* sysNvRamSet - write data to non-volatile RAM
*
* Copy string into non-volatile RAM.
*/

VOID sysNvRamSet (string, strLen, offset)
    FAST char *string;	/* string to be copied into non-volatile RAM */
    int strLen;		/* maximum number of bytes to copy */
    int offset;		/* (even) byte offset into non-volatile RAM */

    {
    FAST char *pNvRam = (char *) (HK_NVRAM_ADDR + ((offset + 1) & ~1));
    FAST int ix;

    strLen--;	/* leave room for terminating EOS */

    /* construct the NV nibbles from string; nibbles are ordered as:
     *
     * <lower addresses> (high nibble)(low nibble)... <higher addresses>
     *
     * Odd addresses contain low order nibbles, even addresses contain
     * high order nibbles. */

    for (ix = 0; ix < HK_NVRAM_SIZE; ix++)
	{
	*(pNvRam++) = (string [ix] >> 4) & 0xf;
	*(pNvRam++) = string [ix] & 0xf;
	}

    /* load from the shadow RAM to the NVRAM */

    vxTas ((char *) HK_NV_STORE);
    }
/*******************************************************************************
*
* sysLED - set the user LED
*
* This routine turns the user LED on or off.
*/

VOID sysLED (on)
    BOOL on;	/* TRUE = turn LED on, FALSE = turn LED off */

    {
    (void) sysBCLSet ((ULONG)HK_BCL_USER_LED,
		      on ? (ULONG)0 : (ULONG)HK_BCL_USER_LED);
    }
