/* sysLib.c - Force SYS68K/CPU-37 system dependent library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;


/*
modification history
--------------------
01a,28nov88,jcf  written by modifying 01o of frc21 version.
*/

/*
DESCRIPTION
This library contains a set of routines to manipulate the primary functions
of the CPU board.  The goal is to provide a board-independant interface on
which VxWorks and application code can be built in a system-independant way.
Not every feature of every board is supported by this library; a particular
board may have various extensions to the capabilities described here.
Also not every board will support all the functions provided by this library.
And some boards provide some of the functions of this library with hardware
switches, jumpers, or PALs, instead of software controllable registers.

The funtions addressed here include:

    initialization functions:
	- initialize hardware to known state
	- identify the system

    memory/address space functions:
	- get on-board memory limit
	- map from local to bus and bus to local address spaces
	- enable/disable cache memory
	- set/get non-volatile RAM

    bus interrupt functions:
	- enable/disable bus interrupt levels
	- generate bus interrupts

    serial channel functions (see tyCoDrv):
	- enable/disable serial channel interrupts
	- set serial channel baud rates
	- get/put bytes from a serial channel

    clock/timer functions:
       - enable/disable timer interrupts
       - set timer periodic rate

    mailbox/location monitor functions:
       - enable mailbox/location monitor interrupts
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "vme.h"
#include "ioLib.h"
#include "sysLib.h"
#include "config.h"

IMPORT char end;		/* end of system, created automatically by ld */

IMPORT VOID logMsg ();

/* globals */

int   sysBus      = BUS;		/* system bus type (VME_BUS, etc) */
int   sysCpu      = CPU;		/* system cpu type (MC680x0) */
char *sysBootLine = BOOT_LINE_ADRS;	/* address of boot line */
char *sysExcMsg   = EXC_MSG_ADRS;	/* catastrophic message area */
int   sysProcNum;			/* processor number of this cpu */
int   sysFlags;				/* boot flags */
char  sysBootHost[BOOT_FIELD_LEN];	/* name of host from which we booted */
char  sysBootFile[BOOT_FIELD_LEN];	/* name of file from which we booted */

/* The force 37 has the licensed inet number of 00.80.42.c0.xx.xx where the
 * last four digits are the last four digits of the serial number.
 * Unfortunately there is no way to read the serial number from software so
 * we synthesize a unique ethernet address by placing the lowest two bytes of
 * the internet number in the lowest two bytes of the ethernet address.
 * This is done in usrRoot() (in all/usrConfig.c).
 */

char lnEnetAddr [6] = { 0x00, 0x80, 0x42, 0xc0, 0x00, 0x00 };

/* locals */

LOCAL FUNCPTR sysClkRoutine     = NULL;	/* routine to call on clock interrupt */
LOCAL int sysClkArg             = NULL;	/* its argument */
LOCAL int sysClkTicksPerSecond  = 60;	/* system clock rate */
LOCAL BOOL sysClkRunning        = FALSE;/* system clock enabled? */

LOCAL FUNCPTR sysAuxClkRoutine  = NULL;	/* routine to call on clock interrupt */
LOCAL int sysAuxClkArg          = NULL;	/* its argument */
LOCAL int auxClkTicksPerSecond	= 60;	/* auxiliary clock rate */
LOCAL BOOL auxClkIsConnected    = FALSE;/* hooked up to interrupt yet? */
LOCAL BOOL auxClkIsRunning      = FALSE;/* auxiliary clock enabled? */

LOCAL BOOL sysPitIsConnected    = FALSE;/* hooked up to interrupt yet? */

LOCAL FUNCPTR sysSysfailRoutine = NULL; /* routine to call on sysfail */
LOCAL int sysSysfailArg         = NULL; /* its argument */
LOCAL FUNCPTR sysAcfailRoutine  = NULL; /* routine to call on acfail */
LOCAL int sysAcfailArg          = NULL; /* its argument */

/* forward declarations */

LOCAL VOID sysPitInt ();
LOCAL VOID sysAuxClkInt ();
LOCAL VOID sysAbortInt ();


/*******************************************************************************
*
* sysModel - return model name of the system CPU
*
* This routine returns a pointer to a string with the model name of this CPU.
*
* RETURNS: pointer to string "Force SYS68K/CPU-37".
*/

char *sysModel ()

    {
    return ("Force SYS68K/CPU-37");
    }
/**********************************************************************
*
* sysHwInit - initialize hardware
*
* This routine initializes the board hardware.
* It is normally called from usrInit (2) in usrConfig (1).
* This routine must be called in supervisor mode.
*
* The timers are initialized and turned off.
*/

VOID sysHwInit ()

    {
    char zero = 0; /* used to force compilers to use move inst instead of clr */

    /* Inhibit cacheing of certain regions of the address space (ie. i/o) */

    sysCacheInhibit ();

    /* Initialize PI/T ports */

    *PIT_TCR (FRC37_PIT_BASE_ADRS)   = TIMER_CTL_5;	/* disable timer */
    *PIT_PGCR (FRC37_PIT_BASE_ADRS)  = H12_ENABLE |     /* h4,h3,h2,h1 ints. */
				       H34_ENABLE;	/*  enabled, pmode=0 */
    *PIT_PSRR (FRC37_PIT_BASE_ADRS)  = PIRQ;            /* pc5 = pirq,  */
							/*  pc6 = fp sense */

    *PIT_PACR (FRC37_PIT_BASE_ADRS)  = PORT_SUBMODE_3;	/* submode 1X */
    *PIT_PADDR (FRC37_PIT_BASE_ADRS) = zero;		/* all inputs */

    *PIT_PBCR (FRC37_PIT_BASE_ADRS)  = PORT_SUBMODE_3;	/* submode 1X */
    *PIT_PBDDR (FRC37_PIT_BASE_ADRS) = zero;		/* all inputs */

    *PIT_PCDR (FRC37_PIT_BASE_ADRS)  = ARB_RAT |	/* rel. after 125 ms */
				       ARB_ROR |	/* rel. on request */
				       ARB_NOT_REC |	/* no rel. every cyc. */
				       ARB_RBCLR |	/* rel. on bus clear */
				       VME_DATA32;	/* vme data port = 32b*/

    *PIT_PCDDR (FRC37_PIT_BASE_ADRS) = 0x97;	/* bits 0-2,4 arbiter func */
						/*  bit 7 dynamic bus sizing */

    *PIT_PIVR(FRC37_PIT_BASE_ADRS)   = INT_VEC_PIT;
    *PIT_TVIR(FRC37_PIT_BASE_ADRS)   = INT_VEC_PIT;

    /* Initialize the DUSCC Ports */

        /* reset all of the gizmo's on the DUSCC chip */

    *DUSCC_CCRA (FRC37_DUSCC_BASE_ADRS)	= DUSCC_CCR_TX_RESET_TX;
    *DUSCC_CCRA (FRC37_DUSCC_BASE_ADRS)	= DUSCC_CCR_RX_RESET_RX;
    *DUSCC_CCRA (FRC37_DUSCC_BASE_ADRS)	= DUSCC_CCR_CT_STOP;
    *DUSCC_CCRA (FRC37_DUSCC_BASE_ADRS)	= DUSCC_CCR_DPLL_DISABLE_DPLL;

    *DUSCC_CCRB (FRC37_DUSCC_BASE_ADRS)	= DUSCC_CCR_TX_RESET_TX;
    *DUSCC_CCRB (FRC37_DUSCC_BASE_ADRS)	= DUSCC_CCR_RX_RESET_RX;
    *DUSCC_CCRB (FRC37_DUSCC_BASE_ADRS)	= DUSCC_CCR_CT_STOP;
    *DUSCC_CCRB (FRC37_DUSCC_BASE_ADRS)	= DUSCC_CCR_DPLL_DISABLE_DPLL;

	/* initialize vector register */

    *DUSCC_IVR(FRC37_DUSCC_BASE_ADRS)  = INT_VEC_DUSCC;

        /* enable a, enable b, modify vec w/stat, present vec to cpu */

    *DUSCC_ICR(FRC37_DUSCC_BASE_ADRS)  = DUSCC_ICR_CHAN_B_MAST_INT_EN |
					 DUSCC_ICR_CHAN_A_MAST_INT_EN |
					 DUSCC_ICR_VECT_INCLDS_STATUS |
					 DUSCC_ICR_VECT_MODE_VECT;

        /* rest of DUSCC initialization done in tyCoHrdInit */


    /* Initialize the MFP Ports */

	/* disable interrupts from MFP chip */

    *MFP_IERA(FRC37_MFP_BASE_ADRS) = 0;	/* clear interrupt enable register A */
    *MFP_IERB(FRC37_MFP_BASE_ADRS) = 0;	/* clear interrupt enable register B */

    *MFP_IMRA(FRC37_MFP_BASE_ADRS) = 0;	/* clear interrupt mask register A */
    *MFP_IMRB(FRC37_MFP_BASE_ADRS) = 0;	/* clear interrupt mask register B */

	/* clear interrupt pending and in-service registers */

    *MFP_IPRA(FRC37_MFP_BASE_ADRS) = 0;	/* interrupt pending register A */
    *MFP_IPRB(FRC37_MFP_BASE_ADRS) = 0;	/* interrupt pending register B */
    *MFP_ISRA(FRC37_MFP_BASE_ADRS) = 0;	/* interrupt in-service register A */
    *MFP_ISRB(FRC37_MFP_BASE_ADRS) = 0;	/* interrupt in-service register B */

	/* clear timers */

    *MFP_TACR(FRC37_MFP_BASE_ADRS) = 0;	/* clear timer A control register */
    *MFP_TBCR(FRC37_MFP_BASE_ADRS) = 0;	/* clear timer B control register */
    *MFP_TCDCR(FRC37_MFP_BASE_ADRS) = 0;/* clear timer C & D control regs. */

    *MFP_TADR(FRC37_MFP_BASE_ADRS) = 0;	/* clear timer A data register */
    *MFP_TBDR(FRC37_MFP_BASE_ADRS) = 0;	/* clear timer B data register */
    *MFP_TCDR(FRC37_MFP_BASE_ADRS) = 0;	/* clear timer C data register */
    *MFP_TDDR(FRC37_MFP_BASE_ADRS) = 0;	/* clear timer D data register */

	/* initialize vector register with software acknowledge */

    *MFP_VR(FRC37_MFP_BASE_ADRS) = INT_VEC_MFP | MFP_VR_SOFT;

	/* initialize data direction register */

    *MFP_DDR(FRC37_MFP_BASE_ADRS) = 0x9e;	/* gpio 5,6 are inputs */
    *MFP_AER(FRC37_MFP_BASE_ADRS) = 0;		/* inputs are asserted high */

    *MFP_GPIP(FRC37_MFP_BASE_ADRS) = 0;	/* RTS/DTR inactive,  */
					/*  FDC low density, drive 0, side 0 */

        /* rest of initialization done in tyMfpHrdInit */

    /* Initialize IRHDL */

	/* vector supplied by interrupt handler */
    *IHDL_PTR (FRC37_IRHDL_BASE_ADRS) = IHDL_CR1;	/* SCSI */
    *IHDL_CR  (FRC37_IRHDL_BASE_ADRS) = IHDL_CR_VECTOR| /* vec by int handler */
					IHDL_CR_HIGH;	/* int. active high */
    *IHDL_PTR (FRC37_IRHDL_BASE_ADRS) = IHDL_CR2;	/* LANCE */
    *IHDL_CR  (FRC37_IRHDL_BASE_ADRS) = IHDL_CR_VECTOR;	/* vec by int handler */
    *IHDL_PTR (FRC37_IRHDL_BASE_ADRS) = IHDL_CR3;	/* MFP */
    *IHDL_CR  (FRC37_IRHDL_BASE_ADRS) = IHDL_CR_VECTOR;	/* vec by int handler */
    *IHDL_PTR (FRC37_IRHDL_BASE_ADRS) = IHDL_CR5;	/* PIT */
    *IHDL_CR  (FRC37_IRHDL_BASE_ADRS) = IHDL_CR_VECTOR;	/* vec by int handler */
    *IHDL_PTR (FRC37_IRHDL_BASE_ADRS) = IHDL_CR6;	/* FDC */
    *IHDL_CR  (FRC37_IRHDL_BASE_ADRS) = IHDL_CR_VECTOR;	/* vec by int handler */

	/* vector supplied by interrupting device */
    *IHDL_PTR (FRC37_IRHDL_BASE_ADRS) = IHDL_CR4;	/* DUSCC */
    *IHDL_CR  (FRC37_IRHDL_BASE_ADRS) = zero;		/* vec by DUSCC */


    *IHDL_VEC (FRC37_IRHDL_BASE_ADRS) = INT_VEC_SCSI - 1;	/* SCSI 2nd */

	/* unmask a few choice local interrupts */
    *IHDL_LOCAL_MASK (FRC37_IRHDL_BASE_ADRS) = IHDL_LOCAL_MASK_3 |  /* MFP */
					       IHDL_LOCAL_MASK_4 |  /* DUSCC */
					       IHDL_LOCAL_MASK_5 |  /* PIT */
					       IHDL_LOCAL_MASK_NMI |/* ABORT */
					       IHDL_LOCAL_MASK_NMI_VEC;

	/* mask out all vme interrupts, they're enabled with sysIntEnable */
    *IHDL_BUS_MASK (FRC37_IRHDL_BASE_ADRS)   = zero;


    /* The SCSI, FDC, LANCE are not initialized here but in their drivers */
    }
/*******************************************************************************
*
* sysMemTop - get top of memory address
*
* This routine returns the address of the first missing byte of memory.
*
* The PIT controller has 3 input bits on port b which have the memory size
* encoded.
*
* RETURNS: address of the first missing byte of memory
*/

char *sysMemTop ()

    {
    static char *memTop;

    switch ((*PIT_PBDR (FRC37_PIT_BASE_ADRS)) & MEM_SIZE_MASK)
	{
	case 0x7:
	    memTop = (char *) 0x400000;	/* 4 MB, no nibble mode */
	    break;
	case 0x6:
	    memTop = (char *) 0x100000;	/* 1 MB, no nibble mode */
	    break;
	default : 			/* calculate adrs */
	    memTop = (char *) (0x2000000 >> ((*PIT_PBDR (FRC37_PIT_BASE_ADRS)) &
     			                     MEM_SIZE_MASK));
	    break;
	}

    return (memTop);
    }
/*******************************************************************************
*
* sysToMonitor - transfer to rom monitor
*
* This routine transfers control to the rom monitor.  It is usually called
* only by the routine reboot, which services control-X, and bus errors at
* interrupt level.  In special circumstances, however, the user may wish
* to introduce a new startType such that a special bootrom facility would be
* enabled.
*
* RETURNS: OK (if we ever continue from the rom monitor)
*
* INTERNAL
* Note that the "WARM" restart address is at:
*    ROM_BASE_ADRS + 16: (8 bytes for vectors + 8 bytes for cold start code)
*/
 
STATUS sysToMonitor (startType)
    int startType;  /* parameter is passed to ROM to tell it how to boot, */
                    /* possible types are defined in sysLib.h             */

    {
    (* ((FUNCPTR) ((int)ROM_BASE_ADRS + 16))) (startType);

    return (OK);    /* in case we ever continue from rom monitor */
    }

/*******************************************************************************
*
* sysClkConnect - connect routine to system clock interrupt
*
* This routine connects the given function to the system clock interrupt.
* System clock interrupts are not enabled.
* It is normally called from usrRoot (2) in usrConfig (1) to connect
* usrClock (2) to the system clock interrupt.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), usrClock (2)
*/

STATUS sysClkConnect (routine, arg)
    FUNCPTR routine;
    int arg;

    {
    if (!sysPitIsConnected &&
	intConnect (INUM_TO_IVEC (INT_VEC_CLOCK), sysPitInt, 0) == ERROR)
	{
	return (ERROR);
	}

    sysClkRoutine = routine;
    sysClkArg     = arg;

    sysPitIsConnected = TRUE;


    /* This routine is called early in vxWorks initialization so we connect
     * other useful interrupt handlers here.
     */

    /* connect abort switch */

    (void) intConnect (INUM_TO_IVEC(INT_VEC_ABORT), sysAbortInt, 0);

    return (OK);
    }
/*******************************************************************************
*
* sysClkDisable - turn off system clock interrupts
*/

VOID sysClkDisable ()

    {
    if (sysClkRunning)
	{
	/* disable the PI/T Timer */

	*PIT_TCR (FRC37_PIT_BASE_ADRS) = TIMER_CTL_5;	/* disable timer */

	sysClkRunning = FALSE;
	}
    }
/*******************************************************************************
*
* sysClkEnable - turn system clock interrupts on
*/

VOID sysClkEnable ()

    {
    unsigned int tc;

    /* A 5-bit prescaler (divide-by-32) is assumed to be used
     * on the PI/T clock input.
     */

    tc = CLOCK_FREQ / (sysClkTicksPerSecond * 32);

    /* write the timer value */

    *PIT_CPRL (FRC37_PIT_BASE_ADRS) = tc;
    *PIT_CPRM (FRC37_PIT_BASE_ADRS) = tc >> 8;
    *PIT_CPRH (FRC37_PIT_BASE_ADRS) = tc >> 16;

    /* enable the PI/T Timer */

    *PIT_TCR (FRC37_PIT_BASE_ADRS) = TIMER_CTL_5 | TIMER_ENABLE;

    sysClkRunning = TRUE;
    }
/*******************************************************************************
*
* sysClkRateGet - get rate of system clock
*
* This routine is used to find out the system clock speed.
*
* RETURNS: number of ticks per second of the system clock
*
* SEE ALSO: sysClkRateSet (2)
*/

int sysClkRateGet ()
    
    {
    return (sysClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysClkRateSet - set rate of system clock
*
* This routine sets the clock rate of the system clock.
* System clock interrupts are not enabled.
* It is normally called by usrRoot (2) in usrConfig (1).
*
* SEE ALSO: sysClkRateGet (2), sysClkEnable (2)
*/

VOID sysClkRateSet (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
	sysClkTicksPerSecond = ticksPerSecond;
	
    if (sysClkRunning)
	{
	sysClkDisable ();
	sysClkEnable ();
	}
    }

/*******************************************************************************
*
* sysAuxClkConnect - connect a routine to the auxiliary clock interrupt
*
* This routine connects a user routine to the auxiliary clock interrupt.
* Auxiliary clock interrupts are not enabled.
*
* RETURNS: OK or ERROR if unable to connnect to interrupt
*
* SEE ALSO: intConnect (2), sysAuxClkDisconnect (2)
*/

STATUS sysAuxClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each auxiliary clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    /* connect the routine to the interrupt vector */

    if (!auxClkIsConnected &&
	intConnect (INUM_TO_IVEC (INT_VEC_AUX_CLOCK),
		    sysAuxClkInt, NULL) == ERROR)
	{
	return (ERROR);
	}

    sysAuxClkRoutine   = routine;
    sysAuxClkArg       = arg;

    auxClkIsConnected  = TRUE;

    return (OK);
    }
/*******************************************************************************
*
* sysAuxClkDisconnect - clear the auxiliary clock routine
* 
* This routine disables the auxiliary clock interrupt and disconnects
* the routine currently connected to the auxiliary clock interrupt.
*
* SEE ALSO: sysAuxClkConnect (2)
*/

VOID sysAuxClkDisconnect ()

    {
    /* disable the auxiliary clock interrupt */
     
    sysAuxClkDisable ();
 
    /* connect the dummy routine, just in case */
 
    sysAuxClkConnect (logMsg, (int)"auxiliary clock interrupt\n");
    }
/********************************************************************************
* sysAuxClkDisable - turn off auxiliary clock interrupts
*/
 
VOID sysAuxClkDisable ()
 
    {
    if (auxClkIsRunning)
	{
	/* stop timer */
     
	*DUSCC_CCRB(FRC37_DUSCC_BASE_ADRS) = DUSCC_CCR_CT_STOP;

	auxClkIsRunning = FALSE;
	}
    }
/********************************************************************************
* sysAuxClkEnable - turn auxiliary clock interrupts on
*/
 
VOID sysAuxClkEnable ()
 
    {
    FAST int tc;

    /* calculate the divide ratio, and write it to the timer chip */

    tc = AUX_CLOCK_FREQ / auxClkTicksPerSecond;

    *DUSCC_CTPRHB(FRC37_DUSCC_BASE_ADRS) = MSB(tc);
    *DUSCC_CTPRLB(FRC37_DUSCC_BASE_ADRS) = LSB(tc);

    /* enable and start clock interrupts */

    *DUSCC_CTCRB(FRC37_DUSCC_BASE_ADRS)  = DUSCC_CTCR_ZERO_DET_INTR_ENABLE |
					   DUSCC_CTCR_CLK_X1_CLK;
    *DUSCC_CCRB(FRC37_DUSCC_BASE_ADRS)	 = DUSCC_CCR_CT_START;

    auxClkIsRunning = TRUE;
    }
/*******************************************************************************
*
* sysAuxClkRateGet - get the auxiliary timer frequency
*
* This routine finds out the auxiliary clock speed.
*
* RETURNS: number of ticks per second of the auxiliary clock
*
* SEE ALSO: sysAuxClkRateSet (2)
*/

int sysAuxClkRateGet ()

    {
    return (auxClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysAuxClkRateSet - set rate of auxiliary clock
*
* This routine sets the clock rate of the auxiliary clock.
* Auxiliary clock interrupts are not enabled.
*
* SEE ALSO: sysAuxClkRateGet (2)
*/

VOID sysAuxClkRateSet (ticksPerSecond)
    FAST int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
        auxClkTicksPerSecond = ticksPerSecond;
     
    if (auxClkIsRunning)
        {
        sysAuxClkDisable ();
        sysAuxClkEnable ();
	}
    }

/*******************************************************************************
*
* sysLocalToBusAdrs - convert local address to bus address
*
* Given a local memory address, this routine returns the VME address
* that would have to be accessed to get to that byte.
*
*
* RETURNS: ERROR since the board is not dual-ported
*
* SEE ALSO: sysBusToLocalAdrs (2)
*
* ARGSUSED
*/

STATUS sysLocalToBusAdrs (adrsSpace, localAdrs, pBusAdrs)
    int adrsSpace;	/* bus address space in which busAdrs resides */
    char *localAdrs;	/* local address to convert */
    char **pBusAdrs;	/* where to return bus address */
 
    {
    return (ERROR);
    }
/*******************************************************************************
*
* sysBusToLocalAdrs - convert bus address to local address
*
* Given a VME memory address, this routine returns the local address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK or ERROR if unknown address space
*
* SEE ALSO: sysLocalToBusAdrs (2)
*
* ARGSUSED
*/

STATUS sysBusToLocalAdrs (adrsSpace, busAdrs, pLocalAdrs)
    int adrsSpace;	/* bus address space in which busAdrs resides */
    char *busAdrs;	/* bus address to convert */
    char **pLocalAdrs;	/* where to return local address */

    {
    switch (adrsSpace)
        {
        case VME_AM_SUP_SHORT_IO:
        case VME_AM_USR_SHORT_IO:
            *pLocalAdrs = (char *) (0xfcff0000 |
			            (0x0000ffff & (unsigned int) busAdrs));
            return (OK);
 
        case VME_AM_STD_SUP_ASCENDING:
        case VME_AM_STD_SUP_PGM:
        case VME_AM_STD_SUP_DATA:
        case VME_AM_STD_USR_ASCENDING:
        case VME_AM_STD_USR_PGM:
        case VME_AM_STD_USR_DATA:
	    if ((unsigned int) busAdrs > (unsigned int) 0x00ffffff)
		return (ERROR);
	    *pLocalAdrs = (char *) (0xfd000000 | (unsigned int) busAdrs);
	    return (OK);
 
        case VME_AM_EXT_SUP_ASCENDING:
        case VME_AM_EXT_SUP_PGM:
        case VME_AM_EXT_SUP_DATA:
        case VME_AM_EXT_USR_ASCENDING:
        case VME_AM_EXT_USR_PGM:
        case VME_AM_EXT_USR_DATA:
            if ((unsigned int) busAdrs < (unsigned int) sysMemTop () ||
		(unsigned int) busAdrs  > (unsigned int) 0xfaffffff)
		{
		return (ERROR);
		}
	    *pLocalAdrs = busAdrs;
	    return (OK);
 
        default:
            return (ERROR);
        }
    }
/*******************************************************************************
*
* sysIntDisable - disable bus interrupt level
*
* This routine disables the specified bus interrupt level.
*
* NOTE:
* Interrupt levels are disabled by masking interrupts on the IRHDL chip.
*
* RETURNS: OK, or ERROR if intLevel not in range 1-7
*/

STATUS sysIntDisable (intLevel)
    int intLevel;	/* interrupt level to disable */

    {
    if ((intLevel < 1) || (intLevel > 7))
	return (ERROR);

    *IHDL_BUS_MASK (FRC37_IRHDL_BASE_ADRS) &= ~(1 << intLevel);
		
    return (OK);
    }
/*******************************************************************************
*
* sysIntEnable - enable bus interrupt level
*
* This routine enables the specified bus interrupt level.
*
* NOTE:
* Interrupt levels are enable by unmasking interrupts on the IRHDL chip.
*
* RETURNS: OK, or ERROR if intLevel not in range 1-7
*/

STATUS sysIntEnable (intLevel)
    int intLevel;	/* interrupt level to enable */

    {
    if ((intLevel < 1) || (intLevel > 7))
	return (ERROR);

    *IHDL_BUS_MASK (FRC37_IRHDL_BASE_ADRS) |= (1 << intLevel);

    return (OK);
    }
/*******************************************************************************
*
* sysBusIntAck - acknowledge interrupt
*
* This routine acknowledges a a specific interrupt.
*
* The IRHDL acknowledges the interrupt for us so this routine has no effect.
*
* RETURNS: NULL
*
* ARGSUSED
*/

int sysBusIntAck (intLevel)
    int intLevel;

    {
    return (NULL);
    }
/*******************************************************************************
*
* sysBusIntGen - generate interrupt
*
* This routine generates a VME bus interrupt.
*
* NOTE:
* This board cannot generate a VME bus interrupt so this routine has no effect.
*
* RETURNS: ERROR as board is unable to generate a VME bus interrupt
*
* ARGSUSED
*/

STATUS sysBusIntGen (level, vector)
    int level;		/* VME bus interrupt level to generate (1-7) */
    int vector;		/* interrupt vector to generate (0-255) */

    {
    return (ERROR);
    }

/*******************************************************************************
*
* sysMailboxConnect - connect routine to the mailbox interrupt
*
* This routine connects the given function to the mailbox interrupt.
*
* RETURNS: ERROR, as there are no mailbox facilities.
*
* SEE ALSO: intConnect (2)
*
* ARGSUSED
*/
 
STATUS sysMailboxConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each mailbox interrupt */
    int arg;		/* argument with which to call routine */
 
    {
    return (ERROR);
    }
/*******************************************************************************
*
* sysMailboxEnable - enable mailbox interrupt
*
* This routine enables the mailbox interrupt.
*
* RETURNS: ERROR, as there are no mailbox facilities.
*
* ARGSUSED
*/
 
STATUS sysMailboxEnable (mailboxAdrs)
    char *mailboxAdrs;

    {
    return (ERROR);
    }
/*******************************************************************************
*
* sysProcNumGet - get processor number
*
* This routine returns the processor number previously set with 
* sysProcNumSet (2).
*
* RETURNS: processor number
*/

int sysProcNumGet ()

    {
    return (sysProcNum);
    }
/*******************************************************************************
*
* sysProcNumSet - set processor number
*
* Set the processor number for this CPU.  Processor numbers should be
* unique on a single backplane.
*/

VOID sysProcNumSet (procNum)
    int procNum;

    {
    sysProcNum = procNum;
    }
/*******************************************************************************
*
* sysBusTas - test and set across VME bus
*
* This routine does a 680x0 test-and-set instruction across the backplane.
*
* RETURNS: TRUE (successful set) or FALSE (failure)
*/

BOOL sysBusTas (addr)
    char *addr;

    {
    return (vxTas (addr));
    }

/* miscellaneous support routines */

/*******************************************************************************
*
* sysSysfailConnect - connect routine to the sysfail signal
*
* This routine connects the given function to the sysfail signal.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2)
*/

STATUS sysSysfailConnect (routine, arg)
    FUNCPTR routine;
    int arg;

    {
    if (!sysPitIsConnected &&
	intConnect (INUM_TO_IVEC (INT_VEC_CLOCK), sysPitInt, 0) == ERROR)
	{
	return (ERROR);
	}

    sysSysfailRoutine = routine;
    sysSysfailArg     = arg;

    sysPitIsConnected = TRUE;

    return (OK);
    }
/*******************************************************************************
*
* sysAcfailConnect - connect routine to the acfail signal
*
* This routine connects the given function to the acfail signal.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2)
*/
STATUS sysAcfailConnect (routine, arg)
    FUNCPTR routine;
    int arg;

    {
    if (!sysPitIsConnected &&
	intConnect (INUM_TO_IVEC (INT_VEC_CLOCK), sysPitInt, 0) == ERROR)
	{
	return (ERROR);
	}

    sysAcfailRoutine = routine;
    sysAcfailArg     = arg;

    sysPitIsConnected = TRUE;

    return (OK);
    }
/*******************************************************************************
*
* sysFrontPanelSwitches - read front panel switches
*
* This routines returns the value of the 8 bit rotary switch on the front panel.
*/

int sysFrontPanelSwitches ()

    {
    return (*PIT_PAAR (FRC37_PIT_BASE_ADRS));
    }
/*******************************************************************************
*
* sysLanIntEnable - enable LAN interrupt level
*
* This routine enables interrupts for the on board LAN chip.
* LAN interrupts are controlled by the IRHDL chip.
*
* RETURNS: OK
* ARGSUSED
*/

STATUS sysLanIntEnable (intLevel)
    int intLevel;	/* interrupt level to enable */

    {
    /* unmask local LANCE interrupt */

    *IHDL_LOCAL_MASK (FRC37_IRHDL_BASE_ADRS) |= IHDL_LOCAL_MASK_2;

    return (OK);
    }
/*******************************************************************************
*
* sysLanIntDisable - disable LAN interrupt
*
* This routine disables interrupts for the on board LAN chip.
* LAN interrupts are controlled by the IRHDL chip.
*
* RETURNS: OK
*/

STATUS sysLanIntDisable ()

    {
    /* mask LANCE interrupt */

    *IHDL_LOCAL_MASK (FRC37_IRHDL_BASE_ADRS) &= ~IHDL_LOCAL_MASK_2;

    return (OK);
    }

/*******************************************************************************
*
* sysPitInt - interrupt level processing for the PI/T chip 
*
* This routine handles the interrupts associated with the PI/T chip.
* It is attached to the interrupt vector by any of the following routines:
* sysClkConnect (2), sysSysfailConnect (2), sysAcfailConnect (2).
* The appropriate routine is called and the interrupt is acknowleged.
*/

LOCAL VOID sysPitInt ()

    {
    if (*PIT_PSR (FRC37_PIT_BASE_ADRS) & H3S)
	{
	if (sysAcfailRoutine != NULL)
	    (* sysAcfailRoutine) (sysAcfailArg);/* call acfail routine */

	*PIT_PSR (FRC37_PIT_BASE_ADRS) = H3S;	/* acknowledge acfail int */
	}

    if (*PIT_PSR (FRC37_PIT_BASE_ADRS) & H1S) 
	{
	if (sysSysfailRoutine != NULL)
	    (* sysSysfailRoutine) (sysSysfailArg);/* call sysfail routine */

	*PIT_PSR (FRC37_PIT_BASE_ADRS) = H1S;	/* acknowledge sysfail int */
	}

    if (*PIT_TSR (FRC37_PIT_BASE_ADRS) & ZERO_DET_STATUS)
	{
	if (sysClkRoutine != NULL)
	    (* sysClkRoutine) (sysClkArg);	/* call system clock routine */

	*PIT_TSR (FRC37_PIT_BASE_ADRS) = 1;	/* acknowledge timer int */
	}

    if (*PIT_PSR (FRC37_PIT_BASE_ADRS) & H4S) 
	*PIT_PSR (FRC37_PIT_BASE_ADRS) = H4S;	/* acknowledge int */

    if (*PIT_PSR (FRC37_PIT_BASE_ADRS) & H2S) 
	*PIT_PSR (FRC37_PIT_BASE_ADRS) = H2S;	/* acknowledge int */
    }
/*******************************************************************************
*
* sysAuxClkInt - interrupt level processing for auxiliary clock
*
* This routine handles the auxiliary clock interrupt.  It is attached to the
* clock interrupt vector by the routine sysAuxClkConnect (2).
* The appropriate routine is called and the interrupt is acknowleged.
*/

LOCAL VOID sysAuxClkInt ()

    {
    if (sysAuxClkRoutine != NULL)
	(*sysAuxClkRoutine) (sysAuxClkArg);	/* call system clock routine */

    /* acknowledge interrupt */

    *DUSCC_ICTSRB(FRC37_DUSCC_BASE_ADRS) = DUSCC_ICTSR_CT_ZERO_COUNT;
    }
/*******************************************************************************
*
* sysAbortInt - handle NMI from ABORT switch on front panel
*/

LOCAL VOID sysAbortInt ()

    {
    sysToMonitor (BOOT_WARM_NO_AUTOBOOT);
    }
/******************************************************************************
*
* sysNvRamGet - get contents out of non-volatile RAM
*
* Copies non-volatile memory into string.
* The string will be terminated with an EOS.
*/

VOID sysNvRamGet (string, strLen, offset)
    char *string;    /* where to copy non-volatile RAM           */
    int strLen;      /* maximum number of bytes to copy          */
    int offset;      /* (even) byte offset into non-volatile RAM */

    {
    bcopyBytes (NV_BOOT_LINE + offset, string, strLen);
    string [strLen] = EOS;

    if (string [0] == EOS)
	strcpy (string, DEFAULT_BOOT_LINE);
    }
/*******************************************************************************
*
* sysNvRamSet - write to non-volatile RAM
*
* Copy 'string' into non-volatile RAM.
*
* INTERNAL
* Note that although the NVRAM looks just like normal memory, it is
* a single byte wide chip and will break long words into four transfers.
* Thus there is no significant speed advantage to longs and bcopyBytes (2)
* is used instead just to be safe.  config.h defines the base
* of the boot line space in the NVRAM.
*/

VOID sysNvRamSet (string, strLen, offset)
    char *string;     /* string to be copied into non-volatile RAM */
    int strLen;       /* maximum number of bytes to copy           */
    int offset;       /* (even) byte offset into non-volatile RAM  */
    {
    bcopyBytes (string, NV_BOOT_LINE + offset, strLen);
    }
